// JigsawServletContext.java
// $Id: JigsawServletContext.java,v 1.23 1999/07/22 12:42:16 bmahe Exp $
// (c) COPYRIGHT MIT and INRIA, 1998.
// Please first read the full copyright statement in file COPYRIGHT.html

package org.w3c.jigsaw.servlet;

import java.io.*;
import java.net.*;
import java.util.*;

import javax.servlet.*;

import org.w3c.util.*;
import org.w3c.tools.resources.store.*;
import org.w3c.tools.resources.event.*;
import org.w3c.tools.resources.*;
import org.w3c.jigsaw.frames.*;
import org.w3c.jigsaw.resources.VirtualHostResource;
import org.w3c.jigsaw.http.httpd;

/**
 * @version $Revision: 1.23 $
 * @author  Benot Mah (bmahe@w3.org)
 */
public class JigsawServletContext extends StructureChangedAdapter
                                  implements ServletContext,  
					     PropertyMonitoring
{

    class Logger {
	File             logfile  = null;
	RandomAccessFile log      = null ;
	byte             msgbuf[] = null ;
	boolean          closed   = true;      

	private final String monthnames[] = {
	    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
	    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
	};

	String getDate() {
	    Date now = new Date();
	    return (now.getDate()
		    + "/" + monthnames[now.getMonth()]
		    + "/" + (now.getYear() + 1900)
		    + ((now.getHours() < 10)
		       ? (":0" + now.getHours())
		       : (":" + now.getHours()))
		    + ((now.getMinutes() < 10)
		       ? (":0" + now.getMinutes())
		       : (":" + now.getMinutes()))
		    + ((now.getSeconds() < 10)
		       ? (":0" + now.getSeconds())
		       : (":" + now.getSeconds()))
		    + ((now.getTimezoneOffset() < 0)
		       ? " " + (now.getTimezoneOffset() / 60)
		       : " +" + (now.getTimezoneOffset() / 60)));
	}

	void log(String msg) {
	    msg = "["+getDate()+"] "+msg+"\n";
	    try {
		if ( log == null || closed)
		    openLogFile();
		if ( log != null ) {
		    int len = msg.length() ;
		    if ( len > msgbuf.length ) 
			msgbuf = new byte[len] ;
		    msg.getBytes (0, len, msgbuf, 0) ;
		    log.write (msgbuf, 0, len) ;
		}
	    } catch (IOException ex) {
		System.out.println("Can't write ("+
				   msg+") to logfile ["+
				   logfile+"] : "+ex.getMessage());
	    }
	}
 
	void log(Exception ex, String msg) {
	    log(msg+" : "+ex.getClass().getName()+" ("+ex.getMessage()+")");
	}

	void log(Throwable throwable, String msg) {
	    ByteArrayOutputStream out = new ByteArrayOutputStream();
	    PrintWriter writer = new PrintWriter(out);
	    throwable.printStackTrace(writer);
	    writer.close();
	    String stacktrace = out.toString();
	    log(msg+" "+stacktrace);
	}

	void openLogFile() 
	    throws IOException
	{
	    RandomAccessFile old = log ;
	    log = new RandomAccessFile (logfile, "rw") ;
	    log.seek (log.length()) ;
	    closed = false;
	    if ( old != null )
		old.close () ;
	}

	void close() {
	    try {
		if (log != null)
		    log.close();
		closed = true;
	    } catch (IOException ex) {
		ex.printStackTrace();
	    }
	}

	Logger(File logfile) {
	    this.logfile = logfile;
	    this.msgbuf  = new byte[128] ;
	    log("Servlet Logger started");
	}
    }

    private ResourceReference reference = null;

    private Logger logger = null;

    private ObservableProperties props = null;

    private File directory = null;

    private Hashtable attributes = null;

    protected static String logdir     = "logs" ;

    protected static String deflogfile = "servlets";

    public boolean propertyChanged (String name) {
	if (name.equals(ServletProps.SERVLET_LOG_FILE_P)) {
	    if (logger != null) {
		logger.close();
		File newlogfile = 
		    new File((String) props.get(
					    ServletProps.SERVLET_LOG_FILE_P));
		if (newlogfile.getPath().length() < 1) 
		    newlogfile = getServletLogFile();
		logger = new Logger(newlogfile);
	    }
	}
	return true;
    }

    public void resourceUnloaded(StructureChangedEvent evt){
	if (logger != null) {
	    logger.close();
	}
    }

    protected long getServletTimeout() {
	return props.getLong(ServletProps.SERVLET_TIMEOUT, -1);
    }

    /**
     * A useful utility routine that tries to guess the content-type
     * of an object based upon its extension.
     */
    protected static String guessContentTypeFromName(String fname) {
	return org.w3c.www.mime.Utils.guessContentTypeFromName(fname);
    }

    /**
     * ServletContext implementation - Get the MIME type for given file.
     */

    public String getMimeType(String filename) {
	return guessContentTypeFromName(filename);
    }

    public ServerInterface getServer() {
	try {
	    Resource res = reference.lock();
	    return ((ServletDirectoryFrame)res).getServer();
	} catch(InvalidResourceException ex) {
	    ex.printStackTrace();
	    return null;
	} finally {
	    reference.unlock();
	}
    }

    public File getServletLogFile() {
	ServerInterface server = getServer();
	File logfile = null;
	String file = (String)
	    server.getProperties().getString(ServletProps.SERVLET_LOG_FILE_P,
					     null);
	if (file != null)
	    logfile = new File(file);
	if ((logfile != null) && (logfile.getPath().length() < 1))
	    logfile = null;
	if (logfile == null) {
	    File root_dir = server.getRootDirectory();
	    if (root_dir == null) {
		throw new RuntimeException("unable to build a default "+
					   "value for the servlet log file.");
	    }
	    logfile = new File(new File(root_dir, logdir), deflogfile);
	    server.getProperties().putValue(ServletProps.SERVLET_LOG_FILE_P,
					    logfile.getAbsolutePath());
	}
	return logfile;
    }

    /**
     * ServletContext implementation - Lookup a given servlet.
     * @deprecated since jsdk2.1
     */

    public Servlet getServlet(String name) {
	try {
	    Resource res = reference.lock();
	    return ((ServletDirectoryFrame)res).getServlet(name);
	} catch(InvalidResourceException ex) {
	    ex.printStackTrace();
	    return null;
	} finally {
	    reference.unlock();
	}
    }

    /**
     * ServletContext implementation - Enumerate all servlets within context.
     * @deprecated since jsdk2.1
     */

    public Enumeration getServlets() {
	try {
	    Resource res = reference.lock();
	    return ((ServletDirectoryFrame)res).getServlets();
	} catch(InvalidResourceException ex) {
	    ex.printStackTrace();
	    return null;
	} finally {
	    reference.unlock();
	}
    }

    /**
     * ServletContext implementation - Enumerate all servlets names 
     * within context.
     * @deprecated since jsdk2.1
     */

    public Enumeration getServletNames() {
	try {
	    Resource res = reference.lock();
	    return ((ServletDirectoryFrame)res).getServletNames();
	} catch(InvalidResourceException ex) {
	    ex.printStackTrace();
	    return null;
	} finally {
	    reference.unlock();
	}
    }

    /**
     * ServletContext implementation - Log a message.
     */
    
    public void log(String msg) {
	logger.log(msg);
    }
    
    /**
     * @deprecated since jsdk2.1
     */
    public void log(Exception ex, String msg) {
	logger.log(ex,msg);
    }

    public void log(String message, Throwable throwable) {
	logger.log(throwable, message);
    }

    /**
     * ServletContext implementation - Translate a piece of path.
     */

    public String getRealPath(String path) {
	httpd             server  = (httpd)getServer();
	ResourceReference rr_root = server.getRootReference();
	try {
	    FramedResource root = (FramedResource)rr_root.lock();
	    if (root instanceof VirtualHostResource) {
		//backward to the virtual host resource
		ResourceReference rr  = null;
		ResourceReference rrp = null;
		FramedResource    res = null;
		try {
		    res = (FramedResource)reference.lock();
		    ServletDirectoryFrame fr = (ServletDirectoryFrame)res;
		    rr = fr.getResource().getResourceReference();
		} catch (InvalidResourceException ex) {
		    return null;
		} finally {
		    reference.unlock();
		}

		while (true) {
		    try {
			res = (FramedResource)rr.lock();
			rrp = res.getParent();
			if ((rrp == rr_root) || (rrp == null)) {
			    if (res.definesAttribute("directory")) {
				File dir = 
				    (File)res.getValue("directory", null);
				if (dir != null)
				    return (new File(dir, path)).
					getAbsolutePath();
			    }
			    return null;
			}
		    } catch (InvalidResourceException ex) {
			return null;
		    } finally {
			rr.unlock();
		    }
		    rr = rrp;
		}
	    } else if (root.definesAttribute("directory")) {
		File dir = (File)root.getValue("directory", null);
		if (dir != null)
		    return (new File(dir, path)).getAbsolutePath();
	    }
	    return null;
	} catch (InvalidResourceException ex) {
	    return null;
	} finally {
	    rr_root.unlock();
	}
    }

    /**
     * ServletContext implementation - Get server informations.
     */

    public String getServerInfo() {
	try {
	    Resource res = reference.lock();
	    return ((ServletDirectoryFrame)res).getServerInfo();
	} catch(InvalidResourceException ex) {
	    ex.printStackTrace();
	    return null;
	} finally {
	    reference.unlock();
	}	
    }

    /**
     * ServletContext implementation - Get an attribute value.
     * We map this into the ServletWrapper attributes, without
     * support for name clashes though.
     * @param name The attribute name.
     */
    
    public Object getAttribute(String name) {
	Object attribute = attributes.get(name);
	if (attribute != null) {
	    return attribute;
	} else {
	    try {
		Resource res = reference.lock();
		return ((ServletDirectoryFrame)res).getAttribute(name);
	    } catch(InvalidResourceException ex) {
		ex.printStackTrace();
		return null;
	    } finally {
		reference.unlock();
	    }
	}
    }

    public void setAttribute(String name, Object object) {
	attributes.put(name, object);
    }

    public void removeAttribute(String name) {
	attributes.remove(name);
    }

    public Enumeration getAttributeNames() {
	return attributes.keys();
    }

    private AutoReloadServletLoader loader = null;

    /** 
     * Get or create a suitable LocalServletLoader instance to load 
     * that servlet.
     * @return A LocalServletLoader instance.
     */
    protected synchronized AutoReloadServletLoader getLocalServletLoader() {
	if ( loader == null ) {
	    loader = new AutoReloadServletLoader(this);
	}
	return loader;
    }

    protected synchronized 
	AutoReloadServletLoader createNewLocalServletLoader (boolean keepold) 
    {
	if ((loader != null) && keepold)
	    loader = new AutoReloadServletLoader(loader);
	else
	    loader = new AutoReloadServletLoader(this);
	return loader;
    }

    public File getServletDirectory() {
	return directory;
    }

    //jsdk2.1

    /**
     * Returns a RequestDispatcher object for the specified URL path if 
     * the context knows of an active source (such as a servlet, JSP page,
     * CGI script, etc) of content for the particular path. This format of
     * the URL path must be of the form /dir/dir/file.ext. The servlet 
     * engine is responsible for implementing whatever functionality is 
     * required to wrap the target source with an implementation of
     * the RequestDispatcher interface. 
     * @param urlpath Path to use to look up the target server resource
     */
    public RequestDispatcher getRequestDispatcher(String urlpath) {
	return 
	    JigsawRequestDispatcher.getRequestDispatcher(urlpath, 
							 (httpd)getServer());
    }

    public int getMajorVersion() {
	return 2;
    }

    public int getMinorVersion() {
	return 1;
    }
    
    public ServletContext getContext(String uripath) {
	if (uripath == null)
	    return null;
	//first, find the ServletDirectoryFrame.
	// Prepare for lookup:
	ResourceReference rr_root = null;
	rr_root = ((httpd) getServer()).getRootReference();

	FramedResource root = null;
	root = ((httpd) getServer()).getRoot();

	// Do the lookup:
	ResourceReference r_target = null;
	try {
	    LookupState  ls = new LookupState(uripath);
	    LookupResult lr = new LookupResult(rr_root);
	    root.lookup(ls, lr);
	    r_target = lr.getTarget();
	} catch (Exception ex) {
	    r_target = null;
	}
	//then return its context
	if (r_target != null) {
	    try {
		Resource target = r_target.lock();
		if (target instanceof FramedResource) {
		    ServletDirectoryFrame frame = (ServletDirectoryFrame)
			((FramedResource) target).
		      getFrame("org.w3c.jigsaw.servlet.ServletDirectoryFrame");
		    if (frame != null)
			return frame.getServletContext();
		}
	    } catch (InvalidResourceException ex) {
		// continue
	    } finally {
		r_target.unlock();
	    }
	}
	return null;
    }

    public URL getResource(String path) 
	throws MalformedURLException
    {
	String url = getServletDirectory().getAbsolutePath();
	if ( File.separatorChar != '/' )
	    url = url.replace(File.separatorChar, '/');
	File file = new File(url,path);
	if (! file.exists())
	    return null;
	return new URL("file", "localhost", url+"/"+path);
    }

    public InputStream getResourceAsStream(String path) {
	try {
	    URL resource = getResource(path);
	    if (resource == null)
		return null;
	    try {
		URLConnection c = resource.openConnection();
		return c.getInputStream();
	    } catch (IOException ex) {
		return null;
	    }
	} catch (MalformedURLException ex) {
	    return null;
	}
    }

    /**
     * Create a new ServletContext.
     * @param ref a ResourceReference pointing on a ServletDirectoryFrame.
     */
    protected JigsawServletContext(ResourceReference ref, 
				   ObservableProperties props) 
    {
	this.reference  = ref;
	this.props      = props;
	this.attributes = new Hashtable(3);
	this.logger     = new Logger(getServletLogFile());
	this.loader     = new AutoReloadServletLoader(this);

	props.registerObserver(this);

	try {
	    Resource res = reference.lock();
	    if (! (res instanceof ServletDirectoryFrame)) {
		throw new RuntimeException("This reference is not pointing on"+
					   " a ServletDirectoryFrame.");
	    } else {
		ServletDirectoryFrame sframe = (ServletDirectoryFrame)res;
		FramedResource resource = (FramedResource)sframe.getResource();
		resource.addStructureChangedListener(this);
		if (resource.definesAttribute("directory"))
		    this.directory = 
			(File) resource.getValue("directory", null);
	    }
	} catch(InvalidResourceException ex) {
	    throw new RuntimeException("This reference is pointing on"+
				       " an Invalid ServletDirectoryFrame.");
	} finally {
	    reference.unlock();
	}
    }

}
