// ActiveStream.java
// $Id: ActiveStream.java,v 1.3 1996/09/03 02:48:55 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html


package w3c.www.protocol.http.cache;

import java.io.*;

class ActiveInputStream extends InputStream {
    byte buffer[] = new byte[4096];
    int  off      = 0;
    int  len      = 0;
    
    boolean closed = false;

    private synchronized void waitForInput() {
	while (( ! closed) && (len == 0)) {
	    try {
		wait();
	    } catch (InterruptedException ex) {
	    }
	}
    }

    public synchronized void receive(byte buf[], int boff, int blen) 
	throws IOException
    {
	if ( closed )
	    throw new IOException("Write to closed stream.");
	while ((! closed) && (len != 0)) {
	    try { wait(); } catch (InterruptedException ex) {} 
	}
	System.arraycopy(buf, boff, buffer, 0, blen);
	this.off    = 0;
	this.len    = blen;
	notifyAll();
    }

    public synchronized void close() {
	closed = true;
	notifyAll();
    }
	

    public synchronized int read() {
	waitForInput();
	if (closed && (len == 0))
	    return -1;
	int b = buffer[off++];
	if ( off >= len ) {
	    buffer = null;
	    notifyAll();
	}
	return b;
    }

    public synchronized int read(byte to[], int toff, int tlen) {
	waitForInput();
	// Check for exhausted stream:
	if (closed && (len == 0))
	    return -1;
	// Send the appropriate stuff:
	if (tlen > len) {
	    int snd = len;
	    System.arraycopy(buffer, off, to, toff, len);
	    len = 0;
	    notifyAll();
	    return snd ;
	} else {
	    System.arraycopy(buffer, off, to, toff, tlen);
	    len -= tlen;
	    return tlen;
	}
    }

}

/**
 * ActiveStream is used to tee a stream to the client, while caching it.
 * This class basically mimics the piped streams provided in the java library
 * in a more efficient manner (well, sort of).
 * <p>If any error occurs while writing data back to the client, then the
 * active thread finishes it works, but only streaming data into the sink,
 */

public class ActiveStream extends Thread {
    ActiveInputStream pout    = null;
    TeeMonitor        monitor = null;
    InputStream       src = null;
    OutputStream      dst = null;

    public void run() {
	byte buffer[] = new byte[2048];
	int  chunksz  = 256;

	try {
	    int count = 0;
	    int total = 0;
	    while ((count = src.read(buffer, 0, chunksz)) > 0) {
		// Try to write to the pipe, is still valid:
		if ( pout != null ) {
		    try {
			pout.receive(buffer, 0, count);
		    } catch (IOException ex) {
			pout = null;
		    }
		}
		// Always write to destination:
		dst.write(buffer, 0, count);
		total += count;
		// Increment the chunk size, for improved performance:
		chunksz = Math.min(buffer.length, chunksz << 1);
	    }
	    src.close();
	    dst.close();
	    if ( pout != null )
		pout.close();
	    monitor.notifyTeeSuccess(total);
	} catch (IOException ex) {
	    ex.printStackTrace();
	    monitor.notifyTeeFailure();
	}
    }

    public static InputStream createTee(TeeMonitor monitor
					, InputStream src
					, OutputStream dst) 
	throws IOException
    {
	ActiveStream tee = new ActiveStream();
	tee.monitor = monitor;
	tee.pout    = new ActiveInputStream();
	tee.src     = src;
	tee.dst     = dst;
	tee.start();
	return tee.pout;
    }

    ActiveStream() {
    }


}
