// SessionInputStream.java
// $Id: SessionInputStream.java,v 1.1 1996/04/10 13:53:02 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.mux ;

import java.io.* ;

class InputQueue {
    InputQueue next  = null ;
    byte       buf[] = null ;
    int        off   = -1 ;
    int        len   = -1 ;

    public String toString() {
	return "[InputQueue l="+len+" o="+off+"]";
    }

    InputQueue (byte buf[], int off, int len, InputQueue next) {
	this.buf  = buf ;
	this.off  = off ;
	this.len  = len ;
	this.next = next ;
    }
}

public class SessionInputStream extends InputStream {
    /**
     * Default size of chunk of buffers.
     */
    public final static int BUFSIZE = 512 ;

    InputQueue tail    = null ;
    InputQueue head    = null ;
    Session    session = null ;
    MPReader   reader  = null ;
    boolean    closed  = false ;

    private final int min(int i, int j) {
	return (i<j) ? i : j ;
    }

    private final int max(int i, int j) {
	return (i<j) ? j : i ;
    }

    /**
     * Check this session for already available input.
     * If input is available, dump it into the provided buffer and
     * return the number of bytes copied.
     */

    protected synchronized int checkInput (byte buf[], int off, int len)
	throws IOException
    {
	if ( closed )
	    return -1 ;
	int emited = 0 ;
// System.out.println("SessionInputStream.checkInput: head="+head) ;
	while ((head != null) && (len > 0)) {
	    if ( head.len <= len ) {
// System.out.println("SessionInputStream.checkInput: fullcopy "+head.len) ;
		System.arraycopy (head.buf, head.off, buf, off, head.len);
		off    += head.len ;
		len    -= head.len ;
		emited += head.len ;
		head    = head.next ;
	    } else {
// System.out.println("SessionInputStream.checkInput: partcopy"+len) ;
		System.arraycopy (head.buf, head.off, buf, off, len) ;
		head.len -= len ;
		head.off += len ;
		emited   += len ;
		off      -= len ;
		len       = 0 ;
	    }
	}
	if ( head == null )
	    tail = null ;
// System.out.println ("SessionInputStream: emited "+emited) ;
	return (emited == 0) ? -2 : emited ;
    }

    /**
     * Add the given data to our input queue.
     * The data gets copied to our own buffer.
     */

    protected synchronized void addInput (byte buf[], int off, int len)
	throws IOException
    {
	// If the input stream has been closed, don't woryy about these bytes:
	if ( closed ) {
	    notify() ;
	    return ;
	}
// System.out.println ("SessionInputStream: queueing "+len+" bytes.") ;
	// If we have some input buffer ready, try filling it first:
	if ( tail != null ) {
	    int buflen = tail.buf.length ;
	    // Rotate the input buffer if possible
	    if (tail.off > (buflen >> 1)) {
		System.arraycopy(tail.buf, off, tail.buf, 0, tail.len) ;
		tail.off = 0 ;
	    }
	    // Fill in as much bytes as possible:
	    int dst = tail.off+tail.len ;
	    if (dst < buflen) {
		int capacity = buflen - dst ;
		int cancopy  = min (capacity, len) ;
		System.arraycopy(buf, off, tail.buf, dst, cancopy) ;
		tail.len += cancopy ;
		len      -= cancopy ;
		off      += cancopy ;
	    }
	}
	// If some bytes remain at this point, we have to create a new chunk:
	if ( len > 0 ) {
// System.out.println("SessionInputStream: creating new chunk.") ;
	    int  bufsize = max(len, BUFSIZE) ;
	    byte copy[]  = new byte[bufsize] ;
	    System.arraycopy (buf, off, copy, 0, len) ;
	    if ( tail == null ) {
		head = tail = new InputQueue (copy, 0, len, null) ;
	    } else {
		tail.next = new InputQueue (copy, 0, len, null) ;
		tail      = tail.next ;
	    }
	}
	// Notify any thread waiting for input on this channel:
	notify() ;
    }

    /**
     * Close this session input stream.
     */

    public synchronized void close() 
	throws IOException
    {
	closed = true ;
	head   = null ;
	tail   = null ;
	notifyAll() ;
    }

    /**
     * Read one byte of input.
     */

    public int read () 
	throws IOException
    {
	byte single[] = new byte[1] ;
	read (single, 0, 1) ;
	return (int) single[0] ;
    }

	
    /**
     * Read some bytes on this session input stream.
     */

    public synchronized int read (byte buf[], int offset, int len) 
	throws IOException
    {
	int cnt = -2 ;
	while ( true ) {
// System.out.println (Thread.currentThread()+" InputStream.read() loop.") ;
	    // Check for available input:
	    if ((cnt = checkInput(buf, offset, len)) > -2 )
		return cnt ;
	    // Register as a reader for the multiplexed stream:
	    if ( (cnt = reader.tryRead(session, buf, offset, len)) > -2 ) {
		return cnt ;
	    }
	    if ( closed ) {
		return -1 ;
	    } else if ( cnt != -3 ) {
		try {
// System.out.println (Thread.currentThread()+" InputStream.read() wait.") ;
		    wait();
		} catch (InterruptedException ex) {

		}
	    }
	}
    }

    SessionInputStream (Session session, boolean closed) {
	this.session = session ;
	this.reader  = session.getReader() ;
	this.head    = null ;
	this.tail    = null ;
	this.closed  = closed ;
    }

}
