// DaemonProperties.java
// $Id: DaemonProperties.java,v 1.7 1996/10/01 18:44:17 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.daemon;

import java.util.* ;
import java.io.File ;

import w3c.util.*;

/**
 * Subclass of Properties, to deal with daemon specific usage of them.
 * This class refines the basic Properties class, in order to tune them for
 * ServerHandler specific needs:
 * <ul>
 * <li>Properties can be observed through the PropertyMonitoring interface,
 * <li>Properties are multiplexed accordiong to the server identifier. Two
 * servers can rely on the same set of properties, but each of them can
 * have its own property value. Eg the w3c.jigsaw.http package host property
 * can be set fro server1 to host1 and for server2 to host2. This is done
 * by defining server1.w3c.jigsaw.http.host and server2.w3c.jigsaw.http.host
 * </ul>
 * <p>Each property can be monitored, to allow for dynamic reconfiguration of
 * the server.
 * @see w3c.jigsaw.daemon.PropertyMonitoring
 */

public class DaemonProperties extends ObservableProperties {

    /**
     * Build up the name of a property, specific to the given server handler.
     * @param handler The server handler which gives the context of the 
     * property.
     * @return The full name of the property relevant to this server.
     */

    public String getPropertyName(ServerHandler handler, String name) {
	return handler.getIdentifier() + "." + name;
    }

    public String getPropertyName(String id, String name) {
	return id+"."+name;
    }

    /**
     * Update a property value.
     * Assign a value to a property. If the property value has really changed
     * notify our observers of the change.
     * @param handler The server handler for which you want the property.
     * @param name The name of the property to assign.
     * @param value The new value for this property, or <strong>null</strong>
     *    if the property setting is to be cancelled.
     * @return A boolean <strong>true</strong> if change was accepted by 
     *    our observers, <strong>false</strong> otherwise.
     */
    
    public synchronized boolean putValue (ServerHandler handler
					  , String name
					  , String value) {
	return super.putValue(getPropertyName(handler, name), value);
    }
    
    /**
     * Get this property value, as a boolean.
     * @param handler The server handler for which you want the property.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return A Boolean instance.
     */

    public boolean getBoolean(ServerHandler handler
			      , String name
			      , boolean def) {
	return super.getBoolean(getPropertyName(handler, name), def);
    }

    /**
     * Get this property value, as a String.
     * @param handler The server handler for which you want the property.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return An instance of String.
     */

    public String getString(ServerHandler handler
			    , String name
			    , String def) {
	return super.getString(getPropertyName(handler, name), def);
    }

    public String getString(String id, String name, String def) {
	return super.getString(getPropertyName(id, name), def);
    }

    /**
     * Get this property as a String array.
     * By convention, properties that are get as string arrays should be
     * encoded as a <strong>|</strong> separated list of Strings.
     * @return A String array, or <strong>null</strong> if the property
     * is undefined.
     */

    public String[] getStringArray(ServerHandler handler
				   , String name
				   , String def[]) {
	return super.getStringArray(getPropertyName(handler, name), def);
    }

    /**
     * Get this property value, as an integer.
     * @param handler The server handler for which you want the property.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return An instance of Integer.
     */

    public int getInteger(ServerHandler handler
			  , String name
			  , int def) {
	return super.getInteger(getPropertyName(handler, name), def);
    }

    /**
     * Get this property as a double value.
     * @param handler The server handler for which you want the property.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return An instance of Integer.
     */

    public double getDouble(ServerHandler handler
			    , String name
			    , double def) {
	return super.getDouble(getPropertyName(handler, name), def);
    }

    /**
     * Get this property value, as a File.
     * @param handler The server handler for which you want the property.
     * @param name The name of the property to be fetched.
     * @param def The default value, if the property isn't defined.
     * @return An instance of File.
     */

    public File getFile(ServerHandler handler
			, String name
			, File def) {
	return super.getFile(getPropertyName(handler, name), def);
    }

    /**
     * Build an httpdProperties instance from a Properties instance.
     * @param props The Properties instance.
     */

    public DaemonProperties (Properties props) {
	super (props) ;
    }
   
}

 
