// CvsDirectoryEntity.java
// $Id: CvsDirectoryResource.java,v 1.7 1996/09/26 21:12:27 abaird Exp $
// (c) COPYRIGHT MIT and INRIA, 1996.
// Please first read the full copyright statement in file COPYRIGHT.html

package w3c.jigsaw.cvs;

import java.io.* ;
import java.util.* ;

import w3c.www.mime.*;
import w3c.www.http.*;
import w3c.jigsaw.http.* ;
import w3c.jigsaw.daemon.*;
import w3c.jigsaw.resources.*;
import w3c.jigsaw.forms.*;
import w3c.jigsaw.html.HtmlGenerator ;
import w3c.cvs.* ;

/**
 * This class implements an HTML interface to the CVS directory package.
 */

public class CvsDirectoryResource extends PostableResource {
    protected static HttpCacheControl CACHE_CONTROL_NOCACHE = null;
    protected static HttpTokenList    PRAGMA_NOCACHE       = null;

    static {
	// Pre-compute the no cache directives:
	CACHE_CONTROL_NOCACHE = HttpFactory.makeCacheControl();
	CACHE_CONTROL_NOCACHE.setNoCache();
	// Pre-compute the no cache directives:
	String nocache[] = { "no-cache" };
	PRAGMA_NOCACHE = HttpFactory.makeStringList(nocache);
    }

    /**
     * Have we already computed our CVS environment ?
     */
    private static boolean      inited   = false ;
    /**
     * CVS environment - The path of the cvs command.
     */
    private static String cvspath = null ;
    /**
     * CVS environment - The path of the CVS shell script wrapper.
     */
    private static String cvswrap = null ;
    /**
     * CVS environment - The absolute path of the repository.
     */
    private static String cvsroot = null ;
     
    private static Hashtable    managers = new Hashtable() ;
    private CvsDirectory        cvs      = null ;
    private CvsHandlerInterface handler  = null ;
    private Hashtable           entities = null ;

    /**
     * Get a suitable HTTP resource to display the given cvs entry.
     * @param entry The CVS entry to be displayed.
     * @return An CvsEntryResource, or <strong>null</strong> if none was
     *    found.
     */

    protected HTTPResource getResourceFor (CvsEntry entry) {
	CvsEntryResource resource = null ;
	if ( entities != null ) {
	    resource = (CvsEntryResource) entities.get (entry.getName()) ;
	    if ( resource != null )
		return resource ;
	} else {
	    entities = new Hashtable(33) ;
	}
	resource = new CvsEntryResource (entry) ;
	entities.put (entry.getName(), resource) ;
	return resource ;
    }
	    
    /**
     * Get the CVS manager associated with this resource, or create it.
     */

    protected final CvsDirectory getCvsManager() {
	// Have we already iniitalize ourself ?
	if ( cvs == null ) {
	    // Initialize the CVS env if not done already:
	    initializeCVS(getServer()) ;
	    // Get a cvs manager for the directory:
	    DirectoryResource directory = (DirectoryResource) getParent();
	    try {
		cvs     = new CvsDirectory(cvspath
					   , cvsroot
					   , cvswrap
					   , directory.getDirectory()) ;
		handler = new CvsDirectoryHandler(cvs) ;
	    } catch (CvsException ex) {
		String msg = "Unable to initialize the CVS directory for the "
		    + " resource: " + ex.getMessage() ;
		throw new RuntimeException (msg) ;
	    }
	}
	return cvs ;
    }

    /**
     * Lookup method for the CVS manager.
     * Lookup for a CvsEntry object having the given name, if found, wrap it
     * into aCvsEntryResource object and return it.
     * @param state The current lookup state.
     */

    public boolean lookup (LookupState ls, LookupResult lr) 
	throws HTTPException
    {
	if ( super.lookup(ls, lr) )
	    return true;
	String   name  = ls.getNextComponent() ;
	CvsEntry entry = getCvsManager().lookup (name) ; 
	if ( entry == null ) {
	    lr.setTarget(null);
	    return true;
	} else {
	    HTTPResource target = getResourceFor(entry);
	    lr.setTarget(target);
	    return target.lookup(ls, lr);
	}
    }

    /**
     * Dump one CVS entry into HTML. 
     * The produced HTML is expected to insert
     * itself into a definition list, were each entry has to produce a
     * definition term, with a corresponding check box for generic mark.
     * @param g The HTML generator to use.
     * @param entry The entry to be dumped.
     * @exception CvsFailure If the CVS access failed to this entry.
     */

    private void dumpEntry (HtmlGenerator g, CvsEntry entry)
	throws CvsException
    {
	String name = entry.getName() ;

	g.append ("<tr><th><input type=\"checkbox\" name=\""
		  + name
		  + "\" value=\"mark\">"
		  + "<th align=left><a href=\""+name+"\">"
		  + name
		  + "</a>");
	// dump one line for the status:
	g.append ("<th>") ;
	switch (entry.getStatus()) {
	  case CvsEntry.STATUS_M:  g.append ("locally modified"); break ;
	  case CvsEntry.STATUS_Q:  g.append ("unknown"); break ;
	  case CvsEntry.STATUS_OK: g.append ("up to date"); break ;
	  case CvsEntry.STATUS_A:  g.append ("locally added"); break ;
	  case CvsEntry.STATUS_R:  g.append ("locally removed"); break ;
	  case CvsEntry.STATUS_U:  g.append ("needs checkout"); break ;
	  case CvsEntry.STATUS_C:  g.append ("conflict !"); break ;
	  case CvsEntry.STATUS_P:  g.append ("needs patch"); break ;
	}
	// Emit a diff/log hyper-link only if this makes sense (entry is known)
	if (entry.getStatus() != CvsEntry.STATUS_Q ) {
	    String eurl = getURL() + "/" + name ;
	    g.append ("<th><a href=\""+eurl+"?log\">log</a>") ;
	    g.append ("<th><a href=\""+eurl+"?diff\">diff</a>") ;
	}
	g.append ("\r\n") ;
    }

    /**
     * Dump the content of the directory as a CVS form.
     * The resulting form allows for trigerring actions on the various files.
     */
    
    public Reply get (Request request) 
	throws HTTPException
    {
	// Force a refresh of the CVS directory state:
	try {
	    try {
		Thread.sleep (500, 0) ;
	    } catch (InterruptedException ex) {
	    }
	    getCvsManager().refresh() ;
	} catch (CvsException ex) {
	    Reply error = request.makeReply(HTTP.INTERNAL_SERVER_ERROR) ;
	    HtmlGenerator g = new HtmlGenerator ("CVS error") ;
	    g.append("<h1>CVS error</h1>") ;
	    g.append("<p>While refreshing the CVS status of "
		     + "<code>"+getCvsManager().getDirectory()+"</code>"
		     + " the CVS manager got the following error:<pre><hr>"
		     + ex.getMessage()
		     + "</pre><hr>"
		     + "<p>Try reloading the document by clicking "
		     + "<a href=\""+getURL()+"\">here</a>");
	    error.setStream(g) ;
	    throw new HTTPException (error) ;
	}
	// Display the form:
	HtmlGenerator g = new HtmlGenerator ("CVS for "
					     + getCvsManager().getDirectory());
	g.append("<form action=\""+request.getURL()+"\" method=\"post\">");
	g.append("<table width=\"100%\">") ;
	Enumeration enum = getCvsManager().getEntries() ;
	while ( enum.hasMoreElements()) {
	    CvsEntry entry = (CvsEntry) enum.nextElement() ;
	    try {
		dumpEntry (g, entry) ;
	    } catch (CvsException e) {
		g.append ("<dt>"
			  + entry.getName()
			  + "</dt><dd><strong>CVS Failed</strong>") ;
	    }
	}
	g.append ("</table><hr>") ;
	// Comment area:
	g.append ("<textarea name=\"comment\" rows=\"10\" cols=\"70\">") ;
	g.append ("</textarea>") ;
	// Marked file actions:
	g.append ("<hr>Perform action on marked files only:<p>") ;
	g.append("<input type=\"radio\" name=\"action\" value=\"add\">"
		 + "Add ");
	g.append("<input type=\"radio\" name=\"action\" value=\"remove\">"
		 + "Remove ");
	g.append("<input type=\"radio\" name=\"action\" value=\"update\">"
		 + "Update ");
	g.append("<input type=\"radio\" name=\"action\" value=\"commit\">"
		 + "Commit ");
	g.append("<input type=\"radio\" name=\"action\" value=\"refresh\">"
		 + "Refresh ");
	g.append ("</p>") ;
	// All file actions:
	g.append ("<hr>Perform action on all files:<p>") ;
	g.append("<input type=\"radio\" name=\"action\" value=\"*add\">"
		 + "Add ");
	g.append("<input type=\"radio\" name=\"action\" value=\"*remove\">"
		 + "Remove ");
	g.append("<input type=\"radio\" name=\"action\" value=\"*update\">"
		 + "Update ");
	g.append("<input type=\"radio\" name=\"action\" value=\"*commit\">"
		 + "Commit ");
	g.append("<input type=\"radio\" name=\"action\" value=\"*refresh\">"
		 + "Refresh ");
	g.append ("</p>") ;
	// Submit button:
	g.append ("<hr>") ;
	g.append("<input type=\"submit\" name=\"submit\" value=\"do !\">") ;
	g.append ("</form>") ;
	g.close() ;
	// Send back the reply:
	Reply reply = request.makeReply(HTTP.OK) ;
	reply.setHeaderValue(reply.H_CACHE_CONTROL, CACHE_CONTROL_NOCACHE);
	reply.setHeaderValue(reply.H_PRAGMA, PRAGMA_NOCACHE);
	reply.setStream(g);
	return reply ;
    }

    /**
     * This is were we handle the big post request.
     */
    
    public Reply handle (Request request, URLDecoder data)
	throws HTTPException
    {
	String  action = data.getValue ("action") ;
	boolean all    = false ;
	// no action, is a bug in the generated form (see get)
	if ( action == null ) {
	    Reply error = request.makeReply(HTTP.INTERNAL_SERVER_ERROR) ;
	    throw new HTTPException (error) ;
	}
	// check for an 'all file' action:
	if ( action.startsWith ("*") ) {
	    all    = true ;
	    action = action.substring(1) ;
	} 
	// Get the list of targets to act on:
	Enumeration enum    = getCvsManager().getEntries() ;
	Vector      targets = new Vector() ;
	while ( enum.hasMoreElements() ) {
	    CvsEntry entry = (CvsEntry) enum.nextElement() ;
	    String   name  = entry.getName() ;
	    if ( all ) {
		targets.addElement (entry) ;
	    } else if ( data.getValue (name) != null ) {
		targets.addElement (entry) ;
	    }
	}
	CvsEntry entries[] = new CvsEntry[targets.size()] ;
	targets.copyInto (entries) ;
	// Perform the comand :
	String comment = data.getValue ("comment") ;
	if ( comment != null )
	    handler.perform (request, action, entries, comment) ;
	else
	    handler.perform (request, action, entries) ;
	return get (request) ;
    }

    /**
     * Get a CVS directory manager for the given directory.
     * @param server The server wanting to create the CVS directory manager.
     * @param url The URL of the directory to examine.
     * @param dir The directory to exmaine.
     */

    public static synchronized void initializeCVS(httpd server) {
	// Initialize our CVS settings :
	if ( ! inited ) {
	    if ( server == null ) 
		return ;
	    DaemonProperties props = server.getProperties() ;
	    // Get the cvs command path from props, or from default:
	    cvspath = props.getString(server
				      , CvsDirectory.CVSPATH_P
				      , CvsDirectory.cvspath_def) ;
	    // Get the repository from props, or from default:
	    cvsroot = props.getString(server
				      , CvsDirectory.CVSROOT_P
				      , CvsDirectory.cvsroot_def) ;
	    // Get the wrapper path from props, or build it.
	    cvswrap = props.getString(server
				      , CvsDirectory.CVSWRAP_P
				      , null) ;
	    if ( cvswrap == null ) {
		String root = props.getString (server, httpd.ROOT_P, null);
		if ( root == null ) {
		    // use default CvsDirectory one:
		    cvswrap = CvsDirectory.cvswrap_def ;
		} else {
		    File bin  = new File(root, "bin") ;
		    File wrap = new File(bin, "cvs_wrapper") ;
		    cvswrap = wrap.getAbsolutePath() ;
		}
	    }
	    inited = true ;
	}
    }

    public CvsDirectoryResource() {
    }

}


