/* emulation of 'SIMPLE VM/CMS MAGTAPE FUNCTIONS' via disk file
 * (corresponding to VMS ZT1D.C)
 *
 * w.j.m. aug 1994 (after ZT1D.UIO_C dated 28-feb-1994)
 *
 * int TAPRD(void *buffer,int size,int *trans)
 * int TAPWR(void *buffer,int size)
 * int TAPBSR()		backspace 1 block
 * int TAPFSR()		forward space 1 block
 * int TAPWTM()		write tape mark
 * int TAPREW()		rewind
 * int TAPRUN()		rewind+unload
 *
 * Disk file specified by <ENVNAME>(default=TAPEFILE) environment variable
 * Result (see VM/SP CMS command & macros manual):
 *       0 => o.k.						** ok.
 *       1 => [invalid function or parameter list] ** NOT USED
 *       2 => EOF or EOT					** TAPRD only
 *       3 => "permanent error"					** any error
 *       4 => [invalid device id] ** NOT USED
 *       5 => [tape not attached] ** returned on open() error	** ok.
 *       6 => write protection
 *       7 => [invalid device attached] ** NOT USED
 *       8 => "incorrect length" ** NOT USED
 */

#ifdef TAP2	/* re-name TAP* routines to TAP2* */
#define TAPRD TAP2RD
#define TAPWR TAP2WR
#define TAPBSR TAP2BSR
#define TAPFSR TAP2FSR
#define TAPWTM TAP2WTM
#define TAPREW TAP2REW
#define TAPRUN TAP2RUN
#define TAPSHOW TAP2SHOW

#define TAPE_hwl TAP2_hwl
#endif	/* TAP2 */

#ifndef ENVNAME
#define ENVNAME "TAPEFILE"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>

#include <fcntl.h>

extern char *getenv();

int TAPE_hwl;


#if TRACE
static void trace_tape(char *,int,int,int);	/* forward */
static int tpos;			/* true position (in blocks) */
#endif


#if !defined(VMS_LITTLEENDIAN)
static int/*logical*/ bigendian;	/* set in tapopen() init. */

static int I4(unsigned char *bp)
{
    unsigned char x[4];

    if(bigendian) {
      x[0] = bp[3]; x[1] = bp[2]; x[2] = bp[1]; x[3] = bp[0];
    } else {
      x[0] = bp[0]; x[1] = bp[1]; x[2] = bp[2]; x[3] = bp[3];
    }
    return *((int *)x);
}
static void I4STORE(unsigned char *bp, int i)
{
    unsigned char *x = (unsigned char *)(&i);

    if(bigendian) {
      bp[0] = x[3]; bp[1] = x[2]; bp[2] = x[1]; bp[3] = x[0];
    } else {
      bp[0] = x[0]; bp[1] = x[1]; bp[2] = x[2]; bp[3] = x[3];
    }
}
#else	/* VMS_LITTLEENDIAN: can load integer from arbitrary byte pointer */
#define I4(bp) (*((int *)(bp)))
#define I4STORE(bp,i) (*((int *)(bp))) = i
#endif


/* 'disk tape' structure (sizeof(int) == 32 assumed, _LITTLE-ENDIAN_):
 *	int l0; char [l0]; int l0; int l1; char [l1]; int l1; ...
 *		l=0 stands for end of written area,
 *		l=1 stands for tape mark
 *
 * This is compatible with a unix [little-endian] FORTRAN BINARY file
 * (checked for ULTRIX & DEC OSF/1).
 *
 * the very first "record" always contains the fixed string "ZT1 "
 *
 * NOTE: no EOT handling (yet)!
 */

#define T_MAXBLOCK 0xFFFF

/* room for tape block & 2 length fields */
static unsigned char tbuf[4 + 4 + T_MAXBLOCK];

/* current byte offset in file (point between 'prev' & 'next' record) */
static int tp = 0;

/* file descriptor */
static int tapfd;

/* read-only flag */
static int/*logical*/ t_ro;

#define T_EOM 0
#define T_MARK 1

static const struct T_magic {
  char l0_1[4];
  char magic[4];
  char l0_2[4];
} t_magic = {{4,0,0,0},{'Z','T','1',' '},{4,0,0,0}};

static int t_nextlen,t_prevlen;

#define T_nextlen t_nextlen
#define T_prevlen t_prevlen
#define T_data tbuf			/* tape block from TOP_read() */

#define T_prevpos (tp - (T_prevlen + 8))
#define T_nextpos (tp + (T_nextlen + 8))

#define T_bot (tp == sizeof(t_magic))
#define T_eom (T_nextlen == T_EOM)

#define T_abort() do {\
	fprintf(stderr,"DSKTAP aborting ...\n");\
	abort();\
} while(0)

#define T_checklen(i) do {\
	if(i < 0 || i > T_MAXBLOCK) {\
		fprintf(stderr,"Bad block length: %d\n",i);\
		T_abort();\
	}\
} while(0)

/*****/

static long diskpos;		/* current file position (byte offset) */
static int diskeof = -1;	/* new EOF position to be set, or -1 */

static void disk_write(int,int l,unsigned char *);	/*forward*/


static void disk_seek(int p)
{
  {
    if(diskeof > 0 && p < diskeof) {
      unsigned char teom[4];

      I4STORE(teom,T_EOM);
      disk_write(diskeof,4,teom);

#if FTRUNCATE
      /* NOTE: `diskeof' has been updated by disk_write() ! */
      if(ftruncate(tapfd,diskeof) != 0) {
	perror("DSKTAP ftruncate()");
      }		/* don't abort, since truncation is just optional */
#endif

      diskeof = -1;
    }
  }
  diskpos = p;
  if(lseek(tapfd,diskpos,SEEK_SET) < 0) {
    perror("lseek()");
    T_abort();
  }
}

static void disk_truncate(int p)
{
  if(p != diskpos) disk_seek(p);
  diskeof = p;
}

static void disk_write(int p, int l, unsigned char *b)
{
  int k;
  int ll = l;
  unsigned char *bb = b;


  if(p != diskpos) disk_seek(p);

  while(ll > 0) {
    if((k = write(tapfd,bb,ll)) < 0) {
      perror("write()");
      T_abort();
    }
    ll -= k;
    bb += k;
  }
  diskpos = diskeof = p + l;
}

static void disk_read(int p, int l, unsigned char *b)  /* ggf. fill in T_EOM */
{
  int k;
  int ll = l;
  unsigned char *bb = b;


  if(p != diskpos) disk_seek(p);

  while(ll > 0) {
    k = read(tapfd,bb,ll);
    if(k < 0) {
      perror("read()");
      T_abort();
    }
    if(k == 0) {			/* disk file EOF */
      if(ll == 4) {			/* 4 bytes remain => ok */
	I4STORE(bb,T_EOM);		/* provide sentinel */
	diskpos = p + (l - 4);
	return;				/* success */
      } else {
	fprintf(stderr,"read() EOF at %d\n",p + l - ll);
	T_abort();
      }
    } else {
      ll -= k;
      bb += k;
    }
  }

  diskpos = p + l;
}
	

static void TOP_rewind()	/* permitted to occur at BOT, verify magic */
{
  if(T_bot) return;

  tp = sizeof(t_magic);
  disk_read(0,sizeof(t_magic) + 4,tbuf);
  T_nextlen = I4(tbuf + sizeof(t_magic));
  T_checklen(T_nextlen);
  if(memcmp(&t_magic,tbuf,sizeof(t_magic))) T_abort();

  T_prevlen = T_EOM;
}

static void TOP_bsr()		/* must not occur at BOT */
{
  tp = T_prevpos;
  if(T_bot) {
    int oldprevlen = T_prevlen;

    TOP_rewind();
    if(T_nextlen != oldprevlen) T_abort();
  } else {
    disk_read(tp - 4,2*4,tbuf);
    if(I4(tbuf + 4) != T_prevlen) T_abort();
    T_nextlen = T_prevlen;
    T_prevlen = I4(tbuf);
    T_checklen(T_prevlen);
  }
}

static void TOP_fsr()		/* must not occur at EOM */
{
  tp = T_nextpos;
  disk_read(tp - 4,2*4,tbuf);
  if(I4(tbuf) != T_nextlen) T_abort();
  T_prevlen = T_nextlen;
  T_nextlen = I4(tbuf + 4);
  T_checklen(T_nextlen);
}

static void TOP_read()		/* must not occur at EOM */
{
  disk_read(tp + 4,T_nextlen + 2*4,tbuf);
  tp = T_nextpos;
  if(I4(tbuf + T_nextlen) != T_nextlen) T_abort();
  T_prevlen = T_nextlen;
  T_nextlen = I4(tbuf + T_nextlen + 4);
  T_checklen(T_nextlen);
}

static void TOP_write(int l, unsigned char *b)	/* also used for writemark */
{
  unsigned char tlen[4];

  
  T_checklen(l);

  if(!T_eom) disk_truncate(tp);

  T_nextlen = l;
  I4STORE(tlen,T_nextlen);
  disk_write(tp,4,tlen);
  disk_write(tp + 4,T_nextlen,b);
  disk_write(tp + 4 + T_nextlen,4,tlen);
  tp = T_nextpos;

  T_prevlen = l;
  T_nextlen = T_EOM;
}


/*****/

static int/*logical*/ opened = 0;


static int tapopen(void)
{
  char *fn,*mp;


  assert(sizeof(int) == 4);	/* we use 'int' for 32 bit values */

  {		/* test for this machine's endian-ness */
    static int c1 = 1;
    
#if !defined(VMS_LITTLEENDIAN)
    bigendian = ( *((char *)(&c1)) != 1 );
#else
    assert(*((char *)c1) == 1);
#endif
  }

  if(!opened) {
    fn = getenv(ENVNAME);
    if(fn == NULL || *fn == '\0') {
      fprintf(stderr,"DSKTAP: %s (disk file name) is undefined\n",ENVNAME);
      return 5;
    }

    tp = 0;
    diskpos = 0;
    diskeof = -1;

    if((tapfd = open(fn,O_RDWR,0)) >= 0) {
      mp = "opened r/w";
      t_ro = 0;
    } else if((tapfd = open(fn,O_RDONLY,0)) >= 0) {
      mp = "opened r/o";
      t_ro = 1;
    } else if((tapfd = open(fn,O_RDWR | O_CREAT,0666)) >= 0) {
      mp = "created";
      t_ro = 0;
      disk_write(0,sizeof(t_magic),(unsigned char *)&t_magic);
    } else {
      fprintf(stderr,"Error opening file %s\n",fn);
      perror("DSKTAP open()/creat()");
      return 5;
    }
    fprintf(stderr,"File %s %s\n",fn,mp);

    TAPE_hwl = t_ro;	/* make read-only status know externally */

    TOP_rewind();

#if TRACE
    tpos = 0;
#endif

    opened = 1;
  }

  return 0;
}

int TAPRD(void *buf, int siz, int *trp)
{
  int k;

  if(!opened) {
    k = tapopen();
    if(k) return k;
  }

#if TRACE
  trace_tape("READ",0,0,0);
#endif

  if(T_eom) return 3;	/* VMS: SS$_TAPEPOSLOST */

  TOP_read();

#if TRACE
    tpos ++;
#endif

  if(T_prevlen == T_MARK) {	/* tape mark read */
    *trp = 0;
    return 2;
  } else {
    *trp = T_prevlen;
    memcpy(buf,T_data,T_prevlen);
    return 0;
  }
}


int TAPWR(void *buf,int siz)
{
  int k;

  if(!opened) {
    k = tapopen();
    if(k) return k;
  }

#if TRACE
  trace_tape("WRITE",0,0,0);
#endif

  if(t_ro) return 6;	/* VMS: SS$_WRITLCK */

  TOP_write(siz,buf);

#if TRACE
  tpos ++;
#endif

  return 0;
}


int TAPBSR(void)
{
  int k;

  if(!opened) {
    k = tapopen();
    if(k) return k;
  }

#if TRACE
  trace_tape("BSR",0,0,0);
#endif

  if(T_bot) {
    return 0;	/* say it worked ... */
  } else {
    TOP_bsr();

#if TRACE
  tpos --;
#endif

    return (T_nextlen == T_MARK) ? 2 : 0;
  }
}


int TAPFSR(void)
{
  int k;

  if(!opened) {
    k = tapopen();
    if(k) return k;
  }

#if TRACE
  trace_tape("FSR",0,0,0);
#endif

  if(T_eom) {
    return 3;	/* VMS: SS$_TAPEPOSLOST */
  } else {
    TOP_fsr();

#if TRACE
  tpos ++;
#endif

    return (T_prevlen == T_MARK) ? 2 : 0;
  }
}

int TAPWTM(void)
{
  int k;

  if(!opened) {
    k = tapopen();
    if(k) return k;
  }

#if TRACE
  trace_tape("WRITEMARK",0,0,0);
#endif

  if(t_ro) return 6;	/* VMS: SS$_WRITLCK */

  TOP_write(T_MARK,(unsigned char *)"\032");	/* ^Z intended, formerly ^V */

#if TRACE
  tpos ++;
#endif

  return 0;
}


int TAPREW(void)
{
  int k;

  if(!opened) {
    k = tapopen();
    if(k) return k;
  }

#if TRACE
  trace_tape("REWIND",0,0,0);
#endif

  TOP_rewind();

#if TRACE
  tpos = 0;
#endif

  return 0;
}


int TAPRUN(void)
{
  int k;

  if(!opened) {
    k = tapopen();
    if(k) return k;
  }

#if TRACE
  trace_tape("UNLOAD",0,0,0);
#endif

  TOP_rewind();

#if TRACE
  tpos = 0;
#endif

  k = close(tapfd); opened = 0;
  if(k != 0) {
    perror("close()");
    T_abort();
  }

  return 0;
}


/* debug only? */

int TAPSHOW(void)
{
  int k;

  if(!opened) {
    k = tapopen();
    if(k) return k;
  }
  
  /* no-op so far */
  return 0;
}


#if TRACE
static void trace_tape(char *fname,int ac,int a1,int a2)
{
  fprintf(stdout,"DT_%s(",fname);
  if(ac > 0) {
    fprintf(stdout,"%d",a1);
    if(ac > 1) {
      fprintf(stdout,",%d",a2);
    }
  }
  fprintf(stdout,") tpos=%d\n",tpos);
}
#endif
