#include "stdsys.h"
#include <windows.h>
#include <wchar.h>
#include <wchar.h>

using namespace std;

// From query_server_comment.cpp compiled with UNICODE
wstring query_server_comment(const string& _root);

// use "internal" as opposite of external.
#define internal static

// we will throw these around in case of an error
typedef string error;

// this is the default command executed in the specified path.  The -e
// option may be used to override this.
internal const char* default_command = "cmd.exe";

// passing this value to exec_command as the script prevents a script
// or command from being executed.
internal const char* no_script = "";

// By defualt, this "magic" value is passed to exec_command as the title string.
// This causes it to use the last component of the specified path name as 
// the command window title. The -t option may be used to specify a specific
// title string.
internal const char* default_title = " ";

// Passing this value to exec_command causes it to use the system default 
// window title (this program's name and path).  Use the -nt option to 
// specify this behavior.
internal const char* system_title = "";

// forward declarations of internal functions
internal void exec_command(const string& cmd, const string& ipath, const string& script, const string& ititle);
internal string long_file_name(const string& _i);
internal string query_network_connection(const string& _ln);
internal string query_file_system(const string& _root);
internal bool get_unc_name(const string& _n, string& _r);
internal bool is_unc_name(const string& _n);
internal bool is_network(const string& _n);
internal bool is_samba(const string& _n);
internal bool has_volume(const string& _n);
internal bool is_command(const string& _e);
internal bool is_option(const char& _c);
internal void pause();
internal void expand(string& _x, const string& _h, const string& _d, const string& _u);

//
// help screen
// 

void usage()
   {
      cout << "cmdat [<option>...] <path>\n"
           << "Purpose:\n"
           << "  Open an NT or Unix command window with the specified path as\n"
           << "  the working directory.  NT ONLY: (if <path> is a file, it is\n"
           << "  executed -- if not an executable file type, then a\n"
           << "  'dir <name>' is issued from the specified working direcory.)\n"
           << "Options:\n"
           << "  -e <exefile>   : specify shell (default: cmd.exe)\n"
           << "  -s <script>    : specify optional script to be called when\n"
           << "                   the new shell starts\n"
           << "  -t <text>      : specify window title text (NT only)\n"
           << "  -nt            : use system window title\n"
           << "Environment:       (used only for Samba volumes)\n"
           << "  CMDATNAME      : specifies comment field that identifies a Samba\n"
           << "                   server.\n"
           << "                   default: 'Samba'\n"
           << "  CMDATEXEC      : specify NT program to start a Unix shell\n"
           << "                   default: 'ssh.exe'\n"
           << "  CMDATARGS    * : specify arguments for Unix shell program\n"
           << "                   default: '%h'\n"
           << "  CMDATFILE      : specify filename for optional login script\n"
           << "                   default: none\n"
           << "  CMDATLINE    * : specify command line CMDATFILE\n"
           << "                   default: 'cd %d'\n"
           << "  *              : these items are subject to the following\n"
           << "                   macro expansions:\n"
           << "              %h : replaced with the name of the Unix host\n"
           << "              %d : replaced with the working directory\n"
           << "              %u : replaced with your userid\n"
           << "            NOTE : host name and userid are extracted from the\n"
           << "                   UNC name obtained from the Samba share (e.g.\n"
           << "                   if M: is mapped to \\\\monolith\\dmiller, then\n"
           << "                   %h would expand to 'monolith', and %u would\n"
           << "                   become 'dmiller'.\n"
           << flush;
      throw error("invalid commmand line");
   }

//
// main program
// 

int main(int argc, char** argv, char** envp)
   {
      string message = "done";

      try {

         --argc, ++argv;

         if (!argc) usage();

         string cmd = "cmd.exe";
         string script = no_script;
         string title = default_title;

         while (argc && is_option(**argv)) {
            string option(*argv + 1);
            if (option == "e") {
               --argc, ++argv;
               if (argc && !is_option(**argv)) {
                  cmd = *argv;
                  --argc, ++argv;
               } else {
                  throw error("-e option requires argument");
               }
            } else if (option == "s") {
               --argc, ++argv;
               if (argc && !is_option(**argv)) {
                  script = *argv;
                  --argc, ++argv;
               } else {
                  throw error("-s option requires argument");
               }
            } else if (option == "t") {
               --argc, ++argv;
               if (argc && !is_option(**argv)) {
                  title = *argv;
                  --argc, ++argv;
               } else {
                  throw error("-t option requires argument");
               }
            } else if (option == "nt") {
               --argc, ++argv;
               title = system_title;
            } else {
               throw error("invalid option " + option);
            }
         }

         string full_name(long_file_name((argc < 1) ? "." : *argv));
         exec_command(cmd, full_name, script, title);

         // END END END -- NO RETURN FROM ABOVE FUNCTION!
         // CODE BELOW SHOULD NEVER BE EXECUTED
         throw error("internal error -- unable launch shell");
      }
         
      catch (const error& _e) {
         message = _e;
      }

      if (!message.empty()) {
         cout << message << endl;
         pause();
      }
      return -1;
   }

//
// exec_command()
//
// Execute the specified command in the specified path.  The command is usually
// CMD.EXE; however, using the -e option, another executable may be specified.
//
// The following command line is constructed:
//
// if path is a directory:
//       CMD /k "[TITLE title&&]CD /d path[&&script]"
// or, if path is a file
//       CMD /k "[TITLE title&&]CD /d path[&&script]&&DIR file"  
//
//

internal void exec_command(
   const string& cmd,      // name of command to execute (should be cmd.exe)
   const string& ipath,    // exact name of initial home directory or file
   const string& script,   // optional script (or command) to run
   const string& ititle)   // optional title for command window
   {
      // now that we have the fully expanded long file name, we will
      // determine if it is a network name or a local name.

      string unc_name;
      bool samba = false;
      if (get_unc_name(ipath, unc_name)) {
         samba = is_samba(unc_name);
         if (!samba && is_unc_name(ipath)) {
            throw error("network names not allowed as current directory");
         }
      }

      // if we have a local name,
      // check to see if it is the name of a directory, or a file.

      char drive[ _MAX_DRIVE ];
      char dir[ _MAX_DIR ];
      char fname[ _MAX_FNAME ];
      char ext[ _MAX_EXT ];
      
      _splitpath(ipath.c_str(), drive, dir, fname, ext);

      string path = ipath;
      string file;

      if (*fname) {
         // have something that could be a file name
         WIN32_FIND_DATA f;
         HANDLE h = ::FindFirstFile(ipath.c_str(), &f);
         if (h != INVALID_HANDLE_VALUE) {
            ::FindClose(h);
            if (!(f.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
               // name is not a directory name, assume is a file
               if (*dir) {
                  // chop off trailing backslash on path component
                  for (char*p = dir; *p; ++p);
                  if (p[-1] == '\\') p[-1] = '\0';
               }
               // use drive and directory for current directory
               path = drive;
               path += dir;
               // use filename for a directory command
               file = fname;
               file += ext;
            }
         } else {
            throw error("unexpected find failure on " + ipath);
         }
      }

      string args;
      string cmdf;

      if (!samba) {
         // for local files, we use the passed command (usually cmd.exe)
         // we build args as: /k "[TITLE=<ititle>&&]cd /d <path>[&&<script>][&&dir <file>]"
         cmdf = cmd;
         args = "/k ";
         args += '"';
         if (ititle != system_title) {
            args += "TITLE ";
            if (ititle != default_title) {
               args += ititle;
            } else {
               string::size_type p = path.rfind('\\');
               if (p != string::npos) {
                  args += path.substr(p + 1);
               } else {
                  args += path;
               }
            }
            args += "&&";
         }
         args += "cd /d ";
         args += path;
         if (script.length()) {
            args += "&&";
            args += script;
         }
         if (file.length()) {
            args += "&&";
            if (!is_command(ext)) {
               args += "dir ";
            }
            args += file;
         }
         args += '"';
      } else {
         // for SAMBA files, we use the environment varaible SAMBA to obtain 
         // an executable file name (the default is SSH.EXE).
         char* e = ::getenv("CMDATEXEC");
         cmdf = e ? e : "ssh.exe";
         string::size_type p = unc_name.find('\\', 2);
         string host(unc_name.substr(2, p - 2));
         string user(unc_name.substr(p + 1));
         // strip the drive from the NT path and convert it to a Unix path
         // the conversion assumes that /home/<user>/<path> is the desired
         // Unix working directory (some SAMBA shares may not work this way).
         path = path.substr(2);
         replace(path.begin(), path.end(), '\\', '/');
         path = "/home/" + user + path;
         // determine if any arguments have been specified in the environment
         e = ::getenv("CMDATARGS");
         // if no arguments, default to passing only the host name
         args = e ? e : "%h";
         expand(args, host, path, user);
         // check for a script file name in the environment
         e = ::getenv("CMDATFILE");
         if (e) {
            // now, we must write an executable script (assumed to be called from the Unix
            // login script) that will change the directory once the login is successful.
            string script_name(drive);
            script_name += "\\";
            script_name += e;
            ofstream script(script_name.c_str(), ios::trunc | ios::binary);
            if (script.good()) {
               // check for specific script command in the environment
               // the default is: "cd <path>"
               e = ::getenv("CMDATLINE");
               string line(e ? e : "cd %d");
               expand(line, host, path, user);
               script << line << endl;
               script.close();
            } else {
               throw error("unable to create script file: " + script_name + ", on " + host);
            }
         }
      }

      _execlp(cmdf.c_str(), cmdf.c_str(), args.c_str(), 0);

      ostrstream b;
      b << "unable to spawn " << cmd << ", errno = " << errno
        << '\n\t' << "args = " << args;
      error e(b.str());
      b.freeze(0);
      throw e;
   }

//
// long_file_name()
//
// Attempt to construct the complete long file name from any file name.
//

internal string long_file_name(const string& _i)
   {
      // first, expand all relative path components
      char full_path[ _MAX_PATH + 1 ];
      LPTSTR file_part;
      long l = ::GetFullPathName(_i.c_str(), sizeof(full_path), full_path, &file_part);
      if (l == 0) {
         throw error("can't find " + _i);
      }
      if (l > _MAX_PATH) {
         throw error("full path name too long!");
      }

      // now break the result into separate parts
      char drive[ _MAX_DRIVE ];
      char dir[ _MAX_DIR ];
      char fname[ _MAX_FNAME ];
      char ext[ _MAX_EXT ];

      _splitpath(full_path, drive, dir, fname, ext);

      string n;               // long name to return
      string x;               // path / name components left to resolve
      string y;               // temporary for partial resolutions
      string::size_type p;

      // we start with the directory and file name components
      // NOTE: fname.ext may actually be a directory
      x = dir;
      x += fname;
      x += ext;

      if (is_unc_name(x)) {
         // name is \\server\share\blahblah\yadayada.yuk
         p = x.find('\\',2);
         if (p != string::npos) {
            // now try to find the share name
            int q = x.find('\\', p+1);
            if (q != string::npos) {
               // found the share name, skip "\\server\share"
               n = x.substr(0, q);
               x = x.substr(q+1);
            } else {
               // could not find any reslovable components
               n = x;
               x = "";
            }
         } else {
            // could not find backslash after server name in UNC same
            throw error("invalid directory: " + x + ", unc names not allowed");
         }
      } else {
         // name is ?:\blahblah\yadayada.yuk
         n = drive;
         if (!x.empty() && x[0] == '\\') {
            // skip leading backslash in path
            x = x.substr(1);
         } else {
            throw error("unexpected path format: " + x);
         }
      }

      while (!x.empty()) {
       //cout << x << endl;
         n += '\\';
         y = n;
         p = x.find('\\');
         if (p != string::npos) {
            y += x.substr(0, p);
            x = x.substr(p+1);
         } else {
            y += x;
            x = "";
         }
         // the only way to resolve short name to long names...
         WIN32_FIND_DATA f;
         HANDLE h = ::FindFirstFile(y.c_str(), &f);
         if (h != INVALID_HANDLE_VALUE) {
            ::FindClose(h);
            // append the resolved path component name onto the result
            n += f.cFileName;
         } else {
          //n = y;
            throw error("can't find partial path " + y);
         }
      }

      return n;
   }

//
// query_network_connection()
//
// Call ::WNetGetConnection() and return the results.  This function only
// works for arguments that begin "?:".
//

internal string query_network_connection(const string& _ln)
   {
      string results;
      if (has_volume(_ln)) {
         DWORD length = MAX_PATH;
         results.resize(length);
         if (::WNetGetConnection(_ln.substr(0,2).c_str(), results.begin(), &length) == NO_ERROR) {
            results.resize(strlen(results.c_str()));
         } else {
            results.resize(0);
         }
      }
      return results;
   }

//
// query_file_system()
//
// Call ::GetVolumeInformation() and return the file system string.
//

internal string query_file_system(const string& _fn)
   {
      string root(has_volume(_fn) ? _fn.substr(0,2) : _fn);
      if (!root.empty() && (root[root.length() - 1] != '\\')) root += '\\';
      DWORD mfnl;
      DWORD flags;
      char results[ MAX_PATH ];
      if (::GetVolumeInformation(root.c_str(), NULL, 0, NULL, &mfnl, &flags, results, sizeof(results))) {
         return results;
      }
      return "";
   }

//
// is_command()
//
// test extension to see if it is a command file type.
//

internal bool is_command(const string& _e)
   {
      string ext(_e);
      transform(ext.begin(), ext.end(), ext.begin(), tolower);
      return ((ext == ".exe") || (ext == ".com") || (ext == ".bat") || (ext == ".cmd"));
   }

//
// has_volume()
//
// check for ':' as second character in filename
//

internal bool has_volume(const string& _n)
   {
      return (_n.length() > 1) && (_n[1] == ':');
   }

//
// is_network()
//
// check for the samba file system
//

internal bool is_network(const string& _n)
   {
      string dummy;
      return get_unc_name(_n, dummy);
   }

//
// is_samba()
//
// check for the samba file system
//

internal bool is_samba(const string& _n)
   {
      const wchar_t* s = ::_wgetenv(L"CMDATNAME");
      if (!s) s = L"Samba";
      wint_t l = ::wcslen(s);
      return query_server_comment(_n).compare(0, l, s) == 0;
   }

//
// is_option()
//
// true if specified character is option prefix.
//

internal bool is_option(const char& _c)
   {
      return (_c == '/') || (_c == '-');
   }

//
// get_unc_name(const string& _n, string& _r)
//
// attempt to obtain the UNC name for a given path
//

internal bool get_unc_name(const string& _n, string& _r)
   {
      _r = query_network_connection(_n);
      return !_r.empty();
   }

//
// is_unc_name()
//
// check for double backslash at the start of name.
//

internal bool is_unc_name(const string& _n)
   {
      return (_n.length() > 1) && (_n.substr(0,2) == "\\\\");
   }

//
// pause()
//
// wait for enter key at console.
//

internal void pause()
   {
      cerr << "Press ENTER to continue..." << flush;
      cin.get();
   }

//
// expand
//
// replace "%" variables.
//

internal void expand(string& _x, const string& _h, const string& _d, const string& _u)
   {
      for (string::size_type p = 0; ((p = _x.find('%', p)) != string::npos) && (_x.length() > p); ++p) {
         switch (_x[p + 1]) {
         case 'h':            // %h expands to the host name
            _x.replace(p, 2, _h);
            break;
         case 'd':            // %d expands to the working directory
            _x.replace(p, 2, _d);
            break;
         case 'u':            // %u expands to the user id (should be home directory name)
            _x.replace(p, 2, _u);
            break;
         default:             // skip invalid expansion codes
            break;
         }
      }

   }
