/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file ******************************************************************
 *
 * \brief Main file of the USB DFU ISP.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a USB module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ***************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */

/*! \mainpage
 * \section intro Introduction
 * This is the documentation for the data structures, functions, variables,
 * defines, enums, and typedefs for the USB DFU bootloader.\n 
 * Refer to http://www.atmel.com/dyn/resources/prod_documents/doc7745.pdf.
 *  
 * \section files Main Files
 * - isp.c: the main file of the USB DFU ISP;
 * - usb_dfu.c: Management of the USB device firmware upgrade;
 *
 * \section compilinfo Compilation Information
 * This software is written for GNU GCC for AVR32 and for IAR Embedded Workbench
 * for Atmel AVR32. Other compilers may or may not work.
 *
 * \section deviceinfo Device Information
 * All AVR32 devices with a USBB module.
 *
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.atmel.com/products/AVR32/">Atmel AVR32</A>.\n
 * Support and FAQ: http://support.atmel.no/
 */


//_____  I N C L U D E S ___________________________________________________

#include "preprocessor.h"
#include "compiler.h"
#include "board.h"
#include "pm.h"
#include "rtc.h"
#include "cycle_counter.h"
#include "flashc.h"
#include "conf_usb.h"
#include "usb_drv.h"
#include "usb_task.h"
#if USB_DEVICE_FEATURE == ENABLED
#include "usb_dfu.h"
#endif
#include "conf_isp.h"
#include "isp.h"
#include "autobaud.h"


//_____ M A C R O S ________________________________________________________


//_____ D E F I N I T I O N S ______________________________________________

#if (defined __GNUC__) && ((defined __AVR32_UC3A364__)   ||                    \
                           (defined __AVR32_UC3A364S__)  ||                    \
                           (defined __AVR32_UC3A3128__)  ||                    \
                           (defined __AVR32_UC3A3128S__) ||                    \
                           (defined __AVR32_UC3A3256__)  ||                    \
                           (defined __AVR32_UC3A3256S__))||                    \
(defined __ICCAVR32__) && ((defined __AT32UC3A364__)     ||                    \
                           (defined __AT32UC3A364S__)    ||                    \
                           (defined __AT32UC3A3128__)    ||                    \
                           (defined __AT32UC3A3128S__)   ||                    \
                           (defined __AT32UC3A3256__)    ||                    \
                           (defined __AT32UC3A3256S__))
#define TC              (&AVR32_TC0)
#else
  #define TC              (&AVR32_TC)
#endif
#define PM                (&AVR32_PM)
#define TC_32KHZ_CHANNEL  0

void wait_10_ms(void)
{
  Set_system_register(AVR32_COUNT, 0);
  while ((U32)Get_system_register(AVR32_COUNT) < (FRCOSC * 10 + 999) / 1000);
}

static void osc_rtc_counter_reset(void)
{
  rtc_set_value(&AVR32_RTC, 0);
}

static int osc_rtc_counter_value(void)
{
  return rtc_get_value(&AVR32_RTC) * 2;
}

static void osc_isp_counter_reset(void)
{
  Set_sys_count(0);
}

static int osc_isp_counter_value(void)
{
  return Get_sys_count();
}

/*!
 *  Start the generation of system clocks with USB autobaud
 */
void sys_clk_gen_start(void)
{
  int cpu_freq_hz, mul;
  const int freq_hz[] = { 8000000, 12000000, 16000000 };
  const struct autobaud_cfg_osc_cmp_t autobaud_cfg = {
    .osc_ref_freq_hz = 115000,
    .osc_ref_counter_reset = osc_rtc_counter_reset,
    .osc_ref_counter_value = osc_rtc_counter_value,
    .osc_target_counter_reset = osc_isp_counter_reset,
    .osc_target_counter_value = osc_isp_counter_value,
    .convergence_rate = 10000
  };
  const struct autobaud_matching_freq_t match_freq = {
    .freq_hz = freq_hz,
    .nb_entries = sizeof(freq_hz)/sizeof(freq_hz[0])
  };
  Bool sav_glob_int_en;

  #define MAX_OSC_FREQ    16000000
  

  // Switch to OSC ISP
  // Set max startup time to make sure any crystal will be supported
  // We cannot use a TC to measure this OSC frequency because the master clock must be faster than the clock selected by the TC
  pm_switch_to_osc0(&AVR32_PM, MAX_OSC_FREQ, ATPASTE3(AVR32_PM_OSCCTRL, ISP_OSC, _STARTUP_16384_RCOSC));
  // Intialize the RTC with the internal RC oscillator
  // RTC will count at the frequency of 115KHz/2
  rtc_init(&AVR32_RTC, RTC_OSC_RC, 0);
  rtc_enable(&AVR32_RTC);

  // Calculate the frequency
  if (!(cpu_freq_hz = autobaud_detect_osc_cmp(&autobaud_cfg)))
    cpu_freq_hz = 12000000;
  cpu_freq_hz = autobaud_match_frequency(cpu_freq_hz, &match_freq);

  switch (cpu_freq_hz)
  {
  case 8000000:
    mul = 5;
    break;
  case 12000000:
    mul = 3;
    break;
  case 16000000:
    mul = 2;
    break;
  default:
    mul = 3;
  }

  Usb_freeze_clock();

  // Set PLL0 VCO @ 96 MHz
  pm_pll_setup(PM, 0,                         // pll
                   mul,                       // mul
                   0,                         // div
                   ISP_OSC,                   // osc
                   63);                       // lockcount

  if ((sav_glob_int_en = Is_global_interrupt_enabled())) Disable_global_interrupt();

  // Set PLL0 @ 48 MHz
  pm_pll_set_option(PM, 0,  // pll
                    1,      // pll_freq
                    1,      // pll_div2
                    0);     // pll_wbwdisable

  // Enable PLL0
  pm_pll_enable(PM, 0);

  // Wait for PLL0 locked with a 10-ms time-out
  pm_wait_for_pll0_locked(PM);

  // Setup USB GCLK
  pm_gc_setup(PM, AVR32_PM_GCLK_USBB, // gc
                      1,              // osc_or_pll: use Osc (if 0) or PLL (if 1)
                      0,              // pll_osc: select Osc0/PLL0 or Osc1/PLL1
#if (defined __GNUC__) && ((defined __AVR32_UC3A364__)   ||                    \
                           (defined __AVR32_UC3A364S__)  ||                    \
                           (defined __AVR32_UC3A3128__)  ||                    \
                           (defined __AVR32_UC3A3128S__) ||                    \
                           (defined __AVR32_UC3A3256__)  ||                    \
                           (defined __AVR32_UC3A3256S__))||                    \
(defined __ICCAVR32__) && ((defined __AT32UC3A364__)     ||                    \
                           (defined __AT32UC3A364S__)    ||                    \
                           (defined __AT32UC3A3128__)    ||                    \
                           (defined __AT32UC3A3128S__)   ||                    \
                           (defined __AT32UC3A3256__)    ||                    \
                           (defined __AT32UC3A3256S__))
                      1,                  // diven
                      1);                 // div
#else
                      0,                  // diven
                      0);                 // div
#endif

  // Enable USB GCLK
  pm_gc_enable(PM, AVR32_PM_GCLK_USBB);

  Usb_unfreeze_clock();

  // Use 1 flash wait state
  flashc_set_wait_state(1);

  // Switch the main clock to PLL0
  pm_switch_to_clock(PM, AVR32_PM_MCCTRL_MCSEL_PLL0);

  // fPBA: 12 MHz
  // fPBB: 12 MHz
  // fHSB: 12 MHz
  pm_cksel(PM, 1,   // pbadiv
               1,   // pbasel
               1,   // pbbdiv
               1,   // pbbsel
               1,   // hsbdiv
               1);  // hsbsel

  // Use 0 flash wait state
  flashc_set_wait_state(0);

  Usb_ack_sof();

  if (sav_glob_int_en) Enable_global_interrupt();
}


/*!
 *  Stop the generation of system clocks and switch to RCOsc
 */
void sys_clk_gen_stop(void)
{
  volatile avr32_pm_t *const pm = &AVR32_PM;

  pm_gc_disable(pm, AVR32_PM_GCLK_USBB);
  pm_gc_setup(pm, AVR32_PM_GCLK_USBB, 0, 0, 0, 0);
  flashc_set_wait_state(1);
  pm_cksel(pm, 0, 0, 0, 0, 0, 0);
  pm_switch_to_clock(pm, AVR32_PM_MCCTRL_MCSEL_SLOW);
  flashc_set_wait_state(0);
  pm_pll_disable(pm, 0);
  pm_pll_set_option(pm, 0, 0, 0, 0);
  pm_pll_setup(pm, 0, 0, 0, 0, 0);
  pm_enable_clk_no_wait(pm, ATPASTE3(AVR32_PM_OSCCTRL, ISP_OSC, _STARTUP_0_RCOSC));
  pm_disable_clk(pm);
  pm_enable_osc_ext_clock(pm);
}


int main(void)
{
  wait_10_ms();

  Usb_force_full_speed_mode();
  usb_task_init();
#if USB_DEVICE_FEATURE == ENABLED
  usb_dfu_init();
#endif

  while (TRUE)
  {
    usb_task();
  }
}
