/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file ******************************************************************
 *
 * \brief Management of the USB device HID task.
 *
 * This file manages the USB device HID task.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a USB module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ***************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. ATMEL grants developer a non-exclusive, limited license to use the Software
 * as a development platform solely in connection with an Atmel AVR product
 * ("Atmel Product").
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 */

// Put following MACRO in comment if you need debug info in this module
//#define UART_voUart0PrintString(...) /* NO DEBUG INFO ON UART */

//_____  I N C L U D E S ___________________________________________________

#include <stdio.h>
#include "usart.h"     // Shall be included before FreeRTOS header files, since 'inline' is defined to ''; leading to
// link errors
#include "conf_usb.h"

#if USB_DEVICE_FEATURE == ENABLED

#include "board.h"
#include "controller.h"
#include "usb_drv.h"
#include "usb_descriptors.h"
#include "usb_standard_request.h"
#include "device_hid_task.h"
#include "audio_example.h"
#include "cycle_counter.h"
#include "debug.h"
#include "hid.h"
#include "et024006dhu.h"

//_____ M A C R O S ________________________________________________________


//_____ D E F I N I T I O N S ______________________________________________

U8 key = 0;

#define TIMER_STARTUP    100
t_cpu_time key_timer;


#define HID_VOL_DN       2
#define HID_VOL_UP       1
#define HID_PLAY_PAUSE   4
#define HID_NEXT         8
#define HID_PREV        16
#define HID_FAST_FWD    32
#define HID_REWIND      64


//_____ D E C L A R A T I O N S ____________________________________________


//!
//! @brief Looks for HID events.
//!
Bool is_usb_hid_event(void) {
  static Bool b_on_off = FALSE;
  static Bool b_prev   = FALSE;
  static Bool b_next   = FALSE;
  static Bool b_vol_up = FALSE;
  static Bool b_vol_dn = FALSE;

  // Management of Play/Pause
  //
  if(is_joystick_pressed() && b_on_off==FALSE)
  {
    b_on_off = TRUE;
    key = HID_PLAY_PAUSE;
    et024006_PrintString("PLAY", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }
  if(!is_joystick_pressed() && b_on_off==TRUE)
  {
    b_on_off = FALSE;
    key = 0;
    et024006_PrintString("    ", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }

  // Management of Prev
  //
  if (is_joystick_left() && b_prev == FALSE)
  {
    b_prev = TRUE;
    key = HID_PREV;

    et024006_PrintString("PREV", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }
  if (!is_joystick_left() && b_prev == TRUE)
  {
    b_prev = FALSE;
    key = 0;

    et024006_PrintString("    ", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }

  // Management of Next
  //
  if (is_joystick_right() && b_next == FALSE)
  {
    b_next = TRUE;
    key = HID_NEXT;

    et024006_PrintString("NEXT", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }
  if (!is_joystick_right() && b_next == TRUE)
  {
    b_next = FALSE;
    key = 0;

    et024006_PrintString("    ", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }


  // Management of Vol -
  //
  if (is_joystick_down() && b_vol_dn == FALSE)
  {
    key = HID_VOL_DN;
    b_vol_dn = TRUE;

    et024006_PrintString("VOL-", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }
  if (!is_joystick_down() && b_vol_dn == TRUE)
  {
    b_vol_dn = FALSE;
    key = 0;

    et024006_PrintString("    ", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }

  // Management of Vol +
  //
  if (is_joystick_up() && b_vol_up == FALSE)
  {
    key = HID_VOL_UP;
    b_vol_up = TRUE;

    et024006_PrintString("VOL+", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }
  if (!is_joystick_up() && b_vol_up == TRUE)
  {
    b_vol_up = FALSE;
    key = 0;

    et024006_PrintString("    ", (const unsigned char *)&FONT8x8, 153, 70+4, BLUE, WHITE);
    return TRUE;
  }

  return FALSE;
}

//!
//! @brief This function initializes the hardware/software resources
//! required for device HID task.
//!
void device_hid_task_init(void)
{
  cpu_set_timeout( cpu_ms_2_cy(TIMER_STARTUP, FCPU_HZ), &key_timer );

#if USB_HOST_FEATURE == ENABLED
  // If both device and host features are enabled, check if device mode is engaged
  // (accessing the USB registers of a non-engaged mode, even with load operations,
  // may corrupt USB FIFO data).
  if (Is_usb_device())
#endif  // USB_HOST_FEATURE == ENABLED
  Usb_enable_sof_interrupt();
}

  //t_cpu_time timer;

  //!
  //! @brief Entry point of the device HID task management
  //!
void device_hid_task(void)
{
  // First, check the device enumeration state
  if (!Is_device_enumerated()) {
    return;
  }

  if (Is_usb_in_ready(EP_KBD_IN)) {
    if (is_usb_hid_event()) {
      Usb_reset_endpoint_fifo_access(EP_KBD_IN);

      //! Write report
      Usb_write_endpoint_data(EP_KBD_IN, 8, key);
      Usb_ack_in_ready_send(EP_KBD_IN);
    }
  }
}

#endif  // USB_DEVICE_FEATURE == ENABLED
