/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file ******************************************************************
 *
 * \brief Memory manager example.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ***************************************************************************/

/*! \mainpage AVR32 Audio decoder over USB
 *
 * <HR>
 * \section license License
 * Use of this program is subject to Atmel's End User License Agreement.
 *
 * Please read the \ref license at the bottom of this page.
 *
 * \section intro Introduction
 * Documentation of the "dlmalloc" memory manager application.
 *
 * \section files Main Files
 * - main.c : Example application that uses only the MSPACE part of the  
 *   "dlmalloc" memory manager. The normal malloc is used from the Newlib
 *   C-library
 *  
 * \section compinfo Compilation Info
 * This software was written for the GNU GCC for AVR32 and IAR Systems compiler
 * for AVR32. Other compilers may or may not work.
 *
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.atmel.com/products/AVR32/">Atmel AVR32</A>.\n
 * Support and FAQ: http://support.atmel.no/
 *
 * \section license Copyright notice
 * Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */


#include <avr32/io.h>
#include "compiler.h"
#include "board.h"
#include "pm.h"
#include "flashc.h"
#include "gpio.h"
#include "sdramc.h"
#include "dlmalloc.h"


#if __GNUC__ && __AVR32__
  #include "nlao_cpu.h"
  #include "nlao_usart.h"
#elif __ICCAVR32__
  #include "usart.h"
#endif

#define FHSB_HZ 66000000
#define FPBA_HZ 66000000
#define SDRAM_START_ADDRESS 0xD0000000
#define MEM_SPACE_SIZE 0x2000000

/*! \brief Initializes MCU exceptions.
 */
static void init_exceptions(void)
{
#if __GNUC__ && __AVR32__
  // Import the Exception Vector Base Address.
  extern void _evba;

  // Load the Exception Vector Base Address in the corresponding system
  // register.
  Set_system_register(AVR32_EVBA, (int)&_evba);
#endif

  // Enable exceptions globally.
  Enable_global_exception();
}


/*! \brief Initializes the HSB bus matrix.
 */
static void init_hmatrix(void)
{
  // For the internal-flash HMATRIX slave, use last master as default.
  union
  {
    unsigned long                 scfg;
    avr32_hmatrix_scfg_t          SCFG;
  } u_avr32_hmatrix_scfg = {AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_FLASH]};
  u_avr32_hmatrix_scfg.SCFG.defmstr_type = AVR32_HMATRIX_DEFMSTR_TYPE_LAST_DEFAULT;
  AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_FLASH] = u_avr32_hmatrix_scfg.scfg;
}


/*! \brief Initializes the MCU system clocks.
 */
static void init_sys_clocks(void)
{
    // Switch to OSC0 to speed up the booting
    pm_switch_to_osc0(&AVR32_PM, FOSC0, OSC0_STARTUP);

    // Start oscillator1 
    pm_enable_osc1_crystal(&AVR32_PM, FOSC1);
    //
    pm_enable_clk1(&AVR32_PM, OSC1_STARTUP);

    // Set PLL0 (fed from OSC0 = 12 MHz) to 132 MHz
    pm_pll_setup(&AVR32_PM, 0,  // pll.
	    10,  // mul.
	    1,   // div.
	    0,   // osc.
	    16); // lockcount.

    // Set PLL operating range and divider (fpll = fvco/2)
    // -> PLL0 output = 66 MHz
    pm_pll_set_option(&AVR32_PM, 0, // pll.
	    1,  // pll_freq.
	    1,  // pll_div2.
	    0); // pll_wbwdisable.

    // start PLL0 and wait for the lock
    pm_pll_enable(&AVR32_PM, 0);
    pm_wait_for_pll0_locked(&AVR32_PM);
    // Set all peripheral clocks torun at master clock rate
    pm_cksel(&AVR32_PM,
	    0,   // pbadiv.
	    0,   // pbasel.
	    0,   // pbbdiv.
	    0,   // pbbsel.
	    0,   // hsbdiv.
	    0);  // hsbsel.

    // Set one waitstate for the flash
    flashc_set_wait_state(1);

    // Switch to PLL0 as the master clock
    pm_switch_to_clock(&AVR32_PM, AVR32_PM_MCCTRL_MCSEL_PLL0);
    
  
#if __GNUC__ && __AVR32__
  // Give the used PBA clock frequency to Newlib, so it can work properly.
  set_cpu_hz(FPBA_HZ);
#endif
}


/*! \brief Initializes the heap.
 */
static void init_heap(void)
{
#ifdef _DEBUG
  #if __GNUC__ && __AVR32__

  // Import the location of the heap.
  extern void __heap_start__;
  extern void __heap_end__;

  U32 *p_heap_word;

  // Initialize the heap.
  for (p_heap_word = &__heap_start__; (void *)p_heap_word < &__heap_end__; p_heap_word++)
    *p_heap_word = DEFAULT_HEAP_INIT_WORD;

  #elif __ICCAVR32__

  // Import the location of the heap.
  #pragma segment = "HEAP"

  U32 *p_heap_word;

  // Initialize the heap.
  for (p_heap_word = __segment_begin("HEAP"); (void *)p_heap_word < __segment_end("HEAP"); p_heap_word++)
    *p_heap_word = DEFAULT_HEAP_INIT_WORD;

  #endif
#endif
}


/*! \brief Initializes STDIO.
 */
#if 0
static void init_stdio(void)
{
#if __GNUC__ && __AVR32__

  static const gpio_map_t STDIO_USART_GPIO_MAP =
  {
    {STDIO_USART_RX_PIN, STDIO_USART_RX_FUNCTION},
    {STDIO_USART_TX_PIN, STDIO_USART_TX_FUNCTION}
  };

  // Initialize the USART used for STDIO.
  set_usart_base((void *)STDIO_USART);
  gpio_enable_module(STDIO_USART_GPIO_MAP,
                     sizeof(STDIO_USART_GPIO_MAP) / sizeof(STDIO_USART_GPIO_MAP[0]));
  usart_init(STDIO_USART_BAUDRATE);

#elif __ICCAVR32__

  static const gpio_map_t STDIO_USART_GPIO_MAP =
  {
    {STDIO_USART_RX_PIN, STDIO_USART_RX_FUNCTION},
    {STDIO_USART_TX_PIN, STDIO_USART_TX_FUNCTION}
  };

  static const usart_options_t STDIO_USART_OPTIONS =
  {
    .baudrate     = STDIO_USART_BAUDRATE,
    .charlength   = 8,
    .paritytype   = USART_NO_PARITY,
    .stopbits     = USART_1_STOPBIT,
    .channelmode  = USART_NORMAL_CHMODE
  };

  // Initialize the USART used for STDIO.
  extern volatile avr32_usart_t *volatile stdio_usart_base;
  stdio_usart_base = STDIO_USART;
  gpio_enable_module(STDIO_USART_GPIO_MAP,
                     sizeof(STDIO_USART_GPIO_MAP) / sizeof(STDIO_USART_GPIO_MAP[0]));
  usart_init_rs232(STDIO_USART, &STDIO_USART_OPTIONS, FPBA_HZ);

#endif
}
#endif

/*! \brief Low-level initialization routine called during startup, before the
 *         \ref main function.
 */
#if __GNUC__ && __AVR32__
int _init_startup(void)
#elif __ICCAVR32__
int __low_level_init(void)
#endif
{
  init_exceptions();
  init_hmatrix();
  init_sys_clocks();
  init_heap();
  //init_stdio();

  // EWAVR32: Request initialization of data segments.
  // GCC: Don't-care value.
  return 1;
}

/*! \brief Main function of the example memory manager.
 * 
 *  This example shows how memory can be allocated from different
 *  memory spaces. 
 *  The default allocation will get memory from the internal SRAM.
 *  By using the "memory space" functionality of the memory manager
 *  it is possible to use other memory spaces as resources like an
 *  attached SDRAM.
 */
int main(void)
{
	void *some_space;
	void *some_more_space;
	void *some_space_in_sdram;
	mspace sdram_msp;
	
	sdramc_init(FHSB_HZ);
	
	// default allocation with C-library malloc from internal SRAM
  	some_space = malloc(512);
	some_more_space = malloc(64);
	
	// Create a new memory space the covers the SDRAM
	sdram_msp = create_mspace_with_base((void*) SDRAM_START_ADDRESS, MEM_SPACE_SIZE, 0);
	
	// allocate memory from the created memroy space 
	some_space_in_sdram = mspace_malloc(sdram_msp, 512);

	while (TRUE)
	{
	}
}
