/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief 32-bit twiddle factors table
 *
 * This file define the twiddle factors table
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
#include "trans_dsp32_twiddle_factors.h"

//! This macro is used to convert the raw data acording to the algorithm's optimization we need
#if DSP_OPTIMIZATION & DSP_OPTI_ACCURACY
#  define DSP32_Q_CONVERT(x)  DSP32_Q(x)
#else
#  define DSP32_Q_CONVERT(x)  DSP32_Q((x)/2.)
#endif

/*
 * Twiddle factors for a 64-point FFT
 * Minimized twiddle factors table: 2*(64/4)*2 = 64 bytes
 * Full twiddle factors table: 2*(64/4)*6 = 192 bytes
 */
#if DSP32_N_TWIDDLE_FACTORS == 64

A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp32_t dsp32_twiddle_factors[DSP32_N_TWIDDLE_FACTORS/2+2] = {
  /*Re(w)   Im(w)  */
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(0.00000000000), DSP32_Q_CONVERT(-1.00000000000)
  };

  // If we need to optimize the code to use small twiddle factors table
#  if !(DSP_OPTIMIZATION & DSP_OPTI_SIZE)
    A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp32_t dsp32_twiddle_factors2[DSP32_N_TWIDDLE_FACTORS] = {
    /*Re(w2)   Im(w2)   Re(w3)   Im(w3)  */
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(-0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(-0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(0.00000000000), DSP32_Q_CONVERT(-1.00000000000),
    DSP32_Q_CONVERT(-0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(-0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(-0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(-0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(-0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(-0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(-0.99518472667), DSP32_Q_CONVERT(0.09801714033),
    DSP32_Q_CONVERT(-0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(-0.92387953251), DSP32_Q_CONVERT(0.38268343237),
    DSP32_Q_CONVERT(-0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(-0.77301045336), DSP32_Q_CONVERT(0.63439328416),
    DSP32_Q_CONVERT(-0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(-0.55557023302), DSP32_Q_CONVERT(0.83146961230),
    DSP32_Q_CONVERT(-0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(-0.29028467725), DSP32_Q_CONVERT(0.95694033573)
    };
#  endif

#endif

/*
 * Twiddle factors for a 256-point FFT
 * Minimized twiddle factors table: 2*(256/4)*2 = 256 bytes
 * Full twiddle factors table: 2*(256/4)*6 = 768 bytes
 */
#if DSP32_N_TWIDDLE_FACTORS == 256

  A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp32_t dsp32_twiddle_factors[DSP32_N_TWIDDLE_FACTORS/2+2] = {
  /*Re(w)   Im(w)  */
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(0.99969881870), DSP32_Q_CONVERT(-0.02454122852),
    DSP32_Q_CONVERT(0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.98527764239), DSP32_Q_CONVERT(-0.17096188876),
    DSP32_Q_CONVERT(0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.94952818059), DSP32_Q_CONVERT(-0.31368174040),
    DSP32_Q_CONVERT(0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.89322430120), DSP32_Q_CONVERT(-0.44961132965),
    DSP32_Q_CONVERT(0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.81758481315), DSP32_Q_CONVERT(-0.57580819142),
    DSP32_Q_CONVERT(0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.72424708295), DSP32_Q_CONVERT(-0.68954054474),
    DSP32_Q_CONVERT(0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.61523159058), DSP32_Q_CONVERT(-0.78834642763),
    DSP32_Q_CONVERT(0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.49289819223), DSP32_Q_CONVERT(-0.87008699111),
    DSP32_Q_CONVERT(0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(0.35989503653), DSP32_Q_CONVERT(-0.93299279883),
    DSP32_Q_CONVERT(0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(0.21910124016), DSP32_Q_CONVERT(-0.97570213004),
    DSP32_Q_CONVERT(0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(0.07356456360), DSP32_Q_CONVERT(-0.99729045668),
    DSP32_Q_CONVERT(0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(0.00000000000), DSP32_Q_CONVERT(-1.00000000000)
  };

  // If we need to optimize the code to use small twiddle factors table
#  if !(DSP_OPTIMIZATION & DSP_OPTI_SIZE)
    A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp32_t dsp32_twiddle_factors2[DSP32_N_TWIDDLE_FACTORS] = {
    /*Re(w2)   Im(w2)   Re(w3)   Im(w3)  */
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(-0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(-0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(-0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(-0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(-0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(-0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(-0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(-0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(-0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(-0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(0.00000000000), DSP32_Q_CONVERT(-1.00000000000),
    DSP32_Q_CONVERT(-0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(-0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(-0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(-0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(-0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(-0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(-0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(-0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(-0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(-0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(-0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(-0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(-0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(-0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(-0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(-0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(-0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(-0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(-0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(-0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(-0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(-0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(-0.99969881870), DSP32_Q_CONVERT(0.02454122852),
    DSP32_Q_CONVERT(-0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(-0.99518472667), DSP32_Q_CONVERT(0.09801714033),
    DSP32_Q_CONVERT(-0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(-0.98527764239), DSP32_Q_CONVERT(0.17096188876),
    DSP32_Q_CONVERT(-0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(-0.97003125319), DSP32_Q_CONVERT(0.24298017990),
    DSP32_Q_CONVERT(-0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(-0.94952818059), DSP32_Q_CONVERT(0.31368174040),
    DSP32_Q_CONVERT(-0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(-0.92387953251), DSP32_Q_CONVERT(0.38268343237),
    DSP32_Q_CONVERT(-0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(-0.89322430120), DSP32_Q_CONVERT(0.44961132965),
    DSP32_Q_CONVERT(-0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(-0.85772861000), DSP32_Q_CONVERT(0.51410274419),
    DSP32_Q_CONVERT(-0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(-0.81758481315), DSP32_Q_CONVERT(0.57580819142),
    DSP32_Q_CONVERT(-0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(-0.77301045336), DSP32_Q_CONVERT(0.63439328416),
    DSP32_Q_CONVERT(-0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(-0.72424708295), DSP32_Q_CONVERT(0.68954054474),
    DSP32_Q_CONVERT(-0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(-0.67155895485), DSP32_Q_CONVERT(0.74095112535),
    DSP32_Q_CONVERT(-0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(-0.61523159058), DSP32_Q_CONVERT(0.78834642763),
    DSP32_Q_CONVERT(-0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(-0.55557023302), DSP32_Q_CONVERT(0.83146961230),
    DSP32_Q_CONVERT(-0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(-0.49289819223), DSP32_Q_CONVERT(0.87008699111),
    DSP32_Q_CONVERT(-0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(-0.42755509343), DSP32_Q_CONVERT(0.90398929312),
    DSP32_Q_CONVERT(-0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(-0.35989503653), DSP32_Q_CONVERT(0.93299279883),
    DSP32_Q_CONVERT(-0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(-0.29028467725), DSP32_Q_CONVERT(0.95694033573),
    DSP32_Q_CONVERT(-0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(-0.21910124016), DSP32_Q_CONVERT(0.97570213004),
    DSP32_Q_CONVERT(-0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(-0.14673047446), DSP32_Q_CONVERT(0.98917650996),
    DSP32_Q_CONVERT(-0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(-0.07356456360), DSP32_Q_CONVERT(0.99729045668)
    };
#  endif

#endif

/*
 * Twiddle factors for a 1024-point FFT
 * Minimized twiddle factors table: 2*(1024/4)*2 = 1024 bytes
 * Full twiddle factors table: 2*(1024/4)*6 = 3072 bytes
 */
#if DSP32_N_TWIDDLE_FACTORS == 1024

  A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp32_t dsp32_twiddle_factors[DSP32_N_TWIDDLE_FACTORS/2+2] = {
  /*Re(w)   Im(w)  */
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(0.99998117528), DSP32_Q_CONVERT(-0.00613588465),
    DSP32_Q_CONVERT(0.99992470184), DSP32_Q_CONVERT(-0.01227153829),
    DSP32_Q_CONVERT(0.99983058180), DSP32_Q_CONVERT(-0.01840672991),
    DSP32_Q_CONVERT(0.99969881870), DSP32_Q_CONVERT(-0.02454122852),
    DSP32_Q_CONVERT(0.99952941750), DSP32_Q_CONVERT(-0.03067480318),
    DSP32_Q_CONVERT(0.99932238459), DSP32_Q_CONVERT(-0.03680722294),
    DSP32_Q_CONVERT(0.99907772775), DSP32_Q_CONVERT(-0.04293825693),
    DSP32_Q_CONVERT(0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(0.99847558057), DSP32_Q_CONVERT(-0.05519524435),
    DSP32_Q_CONVERT(0.99811811290), DSP32_Q_CONVERT(-0.06132073630),
    DSP32_Q_CONVERT(0.99772306664), DSP32_Q_CONVERT(-0.06744391956),
    DSP32_Q_CONVERT(0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(0.99682029929), DSP32_Q_CONVERT(-0.07968243797),
    DSP32_Q_CONVERT(0.99631261218), DSP32_Q_CONVERT(-0.08579731234),
    DSP32_Q_CONVERT(0.99576741447), DSP32_Q_CONVERT(-0.09190895650),
    DSP32_Q_CONVERT(0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(0.99456457073), DSP32_Q_CONVERT(-0.10412163387),
    DSP32_Q_CONVERT(0.99390697000), DSP32_Q_CONVERT(-0.11022220729),
    DSP32_Q_CONVERT(0.99321194923), DSP32_Q_CONVERT(-0.11631863091),
    DSP32_Q_CONVERT(0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(0.99170975367), DSP32_Q_CONVERT(-0.12849811079),
    DSP32_Q_CONVERT(0.99090263543), DSP32_Q_CONVERT(-0.13458070851),
    DSP32_Q_CONVERT(0.99005821026), DSP32_Q_CONVERT(-0.14065823933),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.98825756773), DSP32_Q_CONVERT(-0.15279718526),
    DSP32_Q_CONVERT(0.98730141816), DSP32_Q_CONVERT(-0.15885814333),
    DSP32_Q_CONVERT(0.98630809724), DSP32_Q_CONVERT(-0.16491312049),
    DSP32_Q_CONVERT(0.98527764239), DSP32_Q_CONVERT(-0.17096188876),
    DSP32_Q_CONVERT(0.98421009239), DSP32_Q_CONVERT(-0.17700422041),
    DSP32_Q_CONVERT(0.98310548743), DSP32_Q_CONVERT(-0.18303988796),
    DSP32_Q_CONVERT(0.98196386911), DSP32_Q_CONVERT(-0.18906866415),
    DSP32_Q_CONVERT(0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(0.97956976569), DSP32_Q_CONVERT(-0.20110463484),
    DSP32_Q_CONVERT(0.97831737072), DSP32_Q_CONVERT(-0.20711137619),
    DSP32_Q_CONVERT(0.97702814266), DSP32_Q_CONVERT(-0.21311031992),
    DSP32_Q_CONVERT(0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(0.97433938279), DSP32_Q_CONVERT(-0.22508391136),
    DSP32_Q_CONVERT(0.97293995221), DSP32_Q_CONVERT(-0.23105810828),
    DSP32_Q_CONVERT(0.97150389099), DSP32_Q_CONVERT(-0.23702360599),
    DSP32_Q_CONVERT(0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(0.96852209427), DSP32_Q_CONVERT(-0.24892760575),
    DSP32_Q_CONVERT(0.96697647104), DSP32_Q_CONVERT(-0.25486565960),
    DSP32_Q_CONVERT(0.96539444170), DSP32_Q_CONVERT(-0.26079411792),
    DSP32_Q_CONVERT(0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(0.96212140427), DSP32_Q_CONVERT(-0.27262135545),
    DSP32_Q_CONVERT(0.96043051942), DSP32_Q_CONVERT(-0.27851968939),
    DSP32_Q_CONVERT(0.95870347490), DSP32_Q_CONVERT(-0.28440753721),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.95514116831), DSP32_Q_CONVERT(-0.29615088824),
    DSP32_Q_CONVERT(0.95330604035), DSP32_Q_CONVERT(-0.30200594932),
    DSP32_Q_CONVERT(0.95143502097), DSP32_Q_CONVERT(-0.30784964004),
    DSP32_Q_CONVERT(0.94952818059), DSP32_Q_CONVERT(-0.31368174040),
    DSP32_Q_CONVERT(0.94758559102), DSP32_Q_CONVERT(-0.31950203082),
    DSP32_Q_CONVERT(0.94560732538), DSP32_Q_CONVERT(-0.32531029216),
    DSP32_Q_CONVERT(0.94359345816), DSP32_Q_CONVERT(-0.33110630576),
    DSP32_Q_CONVERT(0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(0.93945922360), DSP32_Q_CONVERT(-0.34266071731),
    DSP32_Q_CONVERT(0.93733901191), DSP32_Q_CONVERT(-0.34841868025),
    DSP32_Q_CONVERT(0.93518350994), DSP32_Q_CONVERT(-0.35416352542),
    DSP32_Q_CONVERT(0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(0.93076696108), DSP32_Q_CONVERT(-0.36561299780),
    DSP32_Q_CONVERT(0.92850608047), DSP32_Q_CONVERT(-0.37131719395),
    DSP32_Q_CONVERT(0.92621024214), DSP32_Q_CONVERT(-0.37700741022),
    DSP32_Q_CONVERT(0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(0.92151403934), DSP32_Q_CONVERT(-0.38834504670),
    DSP32_Q_CONVERT(0.91911385169), DSP32_Q_CONVERT(-0.39399204006),
    DSP32_Q_CONVERT(0.91667905992), DSP32_Q_CONVERT(-0.39962419985),
    DSP32_Q_CONVERT(0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(0.91170603201), DSP32_Q_CONVERT(-0.41084317106),
    DSP32_Q_CONVERT(0.90916798309), DSP32_Q_CONVERT(-0.41642956010),
    DSP32_Q_CONVERT(0.90659570451), DSP32_Q_CONVERT(-0.42200027080),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.90134884705), DSP32_Q_CONVERT(-0.43309381885),
    DSP32_Q_CONVERT(0.89867446569), DSP32_Q_CONVERT(-0.43861623854),
    DSP32_Q_CONVERT(0.89596624976), DSP32_Q_CONVERT(-0.44412214457),
    DSP32_Q_CONVERT(0.89322430120), DSP32_Q_CONVERT(-0.44961132965),
    DSP32_Q_CONVERT(0.89044872324), DSP32_Q_CONVERT(-0.45508358713),
    DSP32_Q_CONVERT(0.88763962040), DSP32_Q_CONVERT(-0.46053871096),
    DSP32_Q_CONVERT(0.88479709843), DSP32_Q_CONVERT(-0.46597649577),
    DSP32_Q_CONVERT(0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(0.87901222643), DSP32_Q_CONVERT(-0.47679923006),
    DSP32_Q_CONVERT(0.87607009420), DSP32_Q_CONVERT(-0.48218377208),
    DSP32_Q_CONVERT(0.87309497842), DSP32_Q_CONVERT(-0.48755016015),
    DSP32_Q_CONVERT(0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(0.86704624552), DSP32_Q_CONVERT(-0.49822766697),
    DSP32_Q_CONVERT(0.86397285612), DSP32_Q_CONVERT(-0.50353838373),
    DSP32_Q_CONVERT(0.86086693864), DSP32_Q_CONVERT(-0.50883014254),
    DSP32_Q_CONVERT(0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(0.85455798837), DSP32_Q_CONVERT(-0.51935599017),
    DSP32_Q_CONVERT(0.85135519311), DSP32_Q_CONVERT(-0.52458968268),
    DSP32_Q_CONVERT(0.84812034480), DSP32_Q_CONVERT(-0.52980362469),
    DSP32_Q_CONVERT(0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(0.84155497744), DSP32_Q_CONVERT(-0.54017147273),
    DSP32_Q_CONVERT(0.83822470555), DSP32_Q_CONVERT(-0.54532498842),
    DSP32_Q_CONVERT(0.83486287499), DSP32_Q_CONVERT(-0.55045797294),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.82804504526), DSP32_Q_CONVERT(-0.56066157620),
    DSP32_Q_CONVERT(0.82458930279), DSP32_Q_CONVERT(-0.56573181078),
    DSP32_Q_CONVERT(0.82110251499), DSP32_Q_CONVERT(-0.57078074589),
    DSP32_Q_CONVERT(0.81758481315), DSP32_Q_CONVERT(-0.57580819142),
    DSP32_Q_CONVERT(0.81403632971), DSP32_Q_CONVERT(-0.58081395810),
    DSP32_Q_CONVERT(0.81045719825), DSP32_Q_CONVERT(-0.58579785746),
    DSP32_Q_CONVERT(0.80684755354), DSP32_Q_CONVERT(-0.59075970186),
    DSP32_Q_CONVERT(0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(0.79953726911), DSP32_Q_CONVERT(-0.60061647938),
    DSP32_Q_CONVERT(0.79583690461), DSP32_Q_CONVERT(-0.60551104140),
    DSP32_Q_CONVERT(0.79210657730), DSP32_Q_CONVERT(-0.61038280628),
    DSP32_Q_CONVERT(0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(0.78455659716), DSP32_Q_CONVERT(-0.62005721176),
    DSP32_Q_CONVERT(0.78073722857), DSP32_Q_CONVERT(-0.62485948814),
    DSP32_Q_CONVERT(0.77688846567), DSP32_Q_CONVERT(-0.62963823891),
    DSP32_Q_CONVERT(0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(0.76910333765), DSP32_Q_CONVERT(-0.63912444486),
    DSP32_Q_CONVERT(0.76516726562), DSP32_Q_CONVERT(-0.64383154289),
    DSP32_Q_CONVERT(0.76120238548), DSP32_Q_CONVERT(-0.64851440102),
    DSP32_Q_CONVERT(0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(0.75318679904), DSP32_Q_CONVERT(-0.65780669330),
    DSP32_Q_CONVERT(0.74913639452), DSP32_Q_CONVERT(-0.66241577759),
    DSP32_Q_CONVERT(0.74505778544), DSP32_Q_CONVERT(-0.66699992230),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.73681656888), DSP32_Q_CONVERT(-0.67609270358),
    DSP32_Q_CONVERT(0.73265427167), DSP32_Q_CONVERT(-0.68060099780),
    DSP32_Q_CONVERT(0.72846439045), DSP32_Q_CONVERT(-0.68508366777),
    DSP32_Q_CONVERT(0.72424708295), DSP32_Q_CONVERT(-0.68954054474),
    DSP32_Q_CONVERT(0.72000250796), DSP32_Q_CONVERT(-0.69397146089),
    DSP32_Q_CONVERT(0.71573082528), DSP32_Q_CONVERT(-0.69837624941),
    DSP32_Q_CONVERT(0.71143219575), DSP32_Q_CONVERT(-0.70275474446),
    DSP32_Q_CONVERT(0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(0.70275474446), DSP32_Q_CONVERT(-0.71143219575),
    DSP32_Q_CONVERT(0.69837624941), DSP32_Q_CONVERT(-0.71573082528),
    DSP32_Q_CONVERT(0.69397146089), DSP32_Q_CONVERT(-0.72000250796),
    DSP32_Q_CONVERT(0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(0.68508366777), DSP32_Q_CONVERT(-0.72846439045),
    DSP32_Q_CONVERT(0.68060099780), DSP32_Q_CONVERT(-0.73265427167),
    DSP32_Q_CONVERT(0.67609270358), DSP32_Q_CONVERT(-0.73681656888),
    DSP32_Q_CONVERT(0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(0.66699992230), DSP32_Q_CONVERT(-0.74505778544),
    DSP32_Q_CONVERT(0.66241577759), DSP32_Q_CONVERT(-0.74913639452),
    DSP32_Q_CONVERT(0.65780669330), DSP32_Q_CONVERT(-0.75318679904),
    DSP32_Q_CONVERT(0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(0.64851440102), DSP32_Q_CONVERT(-0.76120238548),
    DSP32_Q_CONVERT(0.64383154289), DSP32_Q_CONVERT(-0.76516726562),
    DSP32_Q_CONVERT(0.63912444486), DSP32_Q_CONVERT(-0.76910333765),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.62963823891), DSP32_Q_CONVERT(-0.77688846567),
    DSP32_Q_CONVERT(0.62485948814), DSP32_Q_CONVERT(-0.78073722857),
    DSP32_Q_CONVERT(0.62005721176), DSP32_Q_CONVERT(-0.78455659716),
    DSP32_Q_CONVERT(0.61523159058), DSP32_Q_CONVERT(-0.78834642763),
    DSP32_Q_CONVERT(0.61038280628), DSP32_Q_CONVERT(-0.79210657730),
    DSP32_Q_CONVERT(0.60551104140), DSP32_Q_CONVERT(-0.79583690461),
    DSP32_Q_CONVERT(0.60061647938), DSP32_Q_CONVERT(-0.79953726911),
    DSP32_Q_CONVERT(0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.59075970186), DSP32_Q_CONVERT(-0.80684755354),
    DSP32_Q_CONVERT(0.58579785746), DSP32_Q_CONVERT(-0.81045719825),
    DSP32_Q_CONVERT(0.58081395810), DSP32_Q_CONVERT(-0.81403632971),
    DSP32_Q_CONVERT(0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(0.57078074589), DSP32_Q_CONVERT(-0.82110251499),
    DSP32_Q_CONVERT(0.56573181078), DSP32_Q_CONVERT(-0.82458930279),
    DSP32_Q_CONVERT(0.56066157620), DSP32_Q_CONVERT(-0.82804504526),
    DSP32_Q_CONVERT(0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(0.55045797294), DSP32_Q_CONVERT(-0.83486287499),
    DSP32_Q_CONVERT(0.54532498842), DSP32_Q_CONVERT(-0.83822470555),
    DSP32_Q_CONVERT(0.54017147273), DSP32_Q_CONVERT(-0.84155497744),
    DSP32_Q_CONVERT(0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(0.52980362469), DSP32_Q_CONVERT(-0.84812034480),
    DSP32_Q_CONVERT(0.52458968268), DSP32_Q_CONVERT(-0.85135519311),
    DSP32_Q_CONVERT(0.51935599017), DSP32_Q_CONVERT(-0.85455798837),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.50883014254), DSP32_Q_CONVERT(-0.86086693864),
    DSP32_Q_CONVERT(0.50353838373), DSP32_Q_CONVERT(-0.86397285612),
    DSP32_Q_CONVERT(0.49822766697), DSP32_Q_CONVERT(-0.86704624552),
    DSP32_Q_CONVERT(0.49289819223), DSP32_Q_CONVERT(-0.87008699111),
    DSP32_Q_CONVERT(0.48755016015), DSP32_Q_CONVERT(-0.87309497842),
    DSP32_Q_CONVERT(0.48218377208), DSP32_Q_CONVERT(-0.87607009420),
    DSP32_Q_CONVERT(0.47679923006), DSP32_Q_CONVERT(-0.87901222643),
    DSP32_Q_CONVERT(0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(0.46597649577), DSP32_Q_CONVERT(-0.88479709843),
    DSP32_Q_CONVERT(0.46053871096), DSP32_Q_CONVERT(-0.88763962040),
    DSP32_Q_CONVERT(0.45508358713), DSP32_Q_CONVERT(-0.89044872324),
    DSP32_Q_CONVERT(0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(0.44412214457), DSP32_Q_CONVERT(-0.89596624976),
    DSP32_Q_CONVERT(0.43861623854), DSP32_Q_CONVERT(-0.89867446569),
    DSP32_Q_CONVERT(0.43309381885), DSP32_Q_CONVERT(-0.90134884705),
    DSP32_Q_CONVERT(0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(0.42200027080), DSP32_Q_CONVERT(-0.90659570451),
    DSP32_Q_CONVERT(0.41642956010), DSP32_Q_CONVERT(-0.90916798309),
    DSP32_Q_CONVERT(0.41084317106), DSP32_Q_CONVERT(-0.91170603201),
    DSP32_Q_CONVERT(0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(0.39962419985), DSP32_Q_CONVERT(-0.91667905992),
    DSP32_Q_CONVERT(0.39399204006), DSP32_Q_CONVERT(-0.91911385169),
    DSP32_Q_CONVERT(0.38834504670), DSP32_Q_CONVERT(-0.92151403934),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(0.37700741022), DSP32_Q_CONVERT(-0.92621024214),
    DSP32_Q_CONVERT(0.37131719395), DSP32_Q_CONVERT(-0.92850608047),
    DSP32_Q_CONVERT(0.36561299780), DSP32_Q_CONVERT(-0.93076696108),
    DSP32_Q_CONVERT(0.35989503653), DSP32_Q_CONVERT(-0.93299279883),
    DSP32_Q_CONVERT(0.35416352542), DSP32_Q_CONVERT(-0.93518350994),
    DSP32_Q_CONVERT(0.34841868025), DSP32_Q_CONVERT(-0.93733901191),
    DSP32_Q_CONVERT(0.34266071731), DSP32_Q_CONVERT(-0.93945922360),
    DSP32_Q_CONVERT(0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(0.33110630576), DSP32_Q_CONVERT(-0.94359345816),
    DSP32_Q_CONVERT(0.32531029216), DSP32_Q_CONVERT(-0.94560732538),
    DSP32_Q_CONVERT(0.31950203082), DSP32_Q_CONVERT(-0.94758559102),
    DSP32_Q_CONVERT(0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(0.30784964004), DSP32_Q_CONVERT(-0.95143502097),
    DSP32_Q_CONVERT(0.30200594932), DSP32_Q_CONVERT(-0.95330604035),
    DSP32_Q_CONVERT(0.29615088824), DSP32_Q_CONVERT(-0.95514116831),
    DSP32_Q_CONVERT(0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(0.28440753721), DSP32_Q_CONVERT(-0.95870347490),
    DSP32_Q_CONVERT(0.27851968939), DSP32_Q_CONVERT(-0.96043051942),
    DSP32_Q_CONVERT(0.27262135545), DSP32_Q_CONVERT(-0.96212140427),
    DSP32_Q_CONVERT(0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(0.26079411792), DSP32_Q_CONVERT(-0.96539444170),
    DSP32_Q_CONVERT(0.25486565960), DSP32_Q_CONVERT(-0.96697647104),
    DSP32_Q_CONVERT(0.24892760575), DSP32_Q_CONVERT(-0.96852209427),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(0.23702360599), DSP32_Q_CONVERT(-0.97150389099),
    DSP32_Q_CONVERT(0.23105810828), DSP32_Q_CONVERT(-0.97293995221),
    DSP32_Q_CONVERT(0.22508391136), DSP32_Q_CONVERT(-0.97433938279),
    DSP32_Q_CONVERT(0.21910124016), DSP32_Q_CONVERT(-0.97570213004),
    DSP32_Q_CONVERT(0.21311031992), DSP32_Q_CONVERT(-0.97702814266),
    DSP32_Q_CONVERT(0.20711137619), DSP32_Q_CONVERT(-0.97831737072),
    DSP32_Q_CONVERT(0.20110463484), DSP32_Q_CONVERT(-0.97956976569),
    DSP32_Q_CONVERT(0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(0.18906866415), DSP32_Q_CONVERT(-0.98196386911),
    DSP32_Q_CONVERT(0.18303988796), DSP32_Q_CONVERT(-0.98310548743),
    DSP32_Q_CONVERT(0.17700422041), DSP32_Q_CONVERT(-0.98421009239),
    DSP32_Q_CONVERT(0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(0.16491312049), DSP32_Q_CONVERT(-0.98630809724),
    DSP32_Q_CONVERT(0.15885814333), DSP32_Q_CONVERT(-0.98730141816),
    DSP32_Q_CONVERT(0.15279718526), DSP32_Q_CONVERT(-0.98825756773),
    DSP32_Q_CONVERT(0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(0.14065823933), DSP32_Q_CONVERT(-0.99005821026),
    DSP32_Q_CONVERT(0.13458070851), DSP32_Q_CONVERT(-0.99090263543),
    DSP32_Q_CONVERT(0.12849811079), DSP32_Q_CONVERT(-0.99170975367),
    DSP32_Q_CONVERT(0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(0.11631863091), DSP32_Q_CONVERT(-0.99321194923),
    DSP32_Q_CONVERT(0.11022220729), DSP32_Q_CONVERT(-0.99390697000),
    DSP32_Q_CONVERT(0.10412163387), DSP32_Q_CONVERT(-0.99456457073),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(0.09190895650), DSP32_Q_CONVERT(-0.99576741447),
    DSP32_Q_CONVERT(0.08579731234), DSP32_Q_CONVERT(-0.99631261218),
    DSP32_Q_CONVERT(0.07968243797), DSP32_Q_CONVERT(-0.99682029929),
    DSP32_Q_CONVERT(0.07356456360), DSP32_Q_CONVERT(-0.99729045668),
    DSP32_Q_CONVERT(0.06744391956), DSP32_Q_CONVERT(-0.99772306664),
    DSP32_Q_CONVERT(0.06132073630), DSP32_Q_CONVERT(-0.99811811290),
    DSP32_Q_CONVERT(0.05519524435), DSP32_Q_CONVERT(-0.99847558057),
    DSP32_Q_CONVERT(0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(0.04293825693), DSP32_Q_CONVERT(-0.99907772775),
    DSP32_Q_CONVERT(0.03680722294), DSP32_Q_CONVERT(-0.99932238459),
    DSP32_Q_CONVERT(0.03067480318), DSP32_Q_CONVERT(-0.99952941750),
    DSP32_Q_CONVERT(0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(0.01840672991), DSP32_Q_CONVERT(-0.99983058180),
    DSP32_Q_CONVERT(0.01227153829), DSP32_Q_CONVERT(-0.99992470184),
    DSP32_Q_CONVERT(0.00613588465), DSP32_Q_CONVERT(-0.99998117528),
    DSP32_Q_CONVERT(0.00000000000), DSP32_Q_CONVERT(-1.00000000000)
  };

  // If we need to optimize the code to use small twiddle factors table
#  if !(DSP_OPTIMIZATION & DSP_OPTI_SIZE)
    A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp32_t dsp32_twiddle_factors2[DSP32_N_TWIDDLE_FACTORS] = {
    /*Re(w2)   Im(w2)   Re(w3)   Im(w3)  */
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(0.99992470184), DSP32_Q_CONVERT(-0.01227153829),
    DSP32_Q_CONVERT(0.99983058180), DSP32_Q_CONVERT(-0.01840672991),
    DSP32_Q_CONVERT(0.99969881870), DSP32_Q_CONVERT(-0.02454122852),
    DSP32_Q_CONVERT(0.99932238459), DSP32_Q_CONVERT(-0.03680722294),
    DSP32_Q_CONVERT(0.99932238459), DSP32_Q_CONVERT(-0.03680722294),
    DSP32_Q_CONVERT(0.99847558057), DSP32_Q_CONVERT(-0.05519524435),
    DSP32_Q_CONVERT(0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(0.99811811290), DSP32_Q_CONVERT(-0.06132073630),
    DSP32_Q_CONVERT(0.99576741447), DSP32_Q_CONVERT(-0.09190895650),
    DSP32_Q_CONVERT(0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(0.99390697000), DSP32_Q_CONVERT(-0.11022220729),
    DSP32_Q_CONVERT(0.99631261218), DSP32_Q_CONVERT(-0.08579731234),
    DSP32_Q_CONVERT(0.99170975367), DSP32_Q_CONVERT(-0.12849811079),
    DSP32_Q_CONVERT(0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.99390697000), DSP32_Q_CONVERT(-0.11022220729),
    DSP32_Q_CONVERT(0.98630809724), DSP32_Q_CONVERT(-0.16491312049),
    DSP32_Q_CONVERT(0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(0.98310548743), DSP32_Q_CONVERT(-0.18303988796),
    DSP32_Q_CONVERT(0.99090263543), DSP32_Q_CONVERT(-0.13458070851),
    DSP32_Q_CONVERT(0.97956976569), DSP32_Q_CONVERT(-0.20110463484),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(0.98730141816), DSP32_Q_CONVERT(-0.15885814333),
    DSP32_Q_CONVERT(0.97150389099), DSP32_Q_CONVERT(-0.23702360599),
    DSP32_Q_CONVERT(0.98527764239), DSP32_Q_CONVERT(-0.17096188876),
    DSP32_Q_CONVERT(0.96697647104), DSP32_Q_CONVERT(-0.25486565960),
    DSP32_Q_CONVERT(0.98310548743), DSP32_Q_CONVERT(-0.18303988796),
    DSP32_Q_CONVERT(0.96212140427), DSP32_Q_CONVERT(-0.27262135545),
    DSP32_Q_CONVERT(0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.97831737072), DSP32_Q_CONVERT(-0.20711137619),
    DSP32_Q_CONVERT(0.95143502097), DSP32_Q_CONVERT(-0.30784964004),
    DSP32_Q_CONVERT(0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(0.94560732538), DSP32_Q_CONVERT(-0.32531029216),
    DSP32_Q_CONVERT(0.97293995221), DSP32_Q_CONVERT(-0.23105810828),
    DSP32_Q_CONVERT(0.93945922360), DSP32_Q_CONVERT(-0.34266071731),
    DSP32_Q_CONVERT(0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(0.96697647104), DSP32_Q_CONVERT(-0.25486565960),
    DSP32_Q_CONVERT(0.92621024214), DSP32_Q_CONVERT(-0.37700741022),
    DSP32_Q_CONVERT(0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(0.91911385169), DSP32_Q_CONVERT(-0.39399204006),
    DSP32_Q_CONVERT(0.96043051942), DSP32_Q_CONVERT(-0.27851968939),
    DSP32_Q_CONVERT(0.91170603201), DSP32_Q_CONVERT(-0.41084317106),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.95330604035), DSP32_Q_CONVERT(-0.30200594932),
    DSP32_Q_CONVERT(0.89596624976), DSP32_Q_CONVERT(-0.44412214457),
    DSP32_Q_CONVERT(0.94952818059), DSP32_Q_CONVERT(-0.31368174040),
    DSP32_Q_CONVERT(0.88763962040), DSP32_Q_CONVERT(-0.46053871096),
    DSP32_Q_CONVERT(0.94560732538), DSP32_Q_CONVERT(-0.32531029216),
    DSP32_Q_CONVERT(0.87901222643), DSP32_Q_CONVERT(-0.47679923006),
    DSP32_Q_CONVERT(0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(0.93733901191), DSP32_Q_CONVERT(-0.34841868025),
    DSP32_Q_CONVERT(0.86086693864), DSP32_Q_CONVERT(-0.50883014254),
    DSP32_Q_CONVERT(0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(0.85135519311), DSP32_Q_CONVERT(-0.52458968268),
    DSP32_Q_CONVERT(0.92850608047), DSP32_Q_CONVERT(-0.37131719395),
    DSP32_Q_CONVERT(0.84155497744), DSP32_Q_CONVERT(-0.54017147273),
    DSP32_Q_CONVERT(0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.91911385169), DSP32_Q_CONVERT(-0.39399204006),
    DSP32_Q_CONVERT(0.82110251499), DSP32_Q_CONVERT(-0.57078074589),
    DSP32_Q_CONVERT(0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(0.81045719825), DSP32_Q_CONVERT(-0.58579785746),
    DSP32_Q_CONVERT(0.90916798309), DSP32_Q_CONVERT(-0.41642956010),
    DSP32_Q_CONVERT(0.79953726911), DSP32_Q_CONVERT(-0.60061647938),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(0.89867446569), DSP32_Q_CONVERT(-0.43861623854),
    DSP32_Q_CONVERT(0.77688846567), DSP32_Q_CONVERT(-0.62963823891),
    DSP32_Q_CONVERT(0.89322430120), DSP32_Q_CONVERT(-0.44961132965),
    DSP32_Q_CONVERT(0.76516726562), DSP32_Q_CONVERT(-0.64383154289),
    DSP32_Q_CONVERT(0.88763962040), DSP32_Q_CONVERT(-0.46053871096),
    DSP32_Q_CONVERT(0.75318679904), DSP32_Q_CONVERT(-0.65780669330),
    DSP32_Q_CONVERT(0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.87607009420), DSP32_Q_CONVERT(-0.48218377208),
    DSP32_Q_CONVERT(0.72846439045), DSP32_Q_CONVERT(-0.68508366777),
    DSP32_Q_CONVERT(0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(0.71573082528), DSP32_Q_CONVERT(-0.69837624941),
    DSP32_Q_CONVERT(0.86397285612), DSP32_Q_CONVERT(-0.50353838373),
    DSP32_Q_CONVERT(0.70275474446), DSP32_Q_CONVERT(-0.71143219575),
    DSP32_Q_CONVERT(0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(0.85135519311), DSP32_Q_CONVERT(-0.52458968268),
    DSP32_Q_CONVERT(0.67609270358), DSP32_Q_CONVERT(-0.73681656888),
    DSP32_Q_CONVERT(0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(0.66241577759), DSP32_Q_CONVERT(-0.74913639452),
    DSP32_Q_CONVERT(0.83822470555), DSP32_Q_CONVERT(-0.54532498842),
    DSP32_Q_CONVERT(0.64851440102), DSP32_Q_CONVERT(-0.76120238548),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.82458930279), DSP32_Q_CONVERT(-0.56573181078),
    DSP32_Q_CONVERT(0.62005721176), DSP32_Q_CONVERT(-0.78455659716),
    DSP32_Q_CONVERT(0.81758481315), DSP32_Q_CONVERT(-0.57580819142),
    DSP32_Q_CONVERT(0.60551104140), DSP32_Q_CONVERT(-0.79583690461),
    DSP32_Q_CONVERT(0.81045719825), DSP32_Q_CONVERT(-0.58579785746),
    DSP32_Q_CONVERT(0.59075970186), DSP32_Q_CONVERT(-0.80684755354),
    DSP32_Q_CONVERT(0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(0.79583690461), DSP32_Q_CONVERT(-0.60551104140),
    DSP32_Q_CONVERT(0.56066157620), DSP32_Q_CONVERT(-0.82804504526),
    DSP32_Q_CONVERT(0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(0.54532498842), DSP32_Q_CONVERT(-0.83822470555),
    DSP32_Q_CONVERT(0.78073722857), DSP32_Q_CONVERT(-0.62485948814),
    DSP32_Q_CONVERT(0.52980362469), DSP32_Q_CONVERT(-0.84812034480),
    DSP32_Q_CONVERT(0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.76516726562), DSP32_Q_CONVERT(-0.64383154289),
    DSP32_Q_CONVERT(0.49822766697), DSP32_Q_CONVERT(-0.86704624552),
    DSP32_Q_CONVERT(0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(0.48218377208), DSP32_Q_CONVERT(-0.87607009420),
    DSP32_Q_CONVERT(0.74913639452), DSP32_Q_CONVERT(-0.66241577759),
    DSP32_Q_CONVERT(0.46597649577), DSP32_Q_CONVERT(-0.88479709843),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(0.73265427167), DSP32_Q_CONVERT(-0.68060099780),
    DSP32_Q_CONVERT(0.43309381885), DSP32_Q_CONVERT(-0.90134884705),
    DSP32_Q_CONVERT(0.72424708295), DSP32_Q_CONVERT(-0.68954054474),
    DSP32_Q_CONVERT(0.41642956010), DSP32_Q_CONVERT(-0.90916798309),
    DSP32_Q_CONVERT(0.71573082528), DSP32_Q_CONVERT(-0.69837624941),
    DSP32_Q_CONVERT(0.39962419985), DSP32_Q_CONVERT(-0.91667905992),
    DSP32_Q_CONVERT(0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(0.69837624941), DSP32_Q_CONVERT(-0.71573082528),
    DSP32_Q_CONVERT(0.36561299780), DSP32_Q_CONVERT(-0.93076696108),
    DSP32_Q_CONVERT(0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(0.34841868025), DSP32_Q_CONVERT(-0.93733901191),
    DSP32_Q_CONVERT(0.68060099780), DSP32_Q_CONVERT(-0.73265427167),
    DSP32_Q_CONVERT(0.33110630576), DSP32_Q_CONVERT(-0.94359345816),
    DSP32_Q_CONVERT(0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(0.66241577759), DSP32_Q_CONVERT(-0.74913639452),
    DSP32_Q_CONVERT(0.29615088824), DSP32_Q_CONVERT(-0.95514116831),
    DSP32_Q_CONVERT(0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(0.27851968939), DSP32_Q_CONVERT(-0.96043051942),
    DSP32_Q_CONVERT(0.64383154289), DSP32_Q_CONVERT(-0.76516726562),
    DSP32_Q_CONVERT(0.26079411792), DSP32_Q_CONVERT(-0.96539444170),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(0.62485948814), DSP32_Q_CONVERT(-0.78073722857),
    DSP32_Q_CONVERT(0.22508391136), DSP32_Q_CONVERT(-0.97433938279),
    DSP32_Q_CONVERT(0.61523159058), DSP32_Q_CONVERT(-0.78834642763),
    DSP32_Q_CONVERT(0.20711137619), DSP32_Q_CONVERT(-0.97831737072),
    DSP32_Q_CONVERT(0.60551104140), DSP32_Q_CONVERT(-0.79583690461),
    DSP32_Q_CONVERT(0.18906866415), DSP32_Q_CONVERT(-0.98196386911),
    DSP32_Q_CONVERT(0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(0.58579785746), DSP32_Q_CONVERT(-0.81045719825),
    DSP32_Q_CONVERT(0.15279718526), DSP32_Q_CONVERT(-0.98825756773),
    DSP32_Q_CONVERT(0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(0.13458070851), DSP32_Q_CONVERT(-0.99090263543),
    DSP32_Q_CONVERT(0.56573181078), DSP32_Q_CONVERT(-0.82458930279),
    DSP32_Q_CONVERT(0.11631863091), DSP32_Q_CONVERT(-0.99321194923),
    DSP32_Q_CONVERT(0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(0.54532498842), DSP32_Q_CONVERT(-0.83822470555),
    DSP32_Q_CONVERT(0.07968243797), DSP32_Q_CONVERT(-0.99682029929),
    DSP32_Q_CONVERT(0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(0.06132073630), DSP32_Q_CONVERT(-0.99811811290),
    DSP32_Q_CONVERT(0.52458968268), DSP32_Q_CONVERT(-0.85135519311),
    DSP32_Q_CONVERT(0.04293825693), DSP32_Q_CONVERT(-0.99907772775),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(0.50353838373), DSP32_Q_CONVERT(-0.86397285612),
    DSP32_Q_CONVERT(0.00613588465), DSP32_Q_CONVERT(-0.99998117528),
    DSP32_Q_CONVERT(0.49289819223), DSP32_Q_CONVERT(-0.87008699111),
    DSP32_Q_CONVERT(-0.01227153829), DSP32_Q_CONVERT(-0.99992470184),
    DSP32_Q_CONVERT(0.48218377208), DSP32_Q_CONVERT(-0.87607009420),
    DSP32_Q_CONVERT(-0.03067480318), DSP32_Q_CONVERT(-0.99952941750),
    DSP32_Q_CONVERT(0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(-0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(0.46053871096), DSP32_Q_CONVERT(-0.88763962040),
    DSP32_Q_CONVERT(-0.06744391956), DSP32_Q_CONVERT(-0.99772306664),
    DSP32_Q_CONVERT(0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(-0.08579731234), DSP32_Q_CONVERT(-0.99631261218),
    DSP32_Q_CONVERT(0.43861623854), DSP32_Q_CONVERT(-0.89867446569),
    DSP32_Q_CONVERT(-0.10412163387), DSP32_Q_CONVERT(-0.99456457073),
    DSP32_Q_CONVERT(0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(-0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(0.41642956010), DSP32_Q_CONVERT(-0.90916798309),
    DSP32_Q_CONVERT(-0.14065823933), DSP32_Q_CONVERT(-0.99005821026),
    DSP32_Q_CONVERT(0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(-0.15885814333), DSP32_Q_CONVERT(-0.98730141816),
    DSP32_Q_CONVERT(0.39399204006), DSP32_Q_CONVERT(-0.91911385169),
    DSP32_Q_CONVERT(-0.17700422041), DSP32_Q_CONVERT(-0.98421009239),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(-0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(0.37131719395), DSP32_Q_CONVERT(-0.92850608047),
    DSP32_Q_CONVERT(-0.21311031992), DSP32_Q_CONVERT(-0.97702814266),
    DSP32_Q_CONVERT(0.35989503653), DSP32_Q_CONVERT(-0.93299279883),
    DSP32_Q_CONVERT(-0.23105810828), DSP32_Q_CONVERT(-0.97293995221),
    DSP32_Q_CONVERT(0.34841868025), DSP32_Q_CONVERT(-0.93733901191),
    DSP32_Q_CONVERT(-0.24892760575), DSP32_Q_CONVERT(-0.96852209427),
    DSP32_Q_CONVERT(0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(-0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(0.32531029216), DSP32_Q_CONVERT(-0.94560732538),
    DSP32_Q_CONVERT(-0.28440753721), DSP32_Q_CONVERT(-0.95870347490),
    DSP32_Q_CONVERT(0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(-0.30200594932), DSP32_Q_CONVERT(-0.95330604035),
    DSP32_Q_CONVERT(0.30200594932), DSP32_Q_CONVERT(-0.95330604035),
    DSP32_Q_CONVERT(-0.31950203082), DSP32_Q_CONVERT(-0.94758559102),
    DSP32_Q_CONVERT(0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(-0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(0.27851968939), DSP32_Q_CONVERT(-0.96043051942),
    DSP32_Q_CONVERT(-0.35416352542), DSP32_Q_CONVERT(-0.93518350994),
    DSP32_Q_CONVERT(0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(-0.37131719395), DSP32_Q_CONVERT(-0.92850608047),
    DSP32_Q_CONVERT(0.25486565960), DSP32_Q_CONVERT(-0.96697647104),
    DSP32_Q_CONVERT(-0.38834504670), DSP32_Q_CONVERT(-0.92151403934),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(-0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(0.23105810828), DSP32_Q_CONVERT(-0.97293995221),
    DSP32_Q_CONVERT(-0.42200027080), DSP32_Q_CONVERT(-0.90659570451),
    DSP32_Q_CONVERT(0.21910124016), DSP32_Q_CONVERT(-0.97570213004),
    DSP32_Q_CONVERT(-0.43861623854), DSP32_Q_CONVERT(-0.89867446569),
    DSP32_Q_CONVERT(0.20711137619), DSP32_Q_CONVERT(-0.97831737072),
    DSP32_Q_CONVERT(-0.45508358713), DSP32_Q_CONVERT(-0.89044872324),
    DSP32_Q_CONVERT(0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(-0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(0.18303988796), DSP32_Q_CONVERT(-0.98310548743),
    DSP32_Q_CONVERT(-0.48755016015), DSP32_Q_CONVERT(-0.87309497842),
    DSP32_Q_CONVERT(0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(-0.50353838373), DSP32_Q_CONVERT(-0.86397285612),
    DSP32_Q_CONVERT(0.15885814333), DSP32_Q_CONVERT(-0.98730141816),
    DSP32_Q_CONVERT(-0.51935599017), DSP32_Q_CONVERT(-0.85455798837),
    DSP32_Q_CONVERT(0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(-0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(0.13458070851), DSP32_Q_CONVERT(-0.99090263543),
    DSP32_Q_CONVERT(-0.55045797294), DSP32_Q_CONVERT(-0.83486287499),
    DSP32_Q_CONVERT(0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(-0.56573181078), DSP32_Q_CONVERT(-0.82458930279),
    DSP32_Q_CONVERT(0.11022220729), DSP32_Q_CONVERT(-0.99390697000),
    DSP32_Q_CONVERT(-0.58081395810), DSP32_Q_CONVERT(-0.81403632971),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(-0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.08579731234), DSP32_Q_CONVERT(-0.99631261218),
    DSP32_Q_CONVERT(-0.61038280628), DSP32_Q_CONVERT(-0.79210657730),
    DSP32_Q_CONVERT(0.07356456360), DSP32_Q_CONVERT(-0.99729045668),
    DSP32_Q_CONVERT(-0.62485948814), DSP32_Q_CONVERT(-0.78073722857),
    DSP32_Q_CONVERT(0.06132073630), DSP32_Q_CONVERT(-0.99811811290),
    DSP32_Q_CONVERT(-0.63912444486), DSP32_Q_CONVERT(-0.76910333765),
    DSP32_Q_CONVERT(0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(-0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(0.03680722294), DSP32_Q_CONVERT(-0.99932238459),
    DSP32_Q_CONVERT(-0.66699992230), DSP32_Q_CONVERT(-0.74505778544),
    DSP32_Q_CONVERT(0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(-0.68060099780), DSP32_Q_CONVERT(-0.73265427167),
    DSP32_Q_CONVERT(0.01227153829), DSP32_Q_CONVERT(-0.99992470184),
    DSP32_Q_CONVERT(-0.69397146089), DSP32_Q_CONVERT(-0.72000250796),
    DSP32_Q_CONVERT(0.00000000000), DSP32_Q_CONVERT(-1.00000000000),
    DSP32_Q_CONVERT(-0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(-0.01227153829), DSP32_Q_CONVERT(-0.99992470184),
    DSP32_Q_CONVERT(-0.72000250796), DSP32_Q_CONVERT(-0.69397146089),
    DSP32_Q_CONVERT(-0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(-0.73265427167), DSP32_Q_CONVERT(-0.68060099780),
    DSP32_Q_CONVERT(-0.03680722294), DSP32_Q_CONVERT(-0.99932238459),
    DSP32_Q_CONVERT(-0.74505778544), DSP32_Q_CONVERT(-0.66699992230),
    DSP32_Q_CONVERT(-0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(-0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(-0.06132073630), DSP32_Q_CONVERT(-0.99811811290),
    DSP32_Q_CONVERT(-0.76910333765), DSP32_Q_CONVERT(-0.63912444486),
    DSP32_Q_CONVERT(-0.07356456360), DSP32_Q_CONVERT(-0.99729045668),
    DSP32_Q_CONVERT(-0.78073722857), DSP32_Q_CONVERT(-0.62485948814),
    DSP32_Q_CONVERT(-0.08579731234), DSP32_Q_CONVERT(-0.99631261218),
    DSP32_Q_CONVERT(-0.79210657730), DSP32_Q_CONVERT(-0.61038280628),
    DSP32_Q_CONVERT(-0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(-0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(-0.11022220729), DSP32_Q_CONVERT(-0.99390697000),
    DSP32_Q_CONVERT(-0.81403632971), DSP32_Q_CONVERT(-0.58081395810),
    DSP32_Q_CONVERT(-0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(-0.82458930279), DSP32_Q_CONVERT(-0.56573181078),
    DSP32_Q_CONVERT(-0.13458070851), DSP32_Q_CONVERT(-0.99090263543),
    DSP32_Q_CONVERT(-0.83486287499), DSP32_Q_CONVERT(-0.55045797294),
    DSP32_Q_CONVERT(-0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(-0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(-0.15885814333), DSP32_Q_CONVERT(-0.98730141816),
    DSP32_Q_CONVERT(-0.85455798837), DSP32_Q_CONVERT(-0.51935599017),
    DSP32_Q_CONVERT(-0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(-0.86397285612), DSP32_Q_CONVERT(-0.50353838373),
    DSP32_Q_CONVERT(-0.18303988796), DSP32_Q_CONVERT(-0.98310548743),
    DSP32_Q_CONVERT(-0.87309497842), DSP32_Q_CONVERT(-0.48755016015),
    DSP32_Q_CONVERT(-0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(-0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(-0.20711137619), DSP32_Q_CONVERT(-0.97831737072),
    DSP32_Q_CONVERT(-0.89044872324), DSP32_Q_CONVERT(-0.45508358713),
    DSP32_Q_CONVERT(-0.21910124016), DSP32_Q_CONVERT(-0.97570213004),
    DSP32_Q_CONVERT(-0.89867446569), DSP32_Q_CONVERT(-0.43861623854),
    DSP32_Q_CONVERT(-0.23105810828), DSP32_Q_CONVERT(-0.97293995221),
    DSP32_Q_CONVERT(-0.90659570451), DSP32_Q_CONVERT(-0.42200027080),
    DSP32_Q_CONVERT(-0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(-0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(-0.25486565960), DSP32_Q_CONVERT(-0.96697647104),
    DSP32_Q_CONVERT(-0.92151403934), DSP32_Q_CONVERT(-0.38834504670),
    DSP32_Q_CONVERT(-0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(-0.92850608047), DSP32_Q_CONVERT(-0.37131719395),
    DSP32_Q_CONVERT(-0.27851968939), DSP32_Q_CONVERT(-0.96043051942),
    DSP32_Q_CONVERT(-0.93518350994), DSP32_Q_CONVERT(-0.35416352542),
    DSP32_Q_CONVERT(-0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(-0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(-0.30200594932), DSP32_Q_CONVERT(-0.95330604035),
    DSP32_Q_CONVERT(-0.94758559102), DSP32_Q_CONVERT(-0.31950203082),
    DSP32_Q_CONVERT(-0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(-0.95330604035), DSP32_Q_CONVERT(-0.30200594932),
    DSP32_Q_CONVERT(-0.32531029216), DSP32_Q_CONVERT(-0.94560732538),
    DSP32_Q_CONVERT(-0.95870347490), DSP32_Q_CONVERT(-0.28440753721),
    DSP32_Q_CONVERT(-0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(-0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(-0.34841868025), DSP32_Q_CONVERT(-0.93733901191),
    DSP32_Q_CONVERT(-0.96852209427), DSP32_Q_CONVERT(-0.24892760575),
    DSP32_Q_CONVERT(-0.35989503653), DSP32_Q_CONVERT(-0.93299279883),
    DSP32_Q_CONVERT(-0.97293995221), DSP32_Q_CONVERT(-0.23105810828),
    DSP32_Q_CONVERT(-0.37131719395), DSP32_Q_CONVERT(-0.92850608047),
    DSP32_Q_CONVERT(-0.97702814266), DSP32_Q_CONVERT(-0.21311031992),
    DSP32_Q_CONVERT(-0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(-0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(-0.39399204006), DSP32_Q_CONVERT(-0.91911385169),
    DSP32_Q_CONVERT(-0.98421009239), DSP32_Q_CONVERT(-0.17700422041),
    DSP32_Q_CONVERT(-0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(-0.98730141816), DSP32_Q_CONVERT(-0.15885814333),
    DSP32_Q_CONVERT(-0.41642956010), DSP32_Q_CONVERT(-0.90916798309),
    DSP32_Q_CONVERT(-0.99005821026), DSP32_Q_CONVERT(-0.14065823933),
    DSP32_Q_CONVERT(-0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(-0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(-0.43861623854), DSP32_Q_CONVERT(-0.89867446569),
    DSP32_Q_CONVERT(-0.99456457073), DSP32_Q_CONVERT(-0.10412163387),
    DSP32_Q_CONVERT(-0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(-0.99631261218), DSP32_Q_CONVERT(-0.08579731234),
    DSP32_Q_CONVERT(-0.46053871096), DSP32_Q_CONVERT(-0.88763962040),
    DSP32_Q_CONVERT(-0.99772306664), DSP32_Q_CONVERT(-0.06744391956),
    DSP32_Q_CONVERT(-0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(-0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(-0.48218377208), DSP32_Q_CONVERT(-0.87607009420),
    DSP32_Q_CONVERT(-0.99952941750), DSP32_Q_CONVERT(-0.03067480318),
    DSP32_Q_CONVERT(-0.49289819223), DSP32_Q_CONVERT(-0.87008699111),
    DSP32_Q_CONVERT(-0.99992470184), DSP32_Q_CONVERT(-0.01227153829),
    DSP32_Q_CONVERT(-0.50353838373), DSP32_Q_CONVERT(-0.86397285612),
    DSP32_Q_CONVERT(-0.99998117528), DSP32_Q_CONVERT(0.00613588465),
    DSP32_Q_CONVERT(-0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(-0.99969881870), DSP32_Q_CONVERT(0.02454122852),
    DSP32_Q_CONVERT(-0.52458968268), DSP32_Q_CONVERT(-0.85135519311),
    DSP32_Q_CONVERT(-0.99907772775), DSP32_Q_CONVERT(0.04293825693),
    DSP32_Q_CONVERT(-0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(-0.99811811290), DSP32_Q_CONVERT(0.06132073630),
    DSP32_Q_CONVERT(-0.54532498842), DSP32_Q_CONVERT(-0.83822470555),
    DSP32_Q_CONVERT(-0.99682029929), DSP32_Q_CONVERT(0.07968243797),
    DSP32_Q_CONVERT(-0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(-0.99518472667), DSP32_Q_CONVERT(0.09801714033),
    DSP32_Q_CONVERT(-0.56573181078), DSP32_Q_CONVERT(-0.82458930279),
    DSP32_Q_CONVERT(-0.99321194923), DSP32_Q_CONVERT(0.11631863091),
    DSP32_Q_CONVERT(-0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(-0.99090263543), DSP32_Q_CONVERT(0.13458070851),
    DSP32_Q_CONVERT(-0.58579785746), DSP32_Q_CONVERT(-0.81045719825),
    DSP32_Q_CONVERT(-0.98825756773), DSP32_Q_CONVERT(0.15279718526),
    DSP32_Q_CONVERT(-0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(-0.98527764239), DSP32_Q_CONVERT(0.17096188876),
    DSP32_Q_CONVERT(-0.60551104140), DSP32_Q_CONVERT(-0.79583690461),
    DSP32_Q_CONVERT(-0.98196386911), DSP32_Q_CONVERT(0.18906866415),
    DSP32_Q_CONVERT(-0.61523159058), DSP32_Q_CONVERT(-0.78834642763),
    DSP32_Q_CONVERT(-0.97831737072), DSP32_Q_CONVERT(0.20711137619),
    DSP32_Q_CONVERT(-0.62485948814), DSP32_Q_CONVERT(-0.78073722857),
    DSP32_Q_CONVERT(-0.97433938279), DSP32_Q_CONVERT(0.22508391136),
    DSP32_Q_CONVERT(-0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(-0.97003125319), DSP32_Q_CONVERT(0.24298017990),
    DSP32_Q_CONVERT(-0.64383154289), DSP32_Q_CONVERT(-0.76516726562),
    DSP32_Q_CONVERT(-0.96539444170), DSP32_Q_CONVERT(0.26079411792),
    DSP32_Q_CONVERT(-0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(-0.96043051942), DSP32_Q_CONVERT(0.27851968939),
    DSP32_Q_CONVERT(-0.66241577759), DSP32_Q_CONVERT(-0.74913639452),
    DSP32_Q_CONVERT(-0.95514116831), DSP32_Q_CONVERT(0.29615088824),
    DSP32_Q_CONVERT(-0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(-0.94952818059), DSP32_Q_CONVERT(0.31368174040),
    DSP32_Q_CONVERT(-0.68060099780), DSP32_Q_CONVERT(-0.73265427167),
    DSP32_Q_CONVERT(-0.94359345816), DSP32_Q_CONVERT(0.33110630576),
    DSP32_Q_CONVERT(-0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(-0.93733901191), DSP32_Q_CONVERT(0.34841868025),
    DSP32_Q_CONVERT(-0.69837624941), DSP32_Q_CONVERT(-0.71573082528),
    DSP32_Q_CONVERT(-0.93076696108), DSP32_Q_CONVERT(0.36561299780),
    DSP32_Q_CONVERT(-0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(-0.92387953251), DSP32_Q_CONVERT(0.38268343237),
    DSP32_Q_CONVERT(-0.71573082528), DSP32_Q_CONVERT(-0.69837624941),
    DSP32_Q_CONVERT(-0.91667905992), DSP32_Q_CONVERT(0.39962419985),
    DSP32_Q_CONVERT(-0.72424708295), DSP32_Q_CONVERT(-0.68954054474),
    DSP32_Q_CONVERT(-0.90916798309), DSP32_Q_CONVERT(0.41642956010),
    DSP32_Q_CONVERT(-0.73265427167), DSP32_Q_CONVERT(-0.68060099780),
    DSP32_Q_CONVERT(-0.90134884705), DSP32_Q_CONVERT(0.43309381885),
    DSP32_Q_CONVERT(-0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(-0.89322430120), DSP32_Q_CONVERT(0.44961132965),
    DSP32_Q_CONVERT(-0.74913639452), DSP32_Q_CONVERT(-0.66241577759),
    DSP32_Q_CONVERT(-0.88479709843), DSP32_Q_CONVERT(0.46597649577),
    DSP32_Q_CONVERT(-0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(-0.87607009420), DSP32_Q_CONVERT(0.48218377208),
    DSP32_Q_CONVERT(-0.76516726562), DSP32_Q_CONVERT(-0.64383154289),
    DSP32_Q_CONVERT(-0.86704624552), DSP32_Q_CONVERT(0.49822766697),
    DSP32_Q_CONVERT(-0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(-0.85772861000), DSP32_Q_CONVERT(0.51410274419),
    DSP32_Q_CONVERT(-0.78073722857), DSP32_Q_CONVERT(-0.62485948814),
    DSP32_Q_CONVERT(-0.84812034480), DSP32_Q_CONVERT(0.52980362469),
    DSP32_Q_CONVERT(-0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(-0.83822470555), DSP32_Q_CONVERT(0.54532498842),
    DSP32_Q_CONVERT(-0.79583690461), DSP32_Q_CONVERT(-0.60551104140),
    DSP32_Q_CONVERT(-0.82804504526), DSP32_Q_CONVERT(0.56066157620),
    DSP32_Q_CONVERT(-0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(-0.81758481315), DSP32_Q_CONVERT(0.57580819142),
    DSP32_Q_CONVERT(-0.81045719825), DSP32_Q_CONVERT(-0.58579785746),
    DSP32_Q_CONVERT(-0.80684755354), DSP32_Q_CONVERT(0.59075970186),
    DSP32_Q_CONVERT(-0.81758481315), DSP32_Q_CONVERT(-0.57580819142),
    DSP32_Q_CONVERT(-0.79583690461), DSP32_Q_CONVERT(0.60551104140),
    DSP32_Q_CONVERT(-0.82458930279), DSP32_Q_CONVERT(-0.56573181078),
    DSP32_Q_CONVERT(-0.78455659716), DSP32_Q_CONVERT(0.62005721176),
    DSP32_Q_CONVERT(-0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(-0.77301045336), DSP32_Q_CONVERT(0.63439328416),
    DSP32_Q_CONVERT(-0.83822470555), DSP32_Q_CONVERT(-0.54532498842),
    DSP32_Q_CONVERT(-0.76120238548), DSP32_Q_CONVERT(0.64851440102),
    DSP32_Q_CONVERT(-0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(-0.74913639452), DSP32_Q_CONVERT(0.66241577759),
    DSP32_Q_CONVERT(-0.85135519311), DSP32_Q_CONVERT(-0.52458968268),
    DSP32_Q_CONVERT(-0.73681656888), DSP32_Q_CONVERT(0.67609270358),
    DSP32_Q_CONVERT(-0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(-0.72424708295), DSP32_Q_CONVERT(0.68954054474),
    DSP32_Q_CONVERT(-0.86397285612), DSP32_Q_CONVERT(-0.50353838373),
    DSP32_Q_CONVERT(-0.71143219575), DSP32_Q_CONVERT(0.70275474446),
    DSP32_Q_CONVERT(-0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(-0.69837624941), DSP32_Q_CONVERT(0.71573082528),
    DSP32_Q_CONVERT(-0.87607009420), DSP32_Q_CONVERT(-0.48218377208),
    DSP32_Q_CONVERT(-0.68508366777), DSP32_Q_CONVERT(0.72846439045),
    DSP32_Q_CONVERT(-0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(-0.67155895485), DSP32_Q_CONVERT(0.74095112535),
    DSP32_Q_CONVERT(-0.88763962040), DSP32_Q_CONVERT(-0.46053871096),
    DSP32_Q_CONVERT(-0.65780669330), DSP32_Q_CONVERT(0.75318679904),
    DSP32_Q_CONVERT(-0.89322430120), DSP32_Q_CONVERT(-0.44961132965),
    DSP32_Q_CONVERT(-0.64383154289), DSP32_Q_CONVERT(0.76516726562),
    DSP32_Q_CONVERT(-0.89867446569), DSP32_Q_CONVERT(-0.43861623854),
    DSP32_Q_CONVERT(-0.62963823891), DSP32_Q_CONVERT(0.77688846567),
    DSP32_Q_CONVERT(-0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(-0.61523159058), DSP32_Q_CONVERT(0.78834642763),
    DSP32_Q_CONVERT(-0.90916798309), DSP32_Q_CONVERT(-0.41642956010),
    DSP32_Q_CONVERT(-0.60061647938), DSP32_Q_CONVERT(0.79953726911),
    DSP32_Q_CONVERT(-0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(-0.58579785746), DSP32_Q_CONVERT(0.81045719825),
    DSP32_Q_CONVERT(-0.91911385169), DSP32_Q_CONVERT(-0.39399204006),
    DSP32_Q_CONVERT(-0.57078074589), DSP32_Q_CONVERT(0.82110251499),
    DSP32_Q_CONVERT(-0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(-0.55557023302), DSP32_Q_CONVERT(0.83146961230),
    DSP32_Q_CONVERT(-0.92850608047), DSP32_Q_CONVERT(-0.37131719395),
    DSP32_Q_CONVERT(-0.54017147273), DSP32_Q_CONVERT(0.84155497744),
    DSP32_Q_CONVERT(-0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(-0.52458968268), DSP32_Q_CONVERT(0.85135519311),
    DSP32_Q_CONVERT(-0.93733901191), DSP32_Q_CONVERT(-0.34841868025),
    DSP32_Q_CONVERT(-0.50883014254), DSP32_Q_CONVERT(0.86086693864),
    DSP32_Q_CONVERT(-0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(-0.49289819223), DSP32_Q_CONVERT(0.87008699111),
    DSP32_Q_CONVERT(-0.94560732538), DSP32_Q_CONVERT(-0.32531029216),
    DSP32_Q_CONVERT(-0.47679923006), DSP32_Q_CONVERT(0.87901222643),
    DSP32_Q_CONVERT(-0.94952818059), DSP32_Q_CONVERT(-0.31368174040),
    DSP32_Q_CONVERT(-0.46053871096), DSP32_Q_CONVERT(0.88763962040),
    DSP32_Q_CONVERT(-0.95330604035), DSP32_Q_CONVERT(-0.30200594932),
    DSP32_Q_CONVERT(-0.44412214457), DSP32_Q_CONVERT(0.89596624976),
    DSP32_Q_CONVERT(-0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(-0.42755509343), DSP32_Q_CONVERT(0.90398929312),
    DSP32_Q_CONVERT(-0.96043051942), DSP32_Q_CONVERT(-0.27851968939),
    DSP32_Q_CONVERT(-0.41084317106), DSP32_Q_CONVERT(0.91170603201),
    DSP32_Q_CONVERT(-0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(-0.39399204006), DSP32_Q_CONVERT(0.91911385169),
    DSP32_Q_CONVERT(-0.96697647104), DSP32_Q_CONVERT(-0.25486565960),
    DSP32_Q_CONVERT(-0.37700741022), DSP32_Q_CONVERT(0.92621024214),
    DSP32_Q_CONVERT(-0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(-0.35989503653), DSP32_Q_CONVERT(0.93299279883),
    DSP32_Q_CONVERT(-0.97293995221), DSP32_Q_CONVERT(-0.23105810828),
    DSP32_Q_CONVERT(-0.34266071731), DSP32_Q_CONVERT(0.93945922360),
    DSP32_Q_CONVERT(-0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(-0.32531029216), DSP32_Q_CONVERT(0.94560732538),
    DSP32_Q_CONVERT(-0.97831737072), DSP32_Q_CONVERT(-0.20711137619),
    DSP32_Q_CONVERT(-0.30784964004), DSP32_Q_CONVERT(0.95143502097),
    DSP32_Q_CONVERT(-0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(-0.29028467725), DSP32_Q_CONVERT(0.95694033573),
    DSP32_Q_CONVERT(-0.98310548743), DSP32_Q_CONVERT(-0.18303988796),
    DSP32_Q_CONVERT(-0.27262135545), DSP32_Q_CONVERT(0.96212140427),
    DSP32_Q_CONVERT(-0.98527764239), DSP32_Q_CONVERT(-0.17096188876),
    DSP32_Q_CONVERT(-0.25486565960), DSP32_Q_CONVERT(0.96697647104),
    DSP32_Q_CONVERT(-0.98730141816), DSP32_Q_CONVERT(-0.15885814333),
    DSP32_Q_CONVERT(-0.23702360599), DSP32_Q_CONVERT(0.97150389099),
    DSP32_Q_CONVERT(-0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(-0.21910124016), DSP32_Q_CONVERT(0.97570213004),
    DSP32_Q_CONVERT(-0.99090263543), DSP32_Q_CONVERT(-0.13458070851),
    DSP32_Q_CONVERT(-0.20110463484), DSP32_Q_CONVERT(0.97956976569),
    DSP32_Q_CONVERT(-0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(-0.18303988796), DSP32_Q_CONVERT(0.98310548743),
    DSP32_Q_CONVERT(-0.99390697000), DSP32_Q_CONVERT(-0.11022220729),
    DSP32_Q_CONVERT(-0.16491312049), DSP32_Q_CONVERT(0.98630809724),
    DSP32_Q_CONVERT(-0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(-0.14673047446), DSP32_Q_CONVERT(0.98917650996),
    DSP32_Q_CONVERT(-0.99631261218), DSP32_Q_CONVERT(-0.08579731234),
    DSP32_Q_CONVERT(-0.12849811079), DSP32_Q_CONVERT(0.99170975367),
    DSP32_Q_CONVERT(-0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(-0.11022220729), DSP32_Q_CONVERT(0.99390697000),
    DSP32_Q_CONVERT(-0.99811811290), DSP32_Q_CONVERT(-0.06132073630),
    DSP32_Q_CONVERT(-0.09190895650), DSP32_Q_CONVERT(0.99576741447),
    DSP32_Q_CONVERT(-0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(-0.07356456360), DSP32_Q_CONVERT(0.99729045668),
    DSP32_Q_CONVERT(-0.99932238459), DSP32_Q_CONVERT(-0.03680722294),
    DSP32_Q_CONVERT(-0.05519524435), DSP32_Q_CONVERT(0.99847558057),
    DSP32_Q_CONVERT(-0.99969881870), DSP32_Q_CONVERT(-0.02454122852),
    DSP32_Q_CONVERT(-0.03680722294), DSP32_Q_CONVERT(0.99932238459),
    DSP32_Q_CONVERT(-0.99992470184), DSP32_Q_CONVERT(-0.01227153829),
    DSP32_Q_CONVERT(-0.01840672991), DSP32_Q_CONVERT(0.99983058180)
    };
#  endif

#endif

/*
 * Twiddle factors for a 4096-point FFT
 * Minimized twiddle factors table: 2*(4096/4)*2 = 4096 bytes
 * Full twiddle factors table: 2*(4096/4)*6 = 12288 bytes
 */
#if DSP32_N_TWIDDLE_FACTORS == 4096

  A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp32_t dsp32_twiddle_factors[DSP32_N_TWIDDLE_FACTORS/2+2] = {
  /*Re(w)   Im(w)  */
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(0.99999882345), DSP32_Q_CONVERT(-0.00153398019),
    DSP32_Q_CONVERT(0.99999529381), DSP32_Q_CONVERT(-0.00306795676),
    DSP32_Q_CONVERT(0.99998941108), DSP32_Q_CONVERT(-0.00460192612),
    DSP32_Q_CONVERT(0.99998117528), DSP32_Q_CONVERT(-0.00613588465),
    DSP32_Q_CONVERT(0.99997058643), DSP32_Q_CONVERT(-0.00766982874),
    DSP32_Q_CONVERT(0.99995764455), DSP32_Q_CONVERT(-0.00920375478),
    DSP32_Q_CONVERT(0.99994234968), DSP32_Q_CONVERT(-0.01073765917),
    DSP32_Q_CONVERT(0.99992470184), DSP32_Q_CONVERT(-0.01227153829),
    DSP32_Q_CONVERT(0.99990470108), DSP32_Q_CONVERT(-0.01380538853),
    DSP32_Q_CONVERT(0.99988234745), DSP32_Q_CONVERT(-0.01533920628),
    DSP32_Q_CONVERT(0.99985764101), DSP32_Q_CONVERT(-0.01687298795),
    DSP32_Q_CONVERT(0.99983058180), DSP32_Q_CONVERT(-0.01840672991),
    DSP32_Q_CONVERT(0.99980116989), DSP32_Q_CONVERT(-0.01994042855),
    DSP32_Q_CONVERT(0.99976940535), DSP32_Q_CONVERT(-0.02147408028),
    DSP32_Q_CONVERT(0.99973528826), DSP32_Q_CONVERT(-0.02300768147),
    DSP32_Q_CONVERT(0.99969881870), DSP32_Q_CONVERT(-0.02454122852),
    DSP32_Q_CONVERT(0.99965999674), DSP32_Q_CONVERT(-0.02607471783),
    DSP32_Q_CONVERT(0.99961882250), DSP32_Q_CONVERT(-0.02760814578),
    DSP32_Q_CONVERT(0.99957529605), DSP32_Q_CONVERT(-0.02914150876),
    DSP32_Q_CONVERT(0.99952941750), DSP32_Q_CONVERT(-0.03067480318),
    DSP32_Q_CONVERT(0.99948118697), DSP32_Q_CONVERT(-0.03220802541),
    DSP32_Q_CONVERT(0.99943060456), DSP32_Q_CONVERT(-0.03374117185),
    DSP32_Q_CONVERT(0.99937767039), DSP32_Q_CONVERT(-0.03527423890),
    DSP32_Q_CONVERT(0.99932238459), DSP32_Q_CONVERT(-0.03680722294),
    DSP32_Q_CONVERT(0.99926474729), DSP32_Q_CONVERT(-0.03834012037),
    DSP32_Q_CONVERT(0.99920475862), DSP32_Q_CONVERT(-0.03987292759),
    DSP32_Q_CONVERT(0.99914241872), DSP32_Q_CONVERT(-0.04140564098),
    DSP32_Q_CONVERT(0.99907772775), DSP32_Q_CONVERT(-0.04293825693),
    DSP32_Q_CONVERT(0.99901068585), DSP32_Q_CONVERT(-0.04447077185),
    DSP32_Q_CONVERT(0.99894129319), DSP32_Q_CONVERT(-0.04600318213),
    DSP32_Q_CONVERT(0.99886954991), DSP32_Q_CONVERT(-0.04753548416),
    DSP32_Q_CONVERT(0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(0.99871901223), DSP32_Q_CONVERT(-0.05059974904),
    DSP32_Q_CONVERT(0.99864021818), DSP32_Q_CONVERT(-0.05213170468),
    DSP32_Q_CONVERT(0.99855907423), DSP32_Q_CONVERT(-0.05366353765),
    DSP32_Q_CONVERT(0.99847558057), DSP32_Q_CONVERT(-0.05519524435),
    DSP32_Q_CONVERT(0.99838973741), DSP32_Q_CONVERT(-0.05672682117),
    DSP32_Q_CONVERT(0.99830154493), DSP32_Q_CONVERT(-0.05825826450),
    DSP32_Q_CONVERT(0.99821100336), DSP32_Q_CONVERT(-0.05978957075),
    DSP32_Q_CONVERT(0.99811811290), DSP32_Q_CONVERT(-0.06132073630),
    DSP32_Q_CONVERT(0.99802287377), DSP32_Q_CONVERT(-0.06285175756),
    DSP32_Q_CONVERT(0.99792528620), DSP32_Q_CONVERT(-0.06438263093),
    DSP32_Q_CONVERT(0.99782535041), DSP32_Q_CONVERT(-0.06591335280),
    DSP32_Q_CONVERT(0.99772306664), DSP32_Q_CONVERT(-0.06744391956),
    DSP32_Q_CONVERT(0.99761843514), DSP32_Q_CONVERT(-0.06897432763),
    DSP32_Q_CONVERT(0.99751145614), DSP32_Q_CONVERT(-0.07050457339),
    DSP32_Q_CONVERT(0.99740212990), DSP32_Q_CONVERT(-0.07203465325),
    DSP32_Q_CONVERT(0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(0.99717643674), DSP32_Q_CONVERT(-0.07509430085),
    DSP32_Q_CONVERT(0.99706007034), DSP32_Q_CONVERT(-0.07662386139),
    DSP32_Q_CONVERT(0.99694135776), DSP32_Q_CONVERT(-0.07815324163),
    DSP32_Q_CONVERT(0.99682029929), DSP32_Q_CONVERT(-0.07968243797),
    DSP32_Q_CONVERT(0.99669689520), DSP32_Q_CONVERT(-0.08121144681),
    DSP32_Q_CONVERT(0.99657114579), DSP32_Q_CONVERT(-0.08274026455),
    DSP32_Q_CONVERT(0.99644305135), DSP32_Q_CONVERT(-0.08426888759),
    DSP32_Q_CONVERT(0.99631261218), DSP32_Q_CONVERT(-0.08579731234),
    DSP32_Q_CONVERT(0.99617982860), DSP32_Q_CONVERT(-0.08732553521),
    DSP32_Q_CONVERT(0.99604470090), DSP32_Q_CONVERT(-0.08885355258),
    DSP32_Q_CONVERT(0.99590722942), DSP32_Q_CONVERT(-0.09038136088),
    DSP32_Q_CONVERT(0.99576741447), DSP32_Q_CONVERT(-0.09190895650),
    DSP32_Q_CONVERT(0.99562525638), DSP32_Q_CONVERT(-0.09343633585),
    DSP32_Q_CONVERT(0.99548075549), DSP32_Q_CONVERT(-0.09496349533),
    DSP32_Q_CONVERT(0.99533391214), DSP32_Q_CONVERT(-0.09649043136),
    DSP32_Q_CONVERT(0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(0.99503319944), DSP32_Q_CONVERT(-0.09954361866),
    DSP32_Q_CONVERT(0.99487933079), DSP32_Q_CONVERT(-0.10106986275),
    DSP32_Q_CONVERT(0.99472312110), DSP32_Q_CONVERT(-0.10259586902),
    DSP32_Q_CONVERT(0.99456457073), DSP32_Q_CONVERT(-0.10412163387),
    DSP32_Q_CONVERT(0.99440368006), DSP32_Q_CONVERT(-0.10564715371),
    DSP32_Q_CONVERT(0.99424044945), DSP32_Q_CONVERT(-0.10717242496),
    DSP32_Q_CONVERT(0.99407487930), DSP32_Q_CONVERT(-0.10869744401),
    DSP32_Q_CONVERT(0.99390697000), DSP32_Q_CONVERT(-0.11022220729),
    DSP32_Q_CONVERT(0.99373672194), DSP32_Q_CONVERT(-0.11174671121),
    DSP32_Q_CONVERT(0.99356413552), DSP32_Q_CONVERT(-0.11327095218),
    DSP32_Q_CONVERT(0.99338921115), DSP32_Q_CONVERT(-0.11479492661),
    DSP32_Q_CONVERT(0.99321194923), DSP32_Q_CONVERT(-0.11631863091),
    DSP32_Q_CONVERT(0.99303235020), DSP32_Q_CONVERT(-0.11784206151),
    DSP32_Q_CONVERT(0.99285041446), DSP32_Q_CONVERT(-0.11936521481),
    DSP32_Q_CONVERT(0.99266614245), DSP32_Q_CONVERT(-0.12088808724),
    DSP32_Q_CONVERT(0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(0.99229059135), DSP32_Q_CONVERT(-0.12393297512),
    DSP32_Q_CONVERT(0.99209931314), DSP32_Q_CONVERT(-0.12545498341),
    DSP32_Q_CONVERT(0.99190570043), DSP32_Q_CONVERT(-0.12697669650),
    DSP32_Q_CONVERT(0.99170975367), DSP32_Q_CONVERT(-0.12849811079),
    DSP32_Q_CONVERT(0.99151147332), DSP32_Q_CONVERT(-0.13001922272),
    DSP32_Q_CONVERT(0.99131085985), DSP32_Q_CONVERT(-0.13154002870),
    DSP32_Q_CONVERT(0.99110791372), DSP32_Q_CONVERT(-0.13306052516),
    DSP32_Q_CONVERT(0.99090263543), DSP32_Q_CONVERT(-0.13458070851),
    DSP32_Q_CONVERT(0.99069502544), DSP32_Q_CONVERT(-0.13610057518),
    DSP32_Q_CONVERT(0.99048508426), DSP32_Q_CONVERT(-0.13762012159),
    DSP32_Q_CONVERT(0.99027281236), DSP32_Q_CONVERT(-0.13913934416),
    DSP32_Q_CONVERT(0.99005821026), DSP32_Q_CONVERT(-0.14065823933),
    DSP32_Q_CONVERT(0.98984127846), DSP32_Q_CONVERT(-0.14217680352),
    DSP32_Q_CONVERT(0.98962201746), DSP32_Q_CONVERT(-0.14369503315),
    DSP32_Q_CONVERT(0.98940042779), DSP32_Q_CONVERT(-0.14521292465),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.98895026451), DSP32_Q_CONVERT(-0.14824767899),
    DSP32_Q_CONVERT(0.98872169196), DSP32_Q_CONVERT(-0.14976453468),
    DSP32_Q_CONVERT(0.98849079285), DSP32_Q_CONVERT(-0.15128103796),
    DSP32_Q_CONVERT(0.98825756773), DSP32_Q_CONVERT(-0.15279718526),
    DSP32_Q_CONVERT(0.98802201714), DSP32_Q_CONVERT(-0.15431297301),
    DSP32_Q_CONVERT(0.98778414164), DSP32_Q_CONVERT(-0.15582839765),
    DSP32_Q_CONVERT(0.98754394179), DSP32_Q_CONVERT(-0.15734345562),
    DSP32_Q_CONVERT(0.98730141816), DSP32_Q_CONVERT(-0.15885814333),
    DSP32_Q_CONVERT(0.98705657131), DSP32_Q_CONVERT(-0.16037245724),
    DSP32_Q_CONVERT(0.98680940181), DSP32_Q_CONVERT(-0.16188639378),
    DSP32_Q_CONVERT(0.98655991026), DSP32_Q_CONVERT(-0.16339994938),
    DSP32_Q_CONVERT(0.98630809724), DSP32_Q_CONVERT(-0.16491312049),
    DSP32_Q_CONVERT(0.98605396335), DSP32_Q_CONVERT(-0.16642590354),
    DSP32_Q_CONVERT(0.98579750917), DSP32_Q_CONVERT(-0.16793829497),
    DSP32_Q_CONVERT(0.98553873531), DSP32_Q_CONVERT(-0.16945029123),
    DSP32_Q_CONVERT(0.98527764239), DSP32_Q_CONVERT(-0.17096188876),
    DSP32_Q_CONVERT(0.98501423101), DSP32_Q_CONVERT(-0.17247308400),
    DSP32_Q_CONVERT(0.98474850180), DSP32_Q_CONVERT(-0.17398387339),
    DSP32_Q_CONVERT(0.98448045538), DSP32_Q_CONVERT(-0.17549425338),
    DSP32_Q_CONVERT(0.98421009239), DSP32_Q_CONVERT(-0.17700422041),
    DSP32_Q_CONVERT(0.98393741345), DSP32_Q_CONVERT(-0.17851377094),
    DSP32_Q_CONVERT(0.98366241921), DSP32_Q_CONVERT(-0.18002290141),
    DSP32_Q_CONVERT(0.98338511032), DSP32_Q_CONVERT(-0.18153160826),
    DSP32_Q_CONVERT(0.98310548743), DSP32_Q_CONVERT(-0.18303988796),
    DSP32_Q_CONVERT(0.98282355120), DSP32_Q_CONVERT(-0.18454773694),
    DSP32_Q_CONVERT(0.98253930229), DSP32_Q_CONVERT(-0.18605515166),
    DSP32_Q_CONVERT(0.98225274137), DSP32_Q_CONVERT(-0.18756212858),
    DSP32_Q_CONVERT(0.98196386911), DSP32_Q_CONVERT(-0.18906866415),
    DSP32_Q_CONVERT(0.98167268620), DSP32_Q_CONVERT(-0.19057475482),
    DSP32_Q_CONVERT(0.98137919331), DSP32_Q_CONVERT(-0.19208039705),
    DSP32_Q_CONVERT(0.98108339115), DSP32_Q_CONVERT(-0.19358558730),
    DSP32_Q_CONVERT(0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(0.98048486177), DSP32_Q_CONVERT(-0.19659459767),
    DSP32_Q_CONVERT(0.98018213597), DSP32_Q_CONVERT(-0.19809841072),
    DSP32_Q_CONVERT(0.97987710370), DSP32_Q_CONVERT(-0.19960175762),
    DSP32_Q_CONVERT(0.97956976569), DSP32_Q_CONVERT(-0.20110463484),
    DSP32_Q_CONVERT(0.97926012265), DSP32_Q_CONVERT(-0.20260703884),
    DSP32_Q_CONVERT(0.97894817532), DSP32_Q_CONVERT(-0.20410896609),
    DSP32_Q_CONVERT(0.97863392443), DSP32_Q_CONVERT(-0.20561041305),
    DSP32_Q_CONVERT(0.97831737072), DSP32_Q_CONVERT(-0.20711137619),
    DSP32_Q_CONVERT(0.97799851493), DSP32_Q_CONVERT(-0.20861185198),
    DSP32_Q_CONVERT(0.97767735782), DSP32_Q_CONVERT(-0.21011183688),
    DSP32_Q_CONVERT(0.97735390015), DSP32_Q_CONVERT(-0.21161132737),
    DSP32_Q_CONVERT(0.97702814266), DSP32_Q_CONVERT(-0.21311031992),
    DSP32_Q_CONVERT(0.97670008613), DSP32_Q_CONVERT(-0.21460881099),
    DSP32_Q_CONVERT(0.97636973133), DSP32_Q_CONVERT(-0.21610679708),
    DSP32_Q_CONVERT(0.97603707904), DSP32_Q_CONVERT(-0.21760427464),
    DSP32_Q_CONVERT(0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(0.97536488512), DSP32_Q_CONVERT(-0.22059769011),
    DSP32_Q_CONVERT(0.97502534507), DSP32_Q_CONVERT(-0.22209362097),
    DSP32_Q_CONVERT(0.97468351069), DSP32_Q_CONVERT(-0.22358902923),
    DSP32_Q_CONVERT(0.97433938279), DSP32_Q_CONVERT(-0.22508391136),
    DSP32_Q_CONVERT(0.97399296217), DSP32_Q_CONVERT(-0.22657826385),
    DSP32_Q_CONVERT(0.97364424965), DSP32_Q_CONVERT(-0.22807208317),
    DSP32_Q_CONVERT(0.97329324605), DSP32_Q_CONVERT(-0.22956536582),
    DSP32_Q_CONVERT(0.97293995221), DSP32_Q_CONVERT(-0.23105810828),
    DSP32_Q_CONVERT(0.97258436893), DSP32_Q_CONVERT(-0.23255030704),
    DSP32_Q_CONVERT(0.97222649708), DSP32_Q_CONVERT(-0.23404195858),
    DSP32_Q_CONVERT(0.97186633748), DSP32_Q_CONVERT(-0.23553305940),
    DSP32_Q_CONVERT(0.97150389099), DSP32_Q_CONVERT(-0.23702360599),
    DSP32_Q_CONVERT(0.97113915845), DSP32_Q_CONVERT(-0.23851359484),
    DSP32_Q_CONVERT(0.97077214073), DSP32_Q_CONVERT(-0.24000302245),
    DSP32_Q_CONVERT(0.97040283869), DSP32_Q_CONVERT(-0.24149188530),
    DSP32_Q_CONVERT(0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(0.96965738512), DSP32_Q_CONVERT(-0.24446790275),
    DSP32_Q_CONVERT(0.96928123536), DSP32_Q_CONVERT(-0.24595505034),
    DSP32_Q_CONVERT(0.96890280478), DSP32_Q_CONVERT(-0.24744161917),
    DSP32_Q_CONVERT(0.96852209427), DSP32_Q_CONVERT(-0.24892760575),
    DSP32_Q_CONVERT(0.96813910475), DSP32_Q_CONVERT(-0.25041300657),
    DSP32_Q_CONVERT(0.96775383709), DSP32_Q_CONVERT(-0.25189781815),
    DSP32_Q_CONVERT(0.96736629222), DSP32_Q_CONVERT(-0.25338203700),
    DSP32_Q_CONVERT(0.96697647104), DSP32_Q_CONVERT(-0.25486565960),
    DSP32_Q_CONVERT(0.96658437448), DSP32_Q_CONVERT(-0.25634868249),
    DSP32_Q_CONVERT(0.96619000345), DSP32_Q_CONVERT(-0.25783110216),
    DSP32_Q_CONVERT(0.96579335887), DSP32_Q_CONVERT(-0.25931291513),
    DSP32_Q_CONVERT(0.96539444170), DSP32_Q_CONVERT(-0.26079411792),
    DSP32_Q_CONVERT(0.96499325285), DSP32_Q_CONVERT(-0.26227470702),
    DSP32_Q_CONVERT(0.96458979329), DSP32_Q_CONVERT(-0.26375467897),
    DSP32_Q_CONVERT(0.96418406395), DSP32_Q_CONVERT(-0.26523403029),
    DSP32_Q_CONVERT(0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(0.96336579978), DSP32_Q_CONVERT(-0.26819085706),
    DSP32_Q_CONVERT(0.96295326687), DSP32_Q_CONVERT(-0.26966832557),
    DSP32_Q_CONVERT(0.96253846804), DSP32_Q_CONVERT(-0.27114515953),
    DSP32_Q_CONVERT(0.96212140427), DSP32_Q_CONVERT(-0.27262135545),
    DSP32_Q_CONVERT(0.96170207653), DSP32_Q_CONVERT(-0.27409690987),
    DSP32_Q_CONVERT(0.96128048581), DSP32_Q_CONVERT(-0.27557181931),
    DSP32_Q_CONVERT(0.96085663311), DSP32_Q_CONVERT(-0.27704608031),
    DSP32_Q_CONVERT(0.96043051942), DSP32_Q_CONVERT(-0.27851968939),
    DSP32_Q_CONVERT(0.96000214574), DSP32_Q_CONVERT(-0.27999264308),
    DSP32_Q_CONVERT(0.95957151308), DSP32_Q_CONVERT(-0.28146493793),
    DSP32_Q_CONVERT(0.95913862246), DSP32_Q_CONVERT(-0.28293657046),
    DSP32_Q_CONVERT(0.95870347490), DSP32_Q_CONVERT(-0.28440753721),
    DSP32_Q_CONVERT(0.95826607141), DSP32_Q_CONVERT(-0.28587783473),
    DSP32_Q_CONVERT(0.95782641303), DSP32_Q_CONVERT(-0.28734745954),
    DSP32_Q_CONVERT(0.95738450079), DSP32_Q_CONVERT(-0.28881640821),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.95649391890), DSP32_Q_CONVERT(-0.29175226323),
    DSP32_Q_CONVERT(0.95604525135), DSP32_Q_CONVERT(-0.29321916269),
    DSP32_Q_CONVERT(0.95559433413), DSP32_Q_CONVERT(-0.29468537218),
    DSP32_Q_CONVERT(0.95514116831), DSP32_Q_CONVERT(-0.29615088824),
    DSP32_Q_CONVERT(0.95468575494), DSP32_Q_CONVERT(-0.29761570744),
    DSP32_Q_CONVERT(0.95422809511), DSP32_Q_CONVERT(-0.29907982631),
    DSP32_Q_CONVERT(0.95376818989), DSP32_Q_CONVERT(-0.30054324142),
    DSP32_Q_CONVERT(0.95330604035), DSP32_Q_CONVERT(-0.30200594932),
    DSP32_Q_CONVERT(0.95284164760), DSP32_Q_CONVERT(-0.30346794657),
    DSP32_Q_CONVERT(0.95237501272), DSP32_Q_CONVERT(-0.30492922974),
    DSP32_Q_CONVERT(0.95190613681), DSP32_Q_CONVERT(-0.30638979537),
    DSP32_Q_CONVERT(0.95143502097), DSP32_Q_CONVERT(-0.30784964004),
    DSP32_Q_CONVERT(0.95096166631), DSP32_Q_CONVERT(-0.30930876031),
    DSP32_Q_CONVERT(0.95048607395), DSP32_Q_CONVERT(-0.31076715275),
    DSP32_Q_CONVERT(0.95000824500), DSP32_Q_CONVERT(-0.31222481392),
    DSP32_Q_CONVERT(0.94952818059), DSP32_Q_CONVERT(-0.31368174040),
    DSP32_Q_CONVERT(0.94904588185), DSP32_Q_CONVERT(-0.31513792875),
    DSP32_Q_CONVERT(0.94856134992), DSP32_Q_CONVERT(-0.31659337556),
    DSP32_Q_CONVERT(0.94807458592), DSP32_Q_CONVERT(-0.31804807739),
    DSP32_Q_CONVERT(0.94758559102), DSP32_Q_CONVERT(-0.31950203082),
    DSP32_Q_CONVERT(0.94709436635), DSP32_Q_CONVERT(-0.32095523243),
    DSP32_Q_CONVERT(0.94660091308), DSP32_Q_CONVERT(-0.32240767880),
    DSP32_Q_CONVERT(0.94610523237), DSP32_Q_CONVERT(-0.32385936652),
    DSP32_Q_CONVERT(0.94560732538), DSP32_Q_CONVERT(-0.32531029216),
    DSP32_Q_CONVERT(0.94510719329), DSP32_Q_CONVERT(-0.32676045232),
    DSP32_Q_CONVERT(0.94460483726), DSP32_Q_CONVERT(-0.32820984358),
    DSP32_Q_CONVERT(0.94410025849), DSP32_Q_CONVERT(-0.32965846253),
    DSP32_Q_CONVERT(0.94359345816), DSP32_Q_CONVERT(-0.33110630576),
    DSP32_Q_CONVERT(0.94308443747), DSP32_Q_CONVERT(-0.33255336987),
    DSP32_Q_CONVERT(0.94257319760), DSP32_Q_CONVERT(-0.33399965144),
    DSP32_Q_CONVERT(0.94205973977), DSP32_Q_CONVERT(-0.33544514708),
    DSP32_Q_CONVERT(0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(0.94102617505), DSP32_Q_CONVERT(-0.33833376697),
    DSP32_Q_CONVERT(0.94050607059), DSP32_Q_CONVERT(-0.33977688441),
    DSP32_Q_CONVERT(0.93998375303), DSP32_Q_CONVERT(-0.34121920232),
    DSP32_Q_CONVERT(0.93945922360), DSP32_Q_CONVERT(-0.34266071731),
    DSP32_Q_CONVERT(0.93893248353), DSP32_Q_CONVERT(-0.34410142599),
    DSP32_Q_CONVERT(0.93840353406), DSP32_Q_CONVERT(-0.34554132496),
    DSP32_Q_CONVERT(0.93787237644), DSP32_Q_CONVERT(-0.34698041085),
    DSP32_Q_CONVERT(0.93733901191), DSP32_Q_CONVERT(-0.34841868025),
    DSP32_Q_CONVERT(0.93680344174), DSP32_Q_CONVERT(-0.34985612979),
    DSP32_Q_CONVERT(0.93626566717), DSP32_Q_CONVERT(-0.35129275609),
    DSP32_Q_CONVERT(0.93572568948), DSP32_Q_CONVERT(-0.35272855576),
    DSP32_Q_CONVERT(0.93518350994), DSP32_Q_CONVERT(-0.35416352542),
    DSP32_Q_CONVERT(0.93463912982), DSP32_Q_CONVERT(-0.35559766170),
    DSP32_Q_CONVERT(0.93409255040), DSP32_Q_CONVERT(-0.35703096123),
    DSP32_Q_CONVERT(0.93354377298), DSP32_Q_CONVERT(-0.35846342063),
    DSP32_Q_CONVERT(0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(0.93243962927), DSP32_Q_CONVERT(-0.36132580557),
    DSP32_Q_CONVERT(0.93188426558), DSP32_Q_CONVERT(-0.36275572437),
    DSP32_Q_CONVERT(0.93132670908), DSP32_Q_CONVERT(-0.36418478957),
    DSP32_Q_CONVERT(0.93076696108), DSP32_Q_CONVERT(-0.36561299780),
    DSP32_Q_CONVERT(0.93020502289), DSP32_Q_CONVERT(-0.36704034572),
    DSP32_Q_CONVERT(0.92964089584), DSP32_Q_CONVERT(-0.36846682995),
    DSP32_Q_CONVERT(0.92907458126), DSP32_Q_CONVERT(-0.36989244715),
    DSP32_Q_CONVERT(0.92850608047), DSP32_Q_CONVERT(-0.37131719395),
    DSP32_Q_CONVERT(0.92793539482), DSP32_Q_CONVERT(-0.37274106701),
    DSP32_Q_CONVERT(0.92736252565), DSP32_Q_CONVERT(-0.37416406297),
    DSP32_Q_CONVERT(0.92678747430), DSP32_Q_CONVERT(-0.37558617849),
    DSP32_Q_CONVERT(0.92621024214), DSP32_Q_CONVERT(-0.37700741022),
    DSP32_Q_CONVERT(0.92563083051), DSP32_Q_CONVERT(-0.37842775481),
    DSP32_Q_CONVERT(0.92504924078), DSP32_Q_CONVERT(-0.37984720892),
    DSP32_Q_CONVERT(0.92446547433), DSP32_Q_CONVERT(-0.38126576922),
    DSP32_Q_CONVERT(0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(0.92329141672), DSP32_Q_CONVERT(-0.38410019502),
    DSP32_Q_CONVERT(0.92270112833), DSP32_Q_CONVERT(-0.38551605384),
    DSP32_Q_CONVERT(0.92210866874), DSP32_Q_CONVERT(-0.38693100551),
    DSP32_Q_CONVERT(0.92151403934), DSP32_Q_CONVERT(-0.38834504670),
    DSP32_Q_CONVERT(0.92091724153), DSP32_Q_CONVERT(-0.38975817407),
    DSP32_Q_CONVERT(0.92031827671), DSP32_Q_CONVERT(-0.39117038430),
    DSP32_Q_CONVERT(0.91971714629), DSP32_Q_CONVERT(-0.39258167407),
    DSP32_Q_CONVERT(0.91911385169), DSP32_Q_CONVERT(-0.39399204006),
    DSP32_Q_CONVERT(0.91850839433), DSP32_Q_CONVERT(-0.39540147895),
    DSP32_Q_CONVERT(0.91790077562), DSP32_Q_CONVERT(-0.39680998742),
    DSP32_Q_CONVERT(0.91729099701), DSP32_Q_CONVERT(-0.39821756215),
    DSP32_Q_CONVERT(0.91667905992), DSP32_Q_CONVERT(-0.39962419985),
    DSP32_Q_CONVERT(0.91606496580), DSP32_Q_CONVERT(-0.40102989718),
    DSP32_Q_CONVERT(0.91544871609), DSP32_Q_CONVERT(-0.40243465086),
    DSP32_Q_CONVERT(0.91483031224), DSP32_Q_CONVERT(-0.40383845757),
    DSP32_Q_CONVERT(0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(0.91358704795), DSP32_Q_CONVERT(-0.40664321687),
    DSP32_Q_CONVERT(0.91296219043), DSP32_Q_CONVERT(-0.40804416286),
    DSP32_Q_CONVERT(0.91233518462), DSP32_Q_CONVERT(-0.40944414869),
    DSP32_Q_CONVERT(0.91170603201), DSP32_Q_CONVERT(-0.41084317106),
    DSP32_Q_CONVERT(0.91107473406), DSP32_Q_CONVERT(-0.41224122667),
    DSP32_Q_CONVERT(0.91044129226), DSP32_Q_CONVERT(-0.41363831224),
    DSP32_Q_CONVERT(0.90980570810), DSP32_Q_CONVERT(-0.41503442448),
    DSP32_Q_CONVERT(0.90916798309), DSP32_Q_CONVERT(-0.41642956010),
    DSP32_Q_CONVERT(0.90852811872), DSP32_Q_CONVERT(-0.41782371582),
    DSP32_Q_CONVERT(0.90788611649), DSP32_Q_CONVERT(-0.41921688836),
    DSP32_Q_CONVERT(0.90724197792), DSP32_Q_CONVERT(-0.42060907445),
    DSP32_Q_CONVERT(0.90659570451), DSP32_Q_CONVERT(-0.42200027080),
    DSP32_Q_CONVERT(0.90594729781), DSP32_Q_CONVERT(-0.42339047414),
    DSP32_Q_CONVERT(0.90529675932), DSP32_Q_CONVERT(-0.42477968121),
    DSP32_Q_CONVERT(0.90464409058), DSP32_Q_CONVERT(-0.42616788873),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.90333236849), DSP32_Q_CONVERT(-0.42894129206),
    DSP32_Q_CONVERT(0.90267331824), DSP32_Q_CONVERT(-0.43032648134),
    DSP32_Q_CONVERT(0.90201214390), DSP32_Q_CONVERT(-0.43171065803),
    DSP32_Q_CONVERT(0.90134884705), DSP32_Q_CONVERT(-0.43309381885),
    DSP32_Q_CONVERT(0.90068342923), DSP32_Q_CONVERT(-0.43447596057),
    DSP32_Q_CONVERT(0.90001589202), DSP32_Q_CONVERT(-0.43585707992),
    DSP32_Q_CONVERT(0.89934623698), DSP32_Q_CONVERT(-0.43723717366),
    DSP32_Q_CONVERT(0.89867446569), DSP32_Q_CONVERT(-0.43861623854),
    DSP32_Q_CONVERT(0.89800057974), DSP32_Q_CONVERT(-0.43999427131),
    DSP32_Q_CONVERT(0.89732458071), DSP32_Q_CONVERT(-0.44137126873),
    DSP32_Q_CONVERT(0.89664647018), DSP32_Q_CONVERT(-0.44274722756),
    DSP32_Q_CONVERT(0.89596624976), DSP32_Q_CONVERT(-0.44412214457),
    DSP32_Q_CONVERT(0.89528392104), DSP32_Q_CONVERT(-0.44549601651),
    DSP32_Q_CONVERT(0.89459948563), DSP32_Q_CONVERT(-0.44686884016),
    DSP32_Q_CONVERT(0.89391294515), DSP32_Q_CONVERT(-0.44824061229),
    DSP32_Q_CONVERT(0.89322430120), DSP32_Q_CONVERT(-0.44961132965),
    DSP32_Q_CONVERT(0.89253355540), DSP32_Q_CONVERT(-0.45098098905),
    DSP32_Q_CONVERT(0.89184070939), DSP32_Q_CONVERT(-0.45234958723),
    DSP32_Q_CONVERT(0.89114576479), DSP32_Q_CONVERT(-0.45371712100),
    DSP32_Q_CONVERT(0.89044872324), DSP32_Q_CONVERT(-0.45508358713),
    DSP32_Q_CONVERT(0.88974958638), DSP32_Q_CONVERT(-0.45644898240),
    DSP32_Q_CONVERT(0.88904835585), DSP32_Q_CONVERT(-0.45781330360),
    DSP32_Q_CONVERT(0.88834503331), DSP32_Q_CONVERT(-0.45917654752),
    DSP32_Q_CONVERT(0.88763962040), DSP32_Q_CONVERT(-0.46053871096),
    DSP32_Q_CONVERT(0.88693211879), DSP32_Q_CONVERT(-0.46189979070),
    DSP32_Q_CONVERT(0.88622253015), DSP32_Q_CONVERT(-0.46325978355),
    DSP32_Q_CONVERT(0.88551085614), DSP32_Q_CONVERT(-0.46461868631),
    DSP32_Q_CONVERT(0.88479709843), DSP32_Q_CONVERT(-0.46597649577),
    DSP32_Q_CONVERT(0.88408125871), DSP32_Q_CONVERT(-0.46733320874),
    DSP32_Q_CONVERT(0.88336333867), DSP32_Q_CONVERT(-0.46868882204),
    DSP32_Q_CONVERT(0.88264333998), DSP32_Q_CONVERT(-0.47004333246),
    DSP32_Q_CONVERT(0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(0.88119711347), DSP32_Q_CONVERT(-0.47274903195),
    DSP32_Q_CONVERT(0.88047088905), DSP32_Q_CONVERT(-0.47410021465),
    DSP32_Q_CONVERT(0.87974259280), DSP32_Q_CONVERT(-0.47545028175),
    DSP32_Q_CONVERT(0.87901222643), DSP32_Q_CONVERT(-0.47679923006),
    DSP32_Q_CONVERT(0.87827979166), DSP32_Q_CONVERT(-0.47814705642),
    DSP32_Q_CONVERT(0.87754529021), DSP32_Q_CONVERT(-0.47949375766),
    DSP32_Q_CONVERT(0.87680872381), DSP32_Q_CONVERT(-0.48083933060),
    DSP32_Q_CONVERT(0.87607009420), DSP32_Q_CONVERT(-0.48218377208),
    DSP32_Q_CONVERT(0.87532940310), DSP32_Q_CONVERT(-0.48352707893),
    DSP32_Q_CONVERT(0.87458665228), DSP32_Q_CONVERT(-0.48486924800),
    DSP32_Q_CONVERT(0.87384184347), DSP32_Q_CONVERT(-0.48621027612),
    DSP32_Q_CONVERT(0.87309497842), DSP32_Q_CONVERT(-0.48755016015),
    DSP32_Q_CONVERT(0.87234605889), DSP32_Q_CONVERT(-0.48888889692),
    DSP32_Q_CONVERT(0.87159508666), DSP32_Q_CONVERT(-0.49022648329),
    DSP32_Q_CONVERT(0.87084206347), DSP32_Q_CONVERT(-0.49156291611),
    DSP32_Q_CONVERT(0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(0.86932987135), DSP32_Q_CONVERT(-0.49423230852),
    DSP32_Q_CONVERT(0.86857070597), DSP32_Q_CONVERT(-0.49556526183),
    DSP32_Q_CONVERT(0.86780949676), DSP32_Q_CONVERT(-0.49689704902),
    DSP32_Q_CONVERT(0.86704624552), DSP32_Q_CONVERT(-0.49822766697),
    DSP32_Q_CONVERT(0.86628095402), DSP32_Q_CONVERT(-0.49955711255),
    DSP32_Q_CONVERT(0.86551362409), DSP32_Q_CONVERT(-0.50088538261),
    DSP32_Q_CONVERT(0.86474425752), DSP32_Q_CONVERT(-0.50221247405),
    DSP32_Q_CONVERT(0.86397285612), DSP32_Q_CONVERT(-0.50353838373),
    DSP32_Q_CONVERT(0.86319942171), DSP32_Q_CONVERT(-0.50486310853),
    DSP32_Q_CONVERT(0.86242395611), DSP32_Q_CONVERT(-0.50618664535),
    DSP32_Q_CONVERT(0.86164646114), DSP32_Q_CONVERT(-0.50750899105),
    DSP32_Q_CONVERT(0.86086693864), DSP32_Q_CONVERT(-0.50883014254),
    DSP32_Q_CONVERT(0.86008539043), DSP32_Q_CONVERT(-0.51015009671),
    DSP32_Q_CONVERT(0.85930181836), DSP32_Q_CONVERT(-0.51146885044),
    DSP32_Q_CONVERT(0.85851622426), DSP32_Q_CONVERT(-0.51278640063),
    DSP32_Q_CONVERT(0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(0.85693897742), DSP32_Q_CONVERT(-0.51541787802),
    DSP32_Q_CONVERT(0.85614732838), DSP32_Q_CONVERT(-0.51673179902),
    DSP32_Q_CONVERT(0.85535366474), DSP32_Q_CONVERT(-0.51804450410),
    DSP32_Q_CONVERT(0.85455798837), DSP32_Q_CONVERT(-0.51935599017),
    DSP32_Q_CONVERT(0.85376030114), DSP32_Q_CONVERT(-0.52066625414),
    DSP32_Q_CONVERT(0.85296060493), DSP32_Q_CONVERT(-0.52197529294),
    DSP32_Q_CONVERT(0.85215890162), DSP32_Q_CONVERT(-0.52328310348),
    DSP32_Q_CONVERT(0.85135519311), DSP32_Q_CONVERT(-0.52458968268),
    DSP32_Q_CONVERT(0.85054948127), DSP32_Q_CONVERT(-0.52589502747),
    DSP32_Q_CONVERT(0.84974176800), DSP32_Q_CONVERT(-0.52719913478),
    DSP32_Q_CONVERT(0.84893205521), DSP32_Q_CONVERT(-0.52850200154),
    DSP32_Q_CONVERT(0.84812034480), DSP32_Q_CONVERT(-0.52980362469),
    DSP32_Q_CONVERT(0.84730663869), DSP32_Q_CONVERT(-0.53110400115),
    DSP32_Q_CONVERT(0.84649093877), DSP32_Q_CONVERT(-0.53240312788),
    DSP32_Q_CONVERT(0.84567324699), DSP32_Q_CONVERT(-0.53370100181),
    DSP32_Q_CONVERT(0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(0.84403189549), DSP32_Q_CONVERT(-0.53629297907),
    DSP32_Q_CONVERT(0.84320823964), DSP32_Q_CONVERT(-0.53758707630),
    DSP32_Q_CONVERT(0.84238259964), DSP32_Q_CONVERT(-0.53887990853),
    DSP32_Q_CONVERT(0.84155497744), DSP32_Q_CONVERT(-0.54017147273),
    DSP32_Q_CONVERT(0.84072537497), DSP32_Q_CONVERT(-0.54146176585),
    DSP32_Q_CONVERT(0.83989379420), DSP32_Q_CONVERT(-0.54275078486),
    DSP32_Q_CONVERT(0.83906023707), DSP32_Q_CONVERT(-0.54403852673),
    DSP32_Q_CONVERT(0.83822470555), DSP32_Q_CONVERT(-0.54532498842),
    DSP32_Q_CONVERT(0.83738720162), DSP32_Q_CONVERT(-0.54661016691),
    DSP32_Q_CONVERT(0.83654772722), DSP32_Q_CONVERT(-0.54789405917),
    DSP32_Q_CONVERT(0.83570628435), DSP32_Q_CONVERT(-0.54917666219),
    DSP32_Q_CONVERT(0.83486287499), DSP32_Q_CONVERT(-0.55045797294),
    DSP32_Q_CONVERT(0.83401750111), DSP32_Q_CONVERT(-0.55173798840),
    DSP32_Q_CONVERT(0.83317016470), DSP32_Q_CONVERT(-0.55301670558),
    DSP32_Q_CONVERT(0.83232086777), DSP32_Q_CONVERT(-0.55429412145),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.83061640031), DSP32_Q_CONVERT(-0.55684503728),
    DSP32_Q_CONVERT(0.82976123379), DSP32_Q_CONVERT(-0.55811853122),
    DSP32_Q_CONVERT(0.82890411477), DSP32_Q_CONVERT(-0.55939071186),
    DSP32_Q_CONVERT(0.82804504526), DSP32_Q_CONVERT(-0.56066157620),
    DSP32_Q_CONVERT(0.82718402727), DSP32_Q_CONVERT(-0.56193112124),
    DSP32_Q_CONVERT(0.82632106285), DSP32_Q_CONVERT(-0.56319934401),
    DSP32_Q_CONVERT(0.82545615400), DSP32_Q_CONVERT(-0.56446624152),
    DSP32_Q_CONVERT(0.82458930279), DSP32_Q_CONVERT(-0.56573181078),
    DSP32_Q_CONVERT(0.82372051123), DSP32_Q_CONVERT(-0.56699604883),
    DSP32_Q_CONVERT(0.82284978138), DSP32_Q_CONVERT(-0.56825895267),
    DSP32_Q_CONVERT(0.82197711528), DSP32_Q_CONVERT(-0.56952051935),
    DSP32_Q_CONVERT(0.82110251499), DSP32_Q_CONVERT(-0.57078074589),
    DSP32_Q_CONVERT(0.82022598257), DSP32_Q_CONVERT(-0.57203962932),
    DSP32_Q_CONVERT(0.81934752008), DSP32_Q_CONVERT(-0.57329716670),
    DSP32_Q_CONVERT(0.81846712958), DSP32_Q_CONVERT(-0.57455335505),
    DSP32_Q_CONVERT(0.81758481315), DSP32_Q_CONVERT(-0.57580819142),
    DSP32_Q_CONVERT(0.81670057287), DSP32_Q_CONVERT(-0.57706167286),
    DSP32_Q_CONVERT(0.81581441081), DSP32_Q_CONVERT(-0.57831379641),
    DSP32_Q_CONVERT(0.81492632906), DSP32_Q_CONVERT(-0.57956455914),
    DSP32_Q_CONVERT(0.81403632971), DSP32_Q_CONVERT(-0.58081395810),
    DSP32_Q_CONVERT(0.81314441485), DSP32_Q_CONVERT(-0.58206199034),
    DSP32_Q_CONVERT(0.81225058659), DSP32_Q_CONVERT(-0.58330865294),
    DSP32_Q_CONVERT(0.81135484702), DSP32_Q_CONVERT(-0.58455394295),
    DSP32_Q_CONVERT(0.81045719825), DSP32_Q_CONVERT(-0.58579785746),
    DSP32_Q_CONVERT(0.80955764240), DSP32_Q_CONVERT(-0.58704039352),
    DSP32_Q_CONVERT(0.80865618159), DSP32_Q_CONVERT(-0.58828154822),
    DSP32_Q_CONVERT(0.80775281793), DSP32_Q_CONVERT(-0.58952131864),
    DSP32_Q_CONVERT(0.80684755354), DSP32_Q_CONVERT(-0.59075970186),
    DSP32_Q_CONVERT(0.80594039057), DSP32_Q_CONVERT(-0.59199669496),
    DSP32_Q_CONVERT(0.80503133114), DSP32_Q_CONVERT(-0.59323229504),
    DSP32_Q_CONVERT(0.80412037740), DSP32_Q_CONVERT(-0.59446649918),
    DSP32_Q_CONVERT(0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(0.80229279554), DSP32_Q_CONVERT(-0.59693070806),
    DSP32_Q_CONVERT(0.80137617172), DSP32_Q_CONVERT(-0.59816070700),
    DSP32_Q_CONVERT(0.80045766219), DSP32_Q_CONVERT(-0.59938929840),
    DSP32_Q_CONVERT(0.79953726911), DSP32_Q_CONVERT(-0.60061647938),
    DSP32_Q_CONVERT(0.79861499463), DSP32_Q_CONVERT(-0.60184224706),
    DSP32_Q_CONVERT(0.79769084094), DSP32_Q_CONVERT(-0.60306659854),
    DSP32_Q_CONVERT(0.79676481021), DSP32_Q_CONVERT(-0.60428953095),
    DSP32_Q_CONVERT(0.79583690461), DSP32_Q_CONVERT(-0.60551104140),
    DSP32_Q_CONVERT(0.79490712633), DSP32_Q_CONVERT(-0.60673112703),
    DSP32_Q_CONVERT(0.79397547755), DSP32_Q_CONVERT(-0.60794978497),
    DSP32_Q_CONVERT(0.79304196048), DSP32_Q_CONVERT(-0.60916701234),
    DSP32_Q_CONVERT(0.79210657730), DSP32_Q_CONVERT(-0.61038280628),
    DSP32_Q_CONVERT(0.79116933022), DSP32_Q_CONVERT(-0.61159716393),
    DSP32_Q_CONVERT(0.79023022144), DSP32_Q_CONVERT(-0.61281008243),
    DSP32_Q_CONVERT(0.78928925317), DSP32_Q_CONVERT(-0.61402155893),
    DSP32_Q_CONVERT(0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(0.78740174703), DSP32_Q_CONVERT(-0.61644017453),
    DSP32_Q_CONVERT(0.78645521360), DSP32_Q_CONVERT(-0.61764730794),
    DSP32_Q_CONVERT(0.78550682956), DSP32_Q_CONVERT(-0.61885298796),
    DSP32_Q_CONVERT(0.78455659716), DSP32_Q_CONVERT(-0.62005721176),
    DSP32_Q_CONVERT(0.78360451861), DSP32_Q_CONVERT(-0.62125997651),
    DSP32_Q_CONVERT(0.78265059617), DSP32_Q_CONVERT(-0.62246127937),
    DSP32_Q_CONVERT(0.78169483207), DSP32_Q_CONVERT(-0.62366111753),
    DSP32_Q_CONVERT(0.78073722857), DSP32_Q_CONVERT(-0.62485948814),
    DSP32_Q_CONVERT(0.77977778792), DSP32_Q_CONVERT(-0.62605638840),
    DSP32_Q_CONVERT(0.77881651238), DSP32_Q_CONVERT(-0.62725181550),
    DSP32_Q_CONVERT(0.77785340421), DSP32_Q_CONVERT(-0.62844576660),
    DSP32_Q_CONVERT(0.77688846567), DSP32_Q_CONVERT(-0.62963823891),
    DSP32_Q_CONVERT(0.77592169904), DSP32_Q_CONVERT(-0.63082922963),
    DSP32_Q_CONVERT(0.77495310659), DSP32_Q_CONVERT(-0.63201873594),
    DSP32_Q_CONVERT(0.77398269061), DSP32_Q_CONVERT(-0.63320675505),
    DSP32_Q_CONVERT(0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(0.77203639715), DSP32_Q_CONVERT(-0.63557832049),
    DSP32_Q_CONVERT(0.77106052426), DSP32_Q_CONVERT(-0.63676186124),
    DSP32_Q_CONVERT(0.77008283699), DSP32_Q_CONVERT(-0.63794390362),
    DSP32_Q_CONVERT(0.76910333765), DSP32_Q_CONVERT(-0.63912444486),
    DSP32_Q_CONVERT(0.76812202852), DSP32_Q_CONVERT(-0.64030348218),
    DSP32_Q_CONVERT(0.76713891194), DSP32_Q_CONVERT(-0.64148101281),
    DSP32_Q_CONVERT(0.76615399020), DSP32_Q_CONVERT(-0.64265703397),
    DSP32_Q_CONVERT(0.76516726562), DSP32_Q_CONVERT(-0.64383154289),
    DSP32_Q_CONVERT(0.76417874054), DSP32_Q_CONVERT(-0.64500453682),
    DSP32_Q_CONVERT(0.76318841726), DSP32_Q_CONVERT(-0.64617601298),
    DSP32_Q_CONVERT(0.76219629813), DSP32_Q_CONVERT(-0.64734596864),
    DSP32_Q_CONVERT(0.76120238548), DSP32_Q_CONVERT(-0.64851440102),
    DSP32_Q_CONVERT(0.76020668165), DSP32_Q_CONVERT(-0.64968130739),
    DSP32_Q_CONVERT(0.75920918898), DSP32_Q_CONVERT(-0.65084668500),
    DSP32_Q_CONVERT(0.75820990981), DSP32_Q_CONVERT(-0.65201053110),
    DSP32_Q_CONVERT(0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(0.75620600141), DSP32_Q_CONVERT(-0.65433361783),
    DSP32_Q_CONVERT(0.75520137690), DSP32_Q_CONVERT(-0.65549285300),
    DSP32_Q_CONVERT(0.75419497532), DSP32_Q_CONVERT(-0.65665054573),
    DSP32_Q_CONVERT(0.75318679904), DSP32_Q_CONVERT(-0.65780669330),
    DSP32_Q_CONVERT(0.75217685045), DSP32_Q_CONVERT(-0.65896129298),
    DSP32_Q_CONVERT(0.75116513191), DSP32_Q_CONVERT(-0.66011434207),
    DSP32_Q_CONVERT(0.75015164581), DSP32_Q_CONVERT(-0.66126583784),
    DSP32_Q_CONVERT(0.74913639452), DSP32_Q_CONVERT(-0.66241577759),
    DSP32_Q_CONVERT(0.74811938045), DSP32_Q_CONVERT(-0.66356415861),
    DSP32_Q_CONVERT(0.74710060598), DSP32_Q_CONVERT(-0.66471097820),
    DSP32_Q_CONVERT(0.74608007351), DSP32_Q_CONVERT(-0.66585623367),
    DSP32_Q_CONVERT(0.74505778544), DSP32_Q_CONVERT(-0.66699992230),
    DSP32_Q_CONVERT(0.74403374418), DSP32_Q_CONVERT(-0.66814204143),
    DSP32_Q_CONVERT(0.74300795214), DSP32_Q_CONVERT(-0.66928258835),
    DSP32_Q_CONVERT(0.74198041172), DSP32_Q_CONVERT(-0.67042156038),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.73992009546), DSP32_Q_CONVERT(-0.67269476907),
    DSP32_Q_CONVERT(0.73888732446), DSP32_Q_CONVERT(-0.67382900038),
    DSP32_Q_CONVERT(0.73785281479), DSP32_Q_CONVERT(-0.67496164610),
    DSP32_Q_CONVERT(0.73681656888), DSP32_Q_CONVERT(-0.67609270358),
    DSP32_Q_CONVERT(0.73577858917), DSP32_Q_CONVERT(-0.67722217014),
    DSP32_Q_CONVERT(0.73473887810), DSP32_Q_CONVERT(-0.67835004313),
    DSP32_Q_CONVERT(0.73369743811), DSP32_Q_CONVERT(-0.67947631990),
    DSP32_Q_CONVERT(0.73265427167), DSP32_Q_CONVERT(-0.68060099780),
    DSP32_Q_CONVERT(0.73160938122), DSP32_Q_CONVERT(-0.68172407417),
    DSP32_Q_CONVERT(0.73056276923), DSP32_Q_CONVERT(-0.68284554639),
    DSP32_Q_CONVERT(0.72951443815), DSP32_Q_CONVERT(-0.68396541180),
    DSP32_Q_CONVERT(0.72846439045), DSP32_Q_CONVERT(-0.68508366777),
    DSP32_Q_CONVERT(0.72741262860), DSP32_Q_CONVERT(-0.68620031168),
    DSP32_Q_CONVERT(0.72635915508), DSP32_Q_CONVERT(-0.68731534089),
    DSP32_Q_CONVERT(0.72530397237), DSP32_Q_CONVERT(-0.68842875278),
    DSP32_Q_CONVERT(0.72424708295), DSP32_Q_CONVERT(-0.68954054474),
    DSP32_Q_CONVERT(0.72318848931), DSP32_Q_CONVERT(-0.69065071413),
    DSP32_Q_CONVERT(0.72212819393), DSP32_Q_CONVERT(-0.69175925836),
    DSP32_Q_CONVERT(0.72106619931), DSP32_Q_CONVERT(-0.69286617482),
    DSP32_Q_CONVERT(0.72000250796), DSP32_Q_CONVERT(-0.69397146089),
    DSP32_Q_CONVERT(0.71893712237), DSP32_Q_CONVERT(-0.69507511398),
    DSP32_Q_CONVERT(0.71787004506), DSP32_Q_CONVERT(-0.69617713149),
    DSP32_Q_CONVERT(0.71680127852), DSP32_Q_CONVERT(-0.69727751083),
    DSP32_Q_CONVERT(0.71573082528), DSP32_Q_CONVERT(-0.69837624941),
    DSP32_Q_CONVERT(0.71465868786), DSP32_Q_CONVERT(-0.69947334464),
    DSP32_Q_CONVERT(0.71358486878), DSP32_Q_CONVERT(-0.70056879394),
    DSP32_Q_CONVERT(0.71250937056), DSP32_Q_CONVERT(-0.70166259474),
    DSP32_Q_CONVERT(0.71143219575), DSP32_Q_CONVERT(-0.70275474446),
    DSP32_Q_CONVERT(0.71035334686), DSP32_Q_CONVERT(-0.70384524052),
    DSP32_Q_CONVERT(0.70927282644), DSP32_Q_CONVERT(-0.70493408038),
    DSP32_Q_CONVERT(0.70819063703), DSP32_Q_CONVERT(-0.70602126145),
    DSP32_Q_CONVERT(0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(0.70602126145), DSP32_Q_CONVERT(-0.70819063703),
    DSP32_Q_CONVERT(0.70493408038), DSP32_Q_CONVERT(-0.70927282644),
    DSP32_Q_CONVERT(0.70384524052), DSP32_Q_CONVERT(-0.71035334686),
    DSP32_Q_CONVERT(0.70275474446), DSP32_Q_CONVERT(-0.71143219575),
    DSP32_Q_CONVERT(0.70166259474), DSP32_Q_CONVERT(-0.71250937056),
    DSP32_Q_CONVERT(0.70056879394), DSP32_Q_CONVERT(-0.71358486878),
    DSP32_Q_CONVERT(0.69947334464), DSP32_Q_CONVERT(-0.71465868786),
    DSP32_Q_CONVERT(0.69837624941), DSP32_Q_CONVERT(-0.71573082528),
    DSP32_Q_CONVERT(0.69727751083), DSP32_Q_CONVERT(-0.71680127852),
    DSP32_Q_CONVERT(0.69617713149), DSP32_Q_CONVERT(-0.71787004506),
    DSP32_Q_CONVERT(0.69507511398), DSP32_Q_CONVERT(-0.71893712237),
    DSP32_Q_CONVERT(0.69397146089), DSP32_Q_CONVERT(-0.72000250796),
    DSP32_Q_CONVERT(0.69286617482), DSP32_Q_CONVERT(-0.72106619931),
    DSP32_Q_CONVERT(0.69175925836), DSP32_Q_CONVERT(-0.72212819393),
    DSP32_Q_CONVERT(0.69065071413), DSP32_Q_CONVERT(-0.72318848931),
    DSP32_Q_CONVERT(0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(0.68842875278), DSP32_Q_CONVERT(-0.72530397237),
    DSP32_Q_CONVERT(0.68731534089), DSP32_Q_CONVERT(-0.72635915508),
    DSP32_Q_CONVERT(0.68620031168), DSP32_Q_CONVERT(-0.72741262860),
    DSP32_Q_CONVERT(0.68508366777), DSP32_Q_CONVERT(-0.72846439045),
    DSP32_Q_CONVERT(0.68396541180), DSP32_Q_CONVERT(-0.72951443815),
    DSP32_Q_CONVERT(0.68284554639), DSP32_Q_CONVERT(-0.73056276923),
    DSP32_Q_CONVERT(0.68172407417), DSP32_Q_CONVERT(-0.73160938122),
    DSP32_Q_CONVERT(0.68060099780), DSP32_Q_CONVERT(-0.73265427167),
    DSP32_Q_CONVERT(0.67947631990), DSP32_Q_CONVERT(-0.73369743811),
    DSP32_Q_CONVERT(0.67835004313), DSP32_Q_CONVERT(-0.73473887810),
    DSP32_Q_CONVERT(0.67722217014), DSP32_Q_CONVERT(-0.73577858917),
    DSP32_Q_CONVERT(0.67609270358), DSP32_Q_CONVERT(-0.73681656888),
    DSP32_Q_CONVERT(0.67496164610), DSP32_Q_CONVERT(-0.73785281479),
    DSP32_Q_CONVERT(0.67382900038), DSP32_Q_CONVERT(-0.73888732446),
    DSP32_Q_CONVERT(0.67269476907), DSP32_Q_CONVERT(-0.73992009546),
    DSP32_Q_CONVERT(0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(0.67042156038), DSP32_Q_CONVERT(-0.74198041172),
    DSP32_Q_CONVERT(0.66928258835), DSP32_Q_CONVERT(-0.74300795214),
    DSP32_Q_CONVERT(0.66814204143), DSP32_Q_CONVERT(-0.74403374418),
    DSP32_Q_CONVERT(0.66699992230), DSP32_Q_CONVERT(-0.74505778544),
    DSP32_Q_CONVERT(0.66585623367), DSP32_Q_CONVERT(-0.74608007351),
    DSP32_Q_CONVERT(0.66471097820), DSP32_Q_CONVERT(-0.74710060598),
    DSP32_Q_CONVERT(0.66356415861), DSP32_Q_CONVERT(-0.74811938045),
    DSP32_Q_CONVERT(0.66241577759), DSP32_Q_CONVERT(-0.74913639452),
    DSP32_Q_CONVERT(0.66126583784), DSP32_Q_CONVERT(-0.75015164581),
    DSP32_Q_CONVERT(0.66011434207), DSP32_Q_CONVERT(-0.75116513191),
    DSP32_Q_CONVERT(0.65896129298), DSP32_Q_CONVERT(-0.75217685045),
    DSP32_Q_CONVERT(0.65780669330), DSP32_Q_CONVERT(-0.75318679904),
    DSP32_Q_CONVERT(0.65665054573), DSP32_Q_CONVERT(-0.75419497532),
    DSP32_Q_CONVERT(0.65549285300), DSP32_Q_CONVERT(-0.75520137690),
    DSP32_Q_CONVERT(0.65433361783), DSP32_Q_CONVERT(-0.75620600141),
    DSP32_Q_CONVERT(0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(0.65201053110), DSP32_Q_CONVERT(-0.75820990981),
    DSP32_Q_CONVERT(0.65084668500), DSP32_Q_CONVERT(-0.75920918898),
    DSP32_Q_CONVERT(0.64968130739), DSP32_Q_CONVERT(-0.76020668165),
    DSP32_Q_CONVERT(0.64851440102), DSP32_Q_CONVERT(-0.76120238548),
    DSP32_Q_CONVERT(0.64734596864), DSP32_Q_CONVERT(-0.76219629813),
    DSP32_Q_CONVERT(0.64617601298), DSP32_Q_CONVERT(-0.76318841726),
    DSP32_Q_CONVERT(0.64500453682), DSP32_Q_CONVERT(-0.76417874054),
    DSP32_Q_CONVERT(0.64383154289), DSP32_Q_CONVERT(-0.76516726562),
    DSP32_Q_CONVERT(0.64265703397), DSP32_Q_CONVERT(-0.76615399020),
    DSP32_Q_CONVERT(0.64148101281), DSP32_Q_CONVERT(-0.76713891194),
    DSP32_Q_CONVERT(0.64030348218), DSP32_Q_CONVERT(-0.76812202852),
    DSP32_Q_CONVERT(0.63912444486), DSP32_Q_CONVERT(-0.76910333765),
    DSP32_Q_CONVERT(0.63794390362), DSP32_Q_CONVERT(-0.77008283699),
    DSP32_Q_CONVERT(0.63676186124), DSP32_Q_CONVERT(-0.77106052426),
    DSP32_Q_CONVERT(0.63557832049), DSP32_Q_CONVERT(-0.77203639715),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.63320675505), DSP32_Q_CONVERT(-0.77398269061),
    DSP32_Q_CONVERT(0.63201873594), DSP32_Q_CONVERT(-0.77495310659),
    DSP32_Q_CONVERT(0.63082922963), DSP32_Q_CONVERT(-0.77592169904),
    DSP32_Q_CONVERT(0.62963823891), DSP32_Q_CONVERT(-0.77688846567),
    DSP32_Q_CONVERT(0.62844576660), DSP32_Q_CONVERT(-0.77785340421),
    DSP32_Q_CONVERT(0.62725181550), DSP32_Q_CONVERT(-0.77881651238),
    DSP32_Q_CONVERT(0.62605638840), DSP32_Q_CONVERT(-0.77977778792),
    DSP32_Q_CONVERT(0.62485948814), DSP32_Q_CONVERT(-0.78073722857),
    DSP32_Q_CONVERT(0.62366111753), DSP32_Q_CONVERT(-0.78169483207),
    DSP32_Q_CONVERT(0.62246127937), DSP32_Q_CONVERT(-0.78265059617),
    DSP32_Q_CONVERT(0.62125997651), DSP32_Q_CONVERT(-0.78360451861),
    DSP32_Q_CONVERT(0.62005721176), DSP32_Q_CONVERT(-0.78455659716),
    DSP32_Q_CONVERT(0.61885298796), DSP32_Q_CONVERT(-0.78550682956),
    DSP32_Q_CONVERT(0.61764730794), DSP32_Q_CONVERT(-0.78645521360),
    DSP32_Q_CONVERT(0.61644017453), DSP32_Q_CONVERT(-0.78740174703),
    DSP32_Q_CONVERT(0.61523159058), DSP32_Q_CONVERT(-0.78834642763),
    DSP32_Q_CONVERT(0.61402155893), DSP32_Q_CONVERT(-0.78928925317),
    DSP32_Q_CONVERT(0.61281008243), DSP32_Q_CONVERT(-0.79023022144),
    DSP32_Q_CONVERT(0.61159716393), DSP32_Q_CONVERT(-0.79116933022),
    DSP32_Q_CONVERT(0.61038280628), DSP32_Q_CONVERT(-0.79210657730),
    DSP32_Q_CONVERT(0.60916701234), DSP32_Q_CONVERT(-0.79304196048),
    DSP32_Q_CONVERT(0.60794978497), DSP32_Q_CONVERT(-0.79397547755),
    DSP32_Q_CONVERT(0.60673112703), DSP32_Q_CONVERT(-0.79490712633),
    DSP32_Q_CONVERT(0.60551104140), DSP32_Q_CONVERT(-0.79583690461),
    DSP32_Q_CONVERT(0.60428953095), DSP32_Q_CONVERT(-0.79676481021),
    DSP32_Q_CONVERT(0.60306659854), DSP32_Q_CONVERT(-0.79769084094),
    DSP32_Q_CONVERT(0.60184224706), DSP32_Q_CONVERT(-0.79861499463),
    DSP32_Q_CONVERT(0.60061647938), DSP32_Q_CONVERT(-0.79953726911),
    DSP32_Q_CONVERT(0.59938929840), DSP32_Q_CONVERT(-0.80045766219),
    DSP32_Q_CONVERT(0.59816070700), DSP32_Q_CONVERT(-0.80137617172),
    DSP32_Q_CONVERT(0.59693070806), DSP32_Q_CONVERT(-0.80229279554),
    DSP32_Q_CONVERT(0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.59446649918), DSP32_Q_CONVERT(-0.80412037740),
    DSP32_Q_CONVERT(0.59323229504), DSP32_Q_CONVERT(-0.80503133114),
    DSP32_Q_CONVERT(0.59199669496), DSP32_Q_CONVERT(-0.80594039057),
    DSP32_Q_CONVERT(0.59075970186), DSP32_Q_CONVERT(-0.80684755354),
    DSP32_Q_CONVERT(0.58952131864), DSP32_Q_CONVERT(-0.80775281793),
    DSP32_Q_CONVERT(0.58828154822), DSP32_Q_CONVERT(-0.80865618159),
    DSP32_Q_CONVERT(0.58704039352), DSP32_Q_CONVERT(-0.80955764240),
    DSP32_Q_CONVERT(0.58579785746), DSP32_Q_CONVERT(-0.81045719825),
    DSP32_Q_CONVERT(0.58455394295), DSP32_Q_CONVERT(-0.81135484702),
    DSP32_Q_CONVERT(0.58330865294), DSP32_Q_CONVERT(-0.81225058659),
    DSP32_Q_CONVERT(0.58206199034), DSP32_Q_CONVERT(-0.81314441485),
    DSP32_Q_CONVERT(0.58081395810), DSP32_Q_CONVERT(-0.81403632971),
    DSP32_Q_CONVERT(0.57956455914), DSP32_Q_CONVERT(-0.81492632906),
    DSP32_Q_CONVERT(0.57831379641), DSP32_Q_CONVERT(-0.81581441081),
    DSP32_Q_CONVERT(0.57706167286), DSP32_Q_CONVERT(-0.81670057287),
    DSP32_Q_CONVERT(0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(0.57455335505), DSP32_Q_CONVERT(-0.81846712958),
    DSP32_Q_CONVERT(0.57329716670), DSP32_Q_CONVERT(-0.81934752008),
    DSP32_Q_CONVERT(0.57203962932), DSP32_Q_CONVERT(-0.82022598257),
    DSP32_Q_CONVERT(0.57078074589), DSP32_Q_CONVERT(-0.82110251499),
    DSP32_Q_CONVERT(0.56952051935), DSP32_Q_CONVERT(-0.82197711528),
    DSP32_Q_CONVERT(0.56825895267), DSP32_Q_CONVERT(-0.82284978138),
    DSP32_Q_CONVERT(0.56699604883), DSP32_Q_CONVERT(-0.82372051123),
    DSP32_Q_CONVERT(0.56573181078), DSP32_Q_CONVERT(-0.82458930279),
    DSP32_Q_CONVERT(0.56446624152), DSP32_Q_CONVERT(-0.82545615400),
    DSP32_Q_CONVERT(0.56319934401), DSP32_Q_CONVERT(-0.82632106285),
    DSP32_Q_CONVERT(0.56193112124), DSP32_Q_CONVERT(-0.82718402727),
    DSP32_Q_CONVERT(0.56066157620), DSP32_Q_CONVERT(-0.82804504526),
    DSP32_Q_CONVERT(0.55939071186), DSP32_Q_CONVERT(-0.82890411477),
    DSP32_Q_CONVERT(0.55811853122), DSP32_Q_CONVERT(-0.82976123379),
    DSP32_Q_CONVERT(0.55684503728), DSP32_Q_CONVERT(-0.83061640031),
    DSP32_Q_CONVERT(0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(0.55429412145), DSP32_Q_CONVERT(-0.83232086777),
    DSP32_Q_CONVERT(0.55301670558), DSP32_Q_CONVERT(-0.83317016470),
    DSP32_Q_CONVERT(0.55173798840), DSP32_Q_CONVERT(-0.83401750111),
    DSP32_Q_CONVERT(0.55045797294), DSP32_Q_CONVERT(-0.83486287499),
    DSP32_Q_CONVERT(0.54917666219), DSP32_Q_CONVERT(-0.83570628435),
    DSP32_Q_CONVERT(0.54789405917), DSP32_Q_CONVERT(-0.83654772722),
    DSP32_Q_CONVERT(0.54661016691), DSP32_Q_CONVERT(-0.83738720162),
    DSP32_Q_CONVERT(0.54532498842), DSP32_Q_CONVERT(-0.83822470555),
    DSP32_Q_CONVERT(0.54403852673), DSP32_Q_CONVERT(-0.83906023707),
    DSP32_Q_CONVERT(0.54275078486), DSP32_Q_CONVERT(-0.83989379420),
    DSP32_Q_CONVERT(0.54146176585), DSP32_Q_CONVERT(-0.84072537497),
    DSP32_Q_CONVERT(0.54017147273), DSP32_Q_CONVERT(-0.84155497744),
    DSP32_Q_CONVERT(0.53887990853), DSP32_Q_CONVERT(-0.84238259964),
    DSP32_Q_CONVERT(0.53758707630), DSP32_Q_CONVERT(-0.84320823964),
    DSP32_Q_CONVERT(0.53629297907), DSP32_Q_CONVERT(-0.84403189549),
    DSP32_Q_CONVERT(0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(0.53370100181), DSP32_Q_CONVERT(-0.84567324699),
    DSP32_Q_CONVERT(0.53240312788), DSP32_Q_CONVERT(-0.84649093877),
    DSP32_Q_CONVERT(0.53110400115), DSP32_Q_CONVERT(-0.84730663869),
    DSP32_Q_CONVERT(0.52980362469), DSP32_Q_CONVERT(-0.84812034480),
    DSP32_Q_CONVERT(0.52850200154), DSP32_Q_CONVERT(-0.84893205521),
    DSP32_Q_CONVERT(0.52719913478), DSP32_Q_CONVERT(-0.84974176800),
    DSP32_Q_CONVERT(0.52589502747), DSP32_Q_CONVERT(-0.85054948127),
    DSP32_Q_CONVERT(0.52458968268), DSP32_Q_CONVERT(-0.85135519311),
    DSP32_Q_CONVERT(0.52328310348), DSP32_Q_CONVERT(-0.85215890162),
    DSP32_Q_CONVERT(0.52197529294), DSP32_Q_CONVERT(-0.85296060493),
    DSP32_Q_CONVERT(0.52066625414), DSP32_Q_CONVERT(-0.85376030114),
    DSP32_Q_CONVERT(0.51935599017), DSP32_Q_CONVERT(-0.85455798837),
    DSP32_Q_CONVERT(0.51804450410), DSP32_Q_CONVERT(-0.85535366474),
    DSP32_Q_CONVERT(0.51673179902), DSP32_Q_CONVERT(-0.85614732838),
    DSP32_Q_CONVERT(0.51541787802), DSP32_Q_CONVERT(-0.85693897742),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.51278640063), DSP32_Q_CONVERT(-0.85851622426),
    DSP32_Q_CONVERT(0.51146885044), DSP32_Q_CONVERT(-0.85930181836),
    DSP32_Q_CONVERT(0.51015009671), DSP32_Q_CONVERT(-0.86008539043),
    DSP32_Q_CONVERT(0.50883014254), DSP32_Q_CONVERT(-0.86086693864),
    DSP32_Q_CONVERT(0.50750899105), DSP32_Q_CONVERT(-0.86164646114),
    DSP32_Q_CONVERT(0.50618664535), DSP32_Q_CONVERT(-0.86242395611),
    DSP32_Q_CONVERT(0.50486310853), DSP32_Q_CONVERT(-0.86319942171),
    DSP32_Q_CONVERT(0.50353838373), DSP32_Q_CONVERT(-0.86397285612),
    DSP32_Q_CONVERT(0.50221247405), DSP32_Q_CONVERT(-0.86474425752),
    DSP32_Q_CONVERT(0.50088538261), DSP32_Q_CONVERT(-0.86551362409),
    DSP32_Q_CONVERT(0.49955711255), DSP32_Q_CONVERT(-0.86628095402),
    DSP32_Q_CONVERT(0.49822766697), DSP32_Q_CONVERT(-0.86704624552),
    DSP32_Q_CONVERT(0.49689704902), DSP32_Q_CONVERT(-0.86780949676),
    DSP32_Q_CONVERT(0.49556526183), DSP32_Q_CONVERT(-0.86857070597),
    DSP32_Q_CONVERT(0.49423230852), DSP32_Q_CONVERT(-0.86932987135),
    DSP32_Q_CONVERT(0.49289819223), DSP32_Q_CONVERT(-0.87008699111),
    DSP32_Q_CONVERT(0.49156291611), DSP32_Q_CONVERT(-0.87084206347),
    DSP32_Q_CONVERT(0.49022648329), DSP32_Q_CONVERT(-0.87159508666),
    DSP32_Q_CONVERT(0.48888889692), DSP32_Q_CONVERT(-0.87234605889),
    DSP32_Q_CONVERT(0.48755016015), DSP32_Q_CONVERT(-0.87309497842),
    DSP32_Q_CONVERT(0.48621027612), DSP32_Q_CONVERT(-0.87384184347),
    DSP32_Q_CONVERT(0.48486924800), DSP32_Q_CONVERT(-0.87458665228),
    DSP32_Q_CONVERT(0.48352707893), DSP32_Q_CONVERT(-0.87532940310),
    DSP32_Q_CONVERT(0.48218377208), DSP32_Q_CONVERT(-0.87607009420),
    DSP32_Q_CONVERT(0.48083933060), DSP32_Q_CONVERT(-0.87680872381),
    DSP32_Q_CONVERT(0.47949375766), DSP32_Q_CONVERT(-0.87754529021),
    DSP32_Q_CONVERT(0.47814705642), DSP32_Q_CONVERT(-0.87827979166),
    DSP32_Q_CONVERT(0.47679923006), DSP32_Q_CONVERT(-0.87901222643),
    DSP32_Q_CONVERT(0.47545028175), DSP32_Q_CONVERT(-0.87974259280),
    DSP32_Q_CONVERT(0.47410021465), DSP32_Q_CONVERT(-0.88047088905),
    DSP32_Q_CONVERT(0.47274903195), DSP32_Q_CONVERT(-0.88119711347),
    DSP32_Q_CONVERT(0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(0.47004333246), DSP32_Q_CONVERT(-0.88264333998),
    DSP32_Q_CONVERT(0.46868882204), DSP32_Q_CONVERT(-0.88336333867),
    DSP32_Q_CONVERT(0.46733320874), DSP32_Q_CONVERT(-0.88408125871),
    DSP32_Q_CONVERT(0.46597649577), DSP32_Q_CONVERT(-0.88479709843),
    DSP32_Q_CONVERT(0.46461868631), DSP32_Q_CONVERT(-0.88551085614),
    DSP32_Q_CONVERT(0.46325978355), DSP32_Q_CONVERT(-0.88622253015),
    DSP32_Q_CONVERT(0.46189979070), DSP32_Q_CONVERT(-0.88693211879),
    DSP32_Q_CONVERT(0.46053871096), DSP32_Q_CONVERT(-0.88763962040),
    DSP32_Q_CONVERT(0.45917654752), DSP32_Q_CONVERT(-0.88834503331),
    DSP32_Q_CONVERT(0.45781330360), DSP32_Q_CONVERT(-0.88904835585),
    DSP32_Q_CONVERT(0.45644898240), DSP32_Q_CONVERT(-0.88974958638),
    DSP32_Q_CONVERT(0.45508358713), DSP32_Q_CONVERT(-0.89044872324),
    DSP32_Q_CONVERT(0.45371712100), DSP32_Q_CONVERT(-0.89114576479),
    DSP32_Q_CONVERT(0.45234958723), DSP32_Q_CONVERT(-0.89184070939),
    DSP32_Q_CONVERT(0.45098098905), DSP32_Q_CONVERT(-0.89253355540),
    DSP32_Q_CONVERT(0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(0.44824061229), DSP32_Q_CONVERT(-0.89391294515),
    DSP32_Q_CONVERT(0.44686884016), DSP32_Q_CONVERT(-0.89459948563),
    DSP32_Q_CONVERT(0.44549601651), DSP32_Q_CONVERT(-0.89528392104),
    DSP32_Q_CONVERT(0.44412214457), DSP32_Q_CONVERT(-0.89596624976),
    DSP32_Q_CONVERT(0.44274722756), DSP32_Q_CONVERT(-0.89664647018),
    DSP32_Q_CONVERT(0.44137126873), DSP32_Q_CONVERT(-0.89732458071),
    DSP32_Q_CONVERT(0.43999427131), DSP32_Q_CONVERT(-0.89800057974),
    DSP32_Q_CONVERT(0.43861623854), DSP32_Q_CONVERT(-0.89867446569),
    DSP32_Q_CONVERT(0.43723717366), DSP32_Q_CONVERT(-0.89934623698),
    DSP32_Q_CONVERT(0.43585707992), DSP32_Q_CONVERT(-0.90001589202),
    DSP32_Q_CONVERT(0.43447596057), DSP32_Q_CONVERT(-0.90068342923),
    DSP32_Q_CONVERT(0.43309381885), DSP32_Q_CONVERT(-0.90134884705),
    DSP32_Q_CONVERT(0.43171065803), DSP32_Q_CONVERT(-0.90201214390),
    DSP32_Q_CONVERT(0.43032648134), DSP32_Q_CONVERT(-0.90267331824),
    DSP32_Q_CONVERT(0.42894129206), DSP32_Q_CONVERT(-0.90333236849),
    DSP32_Q_CONVERT(0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(0.42616788873), DSP32_Q_CONVERT(-0.90464409058),
    DSP32_Q_CONVERT(0.42477968121), DSP32_Q_CONVERT(-0.90529675932),
    DSP32_Q_CONVERT(0.42339047414), DSP32_Q_CONVERT(-0.90594729781),
    DSP32_Q_CONVERT(0.42200027080), DSP32_Q_CONVERT(-0.90659570451),
    DSP32_Q_CONVERT(0.42060907445), DSP32_Q_CONVERT(-0.90724197792),
    DSP32_Q_CONVERT(0.41921688836), DSP32_Q_CONVERT(-0.90788611649),
    DSP32_Q_CONVERT(0.41782371582), DSP32_Q_CONVERT(-0.90852811872),
    DSP32_Q_CONVERT(0.41642956010), DSP32_Q_CONVERT(-0.90916798309),
    DSP32_Q_CONVERT(0.41503442448), DSP32_Q_CONVERT(-0.90980570810),
    DSP32_Q_CONVERT(0.41363831224), DSP32_Q_CONVERT(-0.91044129226),
    DSP32_Q_CONVERT(0.41224122667), DSP32_Q_CONVERT(-0.91107473406),
    DSP32_Q_CONVERT(0.41084317106), DSP32_Q_CONVERT(-0.91170603201),
    DSP32_Q_CONVERT(0.40944414869), DSP32_Q_CONVERT(-0.91233518462),
    DSP32_Q_CONVERT(0.40804416286), DSP32_Q_CONVERT(-0.91296219043),
    DSP32_Q_CONVERT(0.40664321687), DSP32_Q_CONVERT(-0.91358704795),
    DSP32_Q_CONVERT(0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(0.40383845757), DSP32_Q_CONVERT(-0.91483031224),
    DSP32_Q_CONVERT(0.40243465086), DSP32_Q_CONVERT(-0.91544871609),
    DSP32_Q_CONVERT(0.40102989718), DSP32_Q_CONVERT(-0.91606496580),
    DSP32_Q_CONVERT(0.39962419985), DSP32_Q_CONVERT(-0.91667905992),
    DSP32_Q_CONVERT(0.39821756215), DSP32_Q_CONVERT(-0.91729099701),
    DSP32_Q_CONVERT(0.39680998742), DSP32_Q_CONVERT(-0.91790077562),
    DSP32_Q_CONVERT(0.39540147895), DSP32_Q_CONVERT(-0.91850839433),
    DSP32_Q_CONVERT(0.39399204006), DSP32_Q_CONVERT(-0.91911385169),
    DSP32_Q_CONVERT(0.39258167407), DSP32_Q_CONVERT(-0.91971714629),
    DSP32_Q_CONVERT(0.39117038430), DSP32_Q_CONVERT(-0.92031827671),
    DSP32_Q_CONVERT(0.38975817407), DSP32_Q_CONVERT(-0.92091724153),
    DSP32_Q_CONVERT(0.38834504670), DSP32_Q_CONVERT(-0.92151403934),
    DSP32_Q_CONVERT(0.38693100551), DSP32_Q_CONVERT(-0.92210866874),
    DSP32_Q_CONVERT(0.38551605384), DSP32_Q_CONVERT(-0.92270112833),
    DSP32_Q_CONVERT(0.38410019502), DSP32_Q_CONVERT(-0.92329141672),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(0.38126576922), DSP32_Q_CONVERT(-0.92446547433),
    DSP32_Q_CONVERT(0.37984720892), DSP32_Q_CONVERT(-0.92504924078),
    DSP32_Q_CONVERT(0.37842775481), DSP32_Q_CONVERT(-0.92563083051),
    DSP32_Q_CONVERT(0.37700741022), DSP32_Q_CONVERT(-0.92621024214),
    DSP32_Q_CONVERT(0.37558617849), DSP32_Q_CONVERT(-0.92678747430),
    DSP32_Q_CONVERT(0.37416406297), DSP32_Q_CONVERT(-0.92736252565),
    DSP32_Q_CONVERT(0.37274106701), DSP32_Q_CONVERT(-0.92793539482),
    DSP32_Q_CONVERT(0.37131719395), DSP32_Q_CONVERT(-0.92850608047),
    DSP32_Q_CONVERT(0.36989244715), DSP32_Q_CONVERT(-0.92907458126),
    DSP32_Q_CONVERT(0.36846682995), DSP32_Q_CONVERT(-0.92964089584),
    DSP32_Q_CONVERT(0.36704034572), DSP32_Q_CONVERT(-0.93020502289),
    DSP32_Q_CONVERT(0.36561299780), DSP32_Q_CONVERT(-0.93076696108),
    DSP32_Q_CONVERT(0.36418478957), DSP32_Q_CONVERT(-0.93132670908),
    DSP32_Q_CONVERT(0.36275572437), DSP32_Q_CONVERT(-0.93188426558),
    DSP32_Q_CONVERT(0.36132580557), DSP32_Q_CONVERT(-0.93243962927),
    DSP32_Q_CONVERT(0.35989503653), DSP32_Q_CONVERT(-0.93299279883),
    DSP32_Q_CONVERT(0.35846342063), DSP32_Q_CONVERT(-0.93354377298),
    DSP32_Q_CONVERT(0.35703096123), DSP32_Q_CONVERT(-0.93409255040),
    DSP32_Q_CONVERT(0.35559766170), DSP32_Q_CONVERT(-0.93463912982),
    DSP32_Q_CONVERT(0.35416352542), DSP32_Q_CONVERT(-0.93518350994),
    DSP32_Q_CONVERT(0.35272855576), DSP32_Q_CONVERT(-0.93572568948),
    DSP32_Q_CONVERT(0.35129275609), DSP32_Q_CONVERT(-0.93626566717),
    DSP32_Q_CONVERT(0.34985612979), DSP32_Q_CONVERT(-0.93680344174),
    DSP32_Q_CONVERT(0.34841868025), DSP32_Q_CONVERT(-0.93733901191),
    DSP32_Q_CONVERT(0.34698041085), DSP32_Q_CONVERT(-0.93787237644),
    DSP32_Q_CONVERT(0.34554132496), DSP32_Q_CONVERT(-0.93840353406),
    DSP32_Q_CONVERT(0.34410142599), DSP32_Q_CONVERT(-0.93893248353),
    DSP32_Q_CONVERT(0.34266071731), DSP32_Q_CONVERT(-0.93945922360),
    DSP32_Q_CONVERT(0.34121920232), DSP32_Q_CONVERT(-0.93998375303),
    DSP32_Q_CONVERT(0.33977688441), DSP32_Q_CONVERT(-0.94050607059),
    DSP32_Q_CONVERT(0.33833376697), DSP32_Q_CONVERT(-0.94102617505),
    DSP32_Q_CONVERT(0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(0.33544514708), DSP32_Q_CONVERT(-0.94205973977),
    DSP32_Q_CONVERT(0.33399965144), DSP32_Q_CONVERT(-0.94257319760),
    DSP32_Q_CONVERT(0.33255336987), DSP32_Q_CONVERT(-0.94308443747),
    DSP32_Q_CONVERT(0.33110630576), DSP32_Q_CONVERT(-0.94359345816),
    DSP32_Q_CONVERT(0.32965846253), DSP32_Q_CONVERT(-0.94410025849),
    DSP32_Q_CONVERT(0.32820984358), DSP32_Q_CONVERT(-0.94460483726),
    DSP32_Q_CONVERT(0.32676045232), DSP32_Q_CONVERT(-0.94510719329),
    DSP32_Q_CONVERT(0.32531029216), DSP32_Q_CONVERT(-0.94560732538),
    DSP32_Q_CONVERT(0.32385936652), DSP32_Q_CONVERT(-0.94610523237),
    DSP32_Q_CONVERT(0.32240767880), DSP32_Q_CONVERT(-0.94660091308),
    DSP32_Q_CONVERT(0.32095523243), DSP32_Q_CONVERT(-0.94709436635),
    DSP32_Q_CONVERT(0.31950203082), DSP32_Q_CONVERT(-0.94758559102),
    DSP32_Q_CONVERT(0.31804807739), DSP32_Q_CONVERT(-0.94807458592),
    DSP32_Q_CONVERT(0.31659337556), DSP32_Q_CONVERT(-0.94856134992),
    DSP32_Q_CONVERT(0.31513792875), DSP32_Q_CONVERT(-0.94904588185),
    DSP32_Q_CONVERT(0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(0.31222481392), DSP32_Q_CONVERT(-0.95000824500),
    DSP32_Q_CONVERT(0.31076715275), DSP32_Q_CONVERT(-0.95048607395),
    DSP32_Q_CONVERT(0.30930876031), DSP32_Q_CONVERT(-0.95096166631),
    DSP32_Q_CONVERT(0.30784964004), DSP32_Q_CONVERT(-0.95143502097),
    DSP32_Q_CONVERT(0.30638979537), DSP32_Q_CONVERT(-0.95190613681),
    DSP32_Q_CONVERT(0.30492922974), DSP32_Q_CONVERT(-0.95237501272),
    DSP32_Q_CONVERT(0.30346794657), DSP32_Q_CONVERT(-0.95284164760),
    DSP32_Q_CONVERT(0.30200594932), DSP32_Q_CONVERT(-0.95330604035),
    DSP32_Q_CONVERT(0.30054324142), DSP32_Q_CONVERT(-0.95376818989),
    DSP32_Q_CONVERT(0.29907982631), DSP32_Q_CONVERT(-0.95422809511),
    DSP32_Q_CONVERT(0.29761570744), DSP32_Q_CONVERT(-0.95468575494),
    DSP32_Q_CONVERT(0.29615088824), DSP32_Q_CONVERT(-0.95514116831),
    DSP32_Q_CONVERT(0.29468537218), DSP32_Q_CONVERT(-0.95559433413),
    DSP32_Q_CONVERT(0.29321916269), DSP32_Q_CONVERT(-0.95604525135),
    DSP32_Q_CONVERT(0.29175226323), DSP32_Q_CONVERT(-0.95649391890),
    DSP32_Q_CONVERT(0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(0.28881640821), DSP32_Q_CONVERT(-0.95738450079),
    DSP32_Q_CONVERT(0.28734745954), DSP32_Q_CONVERT(-0.95782641303),
    DSP32_Q_CONVERT(0.28587783473), DSP32_Q_CONVERT(-0.95826607141),
    DSP32_Q_CONVERT(0.28440753721), DSP32_Q_CONVERT(-0.95870347490),
    DSP32_Q_CONVERT(0.28293657046), DSP32_Q_CONVERT(-0.95913862246),
    DSP32_Q_CONVERT(0.28146493793), DSP32_Q_CONVERT(-0.95957151308),
    DSP32_Q_CONVERT(0.27999264308), DSP32_Q_CONVERT(-0.96000214574),
    DSP32_Q_CONVERT(0.27851968939), DSP32_Q_CONVERT(-0.96043051942),
    DSP32_Q_CONVERT(0.27704608031), DSP32_Q_CONVERT(-0.96085663311),
    DSP32_Q_CONVERT(0.27557181931), DSP32_Q_CONVERT(-0.96128048581),
    DSP32_Q_CONVERT(0.27409690987), DSP32_Q_CONVERT(-0.96170207653),
    DSP32_Q_CONVERT(0.27262135545), DSP32_Q_CONVERT(-0.96212140427),
    DSP32_Q_CONVERT(0.27114515953), DSP32_Q_CONVERT(-0.96253846804),
    DSP32_Q_CONVERT(0.26966832557), DSP32_Q_CONVERT(-0.96295326687),
    DSP32_Q_CONVERT(0.26819085706), DSP32_Q_CONVERT(-0.96336579978),
    DSP32_Q_CONVERT(0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(0.26523403029), DSP32_Q_CONVERT(-0.96418406395),
    DSP32_Q_CONVERT(0.26375467897), DSP32_Q_CONVERT(-0.96458979329),
    DSP32_Q_CONVERT(0.26227470702), DSP32_Q_CONVERT(-0.96499325285),
    DSP32_Q_CONVERT(0.26079411792), DSP32_Q_CONVERT(-0.96539444170),
    DSP32_Q_CONVERT(0.25931291513), DSP32_Q_CONVERT(-0.96579335887),
    DSP32_Q_CONVERT(0.25783110216), DSP32_Q_CONVERT(-0.96619000345),
    DSP32_Q_CONVERT(0.25634868249), DSP32_Q_CONVERT(-0.96658437448),
    DSP32_Q_CONVERT(0.25486565960), DSP32_Q_CONVERT(-0.96697647104),
    DSP32_Q_CONVERT(0.25338203700), DSP32_Q_CONVERT(-0.96736629222),
    DSP32_Q_CONVERT(0.25189781815), DSP32_Q_CONVERT(-0.96775383709),
    DSP32_Q_CONVERT(0.25041300657), DSP32_Q_CONVERT(-0.96813910475),
    DSP32_Q_CONVERT(0.24892760575), DSP32_Q_CONVERT(-0.96852209427),
    DSP32_Q_CONVERT(0.24744161917), DSP32_Q_CONVERT(-0.96890280478),
    DSP32_Q_CONVERT(0.24595505034), DSP32_Q_CONVERT(-0.96928123536),
    DSP32_Q_CONVERT(0.24446790275), DSP32_Q_CONVERT(-0.96965738512),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(0.24149188530), DSP32_Q_CONVERT(-0.97040283869),
    DSP32_Q_CONVERT(0.24000302245), DSP32_Q_CONVERT(-0.97077214073),
    DSP32_Q_CONVERT(0.23851359484), DSP32_Q_CONVERT(-0.97113915845),
    DSP32_Q_CONVERT(0.23702360599), DSP32_Q_CONVERT(-0.97150389099),
    DSP32_Q_CONVERT(0.23553305940), DSP32_Q_CONVERT(-0.97186633748),
    DSP32_Q_CONVERT(0.23404195858), DSP32_Q_CONVERT(-0.97222649708),
    DSP32_Q_CONVERT(0.23255030704), DSP32_Q_CONVERT(-0.97258436893),
    DSP32_Q_CONVERT(0.23105810828), DSP32_Q_CONVERT(-0.97293995221),
    DSP32_Q_CONVERT(0.22956536582), DSP32_Q_CONVERT(-0.97329324605),
    DSP32_Q_CONVERT(0.22807208317), DSP32_Q_CONVERT(-0.97364424965),
    DSP32_Q_CONVERT(0.22657826385), DSP32_Q_CONVERT(-0.97399296217),
    DSP32_Q_CONVERT(0.22508391136), DSP32_Q_CONVERT(-0.97433938279),
    DSP32_Q_CONVERT(0.22358902923), DSP32_Q_CONVERT(-0.97468351069),
    DSP32_Q_CONVERT(0.22209362097), DSP32_Q_CONVERT(-0.97502534507),
    DSP32_Q_CONVERT(0.22059769011), DSP32_Q_CONVERT(-0.97536488512),
    DSP32_Q_CONVERT(0.21910124016), DSP32_Q_CONVERT(-0.97570213004),
    DSP32_Q_CONVERT(0.21760427464), DSP32_Q_CONVERT(-0.97603707904),
    DSP32_Q_CONVERT(0.21610679708), DSP32_Q_CONVERT(-0.97636973133),
    DSP32_Q_CONVERT(0.21460881099), DSP32_Q_CONVERT(-0.97670008613),
    DSP32_Q_CONVERT(0.21311031992), DSP32_Q_CONVERT(-0.97702814266),
    DSP32_Q_CONVERT(0.21161132737), DSP32_Q_CONVERT(-0.97735390015),
    DSP32_Q_CONVERT(0.21011183688), DSP32_Q_CONVERT(-0.97767735782),
    DSP32_Q_CONVERT(0.20861185198), DSP32_Q_CONVERT(-0.97799851493),
    DSP32_Q_CONVERT(0.20711137619), DSP32_Q_CONVERT(-0.97831737072),
    DSP32_Q_CONVERT(0.20561041305), DSP32_Q_CONVERT(-0.97863392443),
    DSP32_Q_CONVERT(0.20410896609), DSP32_Q_CONVERT(-0.97894817532),
    DSP32_Q_CONVERT(0.20260703884), DSP32_Q_CONVERT(-0.97926012265),
    DSP32_Q_CONVERT(0.20110463484), DSP32_Q_CONVERT(-0.97956976569),
    DSP32_Q_CONVERT(0.19960175762), DSP32_Q_CONVERT(-0.97987710370),
    DSP32_Q_CONVERT(0.19809841072), DSP32_Q_CONVERT(-0.98018213597),
    DSP32_Q_CONVERT(0.19659459767), DSP32_Q_CONVERT(-0.98048486177),
    DSP32_Q_CONVERT(0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(0.19358558730), DSP32_Q_CONVERT(-0.98108339115),
    DSP32_Q_CONVERT(0.19208039705), DSP32_Q_CONVERT(-0.98137919331),
    DSP32_Q_CONVERT(0.19057475482), DSP32_Q_CONVERT(-0.98167268620),
    DSP32_Q_CONVERT(0.18906866415), DSP32_Q_CONVERT(-0.98196386911),
    DSP32_Q_CONVERT(0.18756212858), DSP32_Q_CONVERT(-0.98225274137),
    DSP32_Q_CONVERT(0.18605515166), DSP32_Q_CONVERT(-0.98253930229),
    DSP32_Q_CONVERT(0.18454773694), DSP32_Q_CONVERT(-0.98282355120),
    DSP32_Q_CONVERT(0.18303988796), DSP32_Q_CONVERT(-0.98310548743),
    DSP32_Q_CONVERT(0.18153160826), DSP32_Q_CONVERT(-0.98338511032),
    DSP32_Q_CONVERT(0.18002290141), DSP32_Q_CONVERT(-0.98366241921),
    DSP32_Q_CONVERT(0.17851377094), DSP32_Q_CONVERT(-0.98393741345),
    DSP32_Q_CONVERT(0.17700422041), DSP32_Q_CONVERT(-0.98421009239),
    DSP32_Q_CONVERT(0.17549425338), DSP32_Q_CONVERT(-0.98448045538),
    DSP32_Q_CONVERT(0.17398387339), DSP32_Q_CONVERT(-0.98474850180),
    DSP32_Q_CONVERT(0.17247308400), DSP32_Q_CONVERT(-0.98501423101),
    DSP32_Q_CONVERT(0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(0.16945029123), DSP32_Q_CONVERT(-0.98553873531),
    DSP32_Q_CONVERT(0.16793829497), DSP32_Q_CONVERT(-0.98579750917),
    DSP32_Q_CONVERT(0.16642590354), DSP32_Q_CONVERT(-0.98605396335),
    DSP32_Q_CONVERT(0.16491312049), DSP32_Q_CONVERT(-0.98630809724),
    DSP32_Q_CONVERT(0.16339994938), DSP32_Q_CONVERT(-0.98655991026),
    DSP32_Q_CONVERT(0.16188639378), DSP32_Q_CONVERT(-0.98680940181),
    DSP32_Q_CONVERT(0.16037245724), DSP32_Q_CONVERT(-0.98705657131),
    DSP32_Q_CONVERT(0.15885814333), DSP32_Q_CONVERT(-0.98730141816),
    DSP32_Q_CONVERT(0.15734345562), DSP32_Q_CONVERT(-0.98754394179),
    DSP32_Q_CONVERT(0.15582839765), DSP32_Q_CONVERT(-0.98778414164),
    DSP32_Q_CONVERT(0.15431297301), DSP32_Q_CONVERT(-0.98802201714),
    DSP32_Q_CONVERT(0.15279718526), DSP32_Q_CONVERT(-0.98825756773),
    DSP32_Q_CONVERT(0.15128103796), DSP32_Q_CONVERT(-0.98849079285),
    DSP32_Q_CONVERT(0.14976453468), DSP32_Q_CONVERT(-0.98872169196),
    DSP32_Q_CONVERT(0.14824767899), DSP32_Q_CONVERT(-0.98895026451),
    DSP32_Q_CONVERT(0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(0.14521292465), DSP32_Q_CONVERT(-0.98940042779),
    DSP32_Q_CONVERT(0.14369503315), DSP32_Q_CONVERT(-0.98962201746),
    DSP32_Q_CONVERT(0.14217680352), DSP32_Q_CONVERT(-0.98984127846),
    DSP32_Q_CONVERT(0.14065823933), DSP32_Q_CONVERT(-0.99005821026),
    DSP32_Q_CONVERT(0.13913934416), DSP32_Q_CONVERT(-0.99027281236),
    DSP32_Q_CONVERT(0.13762012159), DSP32_Q_CONVERT(-0.99048508426),
    DSP32_Q_CONVERT(0.13610057518), DSP32_Q_CONVERT(-0.99069502544),
    DSP32_Q_CONVERT(0.13458070851), DSP32_Q_CONVERT(-0.99090263543),
    DSP32_Q_CONVERT(0.13306052516), DSP32_Q_CONVERT(-0.99110791372),
    DSP32_Q_CONVERT(0.13154002870), DSP32_Q_CONVERT(-0.99131085985),
    DSP32_Q_CONVERT(0.13001922272), DSP32_Q_CONVERT(-0.99151147332),
    DSP32_Q_CONVERT(0.12849811079), DSP32_Q_CONVERT(-0.99170975367),
    DSP32_Q_CONVERT(0.12697669650), DSP32_Q_CONVERT(-0.99190570043),
    DSP32_Q_CONVERT(0.12545498341), DSP32_Q_CONVERT(-0.99209931314),
    DSP32_Q_CONVERT(0.12393297512), DSP32_Q_CONVERT(-0.99229059135),
    DSP32_Q_CONVERT(0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(0.12088808724), DSP32_Q_CONVERT(-0.99266614245),
    DSP32_Q_CONVERT(0.11936521481), DSP32_Q_CONVERT(-0.99285041446),
    DSP32_Q_CONVERT(0.11784206151), DSP32_Q_CONVERT(-0.99303235020),
    DSP32_Q_CONVERT(0.11631863091), DSP32_Q_CONVERT(-0.99321194923),
    DSP32_Q_CONVERT(0.11479492661), DSP32_Q_CONVERT(-0.99338921115),
    DSP32_Q_CONVERT(0.11327095218), DSP32_Q_CONVERT(-0.99356413552),
    DSP32_Q_CONVERT(0.11174671121), DSP32_Q_CONVERT(-0.99373672194),
    DSP32_Q_CONVERT(0.11022220729), DSP32_Q_CONVERT(-0.99390697000),
    DSP32_Q_CONVERT(0.10869744401), DSP32_Q_CONVERT(-0.99407487930),
    DSP32_Q_CONVERT(0.10717242496), DSP32_Q_CONVERT(-0.99424044945),
    DSP32_Q_CONVERT(0.10564715371), DSP32_Q_CONVERT(-0.99440368006),
    DSP32_Q_CONVERT(0.10412163387), DSP32_Q_CONVERT(-0.99456457073),
    DSP32_Q_CONVERT(0.10259586902), DSP32_Q_CONVERT(-0.99472312110),
    DSP32_Q_CONVERT(0.10106986275), DSP32_Q_CONVERT(-0.99487933079),
    DSP32_Q_CONVERT(0.09954361866), DSP32_Q_CONVERT(-0.99503319944),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(0.09649043136), DSP32_Q_CONVERT(-0.99533391214),
    DSP32_Q_CONVERT(0.09496349533), DSP32_Q_CONVERT(-0.99548075549),
    DSP32_Q_CONVERT(0.09343633585), DSP32_Q_CONVERT(-0.99562525638),
    DSP32_Q_CONVERT(0.09190895650), DSP32_Q_CONVERT(-0.99576741447),
    DSP32_Q_CONVERT(0.09038136088), DSP32_Q_CONVERT(-0.99590722942),
    DSP32_Q_CONVERT(0.08885355258), DSP32_Q_CONVERT(-0.99604470090),
    DSP32_Q_CONVERT(0.08732553521), DSP32_Q_CONVERT(-0.99617982860),
    DSP32_Q_CONVERT(0.08579731234), DSP32_Q_CONVERT(-0.99631261218),
    DSP32_Q_CONVERT(0.08426888759), DSP32_Q_CONVERT(-0.99644305135),
    DSP32_Q_CONVERT(0.08274026455), DSP32_Q_CONVERT(-0.99657114579),
    DSP32_Q_CONVERT(0.08121144681), DSP32_Q_CONVERT(-0.99669689520),
    DSP32_Q_CONVERT(0.07968243797), DSP32_Q_CONVERT(-0.99682029929),
    DSP32_Q_CONVERT(0.07815324163), DSP32_Q_CONVERT(-0.99694135776),
    DSP32_Q_CONVERT(0.07662386139), DSP32_Q_CONVERT(-0.99706007034),
    DSP32_Q_CONVERT(0.07509430085), DSP32_Q_CONVERT(-0.99717643674),
    DSP32_Q_CONVERT(0.07356456360), DSP32_Q_CONVERT(-0.99729045668),
    DSP32_Q_CONVERT(0.07203465325), DSP32_Q_CONVERT(-0.99740212990),
    DSP32_Q_CONVERT(0.07050457339), DSP32_Q_CONVERT(-0.99751145614),
    DSP32_Q_CONVERT(0.06897432763), DSP32_Q_CONVERT(-0.99761843514),
    DSP32_Q_CONVERT(0.06744391956), DSP32_Q_CONVERT(-0.99772306664),
    DSP32_Q_CONVERT(0.06591335280), DSP32_Q_CONVERT(-0.99782535041),
    DSP32_Q_CONVERT(0.06438263093), DSP32_Q_CONVERT(-0.99792528620),
    DSP32_Q_CONVERT(0.06285175756), DSP32_Q_CONVERT(-0.99802287377),
    DSP32_Q_CONVERT(0.06132073630), DSP32_Q_CONVERT(-0.99811811290),
    DSP32_Q_CONVERT(0.05978957075), DSP32_Q_CONVERT(-0.99821100336),
    DSP32_Q_CONVERT(0.05825826450), DSP32_Q_CONVERT(-0.99830154493),
    DSP32_Q_CONVERT(0.05672682117), DSP32_Q_CONVERT(-0.99838973741),
    DSP32_Q_CONVERT(0.05519524435), DSP32_Q_CONVERT(-0.99847558057),
    DSP32_Q_CONVERT(0.05366353765), DSP32_Q_CONVERT(-0.99855907423),
    DSP32_Q_CONVERT(0.05213170468), DSP32_Q_CONVERT(-0.99864021818),
    DSP32_Q_CONVERT(0.05059974904), DSP32_Q_CONVERT(-0.99871901223),
    DSP32_Q_CONVERT(0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(0.04753548416), DSP32_Q_CONVERT(-0.99886954991),
    DSP32_Q_CONVERT(0.04600318213), DSP32_Q_CONVERT(-0.99894129319),
    DSP32_Q_CONVERT(0.04447077185), DSP32_Q_CONVERT(-0.99901068585),
    DSP32_Q_CONVERT(0.04293825693), DSP32_Q_CONVERT(-0.99907772775),
    DSP32_Q_CONVERT(0.04140564098), DSP32_Q_CONVERT(-0.99914241872),
    DSP32_Q_CONVERT(0.03987292759), DSP32_Q_CONVERT(-0.99920475862),
    DSP32_Q_CONVERT(0.03834012037), DSP32_Q_CONVERT(-0.99926474729),
    DSP32_Q_CONVERT(0.03680722294), DSP32_Q_CONVERT(-0.99932238459),
    DSP32_Q_CONVERT(0.03527423890), DSP32_Q_CONVERT(-0.99937767039),
    DSP32_Q_CONVERT(0.03374117185), DSP32_Q_CONVERT(-0.99943060456),
    DSP32_Q_CONVERT(0.03220802541), DSP32_Q_CONVERT(-0.99948118697),
    DSP32_Q_CONVERT(0.03067480318), DSP32_Q_CONVERT(-0.99952941750),
    DSP32_Q_CONVERT(0.02914150876), DSP32_Q_CONVERT(-0.99957529605),
    DSP32_Q_CONVERT(0.02760814578), DSP32_Q_CONVERT(-0.99961882250),
    DSP32_Q_CONVERT(0.02607471783), DSP32_Q_CONVERT(-0.99965999674),
    DSP32_Q_CONVERT(0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(0.02300768147), DSP32_Q_CONVERT(-0.99973528826),
    DSP32_Q_CONVERT(0.02147408028), DSP32_Q_CONVERT(-0.99976940535),
    DSP32_Q_CONVERT(0.01994042855), DSP32_Q_CONVERT(-0.99980116989),
    DSP32_Q_CONVERT(0.01840672991), DSP32_Q_CONVERT(-0.99983058180),
    DSP32_Q_CONVERT(0.01687298795), DSP32_Q_CONVERT(-0.99985764101),
    DSP32_Q_CONVERT(0.01533920628), DSP32_Q_CONVERT(-0.99988234745),
    DSP32_Q_CONVERT(0.01380538853), DSP32_Q_CONVERT(-0.99990470108),
    DSP32_Q_CONVERT(0.01227153829), DSP32_Q_CONVERT(-0.99992470184),
    DSP32_Q_CONVERT(0.01073765917), DSP32_Q_CONVERT(-0.99994234968),
    DSP32_Q_CONVERT(0.00920375478), DSP32_Q_CONVERT(-0.99995764455),
    DSP32_Q_CONVERT(0.00766982874), DSP32_Q_CONVERT(-0.99997058643),
    DSP32_Q_CONVERT(0.00613588465), DSP32_Q_CONVERT(-0.99998117528),
    DSP32_Q_CONVERT(0.00460192612), DSP32_Q_CONVERT(-0.99998941108),
    DSP32_Q_CONVERT(0.00306795676), DSP32_Q_CONVERT(-0.99999529381),
    DSP32_Q_CONVERT(0.00153398019), DSP32_Q_CONVERT(-0.99999882345),
    DSP32_Q_CONVERT(0.00000000000), DSP32_Q_CONVERT(-1.00000000000)
  };

  // If we need to optimize the code to use small twiddle factors table
#  if !(DSP_OPTIMIZATION & DSP_OPTI_SIZE)
    A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp32_t dsp32_twiddle_factors2[DSP32_N_TWIDDLE_FACTORS] = {
    /*Re(w2)   Im(w2)   Re(w3)   Im(w3)  */
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(1.00000000000), DSP32_Q_CONVERT(0.00000000000),
    DSP32_Q_CONVERT(0.99999529381), DSP32_Q_CONVERT(-0.00306795676),
    DSP32_Q_CONVERT(0.99998941108), DSP32_Q_CONVERT(-0.00460192612),
    DSP32_Q_CONVERT(0.99998117528), DSP32_Q_CONVERT(-0.00613588465),
    DSP32_Q_CONVERT(0.99995764455), DSP32_Q_CONVERT(-0.00920375478),
    DSP32_Q_CONVERT(0.99995764455), DSP32_Q_CONVERT(-0.00920375478),
    DSP32_Q_CONVERT(0.99990470108), DSP32_Q_CONVERT(-0.01380538853),
    DSP32_Q_CONVERT(0.99992470184), DSP32_Q_CONVERT(-0.01227153829),
    DSP32_Q_CONVERT(0.99983058180), DSP32_Q_CONVERT(-0.01840672991),
    DSP32_Q_CONVERT(0.99988234745), DSP32_Q_CONVERT(-0.01533920628),
    DSP32_Q_CONVERT(0.99973528826), DSP32_Q_CONVERT(-0.02300768147),
    DSP32_Q_CONVERT(0.99983058180), DSP32_Q_CONVERT(-0.01840672991),
    DSP32_Q_CONVERT(0.99961882250), DSP32_Q_CONVERT(-0.02760814578),
    DSP32_Q_CONVERT(0.99976940535), DSP32_Q_CONVERT(-0.02147408028),
    DSP32_Q_CONVERT(0.99948118697), DSP32_Q_CONVERT(-0.03220802541),
    DSP32_Q_CONVERT(0.99969881870), DSP32_Q_CONVERT(-0.02454122852),
    DSP32_Q_CONVERT(0.99932238459), DSP32_Q_CONVERT(-0.03680722294),
    DSP32_Q_CONVERT(0.99961882250), DSP32_Q_CONVERT(-0.02760814578),
    DSP32_Q_CONVERT(0.99914241872), DSP32_Q_CONVERT(-0.04140564098),
    DSP32_Q_CONVERT(0.99952941750), DSP32_Q_CONVERT(-0.03067480318),
    DSP32_Q_CONVERT(0.99894129319), DSP32_Q_CONVERT(-0.04600318213),
    DSP32_Q_CONVERT(0.99943060456), DSP32_Q_CONVERT(-0.03374117185),
    DSP32_Q_CONVERT(0.99871901223), DSP32_Q_CONVERT(-0.05059974904),
    DSP32_Q_CONVERT(0.99932238459), DSP32_Q_CONVERT(-0.03680722294),
    DSP32_Q_CONVERT(0.99847558057), DSP32_Q_CONVERT(-0.05519524435),
    DSP32_Q_CONVERT(0.99920475862), DSP32_Q_CONVERT(-0.03987292759),
    DSP32_Q_CONVERT(0.99821100336), DSP32_Q_CONVERT(-0.05978957075),
    DSP32_Q_CONVERT(0.99907772775), DSP32_Q_CONVERT(-0.04293825693),
    DSP32_Q_CONVERT(0.99792528620), DSP32_Q_CONVERT(-0.06438263093),
    DSP32_Q_CONVERT(0.99894129319), DSP32_Q_CONVERT(-0.04600318213),
    DSP32_Q_CONVERT(0.99761843514), DSP32_Q_CONVERT(-0.06897432763),
    DSP32_Q_CONVERT(0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(0.99864021818), DSP32_Q_CONVERT(-0.05213170468),
    DSP32_Q_CONVERT(0.99694135776), DSP32_Q_CONVERT(-0.07815324163),
    DSP32_Q_CONVERT(0.99847558057), DSP32_Q_CONVERT(-0.05519524435),
    DSP32_Q_CONVERT(0.99657114579), DSP32_Q_CONVERT(-0.08274026455),
    DSP32_Q_CONVERT(0.99830154493), DSP32_Q_CONVERT(-0.05825826450),
    DSP32_Q_CONVERT(0.99617982860), DSP32_Q_CONVERT(-0.08732553521),
    DSP32_Q_CONVERT(0.99811811290), DSP32_Q_CONVERT(-0.06132073630),
    DSP32_Q_CONVERT(0.99576741447), DSP32_Q_CONVERT(-0.09190895650),
    DSP32_Q_CONVERT(0.99792528620), DSP32_Q_CONVERT(-0.06438263093),
    DSP32_Q_CONVERT(0.99533391214), DSP32_Q_CONVERT(-0.09649043136),
    DSP32_Q_CONVERT(0.99772306664), DSP32_Q_CONVERT(-0.06744391956),
    DSP32_Q_CONVERT(0.99487933079), DSP32_Q_CONVERT(-0.10106986275),
    DSP32_Q_CONVERT(0.99751145614), DSP32_Q_CONVERT(-0.07050457339),
    DSP32_Q_CONVERT(0.99440368006), DSP32_Q_CONVERT(-0.10564715371),
    DSP32_Q_CONVERT(0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(0.99390697000), DSP32_Q_CONVERT(-0.11022220729),
    DSP32_Q_CONVERT(0.99706007034), DSP32_Q_CONVERT(-0.07662386139),
    DSP32_Q_CONVERT(0.99338921115), DSP32_Q_CONVERT(-0.11479492661),
    DSP32_Q_CONVERT(0.99682029929), DSP32_Q_CONVERT(-0.07968243797),
    DSP32_Q_CONVERT(0.99285041446), DSP32_Q_CONVERT(-0.11936521481),
    DSP32_Q_CONVERT(0.99657114579), DSP32_Q_CONVERT(-0.08274026455),
    DSP32_Q_CONVERT(0.99229059135), DSP32_Q_CONVERT(-0.12393297512),
    DSP32_Q_CONVERT(0.99631261218), DSP32_Q_CONVERT(-0.08579731234),
    DSP32_Q_CONVERT(0.99170975367), DSP32_Q_CONVERT(-0.12849811079),
    DSP32_Q_CONVERT(0.99604470090), DSP32_Q_CONVERT(-0.08885355258),
    DSP32_Q_CONVERT(0.99110791372), DSP32_Q_CONVERT(-0.13306052516),
    DSP32_Q_CONVERT(0.99576741447), DSP32_Q_CONVERT(-0.09190895650),
    DSP32_Q_CONVERT(0.99048508426), DSP32_Q_CONVERT(-0.13762012159),
    DSP32_Q_CONVERT(0.99548075549), DSP32_Q_CONVERT(-0.09496349533),
    DSP32_Q_CONVERT(0.98984127846), DSP32_Q_CONVERT(-0.14217680352),
    DSP32_Q_CONVERT(0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.99487933079), DSP32_Q_CONVERT(-0.10106986275),
    DSP32_Q_CONVERT(0.98849079285), DSP32_Q_CONVERT(-0.15128103796),
    DSP32_Q_CONVERT(0.99456457073), DSP32_Q_CONVERT(-0.10412163387),
    DSP32_Q_CONVERT(0.98778414164), DSP32_Q_CONVERT(-0.15582839765),
    DSP32_Q_CONVERT(0.99424044945), DSP32_Q_CONVERT(-0.10717242496),
    DSP32_Q_CONVERT(0.98705657131), DSP32_Q_CONVERT(-0.16037245724),
    DSP32_Q_CONVERT(0.99390697000), DSP32_Q_CONVERT(-0.11022220729),
    DSP32_Q_CONVERT(0.98630809724), DSP32_Q_CONVERT(-0.16491312049),
    DSP32_Q_CONVERT(0.99356413552), DSP32_Q_CONVERT(-0.11327095218),
    DSP32_Q_CONVERT(0.98553873531), DSP32_Q_CONVERT(-0.16945029123),
    DSP32_Q_CONVERT(0.99321194923), DSP32_Q_CONVERT(-0.11631863091),
    DSP32_Q_CONVERT(0.98474850180), DSP32_Q_CONVERT(-0.17398387339),
    DSP32_Q_CONVERT(0.99285041446), DSP32_Q_CONVERT(-0.11936521481),
    DSP32_Q_CONVERT(0.98393741345), DSP32_Q_CONVERT(-0.17851377094),
    DSP32_Q_CONVERT(0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(0.98310548743), DSP32_Q_CONVERT(-0.18303988796),
    DSP32_Q_CONVERT(0.99209931314), DSP32_Q_CONVERT(-0.12545498341),
    DSP32_Q_CONVERT(0.98225274137), DSP32_Q_CONVERT(-0.18756212858),
    DSP32_Q_CONVERT(0.99170975367), DSP32_Q_CONVERT(-0.12849811079),
    DSP32_Q_CONVERT(0.98137919331), DSP32_Q_CONVERT(-0.19208039705),
    DSP32_Q_CONVERT(0.99131085985), DSP32_Q_CONVERT(-0.13154002870),
    DSP32_Q_CONVERT(0.98048486177), DSP32_Q_CONVERT(-0.19659459767),
    DSP32_Q_CONVERT(0.99090263543), DSP32_Q_CONVERT(-0.13458070851),
    DSP32_Q_CONVERT(0.97956976569), DSP32_Q_CONVERT(-0.20110463484),
    DSP32_Q_CONVERT(0.99048508426), DSP32_Q_CONVERT(-0.13762012159),
    DSP32_Q_CONVERT(0.97863392443), DSP32_Q_CONVERT(-0.20561041305),
    DSP32_Q_CONVERT(0.99005821026), DSP32_Q_CONVERT(-0.14065823933),
    DSP32_Q_CONVERT(0.97767735782), DSP32_Q_CONVERT(-0.21011183688),
    DSP32_Q_CONVERT(0.98962201746), DSP32_Q_CONVERT(-0.14369503315),
    DSP32_Q_CONVERT(0.97670008613), DSP32_Q_CONVERT(-0.21460881099),
    DSP32_Q_CONVERT(0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(0.98872169196), DSP32_Q_CONVERT(-0.14976453468),
    DSP32_Q_CONVERT(0.97468351069), DSP32_Q_CONVERT(-0.22358902923),
    DSP32_Q_CONVERT(0.98825756773), DSP32_Q_CONVERT(-0.15279718526),
    DSP32_Q_CONVERT(0.97364424965), DSP32_Q_CONVERT(-0.22807208317),
    DSP32_Q_CONVERT(0.98778414164), DSP32_Q_CONVERT(-0.15582839765),
    DSP32_Q_CONVERT(0.97258436893), DSP32_Q_CONVERT(-0.23255030704),
    DSP32_Q_CONVERT(0.98730141816), DSP32_Q_CONVERT(-0.15885814333),
    DSP32_Q_CONVERT(0.97150389099), DSP32_Q_CONVERT(-0.23702360599),
    DSP32_Q_CONVERT(0.98680940181), DSP32_Q_CONVERT(-0.16188639378),
    DSP32_Q_CONVERT(0.97040283869), DSP32_Q_CONVERT(-0.24149188530),
    DSP32_Q_CONVERT(0.98630809724), DSP32_Q_CONVERT(-0.16491312049),
    DSP32_Q_CONVERT(0.96928123536), DSP32_Q_CONVERT(-0.24595505034),
    DSP32_Q_CONVERT(0.98579750917), DSP32_Q_CONVERT(-0.16793829497),
    DSP32_Q_CONVERT(0.96813910475), DSP32_Q_CONVERT(-0.25041300657),
    DSP32_Q_CONVERT(0.98527764239), DSP32_Q_CONVERT(-0.17096188876),
    DSP32_Q_CONVERT(0.96697647104), DSP32_Q_CONVERT(-0.25486565960),
    DSP32_Q_CONVERT(0.98474850180), DSP32_Q_CONVERT(-0.17398387339),
    DSP32_Q_CONVERT(0.96579335887), DSP32_Q_CONVERT(-0.25931291513),
    DSP32_Q_CONVERT(0.98421009239), DSP32_Q_CONVERT(-0.17700422041),
    DSP32_Q_CONVERT(0.96458979329), DSP32_Q_CONVERT(-0.26375467897),
    DSP32_Q_CONVERT(0.98366241921), DSP32_Q_CONVERT(-0.18002290141),
    DSP32_Q_CONVERT(0.96336579978), DSP32_Q_CONVERT(-0.26819085706),
    DSP32_Q_CONVERT(0.98310548743), DSP32_Q_CONVERT(-0.18303988796),
    DSP32_Q_CONVERT(0.96212140427), DSP32_Q_CONVERT(-0.27262135545),
    DSP32_Q_CONVERT(0.98253930229), DSP32_Q_CONVERT(-0.18605515166),
    DSP32_Q_CONVERT(0.96085663311), DSP32_Q_CONVERT(-0.27704608031),
    DSP32_Q_CONVERT(0.98196386911), DSP32_Q_CONVERT(-0.18906866415),
    DSP32_Q_CONVERT(0.95957151308), DSP32_Q_CONVERT(-0.28146493793),
    DSP32_Q_CONVERT(0.98137919331), DSP32_Q_CONVERT(-0.19208039705),
    DSP32_Q_CONVERT(0.95826607141), DSP32_Q_CONVERT(-0.28587783473),
    DSP32_Q_CONVERT(0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.98018213597), DSP32_Q_CONVERT(-0.19809841072),
    DSP32_Q_CONVERT(0.95559433413), DSP32_Q_CONVERT(-0.29468537218),
    DSP32_Q_CONVERT(0.97956976569), DSP32_Q_CONVERT(-0.20110463484),
    DSP32_Q_CONVERT(0.95422809511), DSP32_Q_CONVERT(-0.29907982631),
    DSP32_Q_CONVERT(0.97894817532), DSP32_Q_CONVERT(-0.20410896609),
    DSP32_Q_CONVERT(0.95284164760), DSP32_Q_CONVERT(-0.30346794657),
    DSP32_Q_CONVERT(0.97831737072), DSP32_Q_CONVERT(-0.20711137619),
    DSP32_Q_CONVERT(0.95143502097), DSP32_Q_CONVERT(-0.30784964004),
    DSP32_Q_CONVERT(0.97767735782), DSP32_Q_CONVERT(-0.21011183688),
    DSP32_Q_CONVERT(0.95000824500), DSP32_Q_CONVERT(-0.31222481392),
    DSP32_Q_CONVERT(0.97702814266), DSP32_Q_CONVERT(-0.21311031992),
    DSP32_Q_CONVERT(0.94856134992), DSP32_Q_CONVERT(-0.31659337556),
    DSP32_Q_CONVERT(0.97636973133), DSP32_Q_CONVERT(-0.21610679708),
    DSP32_Q_CONVERT(0.94709436635), DSP32_Q_CONVERT(-0.32095523243),
    DSP32_Q_CONVERT(0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(0.94560732538), DSP32_Q_CONVERT(-0.32531029216),
    DSP32_Q_CONVERT(0.97502534507), DSP32_Q_CONVERT(-0.22209362097),
    DSP32_Q_CONVERT(0.94410025849), DSP32_Q_CONVERT(-0.32965846253),
    DSP32_Q_CONVERT(0.97433938279), DSP32_Q_CONVERT(-0.22508391136),
    DSP32_Q_CONVERT(0.94257319760), DSP32_Q_CONVERT(-0.33399965144),
    DSP32_Q_CONVERT(0.97364424965), DSP32_Q_CONVERT(-0.22807208317),
    DSP32_Q_CONVERT(0.94102617505), DSP32_Q_CONVERT(-0.33833376697),
    DSP32_Q_CONVERT(0.97293995221), DSP32_Q_CONVERT(-0.23105810828),
    DSP32_Q_CONVERT(0.93945922360), DSP32_Q_CONVERT(-0.34266071731),
    DSP32_Q_CONVERT(0.97222649708), DSP32_Q_CONVERT(-0.23404195858),
    DSP32_Q_CONVERT(0.93787237644), DSP32_Q_CONVERT(-0.34698041085),
    DSP32_Q_CONVERT(0.97150389099), DSP32_Q_CONVERT(-0.23702360599),
    DSP32_Q_CONVERT(0.93626566717), DSP32_Q_CONVERT(-0.35129275609),
    DSP32_Q_CONVERT(0.97077214073), DSP32_Q_CONVERT(-0.24000302245),
    DSP32_Q_CONVERT(0.93463912982), DSP32_Q_CONVERT(-0.35559766170),
    DSP32_Q_CONVERT(0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(0.96928123536), DSP32_Q_CONVERT(-0.24595505034),
    DSP32_Q_CONVERT(0.93132670908), DSP32_Q_CONVERT(-0.36418478957),
    DSP32_Q_CONVERT(0.96852209427), DSP32_Q_CONVERT(-0.24892760575),
    DSP32_Q_CONVERT(0.92964089584), DSP32_Q_CONVERT(-0.36846682995),
    DSP32_Q_CONVERT(0.96775383709), DSP32_Q_CONVERT(-0.25189781815),
    DSP32_Q_CONVERT(0.92793539482), DSP32_Q_CONVERT(-0.37274106701),
    DSP32_Q_CONVERT(0.96697647104), DSP32_Q_CONVERT(-0.25486565960),
    DSP32_Q_CONVERT(0.92621024214), DSP32_Q_CONVERT(-0.37700741022),
    DSP32_Q_CONVERT(0.96619000345), DSP32_Q_CONVERT(-0.25783110216),
    DSP32_Q_CONVERT(0.92446547433), DSP32_Q_CONVERT(-0.38126576922),
    DSP32_Q_CONVERT(0.96539444170), DSP32_Q_CONVERT(-0.26079411792),
    DSP32_Q_CONVERT(0.92270112833), DSP32_Q_CONVERT(-0.38551605384),
    DSP32_Q_CONVERT(0.96458979329), DSP32_Q_CONVERT(-0.26375467897),
    DSP32_Q_CONVERT(0.92091724153), DSP32_Q_CONVERT(-0.38975817407),
    DSP32_Q_CONVERT(0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(0.91911385169), DSP32_Q_CONVERT(-0.39399204006),
    DSP32_Q_CONVERT(0.96295326687), DSP32_Q_CONVERT(-0.26966832557),
    DSP32_Q_CONVERT(0.91729099701), DSP32_Q_CONVERT(-0.39821756215),
    DSP32_Q_CONVERT(0.96212140427), DSP32_Q_CONVERT(-0.27262135545),
    DSP32_Q_CONVERT(0.91544871609), DSP32_Q_CONVERT(-0.40243465086),
    DSP32_Q_CONVERT(0.96128048581), DSP32_Q_CONVERT(-0.27557181931),
    DSP32_Q_CONVERT(0.91358704795), DSP32_Q_CONVERT(-0.40664321687),
    DSP32_Q_CONVERT(0.96043051942), DSP32_Q_CONVERT(-0.27851968939),
    DSP32_Q_CONVERT(0.91170603201), DSP32_Q_CONVERT(-0.41084317106),
    DSP32_Q_CONVERT(0.95957151308), DSP32_Q_CONVERT(-0.28146493793),
    DSP32_Q_CONVERT(0.90980570810), DSP32_Q_CONVERT(-0.41503442448),
    DSP32_Q_CONVERT(0.95870347490), DSP32_Q_CONVERT(-0.28440753721),
    DSP32_Q_CONVERT(0.90788611649), DSP32_Q_CONVERT(-0.41921688836),
    DSP32_Q_CONVERT(0.95782641303), DSP32_Q_CONVERT(-0.28734745954),
    DSP32_Q_CONVERT(0.90594729781), DSP32_Q_CONVERT(-0.42339047414),
    DSP32_Q_CONVERT(0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.95604525135), DSP32_Q_CONVERT(-0.29321916269),
    DSP32_Q_CONVERT(0.90201214390), DSP32_Q_CONVERT(-0.43171065803),
    DSP32_Q_CONVERT(0.95514116831), DSP32_Q_CONVERT(-0.29615088824),
    DSP32_Q_CONVERT(0.90001589202), DSP32_Q_CONVERT(-0.43585707992),
    DSP32_Q_CONVERT(0.95422809511), DSP32_Q_CONVERT(-0.29907982631),
    DSP32_Q_CONVERT(0.89800057974), DSP32_Q_CONVERT(-0.43999427131),
    DSP32_Q_CONVERT(0.95330604035), DSP32_Q_CONVERT(-0.30200594932),
    DSP32_Q_CONVERT(0.89596624976), DSP32_Q_CONVERT(-0.44412214457),
    DSP32_Q_CONVERT(0.95237501272), DSP32_Q_CONVERT(-0.30492922974),
    DSP32_Q_CONVERT(0.89391294515), DSP32_Q_CONVERT(-0.44824061229),
    DSP32_Q_CONVERT(0.95143502097), DSP32_Q_CONVERT(-0.30784964004),
    DSP32_Q_CONVERT(0.89184070939), DSP32_Q_CONVERT(-0.45234958723),
    DSP32_Q_CONVERT(0.95048607395), DSP32_Q_CONVERT(-0.31076715275),
    DSP32_Q_CONVERT(0.88974958638), DSP32_Q_CONVERT(-0.45644898240),
    DSP32_Q_CONVERT(0.94952818059), DSP32_Q_CONVERT(-0.31368174040),
    DSP32_Q_CONVERT(0.88763962040), DSP32_Q_CONVERT(-0.46053871096),
    DSP32_Q_CONVERT(0.94856134992), DSP32_Q_CONVERT(-0.31659337556),
    DSP32_Q_CONVERT(0.88551085614), DSP32_Q_CONVERT(-0.46461868631),
    DSP32_Q_CONVERT(0.94758559102), DSP32_Q_CONVERT(-0.31950203082),
    DSP32_Q_CONVERT(0.88336333867), DSP32_Q_CONVERT(-0.46868882204),
    DSP32_Q_CONVERT(0.94660091308), DSP32_Q_CONVERT(-0.32240767880),
    DSP32_Q_CONVERT(0.88119711347), DSP32_Q_CONVERT(-0.47274903195),
    DSP32_Q_CONVERT(0.94560732538), DSP32_Q_CONVERT(-0.32531029216),
    DSP32_Q_CONVERT(0.87901222643), DSP32_Q_CONVERT(-0.47679923006),
    DSP32_Q_CONVERT(0.94460483726), DSP32_Q_CONVERT(-0.32820984358),
    DSP32_Q_CONVERT(0.87680872381), DSP32_Q_CONVERT(-0.48083933060),
    DSP32_Q_CONVERT(0.94359345816), DSP32_Q_CONVERT(-0.33110630576),
    DSP32_Q_CONVERT(0.87458665228), DSP32_Q_CONVERT(-0.48486924800),
    DSP32_Q_CONVERT(0.94257319760), DSP32_Q_CONVERT(-0.33399965144),
    DSP32_Q_CONVERT(0.87234605889), DSP32_Q_CONVERT(-0.48888889692),
    DSP32_Q_CONVERT(0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(0.94050607059), DSP32_Q_CONVERT(-0.33977688441),
    DSP32_Q_CONVERT(0.86780949676), DSP32_Q_CONVERT(-0.49689704902),
    DSP32_Q_CONVERT(0.93945922360), DSP32_Q_CONVERT(-0.34266071731),
    DSP32_Q_CONVERT(0.86551362409), DSP32_Q_CONVERT(-0.50088538261),
    DSP32_Q_CONVERT(0.93840353406), DSP32_Q_CONVERT(-0.34554132496),
    DSP32_Q_CONVERT(0.86319942171), DSP32_Q_CONVERT(-0.50486310853),
    DSP32_Q_CONVERT(0.93733901191), DSP32_Q_CONVERT(-0.34841868025),
    DSP32_Q_CONVERT(0.86086693864), DSP32_Q_CONVERT(-0.50883014254),
    DSP32_Q_CONVERT(0.93626566717), DSP32_Q_CONVERT(-0.35129275609),
    DSP32_Q_CONVERT(0.85851622426), DSP32_Q_CONVERT(-0.51278640063),
    DSP32_Q_CONVERT(0.93518350994), DSP32_Q_CONVERT(-0.35416352542),
    DSP32_Q_CONVERT(0.85614732838), DSP32_Q_CONVERT(-0.51673179902),
    DSP32_Q_CONVERT(0.93409255040), DSP32_Q_CONVERT(-0.35703096123),
    DSP32_Q_CONVERT(0.85376030114), DSP32_Q_CONVERT(-0.52066625414),
    DSP32_Q_CONVERT(0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(0.85135519311), DSP32_Q_CONVERT(-0.52458968268),
    DSP32_Q_CONVERT(0.93188426558), DSP32_Q_CONVERT(-0.36275572437),
    DSP32_Q_CONVERT(0.84893205521), DSP32_Q_CONVERT(-0.52850200154),
    DSP32_Q_CONVERT(0.93076696108), DSP32_Q_CONVERT(-0.36561299780),
    DSP32_Q_CONVERT(0.84649093877), DSP32_Q_CONVERT(-0.53240312788),
    DSP32_Q_CONVERT(0.92964089584), DSP32_Q_CONVERT(-0.36846682995),
    DSP32_Q_CONVERT(0.84403189549), DSP32_Q_CONVERT(-0.53629297907),
    DSP32_Q_CONVERT(0.92850608047), DSP32_Q_CONVERT(-0.37131719395),
    DSP32_Q_CONVERT(0.84155497744), DSP32_Q_CONVERT(-0.54017147273),
    DSP32_Q_CONVERT(0.92736252565), DSP32_Q_CONVERT(-0.37416406297),
    DSP32_Q_CONVERT(0.83906023707), DSP32_Q_CONVERT(-0.54403852673),
    DSP32_Q_CONVERT(0.92621024214), DSP32_Q_CONVERT(-0.37700741022),
    DSP32_Q_CONVERT(0.83654772722), DSP32_Q_CONVERT(-0.54789405917),
    DSP32_Q_CONVERT(0.92504924078), DSP32_Q_CONVERT(-0.37984720892),
    DSP32_Q_CONVERT(0.83401750111), DSP32_Q_CONVERT(-0.55173798840),
    DSP32_Q_CONVERT(0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.92270112833), DSP32_Q_CONVERT(-0.38551605384),
    DSP32_Q_CONVERT(0.82890411477), DSP32_Q_CONVERT(-0.55939071186),
    DSP32_Q_CONVERT(0.92151403934), DSP32_Q_CONVERT(-0.38834504670),
    DSP32_Q_CONVERT(0.82632106285), DSP32_Q_CONVERT(-0.56319934401),
    DSP32_Q_CONVERT(0.92031827671), DSP32_Q_CONVERT(-0.39117038430),
    DSP32_Q_CONVERT(0.82372051123), DSP32_Q_CONVERT(-0.56699604883),
    DSP32_Q_CONVERT(0.91911385169), DSP32_Q_CONVERT(-0.39399204006),
    DSP32_Q_CONVERT(0.82110251499), DSP32_Q_CONVERT(-0.57078074589),
    DSP32_Q_CONVERT(0.91790077562), DSP32_Q_CONVERT(-0.39680998742),
    DSP32_Q_CONVERT(0.81846712958), DSP32_Q_CONVERT(-0.57455335505),
    DSP32_Q_CONVERT(0.91667905992), DSP32_Q_CONVERT(-0.39962419985),
    DSP32_Q_CONVERT(0.81581441081), DSP32_Q_CONVERT(-0.57831379641),
    DSP32_Q_CONVERT(0.91544871609), DSP32_Q_CONVERT(-0.40243465086),
    DSP32_Q_CONVERT(0.81314441485), DSP32_Q_CONVERT(-0.58206199034),
    DSP32_Q_CONVERT(0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(0.81045719825), DSP32_Q_CONVERT(-0.58579785746),
    DSP32_Q_CONVERT(0.91296219043), DSP32_Q_CONVERT(-0.40804416286),
    DSP32_Q_CONVERT(0.80775281793), DSP32_Q_CONVERT(-0.58952131864),
    DSP32_Q_CONVERT(0.91170603201), DSP32_Q_CONVERT(-0.41084317106),
    DSP32_Q_CONVERT(0.80503133114), DSP32_Q_CONVERT(-0.59323229504),
    DSP32_Q_CONVERT(0.91044129226), DSP32_Q_CONVERT(-0.41363831224),
    DSP32_Q_CONVERT(0.80229279554), DSP32_Q_CONVERT(-0.59693070806),
    DSP32_Q_CONVERT(0.90916798309), DSP32_Q_CONVERT(-0.41642956010),
    DSP32_Q_CONVERT(0.79953726911), DSP32_Q_CONVERT(-0.60061647938),
    DSP32_Q_CONVERT(0.90788611649), DSP32_Q_CONVERT(-0.41921688836),
    DSP32_Q_CONVERT(0.79676481021), DSP32_Q_CONVERT(-0.60428953095),
    DSP32_Q_CONVERT(0.90659570451), DSP32_Q_CONVERT(-0.42200027080),
    DSP32_Q_CONVERT(0.79397547755), DSP32_Q_CONVERT(-0.60794978497),
    DSP32_Q_CONVERT(0.90529675932), DSP32_Q_CONVERT(-0.42477968121),
    DSP32_Q_CONVERT(0.79116933022), DSP32_Q_CONVERT(-0.61159716393),
    DSP32_Q_CONVERT(0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(0.90267331824), DSP32_Q_CONVERT(-0.43032648134),
    DSP32_Q_CONVERT(0.78550682956), DSP32_Q_CONVERT(-0.61885298796),
    DSP32_Q_CONVERT(0.90134884705), DSP32_Q_CONVERT(-0.43309381885),
    DSP32_Q_CONVERT(0.78265059617), DSP32_Q_CONVERT(-0.62246127937),
    DSP32_Q_CONVERT(0.90001589202), DSP32_Q_CONVERT(-0.43585707992),
    DSP32_Q_CONVERT(0.77977778792), DSP32_Q_CONVERT(-0.62605638840),
    DSP32_Q_CONVERT(0.89867446569), DSP32_Q_CONVERT(-0.43861623854),
    DSP32_Q_CONVERT(0.77688846567), DSP32_Q_CONVERT(-0.62963823891),
    DSP32_Q_CONVERT(0.89732458071), DSP32_Q_CONVERT(-0.44137126873),
    DSP32_Q_CONVERT(0.77398269061), DSP32_Q_CONVERT(-0.63320675505),
    DSP32_Q_CONVERT(0.89596624976), DSP32_Q_CONVERT(-0.44412214457),
    DSP32_Q_CONVERT(0.77106052426), DSP32_Q_CONVERT(-0.63676186124),
    DSP32_Q_CONVERT(0.89459948563), DSP32_Q_CONVERT(-0.44686884016),
    DSP32_Q_CONVERT(0.76812202852), DSP32_Q_CONVERT(-0.64030348218),
    DSP32_Q_CONVERT(0.89322430120), DSP32_Q_CONVERT(-0.44961132965),
    DSP32_Q_CONVERT(0.76516726562), DSP32_Q_CONVERT(-0.64383154289),
    DSP32_Q_CONVERT(0.89184070939), DSP32_Q_CONVERT(-0.45234958723),
    DSP32_Q_CONVERT(0.76219629813), DSP32_Q_CONVERT(-0.64734596864),
    DSP32_Q_CONVERT(0.89044872324), DSP32_Q_CONVERT(-0.45508358713),
    DSP32_Q_CONVERT(0.75920918898), DSP32_Q_CONVERT(-0.65084668500),
    DSP32_Q_CONVERT(0.88904835585), DSP32_Q_CONVERT(-0.45781330360),
    DSP32_Q_CONVERT(0.75620600141), DSP32_Q_CONVERT(-0.65433361783),
    DSP32_Q_CONVERT(0.88763962040), DSP32_Q_CONVERT(-0.46053871096),
    DSP32_Q_CONVERT(0.75318679904), DSP32_Q_CONVERT(-0.65780669330),
    DSP32_Q_CONVERT(0.88622253015), DSP32_Q_CONVERT(-0.46325978355),
    DSP32_Q_CONVERT(0.75015164581), DSP32_Q_CONVERT(-0.66126583784),
    DSP32_Q_CONVERT(0.88479709843), DSP32_Q_CONVERT(-0.46597649577),
    DSP32_Q_CONVERT(0.74710060598), DSP32_Q_CONVERT(-0.66471097820),
    DSP32_Q_CONVERT(0.88336333867), DSP32_Q_CONVERT(-0.46868882204),
    DSP32_Q_CONVERT(0.74403374418), DSP32_Q_CONVERT(-0.66814204143),
    DSP32_Q_CONVERT(0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.88047088905), DSP32_Q_CONVERT(-0.47410021465),
    DSP32_Q_CONVERT(0.73785281479), DSP32_Q_CONVERT(-0.67496164610),
    DSP32_Q_CONVERT(0.87901222643), DSP32_Q_CONVERT(-0.47679923006),
    DSP32_Q_CONVERT(0.73473887810), DSP32_Q_CONVERT(-0.67835004313),
    DSP32_Q_CONVERT(0.87754529021), DSP32_Q_CONVERT(-0.47949375766),
    DSP32_Q_CONVERT(0.73160938122), DSP32_Q_CONVERT(-0.68172407417),
    DSP32_Q_CONVERT(0.87607009420), DSP32_Q_CONVERT(-0.48218377208),
    DSP32_Q_CONVERT(0.72846439045), DSP32_Q_CONVERT(-0.68508366777),
    DSP32_Q_CONVERT(0.87458665228), DSP32_Q_CONVERT(-0.48486924800),
    DSP32_Q_CONVERT(0.72530397237), DSP32_Q_CONVERT(-0.68842875278),
    DSP32_Q_CONVERT(0.87309497842), DSP32_Q_CONVERT(-0.48755016015),
    DSP32_Q_CONVERT(0.72212819393), DSP32_Q_CONVERT(-0.69175925836),
    DSP32_Q_CONVERT(0.87159508666), DSP32_Q_CONVERT(-0.49022648329),
    DSP32_Q_CONVERT(0.71893712237), DSP32_Q_CONVERT(-0.69507511398),
    DSP32_Q_CONVERT(0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(0.71573082528), DSP32_Q_CONVERT(-0.69837624941),
    DSP32_Q_CONVERT(0.86857070597), DSP32_Q_CONVERT(-0.49556526183),
    DSP32_Q_CONVERT(0.71250937056), DSP32_Q_CONVERT(-0.70166259474),
    DSP32_Q_CONVERT(0.86704624552), DSP32_Q_CONVERT(-0.49822766697),
    DSP32_Q_CONVERT(0.70927282644), DSP32_Q_CONVERT(-0.70493408038),
    DSP32_Q_CONVERT(0.86551362409), DSP32_Q_CONVERT(-0.50088538261),
    DSP32_Q_CONVERT(0.70602126145), DSP32_Q_CONVERT(-0.70819063703),
    DSP32_Q_CONVERT(0.86397285612), DSP32_Q_CONVERT(-0.50353838373),
    DSP32_Q_CONVERT(0.70275474446), DSP32_Q_CONVERT(-0.71143219575),
    DSP32_Q_CONVERT(0.86242395611), DSP32_Q_CONVERT(-0.50618664535),
    DSP32_Q_CONVERT(0.69947334464), DSP32_Q_CONVERT(-0.71465868786),
    DSP32_Q_CONVERT(0.86086693864), DSP32_Q_CONVERT(-0.50883014254),
    DSP32_Q_CONVERT(0.69617713149), DSP32_Q_CONVERT(-0.71787004506),
    DSP32_Q_CONVERT(0.85930181836), DSP32_Q_CONVERT(-0.51146885044),
    DSP32_Q_CONVERT(0.69286617482), DSP32_Q_CONVERT(-0.72106619931),
    DSP32_Q_CONVERT(0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(0.85614732838), DSP32_Q_CONVERT(-0.51673179902),
    DSP32_Q_CONVERT(0.68620031168), DSP32_Q_CONVERT(-0.72741262860),
    DSP32_Q_CONVERT(0.85455798837), DSP32_Q_CONVERT(-0.51935599017),
    DSP32_Q_CONVERT(0.68284554639), DSP32_Q_CONVERT(-0.73056276923),
    DSP32_Q_CONVERT(0.85296060493), DSP32_Q_CONVERT(-0.52197529294),
    DSP32_Q_CONVERT(0.67947631990), DSP32_Q_CONVERT(-0.73369743811),
    DSP32_Q_CONVERT(0.85135519311), DSP32_Q_CONVERT(-0.52458968268),
    DSP32_Q_CONVERT(0.67609270358), DSP32_Q_CONVERT(-0.73681656888),
    DSP32_Q_CONVERT(0.84974176800), DSP32_Q_CONVERT(-0.52719913478),
    DSP32_Q_CONVERT(0.67269476907), DSP32_Q_CONVERT(-0.73992009546),
    DSP32_Q_CONVERT(0.84812034480), DSP32_Q_CONVERT(-0.52980362469),
    DSP32_Q_CONVERT(0.66928258835), DSP32_Q_CONVERT(-0.74300795214),
    DSP32_Q_CONVERT(0.84649093877), DSP32_Q_CONVERT(-0.53240312788),
    DSP32_Q_CONVERT(0.66585623367), DSP32_Q_CONVERT(-0.74608007351),
    DSP32_Q_CONVERT(0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(0.66241577759), DSP32_Q_CONVERT(-0.74913639452),
    DSP32_Q_CONVERT(0.84320823964), DSP32_Q_CONVERT(-0.53758707630),
    DSP32_Q_CONVERT(0.65896129298), DSP32_Q_CONVERT(-0.75217685045),
    DSP32_Q_CONVERT(0.84155497744), DSP32_Q_CONVERT(-0.54017147273),
    DSP32_Q_CONVERT(0.65549285300), DSP32_Q_CONVERT(-0.75520137690),
    DSP32_Q_CONVERT(0.83989379420), DSP32_Q_CONVERT(-0.54275078486),
    DSP32_Q_CONVERT(0.65201053110), DSP32_Q_CONVERT(-0.75820990981),
    DSP32_Q_CONVERT(0.83822470555), DSP32_Q_CONVERT(-0.54532498842),
    DSP32_Q_CONVERT(0.64851440102), DSP32_Q_CONVERT(-0.76120238548),
    DSP32_Q_CONVERT(0.83654772722), DSP32_Q_CONVERT(-0.54789405917),
    DSP32_Q_CONVERT(0.64500453682), DSP32_Q_CONVERT(-0.76417874054),
    DSP32_Q_CONVERT(0.83486287499), DSP32_Q_CONVERT(-0.55045797294),
    DSP32_Q_CONVERT(0.64148101281), DSP32_Q_CONVERT(-0.76713891194),
    DSP32_Q_CONVERT(0.83317016470), DSP32_Q_CONVERT(-0.55301670558),
    DSP32_Q_CONVERT(0.63794390362), DSP32_Q_CONVERT(-0.77008283699),
    DSP32_Q_CONVERT(0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.82976123379), DSP32_Q_CONVERT(-0.55811853122),
    DSP32_Q_CONVERT(0.63082922963), DSP32_Q_CONVERT(-0.77592169904),
    DSP32_Q_CONVERT(0.82804504526), DSP32_Q_CONVERT(-0.56066157620),
    DSP32_Q_CONVERT(0.62725181550), DSP32_Q_CONVERT(-0.77881651238),
    DSP32_Q_CONVERT(0.82632106285), DSP32_Q_CONVERT(-0.56319934401),
    DSP32_Q_CONVERT(0.62366111753), DSP32_Q_CONVERT(-0.78169483207),
    DSP32_Q_CONVERT(0.82458930279), DSP32_Q_CONVERT(-0.56573181078),
    DSP32_Q_CONVERT(0.62005721176), DSP32_Q_CONVERT(-0.78455659716),
    DSP32_Q_CONVERT(0.82284978138), DSP32_Q_CONVERT(-0.56825895267),
    DSP32_Q_CONVERT(0.61644017453), DSP32_Q_CONVERT(-0.78740174703),
    DSP32_Q_CONVERT(0.82110251499), DSP32_Q_CONVERT(-0.57078074589),
    DSP32_Q_CONVERT(0.61281008243), DSP32_Q_CONVERT(-0.79023022144),
    DSP32_Q_CONVERT(0.81934752008), DSP32_Q_CONVERT(-0.57329716670),
    DSP32_Q_CONVERT(0.60916701234), DSP32_Q_CONVERT(-0.79304196048),
    DSP32_Q_CONVERT(0.81758481315), DSP32_Q_CONVERT(-0.57580819142),
    DSP32_Q_CONVERT(0.60551104140), DSP32_Q_CONVERT(-0.79583690461),
    DSP32_Q_CONVERT(0.81581441081), DSP32_Q_CONVERT(-0.57831379641),
    DSP32_Q_CONVERT(0.60184224706), DSP32_Q_CONVERT(-0.79861499463),
    DSP32_Q_CONVERT(0.81403632971), DSP32_Q_CONVERT(-0.58081395810),
    DSP32_Q_CONVERT(0.59816070700), DSP32_Q_CONVERT(-0.80137617172),
    DSP32_Q_CONVERT(0.81225058659), DSP32_Q_CONVERT(-0.58330865294),
    DSP32_Q_CONVERT(0.59446649918), DSP32_Q_CONVERT(-0.80412037740),
    DSP32_Q_CONVERT(0.81045719825), DSP32_Q_CONVERT(-0.58579785746),
    DSP32_Q_CONVERT(0.59075970186), DSP32_Q_CONVERT(-0.80684755354),
    DSP32_Q_CONVERT(0.80865618159), DSP32_Q_CONVERT(-0.58828154822),
    DSP32_Q_CONVERT(0.58704039352), DSP32_Q_CONVERT(-0.80955764240),
    DSP32_Q_CONVERT(0.80684755354), DSP32_Q_CONVERT(-0.59075970186),
    DSP32_Q_CONVERT(0.58330865294), DSP32_Q_CONVERT(-0.81225058659),
    DSP32_Q_CONVERT(0.80503133114), DSP32_Q_CONVERT(-0.59323229504),
    DSP32_Q_CONVERT(0.57956455914), DSP32_Q_CONVERT(-0.81492632906),
    DSP32_Q_CONVERT(0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(0.80137617172), DSP32_Q_CONVERT(-0.59816070700),
    DSP32_Q_CONVERT(0.57203962932), DSP32_Q_CONVERT(-0.82022598257),
    DSP32_Q_CONVERT(0.79953726911), DSP32_Q_CONVERT(-0.60061647938),
    DSP32_Q_CONVERT(0.56825895267), DSP32_Q_CONVERT(-0.82284978138),
    DSP32_Q_CONVERT(0.79769084094), DSP32_Q_CONVERT(-0.60306659854),
    DSP32_Q_CONVERT(0.56446624152), DSP32_Q_CONVERT(-0.82545615400),
    DSP32_Q_CONVERT(0.79583690461), DSP32_Q_CONVERT(-0.60551104140),
    DSP32_Q_CONVERT(0.56066157620), DSP32_Q_CONVERT(-0.82804504526),
    DSP32_Q_CONVERT(0.79397547755), DSP32_Q_CONVERT(-0.60794978497),
    DSP32_Q_CONVERT(0.55684503728), DSP32_Q_CONVERT(-0.83061640031),
    DSP32_Q_CONVERT(0.79210657730), DSP32_Q_CONVERT(-0.61038280628),
    DSP32_Q_CONVERT(0.55301670558), DSP32_Q_CONVERT(-0.83317016470),
    DSP32_Q_CONVERT(0.79023022144), DSP32_Q_CONVERT(-0.61281008243),
    DSP32_Q_CONVERT(0.54917666219), DSP32_Q_CONVERT(-0.83570628435),
    DSP32_Q_CONVERT(0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(0.54532498842), DSP32_Q_CONVERT(-0.83822470555),
    DSP32_Q_CONVERT(0.78645521360), DSP32_Q_CONVERT(-0.61764730794),
    DSP32_Q_CONVERT(0.54146176585), DSP32_Q_CONVERT(-0.84072537497),
    DSP32_Q_CONVERT(0.78455659716), DSP32_Q_CONVERT(-0.62005721176),
    DSP32_Q_CONVERT(0.53758707630), DSP32_Q_CONVERT(-0.84320823964),
    DSP32_Q_CONVERT(0.78265059617), DSP32_Q_CONVERT(-0.62246127937),
    DSP32_Q_CONVERT(0.53370100181), DSP32_Q_CONVERT(-0.84567324699),
    DSP32_Q_CONVERT(0.78073722857), DSP32_Q_CONVERT(-0.62485948814),
    DSP32_Q_CONVERT(0.52980362469), DSP32_Q_CONVERT(-0.84812034480),
    DSP32_Q_CONVERT(0.77881651238), DSP32_Q_CONVERT(-0.62725181550),
    DSP32_Q_CONVERT(0.52589502747), DSP32_Q_CONVERT(-0.85054948127),
    DSP32_Q_CONVERT(0.77688846567), DSP32_Q_CONVERT(-0.62963823891),
    DSP32_Q_CONVERT(0.52197529294), DSP32_Q_CONVERT(-0.85296060493),
    DSP32_Q_CONVERT(0.77495310659), DSP32_Q_CONVERT(-0.63201873594),
    DSP32_Q_CONVERT(0.51804450410), DSP32_Q_CONVERT(-0.85535366474),
    DSP32_Q_CONVERT(0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.77106052426), DSP32_Q_CONVERT(-0.63676186124),
    DSP32_Q_CONVERT(0.51015009671), DSP32_Q_CONVERT(-0.86008539043),
    DSP32_Q_CONVERT(0.76910333765), DSP32_Q_CONVERT(-0.63912444486),
    DSP32_Q_CONVERT(0.50618664535), DSP32_Q_CONVERT(-0.86242395611),
    DSP32_Q_CONVERT(0.76713891194), DSP32_Q_CONVERT(-0.64148101281),
    DSP32_Q_CONVERT(0.50221247405), DSP32_Q_CONVERT(-0.86474425752),
    DSP32_Q_CONVERT(0.76516726562), DSP32_Q_CONVERT(-0.64383154289),
    DSP32_Q_CONVERT(0.49822766697), DSP32_Q_CONVERT(-0.86704624552),
    DSP32_Q_CONVERT(0.76318841726), DSP32_Q_CONVERT(-0.64617601298),
    DSP32_Q_CONVERT(0.49423230852), DSP32_Q_CONVERT(-0.86932987135),
    DSP32_Q_CONVERT(0.76120238548), DSP32_Q_CONVERT(-0.64851440102),
    DSP32_Q_CONVERT(0.49022648329), DSP32_Q_CONVERT(-0.87159508666),
    DSP32_Q_CONVERT(0.75920918898), DSP32_Q_CONVERT(-0.65084668500),
    DSP32_Q_CONVERT(0.48621027612), DSP32_Q_CONVERT(-0.87384184347),
    DSP32_Q_CONVERT(0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(0.48218377208), DSP32_Q_CONVERT(-0.87607009420),
    DSP32_Q_CONVERT(0.75520137690), DSP32_Q_CONVERT(-0.65549285300),
    DSP32_Q_CONVERT(0.47814705642), DSP32_Q_CONVERT(-0.87827979166),
    DSP32_Q_CONVERT(0.75318679904), DSP32_Q_CONVERT(-0.65780669330),
    DSP32_Q_CONVERT(0.47410021465), DSP32_Q_CONVERT(-0.88047088905),
    DSP32_Q_CONVERT(0.75116513191), DSP32_Q_CONVERT(-0.66011434207),
    DSP32_Q_CONVERT(0.47004333246), DSP32_Q_CONVERT(-0.88264333998),
    DSP32_Q_CONVERT(0.74913639452), DSP32_Q_CONVERT(-0.66241577759),
    DSP32_Q_CONVERT(0.46597649577), DSP32_Q_CONVERT(-0.88479709843),
    DSP32_Q_CONVERT(0.74710060598), DSP32_Q_CONVERT(-0.66471097820),
    DSP32_Q_CONVERT(0.46189979070), DSP32_Q_CONVERT(-0.88693211879),
    DSP32_Q_CONVERT(0.74505778544), DSP32_Q_CONVERT(-0.66699992230),
    DSP32_Q_CONVERT(0.45781330360), DSP32_Q_CONVERT(-0.88904835585),
    DSP32_Q_CONVERT(0.74300795214), DSP32_Q_CONVERT(-0.66928258835),
    DSP32_Q_CONVERT(0.45371712100), DSP32_Q_CONVERT(-0.89114576479),
    DSP32_Q_CONVERT(0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(0.73888732446), DSP32_Q_CONVERT(-0.67382900038),
    DSP32_Q_CONVERT(0.44549601651), DSP32_Q_CONVERT(-0.89528392104),
    DSP32_Q_CONVERT(0.73681656888), DSP32_Q_CONVERT(-0.67609270358),
    DSP32_Q_CONVERT(0.44137126873), DSP32_Q_CONVERT(-0.89732458071),
    DSP32_Q_CONVERT(0.73473887810), DSP32_Q_CONVERT(-0.67835004313),
    DSP32_Q_CONVERT(0.43723717366), DSP32_Q_CONVERT(-0.89934623698),
    DSP32_Q_CONVERT(0.73265427167), DSP32_Q_CONVERT(-0.68060099780),
    DSP32_Q_CONVERT(0.43309381885), DSP32_Q_CONVERT(-0.90134884705),
    DSP32_Q_CONVERT(0.73056276923), DSP32_Q_CONVERT(-0.68284554639),
    DSP32_Q_CONVERT(0.42894129206), DSP32_Q_CONVERT(-0.90333236849),
    DSP32_Q_CONVERT(0.72846439045), DSP32_Q_CONVERT(-0.68508366777),
    DSP32_Q_CONVERT(0.42477968121), DSP32_Q_CONVERT(-0.90529675932),
    DSP32_Q_CONVERT(0.72635915508), DSP32_Q_CONVERT(-0.68731534089),
    DSP32_Q_CONVERT(0.42060907445), DSP32_Q_CONVERT(-0.90724197792),
    DSP32_Q_CONVERT(0.72424708295), DSP32_Q_CONVERT(-0.68954054474),
    DSP32_Q_CONVERT(0.41642956010), DSP32_Q_CONVERT(-0.90916798309),
    DSP32_Q_CONVERT(0.72212819393), DSP32_Q_CONVERT(-0.69175925836),
    DSP32_Q_CONVERT(0.41224122667), DSP32_Q_CONVERT(-0.91107473406),
    DSP32_Q_CONVERT(0.72000250796), DSP32_Q_CONVERT(-0.69397146089),
    DSP32_Q_CONVERT(0.40804416286), DSP32_Q_CONVERT(-0.91296219043),
    DSP32_Q_CONVERT(0.71787004506), DSP32_Q_CONVERT(-0.69617713149),
    DSP32_Q_CONVERT(0.40383845757), DSP32_Q_CONVERT(-0.91483031224),
    DSP32_Q_CONVERT(0.71573082528), DSP32_Q_CONVERT(-0.69837624941),
    DSP32_Q_CONVERT(0.39962419985), DSP32_Q_CONVERT(-0.91667905992),
    DSP32_Q_CONVERT(0.71358486878), DSP32_Q_CONVERT(-0.70056879394),
    DSP32_Q_CONVERT(0.39540147895), DSP32_Q_CONVERT(-0.91850839433),
    DSP32_Q_CONVERT(0.71143219575), DSP32_Q_CONVERT(-0.70275474446),
    DSP32_Q_CONVERT(0.39117038430), DSP32_Q_CONVERT(-0.92031827671),
    DSP32_Q_CONVERT(0.70927282644), DSP32_Q_CONVERT(-0.70493408038),
    DSP32_Q_CONVERT(0.38693100551), DSP32_Q_CONVERT(-0.92210866874),
    DSP32_Q_CONVERT(0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(0.70493408038), DSP32_Q_CONVERT(-0.70927282644),
    DSP32_Q_CONVERT(0.37842775481), DSP32_Q_CONVERT(-0.92563083051),
    DSP32_Q_CONVERT(0.70275474446), DSP32_Q_CONVERT(-0.71143219575),
    DSP32_Q_CONVERT(0.37416406297), DSP32_Q_CONVERT(-0.92736252565),
    DSP32_Q_CONVERT(0.70056879394), DSP32_Q_CONVERT(-0.71358486878),
    DSP32_Q_CONVERT(0.36989244715), DSP32_Q_CONVERT(-0.92907458126),
    DSP32_Q_CONVERT(0.69837624941), DSP32_Q_CONVERT(-0.71573082528),
    DSP32_Q_CONVERT(0.36561299780), DSP32_Q_CONVERT(-0.93076696108),
    DSP32_Q_CONVERT(0.69617713149), DSP32_Q_CONVERT(-0.71787004506),
    DSP32_Q_CONVERT(0.36132580557), DSP32_Q_CONVERT(-0.93243962927),
    DSP32_Q_CONVERT(0.69397146089), DSP32_Q_CONVERT(-0.72000250796),
    DSP32_Q_CONVERT(0.35703096123), DSP32_Q_CONVERT(-0.93409255040),
    DSP32_Q_CONVERT(0.69175925836), DSP32_Q_CONVERT(-0.72212819393),
    DSP32_Q_CONVERT(0.35272855576), DSP32_Q_CONVERT(-0.93572568948),
    DSP32_Q_CONVERT(0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(0.34841868025), DSP32_Q_CONVERT(-0.93733901191),
    DSP32_Q_CONVERT(0.68731534089), DSP32_Q_CONVERT(-0.72635915508),
    DSP32_Q_CONVERT(0.34410142599), DSP32_Q_CONVERT(-0.93893248353),
    DSP32_Q_CONVERT(0.68508366777), DSP32_Q_CONVERT(-0.72846439045),
    DSP32_Q_CONVERT(0.33977688441), DSP32_Q_CONVERT(-0.94050607059),
    DSP32_Q_CONVERT(0.68284554639), DSP32_Q_CONVERT(-0.73056276923),
    DSP32_Q_CONVERT(0.33544514708), DSP32_Q_CONVERT(-0.94205973977),
    DSP32_Q_CONVERT(0.68060099780), DSP32_Q_CONVERT(-0.73265427167),
    DSP32_Q_CONVERT(0.33110630576), DSP32_Q_CONVERT(-0.94359345816),
    DSP32_Q_CONVERT(0.67835004313), DSP32_Q_CONVERT(-0.73473887810),
    DSP32_Q_CONVERT(0.32676045232), DSP32_Q_CONVERT(-0.94510719329),
    DSP32_Q_CONVERT(0.67609270358), DSP32_Q_CONVERT(-0.73681656888),
    DSP32_Q_CONVERT(0.32240767880), DSP32_Q_CONVERT(-0.94660091308),
    DSP32_Q_CONVERT(0.67382900038), DSP32_Q_CONVERT(-0.73888732446),
    DSP32_Q_CONVERT(0.31804807739), DSP32_Q_CONVERT(-0.94807458592),
    DSP32_Q_CONVERT(0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(0.66928258835), DSP32_Q_CONVERT(-0.74300795214),
    DSP32_Q_CONVERT(0.30930876031), DSP32_Q_CONVERT(-0.95096166631),
    DSP32_Q_CONVERT(0.66699992230), DSP32_Q_CONVERT(-0.74505778544),
    DSP32_Q_CONVERT(0.30492922974), DSP32_Q_CONVERT(-0.95237501272),
    DSP32_Q_CONVERT(0.66471097820), DSP32_Q_CONVERT(-0.74710060598),
    DSP32_Q_CONVERT(0.30054324142), DSP32_Q_CONVERT(-0.95376818989),
    DSP32_Q_CONVERT(0.66241577759), DSP32_Q_CONVERT(-0.74913639452),
    DSP32_Q_CONVERT(0.29615088824), DSP32_Q_CONVERT(-0.95514116831),
    DSP32_Q_CONVERT(0.66011434207), DSP32_Q_CONVERT(-0.75116513191),
    DSP32_Q_CONVERT(0.29175226323), DSP32_Q_CONVERT(-0.95649391890),
    DSP32_Q_CONVERT(0.65780669330), DSP32_Q_CONVERT(-0.75318679904),
    DSP32_Q_CONVERT(0.28734745954), DSP32_Q_CONVERT(-0.95782641303),
    DSP32_Q_CONVERT(0.65549285300), DSP32_Q_CONVERT(-0.75520137690),
    DSP32_Q_CONVERT(0.28293657046), DSP32_Q_CONVERT(-0.95913862246),
    DSP32_Q_CONVERT(0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(0.27851968939), DSP32_Q_CONVERT(-0.96043051942),
    DSP32_Q_CONVERT(0.65084668500), DSP32_Q_CONVERT(-0.75920918898),
    DSP32_Q_CONVERT(0.27409690987), DSP32_Q_CONVERT(-0.96170207653),
    DSP32_Q_CONVERT(0.64851440102), DSP32_Q_CONVERT(-0.76120238548),
    DSP32_Q_CONVERT(0.26966832557), DSP32_Q_CONVERT(-0.96295326687),
    DSP32_Q_CONVERT(0.64617601298), DSP32_Q_CONVERT(-0.76318841726),
    DSP32_Q_CONVERT(0.26523403029), DSP32_Q_CONVERT(-0.96418406395),
    DSP32_Q_CONVERT(0.64383154289), DSP32_Q_CONVERT(-0.76516726562),
    DSP32_Q_CONVERT(0.26079411792), DSP32_Q_CONVERT(-0.96539444170),
    DSP32_Q_CONVERT(0.64148101281), DSP32_Q_CONVERT(-0.76713891194),
    DSP32_Q_CONVERT(0.25634868249), DSP32_Q_CONVERT(-0.96658437448),
    DSP32_Q_CONVERT(0.63912444486), DSP32_Q_CONVERT(-0.76910333765),
    DSP32_Q_CONVERT(0.25189781815), DSP32_Q_CONVERT(-0.96775383709),
    DSP32_Q_CONVERT(0.63676186124), DSP32_Q_CONVERT(-0.77106052426),
    DSP32_Q_CONVERT(0.24744161917), DSP32_Q_CONVERT(-0.96890280478),
    DSP32_Q_CONVERT(0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(0.63201873594), DSP32_Q_CONVERT(-0.77495310659),
    DSP32_Q_CONVERT(0.23851359484), DSP32_Q_CONVERT(-0.97113915845),
    DSP32_Q_CONVERT(0.62963823891), DSP32_Q_CONVERT(-0.77688846567),
    DSP32_Q_CONVERT(0.23404195858), DSP32_Q_CONVERT(-0.97222649708),
    DSP32_Q_CONVERT(0.62725181550), DSP32_Q_CONVERT(-0.77881651238),
    DSP32_Q_CONVERT(0.22956536582), DSP32_Q_CONVERT(-0.97329324605),
    DSP32_Q_CONVERT(0.62485948814), DSP32_Q_CONVERT(-0.78073722857),
    DSP32_Q_CONVERT(0.22508391136), DSP32_Q_CONVERT(-0.97433938279),
    DSP32_Q_CONVERT(0.62246127937), DSP32_Q_CONVERT(-0.78265059617),
    DSP32_Q_CONVERT(0.22059769011), DSP32_Q_CONVERT(-0.97536488512),
    DSP32_Q_CONVERT(0.62005721176), DSP32_Q_CONVERT(-0.78455659716),
    DSP32_Q_CONVERT(0.21610679708), DSP32_Q_CONVERT(-0.97636973133),
    DSP32_Q_CONVERT(0.61764730794), DSP32_Q_CONVERT(-0.78645521360),
    DSP32_Q_CONVERT(0.21161132737), DSP32_Q_CONVERT(-0.97735390015),
    DSP32_Q_CONVERT(0.61523159058), DSP32_Q_CONVERT(-0.78834642763),
    DSP32_Q_CONVERT(0.20711137619), DSP32_Q_CONVERT(-0.97831737072),
    DSP32_Q_CONVERT(0.61281008243), DSP32_Q_CONVERT(-0.79023022144),
    DSP32_Q_CONVERT(0.20260703884), DSP32_Q_CONVERT(-0.97926012265),
    DSP32_Q_CONVERT(0.61038280628), DSP32_Q_CONVERT(-0.79210657730),
    DSP32_Q_CONVERT(0.19809841072), DSP32_Q_CONVERT(-0.98018213597),
    DSP32_Q_CONVERT(0.60794978497), DSP32_Q_CONVERT(-0.79397547755),
    DSP32_Q_CONVERT(0.19358558730), DSP32_Q_CONVERT(-0.98108339115),
    DSP32_Q_CONVERT(0.60551104140), DSP32_Q_CONVERT(-0.79583690461),
    DSP32_Q_CONVERT(0.18906866415), DSP32_Q_CONVERT(-0.98196386911),
    DSP32_Q_CONVERT(0.60306659854), DSP32_Q_CONVERT(-0.79769084094),
    DSP32_Q_CONVERT(0.18454773694), DSP32_Q_CONVERT(-0.98282355120),
    DSP32_Q_CONVERT(0.60061647938), DSP32_Q_CONVERT(-0.79953726911),
    DSP32_Q_CONVERT(0.18002290141), DSP32_Q_CONVERT(-0.98366241921),
    DSP32_Q_CONVERT(0.59816070700), DSP32_Q_CONVERT(-0.80137617172),
    DSP32_Q_CONVERT(0.17549425338), DSP32_Q_CONVERT(-0.98448045538),
    DSP32_Q_CONVERT(0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(0.59323229504), DSP32_Q_CONVERT(-0.80503133114),
    DSP32_Q_CONVERT(0.16642590354), DSP32_Q_CONVERT(-0.98605396335),
    DSP32_Q_CONVERT(0.59075970186), DSP32_Q_CONVERT(-0.80684755354),
    DSP32_Q_CONVERT(0.16188639378), DSP32_Q_CONVERT(-0.98680940181),
    DSP32_Q_CONVERT(0.58828154822), DSP32_Q_CONVERT(-0.80865618159),
    DSP32_Q_CONVERT(0.15734345562), DSP32_Q_CONVERT(-0.98754394179),
    DSP32_Q_CONVERT(0.58579785746), DSP32_Q_CONVERT(-0.81045719825),
    DSP32_Q_CONVERT(0.15279718526), DSP32_Q_CONVERT(-0.98825756773),
    DSP32_Q_CONVERT(0.58330865294), DSP32_Q_CONVERT(-0.81225058659),
    DSP32_Q_CONVERT(0.14824767899), DSP32_Q_CONVERT(-0.98895026451),
    DSP32_Q_CONVERT(0.58081395810), DSP32_Q_CONVERT(-0.81403632971),
    DSP32_Q_CONVERT(0.14369503315), DSP32_Q_CONVERT(-0.98962201746),
    DSP32_Q_CONVERT(0.57831379641), DSP32_Q_CONVERT(-0.81581441081),
    DSP32_Q_CONVERT(0.13913934416), DSP32_Q_CONVERT(-0.99027281236),
    DSP32_Q_CONVERT(0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(0.13458070851), DSP32_Q_CONVERT(-0.99090263543),
    DSP32_Q_CONVERT(0.57329716670), DSP32_Q_CONVERT(-0.81934752008),
    DSP32_Q_CONVERT(0.13001922272), DSP32_Q_CONVERT(-0.99151147332),
    DSP32_Q_CONVERT(0.57078074589), DSP32_Q_CONVERT(-0.82110251499),
    DSP32_Q_CONVERT(0.12545498341), DSP32_Q_CONVERT(-0.99209931314),
    DSP32_Q_CONVERT(0.56825895267), DSP32_Q_CONVERT(-0.82284978138),
    DSP32_Q_CONVERT(0.12088808724), DSP32_Q_CONVERT(-0.99266614245),
    DSP32_Q_CONVERT(0.56573181078), DSP32_Q_CONVERT(-0.82458930279),
    DSP32_Q_CONVERT(0.11631863091), DSP32_Q_CONVERT(-0.99321194923),
    DSP32_Q_CONVERT(0.56319934401), DSP32_Q_CONVERT(-0.82632106285),
    DSP32_Q_CONVERT(0.11174671121), DSP32_Q_CONVERT(-0.99373672194),
    DSP32_Q_CONVERT(0.56066157620), DSP32_Q_CONVERT(-0.82804504526),
    DSP32_Q_CONVERT(0.10717242496), DSP32_Q_CONVERT(-0.99424044945),
    DSP32_Q_CONVERT(0.55811853122), DSP32_Q_CONVERT(-0.82976123379),
    DSP32_Q_CONVERT(0.10259586902), DSP32_Q_CONVERT(-0.99472312110),
    DSP32_Q_CONVERT(0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(0.55301670558), DSP32_Q_CONVERT(-0.83317016470),
    DSP32_Q_CONVERT(0.09343633585), DSP32_Q_CONVERT(-0.99562525638),
    DSP32_Q_CONVERT(0.55045797294), DSP32_Q_CONVERT(-0.83486287499),
    DSP32_Q_CONVERT(0.08885355258), DSP32_Q_CONVERT(-0.99604470090),
    DSP32_Q_CONVERT(0.54789405917), DSP32_Q_CONVERT(-0.83654772722),
    DSP32_Q_CONVERT(0.08426888759), DSP32_Q_CONVERT(-0.99644305135),
    DSP32_Q_CONVERT(0.54532498842), DSP32_Q_CONVERT(-0.83822470555),
    DSP32_Q_CONVERT(0.07968243797), DSP32_Q_CONVERT(-0.99682029929),
    DSP32_Q_CONVERT(0.54275078486), DSP32_Q_CONVERT(-0.83989379420),
    DSP32_Q_CONVERT(0.07509430085), DSP32_Q_CONVERT(-0.99717643674),
    DSP32_Q_CONVERT(0.54017147273), DSP32_Q_CONVERT(-0.84155497744),
    DSP32_Q_CONVERT(0.07050457339), DSP32_Q_CONVERT(-0.99751145614),
    DSP32_Q_CONVERT(0.53758707630), DSP32_Q_CONVERT(-0.84320823964),
    DSP32_Q_CONVERT(0.06591335280), DSP32_Q_CONVERT(-0.99782535041),
    DSP32_Q_CONVERT(0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(0.06132073630), DSP32_Q_CONVERT(-0.99811811290),
    DSP32_Q_CONVERT(0.53240312788), DSP32_Q_CONVERT(-0.84649093877),
    DSP32_Q_CONVERT(0.05672682117), DSP32_Q_CONVERT(-0.99838973741),
    DSP32_Q_CONVERT(0.52980362469), DSP32_Q_CONVERT(-0.84812034480),
    DSP32_Q_CONVERT(0.05213170468), DSP32_Q_CONVERT(-0.99864021818),
    DSP32_Q_CONVERT(0.52719913478), DSP32_Q_CONVERT(-0.84974176800),
    DSP32_Q_CONVERT(0.04753548416), DSP32_Q_CONVERT(-0.99886954991),
    DSP32_Q_CONVERT(0.52458968268), DSP32_Q_CONVERT(-0.85135519311),
    DSP32_Q_CONVERT(0.04293825693), DSP32_Q_CONVERT(-0.99907772775),
    DSP32_Q_CONVERT(0.52197529294), DSP32_Q_CONVERT(-0.85296060493),
    DSP32_Q_CONVERT(0.03834012037), DSP32_Q_CONVERT(-0.99926474729),
    DSP32_Q_CONVERT(0.51935599017), DSP32_Q_CONVERT(-0.85455798837),
    DSP32_Q_CONVERT(0.03374117185), DSP32_Q_CONVERT(-0.99943060456),
    DSP32_Q_CONVERT(0.51673179902), DSP32_Q_CONVERT(-0.85614732838),
    DSP32_Q_CONVERT(0.02914150876), DSP32_Q_CONVERT(-0.99957529605),
    DSP32_Q_CONVERT(0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(0.51146885044), DSP32_Q_CONVERT(-0.85930181836),
    DSP32_Q_CONVERT(0.01994042855), DSP32_Q_CONVERT(-0.99980116989),
    DSP32_Q_CONVERT(0.50883014254), DSP32_Q_CONVERT(-0.86086693864),
    DSP32_Q_CONVERT(0.01533920628), DSP32_Q_CONVERT(-0.99988234745),
    DSP32_Q_CONVERT(0.50618664535), DSP32_Q_CONVERT(-0.86242395611),
    DSP32_Q_CONVERT(0.01073765917), DSP32_Q_CONVERT(-0.99994234968),
    DSP32_Q_CONVERT(0.50353838373), DSP32_Q_CONVERT(-0.86397285612),
    DSP32_Q_CONVERT(0.00613588465), DSP32_Q_CONVERT(-0.99998117528),
    DSP32_Q_CONVERT(0.50088538261), DSP32_Q_CONVERT(-0.86551362409),
    DSP32_Q_CONVERT(0.00153398019), DSP32_Q_CONVERT(-0.99999882345),
    DSP32_Q_CONVERT(0.49822766697), DSP32_Q_CONVERT(-0.86704624552),
    DSP32_Q_CONVERT(-0.00306795676), DSP32_Q_CONVERT(-0.99999529381),
    DSP32_Q_CONVERT(0.49556526183), DSP32_Q_CONVERT(-0.86857070597),
    DSP32_Q_CONVERT(-0.00766982874), DSP32_Q_CONVERT(-0.99997058643),
    DSP32_Q_CONVERT(0.49289819223), DSP32_Q_CONVERT(-0.87008699111),
    DSP32_Q_CONVERT(-0.01227153829), DSP32_Q_CONVERT(-0.99992470184),
    DSP32_Q_CONVERT(0.49022648329), DSP32_Q_CONVERT(-0.87159508666),
    DSP32_Q_CONVERT(-0.01687298795), DSP32_Q_CONVERT(-0.99985764101),
    DSP32_Q_CONVERT(0.48755016015), DSP32_Q_CONVERT(-0.87309497842),
    DSP32_Q_CONVERT(-0.02147408028), DSP32_Q_CONVERT(-0.99976940535),
    DSP32_Q_CONVERT(0.48486924800), DSP32_Q_CONVERT(-0.87458665228),
    DSP32_Q_CONVERT(-0.02607471783), DSP32_Q_CONVERT(-0.99965999674),
    DSP32_Q_CONVERT(0.48218377208), DSP32_Q_CONVERT(-0.87607009420),
    DSP32_Q_CONVERT(-0.03067480318), DSP32_Q_CONVERT(-0.99952941750),
    DSP32_Q_CONVERT(0.47949375766), DSP32_Q_CONVERT(-0.87754529021),
    DSP32_Q_CONVERT(-0.03527423890), DSP32_Q_CONVERT(-0.99937767039),
    DSP32_Q_CONVERT(0.47679923006), DSP32_Q_CONVERT(-0.87901222643),
    DSP32_Q_CONVERT(-0.03987292759), DSP32_Q_CONVERT(-0.99920475862),
    DSP32_Q_CONVERT(0.47410021465), DSP32_Q_CONVERT(-0.88047088905),
    DSP32_Q_CONVERT(-0.04447077185), DSP32_Q_CONVERT(-0.99901068585),
    DSP32_Q_CONVERT(0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(-0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(0.46868882204), DSP32_Q_CONVERT(-0.88336333867),
    DSP32_Q_CONVERT(-0.05366353765), DSP32_Q_CONVERT(-0.99855907423),
    DSP32_Q_CONVERT(0.46597649577), DSP32_Q_CONVERT(-0.88479709843),
    DSP32_Q_CONVERT(-0.05825826450), DSP32_Q_CONVERT(-0.99830154493),
    DSP32_Q_CONVERT(0.46325978355), DSP32_Q_CONVERT(-0.88622253015),
    DSP32_Q_CONVERT(-0.06285175756), DSP32_Q_CONVERT(-0.99802287377),
    DSP32_Q_CONVERT(0.46053871096), DSP32_Q_CONVERT(-0.88763962040),
    DSP32_Q_CONVERT(-0.06744391956), DSP32_Q_CONVERT(-0.99772306664),
    DSP32_Q_CONVERT(0.45781330360), DSP32_Q_CONVERT(-0.88904835585),
    DSP32_Q_CONVERT(-0.07203465325), DSP32_Q_CONVERT(-0.99740212990),
    DSP32_Q_CONVERT(0.45508358713), DSP32_Q_CONVERT(-0.89044872324),
    DSP32_Q_CONVERT(-0.07662386139), DSP32_Q_CONVERT(-0.99706007034),
    DSP32_Q_CONVERT(0.45234958723), DSP32_Q_CONVERT(-0.89184070939),
    DSP32_Q_CONVERT(-0.08121144681), DSP32_Q_CONVERT(-0.99669689520),
    DSP32_Q_CONVERT(0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(-0.08579731234), DSP32_Q_CONVERT(-0.99631261218),
    DSP32_Q_CONVERT(0.44686884016), DSP32_Q_CONVERT(-0.89459948563),
    DSP32_Q_CONVERT(-0.09038136088), DSP32_Q_CONVERT(-0.99590722942),
    DSP32_Q_CONVERT(0.44412214457), DSP32_Q_CONVERT(-0.89596624976),
    DSP32_Q_CONVERT(-0.09496349533), DSP32_Q_CONVERT(-0.99548075549),
    DSP32_Q_CONVERT(0.44137126873), DSP32_Q_CONVERT(-0.89732458071),
    DSP32_Q_CONVERT(-0.09954361866), DSP32_Q_CONVERT(-0.99503319944),
    DSP32_Q_CONVERT(0.43861623854), DSP32_Q_CONVERT(-0.89867446569),
    DSP32_Q_CONVERT(-0.10412163387), DSP32_Q_CONVERT(-0.99456457073),
    DSP32_Q_CONVERT(0.43585707992), DSP32_Q_CONVERT(-0.90001589202),
    DSP32_Q_CONVERT(-0.10869744401), DSP32_Q_CONVERT(-0.99407487930),
    DSP32_Q_CONVERT(0.43309381885), DSP32_Q_CONVERT(-0.90134884705),
    DSP32_Q_CONVERT(-0.11327095218), DSP32_Q_CONVERT(-0.99356413552),
    DSP32_Q_CONVERT(0.43032648134), DSP32_Q_CONVERT(-0.90267331824),
    DSP32_Q_CONVERT(-0.11784206151), DSP32_Q_CONVERT(-0.99303235020),
    DSP32_Q_CONVERT(0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(-0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(0.42477968121), DSP32_Q_CONVERT(-0.90529675932),
    DSP32_Q_CONVERT(-0.12697669650), DSP32_Q_CONVERT(-0.99190570043),
    DSP32_Q_CONVERT(0.42200027080), DSP32_Q_CONVERT(-0.90659570451),
    DSP32_Q_CONVERT(-0.13154002870), DSP32_Q_CONVERT(-0.99131085985),
    DSP32_Q_CONVERT(0.41921688836), DSP32_Q_CONVERT(-0.90788611649),
    DSP32_Q_CONVERT(-0.13610057518), DSP32_Q_CONVERT(-0.99069502544),
    DSP32_Q_CONVERT(0.41642956010), DSP32_Q_CONVERT(-0.90916798309),
    DSP32_Q_CONVERT(-0.14065823933), DSP32_Q_CONVERT(-0.99005821026),
    DSP32_Q_CONVERT(0.41363831224), DSP32_Q_CONVERT(-0.91044129226),
    DSP32_Q_CONVERT(-0.14521292465), DSP32_Q_CONVERT(-0.98940042779),
    DSP32_Q_CONVERT(0.41084317106), DSP32_Q_CONVERT(-0.91170603201),
    DSP32_Q_CONVERT(-0.14976453468), DSP32_Q_CONVERT(-0.98872169196),
    DSP32_Q_CONVERT(0.40804416286), DSP32_Q_CONVERT(-0.91296219043),
    DSP32_Q_CONVERT(-0.15431297301), DSP32_Q_CONVERT(-0.98802201714),
    DSP32_Q_CONVERT(0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(-0.15885814333), DSP32_Q_CONVERT(-0.98730141816),
    DSP32_Q_CONVERT(0.40243465086), DSP32_Q_CONVERT(-0.91544871609),
    DSP32_Q_CONVERT(-0.16339994938), DSP32_Q_CONVERT(-0.98655991026),
    DSP32_Q_CONVERT(0.39962419985), DSP32_Q_CONVERT(-0.91667905992),
    DSP32_Q_CONVERT(-0.16793829497), DSP32_Q_CONVERT(-0.98579750917),
    DSP32_Q_CONVERT(0.39680998742), DSP32_Q_CONVERT(-0.91790077562),
    DSP32_Q_CONVERT(-0.17247308400), DSP32_Q_CONVERT(-0.98501423101),
    DSP32_Q_CONVERT(0.39399204006), DSP32_Q_CONVERT(-0.91911385169),
    DSP32_Q_CONVERT(-0.17700422041), DSP32_Q_CONVERT(-0.98421009239),
    DSP32_Q_CONVERT(0.39117038430), DSP32_Q_CONVERT(-0.92031827671),
    DSP32_Q_CONVERT(-0.18153160826), DSP32_Q_CONVERT(-0.98338511032),
    DSP32_Q_CONVERT(0.38834504670), DSP32_Q_CONVERT(-0.92151403934),
    DSP32_Q_CONVERT(-0.18605515166), DSP32_Q_CONVERT(-0.98253930229),
    DSP32_Q_CONVERT(0.38551605384), DSP32_Q_CONVERT(-0.92270112833),
    DSP32_Q_CONVERT(-0.19057475482), DSP32_Q_CONVERT(-0.98167268620),
    DSP32_Q_CONVERT(0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(-0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(0.37984720892), DSP32_Q_CONVERT(-0.92504924078),
    DSP32_Q_CONVERT(-0.19960175762), DSP32_Q_CONVERT(-0.97987710370),
    DSP32_Q_CONVERT(0.37700741022), DSP32_Q_CONVERT(-0.92621024214),
    DSP32_Q_CONVERT(-0.20410896609), DSP32_Q_CONVERT(-0.97894817532),
    DSP32_Q_CONVERT(0.37416406297), DSP32_Q_CONVERT(-0.92736252565),
    DSP32_Q_CONVERT(-0.20861185198), DSP32_Q_CONVERT(-0.97799851493),
    DSP32_Q_CONVERT(0.37131719395), DSP32_Q_CONVERT(-0.92850608047),
    DSP32_Q_CONVERT(-0.21311031992), DSP32_Q_CONVERT(-0.97702814266),
    DSP32_Q_CONVERT(0.36846682995), DSP32_Q_CONVERT(-0.92964089584),
    DSP32_Q_CONVERT(-0.21760427464), DSP32_Q_CONVERT(-0.97603707904),
    DSP32_Q_CONVERT(0.36561299780), DSP32_Q_CONVERT(-0.93076696108),
    DSP32_Q_CONVERT(-0.22209362097), DSP32_Q_CONVERT(-0.97502534507),
    DSP32_Q_CONVERT(0.36275572437), DSP32_Q_CONVERT(-0.93188426558),
    DSP32_Q_CONVERT(-0.22657826385), DSP32_Q_CONVERT(-0.97399296217),
    DSP32_Q_CONVERT(0.35989503653), DSP32_Q_CONVERT(-0.93299279883),
    DSP32_Q_CONVERT(-0.23105810828), DSP32_Q_CONVERT(-0.97293995221),
    DSP32_Q_CONVERT(0.35703096123), DSP32_Q_CONVERT(-0.93409255040),
    DSP32_Q_CONVERT(-0.23553305940), DSP32_Q_CONVERT(-0.97186633748),
    DSP32_Q_CONVERT(0.35416352542), DSP32_Q_CONVERT(-0.93518350994),
    DSP32_Q_CONVERT(-0.24000302245), DSP32_Q_CONVERT(-0.97077214073),
    DSP32_Q_CONVERT(0.35129275609), DSP32_Q_CONVERT(-0.93626566717),
    DSP32_Q_CONVERT(-0.24446790275), DSP32_Q_CONVERT(-0.96965738512),
    DSP32_Q_CONVERT(0.34841868025), DSP32_Q_CONVERT(-0.93733901191),
    DSP32_Q_CONVERT(-0.24892760575), DSP32_Q_CONVERT(-0.96852209427),
    DSP32_Q_CONVERT(0.34554132496), DSP32_Q_CONVERT(-0.93840353406),
    DSP32_Q_CONVERT(-0.25338203700), DSP32_Q_CONVERT(-0.96736629222),
    DSP32_Q_CONVERT(0.34266071731), DSP32_Q_CONVERT(-0.93945922360),
    DSP32_Q_CONVERT(-0.25783110216), DSP32_Q_CONVERT(-0.96619000345),
    DSP32_Q_CONVERT(0.33977688441), DSP32_Q_CONVERT(-0.94050607059),
    DSP32_Q_CONVERT(-0.26227470702), DSP32_Q_CONVERT(-0.96499325285),
    DSP32_Q_CONVERT(0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(-0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(0.33399965144), DSP32_Q_CONVERT(-0.94257319760),
    DSP32_Q_CONVERT(-0.27114515953), DSP32_Q_CONVERT(-0.96253846804),
    DSP32_Q_CONVERT(0.33110630576), DSP32_Q_CONVERT(-0.94359345816),
    DSP32_Q_CONVERT(-0.27557181931), DSP32_Q_CONVERT(-0.96128048581),
    DSP32_Q_CONVERT(0.32820984358), DSP32_Q_CONVERT(-0.94460483726),
    DSP32_Q_CONVERT(-0.27999264308), DSP32_Q_CONVERT(-0.96000214574),
    DSP32_Q_CONVERT(0.32531029216), DSP32_Q_CONVERT(-0.94560732538),
    DSP32_Q_CONVERT(-0.28440753721), DSP32_Q_CONVERT(-0.95870347490),
    DSP32_Q_CONVERT(0.32240767880), DSP32_Q_CONVERT(-0.94660091308),
    DSP32_Q_CONVERT(-0.28881640821), DSP32_Q_CONVERT(-0.95738450079),
    DSP32_Q_CONVERT(0.31950203082), DSP32_Q_CONVERT(-0.94758559102),
    DSP32_Q_CONVERT(-0.29321916269), DSP32_Q_CONVERT(-0.95604525135),
    DSP32_Q_CONVERT(0.31659337556), DSP32_Q_CONVERT(-0.94856134992),
    DSP32_Q_CONVERT(-0.29761570744), DSP32_Q_CONVERT(-0.95468575494),
    DSP32_Q_CONVERT(0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(-0.30200594932), DSP32_Q_CONVERT(-0.95330604035),
    DSP32_Q_CONVERT(0.31076715275), DSP32_Q_CONVERT(-0.95048607395),
    DSP32_Q_CONVERT(-0.30638979537), DSP32_Q_CONVERT(-0.95190613681),
    DSP32_Q_CONVERT(0.30784964004), DSP32_Q_CONVERT(-0.95143502097),
    DSP32_Q_CONVERT(-0.31076715275), DSP32_Q_CONVERT(-0.95048607395),
    DSP32_Q_CONVERT(0.30492922974), DSP32_Q_CONVERT(-0.95237501272),
    DSP32_Q_CONVERT(-0.31513792875), DSP32_Q_CONVERT(-0.94904588185),
    DSP32_Q_CONVERT(0.30200594932), DSP32_Q_CONVERT(-0.95330604035),
    DSP32_Q_CONVERT(-0.31950203082), DSP32_Q_CONVERT(-0.94758559102),
    DSP32_Q_CONVERT(0.29907982631), DSP32_Q_CONVERT(-0.95422809511),
    DSP32_Q_CONVERT(-0.32385936652), DSP32_Q_CONVERT(-0.94610523237),
    DSP32_Q_CONVERT(0.29615088824), DSP32_Q_CONVERT(-0.95514116831),
    DSP32_Q_CONVERT(-0.32820984358), DSP32_Q_CONVERT(-0.94460483726),
    DSP32_Q_CONVERT(0.29321916269), DSP32_Q_CONVERT(-0.95604525135),
    DSP32_Q_CONVERT(-0.33255336987), DSP32_Q_CONVERT(-0.94308443747),
    DSP32_Q_CONVERT(0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(-0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(0.28734745954), DSP32_Q_CONVERT(-0.95782641303),
    DSP32_Q_CONVERT(-0.34121920232), DSP32_Q_CONVERT(-0.93998375303),
    DSP32_Q_CONVERT(0.28440753721), DSP32_Q_CONVERT(-0.95870347490),
    DSP32_Q_CONVERT(-0.34554132496), DSP32_Q_CONVERT(-0.93840353406),
    DSP32_Q_CONVERT(0.28146493793), DSP32_Q_CONVERT(-0.95957151308),
    DSP32_Q_CONVERT(-0.34985612979), DSP32_Q_CONVERT(-0.93680344174),
    DSP32_Q_CONVERT(0.27851968939), DSP32_Q_CONVERT(-0.96043051942),
    DSP32_Q_CONVERT(-0.35416352542), DSP32_Q_CONVERT(-0.93518350994),
    DSP32_Q_CONVERT(0.27557181931), DSP32_Q_CONVERT(-0.96128048581),
    DSP32_Q_CONVERT(-0.35846342063), DSP32_Q_CONVERT(-0.93354377298),
    DSP32_Q_CONVERT(0.27262135545), DSP32_Q_CONVERT(-0.96212140427),
    DSP32_Q_CONVERT(-0.36275572437), DSP32_Q_CONVERT(-0.93188426558),
    DSP32_Q_CONVERT(0.26966832557), DSP32_Q_CONVERT(-0.96295326687),
    DSP32_Q_CONVERT(-0.36704034572), DSP32_Q_CONVERT(-0.93020502289),
    DSP32_Q_CONVERT(0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(-0.37131719395), DSP32_Q_CONVERT(-0.92850608047),
    DSP32_Q_CONVERT(0.26375467897), DSP32_Q_CONVERT(-0.96458979329),
    DSP32_Q_CONVERT(-0.37558617849), DSP32_Q_CONVERT(-0.92678747430),
    DSP32_Q_CONVERT(0.26079411792), DSP32_Q_CONVERT(-0.96539444170),
    DSP32_Q_CONVERT(-0.37984720892), DSP32_Q_CONVERT(-0.92504924078),
    DSP32_Q_CONVERT(0.25783110216), DSP32_Q_CONVERT(-0.96619000345),
    DSP32_Q_CONVERT(-0.38410019502), DSP32_Q_CONVERT(-0.92329141672),
    DSP32_Q_CONVERT(0.25486565960), DSP32_Q_CONVERT(-0.96697647104),
    DSP32_Q_CONVERT(-0.38834504670), DSP32_Q_CONVERT(-0.92151403934),
    DSP32_Q_CONVERT(0.25189781815), DSP32_Q_CONVERT(-0.96775383709),
    DSP32_Q_CONVERT(-0.39258167407), DSP32_Q_CONVERT(-0.91971714629),
    DSP32_Q_CONVERT(0.24892760575), DSP32_Q_CONVERT(-0.96852209427),
    DSP32_Q_CONVERT(-0.39680998742), DSP32_Q_CONVERT(-0.91790077562),
    DSP32_Q_CONVERT(0.24595505034), DSP32_Q_CONVERT(-0.96928123536),
    DSP32_Q_CONVERT(-0.40102989718), DSP32_Q_CONVERT(-0.91606496580),
    DSP32_Q_CONVERT(0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(-0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(0.24000302245), DSP32_Q_CONVERT(-0.97077214073),
    DSP32_Q_CONVERT(-0.40944414869), DSP32_Q_CONVERT(-0.91233518462),
    DSP32_Q_CONVERT(0.23702360599), DSP32_Q_CONVERT(-0.97150389099),
    DSP32_Q_CONVERT(-0.41363831224), DSP32_Q_CONVERT(-0.91044129226),
    DSP32_Q_CONVERT(0.23404195858), DSP32_Q_CONVERT(-0.97222649708),
    DSP32_Q_CONVERT(-0.41782371582), DSP32_Q_CONVERT(-0.90852811872),
    DSP32_Q_CONVERT(0.23105810828), DSP32_Q_CONVERT(-0.97293995221),
    DSP32_Q_CONVERT(-0.42200027080), DSP32_Q_CONVERT(-0.90659570451),
    DSP32_Q_CONVERT(0.22807208317), DSP32_Q_CONVERT(-0.97364424965),
    DSP32_Q_CONVERT(-0.42616788873), DSP32_Q_CONVERT(-0.90464409058),
    DSP32_Q_CONVERT(0.22508391136), DSP32_Q_CONVERT(-0.97433938279),
    DSP32_Q_CONVERT(-0.43032648134), DSP32_Q_CONVERT(-0.90267331824),
    DSP32_Q_CONVERT(0.22209362097), DSP32_Q_CONVERT(-0.97502534507),
    DSP32_Q_CONVERT(-0.43447596057), DSP32_Q_CONVERT(-0.90068342923),
    DSP32_Q_CONVERT(0.21910124016), DSP32_Q_CONVERT(-0.97570213004),
    DSP32_Q_CONVERT(-0.43861623854), DSP32_Q_CONVERT(-0.89867446569),
    DSP32_Q_CONVERT(0.21610679708), DSP32_Q_CONVERT(-0.97636973133),
    DSP32_Q_CONVERT(-0.44274722756), DSP32_Q_CONVERT(-0.89664647018),
    DSP32_Q_CONVERT(0.21311031992), DSP32_Q_CONVERT(-0.97702814266),
    DSP32_Q_CONVERT(-0.44686884016), DSP32_Q_CONVERT(-0.89459948563),
    DSP32_Q_CONVERT(0.21011183688), DSP32_Q_CONVERT(-0.97767735782),
    DSP32_Q_CONVERT(-0.45098098905), DSP32_Q_CONVERT(-0.89253355540),
    DSP32_Q_CONVERT(0.20711137619), DSP32_Q_CONVERT(-0.97831737072),
    DSP32_Q_CONVERT(-0.45508358713), DSP32_Q_CONVERT(-0.89044872324),
    DSP32_Q_CONVERT(0.20410896609), DSP32_Q_CONVERT(-0.97894817532),
    DSP32_Q_CONVERT(-0.45917654752), DSP32_Q_CONVERT(-0.88834503331),
    DSP32_Q_CONVERT(0.20110463484), DSP32_Q_CONVERT(-0.97956976569),
    DSP32_Q_CONVERT(-0.46325978355), DSP32_Q_CONVERT(-0.88622253015),
    DSP32_Q_CONVERT(0.19809841072), DSP32_Q_CONVERT(-0.98018213597),
    DSP32_Q_CONVERT(-0.46733320874), DSP32_Q_CONVERT(-0.88408125871),
    DSP32_Q_CONVERT(0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(-0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(0.19208039705), DSP32_Q_CONVERT(-0.98137919331),
    DSP32_Q_CONVERT(-0.47545028175), DSP32_Q_CONVERT(-0.87974259280),
    DSP32_Q_CONVERT(0.18906866415), DSP32_Q_CONVERT(-0.98196386911),
    DSP32_Q_CONVERT(-0.47949375766), DSP32_Q_CONVERT(-0.87754529021),
    DSP32_Q_CONVERT(0.18605515166), DSP32_Q_CONVERT(-0.98253930229),
    DSP32_Q_CONVERT(-0.48352707893), DSP32_Q_CONVERT(-0.87532940310),
    DSP32_Q_CONVERT(0.18303988796), DSP32_Q_CONVERT(-0.98310548743),
    DSP32_Q_CONVERT(-0.48755016015), DSP32_Q_CONVERT(-0.87309497842),
    DSP32_Q_CONVERT(0.18002290141), DSP32_Q_CONVERT(-0.98366241921),
    DSP32_Q_CONVERT(-0.49156291611), DSP32_Q_CONVERT(-0.87084206347),
    DSP32_Q_CONVERT(0.17700422041), DSP32_Q_CONVERT(-0.98421009239),
    DSP32_Q_CONVERT(-0.49556526183), DSP32_Q_CONVERT(-0.86857070597),
    DSP32_Q_CONVERT(0.17398387339), DSP32_Q_CONVERT(-0.98474850180),
    DSP32_Q_CONVERT(-0.49955711255), DSP32_Q_CONVERT(-0.86628095402),
    DSP32_Q_CONVERT(0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(-0.50353838373), DSP32_Q_CONVERT(-0.86397285612),
    DSP32_Q_CONVERT(0.16793829497), DSP32_Q_CONVERT(-0.98579750917),
    DSP32_Q_CONVERT(-0.50750899105), DSP32_Q_CONVERT(-0.86164646114),
    DSP32_Q_CONVERT(0.16491312049), DSP32_Q_CONVERT(-0.98630809724),
    DSP32_Q_CONVERT(-0.51146885044), DSP32_Q_CONVERT(-0.85930181836),
    DSP32_Q_CONVERT(0.16188639378), DSP32_Q_CONVERT(-0.98680940181),
    DSP32_Q_CONVERT(-0.51541787802), DSP32_Q_CONVERT(-0.85693897742),
    DSP32_Q_CONVERT(0.15885814333), DSP32_Q_CONVERT(-0.98730141816),
    DSP32_Q_CONVERT(-0.51935599017), DSP32_Q_CONVERT(-0.85455798837),
    DSP32_Q_CONVERT(0.15582839765), DSP32_Q_CONVERT(-0.98778414164),
    DSP32_Q_CONVERT(-0.52328310348), DSP32_Q_CONVERT(-0.85215890162),
    DSP32_Q_CONVERT(0.15279718526), DSP32_Q_CONVERT(-0.98825756773),
    DSP32_Q_CONVERT(-0.52719913478), DSP32_Q_CONVERT(-0.84974176800),
    DSP32_Q_CONVERT(0.14976453468), DSP32_Q_CONVERT(-0.98872169196),
    DSP32_Q_CONVERT(-0.53110400115), DSP32_Q_CONVERT(-0.84730663869),
    DSP32_Q_CONVERT(0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(-0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(0.14369503315), DSP32_Q_CONVERT(-0.98962201746),
    DSP32_Q_CONVERT(-0.53887990853), DSP32_Q_CONVERT(-0.84238259964),
    DSP32_Q_CONVERT(0.14065823933), DSP32_Q_CONVERT(-0.99005821026),
    DSP32_Q_CONVERT(-0.54275078486), DSP32_Q_CONVERT(-0.83989379420),
    DSP32_Q_CONVERT(0.13762012159), DSP32_Q_CONVERT(-0.99048508426),
    DSP32_Q_CONVERT(-0.54661016691), DSP32_Q_CONVERT(-0.83738720162),
    DSP32_Q_CONVERT(0.13458070851), DSP32_Q_CONVERT(-0.99090263543),
    DSP32_Q_CONVERT(-0.55045797294), DSP32_Q_CONVERT(-0.83486287499),
    DSP32_Q_CONVERT(0.13154002870), DSP32_Q_CONVERT(-0.99131085985),
    DSP32_Q_CONVERT(-0.55429412145), DSP32_Q_CONVERT(-0.83232086777),
    DSP32_Q_CONVERT(0.12849811079), DSP32_Q_CONVERT(-0.99170975367),
    DSP32_Q_CONVERT(-0.55811853122), DSP32_Q_CONVERT(-0.82976123379),
    DSP32_Q_CONVERT(0.12545498341), DSP32_Q_CONVERT(-0.99209931314),
    DSP32_Q_CONVERT(-0.56193112124), DSP32_Q_CONVERT(-0.82718402727),
    DSP32_Q_CONVERT(0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(-0.56573181078), DSP32_Q_CONVERT(-0.82458930279),
    DSP32_Q_CONVERT(0.11936521481), DSP32_Q_CONVERT(-0.99285041446),
    DSP32_Q_CONVERT(-0.56952051935), DSP32_Q_CONVERT(-0.82197711528),
    DSP32_Q_CONVERT(0.11631863091), DSP32_Q_CONVERT(-0.99321194923),
    DSP32_Q_CONVERT(-0.57329716670), DSP32_Q_CONVERT(-0.81934752008),
    DSP32_Q_CONVERT(0.11327095218), DSP32_Q_CONVERT(-0.99356413552),
    DSP32_Q_CONVERT(-0.57706167286), DSP32_Q_CONVERT(-0.81670057287),
    DSP32_Q_CONVERT(0.11022220729), DSP32_Q_CONVERT(-0.99390697000),
    DSP32_Q_CONVERT(-0.58081395810), DSP32_Q_CONVERT(-0.81403632971),
    DSP32_Q_CONVERT(0.10717242496), DSP32_Q_CONVERT(-0.99424044945),
    DSP32_Q_CONVERT(-0.58455394295), DSP32_Q_CONVERT(-0.81135484702),
    DSP32_Q_CONVERT(0.10412163387), DSP32_Q_CONVERT(-0.99456457073),
    DSP32_Q_CONVERT(-0.58828154822), DSP32_Q_CONVERT(-0.80865618159),
    DSP32_Q_CONVERT(0.10106986275), DSP32_Q_CONVERT(-0.99487933079),
    DSP32_Q_CONVERT(-0.59199669496), DSP32_Q_CONVERT(-0.80594039057),
    DSP32_Q_CONVERT(0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(-0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(0.09496349533), DSP32_Q_CONVERT(-0.99548075549),
    DSP32_Q_CONVERT(-0.59938929840), DSP32_Q_CONVERT(-0.80045766219),
    DSP32_Q_CONVERT(0.09190895650), DSP32_Q_CONVERT(-0.99576741447),
    DSP32_Q_CONVERT(-0.60306659854), DSP32_Q_CONVERT(-0.79769084094),
    DSP32_Q_CONVERT(0.08885355258), DSP32_Q_CONVERT(-0.99604470090),
    DSP32_Q_CONVERT(-0.60673112703), DSP32_Q_CONVERT(-0.79490712633),
    DSP32_Q_CONVERT(0.08579731234), DSP32_Q_CONVERT(-0.99631261218),
    DSP32_Q_CONVERT(-0.61038280628), DSP32_Q_CONVERT(-0.79210657730),
    DSP32_Q_CONVERT(0.08274026455), DSP32_Q_CONVERT(-0.99657114579),
    DSP32_Q_CONVERT(-0.61402155893), DSP32_Q_CONVERT(-0.78928925317),
    DSP32_Q_CONVERT(0.07968243797), DSP32_Q_CONVERT(-0.99682029929),
    DSP32_Q_CONVERT(-0.61764730794), DSP32_Q_CONVERT(-0.78645521360),
    DSP32_Q_CONVERT(0.07662386139), DSP32_Q_CONVERT(-0.99706007034),
    DSP32_Q_CONVERT(-0.62125997651), DSP32_Q_CONVERT(-0.78360451861),
    DSP32_Q_CONVERT(0.07356456360), DSP32_Q_CONVERT(-0.99729045668),
    DSP32_Q_CONVERT(-0.62485948814), DSP32_Q_CONVERT(-0.78073722857),
    DSP32_Q_CONVERT(0.07050457339), DSP32_Q_CONVERT(-0.99751145614),
    DSP32_Q_CONVERT(-0.62844576660), DSP32_Q_CONVERT(-0.77785340421),
    DSP32_Q_CONVERT(0.06744391956), DSP32_Q_CONVERT(-0.99772306664),
    DSP32_Q_CONVERT(-0.63201873594), DSP32_Q_CONVERT(-0.77495310659),
    DSP32_Q_CONVERT(0.06438263093), DSP32_Q_CONVERT(-0.99792528620),
    DSP32_Q_CONVERT(-0.63557832049), DSP32_Q_CONVERT(-0.77203639715),
    DSP32_Q_CONVERT(0.06132073630), DSP32_Q_CONVERT(-0.99811811290),
    DSP32_Q_CONVERT(-0.63912444486), DSP32_Q_CONVERT(-0.76910333765),
    DSP32_Q_CONVERT(0.05825826450), DSP32_Q_CONVERT(-0.99830154493),
    DSP32_Q_CONVERT(-0.64265703397), DSP32_Q_CONVERT(-0.76615399020),
    DSP32_Q_CONVERT(0.05519524435), DSP32_Q_CONVERT(-0.99847558057),
    DSP32_Q_CONVERT(-0.64617601298), DSP32_Q_CONVERT(-0.76318841726),
    DSP32_Q_CONVERT(0.05213170468), DSP32_Q_CONVERT(-0.99864021818),
    DSP32_Q_CONVERT(-0.64968130739), DSP32_Q_CONVERT(-0.76020668165),
    DSP32_Q_CONVERT(0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(-0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(0.04600318213), DSP32_Q_CONVERT(-0.99894129319),
    DSP32_Q_CONVERT(-0.65665054573), DSP32_Q_CONVERT(-0.75419497532),
    DSP32_Q_CONVERT(0.04293825693), DSP32_Q_CONVERT(-0.99907772775),
    DSP32_Q_CONVERT(-0.66011434207), DSP32_Q_CONVERT(-0.75116513191),
    DSP32_Q_CONVERT(0.03987292759), DSP32_Q_CONVERT(-0.99920475862),
    DSP32_Q_CONVERT(-0.66356415861), DSP32_Q_CONVERT(-0.74811938045),
    DSP32_Q_CONVERT(0.03680722294), DSP32_Q_CONVERT(-0.99932238459),
    DSP32_Q_CONVERT(-0.66699992230), DSP32_Q_CONVERT(-0.74505778544),
    DSP32_Q_CONVERT(0.03374117185), DSP32_Q_CONVERT(-0.99943060456),
    DSP32_Q_CONVERT(-0.67042156038), DSP32_Q_CONVERT(-0.74198041172),
    DSP32_Q_CONVERT(0.03067480318), DSP32_Q_CONVERT(-0.99952941750),
    DSP32_Q_CONVERT(-0.67382900038), DSP32_Q_CONVERT(-0.73888732446),
    DSP32_Q_CONVERT(0.02760814578), DSP32_Q_CONVERT(-0.99961882250),
    DSP32_Q_CONVERT(-0.67722217014), DSP32_Q_CONVERT(-0.73577858917),
    DSP32_Q_CONVERT(0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(-0.68060099780), DSP32_Q_CONVERT(-0.73265427167),
    DSP32_Q_CONVERT(0.02147408028), DSP32_Q_CONVERT(-0.99976940535),
    DSP32_Q_CONVERT(-0.68396541180), DSP32_Q_CONVERT(-0.72951443815),
    DSP32_Q_CONVERT(0.01840672991), DSP32_Q_CONVERT(-0.99983058180),
    DSP32_Q_CONVERT(-0.68731534089), DSP32_Q_CONVERT(-0.72635915508),
    DSP32_Q_CONVERT(0.01533920628), DSP32_Q_CONVERT(-0.99988234745),
    DSP32_Q_CONVERT(-0.69065071413), DSP32_Q_CONVERT(-0.72318848931),
    DSP32_Q_CONVERT(0.01227153829), DSP32_Q_CONVERT(-0.99992470184),
    DSP32_Q_CONVERT(-0.69397146089), DSP32_Q_CONVERT(-0.72000250796),
    DSP32_Q_CONVERT(0.00920375478), DSP32_Q_CONVERT(-0.99995764455),
    DSP32_Q_CONVERT(-0.69727751083), DSP32_Q_CONVERT(-0.71680127852),
    DSP32_Q_CONVERT(0.00613588465), DSP32_Q_CONVERT(-0.99998117528),
    DSP32_Q_CONVERT(-0.70056879394), DSP32_Q_CONVERT(-0.71358486878),
    DSP32_Q_CONVERT(0.00306795676), DSP32_Q_CONVERT(-0.99999529381),
    DSP32_Q_CONVERT(-0.70384524052), DSP32_Q_CONVERT(-0.71035334686),
    DSP32_Q_CONVERT(0.00000000000), DSP32_Q_CONVERT(-1.00000000000),
    DSP32_Q_CONVERT(-0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(-0.00306795676), DSP32_Q_CONVERT(-0.99999529381),
    DSP32_Q_CONVERT(-0.71035334686), DSP32_Q_CONVERT(-0.70384524052),
    DSP32_Q_CONVERT(-0.00613588465), DSP32_Q_CONVERT(-0.99998117528),
    DSP32_Q_CONVERT(-0.71358486878), DSP32_Q_CONVERT(-0.70056879394),
    DSP32_Q_CONVERT(-0.00920375478), DSP32_Q_CONVERT(-0.99995764455),
    DSP32_Q_CONVERT(-0.71680127852), DSP32_Q_CONVERT(-0.69727751083),
    DSP32_Q_CONVERT(-0.01227153829), DSP32_Q_CONVERT(-0.99992470184),
    DSP32_Q_CONVERT(-0.72000250796), DSP32_Q_CONVERT(-0.69397146089),
    DSP32_Q_CONVERT(-0.01533920628), DSP32_Q_CONVERT(-0.99988234745),
    DSP32_Q_CONVERT(-0.72318848931), DSP32_Q_CONVERT(-0.69065071413),
    DSP32_Q_CONVERT(-0.01840672991), DSP32_Q_CONVERT(-0.99983058180),
    DSP32_Q_CONVERT(-0.72635915508), DSP32_Q_CONVERT(-0.68731534089),
    DSP32_Q_CONVERT(-0.02147408028), DSP32_Q_CONVERT(-0.99976940535),
    DSP32_Q_CONVERT(-0.72951443815), DSP32_Q_CONVERT(-0.68396541180),
    DSP32_Q_CONVERT(-0.02454122852), DSP32_Q_CONVERT(-0.99969881870),
    DSP32_Q_CONVERT(-0.73265427167), DSP32_Q_CONVERT(-0.68060099780),
    DSP32_Q_CONVERT(-0.02760814578), DSP32_Q_CONVERT(-0.99961882250),
    DSP32_Q_CONVERT(-0.73577858917), DSP32_Q_CONVERT(-0.67722217014),
    DSP32_Q_CONVERT(-0.03067480318), DSP32_Q_CONVERT(-0.99952941750),
    DSP32_Q_CONVERT(-0.73888732446), DSP32_Q_CONVERT(-0.67382900038),
    DSP32_Q_CONVERT(-0.03374117185), DSP32_Q_CONVERT(-0.99943060456),
    DSP32_Q_CONVERT(-0.74198041172), DSP32_Q_CONVERT(-0.67042156038),
    DSP32_Q_CONVERT(-0.03680722294), DSP32_Q_CONVERT(-0.99932238459),
    DSP32_Q_CONVERT(-0.74505778544), DSP32_Q_CONVERT(-0.66699992230),
    DSP32_Q_CONVERT(-0.03987292759), DSP32_Q_CONVERT(-0.99920475862),
    DSP32_Q_CONVERT(-0.74811938045), DSP32_Q_CONVERT(-0.66356415861),
    DSP32_Q_CONVERT(-0.04293825693), DSP32_Q_CONVERT(-0.99907772775),
    DSP32_Q_CONVERT(-0.75116513191), DSP32_Q_CONVERT(-0.66011434207),
    DSP32_Q_CONVERT(-0.04600318213), DSP32_Q_CONVERT(-0.99894129319),
    DSP32_Q_CONVERT(-0.75419497532), DSP32_Q_CONVERT(-0.65665054573),
    DSP32_Q_CONVERT(-0.04906767433), DSP32_Q_CONVERT(-0.99879545621),
    DSP32_Q_CONVERT(-0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(-0.05213170468), DSP32_Q_CONVERT(-0.99864021818),
    DSP32_Q_CONVERT(-0.76020668165), DSP32_Q_CONVERT(-0.64968130739),
    DSP32_Q_CONVERT(-0.05519524435), DSP32_Q_CONVERT(-0.99847558057),
    DSP32_Q_CONVERT(-0.76318841726), DSP32_Q_CONVERT(-0.64617601298),
    DSP32_Q_CONVERT(-0.05825826450), DSP32_Q_CONVERT(-0.99830154493),
    DSP32_Q_CONVERT(-0.76615399020), DSP32_Q_CONVERT(-0.64265703397),
    DSP32_Q_CONVERT(-0.06132073630), DSP32_Q_CONVERT(-0.99811811290),
    DSP32_Q_CONVERT(-0.76910333765), DSP32_Q_CONVERT(-0.63912444486),
    DSP32_Q_CONVERT(-0.06438263093), DSP32_Q_CONVERT(-0.99792528620),
    DSP32_Q_CONVERT(-0.77203639715), DSP32_Q_CONVERT(-0.63557832049),
    DSP32_Q_CONVERT(-0.06744391956), DSP32_Q_CONVERT(-0.99772306664),
    DSP32_Q_CONVERT(-0.77495310659), DSP32_Q_CONVERT(-0.63201873594),
    DSP32_Q_CONVERT(-0.07050457339), DSP32_Q_CONVERT(-0.99751145614),
    DSP32_Q_CONVERT(-0.77785340421), DSP32_Q_CONVERT(-0.62844576660),
    DSP32_Q_CONVERT(-0.07356456360), DSP32_Q_CONVERT(-0.99729045668),
    DSP32_Q_CONVERT(-0.78073722857), DSP32_Q_CONVERT(-0.62485948814),
    DSP32_Q_CONVERT(-0.07662386139), DSP32_Q_CONVERT(-0.99706007034),
    DSP32_Q_CONVERT(-0.78360451861), DSP32_Q_CONVERT(-0.62125997651),
    DSP32_Q_CONVERT(-0.07968243797), DSP32_Q_CONVERT(-0.99682029929),
    DSP32_Q_CONVERT(-0.78645521360), DSP32_Q_CONVERT(-0.61764730794),
    DSP32_Q_CONVERT(-0.08274026455), DSP32_Q_CONVERT(-0.99657114579),
    DSP32_Q_CONVERT(-0.78928925317), DSP32_Q_CONVERT(-0.61402155893),
    DSP32_Q_CONVERT(-0.08579731234), DSP32_Q_CONVERT(-0.99631261218),
    DSP32_Q_CONVERT(-0.79210657730), DSP32_Q_CONVERT(-0.61038280628),
    DSP32_Q_CONVERT(-0.08885355258), DSP32_Q_CONVERT(-0.99604470090),
    DSP32_Q_CONVERT(-0.79490712633), DSP32_Q_CONVERT(-0.60673112703),
    DSP32_Q_CONVERT(-0.09190895650), DSP32_Q_CONVERT(-0.99576741447),
    DSP32_Q_CONVERT(-0.79769084094), DSP32_Q_CONVERT(-0.60306659854),
    DSP32_Q_CONVERT(-0.09496349533), DSP32_Q_CONVERT(-0.99548075549),
    DSP32_Q_CONVERT(-0.80045766219), DSP32_Q_CONVERT(-0.59938929840),
    DSP32_Q_CONVERT(-0.09801714033), DSP32_Q_CONVERT(-0.99518472667),
    DSP32_Q_CONVERT(-0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(-0.10106986275), DSP32_Q_CONVERT(-0.99487933079),
    DSP32_Q_CONVERT(-0.80594039057), DSP32_Q_CONVERT(-0.59199669496),
    DSP32_Q_CONVERT(-0.10412163387), DSP32_Q_CONVERT(-0.99456457073),
    DSP32_Q_CONVERT(-0.80865618159), DSP32_Q_CONVERT(-0.58828154822),
    DSP32_Q_CONVERT(-0.10717242496), DSP32_Q_CONVERT(-0.99424044945),
    DSP32_Q_CONVERT(-0.81135484702), DSP32_Q_CONVERT(-0.58455394295),
    DSP32_Q_CONVERT(-0.11022220729), DSP32_Q_CONVERT(-0.99390697000),
    DSP32_Q_CONVERT(-0.81403632971), DSP32_Q_CONVERT(-0.58081395810),
    DSP32_Q_CONVERT(-0.11327095218), DSP32_Q_CONVERT(-0.99356413552),
    DSP32_Q_CONVERT(-0.81670057287), DSP32_Q_CONVERT(-0.57706167286),
    DSP32_Q_CONVERT(-0.11631863091), DSP32_Q_CONVERT(-0.99321194923),
    DSP32_Q_CONVERT(-0.81934752008), DSP32_Q_CONVERT(-0.57329716670),
    DSP32_Q_CONVERT(-0.11936521481), DSP32_Q_CONVERT(-0.99285041446),
    DSP32_Q_CONVERT(-0.82197711528), DSP32_Q_CONVERT(-0.56952051935),
    DSP32_Q_CONVERT(-0.12241067520), DSP32_Q_CONVERT(-0.99247953460),
    DSP32_Q_CONVERT(-0.82458930279), DSP32_Q_CONVERT(-0.56573181078),
    DSP32_Q_CONVERT(-0.12545498341), DSP32_Q_CONVERT(-0.99209931314),
    DSP32_Q_CONVERT(-0.82718402727), DSP32_Q_CONVERT(-0.56193112124),
    DSP32_Q_CONVERT(-0.12849811079), DSP32_Q_CONVERT(-0.99170975367),
    DSP32_Q_CONVERT(-0.82976123379), DSP32_Q_CONVERT(-0.55811853122),
    DSP32_Q_CONVERT(-0.13154002870), DSP32_Q_CONVERT(-0.99131085985),
    DSP32_Q_CONVERT(-0.83232086777), DSP32_Q_CONVERT(-0.55429412145),
    DSP32_Q_CONVERT(-0.13458070851), DSP32_Q_CONVERT(-0.99090263543),
    DSP32_Q_CONVERT(-0.83486287499), DSP32_Q_CONVERT(-0.55045797294),
    DSP32_Q_CONVERT(-0.13762012159), DSP32_Q_CONVERT(-0.99048508426),
    DSP32_Q_CONVERT(-0.83738720162), DSP32_Q_CONVERT(-0.54661016691),
    DSP32_Q_CONVERT(-0.14065823933), DSP32_Q_CONVERT(-0.99005821026),
    DSP32_Q_CONVERT(-0.83989379420), DSP32_Q_CONVERT(-0.54275078486),
    DSP32_Q_CONVERT(-0.14369503315), DSP32_Q_CONVERT(-0.98962201746),
    DSP32_Q_CONVERT(-0.84238259964), DSP32_Q_CONVERT(-0.53887990853),
    DSP32_Q_CONVERT(-0.14673047446), DSP32_Q_CONVERT(-0.98917650996),
    DSP32_Q_CONVERT(-0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(-0.14976453468), DSP32_Q_CONVERT(-0.98872169196),
    DSP32_Q_CONVERT(-0.84730663869), DSP32_Q_CONVERT(-0.53110400115),
    DSP32_Q_CONVERT(-0.15279718526), DSP32_Q_CONVERT(-0.98825756773),
    DSP32_Q_CONVERT(-0.84974176800), DSP32_Q_CONVERT(-0.52719913478),
    DSP32_Q_CONVERT(-0.15582839765), DSP32_Q_CONVERT(-0.98778414164),
    DSP32_Q_CONVERT(-0.85215890162), DSP32_Q_CONVERT(-0.52328310348),
    DSP32_Q_CONVERT(-0.15885814333), DSP32_Q_CONVERT(-0.98730141816),
    DSP32_Q_CONVERT(-0.85455798837), DSP32_Q_CONVERT(-0.51935599017),
    DSP32_Q_CONVERT(-0.16188639378), DSP32_Q_CONVERT(-0.98680940181),
    DSP32_Q_CONVERT(-0.85693897742), DSP32_Q_CONVERT(-0.51541787802),
    DSP32_Q_CONVERT(-0.16491312049), DSP32_Q_CONVERT(-0.98630809724),
    DSP32_Q_CONVERT(-0.85930181836), DSP32_Q_CONVERT(-0.51146885044),
    DSP32_Q_CONVERT(-0.16793829497), DSP32_Q_CONVERT(-0.98579750917),
    DSP32_Q_CONVERT(-0.86164646114), DSP32_Q_CONVERT(-0.50750899105),
    DSP32_Q_CONVERT(-0.17096188876), DSP32_Q_CONVERT(-0.98527764239),
    DSP32_Q_CONVERT(-0.86397285612), DSP32_Q_CONVERT(-0.50353838373),
    DSP32_Q_CONVERT(-0.17398387339), DSP32_Q_CONVERT(-0.98474850180),
    DSP32_Q_CONVERT(-0.86628095402), DSP32_Q_CONVERT(-0.49955711255),
    DSP32_Q_CONVERT(-0.17700422041), DSP32_Q_CONVERT(-0.98421009239),
    DSP32_Q_CONVERT(-0.86857070597), DSP32_Q_CONVERT(-0.49556526183),
    DSP32_Q_CONVERT(-0.18002290141), DSP32_Q_CONVERT(-0.98366241921),
    DSP32_Q_CONVERT(-0.87084206347), DSP32_Q_CONVERT(-0.49156291611),
    DSP32_Q_CONVERT(-0.18303988796), DSP32_Q_CONVERT(-0.98310548743),
    DSP32_Q_CONVERT(-0.87309497842), DSP32_Q_CONVERT(-0.48755016015),
    DSP32_Q_CONVERT(-0.18605515166), DSP32_Q_CONVERT(-0.98253930229),
    DSP32_Q_CONVERT(-0.87532940310), DSP32_Q_CONVERT(-0.48352707893),
    DSP32_Q_CONVERT(-0.18906866415), DSP32_Q_CONVERT(-0.98196386911),
    DSP32_Q_CONVERT(-0.87754529021), DSP32_Q_CONVERT(-0.47949375766),
    DSP32_Q_CONVERT(-0.19208039705), DSP32_Q_CONVERT(-0.98137919331),
    DSP32_Q_CONVERT(-0.87974259280), DSP32_Q_CONVERT(-0.47545028175),
    DSP32_Q_CONVERT(-0.19509032202), DSP32_Q_CONVERT(-0.98078528040),
    DSP32_Q_CONVERT(-0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(-0.19809841072), DSP32_Q_CONVERT(-0.98018213597),
    DSP32_Q_CONVERT(-0.88408125871), DSP32_Q_CONVERT(-0.46733320874),
    DSP32_Q_CONVERT(-0.20110463484), DSP32_Q_CONVERT(-0.97956976569),
    DSP32_Q_CONVERT(-0.88622253015), DSP32_Q_CONVERT(-0.46325978355),
    DSP32_Q_CONVERT(-0.20410896609), DSP32_Q_CONVERT(-0.97894817532),
    DSP32_Q_CONVERT(-0.88834503331), DSP32_Q_CONVERT(-0.45917654752),
    DSP32_Q_CONVERT(-0.20711137619), DSP32_Q_CONVERT(-0.97831737072),
    DSP32_Q_CONVERT(-0.89044872324), DSP32_Q_CONVERT(-0.45508358713),
    DSP32_Q_CONVERT(-0.21011183688), DSP32_Q_CONVERT(-0.97767735782),
    DSP32_Q_CONVERT(-0.89253355540), DSP32_Q_CONVERT(-0.45098098905),
    DSP32_Q_CONVERT(-0.21311031992), DSP32_Q_CONVERT(-0.97702814266),
    DSP32_Q_CONVERT(-0.89459948563), DSP32_Q_CONVERT(-0.44686884016),
    DSP32_Q_CONVERT(-0.21610679708), DSP32_Q_CONVERT(-0.97636973133),
    DSP32_Q_CONVERT(-0.89664647018), DSP32_Q_CONVERT(-0.44274722756),
    DSP32_Q_CONVERT(-0.21910124016), DSP32_Q_CONVERT(-0.97570213004),
    DSP32_Q_CONVERT(-0.89867446569), DSP32_Q_CONVERT(-0.43861623854),
    DSP32_Q_CONVERT(-0.22209362097), DSP32_Q_CONVERT(-0.97502534507),
    DSP32_Q_CONVERT(-0.90068342923), DSP32_Q_CONVERT(-0.43447596057),
    DSP32_Q_CONVERT(-0.22508391136), DSP32_Q_CONVERT(-0.97433938279),
    DSP32_Q_CONVERT(-0.90267331824), DSP32_Q_CONVERT(-0.43032648134),
    DSP32_Q_CONVERT(-0.22807208317), DSP32_Q_CONVERT(-0.97364424965),
    DSP32_Q_CONVERT(-0.90464409058), DSP32_Q_CONVERT(-0.42616788873),
    DSP32_Q_CONVERT(-0.23105810828), DSP32_Q_CONVERT(-0.97293995221),
    DSP32_Q_CONVERT(-0.90659570451), DSP32_Q_CONVERT(-0.42200027080),
    DSP32_Q_CONVERT(-0.23404195858), DSP32_Q_CONVERT(-0.97222649708),
    DSP32_Q_CONVERT(-0.90852811872), DSP32_Q_CONVERT(-0.41782371582),
    DSP32_Q_CONVERT(-0.23702360599), DSP32_Q_CONVERT(-0.97150389099),
    DSP32_Q_CONVERT(-0.91044129226), DSP32_Q_CONVERT(-0.41363831224),
    DSP32_Q_CONVERT(-0.24000302245), DSP32_Q_CONVERT(-0.97077214073),
    DSP32_Q_CONVERT(-0.91233518462), DSP32_Q_CONVERT(-0.40944414869),
    DSP32_Q_CONVERT(-0.24298017990), DSP32_Q_CONVERT(-0.97003125319),
    DSP32_Q_CONVERT(-0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(-0.24595505034), DSP32_Q_CONVERT(-0.96928123536),
    DSP32_Q_CONVERT(-0.91606496580), DSP32_Q_CONVERT(-0.40102989718),
    DSP32_Q_CONVERT(-0.24892760575), DSP32_Q_CONVERT(-0.96852209427),
    DSP32_Q_CONVERT(-0.91790077562), DSP32_Q_CONVERT(-0.39680998742),
    DSP32_Q_CONVERT(-0.25189781815), DSP32_Q_CONVERT(-0.96775383709),
    DSP32_Q_CONVERT(-0.91971714629), DSP32_Q_CONVERT(-0.39258167407),
    DSP32_Q_CONVERT(-0.25486565960), DSP32_Q_CONVERT(-0.96697647104),
    DSP32_Q_CONVERT(-0.92151403934), DSP32_Q_CONVERT(-0.38834504670),
    DSP32_Q_CONVERT(-0.25783110216), DSP32_Q_CONVERT(-0.96619000345),
    DSP32_Q_CONVERT(-0.92329141672), DSP32_Q_CONVERT(-0.38410019502),
    DSP32_Q_CONVERT(-0.26079411792), DSP32_Q_CONVERT(-0.96539444170),
    DSP32_Q_CONVERT(-0.92504924078), DSP32_Q_CONVERT(-0.37984720892),
    DSP32_Q_CONVERT(-0.26375467897), DSP32_Q_CONVERT(-0.96458979329),
    DSP32_Q_CONVERT(-0.92678747430), DSP32_Q_CONVERT(-0.37558617849),
    DSP32_Q_CONVERT(-0.26671275747), DSP32_Q_CONVERT(-0.96377606580),
    DSP32_Q_CONVERT(-0.92850608047), DSP32_Q_CONVERT(-0.37131719395),
    DSP32_Q_CONVERT(-0.26966832557), DSP32_Q_CONVERT(-0.96295326687),
    DSP32_Q_CONVERT(-0.93020502289), DSP32_Q_CONVERT(-0.36704034572),
    DSP32_Q_CONVERT(-0.27262135545), DSP32_Q_CONVERT(-0.96212140427),
    DSP32_Q_CONVERT(-0.93188426558), DSP32_Q_CONVERT(-0.36275572437),
    DSP32_Q_CONVERT(-0.27557181931), DSP32_Q_CONVERT(-0.96128048581),
    DSP32_Q_CONVERT(-0.93354377298), DSP32_Q_CONVERT(-0.35846342063),
    DSP32_Q_CONVERT(-0.27851968939), DSP32_Q_CONVERT(-0.96043051942),
    DSP32_Q_CONVERT(-0.93518350994), DSP32_Q_CONVERT(-0.35416352542),
    DSP32_Q_CONVERT(-0.28146493793), DSP32_Q_CONVERT(-0.95957151308),
    DSP32_Q_CONVERT(-0.93680344174), DSP32_Q_CONVERT(-0.34985612979),
    DSP32_Q_CONVERT(-0.28440753721), DSP32_Q_CONVERT(-0.95870347490),
    DSP32_Q_CONVERT(-0.93840353406), DSP32_Q_CONVERT(-0.34554132496),
    DSP32_Q_CONVERT(-0.28734745954), DSP32_Q_CONVERT(-0.95782641303),
    DSP32_Q_CONVERT(-0.93998375303), DSP32_Q_CONVERT(-0.34121920232),
    DSP32_Q_CONVERT(-0.29028467725), DSP32_Q_CONVERT(-0.95694033573),
    DSP32_Q_CONVERT(-0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(-0.29321916269), DSP32_Q_CONVERT(-0.95604525135),
    DSP32_Q_CONVERT(-0.94308443747), DSP32_Q_CONVERT(-0.33255336987),
    DSP32_Q_CONVERT(-0.29615088824), DSP32_Q_CONVERT(-0.95514116831),
    DSP32_Q_CONVERT(-0.94460483726), DSP32_Q_CONVERT(-0.32820984358),
    DSP32_Q_CONVERT(-0.29907982631), DSP32_Q_CONVERT(-0.95422809511),
    DSP32_Q_CONVERT(-0.94610523237), DSP32_Q_CONVERT(-0.32385936652),
    DSP32_Q_CONVERT(-0.30200594932), DSP32_Q_CONVERT(-0.95330604035),
    DSP32_Q_CONVERT(-0.94758559102), DSP32_Q_CONVERT(-0.31950203082),
    DSP32_Q_CONVERT(-0.30492922974), DSP32_Q_CONVERT(-0.95237501272),
    DSP32_Q_CONVERT(-0.94904588185), DSP32_Q_CONVERT(-0.31513792875),
    DSP32_Q_CONVERT(-0.30784964004), DSP32_Q_CONVERT(-0.95143502097),
    DSP32_Q_CONVERT(-0.95048607395), DSP32_Q_CONVERT(-0.31076715275),
    DSP32_Q_CONVERT(-0.31076715275), DSP32_Q_CONVERT(-0.95048607395),
    DSP32_Q_CONVERT(-0.95190613681), DSP32_Q_CONVERT(-0.30638979537),
    DSP32_Q_CONVERT(-0.31368174040), DSP32_Q_CONVERT(-0.94952818059),
    DSP32_Q_CONVERT(-0.95330604035), DSP32_Q_CONVERT(-0.30200594932),
    DSP32_Q_CONVERT(-0.31659337556), DSP32_Q_CONVERT(-0.94856134992),
    DSP32_Q_CONVERT(-0.95468575494), DSP32_Q_CONVERT(-0.29761570744),
    DSP32_Q_CONVERT(-0.31950203082), DSP32_Q_CONVERT(-0.94758559102),
    DSP32_Q_CONVERT(-0.95604525135), DSP32_Q_CONVERT(-0.29321916269),
    DSP32_Q_CONVERT(-0.32240767880), DSP32_Q_CONVERT(-0.94660091308),
    DSP32_Q_CONVERT(-0.95738450079), DSP32_Q_CONVERT(-0.28881640821),
    DSP32_Q_CONVERT(-0.32531029216), DSP32_Q_CONVERT(-0.94560732538),
    DSP32_Q_CONVERT(-0.95870347490), DSP32_Q_CONVERT(-0.28440753721),
    DSP32_Q_CONVERT(-0.32820984358), DSP32_Q_CONVERT(-0.94460483726),
    DSP32_Q_CONVERT(-0.96000214574), DSP32_Q_CONVERT(-0.27999264308),
    DSP32_Q_CONVERT(-0.33110630576), DSP32_Q_CONVERT(-0.94359345816),
    DSP32_Q_CONVERT(-0.96128048581), DSP32_Q_CONVERT(-0.27557181931),
    DSP32_Q_CONVERT(-0.33399965144), DSP32_Q_CONVERT(-0.94257319760),
    DSP32_Q_CONVERT(-0.96253846804), DSP32_Q_CONVERT(-0.27114515953),
    DSP32_Q_CONVERT(-0.33688985339), DSP32_Q_CONVERT(-0.94154406518),
    DSP32_Q_CONVERT(-0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(-0.33977688441), DSP32_Q_CONVERT(-0.94050607059),
    DSP32_Q_CONVERT(-0.96499325285), DSP32_Q_CONVERT(-0.26227470702),
    DSP32_Q_CONVERT(-0.34266071731), DSP32_Q_CONVERT(-0.93945922360),
    DSP32_Q_CONVERT(-0.96619000345), DSP32_Q_CONVERT(-0.25783110216),
    DSP32_Q_CONVERT(-0.34554132496), DSP32_Q_CONVERT(-0.93840353406),
    DSP32_Q_CONVERT(-0.96736629222), DSP32_Q_CONVERT(-0.25338203700),
    DSP32_Q_CONVERT(-0.34841868025), DSP32_Q_CONVERT(-0.93733901191),
    DSP32_Q_CONVERT(-0.96852209427), DSP32_Q_CONVERT(-0.24892760575),
    DSP32_Q_CONVERT(-0.35129275609), DSP32_Q_CONVERT(-0.93626566717),
    DSP32_Q_CONVERT(-0.96965738512), DSP32_Q_CONVERT(-0.24446790275),
    DSP32_Q_CONVERT(-0.35416352542), DSP32_Q_CONVERT(-0.93518350994),
    DSP32_Q_CONVERT(-0.97077214073), DSP32_Q_CONVERT(-0.24000302245),
    DSP32_Q_CONVERT(-0.35703096123), DSP32_Q_CONVERT(-0.93409255040),
    DSP32_Q_CONVERT(-0.97186633748), DSP32_Q_CONVERT(-0.23553305940),
    DSP32_Q_CONVERT(-0.35989503653), DSP32_Q_CONVERT(-0.93299279883),
    DSP32_Q_CONVERT(-0.97293995221), DSP32_Q_CONVERT(-0.23105810828),
    DSP32_Q_CONVERT(-0.36275572437), DSP32_Q_CONVERT(-0.93188426558),
    DSP32_Q_CONVERT(-0.97399296217), DSP32_Q_CONVERT(-0.22657826385),
    DSP32_Q_CONVERT(-0.36561299780), DSP32_Q_CONVERT(-0.93076696108),
    DSP32_Q_CONVERT(-0.97502534507), DSP32_Q_CONVERT(-0.22209362097),
    DSP32_Q_CONVERT(-0.36846682995), DSP32_Q_CONVERT(-0.92964089584),
    DSP32_Q_CONVERT(-0.97603707904), DSP32_Q_CONVERT(-0.21760427464),
    DSP32_Q_CONVERT(-0.37131719395), DSP32_Q_CONVERT(-0.92850608047),
    DSP32_Q_CONVERT(-0.97702814266), DSP32_Q_CONVERT(-0.21311031992),
    DSP32_Q_CONVERT(-0.37416406297), DSP32_Q_CONVERT(-0.92736252565),
    DSP32_Q_CONVERT(-0.97799851493), DSP32_Q_CONVERT(-0.20861185198),
    DSP32_Q_CONVERT(-0.37700741022), DSP32_Q_CONVERT(-0.92621024214),
    DSP32_Q_CONVERT(-0.97894817532), DSP32_Q_CONVERT(-0.20410896609),
    DSP32_Q_CONVERT(-0.37984720892), DSP32_Q_CONVERT(-0.92504924078),
    DSP32_Q_CONVERT(-0.97987710370), DSP32_Q_CONVERT(-0.19960175762),
    DSP32_Q_CONVERT(-0.38268343237), DSP32_Q_CONVERT(-0.92387953251),
    DSP32_Q_CONVERT(-0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(-0.38551605384), DSP32_Q_CONVERT(-0.92270112833),
    DSP32_Q_CONVERT(-0.98167268620), DSP32_Q_CONVERT(-0.19057475482),
    DSP32_Q_CONVERT(-0.38834504670), DSP32_Q_CONVERT(-0.92151403934),
    DSP32_Q_CONVERT(-0.98253930229), DSP32_Q_CONVERT(-0.18605515166),
    DSP32_Q_CONVERT(-0.39117038430), DSP32_Q_CONVERT(-0.92031827671),
    DSP32_Q_CONVERT(-0.98338511032), DSP32_Q_CONVERT(-0.18153160826),
    DSP32_Q_CONVERT(-0.39399204006), DSP32_Q_CONVERT(-0.91911385169),
    DSP32_Q_CONVERT(-0.98421009239), DSP32_Q_CONVERT(-0.17700422041),
    DSP32_Q_CONVERT(-0.39680998742), DSP32_Q_CONVERT(-0.91790077562),
    DSP32_Q_CONVERT(-0.98501423101), DSP32_Q_CONVERT(-0.17247308400),
    DSP32_Q_CONVERT(-0.39962419985), DSP32_Q_CONVERT(-0.91667905992),
    DSP32_Q_CONVERT(-0.98579750917), DSP32_Q_CONVERT(-0.16793829497),
    DSP32_Q_CONVERT(-0.40243465086), DSP32_Q_CONVERT(-0.91544871609),
    DSP32_Q_CONVERT(-0.98655991026), DSP32_Q_CONVERT(-0.16339994938),
    DSP32_Q_CONVERT(-0.40524131400), DSP32_Q_CONVERT(-0.91420975570),
    DSP32_Q_CONVERT(-0.98730141816), DSP32_Q_CONVERT(-0.15885814333),
    DSP32_Q_CONVERT(-0.40804416286), DSP32_Q_CONVERT(-0.91296219043),
    DSP32_Q_CONVERT(-0.98802201714), DSP32_Q_CONVERT(-0.15431297301),
    DSP32_Q_CONVERT(-0.41084317106), DSP32_Q_CONVERT(-0.91170603201),
    DSP32_Q_CONVERT(-0.98872169196), DSP32_Q_CONVERT(-0.14976453468),
    DSP32_Q_CONVERT(-0.41363831224), DSP32_Q_CONVERT(-0.91044129226),
    DSP32_Q_CONVERT(-0.98940042779), DSP32_Q_CONVERT(-0.14521292465),
    DSP32_Q_CONVERT(-0.41642956010), DSP32_Q_CONVERT(-0.90916798309),
    DSP32_Q_CONVERT(-0.99005821026), DSP32_Q_CONVERT(-0.14065823933),
    DSP32_Q_CONVERT(-0.41921688836), DSP32_Q_CONVERT(-0.90788611649),
    DSP32_Q_CONVERT(-0.99069502544), DSP32_Q_CONVERT(-0.13610057518),
    DSP32_Q_CONVERT(-0.42200027080), DSP32_Q_CONVERT(-0.90659570451),
    DSP32_Q_CONVERT(-0.99131085985), DSP32_Q_CONVERT(-0.13154002870),
    DSP32_Q_CONVERT(-0.42477968121), DSP32_Q_CONVERT(-0.90529675932),
    DSP32_Q_CONVERT(-0.99190570043), DSP32_Q_CONVERT(-0.12697669650),
    DSP32_Q_CONVERT(-0.42755509343), DSP32_Q_CONVERT(-0.90398929312),
    DSP32_Q_CONVERT(-0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(-0.43032648134), DSP32_Q_CONVERT(-0.90267331824),
    DSP32_Q_CONVERT(-0.99303235020), DSP32_Q_CONVERT(-0.11784206151),
    DSP32_Q_CONVERT(-0.43309381885), DSP32_Q_CONVERT(-0.90134884705),
    DSP32_Q_CONVERT(-0.99356413552), DSP32_Q_CONVERT(-0.11327095218),
    DSP32_Q_CONVERT(-0.43585707992), DSP32_Q_CONVERT(-0.90001589202),
    DSP32_Q_CONVERT(-0.99407487930), DSP32_Q_CONVERT(-0.10869744401),
    DSP32_Q_CONVERT(-0.43861623854), DSP32_Q_CONVERT(-0.89867446569),
    DSP32_Q_CONVERT(-0.99456457073), DSP32_Q_CONVERT(-0.10412163387),
    DSP32_Q_CONVERT(-0.44137126873), DSP32_Q_CONVERT(-0.89732458071),
    DSP32_Q_CONVERT(-0.99503319944), DSP32_Q_CONVERT(-0.09954361866),
    DSP32_Q_CONVERT(-0.44412214457), DSP32_Q_CONVERT(-0.89596624976),
    DSP32_Q_CONVERT(-0.99548075549), DSP32_Q_CONVERT(-0.09496349533),
    DSP32_Q_CONVERT(-0.44686884016), DSP32_Q_CONVERT(-0.89459948563),
    DSP32_Q_CONVERT(-0.99590722942), DSP32_Q_CONVERT(-0.09038136088),
    DSP32_Q_CONVERT(-0.44961132965), DSP32_Q_CONVERT(-0.89322430120),
    DSP32_Q_CONVERT(-0.99631261218), DSP32_Q_CONVERT(-0.08579731234),
    DSP32_Q_CONVERT(-0.45234958723), DSP32_Q_CONVERT(-0.89184070939),
    DSP32_Q_CONVERT(-0.99669689520), DSP32_Q_CONVERT(-0.08121144681),
    DSP32_Q_CONVERT(-0.45508358713), DSP32_Q_CONVERT(-0.89044872324),
    DSP32_Q_CONVERT(-0.99706007034), DSP32_Q_CONVERT(-0.07662386139),
    DSP32_Q_CONVERT(-0.45781330360), DSP32_Q_CONVERT(-0.88904835585),
    DSP32_Q_CONVERT(-0.99740212990), DSP32_Q_CONVERT(-0.07203465325),
    DSP32_Q_CONVERT(-0.46053871096), DSP32_Q_CONVERT(-0.88763962040),
    DSP32_Q_CONVERT(-0.99772306664), DSP32_Q_CONVERT(-0.06744391956),
    DSP32_Q_CONVERT(-0.46325978355), DSP32_Q_CONVERT(-0.88622253015),
    DSP32_Q_CONVERT(-0.99802287377), DSP32_Q_CONVERT(-0.06285175756),
    DSP32_Q_CONVERT(-0.46597649577), DSP32_Q_CONVERT(-0.88479709843),
    DSP32_Q_CONVERT(-0.99830154493), DSP32_Q_CONVERT(-0.05825826450),
    DSP32_Q_CONVERT(-0.46868882204), DSP32_Q_CONVERT(-0.88336333867),
    DSP32_Q_CONVERT(-0.99855907423), DSP32_Q_CONVERT(-0.05366353765),
    DSP32_Q_CONVERT(-0.47139673683), DSP32_Q_CONVERT(-0.88192126435),
    DSP32_Q_CONVERT(-0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(-0.47410021465), DSP32_Q_CONVERT(-0.88047088905),
    DSP32_Q_CONVERT(-0.99901068585), DSP32_Q_CONVERT(-0.04447077185),
    DSP32_Q_CONVERT(-0.47679923006), DSP32_Q_CONVERT(-0.87901222643),
    DSP32_Q_CONVERT(-0.99920475862), DSP32_Q_CONVERT(-0.03987292759),
    DSP32_Q_CONVERT(-0.47949375766), DSP32_Q_CONVERT(-0.87754529021),
    DSP32_Q_CONVERT(-0.99937767039), DSP32_Q_CONVERT(-0.03527423890),
    DSP32_Q_CONVERT(-0.48218377208), DSP32_Q_CONVERT(-0.87607009420),
    DSP32_Q_CONVERT(-0.99952941750), DSP32_Q_CONVERT(-0.03067480318),
    DSP32_Q_CONVERT(-0.48486924800), DSP32_Q_CONVERT(-0.87458665228),
    DSP32_Q_CONVERT(-0.99965999674), DSP32_Q_CONVERT(-0.02607471783),
    DSP32_Q_CONVERT(-0.48755016015), DSP32_Q_CONVERT(-0.87309497842),
    DSP32_Q_CONVERT(-0.99976940535), DSP32_Q_CONVERT(-0.02147408028),
    DSP32_Q_CONVERT(-0.49022648329), DSP32_Q_CONVERT(-0.87159508666),
    DSP32_Q_CONVERT(-0.99985764101), DSP32_Q_CONVERT(-0.01687298795),
    DSP32_Q_CONVERT(-0.49289819223), DSP32_Q_CONVERT(-0.87008699111),
    DSP32_Q_CONVERT(-0.99992470184), DSP32_Q_CONVERT(-0.01227153829),
    DSP32_Q_CONVERT(-0.49556526183), DSP32_Q_CONVERT(-0.86857070597),
    DSP32_Q_CONVERT(-0.99997058643), DSP32_Q_CONVERT(-0.00766982874),
    DSP32_Q_CONVERT(-0.49822766697), DSP32_Q_CONVERT(-0.86704624552),
    DSP32_Q_CONVERT(-0.99999529381), DSP32_Q_CONVERT(-0.00306795676),
    DSP32_Q_CONVERT(-0.50088538261), DSP32_Q_CONVERT(-0.86551362409),
    DSP32_Q_CONVERT(-0.99999882345), DSP32_Q_CONVERT(0.00153398019),
    DSP32_Q_CONVERT(-0.50353838373), DSP32_Q_CONVERT(-0.86397285612),
    DSP32_Q_CONVERT(-0.99998117528), DSP32_Q_CONVERT(0.00613588465),
    DSP32_Q_CONVERT(-0.50618664535), DSP32_Q_CONVERT(-0.86242395611),
    DSP32_Q_CONVERT(-0.99994234968), DSP32_Q_CONVERT(0.01073765917),
    DSP32_Q_CONVERT(-0.50883014254), DSP32_Q_CONVERT(-0.86086693864),
    DSP32_Q_CONVERT(-0.99988234745), DSP32_Q_CONVERT(0.01533920628),
    DSP32_Q_CONVERT(-0.51146885044), DSP32_Q_CONVERT(-0.85930181836),
    DSP32_Q_CONVERT(-0.99980116989), DSP32_Q_CONVERT(0.01994042855),
    DSP32_Q_CONVERT(-0.51410274419), DSP32_Q_CONVERT(-0.85772861000),
    DSP32_Q_CONVERT(-0.99969881870), DSP32_Q_CONVERT(0.02454122852),
    DSP32_Q_CONVERT(-0.51673179902), DSP32_Q_CONVERT(-0.85614732838),
    DSP32_Q_CONVERT(-0.99957529605), DSP32_Q_CONVERT(0.02914150876),
    DSP32_Q_CONVERT(-0.51935599017), DSP32_Q_CONVERT(-0.85455798837),
    DSP32_Q_CONVERT(-0.99943060456), DSP32_Q_CONVERT(0.03374117185),
    DSP32_Q_CONVERT(-0.52197529294), DSP32_Q_CONVERT(-0.85296060493),
    DSP32_Q_CONVERT(-0.99926474729), DSP32_Q_CONVERT(0.03834012037),
    DSP32_Q_CONVERT(-0.52458968268), DSP32_Q_CONVERT(-0.85135519311),
    DSP32_Q_CONVERT(-0.99907772775), DSP32_Q_CONVERT(0.04293825693),
    DSP32_Q_CONVERT(-0.52719913478), DSP32_Q_CONVERT(-0.84974176800),
    DSP32_Q_CONVERT(-0.99886954991), DSP32_Q_CONVERT(0.04753548416),
    DSP32_Q_CONVERT(-0.52980362469), DSP32_Q_CONVERT(-0.84812034480),
    DSP32_Q_CONVERT(-0.99864021818), DSP32_Q_CONVERT(0.05213170468),
    DSP32_Q_CONVERT(-0.53240312788), DSP32_Q_CONVERT(-0.84649093877),
    DSP32_Q_CONVERT(-0.99838973741), DSP32_Q_CONVERT(0.05672682117),
    DSP32_Q_CONVERT(-0.53499761989), DSP32_Q_CONVERT(-0.84485356525),
    DSP32_Q_CONVERT(-0.99811811290), DSP32_Q_CONVERT(0.06132073630),
    DSP32_Q_CONVERT(-0.53758707630), DSP32_Q_CONVERT(-0.84320823964),
    DSP32_Q_CONVERT(-0.99782535041), DSP32_Q_CONVERT(0.06591335280),
    DSP32_Q_CONVERT(-0.54017147273), DSP32_Q_CONVERT(-0.84155497744),
    DSP32_Q_CONVERT(-0.99751145614), DSP32_Q_CONVERT(0.07050457339),
    DSP32_Q_CONVERT(-0.54275078486), DSP32_Q_CONVERT(-0.83989379420),
    DSP32_Q_CONVERT(-0.99717643674), DSP32_Q_CONVERT(0.07509430085),
    DSP32_Q_CONVERT(-0.54532498842), DSP32_Q_CONVERT(-0.83822470555),
    DSP32_Q_CONVERT(-0.99682029929), DSP32_Q_CONVERT(0.07968243797),
    DSP32_Q_CONVERT(-0.54789405917), DSP32_Q_CONVERT(-0.83654772722),
    DSP32_Q_CONVERT(-0.99644305135), DSP32_Q_CONVERT(0.08426888759),
    DSP32_Q_CONVERT(-0.55045797294), DSP32_Q_CONVERT(-0.83486287499),
    DSP32_Q_CONVERT(-0.99604470090), DSP32_Q_CONVERT(0.08885355258),
    DSP32_Q_CONVERT(-0.55301670558), DSP32_Q_CONVERT(-0.83317016470),
    DSP32_Q_CONVERT(-0.99562525638), DSP32_Q_CONVERT(0.09343633585),
    DSP32_Q_CONVERT(-0.55557023302), DSP32_Q_CONVERT(-0.83146961230),
    DSP32_Q_CONVERT(-0.99518472667), DSP32_Q_CONVERT(0.09801714033),
    DSP32_Q_CONVERT(-0.55811853122), DSP32_Q_CONVERT(-0.82976123379),
    DSP32_Q_CONVERT(-0.99472312110), DSP32_Q_CONVERT(0.10259586902),
    DSP32_Q_CONVERT(-0.56066157620), DSP32_Q_CONVERT(-0.82804504526),
    DSP32_Q_CONVERT(-0.99424044945), DSP32_Q_CONVERT(0.10717242496),
    DSP32_Q_CONVERT(-0.56319934401), DSP32_Q_CONVERT(-0.82632106285),
    DSP32_Q_CONVERT(-0.99373672194), DSP32_Q_CONVERT(0.11174671121),
    DSP32_Q_CONVERT(-0.56573181078), DSP32_Q_CONVERT(-0.82458930279),
    DSP32_Q_CONVERT(-0.99321194923), DSP32_Q_CONVERT(0.11631863091),
    DSP32_Q_CONVERT(-0.56825895267), DSP32_Q_CONVERT(-0.82284978138),
    DSP32_Q_CONVERT(-0.99266614245), DSP32_Q_CONVERT(0.12088808724),
    DSP32_Q_CONVERT(-0.57078074589), DSP32_Q_CONVERT(-0.82110251499),
    DSP32_Q_CONVERT(-0.99209931314), DSP32_Q_CONVERT(0.12545498341),
    DSP32_Q_CONVERT(-0.57329716670), DSP32_Q_CONVERT(-0.81934752008),
    DSP32_Q_CONVERT(-0.99151147332), DSP32_Q_CONVERT(0.13001922272),
    DSP32_Q_CONVERT(-0.57580819142), DSP32_Q_CONVERT(-0.81758481315),
    DSP32_Q_CONVERT(-0.99090263543), DSP32_Q_CONVERT(0.13458070851),
    DSP32_Q_CONVERT(-0.57831379641), DSP32_Q_CONVERT(-0.81581441081),
    DSP32_Q_CONVERT(-0.99027281236), DSP32_Q_CONVERT(0.13913934416),
    DSP32_Q_CONVERT(-0.58081395810), DSP32_Q_CONVERT(-0.81403632971),
    DSP32_Q_CONVERT(-0.98962201746), DSP32_Q_CONVERT(0.14369503315),
    DSP32_Q_CONVERT(-0.58330865294), DSP32_Q_CONVERT(-0.81225058659),
    DSP32_Q_CONVERT(-0.98895026451), DSP32_Q_CONVERT(0.14824767899),
    DSP32_Q_CONVERT(-0.58579785746), DSP32_Q_CONVERT(-0.81045719825),
    DSP32_Q_CONVERT(-0.98825756773), DSP32_Q_CONVERT(0.15279718526),
    DSP32_Q_CONVERT(-0.58828154822), DSP32_Q_CONVERT(-0.80865618159),
    DSP32_Q_CONVERT(-0.98754394179), DSP32_Q_CONVERT(0.15734345562),
    DSP32_Q_CONVERT(-0.59075970186), DSP32_Q_CONVERT(-0.80684755354),
    DSP32_Q_CONVERT(-0.98680940181), DSP32_Q_CONVERT(0.16188639378),
    DSP32_Q_CONVERT(-0.59323229504), DSP32_Q_CONVERT(-0.80503133114),
    DSP32_Q_CONVERT(-0.98605396335), DSP32_Q_CONVERT(0.16642590354),
    DSP32_Q_CONVERT(-0.59569930449), DSP32_Q_CONVERT(-0.80320753148),
    DSP32_Q_CONVERT(-0.98527764239), DSP32_Q_CONVERT(0.17096188876),
    DSP32_Q_CONVERT(-0.59816070700), DSP32_Q_CONVERT(-0.80137617172),
    DSP32_Q_CONVERT(-0.98448045538), DSP32_Q_CONVERT(0.17549425338),
    DSP32_Q_CONVERT(-0.60061647938), DSP32_Q_CONVERT(-0.79953726911),
    DSP32_Q_CONVERT(-0.98366241921), DSP32_Q_CONVERT(0.18002290141),
    DSP32_Q_CONVERT(-0.60306659854), DSP32_Q_CONVERT(-0.79769084094),
    DSP32_Q_CONVERT(-0.98282355120), DSP32_Q_CONVERT(0.18454773694),
    DSP32_Q_CONVERT(-0.60551104140), DSP32_Q_CONVERT(-0.79583690461),
    DSP32_Q_CONVERT(-0.98196386911), DSP32_Q_CONVERT(0.18906866415),
    DSP32_Q_CONVERT(-0.60794978497), DSP32_Q_CONVERT(-0.79397547755),
    DSP32_Q_CONVERT(-0.98108339115), DSP32_Q_CONVERT(0.19358558730),
    DSP32_Q_CONVERT(-0.61038280628), DSP32_Q_CONVERT(-0.79210657730),
    DSP32_Q_CONVERT(-0.98018213597), DSP32_Q_CONVERT(0.19809841072),
    DSP32_Q_CONVERT(-0.61281008243), DSP32_Q_CONVERT(-0.79023022144),
    DSP32_Q_CONVERT(-0.97926012265), DSP32_Q_CONVERT(0.20260703884),
    DSP32_Q_CONVERT(-0.61523159058), DSP32_Q_CONVERT(-0.78834642763),
    DSP32_Q_CONVERT(-0.97831737072), DSP32_Q_CONVERT(0.20711137619),
    DSP32_Q_CONVERT(-0.61764730794), DSP32_Q_CONVERT(-0.78645521360),
    DSP32_Q_CONVERT(-0.97735390015), DSP32_Q_CONVERT(0.21161132737),
    DSP32_Q_CONVERT(-0.62005721176), DSP32_Q_CONVERT(-0.78455659716),
    DSP32_Q_CONVERT(-0.97636973133), DSP32_Q_CONVERT(0.21610679708),
    DSP32_Q_CONVERT(-0.62246127937), DSP32_Q_CONVERT(-0.78265059617),
    DSP32_Q_CONVERT(-0.97536488512), DSP32_Q_CONVERT(0.22059769011),
    DSP32_Q_CONVERT(-0.62485948814), DSP32_Q_CONVERT(-0.78073722857),
    DSP32_Q_CONVERT(-0.97433938279), DSP32_Q_CONVERT(0.22508391136),
    DSP32_Q_CONVERT(-0.62725181550), DSP32_Q_CONVERT(-0.77881651238),
    DSP32_Q_CONVERT(-0.97329324605), DSP32_Q_CONVERT(0.22956536582),
    DSP32_Q_CONVERT(-0.62963823891), DSP32_Q_CONVERT(-0.77688846567),
    DSP32_Q_CONVERT(-0.97222649708), DSP32_Q_CONVERT(0.23404195858),
    DSP32_Q_CONVERT(-0.63201873594), DSP32_Q_CONVERT(-0.77495310659),
    DSP32_Q_CONVERT(-0.97113915845), DSP32_Q_CONVERT(0.23851359484),
    DSP32_Q_CONVERT(-0.63439328416), DSP32_Q_CONVERT(-0.77301045336),
    DSP32_Q_CONVERT(-0.97003125319), DSP32_Q_CONVERT(0.24298017990),
    DSP32_Q_CONVERT(-0.63676186124), DSP32_Q_CONVERT(-0.77106052426),
    DSP32_Q_CONVERT(-0.96890280478), DSP32_Q_CONVERT(0.24744161917),
    DSP32_Q_CONVERT(-0.63912444486), DSP32_Q_CONVERT(-0.76910333765),
    DSP32_Q_CONVERT(-0.96775383709), DSP32_Q_CONVERT(0.25189781815),
    DSP32_Q_CONVERT(-0.64148101281), DSP32_Q_CONVERT(-0.76713891194),
    DSP32_Q_CONVERT(-0.96658437448), DSP32_Q_CONVERT(0.25634868249),
    DSP32_Q_CONVERT(-0.64383154289), DSP32_Q_CONVERT(-0.76516726562),
    DSP32_Q_CONVERT(-0.96539444170), DSP32_Q_CONVERT(0.26079411792),
    DSP32_Q_CONVERT(-0.64617601298), DSP32_Q_CONVERT(-0.76318841726),
    DSP32_Q_CONVERT(-0.96418406395), DSP32_Q_CONVERT(0.26523403029),
    DSP32_Q_CONVERT(-0.64851440102), DSP32_Q_CONVERT(-0.76120238548),
    DSP32_Q_CONVERT(-0.96295326687), DSP32_Q_CONVERT(0.26966832557),
    DSP32_Q_CONVERT(-0.65084668500), DSP32_Q_CONVERT(-0.75920918898),
    DSP32_Q_CONVERT(-0.96170207653), DSP32_Q_CONVERT(0.27409690987),
    DSP32_Q_CONVERT(-0.65317284295), DSP32_Q_CONVERT(-0.75720884651),
    DSP32_Q_CONVERT(-0.96043051942), DSP32_Q_CONVERT(0.27851968939),
    DSP32_Q_CONVERT(-0.65549285300), DSP32_Q_CONVERT(-0.75520137690),
    DSP32_Q_CONVERT(-0.95913862246), DSP32_Q_CONVERT(0.28293657046),
    DSP32_Q_CONVERT(-0.65780669330), DSP32_Q_CONVERT(-0.75318679904),
    DSP32_Q_CONVERT(-0.95782641303), DSP32_Q_CONVERT(0.28734745954),
    DSP32_Q_CONVERT(-0.66011434207), DSP32_Q_CONVERT(-0.75116513191),
    DSP32_Q_CONVERT(-0.95649391890), DSP32_Q_CONVERT(0.29175226323),
    DSP32_Q_CONVERT(-0.66241577759), DSP32_Q_CONVERT(-0.74913639452),
    DSP32_Q_CONVERT(-0.95514116831), DSP32_Q_CONVERT(0.29615088824),
    DSP32_Q_CONVERT(-0.66471097820), DSP32_Q_CONVERT(-0.74710060598),
    DSP32_Q_CONVERT(-0.95376818989), DSP32_Q_CONVERT(0.30054324142),
    DSP32_Q_CONVERT(-0.66699992230), DSP32_Q_CONVERT(-0.74505778544),
    DSP32_Q_CONVERT(-0.95237501272), DSP32_Q_CONVERT(0.30492922974),
    DSP32_Q_CONVERT(-0.66928258835), DSP32_Q_CONVERT(-0.74300795214),
    DSP32_Q_CONVERT(-0.95096166631), DSP32_Q_CONVERT(0.30930876031),
    DSP32_Q_CONVERT(-0.67155895485), DSP32_Q_CONVERT(-0.74095112535),
    DSP32_Q_CONVERT(-0.94952818059), DSP32_Q_CONVERT(0.31368174040),
    DSP32_Q_CONVERT(-0.67382900038), DSP32_Q_CONVERT(-0.73888732446),
    DSP32_Q_CONVERT(-0.94807458592), DSP32_Q_CONVERT(0.31804807739),
    DSP32_Q_CONVERT(-0.67609270358), DSP32_Q_CONVERT(-0.73681656888),
    DSP32_Q_CONVERT(-0.94660091308), DSP32_Q_CONVERT(0.32240767880),
    DSP32_Q_CONVERT(-0.67835004313), DSP32_Q_CONVERT(-0.73473887810),
    DSP32_Q_CONVERT(-0.94510719329), DSP32_Q_CONVERT(0.32676045232),
    DSP32_Q_CONVERT(-0.68060099780), DSP32_Q_CONVERT(-0.73265427167),
    DSP32_Q_CONVERT(-0.94359345816), DSP32_Q_CONVERT(0.33110630576),
    DSP32_Q_CONVERT(-0.68284554639), DSP32_Q_CONVERT(-0.73056276923),
    DSP32_Q_CONVERT(-0.94205973977), DSP32_Q_CONVERT(0.33544514708),
    DSP32_Q_CONVERT(-0.68508366777), DSP32_Q_CONVERT(-0.72846439045),
    DSP32_Q_CONVERT(-0.94050607059), DSP32_Q_CONVERT(0.33977688441),
    DSP32_Q_CONVERT(-0.68731534089), DSP32_Q_CONVERT(-0.72635915508),
    DSP32_Q_CONVERT(-0.93893248353), DSP32_Q_CONVERT(0.34410142599),
    DSP32_Q_CONVERT(-0.68954054474), DSP32_Q_CONVERT(-0.72424708295),
    DSP32_Q_CONVERT(-0.93733901191), DSP32_Q_CONVERT(0.34841868025),
    DSP32_Q_CONVERT(-0.69175925836), DSP32_Q_CONVERT(-0.72212819393),
    DSP32_Q_CONVERT(-0.93572568948), DSP32_Q_CONVERT(0.35272855576),
    DSP32_Q_CONVERT(-0.69397146089), DSP32_Q_CONVERT(-0.72000250796),
    DSP32_Q_CONVERT(-0.93409255040), DSP32_Q_CONVERT(0.35703096123),
    DSP32_Q_CONVERT(-0.69617713149), DSP32_Q_CONVERT(-0.71787004506),
    DSP32_Q_CONVERT(-0.93243962927), DSP32_Q_CONVERT(0.36132580557),
    DSP32_Q_CONVERT(-0.69837624941), DSP32_Q_CONVERT(-0.71573082528),
    DSP32_Q_CONVERT(-0.93076696108), DSP32_Q_CONVERT(0.36561299780),
    DSP32_Q_CONVERT(-0.70056879394), DSP32_Q_CONVERT(-0.71358486878),
    DSP32_Q_CONVERT(-0.92907458126), DSP32_Q_CONVERT(0.36989244715),
    DSP32_Q_CONVERT(-0.70275474446), DSP32_Q_CONVERT(-0.71143219575),
    DSP32_Q_CONVERT(-0.92736252565), DSP32_Q_CONVERT(0.37416406297),
    DSP32_Q_CONVERT(-0.70493408038), DSP32_Q_CONVERT(-0.70927282644),
    DSP32_Q_CONVERT(-0.92563083051), DSP32_Q_CONVERT(0.37842775481),
    DSP32_Q_CONVERT(-0.70710678119), DSP32_Q_CONVERT(-0.70710678119),
    DSP32_Q_CONVERT(-0.92387953251), DSP32_Q_CONVERT(0.38268343237),
    DSP32_Q_CONVERT(-0.70927282644), DSP32_Q_CONVERT(-0.70493408038),
    DSP32_Q_CONVERT(-0.92210866874), DSP32_Q_CONVERT(0.38693100551),
    DSP32_Q_CONVERT(-0.71143219575), DSP32_Q_CONVERT(-0.70275474446),
    DSP32_Q_CONVERT(-0.92031827671), DSP32_Q_CONVERT(0.39117038430),
    DSP32_Q_CONVERT(-0.71358486878), DSP32_Q_CONVERT(-0.70056879394),
    DSP32_Q_CONVERT(-0.91850839433), DSP32_Q_CONVERT(0.39540147895),
    DSP32_Q_CONVERT(-0.71573082528), DSP32_Q_CONVERT(-0.69837624941),
    DSP32_Q_CONVERT(-0.91667905992), DSP32_Q_CONVERT(0.39962419985),
    DSP32_Q_CONVERT(-0.71787004506), DSP32_Q_CONVERT(-0.69617713149),
    DSP32_Q_CONVERT(-0.91483031224), DSP32_Q_CONVERT(0.40383845757),
    DSP32_Q_CONVERT(-0.72000250796), DSP32_Q_CONVERT(-0.69397146089),
    DSP32_Q_CONVERT(-0.91296219043), DSP32_Q_CONVERT(0.40804416286),
    DSP32_Q_CONVERT(-0.72212819393), DSP32_Q_CONVERT(-0.69175925836),
    DSP32_Q_CONVERT(-0.91107473406), DSP32_Q_CONVERT(0.41224122667),
    DSP32_Q_CONVERT(-0.72424708295), DSP32_Q_CONVERT(-0.68954054474),
    DSP32_Q_CONVERT(-0.90916798309), DSP32_Q_CONVERT(0.41642956010),
    DSP32_Q_CONVERT(-0.72635915508), DSP32_Q_CONVERT(-0.68731534089),
    DSP32_Q_CONVERT(-0.90724197792), DSP32_Q_CONVERT(0.42060907445),
    DSP32_Q_CONVERT(-0.72846439045), DSP32_Q_CONVERT(-0.68508366777),
    DSP32_Q_CONVERT(-0.90529675932), DSP32_Q_CONVERT(0.42477968121),
    DSP32_Q_CONVERT(-0.73056276923), DSP32_Q_CONVERT(-0.68284554639),
    DSP32_Q_CONVERT(-0.90333236849), DSP32_Q_CONVERT(0.42894129206),
    DSP32_Q_CONVERT(-0.73265427167), DSP32_Q_CONVERT(-0.68060099780),
    DSP32_Q_CONVERT(-0.90134884705), DSP32_Q_CONVERT(0.43309381885),
    DSP32_Q_CONVERT(-0.73473887810), DSP32_Q_CONVERT(-0.67835004313),
    DSP32_Q_CONVERT(-0.89934623698), DSP32_Q_CONVERT(0.43723717366),
    DSP32_Q_CONVERT(-0.73681656888), DSP32_Q_CONVERT(-0.67609270358),
    DSP32_Q_CONVERT(-0.89732458071), DSP32_Q_CONVERT(0.44137126873),
    DSP32_Q_CONVERT(-0.73888732446), DSP32_Q_CONVERT(-0.67382900038),
    DSP32_Q_CONVERT(-0.89528392104), DSP32_Q_CONVERT(0.44549601651),
    DSP32_Q_CONVERT(-0.74095112535), DSP32_Q_CONVERT(-0.67155895485),
    DSP32_Q_CONVERT(-0.89322430120), DSP32_Q_CONVERT(0.44961132965),
    DSP32_Q_CONVERT(-0.74300795214), DSP32_Q_CONVERT(-0.66928258835),
    DSP32_Q_CONVERT(-0.89114576479), DSP32_Q_CONVERT(0.45371712100),
    DSP32_Q_CONVERT(-0.74505778544), DSP32_Q_CONVERT(-0.66699992230),
    DSP32_Q_CONVERT(-0.88904835585), DSP32_Q_CONVERT(0.45781330360),
    DSP32_Q_CONVERT(-0.74710060598), DSP32_Q_CONVERT(-0.66471097820),
    DSP32_Q_CONVERT(-0.88693211879), DSP32_Q_CONVERT(0.46189979070),
    DSP32_Q_CONVERT(-0.74913639452), DSP32_Q_CONVERT(-0.66241577759),
    DSP32_Q_CONVERT(-0.88479709843), DSP32_Q_CONVERT(0.46597649577),
    DSP32_Q_CONVERT(-0.75116513191), DSP32_Q_CONVERT(-0.66011434207),
    DSP32_Q_CONVERT(-0.88264333998), DSP32_Q_CONVERT(0.47004333246),
    DSP32_Q_CONVERT(-0.75318679904), DSP32_Q_CONVERT(-0.65780669330),
    DSP32_Q_CONVERT(-0.88047088905), DSP32_Q_CONVERT(0.47410021465),
    DSP32_Q_CONVERT(-0.75520137690), DSP32_Q_CONVERT(-0.65549285300),
    DSP32_Q_CONVERT(-0.87827979166), DSP32_Q_CONVERT(0.47814705642),
    DSP32_Q_CONVERT(-0.75720884651), DSP32_Q_CONVERT(-0.65317284295),
    DSP32_Q_CONVERT(-0.87607009420), DSP32_Q_CONVERT(0.48218377208),
    DSP32_Q_CONVERT(-0.75920918898), DSP32_Q_CONVERT(-0.65084668500),
    DSP32_Q_CONVERT(-0.87384184347), DSP32_Q_CONVERT(0.48621027612),
    DSP32_Q_CONVERT(-0.76120238548), DSP32_Q_CONVERT(-0.64851440102),
    DSP32_Q_CONVERT(-0.87159508666), DSP32_Q_CONVERT(0.49022648329),
    DSP32_Q_CONVERT(-0.76318841726), DSP32_Q_CONVERT(-0.64617601298),
    DSP32_Q_CONVERT(-0.86932987135), DSP32_Q_CONVERT(0.49423230852),
    DSP32_Q_CONVERT(-0.76516726562), DSP32_Q_CONVERT(-0.64383154289),
    DSP32_Q_CONVERT(-0.86704624552), DSP32_Q_CONVERT(0.49822766697),
    DSP32_Q_CONVERT(-0.76713891194), DSP32_Q_CONVERT(-0.64148101281),
    DSP32_Q_CONVERT(-0.86474425752), DSP32_Q_CONVERT(0.50221247405),
    DSP32_Q_CONVERT(-0.76910333765), DSP32_Q_CONVERT(-0.63912444486),
    DSP32_Q_CONVERT(-0.86242395611), DSP32_Q_CONVERT(0.50618664535),
    DSP32_Q_CONVERT(-0.77106052426), DSP32_Q_CONVERT(-0.63676186124),
    DSP32_Q_CONVERT(-0.86008539043), DSP32_Q_CONVERT(0.51015009671),
    DSP32_Q_CONVERT(-0.77301045336), DSP32_Q_CONVERT(-0.63439328416),
    DSP32_Q_CONVERT(-0.85772861000), DSP32_Q_CONVERT(0.51410274419),
    DSP32_Q_CONVERT(-0.77495310659), DSP32_Q_CONVERT(-0.63201873594),
    DSP32_Q_CONVERT(-0.85535366474), DSP32_Q_CONVERT(0.51804450410),
    DSP32_Q_CONVERT(-0.77688846567), DSP32_Q_CONVERT(-0.62963823891),
    DSP32_Q_CONVERT(-0.85296060493), DSP32_Q_CONVERT(0.52197529294),
    DSP32_Q_CONVERT(-0.77881651238), DSP32_Q_CONVERT(-0.62725181550),
    DSP32_Q_CONVERT(-0.85054948127), DSP32_Q_CONVERT(0.52589502747),
    DSP32_Q_CONVERT(-0.78073722857), DSP32_Q_CONVERT(-0.62485948814),
    DSP32_Q_CONVERT(-0.84812034480), DSP32_Q_CONVERT(0.52980362469),
    DSP32_Q_CONVERT(-0.78265059617), DSP32_Q_CONVERT(-0.62246127937),
    DSP32_Q_CONVERT(-0.84567324699), DSP32_Q_CONVERT(0.53370100181),
    DSP32_Q_CONVERT(-0.78455659716), DSP32_Q_CONVERT(-0.62005721176),
    DSP32_Q_CONVERT(-0.84320823964), DSP32_Q_CONVERT(0.53758707630),
    DSP32_Q_CONVERT(-0.78645521360), DSP32_Q_CONVERT(-0.61764730794),
    DSP32_Q_CONVERT(-0.84072537497), DSP32_Q_CONVERT(0.54146176585),
    DSP32_Q_CONVERT(-0.78834642763), DSP32_Q_CONVERT(-0.61523159058),
    DSP32_Q_CONVERT(-0.83822470555), DSP32_Q_CONVERT(0.54532498842),
    DSP32_Q_CONVERT(-0.79023022144), DSP32_Q_CONVERT(-0.61281008243),
    DSP32_Q_CONVERT(-0.83570628435), DSP32_Q_CONVERT(0.54917666219),
    DSP32_Q_CONVERT(-0.79210657730), DSP32_Q_CONVERT(-0.61038280628),
    DSP32_Q_CONVERT(-0.83317016470), DSP32_Q_CONVERT(0.55301670558),
    DSP32_Q_CONVERT(-0.79397547755), DSP32_Q_CONVERT(-0.60794978497),
    DSP32_Q_CONVERT(-0.83061640031), DSP32_Q_CONVERT(0.55684503728),
    DSP32_Q_CONVERT(-0.79583690461), DSP32_Q_CONVERT(-0.60551104140),
    DSP32_Q_CONVERT(-0.82804504526), DSP32_Q_CONVERT(0.56066157620),
    DSP32_Q_CONVERT(-0.79769084094), DSP32_Q_CONVERT(-0.60306659854),
    DSP32_Q_CONVERT(-0.82545615400), DSP32_Q_CONVERT(0.56446624152),
    DSP32_Q_CONVERT(-0.79953726911), DSP32_Q_CONVERT(-0.60061647938),
    DSP32_Q_CONVERT(-0.82284978138), DSP32_Q_CONVERT(0.56825895267),
    DSP32_Q_CONVERT(-0.80137617172), DSP32_Q_CONVERT(-0.59816070700),
    DSP32_Q_CONVERT(-0.82022598257), DSP32_Q_CONVERT(0.57203962932),
    DSP32_Q_CONVERT(-0.80320753148), DSP32_Q_CONVERT(-0.59569930449),
    DSP32_Q_CONVERT(-0.81758481315), DSP32_Q_CONVERT(0.57580819142),
    DSP32_Q_CONVERT(-0.80503133114), DSP32_Q_CONVERT(-0.59323229504),
    DSP32_Q_CONVERT(-0.81492632906), DSP32_Q_CONVERT(0.57956455914),
    DSP32_Q_CONVERT(-0.80684755354), DSP32_Q_CONVERT(-0.59075970186),
    DSP32_Q_CONVERT(-0.81225058659), DSP32_Q_CONVERT(0.58330865294),
    DSP32_Q_CONVERT(-0.80865618159), DSP32_Q_CONVERT(-0.58828154822),
    DSP32_Q_CONVERT(-0.80955764240), DSP32_Q_CONVERT(0.58704039352),
    DSP32_Q_CONVERT(-0.81045719825), DSP32_Q_CONVERT(-0.58579785746),
    DSP32_Q_CONVERT(-0.80684755354), DSP32_Q_CONVERT(0.59075970186),
    DSP32_Q_CONVERT(-0.81225058659), DSP32_Q_CONVERT(-0.58330865294),
    DSP32_Q_CONVERT(-0.80412037740), DSP32_Q_CONVERT(0.59446649918),
    DSP32_Q_CONVERT(-0.81403632971), DSP32_Q_CONVERT(-0.58081395810),
    DSP32_Q_CONVERT(-0.80137617172), DSP32_Q_CONVERT(0.59816070700),
    DSP32_Q_CONVERT(-0.81581441081), DSP32_Q_CONVERT(-0.57831379641),
    DSP32_Q_CONVERT(-0.79861499463), DSP32_Q_CONVERT(0.60184224706),
    DSP32_Q_CONVERT(-0.81758481315), DSP32_Q_CONVERT(-0.57580819142),
    DSP32_Q_CONVERT(-0.79583690461), DSP32_Q_CONVERT(0.60551104140),
    DSP32_Q_CONVERT(-0.81934752008), DSP32_Q_CONVERT(-0.57329716670),
    DSP32_Q_CONVERT(-0.79304196048), DSP32_Q_CONVERT(0.60916701234),
    DSP32_Q_CONVERT(-0.82110251499), DSP32_Q_CONVERT(-0.57078074589),
    DSP32_Q_CONVERT(-0.79023022144), DSP32_Q_CONVERT(0.61281008243),
    DSP32_Q_CONVERT(-0.82284978138), DSP32_Q_CONVERT(-0.56825895267),
    DSP32_Q_CONVERT(-0.78740174703), DSP32_Q_CONVERT(0.61644017453),
    DSP32_Q_CONVERT(-0.82458930279), DSP32_Q_CONVERT(-0.56573181078),
    DSP32_Q_CONVERT(-0.78455659716), DSP32_Q_CONVERT(0.62005721176),
    DSP32_Q_CONVERT(-0.82632106285), DSP32_Q_CONVERT(-0.56319934401),
    DSP32_Q_CONVERT(-0.78169483207), DSP32_Q_CONVERT(0.62366111753),
    DSP32_Q_CONVERT(-0.82804504526), DSP32_Q_CONVERT(-0.56066157620),
    DSP32_Q_CONVERT(-0.77881651238), DSP32_Q_CONVERT(0.62725181550),
    DSP32_Q_CONVERT(-0.82976123379), DSP32_Q_CONVERT(-0.55811853122),
    DSP32_Q_CONVERT(-0.77592169904), DSP32_Q_CONVERT(0.63082922963),
    DSP32_Q_CONVERT(-0.83146961230), DSP32_Q_CONVERT(-0.55557023302),
    DSP32_Q_CONVERT(-0.77301045336), DSP32_Q_CONVERT(0.63439328416),
    DSP32_Q_CONVERT(-0.83317016470), DSP32_Q_CONVERT(-0.55301670558),
    DSP32_Q_CONVERT(-0.77008283699), DSP32_Q_CONVERT(0.63794390362),
    DSP32_Q_CONVERT(-0.83486287499), DSP32_Q_CONVERT(-0.55045797294),
    DSP32_Q_CONVERT(-0.76713891194), DSP32_Q_CONVERT(0.64148101281),
    DSP32_Q_CONVERT(-0.83654772722), DSP32_Q_CONVERT(-0.54789405917),
    DSP32_Q_CONVERT(-0.76417874054), DSP32_Q_CONVERT(0.64500453682),
    DSP32_Q_CONVERT(-0.83822470555), DSP32_Q_CONVERT(-0.54532498842),
    DSP32_Q_CONVERT(-0.76120238548), DSP32_Q_CONVERT(0.64851440102),
    DSP32_Q_CONVERT(-0.83989379420), DSP32_Q_CONVERT(-0.54275078486),
    DSP32_Q_CONVERT(-0.75820990981), DSP32_Q_CONVERT(0.65201053110),
    DSP32_Q_CONVERT(-0.84155497744), DSP32_Q_CONVERT(-0.54017147273),
    DSP32_Q_CONVERT(-0.75520137690), DSP32_Q_CONVERT(0.65549285300),
    DSP32_Q_CONVERT(-0.84320823964), DSP32_Q_CONVERT(-0.53758707630),
    DSP32_Q_CONVERT(-0.75217685045), DSP32_Q_CONVERT(0.65896129298),
    DSP32_Q_CONVERT(-0.84485356525), DSP32_Q_CONVERT(-0.53499761989),
    DSP32_Q_CONVERT(-0.74913639452), DSP32_Q_CONVERT(0.66241577759),
    DSP32_Q_CONVERT(-0.84649093877), DSP32_Q_CONVERT(-0.53240312788),
    DSP32_Q_CONVERT(-0.74608007351), DSP32_Q_CONVERT(0.66585623367),
    DSP32_Q_CONVERT(-0.84812034480), DSP32_Q_CONVERT(-0.52980362469),
    DSP32_Q_CONVERT(-0.74300795214), DSP32_Q_CONVERT(0.66928258835),
    DSP32_Q_CONVERT(-0.84974176800), DSP32_Q_CONVERT(-0.52719913478),
    DSP32_Q_CONVERT(-0.73992009546), DSP32_Q_CONVERT(0.67269476907),
    DSP32_Q_CONVERT(-0.85135519311), DSP32_Q_CONVERT(-0.52458968268),
    DSP32_Q_CONVERT(-0.73681656888), DSP32_Q_CONVERT(0.67609270358),
    DSP32_Q_CONVERT(-0.85296060493), DSP32_Q_CONVERT(-0.52197529294),
    DSP32_Q_CONVERT(-0.73369743811), DSP32_Q_CONVERT(0.67947631990),
    DSP32_Q_CONVERT(-0.85455798837), DSP32_Q_CONVERT(-0.51935599017),
    DSP32_Q_CONVERT(-0.73056276923), DSP32_Q_CONVERT(0.68284554639),
    DSP32_Q_CONVERT(-0.85614732838), DSP32_Q_CONVERT(-0.51673179902),
    DSP32_Q_CONVERT(-0.72741262860), DSP32_Q_CONVERT(0.68620031168),
    DSP32_Q_CONVERT(-0.85772861000), DSP32_Q_CONVERT(-0.51410274419),
    DSP32_Q_CONVERT(-0.72424708295), DSP32_Q_CONVERT(0.68954054474),
    DSP32_Q_CONVERT(-0.85930181836), DSP32_Q_CONVERT(-0.51146885044),
    DSP32_Q_CONVERT(-0.72106619931), DSP32_Q_CONVERT(0.69286617482),
    DSP32_Q_CONVERT(-0.86086693864), DSP32_Q_CONVERT(-0.50883014254),
    DSP32_Q_CONVERT(-0.71787004506), DSP32_Q_CONVERT(0.69617713149),
    DSP32_Q_CONVERT(-0.86242395611), DSP32_Q_CONVERT(-0.50618664535),
    DSP32_Q_CONVERT(-0.71465868786), DSP32_Q_CONVERT(0.69947334464),
    DSP32_Q_CONVERT(-0.86397285612), DSP32_Q_CONVERT(-0.50353838373),
    DSP32_Q_CONVERT(-0.71143219575), DSP32_Q_CONVERT(0.70275474446),
    DSP32_Q_CONVERT(-0.86551362409), DSP32_Q_CONVERT(-0.50088538261),
    DSP32_Q_CONVERT(-0.70819063703), DSP32_Q_CONVERT(0.70602126145),
    DSP32_Q_CONVERT(-0.86704624552), DSP32_Q_CONVERT(-0.49822766697),
    DSP32_Q_CONVERT(-0.70493408038), DSP32_Q_CONVERT(0.70927282644),
    DSP32_Q_CONVERT(-0.86857070597), DSP32_Q_CONVERT(-0.49556526183),
    DSP32_Q_CONVERT(-0.70166259474), DSP32_Q_CONVERT(0.71250937056),
    DSP32_Q_CONVERT(-0.87008699111), DSP32_Q_CONVERT(-0.49289819223),
    DSP32_Q_CONVERT(-0.69837624941), DSP32_Q_CONVERT(0.71573082528),
    DSP32_Q_CONVERT(-0.87159508666), DSP32_Q_CONVERT(-0.49022648329),
    DSP32_Q_CONVERT(-0.69507511398), DSP32_Q_CONVERT(0.71893712237),
    DSP32_Q_CONVERT(-0.87309497842), DSP32_Q_CONVERT(-0.48755016015),
    DSP32_Q_CONVERT(-0.69175925836), DSP32_Q_CONVERT(0.72212819393),
    DSP32_Q_CONVERT(-0.87458665228), DSP32_Q_CONVERT(-0.48486924800),
    DSP32_Q_CONVERT(-0.68842875278), DSP32_Q_CONVERT(0.72530397237),
    DSP32_Q_CONVERT(-0.87607009420), DSP32_Q_CONVERT(-0.48218377208),
    DSP32_Q_CONVERT(-0.68508366777), DSP32_Q_CONVERT(0.72846439045),
    DSP32_Q_CONVERT(-0.87754529021), DSP32_Q_CONVERT(-0.47949375766),
    DSP32_Q_CONVERT(-0.68172407417), DSP32_Q_CONVERT(0.73160938122),
    DSP32_Q_CONVERT(-0.87901222643), DSP32_Q_CONVERT(-0.47679923006),
    DSP32_Q_CONVERT(-0.67835004313), DSP32_Q_CONVERT(0.73473887810),
    DSP32_Q_CONVERT(-0.88047088905), DSP32_Q_CONVERT(-0.47410021465),
    DSP32_Q_CONVERT(-0.67496164610), DSP32_Q_CONVERT(0.73785281479),
    DSP32_Q_CONVERT(-0.88192126435), DSP32_Q_CONVERT(-0.47139673683),
    DSP32_Q_CONVERT(-0.67155895485), DSP32_Q_CONVERT(0.74095112535),
    DSP32_Q_CONVERT(-0.88336333867), DSP32_Q_CONVERT(-0.46868882204),
    DSP32_Q_CONVERT(-0.66814204143), DSP32_Q_CONVERT(0.74403374418),
    DSP32_Q_CONVERT(-0.88479709843), DSP32_Q_CONVERT(-0.46597649577),
    DSP32_Q_CONVERT(-0.66471097820), DSP32_Q_CONVERT(0.74710060598),
    DSP32_Q_CONVERT(-0.88622253015), DSP32_Q_CONVERT(-0.46325978355),
    DSP32_Q_CONVERT(-0.66126583784), DSP32_Q_CONVERT(0.75015164581),
    DSP32_Q_CONVERT(-0.88763962040), DSP32_Q_CONVERT(-0.46053871096),
    DSP32_Q_CONVERT(-0.65780669330), DSP32_Q_CONVERT(0.75318679904),
    DSP32_Q_CONVERT(-0.88904835585), DSP32_Q_CONVERT(-0.45781330360),
    DSP32_Q_CONVERT(-0.65433361783), DSP32_Q_CONVERT(0.75620600141),
    DSP32_Q_CONVERT(-0.89044872324), DSP32_Q_CONVERT(-0.45508358713),
    DSP32_Q_CONVERT(-0.65084668500), DSP32_Q_CONVERT(0.75920918898),
    DSP32_Q_CONVERT(-0.89184070939), DSP32_Q_CONVERT(-0.45234958723),
    DSP32_Q_CONVERT(-0.64734596864), DSP32_Q_CONVERT(0.76219629813),
    DSP32_Q_CONVERT(-0.89322430120), DSP32_Q_CONVERT(-0.44961132965),
    DSP32_Q_CONVERT(-0.64383154289), DSP32_Q_CONVERT(0.76516726562),
    DSP32_Q_CONVERT(-0.89459948563), DSP32_Q_CONVERT(-0.44686884016),
    DSP32_Q_CONVERT(-0.64030348218), DSP32_Q_CONVERT(0.76812202852),
    DSP32_Q_CONVERT(-0.89596624976), DSP32_Q_CONVERT(-0.44412214457),
    DSP32_Q_CONVERT(-0.63676186124), DSP32_Q_CONVERT(0.77106052426),
    DSP32_Q_CONVERT(-0.89732458071), DSP32_Q_CONVERT(-0.44137126873),
    DSP32_Q_CONVERT(-0.63320675505), DSP32_Q_CONVERT(0.77398269061),
    DSP32_Q_CONVERT(-0.89867446569), DSP32_Q_CONVERT(-0.43861623854),
    DSP32_Q_CONVERT(-0.62963823891), DSP32_Q_CONVERT(0.77688846567),
    DSP32_Q_CONVERT(-0.90001589202), DSP32_Q_CONVERT(-0.43585707992),
    DSP32_Q_CONVERT(-0.62605638840), DSP32_Q_CONVERT(0.77977778792),
    DSP32_Q_CONVERT(-0.90134884705), DSP32_Q_CONVERT(-0.43309381885),
    DSP32_Q_CONVERT(-0.62246127937), DSP32_Q_CONVERT(0.78265059617),
    DSP32_Q_CONVERT(-0.90267331824), DSP32_Q_CONVERT(-0.43032648134),
    DSP32_Q_CONVERT(-0.61885298796), DSP32_Q_CONVERT(0.78550682956),
    DSP32_Q_CONVERT(-0.90398929312), DSP32_Q_CONVERT(-0.42755509343),
    DSP32_Q_CONVERT(-0.61523159058), DSP32_Q_CONVERT(0.78834642763),
    DSP32_Q_CONVERT(-0.90529675932), DSP32_Q_CONVERT(-0.42477968121),
    DSP32_Q_CONVERT(-0.61159716393), DSP32_Q_CONVERT(0.79116933022),
    DSP32_Q_CONVERT(-0.90659570451), DSP32_Q_CONVERT(-0.42200027080),
    DSP32_Q_CONVERT(-0.60794978497), DSP32_Q_CONVERT(0.79397547755),
    DSP32_Q_CONVERT(-0.90788611649), DSP32_Q_CONVERT(-0.41921688836),
    DSP32_Q_CONVERT(-0.60428953095), DSP32_Q_CONVERT(0.79676481021),
    DSP32_Q_CONVERT(-0.90916798309), DSP32_Q_CONVERT(-0.41642956010),
    DSP32_Q_CONVERT(-0.60061647938), DSP32_Q_CONVERT(0.79953726911),
    DSP32_Q_CONVERT(-0.91044129226), DSP32_Q_CONVERT(-0.41363831224),
    DSP32_Q_CONVERT(-0.59693070806), DSP32_Q_CONVERT(0.80229279554),
    DSP32_Q_CONVERT(-0.91170603201), DSP32_Q_CONVERT(-0.41084317106),
    DSP32_Q_CONVERT(-0.59323229504), DSP32_Q_CONVERT(0.80503133114),
    DSP32_Q_CONVERT(-0.91296219043), DSP32_Q_CONVERT(-0.40804416286),
    DSP32_Q_CONVERT(-0.58952131864), DSP32_Q_CONVERT(0.80775281793),
    DSP32_Q_CONVERT(-0.91420975570), DSP32_Q_CONVERT(-0.40524131400),
    DSP32_Q_CONVERT(-0.58579785746), DSP32_Q_CONVERT(0.81045719825),
    DSP32_Q_CONVERT(-0.91544871609), DSP32_Q_CONVERT(-0.40243465086),
    DSP32_Q_CONVERT(-0.58206199034), DSP32_Q_CONVERT(0.81314441485),
    DSP32_Q_CONVERT(-0.91667905992), DSP32_Q_CONVERT(-0.39962419985),
    DSP32_Q_CONVERT(-0.57831379641), DSP32_Q_CONVERT(0.81581441081),
    DSP32_Q_CONVERT(-0.91790077562), DSP32_Q_CONVERT(-0.39680998742),
    DSP32_Q_CONVERT(-0.57455335505), DSP32_Q_CONVERT(0.81846712958),
    DSP32_Q_CONVERT(-0.91911385169), DSP32_Q_CONVERT(-0.39399204006),
    DSP32_Q_CONVERT(-0.57078074589), DSP32_Q_CONVERT(0.82110251499),
    DSP32_Q_CONVERT(-0.92031827671), DSP32_Q_CONVERT(-0.39117038430),
    DSP32_Q_CONVERT(-0.56699604883), DSP32_Q_CONVERT(0.82372051123),
    DSP32_Q_CONVERT(-0.92151403934), DSP32_Q_CONVERT(-0.38834504670),
    DSP32_Q_CONVERT(-0.56319934401), DSP32_Q_CONVERT(0.82632106285),
    DSP32_Q_CONVERT(-0.92270112833), DSP32_Q_CONVERT(-0.38551605384),
    DSP32_Q_CONVERT(-0.55939071186), DSP32_Q_CONVERT(0.82890411477),
    DSP32_Q_CONVERT(-0.92387953251), DSP32_Q_CONVERT(-0.38268343237),
    DSP32_Q_CONVERT(-0.55557023302), DSP32_Q_CONVERT(0.83146961230),
    DSP32_Q_CONVERT(-0.92504924078), DSP32_Q_CONVERT(-0.37984720892),
    DSP32_Q_CONVERT(-0.55173798840), DSP32_Q_CONVERT(0.83401750111),
    DSP32_Q_CONVERT(-0.92621024214), DSP32_Q_CONVERT(-0.37700741022),
    DSP32_Q_CONVERT(-0.54789405917), DSP32_Q_CONVERT(0.83654772722),
    DSP32_Q_CONVERT(-0.92736252565), DSP32_Q_CONVERT(-0.37416406297),
    DSP32_Q_CONVERT(-0.54403852673), DSP32_Q_CONVERT(0.83906023707),
    DSP32_Q_CONVERT(-0.92850608047), DSP32_Q_CONVERT(-0.37131719395),
    DSP32_Q_CONVERT(-0.54017147273), DSP32_Q_CONVERT(0.84155497744),
    DSP32_Q_CONVERT(-0.92964089584), DSP32_Q_CONVERT(-0.36846682995),
    DSP32_Q_CONVERT(-0.53629297907), DSP32_Q_CONVERT(0.84403189549),
    DSP32_Q_CONVERT(-0.93076696108), DSP32_Q_CONVERT(-0.36561299780),
    DSP32_Q_CONVERT(-0.53240312788), DSP32_Q_CONVERT(0.84649093877),
    DSP32_Q_CONVERT(-0.93188426558), DSP32_Q_CONVERT(-0.36275572437),
    DSP32_Q_CONVERT(-0.52850200154), DSP32_Q_CONVERT(0.84893205521),
    DSP32_Q_CONVERT(-0.93299279883), DSP32_Q_CONVERT(-0.35989503653),
    DSP32_Q_CONVERT(-0.52458968268), DSP32_Q_CONVERT(0.85135519311),
    DSP32_Q_CONVERT(-0.93409255040), DSP32_Q_CONVERT(-0.35703096123),
    DSP32_Q_CONVERT(-0.52066625414), DSP32_Q_CONVERT(0.85376030114),
    DSP32_Q_CONVERT(-0.93518350994), DSP32_Q_CONVERT(-0.35416352542),
    DSP32_Q_CONVERT(-0.51673179902), DSP32_Q_CONVERT(0.85614732838),
    DSP32_Q_CONVERT(-0.93626566717), DSP32_Q_CONVERT(-0.35129275609),
    DSP32_Q_CONVERT(-0.51278640063), DSP32_Q_CONVERT(0.85851622426),
    DSP32_Q_CONVERT(-0.93733901191), DSP32_Q_CONVERT(-0.34841868025),
    DSP32_Q_CONVERT(-0.50883014254), DSP32_Q_CONVERT(0.86086693864),
    DSP32_Q_CONVERT(-0.93840353406), DSP32_Q_CONVERT(-0.34554132496),
    DSP32_Q_CONVERT(-0.50486310853), DSP32_Q_CONVERT(0.86319942171),
    DSP32_Q_CONVERT(-0.93945922360), DSP32_Q_CONVERT(-0.34266071731),
    DSP32_Q_CONVERT(-0.50088538261), DSP32_Q_CONVERT(0.86551362409),
    DSP32_Q_CONVERT(-0.94050607059), DSP32_Q_CONVERT(-0.33977688441),
    DSP32_Q_CONVERT(-0.49689704902), DSP32_Q_CONVERT(0.86780949676),
    DSP32_Q_CONVERT(-0.94154406518), DSP32_Q_CONVERT(-0.33688985339),
    DSP32_Q_CONVERT(-0.49289819223), DSP32_Q_CONVERT(0.87008699111),
    DSP32_Q_CONVERT(-0.94257319760), DSP32_Q_CONVERT(-0.33399965144),
    DSP32_Q_CONVERT(-0.48888889692), DSP32_Q_CONVERT(0.87234605889),
    DSP32_Q_CONVERT(-0.94359345816), DSP32_Q_CONVERT(-0.33110630576),
    DSP32_Q_CONVERT(-0.48486924800), DSP32_Q_CONVERT(0.87458665228),
    DSP32_Q_CONVERT(-0.94460483726), DSP32_Q_CONVERT(-0.32820984358),
    DSP32_Q_CONVERT(-0.48083933060), DSP32_Q_CONVERT(0.87680872381),
    DSP32_Q_CONVERT(-0.94560732538), DSP32_Q_CONVERT(-0.32531029216),
    DSP32_Q_CONVERT(-0.47679923006), DSP32_Q_CONVERT(0.87901222643),
    DSP32_Q_CONVERT(-0.94660091308), DSP32_Q_CONVERT(-0.32240767880),
    DSP32_Q_CONVERT(-0.47274903195), DSP32_Q_CONVERT(0.88119711347),
    DSP32_Q_CONVERT(-0.94758559102), DSP32_Q_CONVERT(-0.31950203082),
    DSP32_Q_CONVERT(-0.46868882204), DSP32_Q_CONVERT(0.88336333867),
    DSP32_Q_CONVERT(-0.94856134992), DSP32_Q_CONVERT(-0.31659337556),
    DSP32_Q_CONVERT(-0.46461868631), DSP32_Q_CONVERT(0.88551085614),
    DSP32_Q_CONVERT(-0.94952818059), DSP32_Q_CONVERT(-0.31368174040),
    DSP32_Q_CONVERT(-0.46053871096), DSP32_Q_CONVERT(0.88763962040),
    DSP32_Q_CONVERT(-0.95048607395), DSP32_Q_CONVERT(-0.31076715275),
    DSP32_Q_CONVERT(-0.45644898240), DSP32_Q_CONVERT(0.88974958638),
    DSP32_Q_CONVERT(-0.95143502097), DSP32_Q_CONVERT(-0.30784964004),
    DSP32_Q_CONVERT(-0.45234958723), DSP32_Q_CONVERT(0.89184070939),
    DSP32_Q_CONVERT(-0.95237501272), DSP32_Q_CONVERT(-0.30492922974),
    DSP32_Q_CONVERT(-0.44824061229), DSP32_Q_CONVERT(0.89391294515),
    DSP32_Q_CONVERT(-0.95330604035), DSP32_Q_CONVERT(-0.30200594932),
    DSP32_Q_CONVERT(-0.44412214457), DSP32_Q_CONVERT(0.89596624976),
    DSP32_Q_CONVERT(-0.95422809511), DSP32_Q_CONVERT(-0.29907982631),
    DSP32_Q_CONVERT(-0.43999427131), DSP32_Q_CONVERT(0.89800057974),
    DSP32_Q_CONVERT(-0.95514116831), DSP32_Q_CONVERT(-0.29615088824),
    DSP32_Q_CONVERT(-0.43585707992), DSP32_Q_CONVERT(0.90001589202),
    DSP32_Q_CONVERT(-0.95604525135), DSP32_Q_CONVERT(-0.29321916269),
    DSP32_Q_CONVERT(-0.43171065803), DSP32_Q_CONVERT(0.90201214390),
    DSP32_Q_CONVERT(-0.95694033573), DSP32_Q_CONVERT(-0.29028467725),
    DSP32_Q_CONVERT(-0.42755509343), DSP32_Q_CONVERT(0.90398929312),
    DSP32_Q_CONVERT(-0.95782641303), DSP32_Q_CONVERT(-0.28734745954),
    DSP32_Q_CONVERT(-0.42339047414), DSP32_Q_CONVERT(0.90594729781),
    DSP32_Q_CONVERT(-0.95870347490), DSP32_Q_CONVERT(-0.28440753721),
    DSP32_Q_CONVERT(-0.41921688836), DSP32_Q_CONVERT(0.90788611649),
    DSP32_Q_CONVERT(-0.95957151308), DSP32_Q_CONVERT(-0.28146493793),
    DSP32_Q_CONVERT(-0.41503442448), DSP32_Q_CONVERT(0.90980570810),
    DSP32_Q_CONVERT(-0.96043051942), DSP32_Q_CONVERT(-0.27851968939),
    DSP32_Q_CONVERT(-0.41084317106), DSP32_Q_CONVERT(0.91170603201),
    DSP32_Q_CONVERT(-0.96128048581), DSP32_Q_CONVERT(-0.27557181931),
    DSP32_Q_CONVERT(-0.40664321687), DSP32_Q_CONVERT(0.91358704795),
    DSP32_Q_CONVERT(-0.96212140427), DSP32_Q_CONVERT(-0.27262135545),
    DSP32_Q_CONVERT(-0.40243465086), DSP32_Q_CONVERT(0.91544871609),
    DSP32_Q_CONVERT(-0.96295326687), DSP32_Q_CONVERT(-0.26966832557),
    DSP32_Q_CONVERT(-0.39821756215), DSP32_Q_CONVERT(0.91729099701),
    DSP32_Q_CONVERT(-0.96377606580), DSP32_Q_CONVERT(-0.26671275747),
    DSP32_Q_CONVERT(-0.39399204006), DSP32_Q_CONVERT(0.91911385169),
    DSP32_Q_CONVERT(-0.96458979329), DSP32_Q_CONVERT(-0.26375467897),
    DSP32_Q_CONVERT(-0.38975817407), DSP32_Q_CONVERT(0.92091724153),
    DSP32_Q_CONVERT(-0.96539444170), DSP32_Q_CONVERT(-0.26079411792),
    DSP32_Q_CONVERT(-0.38551605384), DSP32_Q_CONVERT(0.92270112833),
    DSP32_Q_CONVERT(-0.96619000345), DSP32_Q_CONVERT(-0.25783110216),
    DSP32_Q_CONVERT(-0.38126576922), DSP32_Q_CONVERT(0.92446547433),
    DSP32_Q_CONVERT(-0.96697647104), DSP32_Q_CONVERT(-0.25486565960),
    DSP32_Q_CONVERT(-0.37700741022), DSP32_Q_CONVERT(0.92621024214),
    DSP32_Q_CONVERT(-0.96775383709), DSP32_Q_CONVERT(-0.25189781815),
    DSP32_Q_CONVERT(-0.37274106701), DSP32_Q_CONVERT(0.92793539482),
    DSP32_Q_CONVERT(-0.96852209427), DSP32_Q_CONVERT(-0.24892760575),
    DSP32_Q_CONVERT(-0.36846682995), DSP32_Q_CONVERT(0.92964089584),
    DSP32_Q_CONVERT(-0.96928123536), DSP32_Q_CONVERT(-0.24595505034),
    DSP32_Q_CONVERT(-0.36418478957), DSP32_Q_CONVERT(0.93132670908),
    DSP32_Q_CONVERT(-0.97003125319), DSP32_Q_CONVERT(-0.24298017990),
    DSP32_Q_CONVERT(-0.35989503653), DSP32_Q_CONVERT(0.93299279883),
    DSP32_Q_CONVERT(-0.97077214073), DSP32_Q_CONVERT(-0.24000302245),
    DSP32_Q_CONVERT(-0.35559766170), DSP32_Q_CONVERT(0.93463912982),
    DSP32_Q_CONVERT(-0.97150389099), DSP32_Q_CONVERT(-0.23702360599),
    DSP32_Q_CONVERT(-0.35129275609), DSP32_Q_CONVERT(0.93626566717),
    DSP32_Q_CONVERT(-0.97222649708), DSP32_Q_CONVERT(-0.23404195858),
    DSP32_Q_CONVERT(-0.34698041085), DSP32_Q_CONVERT(0.93787237644),
    DSP32_Q_CONVERT(-0.97293995221), DSP32_Q_CONVERT(-0.23105810828),
    DSP32_Q_CONVERT(-0.34266071731), DSP32_Q_CONVERT(0.93945922360),
    DSP32_Q_CONVERT(-0.97364424965), DSP32_Q_CONVERT(-0.22807208317),
    DSP32_Q_CONVERT(-0.33833376697), DSP32_Q_CONVERT(0.94102617505),
    DSP32_Q_CONVERT(-0.97433938279), DSP32_Q_CONVERT(-0.22508391136),
    DSP32_Q_CONVERT(-0.33399965144), DSP32_Q_CONVERT(0.94257319760),
    DSP32_Q_CONVERT(-0.97502534507), DSP32_Q_CONVERT(-0.22209362097),
    DSP32_Q_CONVERT(-0.32965846253), DSP32_Q_CONVERT(0.94410025849),
    DSP32_Q_CONVERT(-0.97570213004), DSP32_Q_CONVERT(-0.21910124016),
    DSP32_Q_CONVERT(-0.32531029216), DSP32_Q_CONVERT(0.94560732538),
    DSP32_Q_CONVERT(-0.97636973133), DSP32_Q_CONVERT(-0.21610679708),
    DSP32_Q_CONVERT(-0.32095523243), DSP32_Q_CONVERT(0.94709436635),
    DSP32_Q_CONVERT(-0.97702814266), DSP32_Q_CONVERT(-0.21311031992),
    DSP32_Q_CONVERT(-0.31659337556), DSP32_Q_CONVERT(0.94856134992),
    DSP32_Q_CONVERT(-0.97767735782), DSP32_Q_CONVERT(-0.21011183688),
    DSP32_Q_CONVERT(-0.31222481392), DSP32_Q_CONVERT(0.95000824500),
    DSP32_Q_CONVERT(-0.97831737072), DSP32_Q_CONVERT(-0.20711137619),
    DSP32_Q_CONVERT(-0.30784964004), DSP32_Q_CONVERT(0.95143502097),
    DSP32_Q_CONVERT(-0.97894817532), DSP32_Q_CONVERT(-0.20410896609),
    DSP32_Q_CONVERT(-0.30346794657), DSP32_Q_CONVERT(0.95284164760),
    DSP32_Q_CONVERT(-0.97956976569), DSP32_Q_CONVERT(-0.20110463484),
    DSP32_Q_CONVERT(-0.29907982631), DSP32_Q_CONVERT(0.95422809511),
    DSP32_Q_CONVERT(-0.98018213597), DSP32_Q_CONVERT(-0.19809841072),
    DSP32_Q_CONVERT(-0.29468537218), DSP32_Q_CONVERT(0.95559433413),
    DSP32_Q_CONVERT(-0.98078528040), DSP32_Q_CONVERT(-0.19509032202),
    DSP32_Q_CONVERT(-0.29028467725), DSP32_Q_CONVERT(0.95694033573),
    DSP32_Q_CONVERT(-0.98137919331), DSP32_Q_CONVERT(-0.19208039705),
    DSP32_Q_CONVERT(-0.28587783473), DSP32_Q_CONVERT(0.95826607141),
    DSP32_Q_CONVERT(-0.98196386911), DSP32_Q_CONVERT(-0.18906866415),
    DSP32_Q_CONVERT(-0.28146493793), DSP32_Q_CONVERT(0.95957151308),
    DSP32_Q_CONVERT(-0.98253930229), DSP32_Q_CONVERT(-0.18605515166),
    DSP32_Q_CONVERT(-0.27704608031), DSP32_Q_CONVERT(0.96085663311),
    DSP32_Q_CONVERT(-0.98310548743), DSP32_Q_CONVERT(-0.18303988796),
    DSP32_Q_CONVERT(-0.27262135545), DSP32_Q_CONVERT(0.96212140427),
    DSP32_Q_CONVERT(-0.98366241921), DSP32_Q_CONVERT(-0.18002290141),
    DSP32_Q_CONVERT(-0.26819085706), DSP32_Q_CONVERT(0.96336579978),
    DSP32_Q_CONVERT(-0.98421009239), DSP32_Q_CONVERT(-0.17700422041),
    DSP32_Q_CONVERT(-0.26375467897), DSP32_Q_CONVERT(0.96458979329),
    DSP32_Q_CONVERT(-0.98474850180), DSP32_Q_CONVERT(-0.17398387339),
    DSP32_Q_CONVERT(-0.25931291513), DSP32_Q_CONVERT(0.96579335887),
    DSP32_Q_CONVERT(-0.98527764239), DSP32_Q_CONVERT(-0.17096188876),
    DSP32_Q_CONVERT(-0.25486565960), DSP32_Q_CONVERT(0.96697647104),
    DSP32_Q_CONVERT(-0.98579750917), DSP32_Q_CONVERT(-0.16793829497),
    DSP32_Q_CONVERT(-0.25041300657), DSP32_Q_CONVERT(0.96813910475),
    DSP32_Q_CONVERT(-0.98630809724), DSP32_Q_CONVERT(-0.16491312049),
    DSP32_Q_CONVERT(-0.24595505034), DSP32_Q_CONVERT(0.96928123536),
    DSP32_Q_CONVERT(-0.98680940181), DSP32_Q_CONVERT(-0.16188639378),
    DSP32_Q_CONVERT(-0.24149188530), DSP32_Q_CONVERT(0.97040283869),
    DSP32_Q_CONVERT(-0.98730141816), DSP32_Q_CONVERT(-0.15885814333),
    DSP32_Q_CONVERT(-0.23702360599), DSP32_Q_CONVERT(0.97150389099),
    DSP32_Q_CONVERT(-0.98778414164), DSP32_Q_CONVERT(-0.15582839765),
    DSP32_Q_CONVERT(-0.23255030704), DSP32_Q_CONVERT(0.97258436893),
    DSP32_Q_CONVERT(-0.98825756773), DSP32_Q_CONVERT(-0.15279718526),
    DSP32_Q_CONVERT(-0.22807208317), DSP32_Q_CONVERT(0.97364424965),
    DSP32_Q_CONVERT(-0.98872169196), DSP32_Q_CONVERT(-0.14976453468),
    DSP32_Q_CONVERT(-0.22358902923), DSP32_Q_CONVERT(0.97468351069),
    DSP32_Q_CONVERT(-0.98917650996), DSP32_Q_CONVERT(-0.14673047446),
    DSP32_Q_CONVERT(-0.21910124016), DSP32_Q_CONVERT(0.97570213004),
    DSP32_Q_CONVERT(-0.98962201746), DSP32_Q_CONVERT(-0.14369503315),
    DSP32_Q_CONVERT(-0.21460881099), DSP32_Q_CONVERT(0.97670008613),
    DSP32_Q_CONVERT(-0.99005821026), DSP32_Q_CONVERT(-0.14065823933),
    DSP32_Q_CONVERT(-0.21011183688), DSP32_Q_CONVERT(0.97767735782),
    DSP32_Q_CONVERT(-0.99048508426), DSP32_Q_CONVERT(-0.13762012159),
    DSP32_Q_CONVERT(-0.20561041305), DSP32_Q_CONVERT(0.97863392443),
    DSP32_Q_CONVERT(-0.99090263543), DSP32_Q_CONVERT(-0.13458070851),
    DSP32_Q_CONVERT(-0.20110463484), DSP32_Q_CONVERT(0.97956976569),
    DSP32_Q_CONVERT(-0.99131085985), DSP32_Q_CONVERT(-0.13154002870),
    DSP32_Q_CONVERT(-0.19659459767), DSP32_Q_CONVERT(0.98048486177),
    DSP32_Q_CONVERT(-0.99170975367), DSP32_Q_CONVERT(-0.12849811079),
    DSP32_Q_CONVERT(-0.19208039705), DSP32_Q_CONVERT(0.98137919331),
    DSP32_Q_CONVERT(-0.99209931314), DSP32_Q_CONVERT(-0.12545498341),
    DSP32_Q_CONVERT(-0.18756212858), DSP32_Q_CONVERT(0.98225274137),
    DSP32_Q_CONVERT(-0.99247953460), DSP32_Q_CONVERT(-0.12241067520),
    DSP32_Q_CONVERT(-0.18303988796), DSP32_Q_CONVERT(0.98310548743),
    DSP32_Q_CONVERT(-0.99285041446), DSP32_Q_CONVERT(-0.11936521481),
    DSP32_Q_CONVERT(-0.17851377094), DSP32_Q_CONVERT(0.98393741345),
    DSP32_Q_CONVERT(-0.99321194923), DSP32_Q_CONVERT(-0.11631863091),
    DSP32_Q_CONVERT(-0.17398387339), DSP32_Q_CONVERT(0.98474850180),
    DSP32_Q_CONVERT(-0.99356413552), DSP32_Q_CONVERT(-0.11327095218),
    DSP32_Q_CONVERT(-0.16945029123), DSP32_Q_CONVERT(0.98553873531),
    DSP32_Q_CONVERT(-0.99390697000), DSP32_Q_CONVERT(-0.11022220729),
    DSP32_Q_CONVERT(-0.16491312049), DSP32_Q_CONVERT(0.98630809724),
    DSP32_Q_CONVERT(-0.99424044945), DSP32_Q_CONVERT(-0.10717242496),
    DSP32_Q_CONVERT(-0.16037245724), DSP32_Q_CONVERT(0.98705657131),
    DSP32_Q_CONVERT(-0.99456457073), DSP32_Q_CONVERT(-0.10412163387),
    DSP32_Q_CONVERT(-0.15582839765), DSP32_Q_CONVERT(0.98778414164),
    DSP32_Q_CONVERT(-0.99487933079), DSP32_Q_CONVERT(-0.10106986275),
    DSP32_Q_CONVERT(-0.15128103796), DSP32_Q_CONVERT(0.98849079285),
    DSP32_Q_CONVERT(-0.99518472667), DSP32_Q_CONVERT(-0.09801714033),
    DSP32_Q_CONVERT(-0.14673047446), DSP32_Q_CONVERT(0.98917650996),
    DSP32_Q_CONVERT(-0.99548075549), DSP32_Q_CONVERT(-0.09496349533),
    DSP32_Q_CONVERT(-0.14217680352), DSP32_Q_CONVERT(0.98984127846),
    DSP32_Q_CONVERT(-0.99576741447), DSP32_Q_CONVERT(-0.09190895650),
    DSP32_Q_CONVERT(-0.13762012159), DSP32_Q_CONVERT(0.99048508426),
    DSP32_Q_CONVERT(-0.99604470090), DSP32_Q_CONVERT(-0.08885355258),
    DSP32_Q_CONVERT(-0.13306052516), DSP32_Q_CONVERT(0.99110791372),
    DSP32_Q_CONVERT(-0.99631261218), DSP32_Q_CONVERT(-0.08579731234),
    DSP32_Q_CONVERT(-0.12849811079), DSP32_Q_CONVERT(0.99170975367),
    DSP32_Q_CONVERT(-0.99657114579), DSP32_Q_CONVERT(-0.08274026455),
    DSP32_Q_CONVERT(-0.12393297512), DSP32_Q_CONVERT(0.99229059135),
    DSP32_Q_CONVERT(-0.99682029929), DSP32_Q_CONVERT(-0.07968243797),
    DSP32_Q_CONVERT(-0.11936521481), DSP32_Q_CONVERT(0.99285041446),
    DSP32_Q_CONVERT(-0.99706007034), DSP32_Q_CONVERT(-0.07662386139),
    DSP32_Q_CONVERT(-0.11479492661), DSP32_Q_CONVERT(0.99338921115),
    DSP32_Q_CONVERT(-0.99729045668), DSP32_Q_CONVERT(-0.07356456360),
    DSP32_Q_CONVERT(-0.11022220729), DSP32_Q_CONVERT(0.99390697000),
    DSP32_Q_CONVERT(-0.99751145614), DSP32_Q_CONVERT(-0.07050457339),
    DSP32_Q_CONVERT(-0.10564715371), DSP32_Q_CONVERT(0.99440368006),
    DSP32_Q_CONVERT(-0.99772306664), DSP32_Q_CONVERT(-0.06744391956),
    DSP32_Q_CONVERT(-0.10106986275), DSP32_Q_CONVERT(0.99487933079),
    DSP32_Q_CONVERT(-0.99792528620), DSP32_Q_CONVERT(-0.06438263093),
    DSP32_Q_CONVERT(-0.09649043136), DSP32_Q_CONVERT(0.99533391214),
    DSP32_Q_CONVERT(-0.99811811290), DSP32_Q_CONVERT(-0.06132073630),
    DSP32_Q_CONVERT(-0.09190895650), DSP32_Q_CONVERT(0.99576741447),
    DSP32_Q_CONVERT(-0.99830154493), DSP32_Q_CONVERT(-0.05825826450),
    DSP32_Q_CONVERT(-0.08732553521), DSP32_Q_CONVERT(0.99617982860),
    DSP32_Q_CONVERT(-0.99847558057), DSP32_Q_CONVERT(-0.05519524435),
    DSP32_Q_CONVERT(-0.08274026455), DSP32_Q_CONVERT(0.99657114579),
    DSP32_Q_CONVERT(-0.99864021818), DSP32_Q_CONVERT(-0.05213170468),
    DSP32_Q_CONVERT(-0.07815324163), DSP32_Q_CONVERT(0.99694135776),
    DSP32_Q_CONVERT(-0.99879545621), DSP32_Q_CONVERT(-0.04906767433),
    DSP32_Q_CONVERT(-0.07356456360), DSP32_Q_CONVERT(0.99729045668),
    DSP32_Q_CONVERT(-0.99894129319), DSP32_Q_CONVERT(-0.04600318213),
    DSP32_Q_CONVERT(-0.06897432763), DSP32_Q_CONVERT(0.99761843514),
    DSP32_Q_CONVERT(-0.99907772775), DSP32_Q_CONVERT(-0.04293825693),
    DSP32_Q_CONVERT(-0.06438263093), DSP32_Q_CONVERT(0.99792528620),
    DSP32_Q_CONVERT(-0.99920475862), DSP32_Q_CONVERT(-0.03987292759),
    DSP32_Q_CONVERT(-0.05978957075), DSP32_Q_CONVERT(0.99821100336),
    DSP32_Q_CONVERT(-0.99932238459), DSP32_Q_CONVERT(-0.03680722294),
    DSP32_Q_CONVERT(-0.05519524435), DSP32_Q_CONVERT(0.99847558057),
    DSP32_Q_CONVERT(-0.99943060456), DSP32_Q_CONVERT(-0.03374117185),
    DSP32_Q_CONVERT(-0.05059974904), DSP32_Q_CONVERT(0.99871901223),
    DSP32_Q_CONVERT(-0.99952941750), DSP32_Q_CONVERT(-0.03067480318),
    DSP32_Q_CONVERT(-0.04600318213), DSP32_Q_CONVERT(0.99894129319),
    DSP32_Q_CONVERT(-0.99961882250), DSP32_Q_CONVERT(-0.02760814578),
    DSP32_Q_CONVERT(-0.04140564098), DSP32_Q_CONVERT(0.99914241872),
    DSP32_Q_CONVERT(-0.99969881870), DSP32_Q_CONVERT(-0.02454122852),
    DSP32_Q_CONVERT(-0.03680722294), DSP32_Q_CONVERT(0.99932238459),
    DSP32_Q_CONVERT(-0.99976940535), DSP32_Q_CONVERT(-0.02147408028),
    DSP32_Q_CONVERT(-0.03220802541), DSP32_Q_CONVERT(0.99948118697),
    DSP32_Q_CONVERT(-0.99983058180), DSP32_Q_CONVERT(-0.01840672991),
    DSP32_Q_CONVERT(-0.02760814578), DSP32_Q_CONVERT(0.99961882250),
    DSP32_Q_CONVERT(-0.99988234745), DSP32_Q_CONVERT(-0.01533920628),
    DSP32_Q_CONVERT(-0.02300768147), DSP32_Q_CONVERT(0.99973528826),
    DSP32_Q_CONVERT(-0.99992470184), DSP32_Q_CONVERT(-0.01227153829),
    DSP32_Q_CONVERT(-0.01840672991), DSP32_Q_CONVERT(0.99983058180),
    DSP32_Q_CONVERT(-0.99995764455), DSP32_Q_CONVERT(-0.00920375478),
    DSP32_Q_CONVERT(-0.01380538853), DSP32_Q_CONVERT(0.99990470108),
    DSP32_Q_CONVERT(-0.99998117528), DSP32_Q_CONVERT(-0.00613588465),
    DSP32_Q_CONVERT(-0.00920375478), DSP32_Q_CONVERT(0.99995764455),
    DSP32_Q_CONVERT(-0.99999529381), DSP32_Q_CONVERT(-0.00306795676),
    DSP32_Q_CONVERT(-0.00460192612), DSP32_Q_CONVERT(0.99998941108)
    };
#  endif

#endif
