/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief 16-bit twiddle factors table
 *
 * This file define the twiddle factors table
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
#include "trans_dsp16_twiddle_factors.h"

//! This macro is used to convert the raw data acording to the algorithm's optimization we need
#if DSP_OPTIMIZATION & DSP_OPTI_ACCURACY
# define DSP16_Q_CONVERT(x)  DSP16_Q(x)
#else
# define DSP16_Q_CONVERT(x)  DSP16_Q((x)/2.)
#endif

/*
 * Twiddle factors for a 64-point FFT
 * Minimized twiddle factors table: 2*(64/4)*2 + 2 = 66 bytes
 * Full twiddle factors table: 2*(64/4)*6 + 2 = 194 bytes
 */
#if DSP16_N_TWIDDLE_FACTORS == 64

/*! \name 64 elements twiddle factors table
 */
//! \{
/*! \brief This table has been generated using the following algorithm:
 * \code
 * for i=0:4:DSP16_N_TWIDDLE_FACTORS,
 * k = i/(DSP16_N_TWIDDLE_FACTORS*4);
 * w = exp(-2*PI*%i*k);
 * end;
 * \endcode
 * It is a one dimensional array containing the real parts (even parts) and imaginary parts (odd parts) of the \b w value.
 */
  A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp16_t dsp16_twiddle_factors[DSP16_N_TWIDDLE_FACTORS/2+2] = {
  /*Re(w)   Im(w)  */
    DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000),
    DSP16_Q_CONVERT(0.995185), DSP16_Q_CONVERT(-0.098017),
    DSP16_Q_CONVERT(0.980785), DSP16_Q_CONVERT(-0.195090),
    DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285),
    DSP16_Q_CONVERT(0.923880), DSP16_Q_CONVERT(-0.382683),
    DSP16_Q_CONVERT(0.881921), DSP16_Q_CONVERT(-0.471397),
    DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570),
    DSP16_Q_CONVERT(0.773010), DSP16_Q_CONVERT(-0.634393),
    DSP16_Q_CONVERT(0.707107), DSP16_Q_CONVERT(-0.707107),
    DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010),
    DSP16_Q_CONVERT(0.555570), DSP16_Q_CONVERT(-0.831470),
    DSP16_Q_CONVERT(0.471397), DSP16_Q_CONVERT(-0.881921),
    DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880),
    DSP16_Q_CONVERT(0.290285), DSP16_Q_CONVERT(-0.956940),
    DSP16_Q_CONVERT(0.195090), DSP16_Q_CONVERT(-0.980785),
    DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185),
    DSP16_Q_CONVERT(-0.000000), DSP16_Q_CONVERT(-1.000000)
  };

  // If we need to optimize the code to use small twiddle factors table
#  if !(DSP_OPTIMIZATION & DSP_OPTI_SIZE)
/*! \brief This table has been generated using the following algorithm:
 * \code
 * for i=0:4:DSP16_N_TWIDDLE_FACTORS-4,
 * k = i/(DSP16_N_TWIDDLE_FACTORS*4);
 * w2 = exp(-2*PI*%i*k*2);
 * w3 = exp(-2*PI*%i*k*3);
 * end;
 * \endcode
 * It is a one dimensional array containing the real parts (4*i parts) and imaginary parts (4*i+1 parts) of the \b w2 value
 * and the real parts (4*i+2 parts) and imaginary parts (4*i+3 parts) of the \b w3 value.
 */
    A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp16_t dsp16_twiddle_factors2[DSP16_N_TWIDDLE_FACTORS] = {
    /*Re(w2)   Im(w2)   Re(w3)   Im(w3)  */
      DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000),
      DSP16_Q_CONVERT(0.980785), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285),
      DSP16_Q_CONVERT(0.923880), DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570),
      DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010),
      DSP16_Q_CONVERT(0.707107), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880),
      DSP16_Q_CONVERT(0.555570), DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185),
      DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.980785),
      DSP16_Q_CONVERT(0.195090), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.881921),
      DSP16_Q_CONVERT(-0.000000), DSP16_Q_CONVERT(-1.000000), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.707107),
      DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.471397),
      DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.195090),
      DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(0.098017),
      DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(0.382683),
      DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(0.634393),
      DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(0.831470),
      DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(0.956940)
    };
#  endif

//! \}

/*
 * Twiddle factors for a 256-point FFT
 * Minimized twiddle factors table: 2*(256/4)*2 + 2= 258 bytes
 * Full twiddle factors table: 2*(256/4)*6 + 2= 770 bytes
 */
#elif DSP16_N_TWIDDLE_FACTORS == 256

/*! \name 256 elements twiddle factors table
 */
//! \{
/*! \brief This table has been generated using the following algorithm:
 * \code
 * for i=0:4:DSP16_N_TWIDDLE_FACTORS,
 * k = i/(DSP16_N_TWIDDLE_FACTORS*4);
 * w = exp(-2*PI*%i*k);
 * end;
 * \endcode
 * It is a one dimensional array containing the real parts (even parts) and imaginary parts (odd parts) of the \b w value.
 */
  A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp16_t dsp16_twiddle_factors[DSP16_N_TWIDDLE_FACTORS/2+2] = {
  /*Re(w)   Im(w)  */
    DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000),
    DSP16_Q_CONVERT(0.999699), DSP16_Q_CONVERT(-0.024541),
    DSP16_Q_CONVERT(0.998795), DSP16_Q_CONVERT(-0.049068),
    DSP16_Q_CONVERT(0.997290), DSP16_Q_CONVERT(-0.073565),
    DSP16_Q_CONVERT(0.995185), DSP16_Q_CONVERT(-0.098017),
    DSP16_Q_CONVERT(0.992480), DSP16_Q_CONVERT(-0.122411),
    DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730),
    DSP16_Q_CONVERT(0.985278), DSP16_Q_CONVERT(-0.170962),
    DSP16_Q_CONVERT(0.980785), DSP16_Q_CONVERT(-0.195090),
    DSP16_Q_CONVERT(0.975702), DSP16_Q_CONVERT(-0.219101),
    DSP16_Q_CONVERT(0.970031), DSP16_Q_CONVERT(-0.242980),
    DSP16_Q_CONVERT(0.963776), DSP16_Q_CONVERT(-0.266713),
    DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285),
    DSP16_Q_CONVERT(0.949528), DSP16_Q_CONVERT(-0.313682),
    DSP16_Q_CONVERT(0.941544), DSP16_Q_CONVERT(-0.336890),
    DSP16_Q_CONVERT(0.932993), DSP16_Q_CONVERT(-0.359895),
    DSP16_Q_CONVERT(0.923880), DSP16_Q_CONVERT(-0.382683),
    DSP16_Q_CONVERT(0.914210), DSP16_Q_CONVERT(-0.405241),
    DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555),
    DSP16_Q_CONVERT(0.893224), DSP16_Q_CONVERT(-0.449611),
    DSP16_Q_CONVERT(0.881921), DSP16_Q_CONVERT(-0.471397),
    DSP16_Q_CONVERT(0.870087), DSP16_Q_CONVERT(-0.492898),
    DSP16_Q_CONVERT(0.857729), DSP16_Q_CONVERT(-0.514103),
    DSP16_Q_CONVERT(0.844854), DSP16_Q_CONVERT(-0.534998),
    DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570),
    DSP16_Q_CONVERT(0.817585), DSP16_Q_CONVERT(-0.575808),
    DSP16_Q_CONVERT(0.803208), DSP16_Q_CONVERT(-0.595699),
    DSP16_Q_CONVERT(0.788346), DSP16_Q_CONVERT(-0.615232),
    DSP16_Q_CONVERT(0.773010), DSP16_Q_CONVERT(-0.634393),
    DSP16_Q_CONVERT(0.757209), DSP16_Q_CONVERT(-0.653173),
    DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559),
    DSP16_Q_CONVERT(0.724247), DSP16_Q_CONVERT(-0.689541),
    DSP16_Q_CONVERT(0.707107), DSP16_Q_CONVERT(-0.707107),
    DSP16_Q_CONVERT(0.689541), DSP16_Q_CONVERT(-0.724247),
    DSP16_Q_CONVERT(0.671559), DSP16_Q_CONVERT(-0.740951),
    DSP16_Q_CONVERT(0.653173), DSP16_Q_CONVERT(-0.757209),
    DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010),
    DSP16_Q_CONVERT(0.615232), DSP16_Q_CONVERT(-0.788346),
    DSP16_Q_CONVERT(0.595699), DSP16_Q_CONVERT(-0.803208),
    DSP16_Q_CONVERT(0.575808), DSP16_Q_CONVERT(-0.817585),
    DSP16_Q_CONVERT(0.555570), DSP16_Q_CONVERT(-0.831470),
    DSP16_Q_CONVERT(0.534998), DSP16_Q_CONVERT(-0.844854),
    DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729),
    DSP16_Q_CONVERT(0.492898), DSP16_Q_CONVERT(-0.870087),
    DSP16_Q_CONVERT(0.471397), DSP16_Q_CONVERT(-0.881921),
    DSP16_Q_CONVERT(0.449611), DSP16_Q_CONVERT(-0.893224),
    DSP16_Q_CONVERT(0.427555), DSP16_Q_CONVERT(-0.903989),
    DSP16_Q_CONVERT(0.405241), DSP16_Q_CONVERT(-0.914210),
    DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880),
    DSP16_Q_CONVERT(0.359895), DSP16_Q_CONVERT(-0.932993),
    DSP16_Q_CONVERT(0.336890), DSP16_Q_CONVERT(-0.941544),
    DSP16_Q_CONVERT(0.313682), DSP16_Q_CONVERT(-0.949528),
    DSP16_Q_CONVERT(0.290285), DSP16_Q_CONVERT(-0.956940),
    DSP16_Q_CONVERT(0.266713), DSP16_Q_CONVERT(-0.963776),
    DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031),
    DSP16_Q_CONVERT(0.219101), DSP16_Q_CONVERT(-0.975702),
    DSP16_Q_CONVERT(0.195090), DSP16_Q_CONVERT(-0.980785),
    DSP16_Q_CONVERT(0.170962), DSP16_Q_CONVERT(-0.985278),
    DSP16_Q_CONVERT(0.146730), DSP16_Q_CONVERT(-0.989177),
    DSP16_Q_CONVERT(0.122411), DSP16_Q_CONVERT(-0.992480),
    DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185),
    DSP16_Q_CONVERT(0.073565), DSP16_Q_CONVERT(-0.997290),
    DSP16_Q_CONVERT(0.049068), DSP16_Q_CONVERT(-0.998795),
    DSP16_Q_CONVERT(0.024541), DSP16_Q_CONVERT(-0.999699),
    DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(-1.000000)
  };

  // If we need to optimize the code to use small twiddle factors table
#  if !(DSP_OPTIMIZATION & DSP_OPTI_SIZE)

/*! \brief This table has been generated using the following algorithm:
 * \code
 * for i=0:4:DSP16_N_TWIDDLE_FACTORS-4,
 * k = i/(DSP16_N_TWIDDLE_FACTORS*4);
 * w2 = exp(-2*PI*%i*k*2);
 * w3 = exp(-2*PI*%i*k*3);
 * end;
 * \endcode
 * It is a one dimensional array containing the real parts (4*i parts) and imaginary parts (4*i+1 parts) of the \b w2 value
 * and the real parts (4*i+2 parts) and imaginary parts (4*i+3 parts) of the \b w3 value.
 */
    A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp16_t dsp16_twiddle_factors2[DSP16_N_TWIDDLE_FACTORS] = {
    /*Re(w2)   Im(w2)   Re(w3)   Im(w3)  */
      DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000),
      DSP16_Q_CONVERT(0.998795), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(0.997290), DSP16_Q_CONVERT(-0.073565),
      DSP16_Q_CONVERT(0.995185), DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730),
      DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(0.975702), DSP16_Q_CONVERT(-0.219101),
      DSP16_Q_CONVERT(0.980785), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285),
      DSP16_Q_CONVERT(0.970031), DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(0.932993), DSP16_Q_CONVERT(-0.359895),
      DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555),
      DSP16_Q_CONVERT(0.941544), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(0.870087), DSP16_Q_CONVERT(-0.492898),
      DSP16_Q_CONVERT(0.923880), DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570),
      DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(0.788346), DSP16_Q_CONVERT(-0.615232),
      DSP16_Q_CONVERT(0.881921), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559),
      DSP16_Q_CONVERT(0.857729), DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(0.689541), DSP16_Q_CONVERT(-0.724247),
      DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010),
      DSP16_Q_CONVERT(0.803208), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(0.575808), DSP16_Q_CONVERT(-0.817585),
      DSP16_Q_CONVERT(0.773010), DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729),
      DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(0.449611), DSP16_Q_CONVERT(-0.893224),
      DSP16_Q_CONVERT(0.707107), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880),
      DSP16_Q_CONVERT(0.671559), DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(0.313682), DSP16_Q_CONVERT(-0.949528),
      DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031),
      DSP16_Q_CONVERT(0.595699), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(0.170962), DSP16_Q_CONVERT(-0.985278),
      DSP16_Q_CONVERT(0.555570), DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185),
      DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(0.024541), DSP16_Q_CONVERT(-0.999699),
      DSP16_Q_CONVERT(0.471397), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.998795),
      DSP16_Q_CONVERT(0.427555), DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(-0.992480),
      DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.980785),
      DSP16_Q_CONVERT(0.336890), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(-0.963776),
      DSP16_Q_CONVERT(0.290285), DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.941544),
      DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(-0.914210),
      DSP16_Q_CONVERT(0.195090), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.881921),
      DSP16_Q_CONVERT(0.146730), DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(-0.844854),
      DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.803208),
      DSP16_Q_CONVERT(0.049068), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(-0.757209),
      DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(-1.000000), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.707107),
      DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(-0.653173),
      DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.595699),
      DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(-0.534998),
      DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.471397),
      DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.405241),
      DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.336890),
      DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.266713),
      DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.195090),
      DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.122411),
      DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.049068),
      DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(0.024541),
      DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(0.098017),
      DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(0.170962),
      DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(0.242980),
      DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(0.313682),
      DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(0.382683),
      DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(0.449611),
      DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(0.514103),
      DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(0.575808),
      DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(0.634393),
      DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(0.689541),
      DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(0.740951),
      DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(0.788346),
      DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(0.831470),
      DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(0.870087),
      DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(0.903989),
      DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(0.932993),
      DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(0.956940),
      DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(0.975702),
      DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(0.989177),
      DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(0.997290)
    };
#  endif

//! \}

/*
 * Twiddle factors for a 1024-point FFT
 * Minimized twiddle factors table: 2*(1024/4)*2 + 2 = 1 Kbytes
 * Full twiddle factors table: 2*(1024/4)*6 + 2 = 3 Kbytes
 */
#elif DSP16_N_TWIDDLE_FACTORS == 1024

/*! \name 1024 elements twiddle factors table
 */
//! \{
/*! \brief This table has been generated using the following algorithm:
 * \code
 * for i=0:4:DSP16_N_TWIDDLE_FACTORS,
 * k = i/(DSP16_N_TWIDDLE_FACTORS*4);
 * w = exp(-2*PI*%i*k);
 * end;
 * \endcode
 * It is a one dimensional array containing the real parts (even parts) and imaginary parts (odd parts) of the \b w value.
 */
  A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp16_t dsp16_twiddle_factors[DSP16_N_TWIDDLE_FACTORS/2+2] = {
  /*Re(w)   Im(w)  */
    DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000),
    DSP16_Q_CONVERT(0.999981), DSP16_Q_CONVERT(-0.006136),
    DSP16_Q_CONVERT(0.999925), DSP16_Q_CONVERT(-0.012272),
    DSP16_Q_CONVERT(0.999831), DSP16_Q_CONVERT(-0.018407),
    DSP16_Q_CONVERT(0.999699), DSP16_Q_CONVERT(-0.024541),
    DSP16_Q_CONVERT(0.999529), DSP16_Q_CONVERT(-0.030675),
    DSP16_Q_CONVERT(0.999322), DSP16_Q_CONVERT(-0.036807),
    DSP16_Q_CONVERT(0.999078), DSP16_Q_CONVERT(-0.042938),
    DSP16_Q_CONVERT(0.998795), DSP16_Q_CONVERT(-0.049068),
    DSP16_Q_CONVERT(0.998476), DSP16_Q_CONVERT(-0.055195),
    DSP16_Q_CONVERT(0.998118), DSP16_Q_CONVERT(-0.061321),
    DSP16_Q_CONVERT(0.997723), DSP16_Q_CONVERT(-0.067444),
    DSP16_Q_CONVERT(0.997290), DSP16_Q_CONVERT(-0.073565),
    DSP16_Q_CONVERT(0.996820), DSP16_Q_CONVERT(-0.079682),
    DSP16_Q_CONVERT(0.996313), DSP16_Q_CONVERT(-0.085797),
    DSP16_Q_CONVERT(0.995767), DSP16_Q_CONVERT(-0.091909),
    DSP16_Q_CONVERT(0.995185), DSP16_Q_CONVERT(-0.098017),
    DSP16_Q_CONVERT(0.994565), DSP16_Q_CONVERT(-0.104122),
    DSP16_Q_CONVERT(0.993907), DSP16_Q_CONVERT(-0.110222),
    DSP16_Q_CONVERT(0.993212), DSP16_Q_CONVERT(-0.116319),
    DSP16_Q_CONVERT(0.992480), DSP16_Q_CONVERT(-0.122411),
    DSP16_Q_CONVERT(0.991710), DSP16_Q_CONVERT(-0.128498),
    DSP16_Q_CONVERT(0.990903), DSP16_Q_CONVERT(-0.134581),
    DSP16_Q_CONVERT(0.990058), DSP16_Q_CONVERT(-0.140658),
    DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730),
    DSP16_Q_CONVERT(0.988258), DSP16_Q_CONVERT(-0.152797),
    DSP16_Q_CONVERT(0.987301), DSP16_Q_CONVERT(-0.158858),
    DSP16_Q_CONVERT(0.986308), DSP16_Q_CONVERT(-0.164913),
    DSP16_Q_CONVERT(0.985278), DSP16_Q_CONVERT(-0.170962),
    DSP16_Q_CONVERT(0.984210), DSP16_Q_CONVERT(-0.177004),
    DSP16_Q_CONVERT(0.983105), DSP16_Q_CONVERT(-0.183040),
    DSP16_Q_CONVERT(0.981964), DSP16_Q_CONVERT(-0.189069),
    DSP16_Q_CONVERT(0.980785), DSP16_Q_CONVERT(-0.195090),
    DSP16_Q_CONVERT(0.979570), DSP16_Q_CONVERT(-0.201105),
    DSP16_Q_CONVERT(0.978317), DSP16_Q_CONVERT(-0.207111),
    DSP16_Q_CONVERT(0.977028), DSP16_Q_CONVERT(-0.213110),
    DSP16_Q_CONVERT(0.975702), DSP16_Q_CONVERT(-0.219101),
    DSP16_Q_CONVERT(0.974339), DSP16_Q_CONVERT(-0.225084),
    DSP16_Q_CONVERT(0.972940), DSP16_Q_CONVERT(-0.231058),
    DSP16_Q_CONVERT(0.971504), DSP16_Q_CONVERT(-0.237024),
    DSP16_Q_CONVERT(0.970031), DSP16_Q_CONVERT(-0.242980),
    DSP16_Q_CONVERT(0.968522), DSP16_Q_CONVERT(-0.248928),
    DSP16_Q_CONVERT(0.966976), DSP16_Q_CONVERT(-0.254866),
    DSP16_Q_CONVERT(0.965394), DSP16_Q_CONVERT(-0.260794),
    DSP16_Q_CONVERT(0.963776), DSP16_Q_CONVERT(-0.266713),
    DSP16_Q_CONVERT(0.962121), DSP16_Q_CONVERT(-0.272621),
    DSP16_Q_CONVERT(0.960431), DSP16_Q_CONVERT(-0.278520),
    DSP16_Q_CONVERT(0.958703), DSP16_Q_CONVERT(-0.284408),
    DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285),
    DSP16_Q_CONVERT(0.955141), DSP16_Q_CONVERT(-0.296151),
    DSP16_Q_CONVERT(0.953306), DSP16_Q_CONVERT(-0.302006),
    DSP16_Q_CONVERT(0.951435), DSP16_Q_CONVERT(-0.307850),
    DSP16_Q_CONVERT(0.949528), DSP16_Q_CONVERT(-0.313682),
    DSP16_Q_CONVERT(0.947586), DSP16_Q_CONVERT(-0.319502),
    DSP16_Q_CONVERT(0.945607), DSP16_Q_CONVERT(-0.325310),
    DSP16_Q_CONVERT(0.943593), DSP16_Q_CONVERT(-0.331106),
    DSP16_Q_CONVERT(0.941544), DSP16_Q_CONVERT(-0.336890),
    DSP16_Q_CONVERT(0.939459), DSP16_Q_CONVERT(-0.342661),
    DSP16_Q_CONVERT(0.937339), DSP16_Q_CONVERT(-0.348419),
    DSP16_Q_CONVERT(0.935184), DSP16_Q_CONVERT(-0.354164),
    DSP16_Q_CONVERT(0.932993), DSP16_Q_CONVERT(-0.359895),
    DSP16_Q_CONVERT(0.930767), DSP16_Q_CONVERT(-0.365613),
    DSP16_Q_CONVERT(0.928506), DSP16_Q_CONVERT(-0.371317),
    DSP16_Q_CONVERT(0.926210), DSP16_Q_CONVERT(-0.377007),
    DSP16_Q_CONVERT(0.923880), DSP16_Q_CONVERT(-0.382683),
    DSP16_Q_CONVERT(0.921514), DSP16_Q_CONVERT(-0.388345),
    DSP16_Q_CONVERT(0.919114), DSP16_Q_CONVERT(-0.393992),
    DSP16_Q_CONVERT(0.916679), DSP16_Q_CONVERT(-0.399624),
    DSP16_Q_CONVERT(0.914210), DSP16_Q_CONVERT(-0.405241),
    DSP16_Q_CONVERT(0.911706), DSP16_Q_CONVERT(-0.410843),
    DSP16_Q_CONVERT(0.909168), DSP16_Q_CONVERT(-0.416430),
    DSP16_Q_CONVERT(0.906596), DSP16_Q_CONVERT(-0.422000),
    DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555),
    DSP16_Q_CONVERT(0.901349), DSP16_Q_CONVERT(-0.433094),
    DSP16_Q_CONVERT(0.898674), DSP16_Q_CONVERT(-0.438616),
    DSP16_Q_CONVERT(0.895966), DSP16_Q_CONVERT(-0.444122),
    DSP16_Q_CONVERT(0.893224), DSP16_Q_CONVERT(-0.449611),
    DSP16_Q_CONVERT(0.890449), DSP16_Q_CONVERT(-0.455084),
    DSP16_Q_CONVERT(0.887640), DSP16_Q_CONVERT(-0.460539),
    DSP16_Q_CONVERT(0.884797), DSP16_Q_CONVERT(-0.465976),
    DSP16_Q_CONVERT(0.881921), DSP16_Q_CONVERT(-0.471397),
    DSP16_Q_CONVERT(0.879012), DSP16_Q_CONVERT(-0.476799),
    DSP16_Q_CONVERT(0.876070), DSP16_Q_CONVERT(-0.482184),
    DSP16_Q_CONVERT(0.873095), DSP16_Q_CONVERT(-0.487550),
    DSP16_Q_CONVERT(0.870087), DSP16_Q_CONVERT(-0.492898),
    DSP16_Q_CONVERT(0.867046), DSP16_Q_CONVERT(-0.498228),
    DSP16_Q_CONVERT(0.863973), DSP16_Q_CONVERT(-0.503538),
    DSP16_Q_CONVERT(0.860867), DSP16_Q_CONVERT(-0.508830),
    DSP16_Q_CONVERT(0.857729), DSP16_Q_CONVERT(-0.514103),
    DSP16_Q_CONVERT(0.854558), DSP16_Q_CONVERT(-0.519356),
    DSP16_Q_CONVERT(0.851355), DSP16_Q_CONVERT(-0.524590),
    DSP16_Q_CONVERT(0.848120), DSP16_Q_CONVERT(-0.529804),
    DSP16_Q_CONVERT(0.844854), DSP16_Q_CONVERT(-0.534998),
    DSP16_Q_CONVERT(0.841555), DSP16_Q_CONVERT(-0.540171),
    DSP16_Q_CONVERT(0.838225), DSP16_Q_CONVERT(-0.545325),
    DSP16_Q_CONVERT(0.834863), DSP16_Q_CONVERT(-0.550458),
    DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570),
    DSP16_Q_CONVERT(0.828045), DSP16_Q_CONVERT(-0.560662),
    DSP16_Q_CONVERT(0.824589), DSP16_Q_CONVERT(-0.565732),
    DSP16_Q_CONVERT(0.821103), DSP16_Q_CONVERT(-0.570781),
    DSP16_Q_CONVERT(0.817585), DSP16_Q_CONVERT(-0.575808),
    DSP16_Q_CONVERT(0.814036), DSP16_Q_CONVERT(-0.580814),
    DSP16_Q_CONVERT(0.810457), DSP16_Q_CONVERT(-0.585798),
    DSP16_Q_CONVERT(0.806848), DSP16_Q_CONVERT(-0.590760),
    DSP16_Q_CONVERT(0.803208), DSP16_Q_CONVERT(-0.595699),
    DSP16_Q_CONVERT(0.799537), DSP16_Q_CONVERT(-0.600616),
    DSP16_Q_CONVERT(0.795837), DSP16_Q_CONVERT(-0.605511),
    DSP16_Q_CONVERT(0.792107), DSP16_Q_CONVERT(-0.610383),
    DSP16_Q_CONVERT(0.788346), DSP16_Q_CONVERT(-0.615232),
    DSP16_Q_CONVERT(0.784557), DSP16_Q_CONVERT(-0.620057),
    DSP16_Q_CONVERT(0.780737), DSP16_Q_CONVERT(-0.624859),
    DSP16_Q_CONVERT(0.776888), DSP16_Q_CONVERT(-0.629638),
    DSP16_Q_CONVERT(0.773010), DSP16_Q_CONVERT(-0.634393),
    DSP16_Q_CONVERT(0.769103), DSP16_Q_CONVERT(-0.639124),
    DSP16_Q_CONVERT(0.765167), DSP16_Q_CONVERT(-0.643832),
    DSP16_Q_CONVERT(0.761202), DSP16_Q_CONVERT(-0.648514),
    DSP16_Q_CONVERT(0.757209), DSP16_Q_CONVERT(-0.653173),
    DSP16_Q_CONVERT(0.753187), DSP16_Q_CONVERT(-0.657807),
    DSP16_Q_CONVERT(0.749136), DSP16_Q_CONVERT(-0.662416),
    DSP16_Q_CONVERT(0.745058), DSP16_Q_CONVERT(-0.667000),
    DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559),
    DSP16_Q_CONVERT(0.736817), DSP16_Q_CONVERT(-0.676093),
    DSP16_Q_CONVERT(0.732654), DSP16_Q_CONVERT(-0.680601),
    DSP16_Q_CONVERT(0.728464), DSP16_Q_CONVERT(-0.685084),
    DSP16_Q_CONVERT(0.724247), DSP16_Q_CONVERT(-0.689541),
    DSP16_Q_CONVERT(0.720003), DSP16_Q_CONVERT(-0.693971),
    DSP16_Q_CONVERT(0.715731), DSP16_Q_CONVERT(-0.698376),
    DSP16_Q_CONVERT(0.711432), DSP16_Q_CONVERT(-0.702755),
    DSP16_Q_CONVERT(0.707107), DSP16_Q_CONVERT(-0.707107),
    DSP16_Q_CONVERT(0.702755), DSP16_Q_CONVERT(-0.711432),
    DSP16_Q_CONVERT(0.698376), DSP16_Q_CONVERT(-0.715731),
    DSP16_Q_CONVERT(0.693971), DSP16_Q_CONVERT(-0.720003),
    DSP16_Q_CONVERT(0.689541), DSP16_Q_CONVERT(-0.724247),
    DSP16_Q_CONVERT(0.685084), DSP16_Q_CONVERT(-0.728464),
    DSP16_Q_CONVERT(0.680601), DSP16_Q_CONVERT(-0.732654),
    DSP16_Q_CONVERT(0.676093), DSP16_Q_CONVERT(-0.736817),
    DSP16_Q_CONVERT(0.671559), DSP16_Q_CONVERT(-0.740951),
    DSP16_Q_CONVERT(0.667000), DSP16_Q_CONVERT(-0.745058),
    DSP16_Q_CONVERT(0.662416), DSP16_Q_CONVERT(-0.749136),
    DSP16_Q_CONVERT(0.657807), DSP16_Q_CONVERT(-0.753187),
    DSP16_Q_CONVERT(0.653173), DSP16_Q_CONVERT(-0.757209),
    DSP16_Q_CONVERT(0.648514), DSP16_Q_CONVERT(-0.761202),
    DSP16_Q_CONVERT(0.643832), DSP16_Q_CONVERT(-0.765167),
    DSP16_Q_CONVERT(0.639124), DSP16_Q_CONVERT(-0.769103),
    DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010),
    DSP16_Q_CONVERT(0.629638), DSP16_Q_CONVERT(-0.776888),
    DSP16_Q_CONVERT(0.624859), DSP16_Q_CONVERT(-0.780737),
    DSP16_Q_CONVERT(0.620057), DSP16_Q_CONVERT(-0.784557),
    DSP16_Q_CONVERT(0.615232), DSP16_Q_CONVERT(-0.788346),
    DSP16_Q_CONVERT(0.610383), DSP16_Q_CONVERT(-0.792107),
    DSP16_Q_CONVERT(0.605511), DSP16_Q_CONVERT(-0.795837),
    DSP16_Q_CONVERT(0.600616), DSP16_Q_CONVERT(-0.799537),
    DSP16_Q_CONVERT(0.595699), DSP16_Q_CONVERT(-0.803208),
    DSP16_Q_CONVERT(0.590760), DSP16_Q_CONVERT(-0.806848),
    DSP16_Q_CONVERT(0.585798), DSP16_Q_CONVERT(-0.810457),
    DSP16_Q_CONVERT(0.580814), DSP16_Q_CONVERT(-0.814036),
    DSP16_Q_CONVERT(0.575808), DSP16_Q_CONVERT(-0.817585),
    DSP16_Q_CONVERT(0.570781), DSP16_Q_CONVERT(-0.821103),
    DSP16_Q_CONVERT(0.565732), DSP16_Q_CONVERT(-0.824589),
    DSP16_Q_CONVERT(0.560662), DSP16_Q_CONVERT(-0.828045),
    DSP16_Q_CONVERT(0.555570), DSP16_Q_CONVERT(-0.831470),
    DSP16_Q_CONVERT(0.550458), DSP16_Q_CONVERT(-0.834863),
    DSP16_Q_CONVERT(0.545325), DSP16_Q_CONVERT(-0.838225),
    DSP16_Q_CONVERT(0.540171), DSP16_Q_CONVERT(-0.841555),
    DSP16_Q_CONVERT(0.534998), DSP16_Q_CONVERT(-0.844854),
    DSP16_Q_CONVERT(0.529804), DSP16_Q_CONVERT(-0.848120),
    DSP16_Q_CONVERT(0.524590), DSP16_Q_CONVERT(-0.851355),
    DSP16_Q_CONVERT(0.519356), DSP16_Q_CONVERT(-0.854558),
    DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729),
    DSP16_Q_CONVERT(0.508830), DSP16_Q_CONVERT(-0.860867),
    DSP16_Q_CONVERT(0.503538), DSP16_Q_CONVERT(-0.863973),
    DSP16_Q_CONVERT(0.498228), DSP16_Q_CONVERT(-0.867046),
    DSP16_Q_CONVERT(0.492898), DSP16_Q_CONVERT(-0.870087),
    DSP16_Q_CONVERT(0.487550), DSP16_Q_CONVERT(-0.873095),
    DSP16_Q_CONVERT(0.482184), DSP16_Q_CONVERT(-0.876070),
    DSP16_Q_CONVERT(0.476799), DSP16_Q_CONVERT(-0.879012),
    DSP16_Q_CONVERT(0.471397), DSP16_Q_CONVERT(-0.881921),
    DSP16_Q_CONVERT(0.465976), DSP16_Q_CONVERT(-0.884797),
    DSP16_Q_CONVERT(0.460539), DSP16_Q_CONVERT(-0.887640),
    DSP16_Q_CONVERT(0.455084), DSP16_Q_CONVERT(-0.890449),
    DSP16_Q_CONVERT(0.449611), DSP16_Q_CONVERT(-0.893224),
    DSP16_Q_CONVERT(0.444122), DSP16_Q_CONVERT(-0.895966),
    DSP16_Q_CONVERT(0.438616), DSP16_Q_CONVERT(-0.898674),
    DSP16_Q_CONVERT(0.433094), DSP16_Q_CONVERT(-0.901349),
    DSP16_Q_CONVERT(0.427555), DSP16_Q_CONVERT(-0.903989),
    DSP16_Q_CONVERT(0.422000), DSP16_Q_CONVERT(-0.906596),
    DSP16_Q_CONVERT(0.416430), DSP16_Q_CONVERT(-0.909168),
    DSP16_Q_CONVERT(0.410843), DSP16_Q_CONVERT(-0.911706),
    DSP16_Q_CONVERT(0.405241), DSP16_Q_CONVERT(-0.914210),
    DSP16_Q_CONVERT(0.399624), DSP16_Q_CONVERT(-0.916679),
    DSP16_Q_CONVERT(0.393992), DSP16_Q_CONVERT(-0.919114),
    DSP16_Q_CONVERT(0.388345), DSP16_Q_CONVERT(-0.921514),
    DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880),
    DSP16_Q_CONVERT(0.377007), DSP16_Q_CONVERT(-0.926210),
    DSP16_Q_CONVERT(0.371317), DSP16_Q_CONVERT(-0.928506),
    DSP16_Q_CONVERT(0.365613), DSP16_Q_CONVERT(-0.930767),
    DSP16_Q_CONVERT(0.359895), DSP16_Q_CONVERT(-0.932993),
    DSP16_Q_CONVERT(0.354164), DSP16_Q_CONVERT(-0.935184),
    DSP16_Q_CONVERT(0.348419), DSP16_Q_CONVERT(-0.937339),
    DSP16_Q_CONVERT(0.342661), DSP16_Q_CONVERT(-0.939459),
    DSP16_Q_CONVERT(0.336890), DSP16_Q_CONVERT(-0.941544),
    DSP16_Q_CONVERT(0.331106), DSP16_Q_CONVERT(-0.943593),
    DSP16_Q_CONVERT(0.325310), DSP16_Q_CONVERT(-0.945607),
    DSP16_Q_CONVERT(0.319502), DSP16_Q_CONVERT(-0.947586),
    DSP16_Q_CONVERT(0.313682), DSP16_Q_CONVERT(-0.949528),
    DSP16_Q_CONVERT(0.307850), DSP16_Q_CONVERT(-0.951435),
    DSP16_Q_CONVERT(0.302006), DSP16_Q_CONVERT(-0.953306),
    DSP16_Q_CONVERT(0.296151), DSP16_Q_CONVERT(-0.955141),
    DSP16_Q_CONVERT(0.290285), DSP16_Q_CONVERT(-0.956940),
    DSP16_Q_CONVERT(0.284408), DSP16_Q_CONVERT(-0.958703),
    DSP16_Q_CONVERT(0.278520), DSP16_Q_CONVERT(-0.960431),
    DSP16_Q_CONVERT(0.272621), DSP16_Q_CONVERT(-0.962121),
    DSP16_Q_CONVERT(0.266713), DSP16_Q_CONVERT(-0.963776),
    DSP16_Q_CONVERT(0.260794), DSP16_Q_CONVERT(-0.965394),
    DSP16_Q_CONVERT(0.254866), DSP16_Q_CONVERT(-0.966976),
    DSP16_Q_CONVERT(0.248928), DSP16_Q_CONVERT(-0.968522),
    DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031),
    DSP16_Q_CONVERT(0.237024), DSP16_Q_CONVERT(-0.971504),
    DSP16_Q_CONVERT(0.231058), DSP16_Q_CONVERT(-0.972940),
    DSP16_Q_CONVERT(0.225084), DSP16_Q_CONVERT(-0.974339),
    DSP16_Q_CONVERT(0.219101), DSP16_Q_CONVERT(-0.975702),
    DSP16_Q_CONVERT(0.213110), DSP16_Q_CONVERT(-0.977028),
    DSP16_Q_CONVERT(0.207111), DSP16_Q_CONVERT(-0.978317),
    DSP16_Q_CONVERT(0.201105), DSP16_Q_CONVERT(-0.979570),
    DSP16_Q_CONVERT(0.195090), DSP16_Q_CONVERT(-0.980785),
    DSP16_Q_CONVERT(0.189069), DSP16_Q_CONVERT(-0.981964),
    DSP16_Q_CONVERT(0.183040), DSP16_Q_CONVERT(-0.983105),
    DSP16_Q_CONVERT(0.177004), DSP16_Q_CONVERT(-0.984210),
    DSP16_Q_CONVERT(0.170962), DSP16_Q_CONVERT(-0.985278),
    DSP16_Q_CONVERT(0.164913), DSP16_Q_CONVERT(-0.986308),
    DSP16_Q_CONVERT(0.158858), DSP16_Q_CONVERT(-0.987301),
    DSP16_Q_CONVERT(0.152797), DSP16_Q_CONVERT(-0.988258),
    DSP16_Q_CONVERT(0.146730), DSP16_Q_CONVERT(-0.989177),
    DSP16_Q_CONVERT(0.140658), DSP16_Q_CONVERT(-0.990058),
    DSP16_Q_CONVERT(0.134581), DSP16_Q_CONVERT(-0.990903),
    DSP16_Q_CONVERT(0.128498), DSP16_Q_CONVERT(-0.991710),
    DSP16_Q_CONVERT(0.122411), DSP16_Q_CONVERT(-0.992480),
    DSP16_Q_CONVERT(0.116319), DSP16_Q_CONVERT(-0.993212),
    DSP16_Q_CONVERT(0.110222), DSP16_Q_CONVERT(-0.993907),
    DSP16_Q_CONVERT(0.104122), DSP16_Q_CONVERT(-0.994565),
    DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185),
    DSP16_Q_CONVERT(0.091909), DSP16_Q_CONVERT(-0.995767),
    DSP16_Q_CONVERT(0.085797), DSP16_Q_CONVERT(-0.996313),
    DSP16_Q_CONVERT(0.079682), DSP16_Q_CONVERT(-0.996820),
    DSP16_Q_CONVERT(0.073565), DSP16_Q_CONVERT(-0.997290),
    DSP16_Q_CONVERT(0.067444), DSP16_Q_CONVERT(-0.997723),
    DSP16_Q_CONVERT(0.061321), DSP16_Q_CONVERT(-0.998118),
    DSP16_Q_CONVERT(0.055195), DSP16_Q_CONVERT(-0.998476),
    DSP16_Q_CONVERT(0.049068), DSP16_Q_CONVERT(-0.998795),
    DSP16_Q_CONVERT(0.042938), DSP16_Q_CONVERT(-0.999078),
    DSP16_Q_CONVERT(0.036807), DSP16_Q_CONVERT(-0.999322),
    DSP16_Q_CONVERT(0.030675), DSP16_Q_CONVERT(-0.999529),
    DSP16_Q_CONVERT(0.024541), DSP16_Q_CONVERT(-0.999699),
    DSP16_Q_CONVERT(0.018407), DSP16_Q_CONVERT(-0.999831),
    DSP16_Q_CONVERT(0.012272), DSP16_Q_CONVERT(-0.999925),
    DSP16_Q_CONVERT(0.006136), DSP16_Q_CONVERT(-0.999981),
    DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(-1.000000)
  };

  // If we need to optimize the code to use small twiddle factors table
#  if !(DSP_OPTIMIZATION & DSP_OPTI_SIZE)
/*! \brief This table has been generated using the following algorithm:
 * \code
 * for i=0:4:DSP16_N_TWIDDLE_FACTORS-4,
 * k = i/(DSP16_N_TWIDDLE_FACTORS*4);
 * w2 = exp(-2*PI*%i*k*2);
 * w3 = exp(-2*PI*%i*k*3);
 * end;
 * \endcode
 * It is a one dimensional array containing the real parts (4*i parts) and imaginary parts (4*i+1 parts) of the \b w2 value
 * and the real parts (4*i+2 parts) and imaginary parts (4*i+3 parts) of the \b w3 value.
 */
    A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp16_t dsp16_twiddle_factors2[DSP16_N_TWIDDLE_FACTORS] = {
    /*Re(w2)   Im(w2)   Re(w3)   Im(w3)  */
      DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000),
      DSP16_Q_CONVERT(0.999925), DSP16_Q_CONVERT(-0.012272), DSP16_Q_CONVERT(0.999831), DSP16_Q_CONVERT(-0.018407),
      DSP16_Q_CONVERT(0.999699), DSP16_Q_CONVERT(-0.024541), DSP16_Q_CONVERT(0.999322), DSP16_Q_CONVERT(-0.036807),
      DSP16_Q_CONVERT(0.999322), DSP16_Q_CONVERT(-0.036807), DSP16_Q_CONVERT(0.998476), DSP16_Q_CONVERT(-0.055195),
      DSP16_Q_CONVERT(0.998795), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(0.997290), DSP16_Q_CONVERT(-0.073565),
      DSP16_Q_CONVERT(0.998118), DSP16_Q_CONVERT(-0.061321), DSP16_Q_CONVERT(0.995767), DSP16_Q_CONVERT(-0.091909),
      DSP16_Q_CONVERT(0.997290), DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(0.993907), DSP16_Q_CONVERT(-0.110222),
      DSP16_Q_CONVERT(0.996313), DSP16_Q_CONVERT(-0.085797), DSP16_Q_CONVERT(0.991710), DSP16_Q_CONVERT(-0.128498),
      DSP16_Q_CONVERT(0.995185), DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730),
      DSP16_Q_CONVERT(0.993907), DSP16_Q_CONVERT(-0.110222), DSP16_Q_CONVERT(0.986308), DSP16_Q_CONVERT(-0.164913),
      DSP16_Q_CONVERT(0.992480), DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(0.983105), DSP16_Q_CONVERT(-0.183040),
      DSP16_Q_CONVERT(0.990903), DSP16_Q_CONVERT(-0.134581), DSP16_Q_CONVERT(0.979570), DSP16_Q_CONVERT(-0.201105),
      DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(0.975702), DSP16_Q_CONVERT(-0.219101),
      DSP16_Q_CONVERT(0.987301), DSP16_Q_CONVERT(-0.158858), DSP16_Q_CONVERT(0.971504), DSP16_Q_CONVERT(-0.237024),
      DSP16_Q_CONVERT(0.985278), DSP16_Q_CONVERT(-0.170962), DSP16_Q_CONVERT(0.966976), DSP16_Q_CONVERT(-0.254866),
      DSP16_Q_CONVERT(0.983105), DSP16_Q_CONVERT(-0.183040), DSP16_Q_CONVERT(0.962121), DSP16_Q_CONVERT(-0.272621),
      DSP16_Q_CONVERT(0.980785), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285),
      DSP16_Q_CONVERT(0.978317), DSP16_Q_CONVERT(-0.207111), DSP16_Q_CONVERT(0.951435), DSP16_Q_CONVERT(-0.307850),
      DSP16_Q_CONVERT(0.975702), DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(0.945607), DSP16_Q_CONVERT(-0.325310),
      DSP16_Q_CONVERT(0.972940), DSP16_Q_CONVERT(-0.231058), DSP16_Q_CONVERT(0.939459), DSP16_Q_CONVERT(-0.342661),
      DSP16_Q_CONVERT(0.970031), DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(0.932993), DSP16_Q_CONVERT(-0.359895),
      DSP16_Q_CONVERT(0.966976), DSP16_Q_CONVERT(-0.254866), DSP16_Q_CONVERT(0.926210), DSP16_Q_CONVERT(-0.377007),
      DSP16_Q_CONVERT(0.963776), DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(0.919114), DSP16_Q_CONVERT(-0.393992),
      DSP16_Q_CONVERT(0.960431), DSP16_Q_CONVERT(-0.278520), DSP16_Q_CONVERT(0.911706), DSP16_Q_CONVERT(-0.410843),
      DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555),
      DSP16_Q_CONVERT(0.953306), DSP16_Q_CONVERT(-0.302006), DSP16_Q_CONVERT(0.895966), DSP16_Q_CONVERT(-0.444122),
      DSP16_Q_CONVERT(0.949528), DSP16_Q_CONVERT(-0.313682), DSP16_Q_CONVERT(0.887640), DSP16_Q_CONVERT(-0.460539),
      DSP16_Q_CONVERT(0.945607), DSP16_Q_CONVERT(-0.325310), DSP16_Q_CONVERT(0.879012), DSP16_Q_CONVERT(-0.476799),
      DSP16_Q_CONVERT(0.941544), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(0.870087), DSP16_Q_CONVERT(-0.492898),
      DSP16_Q_CONVERT(0.937339), DSP16_Q_CONVERT(-0.348419), DSP16_Q_CONVERT(0.860867), DSP16_Q_CONVERT(-0.508830),
      DSP16_Q_CONVERT(0.932993), DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(0.851355), DSP16_Q_CONVERT(-0.524590),
      DSP16_Q_CONVERT(0.928506), DSP16_Q_CONVERT(-0.371317), DSP16_Q_CONVERT(0.841555), DSP16_Q_CONVERT(-0.540171),
      DSP16_Q_CONVERT(0.923880), DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570),
      DSP16_Q_CONVERT(0.919114), DSP16_Q_CONVERT(-0.393992), DSP16_Q_CONVERT(0.821103), DSP16_Q_CONVERT(-0.570781),
      DSP16_Q_CONVERT(0.914210), DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(0.810457), DSP16_Q_CONVERT(-0.585798),
      DSP16_Q_CONVERT(0.909168), DSP16_Q_CONVERT(-0.416430), DSP16_Q_CONVERT(0.799537), DSP16_Q_CONVERT(-0.600616),
      DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(0.788346), DSP16_Q_CONVERT(-0.615232),
      DSP16_Q_CONVERT(0.898674), DSP16_Q_CONVERT(-0.438616), DSP16_Q_CONVERT(0.776888), DSP16_Q_CONVERT(-0.629638),
      DSP16_Q_CONVERT(0.893224), DSP16_Q_CONVERT(-0.449611), DSP16_Q_CONVERT(0.765167), DSP16_Q_CONVERT(-0.643832),
      DSP16_Q_CONVERT(0.887640), DSP16_Q_CONVERT(-0.460539), DSP16_Q_CONVERT(0.753187), DSP16_Q_CONVERT(-0.657807),
      DSP16_Q_CONVERT(0.881921), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559),
      DSP16_Q_CONVERT(0.876070), DSP16_Q_CONVERT(-0.482184), DSP16_Q_CONVERT(0.728464), DSP16_Q_CONVERT(-0.685084),
      DSP16_Q_CONVERT(0.870087), DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(0.715731), DSP16_Q_CONVERT(-0.698376),
      DSP16_Q_CONVERT(0.863973), DSP16_Q_CONVERT(-0.503538), DSP16_Q_CONVERT(0.702755), DSP16_Q_CONVERT(-0.711432),
      DSP16_Q_CONVERT(0.857729), DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(0.689541), DSP16_Q_CONVERT(-0.724247),
      DSP16_Q_CONVERT(0.851355), DSP16_Q_CONVERT(-0.524590), DSP16_Q_CONVERT(0.676093), DSP16_Q_CONVERT(-0.736817),
      DSP16_Q_CONVERT(0.844854), DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(0.662416), DSP16_Q_CONVERT(-0.749136),
      DSP16_Q_CONVERT(0.838225), DSP16_Q_CONVERT(-0.545325), DSP16_Q_CONVERT(0.648514), DSP16_Q_CONVERT(-0.761202),
      DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010),
      DSP16_Q_CONVERT(0.824589), DSP16_Q_CONVERT(-0.565732), DSP16_Q_CONVERT(0.620057), DSP16_Q_CONVERT(-0.784557),
      DSP16_Q_CONVERT(0.817585), DSP16_Q_CONVERT(-0.575808), DSP16_Q_CONVERT(0.605511), DSP16_Q_CONVERT(-0.795837),
      DSP16_Q_CONVERT(0.810457), DSP16_Q_CONVERT(-0.585798), DSP16_Q_CONVERT(0.590760), DSP16_Q_CONVERT(-0.806848),
      DSP16_Q_CONVERT(0.803208), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(0.575808), DSP16_Q_CONVERT(-0.817585),
      DSP16_Q_CONVERT(0.795837), DSP16_Q_CONVERT(-0.605511), DSP16_Q_CONVERT(0.560662), DSP16_Q_CONVERT(-0.828045),
      DSP16_Q_CONVERT(0.788346), DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(0.545325), DSP16_Q_CONVERT(-0.838225),
      DSP16_Q_CONVERT(0.780737), DSP16_Q_CONVERT(-0.624859), DSP16_Q_CONVERT(0.529804), DSP16_Q_CONVERT(-0.848120),
      DSP16_Q_CONVERT(0.773010), DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729),
      DSP16_Q_CONVERT(0.765167), DSP16_Q_CONVERT(-0.643832), DSP16_Q_CONVERT(0.498228), DSP16_Q_CONVERT(-0.867046),
      DSP16_Q_CONVERT(0.757209), DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(0.482184), DSP16_Q_CONVERT(-0.876070),
      DSP16_Q_CONVERT(0.749136), DSP16_Q_CONVERT(-0.662416), DSP16_Q_CONVERT(0.465976), DSP16_Q_CONVERT(-0.884797),
      DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(0.449611), DSP16_Q_CONVERT(-0.893224),
      DSP16_Q_CONVERT(0.732654), DSP16_Q_CONVERT(-0.680601), DSP16_Q_CONVERT(0.433094), DSP16_Q_CONVERT(-0.901349),
      DSP16_Q_CONVERT(0.724247), DSP16_Q_CONVERT(-0.689541), DSP16_Q_CONVERT(0.416430), DSP16_Q_CONVERT(-0.909168),
      DSP16_Q_CONVERT(0.715731), DSP16_Q_CONVERT(-0.698376), DSP16_Q_CONVERT(0.399624), DSP16_Q_CONVERT(-0.916679),
      DSP16_Q_CONVERT(0.707107), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880),
      DSP16_Q_CONVERT(0.698376), DSP16_Q_CONVERT(-0.715731), DSP16_Q_CONVERT(0.365613), DSP16_Q_CONVERT(-0.930767),
      DSP16_Q_CONVERT(0.689541), DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(0.348419), DSP16_Q_CONVERT(-0.937339),
      DSP16_Q_CONVERT(0.680601), DSP16_Q_CONVERT(-0.732654), DSP16_Q_CONVERT(0.331106), DSP16_Q_CONVERT(-0.943593),
      DSP16_Q_CONVERT(0.671559), DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(0.313682), DSP16_Q_CONVERT(-0.949528),
      DSP16_Q_CONVERT(0.662416), DSP16_Q_CONVERT(-0.749136), DSP16_Q_CONVERT(0.296151), DSP16_Q_CONVERT(-0.955141),
      DSP16_Q_CONVERT(0.653173), DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(0.278520), DSP16_Q_CONVERT(-0.960431),
      DSP16_Q_CONVERT(0.643832), DSP16_Q_CONVERT(-0.765167), DSP16_Q_CONVERT(0.260794), DSP16_Q_CONVERT(-0.965394),
      DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031),
      DSP16_Q_CONVERT(0.624859), DSP16_Q_CONVERT(-0.780737), DSP16_Q_CONVERT(0.225084), DSP16_Q_CONVERT(-0.974339),
      DSP16_Q_CONVERT(0.615232), DSP16_Q_CONVERT(-0.788346), DSP16_Q_CONVERT(0.207111), DSP16_Q_CONVERT(-0.978317),
      DSP16_Q_CONVERT(0.605511), DSP16_Q_CONVERT(-0.795837), DSP16_Q_CONVERT(0.189069), DSP16_Q_CONVERT(-0.981964),
      DSP16_Q_CONVERT(0.595699), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(0.170962), DSP16_Q_CONVERT(-0.985278),
      DSP16_Q_CONVERT(0.585798), DSP16_Q_CONVERT(-0.810457), DSP16_Q_CONVERT(0.152797), DSP16_Q_CONVERT(-0.988258),
      DSP16_Q_CONVERT(0.575808), DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(0.134581), DSP16_Q_CONVERT(-0.990903),
      DSP16_Q_CONVERT(0.565732), DSP16_Q_CONVERT(-0.824589), DSP16_Q_CONVERT(0.116319), DSP16_Q_CONVERT(-0.993212),
      DSP16_Q_CONVERT(0.555570), DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185),
      DSP16_Q_CONVERT(0.545325), DSP16_Q_CONVERT(-0.838225), DSP16_Q_CONVERT(0.079682), DSP16_Q_CONVERT(-0.996820),
      DSP16_Q_CONVERT(0.534998), DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(0.061321), DSP16_Q_CONVERT(-0.998118),
      DSP16_Q_CONVERT(0.524590), DSP16_Q_CONVERT(-0.851355), DSP16_Q_CONVERT(0.042938), DSP16_Q_CONVERT(-0.999078),
      DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(0.024541), DSP16_Q_CONVERT(-0.999699),
      DSP16_Q_CONVERT(0.503538), DSP16_Q_CONVERT(-0.863973), DSP16_Q_CONVERT(0.006136), DSP16_Q_CONVERT(-0.999981),
      DSP16_Q_CONVERT(0.492898), DSP16_Q_CONVERT(-0.870087), DSP16_Q_CONVERT(-0.012272), DSP16_Q_CONVERT(-0.999925),
      DSP16_Q_CONVERT(0.482184), DSP16_Q_CONVERT(-0.876070), DSP16_Q_CONVERT(-0.030675), DSP16_Q_CONVERT(-0.999529),
      DSP16_Q_CONVERT(0.471397), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.998795),
      DSP16_Q_CONVERT(0.460539), DSP16_Q_CONVERT(-0.887640), DSP16_Q_CONVERT(-0.067444), DSP16_Q_CONVERT(-0.997723),
      DSP16_Q_CONVERT(0.449611), DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(-0.085797), DSP16_Q_CONVERT(-0.996313),
      DSP16_Q_CONVERT(0.438616), DSP16_Q_CONVERT(-0.898674), DSP16_Q_CONVERT(-0.104122), DSP16_Q_CONVERT(-0.994565),
      DSP16_Q_CONVERT(0.427555), DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(-0.992480),
      DSP16_Q_CONVERT(0.416430), DSP16_Q_CONVERT(-0.909168), DSP16_Q_CONVERT(-0.140658), DSP16_Q_CONVERT(-0.990058),
      DSP16_Q_CONVERT(0.405241), DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.158858), DSP16_Q_CONVERT(-0.987301),
      DSP16_Q_CONVERT(0.393992), DSP16_Q_CONVERT(-0.919114), DSP16_Q_CONVERT(-0.177004), DSP16_Q_CONVERT(-0.984210),
      DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.980785),
      DSP16_Q_CONVERT(0.371317), DSP16_Q_CONVERT(-0.928506), DSP16_Q_CONVERT(-0.213110), DSP16_Q_CONVERT(-0.977028),
      DSP16_Q_CONVERT(0.359895), DSP16_Q_CONVERT(-0.932993), DSP16_Q_CONVERT(-0.231058), DSP16_Q_CONVERT(-0.972940),
      DSP16_Q_CONVERT(0.348419), DSP16_Q_CONVERT(-0.937339), DSP16_Q_CONVERT(-0.248928), DSP16_Q_CONVERT(-0.968522),
      DSP16_Q_CONVERT(0.336890), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(-0.963776),
      DSP16_Q_CONVERT(0.325310), DSP16_Q_CONVERT(-0.945607), DSP16_Q_CONVERT(-0.284408), DSP16_Q_CONVERT(-0.958703),
      DSP16_Q_CONVERT(0.313682), DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(-0.302006), DSP16_Q_CONVERT(-0.953306),
      DSP16_Q_CONVERT(0.302006), DSP16_Q_CONVERT(-0.953306), DSP16_Q_CONVERT(-0.319502), DSP16_Q_CONVERT(-0.947586),
      DSP16_Q_CONVERT(0.290285), DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.941544),
      DSP16_Q_CONVERT(0.278520), DSP16_Q_CONVERT(-0.960431), DSP16_Q_CONVERT(-0.354164), DSP16_Q_CONVERT(-0.935184),
      DSP16_Q_CONVERT(0.266713), DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.371317), DSP16_Q_CONVERT(-0.928506),
      DSP16_Q_CONVERT(0.254866), DSP16_Q_CONVERT(-0.966976), DSP16_Q_CONVERT(-0.388345), DSP16_Q_CONVERT(-0.921514),
      DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(-0.914210),
      DSP16_Q_CONVERT(0.231058), DSP16_Q_CONVERT(-0.972940), DSP16_Q_CONVERT(-0.422000), DSP16_Q_CONVERT(-0.906596),
      DSP16_Q_CONVERT(0.219101), DSP16_Q_CONVERT(-0.975702), DSP16_Q_CONVERT(-0.438616), DSP16_Q_CONVERT(-0.898674),
      DSP16_Q_CONVERT(0.207111), DSP16_Q_CONVERT(-0.978317), DSP16_Q_CONVERT(-0.455084), DSP16_Q_CONVERT(-0.890449),
      DSP16_Q_CONVERT(0.195090), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.881921),
      DSP16_Q_CONVERT(0.183040), DSP16_Q_CONVERT(-0.983105), DSP16_Q_CONVERT(-0.487550), DSP16_Q_CONVERT(-0.873095),
      DSP16_Q_CONVERT(0.170962), DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(-0.503538), DSP16_Q_CONVERT(-0.863973),
      DSP16_Q_CONVERT(0.158858), DSP16_Q_CONVERT(-0.987301), DSP16_Q_CONVERT(-0.519356), DSP16_Q_CONVERT(-0.854558),
      DSP16_Q_CONVERT(0.146730), DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(-0.844854),
      DSP16_Q_CONVERT(0.134581), DSP16_Q_CONVERT(-0.990903), DSP16_Q_CONVERT(-0.550458), DSP16_Q_CONVERT(-0.834863),
      DSP16_Q_CONVERT(0.122411), DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.565732), DSP16_Q_CONVERT(-0.824589),
      DSP16_Q_CONVERT(0.110222), DSP16_Q_CONVERT(-0.993907), DSP16_Q_CONVERT(-0.580814), DSP16_Q_CONVERT(-0.814036),
      DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.803208),
      DSP16_Q_CONVERT(0.085797), DSP16_Q_CONVERT(-0.996313), DSP16_Q_CONVERT(-0.610383), DSP16_Q_CONVERT(-0.792107),
      DSP16_Q_CONVERT(0.073565), DSP16_Q_CONVERT(-0.997290), DSP16_Q_CONVERT(-0.624859), DSP16_Q_CONVERT(-0.780737),
      DSP16_Q_CONVERT(0.061321), DSP16_Q_CONVERT(-0.998118), DSP16_Q_CONVERT(-0.639124), DSP16_Q_CONVERT(-0.769103),
      DSP16_Q_CONVERT(0.049068), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(-0.757209),
      DSP16_Q_CONVERT(0.036807), DSP16_Q_CONVERT(-0.999322), DSP16_Q_CONVERT(-0.667000), DSP16_Q_CONVERT(-0.745058),
      DSP16_Q_CONVERT(0.024541), DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(-0.680601), DSP16_Q_CONVERT(-0.732654),
      DSP16_Q_CONVERT(0.012272), DSP16_Q_CONVERT(-0.999925), DSP16_Q_CONVERT(-0.693971), DSP16_Q_CONVERT(-0.720003),
      DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(-1.000000), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.707107),
      DSP16_Q_CONVERT(-0.012272), DSP16_Q_CONVERT(-0.999925), DSP16_Q_CONVERT(-0.720003), DSP16_Q_CONVERT(-0.693971),
      DSP16_Q_CONVERT(-0.024541), DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(-0.732654), DSP16_Q_CONVERT(-0.680601),
      DSP16_Q_CONVERT(-0.036807), DSP16_Q_CONVERT(-0.999322), DSP16_Q_CONVERT(-0.745058), DSP16_Q_CONVERT(-0.667000),
      DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(-0.653173),
      DSP16_Q_CONVERT(-0.061321), DSP16_Q_CONVERT(-0.998118), DSP16_Q_CONVERT(-0.769103), DSP16_Q_CONVERT(-0.639124),
      DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(-0.997290), DSP16_Q_CONVERT(-0.780737), DSP16_Q_CONVERT(-0.624859),
      DSP16_Q_CONVERT(-0.085797), DSP16_Q_CONVERT(-0.996313), DSP16_Q_CONVERT(-0.792107), DSP16_Q_CONVERT(-0.610383),
      DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.595699),
      DSP16_Q_CONVERT(-0.110222), DSP16_Q_CONVERT(-0.993907), DSP16_Q_CONVERT(-0.814036), DSP16_Q_CONVERT(-0.580814),
      DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.824589), DSP16_Q_CONVERT(-0.565732),
      DSP16_Q_CONVERT(-0.134581), DSP16_Q_CONVERT(-0.990903), DSP16_Q_CONVERT(-0.834863), DSP16_Q_CONVERT(-0.550458),
      DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(-0.534998),
      DSP16_Q_CONVERT(-0.158858), DSP16_Q_CONVERT(-0.987301), DSP16_Q_CONVERT(-0.854558), DSP16_Q_CONVERT(-0.519356),
      DSP16_Q_CONVERT(-0.170962), DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(-0.863973), DSP16_Q_CONVERT(-0.503538),
      DSP16_Q_CONVERT(-0.183040), DSP16_Q_CONVERT(-0.983105), DSP16_Q_CONVERT(-0.873095), DSP16_Q_CONVERT(-0.487550),
      DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.471397),
      DSP16_Q_CONVERT(-0.207111), DSP16_Q_CONVERT(-0.978317), DSP16_Q_CONVERT(-0.890449), DSP16_Q_CONVERT(-0.455084),
      DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(-0.975702), DSP16_Q_CONVERT(-0.898674), DSP16_Q_CONVERT(-0.438616),
      DSP16_Q_CONVERT(-0.231058), DSP16_Q_CONVERT(-0.972940), DSP16_Q_CONVERT(-0.906596), DSP16_Q_CONVERT(-0.422000),
      DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.405241),
      DSP16_Q_CONVERT(-0.254866), DSP16_Q_CONVERT(-0.966976), DSP16_Q_CONVERT(-0.921514), DSP16_Q_CONVERT(-0.388345),
      DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.928506), DSP16_Q_CONVERT(-0.371317),
      DSP16_Q_CONVERT(-0.278520), DSP16_Q_CONVERT(-0.960431), DSP16_Q_CONVERT(-0.935184), DSP16_Q_CONVERT(-0.354164),
      DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.336890),
      DSP16_Q_CONVERT(-0.302006), DSP16_Q_CONVERT(-0.953306), DSP16_Q_CONVERT(-0.947586), DSP16_Q_CONVERT(-0.319502),
      DSP16_Q_CONVERT(-0.313682), DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(-0.953306), DSP16_Q_CONVERT(-0.302006),
      DSP16_Q_CONVERT(-0.325310), DSP16_Q_CONVERT(-0.945607), DSP16_Q_CONVERT(-0.958703), DSP16_Q_CONVERT(-0.284408),
      DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.266713),
      DSP16_Q_CONVERT(-0.348419), DSP16_Q_CONVERT(-0.937339), DSP16_Q_CONVERT(-0.968522), DSP16_Q_CONVERT(-0.248928),
      DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(-0.932993), DSP16_Q_CONVERT(-0.972940), DSP16_Q_CONVERT(-0.231058),
      DSP16_Q_CONVERT(-0.371317), DSP16_Q_CONVERT(-0.928506), DSP16_Q_CONVERT(-0.977028), DSP16_Q_CONVERT(-0.213110),
      DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.195090),
      DSP16_Q_CONVERT(-0.393992), DSP16_Q_CONVERT(-0.919114), DSP16_Q_CONVERT(-0.984210), DSP16_Q_CONVERT(-0.177004),
      DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.987301), DSP16_Q_CONVERT(-0.158858),
      DSP16_Q_CONVERT(-0.416430), DSP16_Q_CONVERT(-0.909168), DSP16_Q_CONVERT(-0.990058), DSP16_Q_CONVERT(-0.140658),
      DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.122411),
      DSP16_Q_CONVERT(-0.438616), DSP16_Q_CONVERT(-0.898674), DSP16_Q_CONVERT(-0.994565), DSP16_Q_CONVERT(-0.104122),
      DSP16_Q_CONVERT(-0.449611), DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(-0.996313), DSP16_Q_CONVERT(-0.085797),
      DSP16_Q_CONVERT(-0.460539), DSP16_Q_CONVERT(-0.887640), DSP16_Q_CONVERT(-0.997723), DSP16_Q_CONVERT(-0.067444),
      DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.049068),
      DSP16_Q_CONVERT(-0.482184), DSP16_Q_CONVERT(-0.876070), DSP16_Q_CONVERT(-0.999529), DSP16_Q_CONVERT(-0.030675),
      DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(-0.870087), DSP16_Q_CONVERT(-0.999925), DSP16_Q_CONVERT(-0.012272),
      DSP16_Q_CONVERT(-0.503538), DSP16_Q_CONVERT(-0.863973), DSP16_Q_CONVERT(-0.999981), DSP16_Q_CONVERT(0.006136),
      DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(0.024541),
      DSP16_Q_CONVERT(-0.524590), DSP16_Q_CONVERT(-0.851355), DSP16_Q_CONVERT(-0.999078), DSP16_Q_CONVERT(0.042938),
      DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(-0.998118), DSP16_Q_CONVERT(0.061321),
      DSP16_Q_CONVERT(-0.545325), DSP16_Q_CONVERT(-0.838225), DSP16_Q_CONVERT(-0.996820), DSP16_Q_CONVERT(0.079682),
      DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(0.098017),
      DSP16_Q_CONVERT(-0.565732), DSP16_Q_CONVERT(-0.824589), DSP16_Q_CONVERT(-0.993212), DSP16_Q_CONVERT(0.116319),
      DSP16_Q_CONVERT(-0.575808), DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(-0.990903), DSP16_Q_CONVERT(0.134581),
      DSP16_Q_CONVERT(-0.585798), DSP16_Q_CONVERT(-0.810457), DSP16_Q_CONVERT(-0.988258), DSP16_Q_CONVERT(0.152797),
      DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(0.170962),
      DSP16_Q_CONVERT(-0.605511), DSP16_Q_CONVERT(-0.795837), DSP16_Q_CONVERT(-0.981964), DSP16_Q_CONVERT(0.189069),
      DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(-0.788346), DSP16_Q_CONVERT(-0.978317), DSP16_Q_CONVERT(0.207111),
      DSP16_Q_CONVERT(-0.624859), DSP16_Q_CONVERT(-0.780737), DSP16_Q_CONVERT(-0.974339), DSP16_Q_CONVERT(0.225084),
      DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(0.242980),
      DSP16_Q_CONVERT(-0.643832), DSP16_Q_CONVERT(-0.765167), DSP16_Q_CONVERT(-0.965394), DSP16_Q_CONVERT(0.260794),
      DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(-0.960431), DSP16_Q_CONVERT(0.278520),
      DSP16_Q_CONVERT(-0.662416), DSP16_Q_CONVERT(-0.749136), DSP16_Q_CONVERT(-0.955141), DSP16_Q_CONVERT(0.296151),
      DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(0.313682),
      DSP16_Q_CONVERT(-0.680601), DSP16_Q_CONVERT(-0.732654), DSP16_Q_CONVERT(-0.943593), DSP16_Q_CONVERT(0.331106),
      DSP16_Q_CONVERT(-0.689541), DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(-0.937339), DSP16_Q_CONVERT(0.348419),
      DSP16_Q_CONVERT(-0.698376), DSP16_Q_CONVERT(-0.715731), DSP16_Q_CONVERT(-0.930767), DSP16_Q_CONVERT(0.365613),
      DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(0.382683),
      DSP16_Q_CONVERT(-0.715731), DSP16_Q_CONVERT(-0.698376), DSP16_Q_CONVERT(-0.916679), DSP16_Q_CONVERT(0.399624),
      DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(-0.689541), DSP16_Q_CONVERT(-0.909168), DSP16_Q_CONVERT(0.416430),
      DSP16_Q_CONVERT(-0.732654), DSP16_Q_CONVERT(-0.680601), DSP16_Q_CONVERT(-0.901349), DSP16_Q_CONVERT(0.433094),
      DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(0.449611),
      DSP16_Q_CONVERT(-0.749136), DSP16_Q_CONVERT(-0.662416), DSP16_Q_CONVERT(-0.884797), DSP16_Q_CONVERT(0.465976),
      DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(-0.876070), DSP16_Q_CONVERT(0.482184),
      DSP16_Q_CONVERT(-0.765167), DSP16_Q_CONVERT(-0.643832), DSP16_Q_CONVERT(-0.867046), DSP16_Q_CONVERT(0.498228),
      DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(0.514103),
      DSP16_Q_CONVERT(-0.780737), DSP16_Q_CONVERT(-0.624859), DSP16_Q_CONVERT(-0.848120), DSP16_Q_CONVERT(0.529804),
      DSP16_Q_CONVERT(-0.788346), DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(-0.838225), DSP16_Q_CONVERT(0.545325),
      DSP16_Q_CONVERT(-0.795837), DSP16_Q_CONVERT(-0.605511), DSP16_Q_CONVERT(-0.828045), DSP16_Q_CONVERT(0.560662),
      DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(0.575808),
      DSP16_Q_CONVERT(-0.810457), DSP16_Q_CONVERT(-0.585798), DSP16_Q_CONVERT(-0.806848), DSP16_Q_CONVERT(0.590760),
      DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(-0.575808), DSP16_Q_CONVERT(-0.795837), DSP16_Q_CONVERT(0.605511),
      DSP16_Q_CONVERT(-0.824589), DSP16_Q_CONVERT(-0.565732), DSP16_Q_CONVERT(-0.784557), DSP16_Q_CONVERT(0.620057),
      DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(0.634393),
      DSP16_Q_CONVERT(-0.838225), DSP16_Q_CONVERT(-0.545325), DSP16_Q_CONVERT(-0.761202), DSP16_Q_CONVERT(0.648514),
      DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(-0.749136), DSP16_Q_CONVERT(0.662416),
      DSP16_Q_CONVERT(-0.851355), DSP16_Q_CONVERT(-0.524590), DSP16_Q_CONVERT(-0.736817), DSP16_Q_CONVERT(0.676093),
      DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(0.689541),
      DSP16_Q_CONVERT(-0.863973), DSP16_Q_CONVERT(-0.503538), DSP16_Q_CONVERT(-0.711432), DSP16_Q_CONVERT(0.702755),
      DSP16_Q_CONVERT(-0.870087), DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(-0.698376), DSP16_Q_CONVERT(0.715731),
      DSP16_Q_CONVERT(-0.876070), DSP16_Q_CONVERT(-0.482184), DSP16_Q_CONVERT(-0.685084), DSP16_Q_CONVERT(0.728464),
      DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(0.740951),
      DSP16_Q_CONVERT(-0.887640), DSP16_Q_CONVERT(-0.460539), DSP16_Q_CONVERT(-0.657807), DSP16_Q_CONVERT(0.753187),
      DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(-0.449611), DSP16_Q_CONVERT(-0.643832), DSP16_Q_CONVERT(0.765167),
      DSP16_Q_CONVERT(-0.898674), DSP16_Q_CONVERT(-0.438616), DSP16_Q_CONVERT(-0.629638), DSP16_Q_CONVERT(0.776888),
      DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(0.788346),
      DSP16_Q_CONVERT(-0.909168), DSP16_Q_CONVERT(-0.416430), DSP16_Q_CONVERT(-0.600616), DSP16_Q_CONVERT(0.799537),
      DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(-0.585798), DSP16_Q_CONVERT(0.810457),
      DSP16_Q_CONVERT(-0.919114), DSP16_Q_CONVERT(-0.393992), DSP16_Q_CONVERT(-0.570781), DSP16_Q_CONVERT(0.821103),
      DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(0.831470),
      DSP16_Q_CONVERT(-0.928506), DSP16_Q_CONVERT(-0.371317), DSP16_Q_CONVERT(-0.540171), DSP16_Q_CONVERT(0.841555),
      DSP16_Q_CONVERT(-0.932993), DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(-0.524590), DSP16_Q_CONVERT(0.851355),
      DSP16_Q_CONVERT(-0.937339), DSP16_Q_CONVERT(-0.348419), DSP16_Q_CONVERT(-0.508830), DSP16_Q_CONVERT(0.860867),
      DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(0.870087),
      DSP16_Q_CONVERT(-0.945607), DSP16_Q_CONVERT(-0.325310), DSP16_Q_CONVERT(-0.476799), DSP16_Q_CONVERT(0.879012),
      DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(-0.313682), DSP16_Q_CONVERT(-0.460539), DSP16_Q_CONVERT(0.887640),
      DSP16_Q_CONVERT(-0.953306), DSP16_Q_CONVERT(-0.302006), DSP16_Q_CONVERT(-0.444122), DSP16_Q_CONVERT(0.895966),
      DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(0.903989),
      DSP16_Q_CONVERT(-0.960431), DSP16_Q_CONVERT(-0.278520), DSP16_Q_CONVERT(-0.410843), DSP16_Q_CONVERT(0.911706),
      DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(-0.393992), DSP16_Q_CONVERT(0.919114),
      DSP16_Q_CONVERT(-0.966976), DSP16_Q_CONVERT(-0.254866), DSP16_Q_CONVERT(-0.377007), DSP16_Q_CONVERT(0.926210),
      DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(0.932993),
      DSP16_Q_CONVERT(-0.972940), DSP16_Q_CONVERT(-0.231058), DSP16_Q_CONVERT(-0.342661), DSP16_Q_CONVERT(0.939459),
      DSP16_Q_CONVERT(-0.975702), DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(-0.325310), DSP16_Q_CONVERT(0.945607),
      DSP16_Q_CONVERT(-0.978317), DSP16_Q_CONVERT(-0.207111), DSP16_Q_CONVERT(-0.307850), DSP16_Q_CONVERT(0.951435),
      DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(0.956940),
      DSP16_Q_CONVERT(-0.983105), DSP16_Q_CONVERT(-0.183040), DSP16_Q_CONVERT(-0.272621), DSP16_Q_CONVERT(0.962121),
      DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(-0.170962), DSP16_Q_CONVERT(-0.254866), DSP16_Q_CONVERT(0.966976),
      DSP16_Q_CONVERT(-0.987301), DSP16_Q_CONVERT(-0.158858), DSP16_Q_CONVERT(-0.237024), DSP16_Q_CONVERT(0.971504),
      DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(0.975702),
      DSP16_Q_CONVERT(-0.990903), DSP16_Q_CONVERT(-0.134581), DSP16_Q_CONVERT(-0.201105), DSP16_Q_CONVERT(0.979570),
      DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(-0.183040), DSP16_Q_CONVERT(0.983105),
      DSP16_Q_CONVERT(-0.993907), DSP16_Q_CONVERT(-0.110222), DSP16_Q_CONVERT(-0.164913), DSP16_Q_CONVERT(0.986308),
      DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(0.989177),
      DSP16_Q_CONVERT(-0.996313), DSP16_Q_CONVERT(-0.085797), DSP16_Q_CONVERT(-0.128498), DSP16_Q_CONVERT(0.991710),
      DSP16_Q_CONVERT(-0.997290), DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(-0.110222), DSP16_Q_CONVERT(0.993907),
      DSP16_Q_CONVERT(-0.998118), DSP16_Q_CONVERT(-0.061321), DSP16_Q_CONVERT(-0.091909), DSP16_Q_CONVERT(0.995767),
      DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(0.997290),
      DSP16_Q_CONVERT(-0.999322), DSP16_Q_CONVERT(-0.036807), DSP16_Q_CONVERT(-0.055195), DSP16_Q_CONVERT(0.998476),
      DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(-0.024541), DSP16_Q_CONVERT(-0.036807), DSP16_Q_CONVERT(0.999322),
      DSP16_Q_CONVERT(-0.999925), DSP16_Q_CONVERT(-0.012272), DSP16_Q_CONVERT(-0.018407), DSP16_Q_CONVERT(0.999831)
    };
#  endif

//! \}

/*
 * Twiddle factors for a 4096-point FFT
 * Minimized twiddle factors table: 2*(4096/4)*2 + 2 = 4 Kbytes
 * Full twiddle factors table: 2*(4096/4)*6 + 2 = 12 Kbytes
 */
#elif DSP16_N_TWIDDLE_FACTORS == 4096

/*! \name 4096 elements twiddle factors table
 */
//! \{
/*! \brief This table has been generated using the following algorithm:
 * \code
 * for i=0:4:DSP16_N_TWIDDLE_FACTORS,
 * k = i/(DSP16_N_TWIDDLE_FACTORS*4);
 * w = exp(-2*PI*%i*k);
 * end;
 * \endcode
 * It is a one dimensional array containing the real parts (even parts) and imaginary parts (odd parts) of the \b w value.
 */
  A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp16_t dsp16_twiddle_factors[DSP16_N_TWIDDLE_FACTORS/2+2] = {
  /*Re(w)   Im(w)  */
    DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000),
    DSP16_Q_CONVERT(0.999999), DSP16_Q_CONVERT(-0.001534),
    DSP16_Q_CONVERT(0.999995), DSP16_Q_CONVERT(-0.003068),
    DSP16_Q_CONVERT(0.999989), DSP16_Q_CONVERT(-0.004602),
    DSP16_Q_CONVERT(0.999981), DSP16_Q_CONVERT(-0.006136),
    DSP16_Q_CONVERT(0.999971), DSP16_Q_CONVERT(-0.007670),
    DSP16_Q_CONVERT(0.999958), DSP16_Q_CONVERT(-0.009204),
    DSP16_Q_CONVERT(0.999942), DSP16_Q_CONVERT(-0.010738),
    DSP16_Q_CONVERT(0.999925), DSP16_Q_CONVERT(-0.012272),
    DSP16_Q_CONVERT(0.999905), DSP16_Q_CONVERT(-0.013805),
    DSP16_Q_CONVERT(0.999882), DSP16_Q_CONVERT(-0.015339),
    DSP16_Q_CONVERT(0.999858), DSP16_Q_CONVERT(-0.016873),
    DSP16_Q_CONVERT(0.999831), DSP16_Q_CONVERT(-0.018407),
    DSP16_Q_CONVERT(0.999801), DSP16_Q_CONVERT(-0.019940),
    DSP16_Q_CONVERT(0.999769), DSP16_Q_CONVERT(-0.021474),
    DSP16_Q_CONVERT(0.999735), DSP16_Q_CONVERT(-0.023008),
    DSP16_Q_CONVERT(0.999699), DSP16_Q_CONVERT(-0.024541),
    DSP16_Q_CONVERT(0.999660), DSP16_Q_CONVERT(-0.026075),
    DSP16_Q_CONVERT(0.999619), DSP16_Q_CONVERT(-0.027608),
    DSP16_Q_CONVERT(0.999575), DSP16_Q_CONVERT(-0.029142),
    DSP16_Q_CONVERT(0.999529), DSP16_Q_CONVERT(-0.030675),
    DSP16_Q_CONVERT(0.999481), DSP16_Q_CONVERT(-0.032208),
    DSP16_Q_CONVERT(0.999431), DSP16_Q_CONVERT(-0.033741),
    DSP16_Q_CONVERT(0.999378), DSP16_Q_CONVERT(-0.035274),
    DSP16_Q_CONVERT(0.999322), DSP16_Q_CONVERT(-0.036807),
    DSP16_Q_CONVERT(0.999265), DSP16_Q_CONVERT(-0.038340),
    DSP16_Q_CONVERT(0.999205), DSP16_Q_CONVERT(-0.039873),
    DSP16_Q_CONVERT(0.999142), DSP16_Q_CONVERT(-0.041406),
    DSP16_Q_CONVERT(0.999078), DSP16_Q_CONVERT(-0.042938),
    DSP16_Q_CONVERT(0.999011), DSP16_Q_CONVERT(-0.044471),
    DSP16_Q_CONVERT(0.998941), DSP16_Q_CONVERT(-0.046003),
    DSP16_Q_CONVERT(0.998870), DSP16_Q_CONVERT(-0.047535),
    DSP16_Q_CONVERT(0.998795), DSP16_Q_CONVERT(-0.049068),
    DSP16_Q_CONVERT(0.998719), DSP16_Q_CONVERT(-0.050600),
    DSP16_Q_CONVERT(0.998640), DSP16_Q_CONVERT(-0.052132),
    DSP16_Q_CONVERT(0.998559), DSP16_Q_CONVERT(-0.053664),
    DSP16_Q_CONVERT(0.998476), DSP16_Q_CONVERT(-0.055195),
    DSP16_Q_CONVERT(0.998390), DSP16_Q_CONVERT(-0.056727),
    DSP16_Q_CONVERT(0.998302), DSP16_Q_CONVERT(-0.058258),
    DSP16_Q_CONVERT(0.998211), DSP16_Q_CONVERT(-0.059790),
    DSP16_Q_CONVERT(0.998118), DSP16_Q_CONVERT(-0.061321),
    DSP16_Q_CONVERT(0.998023), DSP16_Q_CONVERT(-0.062852),
    DSP16_Q_CONVERT(0.997925), DSP16_Q_CONVERT(-0.064383),
    DSP16_Q_CONVERT(0.997825), DSP16_Q_CONVERT(-0.065913),
    DSP16_Q_CONVERT(0.997723), DSP16_Q_CONVERT(-0.067444),
    DSP16_Q_CONVERT(0.997618), DSP16_Q_CONVERT(-0.068974),
    DSP16_Q_CONVERT(0.997511), DSP16_Q_CONVERT(-0.070505),
    DSP16_Q_CONVERT(0.997402), DSP16_Q_CONVERT(-0.072035),
    DSP16_Q_CONVERT(0.997290), DSP16_Q_CONVERT(-0.073565),
    DSP16_Q_CONVERT(0.997176), DSP16_Q_CONVERT(-0.075094),
    DSP16_Q_CONVERT(0.997060), DSP16_Q_CONVERT(-0.076624),
    DSP16_Q_CONVERT(0.996941), DSP16_Q_CONVERT(-0.078153),
    DSP16_Q_CONVERT(0.996820), DSP16_Q_CONVERT(-0.079682),
    DSP16_Q_CONVERT(0.996697), DSP16_Q_CONVERT(-0.081211),
    DSP16_Q_CONVERT(0.996571), DSP16_Q_CONVERT(-0.082740),
    DSP16_Q_CONVERT(0.996443), DSP16_Q_CONVERT(-0.084269),
    DSP16_Q_CONVERT(0.996313), DSP16_Q_CONVERT(-0.085797),
    DSP16_Q_CONVERT(0.996180), DSP16_Q_CONVERT(-0.087326),
    DSP16_Q_CONVERT(0.996045), DSP16_Q_CONVERT(-0.088854),
    DSP16_Q_CONVERT(0.995907), DSP16_Q_CONVERT(-0.090381),
    DSP16_Q_CONVERT(0.995767), DSP16_Q_CONVERT(-0.091909),
    DSP16_Q_CONVERT(0.995625), DSP16_Q_CONVERT(-0.093436),
    DSP16_Q_CONVERT(0.995481), DSP16_Q_CONVERT(-0.094963),
    DSP16_Q_CONVERT(0.995334), DSP16_Q_CONVERT(-0.096490),
    DSP16_Q_CONVERT(0.995185), DSP16_Q_CONVERT(-0.098017),
    DSP16_Q_CONVERT(0.995033), DSP16_Q_CONVERT(-0.099544),
    DSP16_Q_CONVERT(0.994879), DSP16_Q_CONVERT(-0.101070),
    DSP16_Q_CONVERT(0.994723), DSP16_Q_CONVERT(-0.102596),
    DSP16_Q_CONVERT(0.994565), DSP16_Q_CONVERT(-0.104122),
    DSP16_Q_CONVERT(0.994404), DSP16_Q_CONVERT(-0.105647),
    DSP16_Q_CONVERT(0.994240), DSP16_Q_CONVERT(-0.107172),
    DSP16_Q_CONVERT(0.994075), DSP16_Q_CONVERT(-0.108697),
    DSP16_Q_CONVERT(0.993907), DSP16_Q_CONVERT(-0.110222),
    DSP16_Q_CONVERT(0.993737), DSP16_Q_CONVERT(-0.111747),
    DSP16_Q_CONVERT(0.993564), DSP16_Q_CONVERT(-0.113271),
    DSP16_Q_CONVERT(0.993389), DSP16_Q_CONVERT(-0.114795),
    DSP16_Q_CONVERT(0.993212), DSP16_Q_CONVERT(-0.116319),
    DSP16_Q_CONVERT(0.993032), DSP16_Q_CONVERT(-0.117842),
    DSP16_Q_CONVERT(0.992850), DSP16_Q_CONVERT(-0.119365),
    DSP16_Q_CONVERT(0.992666), DSP16_Q_CONVERT(-0.120888),
    DSP16_Q_CONVERT(0.992480), DSP16_Q_CONVERT(-0.122411),
    DSP16_Q_CONVERT(0.992291), DSP16_Q_CONVERT(-0.123933),
    DSP16_Q_CONVERT(0.992099), DSP16_Q_CONVERT(-0.125455),
    DSP16_Q_CONVERT(0.991906), DSP16_Q_CONVERT(-0.126977),
    DSP16_Q_CONVERT(0.991710), DSP16_Q_CONVERT(-0.128498),
    DSP16_Q_CONVERT(0.991511), DSP16_Q_CONVERT(-0.130019),
    DSP16_Q_CONVERT(0.991311), DSP16_Q_CONVERT(-0.131540),
    DSP16_Q_CONVERT(0.991108), DSP16_Q_CONVERT(-0.133061),
    DSP16_Q_CONVERT(0.990903), DSP16_Q_CONVERT(-0.134581),
    DSP16_Q_CONVERT(0.990695), DSP16_Q_CONVERT(-0.136101),
    DSP16_Q_CONVERT(0.990485), DSP16_Q_CONVERT(-0.137620),
    DSP16_Q_CONVERT(0.990273), DSP16_Q_CONVERT(-0.139139),
    DSP16_Q_CONVERT(0.990058), DSP16_Q_CONVERT(-0.140658),
    DSP16_Q_CONVERT(0.989841), DSP16_Q_CONVERT(-0.142177),
    DSP16_Q_CONVERT(0.989622), DSP16_Q_CONVERT(-0.143695),
    DSP16_Q_CONVERT(0.989400), DSP16_Q_CONVERT(-0.145213),
    DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730),
    DSP16_Q_CONVERT(0.988950), DSP16_Q_CONVERT(-0.148248),
    DSP16_Q_CONVERT(0.988722), DSP16_Q_CONVERT(-0.149765),
    DSP16_Q_CONVERT(0.988491), DSP16_Q_CONVERT(-0.151281),
    DSP16_Q_CONVERT(0.988258), DSP16_Q_CONVERT(-0.152797),
    DSP16_Q_CONVERT(0.988022), DSP16_Q_CONVERT(-0.154313),
    DSP16_Q_CONVERT(0.987784), DSP16_Q_CONVERT(-0.155828),
    DSP16_Q_CONVERT(0.987544), DSP16_Q_CONVERT(-0.157343),
    DSP16_Q_CONVERT(0.987301), DSP16_Q_CONVERT(-0.158858),
    DSP16_Q_CONVERT(0.987057), DSP16_Q_CONVERT(-0.160372),
    DSP16_Q_CONVERT(0.986809), DSP16_Q_CONVERT(-0.161886),
    DSP16_Q_CONVERT(0.986560), DSP16_Q_CONVERT(-0.163400),
    DSP16_Q_CONVERT(0.986308), DSP16_Q_CONVERT(-0.164913),
    DSP16_Q_CONVERT(0.986054), DSP16_Q_CONVERT(-0.166426),
    DSP16_Q_CONVERT(0.985798), DSP16_Q_CONVERT(-0.167938),
    DSP16_Q_CONVERT(0.985539), DSP16_Q_CONVERT(-0.169450),
    DSP16_Q_CONVERT(0.985278), DSP16_Q_CONVERT(-0.170962),
    DSP16_Q_CONVERT(0.985014), DSP16_Q_CONVERT(-0.172473),
    DSP16_Q_CONVERT(0.984749), DSP16_Q_CONVERT(-0.173984),
    DSP16_Q_CONVERT(0.984480), DSP16_Q_CONVERT(-0.175494),
    DSP16_Q_CONVERT(0.984210), DSP16_Q_CONVERT(-0.177004),
    DSP16_Q_CONVERT(0.983937), DSP16_Q_CONVERT(-0.178514),
    DSP16_Q_CONVERT(0.983662), DSP16_Q_CONVERT(-0.180023),
    DSP16_Q_CONVERT(0.983385), DSP16_Q_CONVERT(-0.181532),
    DSP16_Q_CONVERT(0.983105), DSP16_Q_CONVERT(-0.183040),
    DSP16_Q_CONVERT(0.982824), DSP16_Q_CONVERT(-0.184548),
    DSP16_Q_CONVERT(0.982539), DSP16_Q_CONVERT(-0.186055),
    DSP16_Q_CONVERT(0.982253), DSP16_Q_CONVERT(-0.187562),
    DSP16_Q_CONVERT(0.981964), DSP16_Q_CONVERT(-0.189069),
    DSP16_Q_CONVERT(0.981673), DSP16_Q_CONVERT(-0.190575),
    DSP16_Q_CONVERT(0.981379), DSP16_Q_CONVERT(-0.192080),
    DSP16_Q_CONVERT(0.981083), DSP16_Q_CONVERT(-0.193586),
    DSP16_Q_CONVERT(0.980785), DSP16_Q_CONVERT(-0.195090),
    DSP16_Q_CONVERT(0.980485), DSP16_Q_CONVERT(-0.196595),
    DSP16_Q_CONVERT(0.980182), DSP16_Q_CONVERT(-0.198098),
    DSP16_Q_CONVERT(0.979877), DSP16_Q_CONVERT(-0.199602),
    DSP16_Q_CONVERT(0.979570), DSP16_Q_CONVERT(-0.201105),
    DSP16_Q_CONVERT(0.979260), DSP16_Q_CONVERT(-0.202607),
    DSP16_Q_CONVERT(0.978948), DSP16_Q_CONVERT(-0.204109),
    DSP16_Q_CONVERT(0.978634), DSP16_Q_CONVERT(-0.205610),
    DSP16_Q_CONVERT(0.978317), DSP16_Q_CONVERT(-0.207111),
    DSP16_Q_CONVERT(0.977999), DSP16_Q_CONVERT(-0.208612),
    DSP16_Q_CONVERT(0.977677), DSP16_Q_CONVERT(-0.210112),
    DSP16_Q_CONVERT(0.977354), DSP16_Q_CONVERT(-0.211611),
    DSP16_Q_CONVERT(0.977028), DSP16_Q_CONVERT(-0.213110),
    DSP16_Q_CONVERT(0.976700), DSP16_Q_CONVERT(-0.214609),
    DSP16_Q_CONVERT(0.976370), DSP16_Q_CONVERT(-0.216107),
    DSP16_Q_CONVERT(0.976037), DSP16_Q_CONVERT(-0.217604),
    DSP16_Q_CONVERT(0.975702), DSP16_Q_CONVERT(-0.219101),
    DSP16_Q_CONVERT(0.975365), DSP16_Q_CONVERT(-0.220598),
    DSP16_Q_CONVERT(0.975025), DSP16_Q_CONVERT(-0.222094),
    DSP16_Q_CONVERT(0.974684), DSP16_Q_CONVERT(-0.223589),
    DSP16_Q_CONVERT(0.974339), DSP16_Q_CONVERT(-0.225084),
    DSP16_Q_CONVERT(0.973993), DSP16_Q_CONVERT(-0.226578),
    DSP16_Q_CONVERT(0.973644), DSP16_Q_CONVERT(-0.228072),
    DSP16_Q_CONVERT(0.973293), DSP16_Q_CONVERT(-0.229565),
    DSP16_Q_CONVERT(0.972940), DSP16_Q_CONVERT(-0.231058),
    DSP16_Q_CONVERT(0.972584), DSP16_Q_CONVERT(-0.232550),
    DSP16_Q_CONVERT(0.972226), DSP16_Q_CONVERT(-0.234042),
    DSP16_Q_CONVERT(0.971866), DSP16_Q_CONVERT(-0.235533),
    DSP16_Q_CONVERT(0.971504), DSP16_Q_CONVERT(-0.237024),
    DSP16_Q_CONVERT(0.971139), DSP16_Q_CONVERT(-0.238514),
    DSP16_Q_CONVERT(0.970772), DSP16_Q_CONVERT(-0.240003),
    DSP16_Q_CONVERT(0.970403), DSP16_Q_CONVERT(-0.241492),
    DSP16_Q_CONVERT(0.970031), DSP16_Q_CONVERT(-0.242980),
    DSP16_Q_CONVERT(0.969657), DSP16_Q_CONVERT(-0.244468),
    DSP16_Q_CONVERT(0.969281), DSP16_Q_CONVERT(-0.245955),
    DSP16_Q_CONVERT(0.968903), DSP16_Q_CONVERT(-0.247442),
    DSP16_Q_CONVERT(0.968522), DSP16_Q_CONVERT(-0.248928),
    DSP16_Q_CONVERT(0.968139), DSP16_Q_CONVERT(-0.250413),
    DSP16_Q_CONVERT(0.967754), DSP16_Q_CONVERT(-0.251898),
    DSP16_Q_CONVERT(0.967366), DSP16_Q_CONVERT(-0.253382),
    DSP16_Q_CONVERT(0.966976), DSP16_Q_CONVERT(-0.254866),
    DSP16_Q_CONVERT(0.966584), DSP16_Q_CONVERT(-0.256349),
    DSP16_Q_CONVERT(0.966190), DSP16_Q_CONVERT(-0.257831),
    DSP16_Q_CONVERT(0.965793), DSP16_Q_CONVERT(-0.259313),
    DSP16_Q_CONVERT(0.965394), DSP16_Q_CONVERT(-0.260794),
    DSP16_Q_CONVERT(0.964993), DSP16_Q_CONVERT(-0.262275),
    DSP16_Q_CONVERT(0.964590), DSP16_Q_CONVERT(-0.263755),
    DSP16_Q_CONVERT(0.964184), DSP16_Q_CONVERT(-0.265234),
    DSP16_Q_CONVERT(0.963776), DSP16_Q_CONVERT(-0.266713),
    DSP16_Q_CONVERT(0.963366), DSP16_Q_CONVERT(-0.268191),
    DSP16_Q_CONVERT(0.962953), DSP16_Q_CONVERT(-0.269668),
    DSP16_Q_CONVERT(0.962538), DSP16_Q_CONVERT(-0.271145),
    DSP16_Q_CONVERT(0.962121), DSP16_Q_CONVERT(-0.272621),
    DSP16_Q_CONVERT(0.961702), DSP16_Q_CONVERT(-0.274097),
    DSP16_Q_CONVERT(0.961280), DSP16_Q_CONVERT(-0.275572),
    DSP16_Q_CONVERT(0.960857), DSP16_Q_CONVERT(-0.277046),
    DSP16_Q_CONVERT(0.960431), DSP16_Q_CONVERT(-0.278520),
    DSP16_Q_CONVERT(0.960002), DSP16_Q_CONVERT(-0.279993),
    DSP16_Q_CONVERT(0.959572), DSP16_Q_CONVERT(-0.281465),
    DSP16_Q_CONVERT(0.959139), DSP16_Q_CONVERT(-0.282937),
    DSP16_Q_CONVERT(0.958703), DSP16_Q_CONVERT(-0.284408),
    DSP16_Q_CONVERT(0.958266), DSP16_Q_CONVERT(-0.285878),
    DSP16_Q_CONVERT(0.957826), DSP16_Q_CONVERT(-0.287347),
    DSP16_Q_CONVERT(0.957385), DSP16_Q_CONVERT(-0.288816),
    DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285),
    DSP16_Q_CONVERT(0.956494), DSP16_Q_CONVERT(-0.291752),
    DSP16_Q_CONVERT(0.956045), DSP16_Q_CONVERT(-0.293219),
    DSP16_Q_CONVERT(0.955594), DSP16_Q_CONVERT(-0.294685),
    DSP16_Q_CONVERT(0.955141), DSP16_Q_CONVERT(-0.296151),
    DSP16_Q_CONVERT(0.954686), DSP16_Q_CONVERT(-0.297616),
    DSP16_Q_CONVERT(0.954228), DSP16_Q_CONVERT(-0.299080),
    DSP16_Q_CONVERT(0.953768), DSP16_Q_CONVERT(-0.300543),
    DSP16_Q_CONVERT(0.953306), DSP16_Q_CONVERT(-0.302006),
    DSP16_Q_CONVERT(0.952842), DSP16_Q_CONVERT(-0.303468),
    DSP16_Q_CONVERT(0.952375), DSP16_Q_CONVERT(-0.304929),
    DSP16_Q_CONVERT(0.951906), DSP16_Q_CONVERT(-0.306390),
    DSP16_Q_CONVERT(0.951435), DSP16_Q_CONVERT(-0.307850),
    DSP16_Q_CONVERT(0.950962), DSP16_Q_CONVERT(-0.309309),
    DSP16_Q_CONVERT(0.950486), DSP16_Q_CONVERT(-0.310767),
    DSP16_Q_CONVERT(0.950008), DSP16_Q_CONVERT(-0.312225),
    DSP16_Q_CONVERT(0.949528), DSP16_Q_CONVERT(-0.313682),
    DSP16_Q_CONVERT(0.949046), DSP16_Q_CONVERT(-0.315138),
    DSP16_Q_CONVERT(0.948561), DSP16_Q_CONVERT(-0.316593),
    DSP16_Q_CONVERT(0.948075), DSP16_Q_CONVERT(-0.318048),
    DSP16_Q_CONVERT(0.947586), DSP16_Q_CONVERT(-0.319502),
    DSP16_Q_CONVERT(0.947094), DSP16_Q_CONVERT(-0.320955),
    DSP16_Q_CONVERT(0.946601), DSP16_Q_CONVERT(-0.322408),
    DSP16_Q_CONVERT(0.946105), DSP16_Q_CONVERT(-0.323859),
    DSP16_Q_CONVERT(0.945607), DSP16_Q_CONVERT(-0.325310),
    DSP16_Q_CONVERT(0.945107), DSP16_Q_CONVERT(-0.326760),
    DSP16_Q_CONVERT(0.944605), DSP16_Q_CONVERT(-0.328210),
    DSP16_Q_CONVERT(0.944100), DSP16_Q_CONVERT(-0.329658),
    DSP16_Q_CONVERT(0.943593), DSP16_Q_CONVERT(-0.331106),
    DSP16_Q_CONVERT(0.943084), DSP16_Q_CONVERT(-0.332553),
    DSP16_Q_CONVERT(0.942573), DSP16_Q_CONVERT(-0.334000),
    DSP16_Q_CONVERT(0.942060), DSP16_Q_CONVERT(-0.335445),
    DSP16_Q_CONVERT(0.941544), DSP16_Q_CONVERT(-0.336890),
    DSP16_Q_CONVERT(0.941026), DSP16_Q_CONVERT(-0.338334),
    DSP16_Q_CONVERT(0.940506), DSP16_Q_CONVERT(-0.339777),
    DSP16_Q_CONVERT(0.939984), DSP16_Q_CONVERT(-0.341219),
    DSP16_Q_CONVERT(0.939459), DSP16_Q_CONVERT(-0.342661),
    DSP16_Q_CONVERT(0.938932), DSP16_Q_CONVERT(-0.344101),
    DSP16_Q_CONVERT(0.938404), DSP16_Q_CONVERT(-0.345541),
    DSP16_Q_CONVERT(0.937872), DSP16_Q_CONVERT(-0.346980),
    DSP16_Q_CONVERT(0.937339), DSP16_Q_CONVERT(-0.348419),
    DSP16_Q_CONVERT(0.936803), DSP16_Q_CONVERT(-0.349856),
    DSP16_Q_CONVERT(0.936266), DSP16_Q_CONVERT(-0.351293),
    DSP16_Q_CONVERT(0.935726), DSP16_Q_CONVERT(-0.352729),
    DSP16_Q_CONVERT(0.935184), DSP16_Q_CONVERT(-0.354164),
    DSP16_Q_CONVERT(0.934639), DSP16_Q_CONVERT(-0.355598),
    DSP16_Q_CONVERT(0.934093), DSP16_Q_CONVERT(-0.357031),
    DSP16_Q_CONVERT(0.933544), DSP16_Q_CONVERT(-0.358463),
    DSP16_Q_CONVERT(0.932993), DSP16_Q_CONVERT(-0.359895),
    DSP16_Q_CONVERT(0.932440), DSP16_Q_CONVERT(-0.361326),
    DSP16_Q_CONVERT(0.931884), DSP16_Q_CONVERT(-0.362756),
    DSP16_Q_CONVERT(0.931327), DSP16_Q_CONVERT(-0.364185),
    DSP16_Q_CONVERT(0.930767), DSP16_Q_CONVERT(-0.365613),
    DSP16_Q_CONVERT(0.930205), DSP16_Q_CONVERT(-0.367040),
    DSP16_Q_CONVERT(0.929641), DSP16_Q_CONVERT(-0.368467),
    DSP16_Q_CONVERT(0.929075), DSP16_Q_CONVERT(-0.369892),
    DSP16_Q_CONVERT(0.928506), DSP16_Q_CONVERT(-0.371317),
    DSP16_Q_CONVERT(0.927935), DSP16_Q_CONVERT(-0.372741),
    DSP16_Q_CONVERT(0.927363), DSP16_Q_CONVERT(-0.374164),
    DSP16_Q_CONVERT(0.926787), DSP16_Q_CONVERT(-0.375586),
    DSP16_Q_CONVERT(0.926210), DSP16_Q_CONVERT(-0.377007),
    DSP16_Q_CONVERT(0.925631), DSP16_Q_CONVERT(-0.378428),
    DSP16_Q_CONVERT(0.925049), DSP16_Q_CONVERT(-0.379847),
    DSP16_Q_CONVERT(0.924465), DSP16_Q_CONVERT(-0.381266),
    DSP16_Q_CONVERT(0.923880), DSP16_Q_CONVERT(-0.382683),
    DSP16_Q_CONVERT(0.923291), DSP16_Q_CONVERT(-0.384100),
    DSP16_Q_CONVERT(0.922701), DSP16_Q_CONVERT(-0.385516),
    DSP16_Q_CONVERT(0.922109), DSP16_Q_CONVERT(-0.386931),
    DSP16_Q_CONVERT(0.921514), DSP16_Q_CONVERT(-0.388345),
    DSP16_Q_CONVERT(0.920917), DSP16_Q_CONVERT(-0.389758),
    DSP16_Q_CONVERT(0.920318), DSP16_Q_CONVERT(-0.391170),
    DSP16_Q_CONVERT(0.919717), DSP16_Q_CONVERT(-0.392582),
    DSP16_Q_CONVERT(0.919114), DSP16_Q_CONVERT(-0.393992),
    DSP16_Q_CONVERT(0.918508), DSP16_Q_CONVERT(-0.395401),
    DSP16_Q_CONVERT(0.917901), DSP16_Q_CONVERT(-0.396810),
    DSP16_Q_CONVERT(0.917291), DSP16_Q_CONVERT(-0.398218),
    DSP16_Q_CONVERT(0.916679), DSP16_Q_CONVERT(-0.399624),
    DSP16_Q_CONVERT(0.916065), DSP16_Q_CONVERT(-0.401030),
    DSP16_Q_CONVERT(0.915449), DSP16_Q_CONVERT(-0.402435),
    DSP16_Q_CONVERT(0.914830), DSP16_Q_CONVERT(-0.403838),
    DSP16_Q_CONVERT(0.914210), DSP16_Q_CONVERT(-0.405241),
    DSP16_Q_CONVERT(0.913587), DSP16_Q_CONVERT(-0.406643),
    DSP16_Q_CONVERT(0.912962), DSP16_Q_CONVERT(-0.408044),
    DSP16_Q_CONVERT(0.912335), DSP16_Q_CONVERT(-0.409444),
    DSP16_Q_CONVERT(0.911706), DSP16_Q_CONVERT(-0.410843),
    DSP16_Q_CONVERT(0.911075), DSP16_Q_CONVERT(-0.412241),
    DSP16_Q_CONVERT(0.910441), DSP16_Q_CONVERT(-0.413638),
    DSP16_Q_CONVERT(0.909806), DSP16_Q_CONVERT(-0.415034),
    DSP16_Q_CONVERT(0.909168), DSP16_Q_CONVERT(-0.416430),
    DSP16_Q_CONVERT(0.908528), DSP16_Q_CONVERT(-0.417824),
    DSP16_Q_CONVERT(0.907886), DSP16_Q_CONVERT(-0.419217),
    DSP16_Q_CONVERT(0.907242), DSP16_Q_CONVERT(-0.420609),
    DSP16_Q_CONVERT(0.906596), DSP16_Q_CONVERT(-0.422000),
    DSP16_Q_CONVERT(0.905947), DSP16_Q_CONVERT(-0.423390),
    DSP16_Q_CONVERT(0.905297), DSP16_Q_CONVERT(-0.424780),
    DSP16_Q_CONVERT(0.904644), DSP16_Q_CONVERT(-0.426168),
    DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555),
    DSP16_Q_CONVERT(0.903332), DSP16_Q_CONVERT(-0.428941),
    DSP16_Q_CONVERT(0.902673), DSP16_Q_CONVERT(-0.430326),
    DSP16_Q_CONVERT(0.902012), DSP16_Q_CONVERT(-0.431711),
    DSP16_Q_CONVERT(0.901349), DSP16_Q_CONVERT(-0.433094),
    DSP16_Q_CONVERT(0.900683), DSP16_Q_CONVERT(-0.434476),
    DSP16_Q_CONVERT(0.900016), DSP16_Q_CONVERT(-0.435857),
    DSP16_Q_CONVERT(0.899346), DSP16_Q_CONVERT(-0.437237),
    DSP16_Q_CONVERT(0.898674), DSP16_Q_CONVERT(-0.438616),
    DSP16_Q_CONVERT(0.898001), DSP16_Q_CONVERT(-0.439994),
    DSP16_Q_CONVERT(0.897325), DSP16_Q_CONVERT(-0.441371),
    DSP16_Q_CONVERT(0.896646), DSP16_Q_CONVERT(-0.442747),
    DSP16_Q_CONVERT(0.895966), DSP16_Q_CONVERT(-0.444122),
    DSP16_Q_CONVERT(0.895284), DSP16_Q_CONVERT(-0.445496),
    DSP16_Q_CONVERT(0.894599), DSP16_Q_CONVERT(-0.446869),
    DSP16_Q_CONVERT(0.893913), DSP16_Q_CONVERT(-0.448241),
    DSP16_Q_CONVERT(0.893224), DSP16_Q_CONVERT(-0.449611),
    DSP16_Q_CONVERT(0.892534), DSP16_Q_CONVERT(-0.450981),
    DSP16_Q_CONVERT(0.891841), DSP16_Q_CONVERT(-0.452350),
    DSP16_Q_CONVERT(0.891146), DSP16_Q_CONVERT(-0.453717),
    DSP16_Q_CONVERT(0.890449), DSP16_Q_CONVERT(-0.455084),
    DSP16_Q_CONVERT(0.889750), DSP16_Q_CONVERT(-0.456449),
    DSP16_Q_CONVERT(0.889048), DSP16_Q_CONVERT(-0.457813),
    DSP16_Q_CONVERT(0.888345), DSP16_Q_CONVERT(-0.459177),
    DSP16_Q_CONVERT(0.887640), DSP16_Q_CONVERT(-0.460539),
    DSP16_Q_CONVERT(0.886932), DSP16_Q_CONVERT(-0.461900),
    DSP16_Q_CONVERT(0.886223), DSP16_Q_CONVERT(-0.463260),
    DSP16_Q_CONVERT(0.885511), DSP16_Q_CONVERT(-0.464619),
    DSP16_Q_CONVERT(0.884797), DSP16_Q_CONVERT(-0.465976),
    DSP16_Q_CONVERT(0.884081), DSP16_Q_CONVERT(-0.467333),
    DSP16_Q_CONVERT(0.883363), DSP16_Q_CONVERT(-0.468689),
    DSP16_Q_CONVERT(0.882643), DSP16_Q_CONVERT(-0.470043),
    DSP16_Q_CONVERT(0.881921), DSP16_Q_CONVERT(-0.471397),
    DSP16_Q_CONVERT(0.881197), DSP16_Q_CONVERT(-0.472749),
    DSP16_Q_CONVERT(0.880471), DSP16_Q_CONVERT(-0.474100),
    DSP16_Q_CONVERT(0.879743), DSP16_Q_CONVERT(-0.475450),
    DSP16_Q_CONVERT(0.879012), DSP16_Q_CONVERT(-0.476799),
    DSP16_Q_CONVERT(0.878280), DSP16_Q_CONVERT(-0.478147),
    DSP16_Q_CONVERT(0.877545), DSP16_Q_CONVERT(-0.479494),
    DSP16_Q_CONVERT(0.876809), DSP16_Q_CONVERT(-0.480839),
    DSP16_Q_CONVERT(0.876070), DSP16_Q_CONVERT(-0.482184),
    DSP16_Q_CONVERT(0.875329), DSP16_Q_CONVERT(-0.483527),
    DSP16_Q_CONVERT(0.874587), DSP16_Q_CONVERT(-0.484869),
    DSP16_Q_CONVERT(0.873842), DSP16_Q_CONVERT(-0.486210),
    DSP16_Q_CONVERT(0.873095), DSP16_Q_CONVERT(-0.487550),
    DSP16_Q_CONVERT(0.872346), DSP16_Q_CONVERT(-0.488889),
    DSP16_Q_CONVERT(0.871595), DSP16_Q_CONVERT(-0.490226),
    DSP16_Q_CONVERT(0.870842), DSP16_Q_CONVERT(-0.491563),
    DSP16_Q_CONVERT(0.870087), DSP16_Q_CONVERT(-0.492898),
    DSP16_Q_CONVERT(0.869330), DSP16_Q_CONVERT(-0.494232),
    DSP16_Q_CONVERT(0.868571), DSP16_Q_CONVERT(-0.495565),
    DSP16_Q_CONVERT(0.867809), DSP16_Q_CONVERT(-0.496897),
    DSP16_Q_CONVERT(0.867046), DSP16_Q_CONVERT(-0.498228),
    DSP16_Q_CONVERT(0.866281), DSP16_Q_CONVERT(-0.499557),
    DSP16_Q_CONVERT(0.865514), DSP16_Q_CONVERT(-0.500885),
    DSP16_Q_CONVERT(0.864744), DSP16_Q_CONVERT(-0.502212),
    DSP16_Q_CONVERT(0.863973), DSP16_Q_CONVERT(-0.503538),
    DSP16_Q_CONVERT(0.863199), DSP16_Q_CONVERT(-0.504863),
    DSP16_Q_CONVERT(0.862424), DSP16_Q_CONVERT(-0.506187),
    DSP16_Q_CONVERT(0.861646), DSP16_Q_CONVERT(-0.507509),
    DSP16_Q_CONVERT(0.860867), DSP16_Q_CONVERT(-0.508830),
    DSP16_Q_CONVERT(0.860085), DSP16_Q_CONVERT(-0.510150),
    DSP16_Q_CONVERT(0.859302), DSP16_Q_CONVERT(-0.511469),
    DSP16_Q_CONVERT(0.858516), DSP16_Q_CONVERT(-0.512786),
    DSP16_Q_CONVERT(0.857729), DSP16_Q_CONVERT(-0.514103),
    DSP16_Q_CONVERT(0.856939), DSP16_Q_CONVERT(-0.515418),
    DSP16_Q_CONVERT(0.856147), DSP16_Q_CONVERT(-0.516732),
    DSP16_Q_CONVERT(0.855354), DSP16_Q_CONVERT(-0.518045),
    DSP16_Q_CONVERT(0.854558), DSP16_Q_CONVERT(-0.519356),
    DSP16_Q_CONVERT(0.853760), DSP16_Q_CONVERT(-0.520666),
    DSP16_Q_CONVERT(0.852961), DSP16_Q_CONVERT(-0.521975),
    DSP16_Q_CONVERT(0.852159), DSP16_Q_CONVERT(-0.523283),
    DSP16_Q_CONVERT(0.851355), DSP16_Q_CONVERT(-0.524590),
    DSP16_Q_CONVERT(0.850549), DSP16_Q_CONVERT(-0.525895),
    DSP16_Q_CONVERT(0.849742), DSP16_Q_CONVERT(-0.527199),
    DSP16_Q_CONVERT(0.848932), DSP16_Q_CONVERT(-0.528502),
    DSP16_Q_CONVERT(0.848120), DSP16_Q_CONVERT(-0.529804),
    DSP16_Q_CONVERT(0.847307), DSP16_Q_CONVERT(-0.531104),
    DSP16_Q_CONVERT(0.846491), DSP16_Q_CONVERT(-0.532403),
    DSP16_Q_CONVERT(0.845673), DSP16_Q_CONVERT(-0.533701),
    DSP16_Q_CONVERT(0.844854), DSP16_Q_CONVERT(-0.534998),
    DSP16_Q_CONVERT(0.844032), DSP16_Q_CONVERT(-0.536293),
    DSP16_Q_CONVERT(0.843208), DSP16_Q_CONVERT(-0.537587),
    DSP16_Q_CONVERT(0.842383), DSP16_Q_CONVERT(-0.538880),
    DSP16_Q_CONVERT(0.841555), DSP16_Q_CONVERT(-0.540171),
    DSP16_Q_CONVERT(0.840725), DSP16_Q_CONVERT(-0.541462),
    DSP16_Q_CONVERT(0.839894), DSP16_Q_CONVERT(-0.542751),
    DSP16_Q_CONVERT(0.839060), DSP16_Q_CONVERT(-0.544039),
    DSP16_Q_CONVERT(0.838225), DSP16_Q_CONVERT(-0.545325),
    DSP16_Q_CONVERT(0.837387), DSP16_Q_CONVERT(-0.546610),
    DSP16_Q_CONVERT(0.836548), DSP16_Q_CONVERT(-0.547894),
    DSP16_Q_CONVERT(0.835706), DSP16_Q_CONVERT(-0.549177),
    DSP16_Q_CONVERT(0.834863), DSP16_Q_CONVERT(-0.550458),
    DSP16_Q_CONVERT(0.834018), DSP16_Q_CONVERT(-0.551738),
    DSP16_Q_CONVERT(0.833170), DSP16_Q_CONVERT(-0.553017),
    DSP16_Q_CONVERT(0.832321), DSP16_Q_CONVERT(-0.554294),
    DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570),
    DSP16_Q_CONVERT(0.830616), DSP16_Q_CONVERT(-0.556845),
    DSP16_Q_CONVERT(0.829761), DSP16_Q_CONVERT(-0.558119),
    DSP16_Q_CONVERT(0.828904), DSP16_Q_CONVERT(-0.559391),
    DSP16_Q_CONVERT(0.828045), DSP16_Q_CONVERT(-0.560662),
    DSP16_Q_CONVERT(0.827184), DSP16_Q_CONVERT(-0.561931),
    DSP16_Q_CONVERT(0.826321), DSP16_Q_CONVERT(-0.563199),
    DSP16_Q_CONVERT(0.825456), DSP16_Q_CONVERT(-0.564466),
    DSP16_Q_CONVERT(0.824589), DSP16_Q_CONVERT(-0.565732),
    DSP16_Q_CONVERT(0.823721), DSP16_Q_CONVERT(-0.566996),
    DSP16_Q_CONVERT(0.822850), DSP16_Q_CONVERT(-0.568259),
    DSP16_Q_CONVERT(0.821977), DSP16_Q_CONVERT(-0.569521),
    DSP16_Q_CONVERT(0.821103), DSP16_Q_CONVERT(-0.570781),
    DSP16_Q_CONVERT(0.820226), DSP16_Q_CONVERT(-0.572040),
    DSP16_Q_CONVERT(0.819348), DSP16_Q_CONVERT(-0.573297),
    DSP16_Q_CONVERT(0.818467), DSP16_Q_CONVERT(-0.574553),
    DSP16_Q_CONVERT(0.817585), DSP16_Q_CONVERT(-0.575808),
    DSP16_Q_CONVERT(0.816701), DSP16_Q_CONVERT(-0.577062),
    DSP16_Q_CONVERT(0.815814), DSP16_Q_CONVERT(-0.578314),
    DSP16_Q_CONVERT(0.814926), DSP16_Q_CONVERT(-0.579565),
    DSP16_Q_CONVERT(0.814036), DSP16_Q_CONVERT(-0.580814),
    DSP16_Q_CONVERT(0.813144), DSP16_Q_CONVERT(-0.582062),
    DSP16_Q_CONVERT(0.812251), DSP16_Q_CONVERT(-0.583309),
    DSP16_Q_CONVERT(0.811355), DSP16_Q_CONVERT(-0.584554),
    DSP16_Q_CONVERT(0.810457), DSP16_Q_CONVERT(-0.585798),
    DSP16_Q_CONVERT(0.809558), DSP16_Q_CONVERT(-0.587040),
    DSP16_Q_CONVERT(0.808656), DSP16_Q_CONVERT(-0.588282),
    DSP16_Q_CONVERT(0.807753), DSP16_Q_CONVERT(-0.589521),
    DSP16_Q_CONVERT(0.806848), DSP16_Q_CONVERT(-0.590760),
    DSP16_Q_CONVERT(0.805940), DSP16_Q_CONVERT(-0.591997),
    DSP16_Q_CONVERT(0.805031), DSP16_Q_CONVERT(-0.593232),
    DSP16_Q_CONVERT(0.804120), DSP16_Q_CONVERT(-0.594466),
    DSP16_Q_CONVERT(0.803208), DSP16_Q_CONVERT(-0.595699),
    DSP16_Q_CONVERT(0.802293), DSP16_Q_CONVERT(-0.596931),
    DSP16_Q_CONVERT(0.801376), DSP16_Q_CONVERT(-0.598161),
    DSP16_Q_CONVERT(0.800458), DSP16_Q_CONVERT(-0.599389),
    DSP16_Q_CONVERT(0.799537), DSP16_Q_CONVERT(-0.600616),
    DSP16_Q_CONVERT(0.798615), DSP16_Q_CONVERT(-0.601842),
    DSP16_Q_CONVERT(0.797691), DSP16_Q_CONVERT(-0.603067),
    DSP16_Q_CONVERT(0.796765), DSP16_Q_CONVERT(-0.604290),
    DSP16_Q_CONVERT(0.795837), DSP16_Q_CONVERT(-0.605511),
    DSP16_Q_CONVERT(0.794907), DSP16_Q_CONVERT(-0.606731),
    DSP16_Q_CONVERT(0.793975), DSP16_Q_CONVERT(-0.607950),
    DSP16_Q_CONVERT(0.793042), DSP16_Q_CONVERT(-0.609167),
    DSP16_Q_CONVERT(0.792107), DSP16_Q_CONVERT(-0.610383),
    DSP16_Q_CONVERT(0.791169), DSP16_Q_CONVERT(-0.611597),
    DSP16_Q_CONVERT(0.790230), DSP16_Q_CONVERT(-0.612810),
    DSP16_Q_CONVERT(0.789289), DSP16_Q_CONVERT(-0.614022),
    DSP16_Q_CONVERT(0.788346), DSP16_Q_CONVERT(-0.615232),
    DSP16_Q_CONVERT(0.787402), DSP16_Q_CONVERT(-0.616440),
    DSP16_Q_CONVERT(0.786455), DSP16_Q_CONVERT(-0.617647),
    DSP16_Q_CONVERT(0.785507), DSP16_Q_CONVERT(-0.618853),
    DSP16_Q_CONVERT(0.784557), DSP16_Q_CONVERT(-0.620057),
    DSP16_Q_CONVERT(0.783605), DSP16_Q_CONVERT(-0.621260),
    DSP16_Q_CONVERT(0.782651), DSP16_Q_CONVERT(-0.622461),
    DSP16_Q_CONVERT(0.781695), DSP16_Q_CONVERT(-0.623661),
    DSP16_Q_CONVERT(0.780737), DSP16_Q_CONVERT(-0.624859),
    DSP16_Q_CONVERT(0.779778), DSP16_Q_CONVERT(-0.626056),
    DSP16_Q_CONVERT(0.778817), DSP16_Q_CONVERT(-0.627252),
    DSP16_Q_CONVERT(0.777853), DSP16_Q_CONVERT(-0.628446),
    DSP16_Q_CONVERT(0.776888), DSP16_Q_CONVERT(-0.629638),
    DSP16_Q_CONVERT(0.775922), DSP16_Q_CONVERT(-0.630829),
    DSP16_Q_CONVERT(0.774953), DSP16_Q_CONVERT(-0.632019),
    DSP16_Q_CONVERT(0.773983), DSP16_Q_CONVERT(-0.633207),
    DSP16_Q_CONVERT(0.773010), DSP16_Q_CONVERT(-0.634393),
    DSP16_Q_CONVERT(0.772036), DSP16_Q_CONVERT(-0.635578),
    DSP16_Q_CONVERT(0.771061), DSP16_Q_CONVERT(-0.636762),
    DSP16_Q_CONVERT(0.770083), DSP16_Q_CONVERT(-0.637944),
    DSP16_Q_CONVERT(0.769103), DSP16_Q_CONVERT(-0.639124),
    DSP16_Q_CONVERT(0.768122), DSP16_Q_CONVERT(-0.640303),
    DSP16_Q_CONVERT(0.767139), DSP16_Q_CONVERT(-0.641481),
    DSP16_Q_CONVERT(0.766154), DSP16_Q_CONVERT(-0.642657),
    DSP16_Q_CONVERT(0.765167), DSP16_Q_CONVERT(-0.643832),
    DSP16_Q_CONVERT(0.764179), DSP16_Q_CONVERT(-0.645005),
    DSP16_Q_CONVERT(0.763188), DSP16_Q_CONVERT(-0.646176),
    DSP16_Q_CONVERT(0.762196), DSP16_Q_CONVERT(-0.647346),
    DSP16_Q_CONVERT(0.761202), DSP16_Q_CONVERT(-0.648514),
    DSP16_Q_CONVERT(0.760207), DSP16_Q_CONVERT(-0.649681),
    DSP16_Q_CONVERT(0.759209), DSP16_Q_CONVERT(-0.650847),
    DSP16_Q_CONVERT(0.758210), DSP16_Q_CONVERT(-0.652011),
    DSP16_Q_CONVERT(0.757209), DSP16_Q_CONVERT(-0.653173),
    DSP16_Q_CONVERT(0.756206), DSP16_Q_CONVERT(-0.654334),
    DSP16_Q_CONVERT(0.755201), DSP16_Q_CONVERT(-0.655493),
    DSP16_Q_CONVERT(0.754195), DSP16_Q_CONVERT(-0.656651),
    DSP16_Q_CONVERT(0.753187), DSP16_Q_CONVERT(-0.657807),
    DSP16_Q_CONVERT(0.752177), DSP16_Q_CONVERT(-0.658961),
    DSP16_Q_CONVERT(0.751165), DSP16_Q_CONVERT(-0.660114),
    DSP16_Q_CONVERT(0.750152), DSP16_Q_CONVERT(-0.661266),
    DSP16_Q_CONVERT(0.749136), DSP16_Q_CONVERT(-0.662416),
    DSP16_Q_CONVERT(0.748119), DSP16_Q_CONVERT(-0.663564),
    DSP16_Q_CONVERT(0.747101), DSP16_Q_CONVERT(-0.664711),
    DSP16_Q_CONVERT(0.746080), DSP16_Q_CONVERT(-0.665856),
    DSP16_Q_CONVERT(0.745058), DSP16_Q_CONVERT(-0.667000),
    DSP16_Q_CONVERT(0.744034), DSP16_Q_CONVERT(-0.668142),
    DSP16_Q_CONVERT(0.743008), DSP16_Q_CONVERT(-0.669283),
    DSP16_Q_CONVERT(0.741980), DSP16_Q_CONVERT(-0.670422),
    DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559),
    DSP16_Q_CONVERT(0.739920), DSP16_Q_CONVERT(-0.672695),
    DSP16_Q_CONVERT(0.738887), DSP16_Q_CONVERT(-0.673829),
    DSP16_Q_CONVERT(0.737853), DSP16_Q_CONVERT(-0.674962),
    DSP16_Q_CONVERT(0.736817), DSP16_Q_CONVERT(-0.676093),
    DSP16_Q_CONVERT(0.735779), DSP16_Q_CONVERT(-0.677222),
    DSP16_Q_CONVERT(0.734739), DSP16_Q_CONVERT(-0.678350),
    DSP16_Q_CONVERT(0.733697), DSP16_Q_CONVERT(-0.679476),
    DSP16_Q_CONVERT(0.732654), DSP16_Q_CONVERT(-0.680601),
    DSP16_Q_CONVERT(0.731609), DSP16_Q_CONVERT(-0.681724),
    DSP16_Q_CONVERT(0.730563), DSP16_Q_CONVERT(-0.682846),
    DSP16_Q_CONVERT(0.729514), DSP16_Q_CONVERT(-0.683965),
    DSP16_Q_CONVERT(0.728464), DSP16_Q_CONVERT(-0.685084),
    DSP16_Q_CONVERT(0.727413), DSP16_Q_CONVERT(-0.686200),
    DSP16_Q_CONVERT(0.726359), DSP16_Q_CONVERT(-0.687315),
    DSP16_Q_CONVERT(0.725304), DSP16_Q_CONVERT(-0.688429),
    DSP16_Q_CONVERT(0.724247), DSP16_Q_CONVERT(-0.689541),
    DSP16_Q_CONVERT(0.723188), DSP16_Q_CONVERT(-0.690651),
    DSP16_Q_CONVERT(0.722128), DSP16_Q_CONVERT(-0.691759),
    DSP16_Q_CONVERT(0.721066), DSP16_Q_CONVERT(-0.692866),
    DSP16_Q_CONVERT(0.720003), DSP16_Q_CONVERT(-0.693971),
    DSP16_Q_CONVERT(0.718937), DSP16_Q_CONVERT(-0.695075),
    DSP16_Q_CONVERT(0.717870), DSP16_Q_CONVERT(-0.696177),
    DSP16_Q_CONVERT(0.716801), DSP16_Q_CONVERT(-0.697278),
    DSP16_Q_CONVERT(0.715731), DSP16_Q_CONVERT(-0.698376),
    DSP16_Q_CONVERT(0.714659), DSP16_Q_CONVERT(-0.699473),
    DSP16_Q_CONVERT(0.713585), DSP16_Q_CONVERT(-0.700569),
    DSP16_Q_CONVERT(0.712509), DSP16_Q_CONVERT(-0.701663),
    DSP16_Q_CONVERT(0.711432), DSP16_Q_CONVERT(-0.702755),
    DSP16_Q_CONVERT(0.710353), DSP16_Q_CONVERT(-0.703845),
    DSP16_Q_CONVERT(0.709273), DSP16_Q_CONVERT(-0.704934),
    DSP16_Q_CONVERT(0.708191), DSP16_Q_CONVERT(-0.706021),
    DSP16_Q_CONVERT(0.707107), DSP16_Q_CONVERT(-0.707107),
    DSP16_Q_CONVERT(0.706021), DSP16_Q_CONVERT(-0.708191),
    DSP16_Q_CONVERT(0.704934), DSP16_Q_CONVERT(-0.709273),
    DSP16_Q_CONVERT(0.703845), DSP16_Q_CONVERT(-0.710353),
    DSP16_Q_CONVERT(0.702755), DSP16_Q_CONVERT(-0.711432),
    DSP16_Q_CONVERT(0.701663), DSP16_Q_CONVERT(-0.712509),
    DSP16_Q_CONVERT(0.700569), DSP16_Q_CONVERT(-0.713585),
    DSP16_Q_CONVERT(0.699473), DSP16_Q_CONVERT(-0.714659),
    DSP16_Q_CONVERT(0.698376), DSP16_Q_CONVERT(-0.715731),
    DSP16_Q_CONVERT(0.697278), DSP16_Q_CONVERT(-0.716801),
    DSP16_Q_CONVERT(0.696177), DSP16_Q_CONVERT(-0.717870),
    DSP16_Q_CONVERT(0.695075), DSP16_Q_CONVERT(-0.718937),
    DSP16_Q_CONVERT(0.693971), DSP16_Q_CONVERT(-0.720003),
    DSP16_Q_CONVERT(0.692866), DSP16_Q_CONVERT(-0.721066),
    DSP16_Q_CONVERT(0.691759), DSP16_Q_CONVERT(-0.722128),
    DSP16_Q_CONVERT(0.690651), DSP16_Q_CONVERT(-0.723188),
    DSP16_Q_CONVERT(0.689541), DSP16_Q_CONVERT(-0.724247),
    DSP16_Q_CONVERT(0.688429), DSP16_Q_CONVERT(-0.725304),
    DSP16_Q_CONVERT(0.687315), DSP16_Q_CONVERT(-0.726359),
    DSP16_Q_CONVERT(0.686200), DSP16_Q_CONVERT(-0.727413),
    DSP16_Q_CONVERT(0.685084), DSP16_Q_CONVERT(-0.728464),
    DSP16_Q_CONVERT(0.683965), DSP16_Q_CONVERT(-0.729514),
    DSP16_Q_CONVERT(0.682846), DSP16_Q_CONVERT(-0.730563),
    DSP16_Q_CONVERT(0.681724), DSP16_Q_CONVERT(-0.731609),
    DSP16_Q_CONVERT(0.680601), DSP16_Q_CONVERT(-0.732654),
    DSP16_Q_CONVERT(0.679476), DSP16_Q_CONVERT(-0.733697),
    DSP16_Q_CONVERT(0.678350), DSP16_Q_CONVERT(-0.734739),
    DSP16_Q_CONVERT(0.677222), DSP16_Q_CONVERT(-0.735779),
    DSP16_Q_CONVERT(0.676093), DSP16_Q_CONVERT(-0.736817),
    DSP16_Q_CONVERT(0.674962), DSP16_Q_CONVERT(-0.737853),
    DSP16_Q_CONVERT(0.673829), DSP16_Q_CONVERT(-0.738887),
    DSP16_Q_CONVERT(0.672695), DSP16_Q_CONVERT(-0.739920),
    DSP16_Q_CONVERT(0.671559), DSP16_Q_CONVERT(-0.740951),
    DSP16_Q_CONVERT(0.670422), DSP16_Q_CONVERT(-0.741980),
    DSP16_Q_CONVERT(0.669283), DSP16_Q_CONVERT(-0.743008),
    DSP16_Q_CONVERT(0.668142), DSP16_Q_CONVERT(-0.744034),
    DSP16_Q_CONVERT(0.667000), DSP16_Q_CONVERT(-0.745058),
    DSP16_Q_CONVERT(0.665856), DSP16_Q_CONVERT(-0.746080),
    DSP16_Q_CONVERT(0.664711), DSP16_Q_CONVERT(-0.747101),
    DSP16_Q_CONVERT(0.663564), DSP16_Q_CONVERT(-0.748119),
    DSP16_Q_CONVERT(0.662416), DSP16_Q_CONVERT(-0.749136),
    DSP16_Q_CONVERT(0.661266), DSP16_Q_CONVERT(-0.750152),
    DSP16_Q_CONVERT(0.660114), DSP16_Q_CONVERT(-0.751165),
    DSP16_Q_CONVERT(0.658961), DSP16_Q_CONVERT(-0.752177),
    DSP16_Q_CONVERT(0.657807), DSP16_Q_CONVERT(-0.753187),
    DSP16_Q_CONVERT(0.656651), DSP16_Q_CONVERT(-0.754195),
    DSP16_Q_CONVERT(0.655493), DSP16_Q_CONVERT(-0.755201),
    DSP16_Q_CONVERT(0.654334), DSP16_Q_CONVERT(-0.756206),
    DSP16_Q_CONVERT(0.653173), DSP16_Q_CONVERT(-0.757209),
    DSP16_Q_CONVERT(0.652011), DSP16_Q_CONVERT(-0.758210),
    DSP16_Q_CONVERT(0.650847), DSP16_Q_CONVERT(-0.759209),
    DSP16_Q_CONVERT(0.649681), DSP16_Q_CONVERT(-0.760207),
    DSP16_Q_CONVERT(0.648514), DSP16_Q_CONVERT(-0.761202),
    DSP16_Q_CONVERT(0.647346), DSP16_Q_CONVERT(-0.762196),
    DSP16_Q_CONVERT(0.646176), DSP16_Q_CONVERT(-0.763188),
    DSP16_Q_CONVERT(0.645005), DSP16_Q_CONVERT(-0.764179),
    DSP16_Q_CONVERT(0.643832), DSP16_Q_CONVERT(-0.765167),
    DSP16_Q_CONVERT(0.642657), DSP16_Q_CONVERT(-0.766154),
    DSP16_Q_CONVERT(0.641481), DSP16_Q_CONVERT(-0.767139),
    DSP16_Q_CONVERT(0.640303), DSP16_Q_CONVERT(-0.768122),
    DSP16_Q_CONVERT(0.639124), DSP16_Q_CONVERT(-0.769103),
    DSP16_Q_CONVERT(0.637944), DSP16_Q_CONVERT(-0.770083),
    DSP16_Q_CONVERT(0.636762), DSP16_Q_CONVERT(-0.771061),
    DSP16_Q_CONVERT(0.635578), DSP16_Q_CONVERT(-0.772036),
    DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010),
    DSP16_Q_CONVERT(0.633207), DSP16_Q_CONVERT(-0.773983),
    DSP16_Q_CONVERT(0.632019), DSP16_Q_CONVERT(-0.774953),
    DSP16_Q_CONVERT(0.630829), DSP16_Q_CONVERT(-0.775922),
    DSP16_Q_CONVERT(0.629638), DSP16_Q_CONVERT(-0.776888),
    DSP16_Q_CONVERT(0.628446), DSP16_Q_CONVERT(-0.777853),
    DSP16_Q_CONVERT(0.627252), DSP16_Q_CONVERT(-0.778817),
    DSP16_Q_CONVERT(0.626056), DSP16_Q_CONVERT(-0.779778),
    DSP16_Q_CONVERT(0.624859), DSP16_Q_CONVERT(-0.780737),
    DSP16_Q_CONVERT(0.623661), DSP16_Q_CONVERT(-0.781695),
    DSP16_Q_CONVERT(0.622461), DSP16_Q_CONVERT(-0.782651),
    DSP16_Q_CONVERT(0.621260), DSP16_Q_CONVERT(-0.783605),
    DSP16_Q_CONVERT(0.620057), DSP16_Q_CONVERT(-0.784557),
    DSP16_Q_CONVERT(0.618853), DSP16_Q_CONVERT(-0.785507),
    DSP16_Q_CONVERT(0.617647), DSP16_Q_CONVERT(-0.786455),
    DSP16_Q_CONVERT(0.616440), DSP16_Q_CONVERT(-0.787402),
    DSP16_Q_CONVERT(0.615232), DSP16_Q_CONVERT(-0.788346),
    DSP16_Q_CONVERT(0.614022), DSP16_Q_CONVERT(-0.789289),
    DSP16_Q_CONVERT(0.612810), DSP16_Q_CONVERT(-0.790230),
    DSP16_Q_CONVERT(0.611597), DSP16_Q_CONVERT(-0.791169),
    DSP16_Q_CONVERT(0.610383), DSP16_Q_CONVERT(-0.792107),
    DSP16_Q_CONVERT(0.609167), DSP16_Q_CONVERT(-0.793042),
    DSP16_Q_CONVERT(0.607950), DSP16_Q_CONVERT(-0.793975),
    DSP16_Q_CONVERT(0.606731), DSP16_Q_CONVERT(-0.794907),
    DSP16_Q_CONVERT(0.605511), DSP16_Q_CONVERT(-0.795837),
    DSP16_Q_CONVERT(0.604290), DSP16_Q_CONVERT(-0.796765),
    DSP16_Q_CONVERT(0.603067), DSP16_Q_CONVERT(-0.797691),
    DSP16_Q_CONVERT(0.601842), DSP16_Q_CONVERT(-0.798615),
    DSP16_Q_CONVERT(0.600616), DSP16_Q_CONVERT(-0.799537),
    DSP16_Q_CONVERT(0.599389), DSP16_Q_CONVERT(-0.800458),
    DSP16_Q_CONVERT(0.598161), DSP16_Q_CONVERT(-0.801376),
    DSP16_Q_CONVERT(0.596931), DSP16_Q_CONVERT(-0.802293),
    DSP16_Q_CONVERT(0.595699), DSP16_Q_CONVERT(-0.803208),
    DSP16_Q_CONVERT(0.594466), DSP16_Q_CONVERT(-0.804120),
    DSP16_Q_CONVERT(0.593232), DSP16_Q_CONVERT(-0.805031),
    DSP16_Q_CONVERT(0.591997), DSP16_Q_CONVERT(-0.805940),
    DSP16_Q_CONVERT(0.590760), DSP16_Q_CONVERT(-0.806848),
    DSP16_Q_CONVERT(0.589521), DSP16_Q_CONVERT(-0.807753),
    DSP16_Q_CONVERT(0.588282), DSP16_Q_CONVERT(-0.808656),
    DSP16_Q_CONVERT(0.587040), DSP16_Q_CONVERT(-0.809558),
    DSP16_Q_CONVERT(0.585798), DSP16_Q_CONVERT(-0.810457),
    DSP16_Q_CONVERT(0.584554), DSP16_Q_CONVERT(-0.811355),
    DSP16_Q_CONVERT(0.583309), DSP16_Q_CONVERT(-0.812251),
    DSP16_Q_CONVERT(0.582062), DSP16_Q_CONVERT(-0.813144),
    DSP16_Q_CONVERT(0.580814), DSP16_Q_CONVERT(-0.814036),
    DSP16_Q_CONVERT(0.579565), DSP16_Q_CONVERT(-0.814926),
    DSP16_Q_CONVERT(0.578314), DSP16_Q_CONVERT(-0.815814),
    DSP16_Q_CONVERT(0.577062), DSP16_Q_CONVERT(-0.816701),
    DSP16_Q_CONVERT(0.575808), DSP16_Q_CONVERT(-0.817585),
    DSP16_Q_CONVERT(0.574553), DSP16_Q_CONVERT(-0.818467),
    DSP16_Q_CONVERT(0.573297), DSP16_Q_CONVERT(-0.819348),
    DSP16_Q_CONVERT(0.572040), DSP16_Q_CONVERT(-0.820226),
    DSP16_Q_CONVERT(0.570781), DSP16_Q_CONVERT(-0.821103),
    DSP16_Q_CONVERT(0.569521), DSP16_Q_CONVERT(-0.821977),
    DSP16_Q_CONVERT(0.568259), DSP16_Q_CONVERT(-0.822850),
    DSP16_Q_CONVERT(0.566996), DSP16_Q_CONVERT(-0.823721),
    DSP16_Q_CONVERT(0.565732), DSP16_Q_CONVERT(-0.824589),
    DSP16_Q_CONVERT(0.564466), DSP16_Q_CONVERT(-0.825456),
    DSP16_Q_CONVERT(0.563199), DSP16_Q_CONVERT(-0.826321),
    DSP16_Q_CONVERT(0.561931), DSP16_Q_CONVERT(-0.827184),
    DSP16_Q_CONVERT(0.560662), DSP16_Q_CONVERT(-0.828045),
    DSP16_Q_CONVERT(0.559391), DSP16_Q_CONVERT(-0.828904),
    DSP16_Q_CONVERT(0.558119), DSP16_Q_CONVERT(-0.829761),
    DSP16_Q_CONVERT(0.556845), DSP16_Q_CONVERT(-0.830616),
    DSP16_Q_CONVERT(0.555570), DSP16_Q_CONVERT(-0.831470),
    DSP16_Q_CONVERT(0.554294), DSP16_Q_CONVERT(-0.832321),
    DSP16_Q_CONVERT(0.553017), DSP16_Q_CONVERT(-0.833170),
    DSP16_Q_CONVERT(0.551738), DSP16_Q_CONVERT(-0.834018),
    DSP16_Q_CONVERT(0.550458), DSP16_Q_CONVERT(-0.834863),
    DSP16_Q_CONVERT(0.549177), DSP16_Q_CONVERT(-0.835706),
    DSP16_Q_CONVERT(0.547894), DSP16_Q_CONVERT(-0.836548),
    DSP16_Q_CONVERT(0.546610), DSP16_Q_CONVERT(-0.837387),
    DSP16_Q_CONVERT(0.545325), DSP16_Q_CONVERT(-0.838225),
    DSP16_Q_CONVERT(0.544039), DSP16_Q_CONVERT(-0.839060),
    DSP16_Q_CONVERT(0.542751), DSP16_Q_CONVERT(-0.839894),
    DSP16_Q_CONVERT(0.541462), DSP16_Q_CONVERT(-0.840725),
    DSP16_Q_CONVERT(0.540171), DSP16_Q_CONVERT(-0.841555),
    DSP16_Q_CONVERT(0.538880), DSP16_Q_CONVERT(-0.842383),
    DSP16_Q_CONVERT(0.537587), DSP16_Q_CONVERT(-0.843208),
    DSP16_Q_CONVERT(0.536293), DSP16_Q_CONVERT(-0.844032),
    DSP16_Q_CONVERT(0.534998), DSP16_Q_CONVERT(-0.844854),
    DSP16_Q_CONVERT(0.533701), DSP16_Q_CONVERT(-0.845673),
    DSP16_Q_CONVERT(0.532403), DSP16_Q_CONVERT(-0.846491),
    DSP16_Q_CONVERT(0.531104), DSP16_Q_CONVERT(-0.847307),
    DSP16_Q_CONVERT(0.529804), DSP16_Q_CONVERT(-0.848120),
    DSP16_Q_CONVERT(0.528502), DSP16_Q_CONVERT(-0.848932),
    DSP16_Q_CONVERT(0.527199), DSP16_Q_CONVERT(-0.849742),
    DSP16_Q_CONVERT(0.525895), DSP16_Q_CONVERT(-0.850549),
    DSP16_Q_CONVERT(0.524590), DSP16_Q_CONVERT(-0.851355),
    DSP16_Q_CONVERT(0.523283), DSP16_Q_CONVERT(-0.852159),
    DSP16_Q_CONVERT(0.521975), DSP16_Q_CONVERT(-0.852961),
    DSP16_Q_CONVERT(0.520666), DSP16_Q_CONVERT(-0.853760),
    DSP16_Q_CONVERT(0.519356), DSP16_Q_CONVERT(-0.854558),
    DSP16_Q_CONVERT(0.518045), DSP16_Q_CONVERT(-0.855354),
    DSP16_Q_CONVERT(0.516732), DSP16_Q_CONVERT(-0.856147),
    DSP16_Q_CONVERT(0.515418), DSP16_Q_CONVERT(-0.856939),
    DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729),
    DSP16_Q_CONVERT(0.512786), DSP16_Q_CONVERT(-0.858516),
    DSP16_Q_CONVERT(0.511469), DSP16_Q_CONVERT(-0.859302),
    DSP16_Q_CONVERT(0.510150), DSP16_Q_CONVERT(-0.860085),
    DSP16_Q_CONVERT(0.508830), DSP16_Q_CONVERT(-0.860867),
    DSP16_Q_CONVERT(0.507509), DSP16_Q_CONVERT(-0.861646),
    DSP16_Q_CONVERT(0.506187), DSP16_Q_CONVERT(-0.862424),
    DSP16_Q_CONVERT(0.504863), DSP16_Q_CONVERT(-0.863199),
    DSP16_Q_CONVERT(0.503538), DSP16_Q_CONVERT(-0.863973),
    DSP16_Q_CONVERT(0.502212), DSP16_Q_CONVERT(-0.864744),
    DSP16_Q_CONVERT(0.500885), DSP16_Q_CONVERT(-0.865514),
    DSP16_Q_CONVERT(0.499557), DSP16_Q_CONVERT(-0.866281),
    DSP16_Q_CONVERT(0.498228), DSP16_Q_CONVERT(-0.867046),
    DSP16_Q_CONVERT(0.496897), DSP16_Q_CONVERT(-0.867809),
    DSP16_Q_CONVERT(0.495565), DSP16_Q_CONVERT(-0.868571),
    DSP16_Q_CONVERT(0.494232), DSP16_Q_CONVERT(-0.869330),
    DSP16_Q_CONVERT(0.492898), DSP16_Q_CONVERT(-0.870087),
    DSP16_Q_CONVERT(0.491563), DSP16_Q_CONVERT(-0.870842),
    DSP16_Q_CONVERT(0.490226), DSP16_Q_CONVERT(-0.871595),
    DSP16_Q_CONVERT(0.488889), DSP16_Q_CONVERT(-0.872346),
    DSP16_Q_CONVERT(0.487550), DSP16_Q_CONVERT(-0.873095),
    DSP16_Q_CONVERT(0.486210), DSP16_Q_CONVERT(-0.873842),
    DSP16_Q_CONVERT(0.484869), DSP16_Q_CONVERT(-0.874587),
    DSP16_Q_CONVERT(0.483527), DSP16_Q_CONVERT(-0.875329),
    DSP16_Q_CONVERT(0.482184), DSP16_Q_CONVERT(-0.876070),
    DSP16_Q_CONVERT(0.480839), DSP16_Q_CONVERT(-0.876809),
    DSP16_Q_CONVERT(0.479494), DSP16_Q_CONVERT(-0.877545),
    DSP16_Q_CONVERT(0.478147), DSP16_Q_CONVERT(-0.878280),
    DSP16_Q_CONVERT(0.476799), DSP16_Q_CONVERT(-0.879012),
    DSP16_Q_CONVERT(0.475450), DSP16_Q_CONVERT(-0.879743),
    DSP16_Q_CONVERT(0.474100), DSP16_Q_CONVERT(-0.880471),
    DSP16_Q_CONVERT(0.472749), DSP16_Q_CONVERT(-0.881197),
    DSP16_Q_CONVERT(0.471397), DSP16_Q_CONVERT(-0.881921),
    DSP16_Q_CONVERT(0.470043), DSP16_Q_CONVERT(-0.882643),
    DSP16_Q_CONVERT(0.468689), DSP16_Q_CONVERT(-0.883363),
    DSP16_Q_CONVERT(0.467333), DSP16_Q_CONVERT(-0.884081),
    DSP16_Q_CONVERT(0.465976), DSP16_Q_CONVERT(-0.884797),
    DSP16_Q_CONVERT(0.464619), DSP16_Q_CONVERT(-0.885511),
    DSP16_Q_CONVERT(0.463260), DSP16_Q_CONVERT(-0.886223),
    DSP16_Q_CONVERT(0.461900), DSP16_Q_CONVERT(-0.886932),
    DSP16_Q_CONVERT(0.460539), DSP16_Q_CONVERT(-0.887640),
    DSP16_Q_CONVERT(0.459177), DSP16_Q_CONVERT(-0.888345),
    DSP16_Q_CONVERT(0.457813), DSP16_Q_CONVERT(-0.889048),
    DSP16_Q_CONVERT(0.456449), DSP16_Q_CONVERT(-0.889750),
    DSP16_Q_CONVERT(0.455084), DSP16_Q_CONVERT(-0.890449),
    DSP16_Q_CONVERT(0.453717), DSP16_Q_CONVERT(-0.891146),
    DSP16_Q_CONVERT(0.452350), DSP16_Q_CONVERT(-0.891841),
    DSP16_Q_CONVERT(0.450981), DSP16_Q_CONVERT(-0.892534),
    DSP16_Q_CONVERT(0.449611), DSP16_Q_CONVERT(-0.893224),
    DSP16_Q_CONVERT(0.448241), DSP16_Q_CONVERT(-0.893913),
    DSP16_Q_CONVERT(0.446869), DSP16_Q_CONVERT(-0.894599),
    DSP16_Q_CONVERT(0.445496), DSP16_Q_CONVERT(-0.895284),
    DSP16_Q_CONVERT(0.444122), DSP16_Q_CONVERT(-0.895966),
    DSP16_Q_CONVERT(0.442747), DSP16_Q_CONVERT(-0.896646),
    DSP16_Q_CONVERT(0.441371), DSP16_Q_CONVERT(-0.897325),
    DSP16_Q_CONVERT(0.439994), DSP16_Q_CONVERT(-0.898001),
    DSP16_Q_CONVERT(0.438616), DSP16_Q_CONVERT(-0.898674),
    DSP16_Q_CONVERT(0.437237), DSP16_Q_CONVERT(-0.899346),
    DSP16_Q_CONVERT(0.435857), DSP16_Q_CONVERT(-0.900016),
    DSP16_Q_CONVERT(0.434476), DSP16_Q_CONVERT(-0.900683),
    DSP16_Q_CONVERT(0.433094), DSP16_Q_CONVERT(-0.901349),
    DSP16_Q_CONVERT(0.431711), DSP16_Q_CONVERT(-0.902012),
    DSP16_Q_CONVERT(0.430326), DSP16_Q_CONVERT(-0.902673),
    DSP16_Q_CONVERT(0.428941), DSP16_Q_CONVERT(-0.903332),
    DSP16_Q_CONVERT(0.427555), DSP16_Q_CONVERT(-0.903989),
    DSP16_Q_CONVERT(0.426168), DSP16_Q_CONVERT(-0.904644),
    DSP16_Q_CONVERT(0.424780), DSP16_Q_CONVERT(-0.905297),
    DSP16_Q_CONVERT(0.423390), DSP16_Q_CONVERT(-0.905947),
    DSP16_Q_CONVERT(0.422000), DSP16_Q_CONVERT(-0.906596),
    DSP16_Q_CONVERT(0.420609), DSP16_Q_CONVERT(-0.907242),
    DSP16_Q_CONVERT(0.419217), DSP16_Q_CONVERT(-0.907886),
    DSP16_Q_CONVERT(0.417824), DSP16_Q_CONVERT(-0.908528),
    DSP16_Q_CONVERT(0.416430), DSP16_Q_CONVERT(-0.909168),
    DSP16_Q_CONVERT(0.415034), DSP16_Q_CONVERT(-0.909806),
    DSP16_Q_CONVERT(0.413638), DSP16_Q_CONVERT(-0.910441),
    DSP16_Q_CONVERT(0.412241), DSP16_Q_CONVERT(-0.911075),
    DSP16_Q_CONVERT(0.410843), DSP16_Q_CONVERT(-0.911706),
    DSP16_Q_CONVERT(0.409444), DSP16_Q_CONVERT(-0.912335),
    DSP16_Q_CONVERT(0.408044), DSP16_Q_CONVERT(-0.912962),
    DSP16_Q_CONVERT(0.406643), DSP16_Q_CONVERT(-0.913587),
    DSP16_Q_CONVERT(0.405241), DSP16_Q_CONVERT(-0.914210),
    DSP16_Q_CONVERT(0.403838), DSP16_Q_CONVERT(-0.914830),
    DSP16_Q_CONVERT(0.402435), DSP16_Q_CONVERT(-0.915449),
    DSP16_Q_CONVERT(0.401030), DSP16_Q_CONVERT(-0.916065),
    DSP16_Q_CONVERT(0.399624), DSP16_Q_CONVERT(-0.916679),
    DSP16_Q_CONVERT(0.398218), DSP16_Q_CONVERT(-0.917291),
    DSP16_Q_CONVERT(0.396810), DSP16_Q_CONVERT(-0.917901),
    DSP16_Q_CONVERT(0.395401), DSP16_Q_CONVERT(-0.918508),
    DSP16_Q_CONVERT(0.393992), DSP16_Q_CONVERT(-0.919114),
    DSP16_Q_CONVERT(0.392582), DSP16_Q_CONVERT(-0.919717),
    DSP16_Q_CONVERT(0.391170), DSP16_Q_CONVERT(-0.920318),
    DSP16_Q_CONVERT(0.389758), DSP16_Q_CONVERT(-0.920917),
    DSP16_Q_CONVERT(0.388345), DSP16_Q_CONVERT(-0.921514),
    DSP16_Q_CONVERT(0.386931), DSP16_Q_CONVERT(-0.922109),
    DSP16_Q_CONVERT(0.385516), DSP16_Q_CONVERT(-0.922701),
    DSP16_Q_CONVERT(0.384100), DSP16_Q_CONVERT(-0.923291),
    DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880),
    DSP16_Q_CONVERT(0.381266), DSP16_Q_CONVERT(-0.924465),
    DSP16_Q_CONVERT(0.379847), DSP16_Q_CONVERT(-0.925049),
    DSP16_Q_CONVERT(0.378428), DSP16_Q_CONVERT(-0.925631),
    DSP16_Q_CONVERT(0.377007), DSP16_Q_CONVERT(-0.926210),
    DSP16_Q_CONVERT(0.375586), DSP16_Q_CONVERT(-0.926787),
    DSP16_Q_CONVERT(0.374164), DSP16_Q_CONVERT(-0.927363),
    DSP16_Q_CONVERT(0.372741), DSP16_Q_CONVERT(-0.927935),
    DSP16_Q_CONVERT(0.371317), DSP16_Q_CONVERT(-0.928506),
    DSP16_Q_CONVERT(0.369892), DSP16_Q_CONVERT(-0.929075),
    DSP16_Q_CONVERT(0.368467), DSP16_Q_CONVERT(-0.929641),
    DSP16_Q_CONVERT(0.367040), DSP16_Q_CONVERT(-0.930205),
    DSP16_Q_CONVERT(0.365613), DSP16_Q_CONVERT(-0.930767),
    DSP16_Q_CONVERT(0.364185), DSP16_Q_CONVERT(-0.931327),
    DSP16_Q_CONVERT(0.362756), DSP16_Q_CONVERT(-0.931884),
    DSP16_Q_CONVERT(0.361326), DSP16_Q_CONVERT(-0.932440),
    DSP16_Q_CONVERT(0.359895), DSP16_Q_CONVERT(-0.932993),
    DSP16_Q_CONVERT(0.358463), DSP16_Q_CONVERT(-0.933544),
    DSP16_Q_CONVERT(0.357031), DSP16_Q_CONVERT(-0.934093),
    DSP16_Q_CONVERT(0.355598), DSP16_Q_CONVERT(-0.934639),
    DSP16_Q_CONVERT(0.354164), DSP16_Q_CONVERT(-0.935184),
    DSP16_Q_CONVERT(0.352729), DSP16_Q_CONVERT(-0.935726),
    DSP16_Q_CONVERT(0.351293), DSP16_Q_CONVERT(-0.936266),
    DSP16_Q_CONVERT(0.349856), DSP16_Q_CONVERT(-0.936803),
    DSP16_Q_CONVERT(0.348419), DSP16_Q_CONVERT(-0.937339),
    DSP16_Q_CONVERT(0.346980), DSP16_Q_CONVERT(-0.937872),
    DSP16_Q_CONVERT(0.345541), DSP16_Q_CONVERT(-0.938404),
    DSP16_Q_CONVERT(0.344101), DSP16_Q_CONVERT(-0.938932),
    DSP16_Q_CONVERT(0.342661), DSP16_Q_CONVERT(-0.939459),
    DSP16_Q_CONVERT(0.341219), DSP16_Q_CONVERT(-0.939984),
    DSP16_Q_CONVERT(0.339777), DSP16_Q_CONVERT(-0.940506),
    DSP16_Q_CONVERT(0.338334), DSP16_Q_CONVERT(-0.941026),
    DSP16_Q_CONVERT(0.336890), DSP16_Q_CONVERT(-0.941544),
    DSP16_Q_CONVERT(0.335445), DSP16_Q_CONVERT(-0.942060),
    DSP16_Q_CONVERT(0.334000), DSP16_Q_CONVERT(-0.942573),
    DSP16_Q_CONVERT(0.332553), DSP16_Q_CONVERT(-0.943084),
    DSP16_Q_CONVERT(0.331106), DSP16_Q_CONVERT(-0.943593),
    DSP16_Q_CONVERT(0.329658), DSP16_Q_CONVERT(-0.944100),
    DSP16_Q_CONVERT(0.328210), DSP16_Q_CONVERT(-0.944605),
    DSP16_Q_CONVERT(0.326760), DSP16_Q_CONVERT(-0.945107),
    DSP16_Q_CONVERT(0.325310), DSP16_Q_CONVERT(-0.945607),
    DSP16_Q_CONVERT(0.323859), DSP16_Q_CONVERT(-0.946105),
    DSP16_Q_CONVERT(0.322408), DSP16_Q_CONVERT(-0.946601),
    DSP16_Q_CONVERT(0.320955), DSP16_Q_CONVERT(-0.947094),
    DSP16_Q_CONVERT(0.319502), DSP16_Q_CONVERT(-0.947586),
    DSP16_Q_CONVERT(0.318048), DSP16_Q_CONVERT(-0.948075),
    DSP16_Q_CONVERT(0.316593), DSP16_Q_CONVERT(-0.948561),
    DSP16_Q_CONVERT(0.315138), DSP16_Q_CONVERT(-0.949046),
    DSP16_Q_CONVERT(0.313682), DSP16_Q_CONVERT(-0.949528),
    DSP16_Q_CONVERT(0.312225), DSP16_Q_CONVERT(-0.950008),
    DSP16_Q_CONVERT(0.310767), DSP16_Q_CONVERT(-0.950486),
    DSP16_Q_CONVERT(0.309309), DSP16_Q_CONVERT(-0.950962),
    DSP16_Q_CONVERT(0.307850), DSP16_Q_CONVERT(-0.951435),
    DSP16_Q_CONVERT(0.306390), DSP16_Q_CONVERT(-0.951906),
    DSP16_Q_CONVERT(0.304929), DSP16_Q_CONVERT(-0.952375),
    DSP16_Q_CONVERT(0.303468), DSP16_Q_CONVERT(-0.952842),
    DSP16_Q_CONVERT(0.302006), DSP16_Q_CONVERT(-0.953306),
    DSP16_Q_CONVERT(0.300543), DSP16_Q_CONVERT(-0.953768),
    DSP16_Q_CONVERT(0.299080), DSP16_Q_CONVERT(-0.954228),
    DSP16_Q_CONVERT(0.297616), DSP16_Q_CONVERT(-0.954686),
    DSP16_Q_CONVERT(0.296151), DSP16_Q_CONVERT(-0.955141),
    DSP16_Q_CONVERT(0.294685), DSP16_Q_CONVERT(-0.955594),
    DSP16_Q_CONVERT(0.293219), DSP16_Q_CONVERT(-0.956045),
    DSP16_Q_CONVERT(0.291752), DSP16_Q_CONVERT(-0.956494),
    DSP16_Q_CONVERT(0.290285), DSP16_Q_CONVERT(-0.956940),
    DSP16_Q_CONVERT(0.288816), DSP16_Q_CONVERT(-0.957385),
    DSP16_Q_CONVERT(0.287347), DSP16_Q_CONVERT(-0.957826),
    DSP16_Q_CONVERT(0.285878), DSP16_Q_CONVERT(-0.958266),
    DSP16_Q_CONVERT(0.284408), DSP16_Q_CONVERT(-0.958703),
    DSP16_Q_CONVERT(0.282937), DSP16_Q_CONVERT(-0.959139),
    DSP16_Q_CONVERT(0.281465), DSP16_Q_CONVERT(-0.959572),
    DSP16_Q_CONVERT(0.279993), DSP16_Q_CONVERT(-0.960002),
    DSP16_Q_CONVERT(0.278520), DSP16_Q_CONVERT(-0.960431),
    DSP16_Q_CONVERT(0.277046), DSP16_Q_CONVERT(-0.960857),
    DSP16_Q_CONVERT(0.275572), DSP16_Q_CONVERT(-0.961280),
    DSP16_Q_CONVERT(0.274097), DSP16_Q_CONVERT(-0.961702),
    DSP16_Q_CONVERT(0.272621), DSP16_Q_CONVERT(-0.962121),
    DSP16_Q_CONVERT(0.271145), DSP16_Q_CONVERT(-0.962538),
    DSP16_Q_CONVERT(0.269668), DSP16_Q_CONVERT(-0.962953),
    DSP16_Q_CONVERT(0.268191), DSP16_Q_CONVERT(-0.963366),
    DSP16_Q_CONVERT(0.266713), DSP16_Q_CONVERT(-0.963776),
    DSP16_Q_CONVERT(0.265234), DSP16_Q_CONVERT(-0.964184),
    DSP16_Q_CONVERT(0.263755), DSP16_Q_CONVERT(-0.964590),
    DSP16_Q_CONVERT(0.262275), DSP16_Q_CONVERT(-0.964993),
    DSP16_Q_CONVERT(0.260794), DSP16_Q_CONVERT(-0.965394),
    DSP16_Q_CONVERT(0.259313), DSP16_Q_CONVERT(-0.965793),
    DSP16_Q_CONVERT(0.257831), DSP16_Q_CONVERT(-0.966190),
    DSP16_Q_CONVERT(0.256349), DSP16_Q_CONVERT(-0.966584),
    DSP16_Q_CONVERT(0.254866), DSP16_Q_CONVERT(-0.966976),
    DSP16_Q_CONVERT(0.253382), DSP16_Q_CONVERT(-0.967366),
    DSP16_Q_CONVERT(0.251898), DSP16_Q_CONVERT(-0.967754),
    DSP16_Q_CONVERT(0.250413), DSP16_Q_CONVERT(-0.968139),
    DSP16_Q_CONVERT(0.248928), DSP16_Q_CONVERT(-0.968522),
    DSP16_Q_CONVERT(0.247442), DSP16_Q_CONVERT(-0.968903),
    DSP16_Q_CONVERT(0.245955), DSP16_Q_CONVERT(-0.969281),
    DSP16_Q_CONVERT(0.244468), DSP16_Q_CONVERT(-0.969657),
    DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031),
    DSP16_Q_CONVERT(0.241492), DSP16_Q_CONVERT(-0.970403),
    DSP16_Q_CONVERT(0.240003), DSP16_Q_CONVERT(-0.970772),
    DSP16_Q_CONVERT(0.238514), DSP16_Q_CONVERT(-0.971139),
    DSP16_Q_CONVERT(0.237024), DSP16_Q_CONVERT(-0.971504),
    DSP16_Q_CONVERT(0.235533), DSP16_Q_CONVERT(-0.971866),
    DSP16_Q_CONVERT(0.234042), DSP16_Q_CONVERT(-0.972226),
    DSP16_Q_CONVERT(0.232550), DSP16_Q_CONVERT(-0.972584),
    DSP16_Q_CONVERT(0.231058), DSP16_Q_CONVERT(-0.972940),
    DSP16_Q_CONVERT(0.229565), DSP16_Q_CONVERT(-0.973293),
    DSP16_Q_CONVERT(0.228072), DSP16_Q_CONVERT(-0.973644),
    DSP16_Q_CONVERT(0.226578), DSP16_Q_CONVERT(-0.973993),
    DSP16_Q_CONVERT(0.225084), DSP16_Q_CONVERT(-0.974339),
    DSP16_Q_CONVERT(0.223589), DSP16_Q_CONVERT(-0.974684),
    DSP16_Q_CONVERT(0.222094), DSP16_Q_CONVERT(-0.975025),
    DSP16_Q_CONVERT(0.220598), DSP16_Q_CONVERT(-0.975365),
    DSP16_Q_CONVERT(0.219101), DSP16_Q_CONVERT(-0.975702),
    DSP16_Q_CONVERT(0.217604), DSP16_Q_CONVERT(-0.976037),
    DSP16_Q_CONVERT(0.216107), DSP16_Q_CONVERT(-0.976370),
    DSP16_Q_CONVERT(0.214609), DSP16_Q_CONVERT(-0.976700),
    DSP16_Q_CONVERT(0.213110), DSP16_Q_CONVERT(-0.977028),
    DSP16_Q_CONVERT(0.211611), DSP16_Q_CONVERT(-0.977354),
    DSP16_Q_CONVERT(0.210112), DSP16_Q_CONVERT(-0.977677),
    DSP16_Q_CONVERT(0.208612), DSP16_Q_CONVERT(-0.977999),
    DSP16_Q_CONVERT(0.207111), DSP16_Q_CONVERT(-0.978317),
    DSP16_Q_CONVERT(0.205610), DSP16_Q_CONVERT(-0.978634),
    DSP16_Q_CONVERT(0.204109), DSP16_Q_CONVERT(-0.978948),
    DSP16_Q_CONVERT(0.202607), DSP16_Q_CONVERT(-0.979260),
    DSP16_Q_CONVERT(0.201105), DSP16_Q_CONVERT(-0.979570),
    DSP16_Q_CONVERT(0.199602), DSP16_Q_CONVERT(-0.979877),
    DSP16_Q_CONVERT(0.198098), DSP16_Q_CONVERT(-0.980182),
    DSP16_Q_CONVERT(0.196595), DSP16_Q_CONVERT(-0.980485),
    DSP16_Q_CONVERT(0.195090), DSP16_Q_CONVERT(-0.980785),
    DSP16_Q_CONVERT(0.193586), DSP16_Q_CONVERT(-0.981083),
    DSP16_Q_CONVERT(0.192080), DSP16_Q_CONVERT(-0.981379),
    DSP16_Q_CONVERT(0.190575), DSP16_Q_CONVERT(-0.981673),
    DSP16_Q_CONVERT(0.189069), DSP16_Q_CONVERT(-0.981964),
    DSP16_Q_CONVERT(0.187562), DSP16_Q_CONVERT(-0.982253),
    DSP16_Q_CONVERT(0.186055), DSP16_Q_CONVERT(-0.982539),
    DSP16_Q_CONVERT(0.184548), DSP16_Q_CONVERT(-0.982824),
    DSP16_Q_CONVERT(0.183040), DSP16_Q_CONVERT(-0.983105),
    DSP16_Q_CONVERT(0.181532), DSP16_Q_CONVERT(-0.983385),
    DSP16_Q_CONVERT(0.180023), DSP16_Q_CONVERT(-0.983662),
    DSP16_Q_CONVERT(0.178514), DSP16_Q_CONVERT(-0.983937),
    DSP16_Q_CONVERT(0.177004), DSP16_Q_CONVERT(-0.984210),
    DSP16_Q_CONVERT(0.175494), DSP16_Q_CONVERT(-0.984480),
    DSP16_Q_CONVERT(0.173984), DSP16_Q_CONVERT(-0.984749),
    DSP16_Q_CONVERT(0.172473), DSP16_Q_CONVERT(-0.985014),
    DSP16_Q_CONVERT(0.170962), DSP16_Q_CONVERT(-0.985278),
    DSP16_Q_CONVERT(0.169450), DSP16_Q_CONVERT(-0.985539),
    DSP16_Q_CONVERT(0.167938), DSP16_Q_CONVERT(-0.985798),
    DSP16_Q_CONVERT(0.166426), DSP16_Q_CONVERT(-0.986054),
    DSP16_Q_CONVERT(0.164913), DSP16_Q_CONVERT(-0.986308),
    DSP16_Q_CONVERT(0.163400), DSP16_Q_CONVERT(-0.986560),
    DSP16_Q_CONVERT(0.161886), DSP16_Q_CONVERT(-0.986809),
    DSP16_Q_CONVERT(0.160372), DSP16_Q_CONVERT(-0.987057),
    DSP16_Q_CONVERT(0.158858), DSP16_Q_CONVERT(-0.987301),
    DSP16_Q_CONVERT(0.157343), DSP16_Q_CONVERT(-0.987544),
    DSP16_Q_CONVERT(0.155828), DSP16_Q_CONVERT(-0.987784),
    DSP16_Q_CONVERT(0.154313), DSP16_Q_CONVERT(-0.988022),
    DSP16_Q_CONVERT(0.152797), DSP16_Q_CONVERT(-0.988258),
    DSP16_Q_CONVERT(0.151281), DSP16_Q_CONVERT(-0.988491),
    DSP16_Q_CONVERT(0.149765), DSP16_Q_CONVERT(-0.988722),
    DSP16_Q_CONVERT(0.148248), DSP16_Q_CONVERT(-0.988950),
    DSP16_Q_CONVERT(0.146730), DSP16_Q_CONVERT(-0.989177),
    DSP16_Q_CONVERT(0.145213), DSP16_Q_CONVERT(-0.989400),
    DSP16_Q_CONVERT(0.143695), DSP16_Q_CONVERT(-0.989622),
    DSP16_Q_CONVERT(0.142177), DSP16_Q_CONVERT(-0.989841),
    DSP16_Q_CONVERT(0.140658), DSP16_Q_CONVERT(-0.990058),
    DSP16_Q_CONVERT(0.139139), DSP16_Q_CONVERT(-0.990273),
    DSP16_Q_CONVERT(0.137620), DSP16_Q_CONVERT(-0.990485),
    DSP16_Q_CONVERT(0.136101), DSP16_Q_CONVERT(-0.990695),
    DSP16_Q_CONVERT(0.134581), DSP16_Q_CONVERT(-0.990903),
    DSP16_Q_CONVERT(0.133061), DSP16_Q_CONVERT(-0.991108),
    DSP16_Q_CONVERT(0.131540), DSP16_Q_CONVERT(-0.991311),
    DSP16_Q_CONVERT(0.130019), DSP16_Q_CONVERT(-0.991511),
    DSP16_Q_CONVERT(0.128498), DSP16_Q_CONVERT(-0.991710),
    DSP16_Q_CONVERT(0.126977), DSP16_Q_CONVERT(-0.991906),
    DSP16_Q_CONVERT(0.125455), DSP16_Q_CONVERT(-0.992099),
    DSP16_Q_CONVERT(0.123933), DSP16_Q_CONVERT(-0.992291),
    DSP16_Q_CONVERT(0.122411), DSP16_Q_CONVERT(-0.992480),
    DSP16_Q_CONVERT(0.120888), DSP16_Q_CONVERT(-0.992666),
    DSP16_Q_CONVERT(0.119365), DSP16_Q_CONVERT(-0.992850),
    DSP16_Q_CONVERT(0.117842), DSP16_Q_CONVERT(-0.993032),
    DSP16_Q_CONVERT(0.116319), DSP16_Q_CONVERT(-0.993212),
    DSP16_Q_CONVERT(0.114795), DSP16_Q_CONVERT(-0.993389),
    DSP16_Q_CONVERT(0.113271), DSP16_Q_CONVERT(-0.993564),
    DSP16_Q_CONVERT(0.111747), DSP16_Q_CONVERT(-0.993737),
    DSP16_Q_CONVERT(0.110222), DSP16_Q_CONVERT(-0.993907),
    DSP16_Q_CONVERT(0.108697), DSP16_Q_CONVERT(-0.994075),
    DSP16_Q_CONVERT(0.107172), DSP16_Q_CONVERT(-0.994240),
    DSP16_Q_CONVERT(0.105647), DSP16_Q_CONVERT(-0.994404),
    DSP16_Q_CONVERT(0.104122), DSP16_Q_CONVERT(-0.994565),
    DSP16_Q_CONVERT(0.102596), DSP16_Q_CONVERT(-0.994723),
    DSP16_Q_CONVERT(0.101070), DSP16_Q_CONVERT(-0.994879),
    DSP16_Q_CONVERT(0.099544), DSP16_Q_CONVERT(-0.995033),
    DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185),
    DSP16_Q_CONVERT(0.096490), DSP16_Q_CONVERT(-0.995334),
    DSP16_Q_CONVERT(0.094963), DSP16_Q_CONVERT(-0.995481),
    DSP16_Q_CONVERT(0.093436), DSP16_Q_CONVERT(-0.995625),
    DSP16_Q_CONVERT(0.091909), DSP16_Q_CONVERT(-0.995767),
    DSP16_Q_CONVERT(0.090381), DSP16_Q_CONVERT(-0.995907),
    DSP16_Q_CONVERT(0.088854), DSP16_Q_CONVERT(-0.996045),
    DSP16_Q_CONVERT(0.087326), DSP16_Q_CONVERT(-0.996180),
    DSP16_Q_CONVERT(0.085797), DSP16_Q_CONVERT(-0.996313),
    DSP16_Q_CONVERT(0.084269), DSP16_Q_CONVERT(-0.996443),
    DSP16_Q_CONVERT(0.082740), DSP16_Q_CONVERT(-0.996571),
    DSP16_Q_CONVERT(0.081211), DSP16_Q_CONVERT(-0.996697),
    DSP16_Q_CONVERT(0.079682), DSP16_Q_CONVERT(-0.996820),
    DSP16_Q_CONVERT(0.078153), DSP16_Q_CONVERT(-0.996941),
    DSP16_Q_CONVERT(0.076624), DSP16_Q_CONVERT(-0.997060),
    DSP16_Q_CONVERT(0.075094), DSP16_Q_CONVERT(-0.997176),
    DSP16_Q_CONVERT(0.073565), DSP16_Q_CONVERT(-0.997290),
    DSP16_Q_CONVERT(0.072035), DSP16_Q_CONVERT(-0.997402),
    DSP16_Q_CONVERT(0.070505), DSP16_Q_CONVERT(-0.997511),
    DSP16_Q_CONVERT(0.068974), DSP16_Q_CONVERT(-0.997618),
    DSP16_Q_CONVERT(0.067444), DSP16_Q_CONVERT(-0.997723),
    DSP16_Q_CONVERT(0.065913), DSP16_Q_CONVERT(-0.997825),
    DSP16_Q_CONVERT(0.064383), DSP16_Q_CONVERT(-0.997925),
    DSP16_Q_CONVERT(0.062852), DSP16_Q_CONVERT(-0.998023),
    DSP16_Q_CONVERT(0.061321), DSP16_Q_CONVERT(-0.998118),
    DSP16_Q_CONVERT(0.059790), DSP16_Q_CONVERT(-0.998211),
    DSP16_Q_CONVERT(0.058258), DSP16_Q_CONVERT(-0.998302),
    DSP16_Q_CONVERT(0.056727), DSP16_Q_CONVERT(-0.998390),
    DSP16_Q_CONVERT(0.055195), DSP16_Q_CONVERT(-0.998476),
    DSP16_Q_CONVERT(0.053664), DSP16_Q_CONVERT(-0.998559),
    DSP16_Q_CONVERT(0.052132), DSP16_Q_CONVERT(-0.998640),
    DSP16_Q_CONVERT(0.050600), DSP16_Q_CONVERT(-0.998719),
    DSP16_Q_CONVERT(0.049068), DSP16_Q_CONVERT(-0.998795),
    DSP16_Q_CONVERT(0.047535), DSP16_Q_CONVERT(-0.998870),
    DSP16_Q_CONVERT(0.046003), DSP16_Q_CONVERT(-0.998941),
    DSP16_Q_CONVERT(0.044471), DSP16_Q_CONVERT(-0.999011),
    DSP16_Q_CONVERT(0.042938), DSP16_Q_CONVERT(-0.999078),
    DSP16_Q_CONVERT(0.041406), DSP16_Q_CONVERT(-0.999142),
    DSP16_Q_CONVERT(0.039873), DSP16_Q_CONVERT(-0.999205),
    DSP16_Q_CONVERT(0.038340), DSP16_Q_CONVERT(-0.999265),
    DSP16_Q_CONVERT(0.036807), DSP16_Q_CONVERT(-0.999322),
    DSP16_Q_CONVERT(0.035274), DSP16_Q_CONVERT(-0.999378),
    DSP16_Q_CONVERT(0.033741), DSP16_Q_CONVERT(-0.999431),
    DSP16_Q_CONVERT(0.032208), DSP16_Q_CONVERT(-0.999481),
    DSP16_Q_CONVERT(0.030675), DSP16_Q_CONVERT(-0.999529),
    DSP16_Q_CONVERT(0.029142), DSP16_Q_CONVERT(-0.999575),
    DSP16_Q_CONVERT(0.027608), DSP16_Q_CONVERT(-0.999619),
    DSP16_Q_CONVERT(0.026075), DSP16_Q_CONVERT(-0.999660),
    DSP16_Q_CONVERT(0.024541), DSP16_Q_CONVERT(-0.999699),
    DSP16_Q_CONVERT(0.023008), DSP16_Q_CONVERT(-0.999735),
    DSP16_Q_CONVERT(0.021474), DSP16_Q_CONVERT(-0.999769),
    DSP16_Q_CONVERT(0.019940), DSP16_Q_CONVERT(-0.999801),
    DSP16_Q_CONVERT(0.018407), DSP16_Q_CONVERT(-0.999831),
    DSP16_Q_CONVERT(0.016873), DSP16_Q_CONVERT(-0.999858),
    DSP16_Q_CONVERT(0.015339), DSP16_Q_CONVERT(-0.999882),
    DSP16_Q_CONVERT(0.013805), DSP16_Q_CONVERT(-0.999905),
    DSP16_Q_CONVERT(0.012272), DSP16_Q_CONVERT(-0.999925),
    DSP16_Q_CONVERT(0.010738), DSP16_Q_CONVERT(-0.999942),
    DSP16_Q_CONVERT(0.009204), DSP16_Q_CONVERT(-0.999958),
    DSP16_Q_CONVERT(0.007670), DSP16_Q_CONVERT(-0.999971),
    DSP16_Q_CONVERT(0.006136), DSP16_Q_CONVERT(-0.999981),
    DSP16_Q_CONVERT(0.004602), DSP16_Q_CONVERT(-0.999989),
    DSP16_Q_CONVERT(0.003068), DSP16_Q_CONVERT(-0.999995),
    DSP16_Q_CONVERT(0.001534), DSP16_Q_CONVERT(-0.999999),
    DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(-1.000000)
  };

  // If we need to optimize the code to use small twiddle factors table
#  if !(DSP_OPTIMIZATION & DSP_OPTI_SIZE)
/*! \brief This table has been generated using the following algorithm:
 * \code
 * for i=0:4:DSP16_N_TWIDDLE_FACTORS-4,
 * k = i/(DSP16_N_TWIDDLE_FACTORS*4);
 * w2 = exp(-2*PI*%i*k*2);
 * w3 = exp(-2*PI*%i*k*3);
 * end;
 * \endcode
 * It is a one dimensional array containing the real parts (4*i parts) and imaginary parts (4*i+1 parts) of the \b w2 value
 * and the real parts (4*i+2 parts) and imaginary parts (4*i+3 parts) of the \b w3 value.
 */
    A_ALIGNED TWIDDLE_FACTORS_PREFIX_TAB dsp16_t dsp16_twiddle_factors2[DSP16_N_TWIDDLE_FACTORS] = {
    /*Re(w2)   Im(w2)   Re(w3)   Im(w3)  */
      DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(1.000000), DSP16_Q_CONVERT(0.000000),
      DSP16_Q_CONVERT(0.999995), DSP16_Q_CONVERT(-0.003068), DSP16_Q_CONVERT(0.999989), DSP16_Q_CONVERT(-0.004602),
      DSP16_Q_CONVERT(0.999981), DSP16_Q_CONVERT(-0.006136), DSP16_Q_CONVERT(0.999958), DSP16_Q_CONVERT(-0.009204),
      DSP16_Q_CONVERT(0.999958), DSP16_Q_CONVERT(-0.009204), DSP16_Q_CONVERT(0.999905), DSP16_Q_CONVERT(-0.013805),
      DSP16_Q_CONVERT(0.999925), DSP16_Q_CONVERT(-0.012272), DSP16_Q_CONVERT(0.999831), DSP16_Q_CONVERT(-0.018407),
      DSP16_Q_CONVERT(0.999882), DSP16_Q_CONVERT(-0.015339), DSP16_Q_CONVERT(0.999735), DSP16_Q_CONVERT(-0.023008),
      DSP16_Q_CONVERT(0.999831), DSP16_Q_CONVERT(-0.018407), DSP16_Q_CONVERT(0.999619), DSP16_Q_CONVERT(-0.027608),
      DSP16_Q_CONVERT(0.999769), DSP16_Q_CONVERT(-0.021474), DSP16_Q_CONVERT(0.999481), DSP16_Q_CONVERT(-0.032208),
      DSP16_Q_CONVERT(0.999699), DSP16_Q_CONVERT(-0.024541), DSP16_Q_CONVERT(0.999322), DSP16_Q_CONVERT(-0.036807),
      DSP16_Q_CONVERT(0.999619), DSP16_Q_CONVERT(-0.027608), DSP16_Q_CONVERT(0.999142), DSP16_Q_CONVERT(-0.041406),
      DSP16_Q_CONVERT(0.999529), DSP16_Q_CONVERT(-0.030675), DSP16_Q_CONVERT(0.998941), DSP16_Q_CONVERT(-0.046003),
      DSP16_Q_CONVERT(0.999431), DSP16_Q_CONVERT(-0.033741), DSP16_Q_CONVERT(0.998719), DSP16_Q_CONVERT(-0.050600),
      DSP16_Q_CONVERT(0.999322), DSP16_Q_CONVERT(-0.036807), DSP16_Q_CONVERT(0.998476), DSP16_Q_CONVERT(-0.055195),
      DSP16_Q_CONVERT(0.999205), DSP16_Q_CONVERT(-0.039873), DSP16_Q_CONVERT(0.998211), DSP16_Q_CONVERT(-0.059790),
      DSP16_Q_CONVERT(0.999078), DSP16_Q_CONVERT(-0.042938), DSP16_Q_CONVERT(0.997925), DSP16_Q_CONVERT(-0.064383),
      DSP16_Q_CONVERT(0.998941), DSP16_Q_CONVERT(-0.046003), DSP16_Q_CONVERT(0.997618), DSP16_Q_CONVERT(-0.068974),
      DSP16_Q_CONVERT(0.998795), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(0.997290), DSP16_Q_CONVERT(-0.073565),
      DSP16_Q_CONVERT(0.998640), DSP16_Q_CONVERT(-0.052132), DSP16_Q_CONVERT(0.996941), DSP16_Q_CONVERT(-0.078153),
      DSP16_Q_CONVERT(0.998476), DSP16_Q_CONVERT(-0.055195), DSP16_Q_CONVERT(0.996571), DSP16_Q_CONVERT(-0.082740),
      DSP16_Q_CONVERT(0.998302), DSP16_Q_CONVERT(-0.058258), DSP16_Q_CONVERT(0.996180), DSP16_Q_CONVERT(-0.087326),
      DSP16_Q_CONVERT(0.998118), DSP16_Q_CONVERT(-0.061321), DSP16_Q_CONVERT(0.995767), DSP16_Q_CONVERT(-0.091909),
      DSP16_Q_CONVERT(0.997925), DSP16_Q_CONVERT(-0.064383), DSP16_Q_CONVERT(0.995334), DSP16_Q_CONVERT(-0.096490),
      DSP16_Q_CONVERT(0.997723), DSP16_Q_CONVERT(-0.067444), DSP16_Q_CONVERT(0.994879), DSP16_Q_CONVERT(-0.101070),
      DSP16_Q_CONVERT(0.997511), DSP16_Q_CONVERT(-0.070505), DSP16_Q_CONVERT(0.994404), DSP16_Q_CONVERT(-0.105647),
      DSP16_Q_CONVERT(0.997290), DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(0.993907), DSP16_Q_CONVERT(-0.110222),
      DSP16_Q_CONVERT(0.997060), DSP16_Q_CONVERT(-0.076624), DSP16_Q_CONVERT(0.993389), DSP16_Q_CONVERT(-0.114795),
      DSP16_Q_CONVERT(0.996820), DSP16_Q_CONVERT(-0.079682), DSP16_Q_CONVERT(0.992850), DSP16_Q_CONVERT(-0.119365),
      DSP16_Q_CONVERT(0.996571), DSP16_Q_CONVERT(-0.082740), DSP16_Q_CONVERT(0.992291), DSP16_Q_CONVERT(-0.123933),
      DSP16_Q_CONVERT(0.996313), DSP16_Q_CONVERT(-0.085797), DSP16_Q_CONVERT(0.991710), DSP16_Q_CONVERT(-0.128498),
      DSP16_Q_CONVERT(0.996045), DSP16_Q_CONVERT(-0.088854), DSP16_Q_CONVERT(0.991108), DSP16_Q_CONVERT(-0.133061),
      DSP16_Q_CONVERT(0.995767), DSP16_Q_CONVERT(-0.091909), DSP16_Q_CONVERT(0.990485), DSP16_Q_CONVERT(-0.137620),
      DSP16_Q_CONVERT(0.995481), DSP16_Q_CONVERT(-0.094963), DSP16_Q_CONVERT(0.989841), DSP16_Q_CONVERT(-0.142177),
      DSP16_Q_CONVERT(0.995185), DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730),
      DSP16_Q_CONVERT(0.994879), DSP16_Q_CONVERT(-0.101070), DSP16_Q_CONVERT(0.988491), DSP16_Q_CONVERT(-0.151281),
      DSP16_Q_CONVERT(0.994565), DSP16_Q_CONVERT(-0.104122), DSP16_Q_CONVERT(0.987784), DSP16_Q_CONVERT(-0.155828),
      DSP16_Q_CONVERT(0.994240), DSP16_Q_CONVERT(-0.107172), DSP16_Q_CONVERT(0.987057), DSP16_Q_CONVERT(-0.160372),
      DSP16_Q_CONVERT(0.993907), DSP16_Q_CONVERT(-0.110222), DSP16_Q_CONVERT(0.986308), DSP16_Q_CONVERT(-0.164913),
      DSP16_Q_CONVERT(0.993564), DSP16_Q_CONVERT(-0.113271), DSP16_Q_CONVERT(0.985539), DSP16_Q_CONVERT(-0.169450),
      DSP16_Q_CONVERT(0.993212), DSP16_Q_CONVERT(-0.116319), DSP16_Q_CONVERT(0.984749), DSP16_Q_CONVERT(-0.173984),
      DSP16_Q_CONVERT(0.992850), DSP16_Q_CONVERT(-0.119365), DSP16_Q_CONVERT(0.983937), DSP16_Q_CONVERT(-0.178514),
      DSP16_Q_CONVERT(0.992480), DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(0.983105), DSP16_Q_CONVERT(-0.183040),
      DSP16_Q_CONVERT(0.992099), DSP16_Q_CONVERT(-0.125455), DSP16_Q_CONVERT(0.982253), DSP16_Q_CONVERT(-0.187562),
      DSP16_Q_CONVERT(0.991710), DSP16_Q_CONVERT(-0.128498), DSP16_Q_CONVERT(0.981379), DSP16_Q_CONVERT(-0.192080),
      DSP16_Q_CONVERT(0.991311), DSP16_Q_CONVERT(-0.131540), DSP16_Q_CONVERT(0.980485), DSP16_Q_CONVERT(-0.196595),
      DSP16_Q_CONVERT(0.990903), DSP16_Q_CONVERT(-0.134581), DSP16_Q_CONVERT(0.979570), DSP16_Q_CONVERT(-0.201105),
      DSP16_Q_CONVERT(0.990485), DSP16_Q_CONVERT(-0.137620), DSP16_Q_CONVERT(0.978634), DSP16_Q_CONVERT(-0.205610),
      DSP16_Q_CONVERT(0.990058), DSP16_Q_CONVERT(-0.140658), DSP16_Q_CONVERT(0.977677), DSP16_Q_CONVERT(-0.210112),
      DSP16_Q_CONVERT(0.989622), DSP16_Q_CONVERT(-0.143695), DSP16_Q_CONVERT(0.976700), DSP16_Q_CONVERT(-0.214609),
      DSP16_Q_CONVERT(0.989177), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(0.975702), DSP16_Q_CONVERT(-0.219101),
      DSP16_Q_CONVERT(0.988722), DSP16_Q_CONVERT(-0.149765), DSP16_Q_CONVERT(0.974684), DSP16_Q_CONVERT(-0.223589),
      DSP16_Q_CONVERT(0.988258), DSP16_Q_CONVERT(-0.152797), DSP16_Q_CONVERT(0.973644), DSP16_Q_CONVERT(-0.228072),
      DSP16_Q_CONVERT(0.987784), DSP16_Q_CONVERT(-0.155828), DSP16_Q_CONVERT(0.972584), DSP16_Q_CONVERT(-0.232550),
      DSP16_Q_CONVERT(0.987301), DSP16_Q_CONVERT(-0.158858), DSP16_Q_CONVERT(0.971504), DSP16_Q_CONVERT(-0.237024),
      DSP16_Q_CONVERT(0.986809), DSP16_Q_CONVERT(-0.161886), DSP16_Q_CONVERT(0.970403), DSP16_Q_CONVERT(-0.241492),
      DSP16_Q_CONVERT(0.986308), DSP16_Q_CONVERT(-0.164913), DSP16_Q_CONVERT(0.969281), DSP16_Q_CONVERT(-0.245955),
      DSP16_Q_CONVERT(0.985798), DSP16_Q_CONVERT(-0.167938), DSP16_Q_CONVERT(0.968139), DSP16_Q_CONVERT(-0.250413),
      DSP16_Q_CONVERT(0.985278), DSP16_Q_CONVERT(-0.170962), DSP16_Q_CONVERT(0.966976), DSP16_Q_CONVERT(-0.254866),
      DSP16_Q_CONVERT(0.984749), DSP16_Q_CONVERT(-0.173984), DSP16_Q_CONVERT(0.965793), DSP16_Q_CONVERT(-0.259313),
      DSP16_Q_CONVERT(0.984210), DSP16_Q_CONVERT(-0.177004), DSP16_Q_CONVERT(0.964590), DSP16_Q_CONVERT(-0.263755),
      DSP16_Q_CONVERT(0.983662), DSP16_Q_CONVERT(-0.180023), DSP16_Q_CONVERT(0.963366), DSP16_Q_CONVERT(-0.268191),
      DSP16_Q_CONVERT(0.983105), DSP16_Q_CONVERT(-0.183040), DSP16_Q_CONVERT(0.962121), DSP16_Q_CONVERT(-0.272621),
      DSP16_Q_CONVERT(0.982539), DSP16_Q_CONVERT(-0.186055), DSP16_Q_CONVERT(0.960857), DSP16_Q_CONVERT(-0.277046),
      DSP16_Q_CONVERT(0.981964), DSP16_Q_CONVERT(-0.189069), DSP16_Q_CONVERT(0.959572), DSP16_Q_CONVERT(-0.281465),
      DSP16_Q_CONVERT(0.981379), DSP16_Q_CONVERT(-0.192080), DSP16_Q_CONVERT(0.958266), DSP16_Q_CONVERT(-0.285878),
      DSP16_Q_CONVERT(0.980785), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285),
      DSP16_Q_CONVERT(0.980182), DSP16_Q_CONVERT(-0.198098), DSP16_Q_CONVERT(0.955594), DSP16_Q_CONVERT(-0.294685),
      DSP16_Q_CONVERT(0.979570), DSP16_Q_CONVERT(-0.201105), DSP16_Q_CONVERT(0.954228), DSP16_Q_CONVERT(-0.299080),
      DSP16_Q_CONVERT(0.978948), DSP16_Q_CONVERT(-0.204109), DSP16_Q_CONVERT(0.952842), DSP16_Q_CONVERT(-0.303468),
      DSP16_Q_CONVERT(0.978317), DSP16_Q_CONVERT(-0.207111), DSP16_Q_CONVERT(0.951435), DSP16_Q_CONVERT(-0.307850),
      DSP16_Q_CONVERT(0.977677), DSP16_Q_CONVERT(-0.210112), DSP16_Q_CONVERT(0.950008), DSP16_Q_CONVERT(-0.312225),
      DSP16_Q_CONVERT(0.977028), DSP16_Q_CONVERT(-0.213110), DSP16_Q_CONVERT(0.948561), DSP16_Q_CONVERT(-0.316593),
      DSP16_Q_CONVERT(0.976370), DSP16_Q_CONVERT(-0.216107), DSP16_Q_CONVERT(0.947094), DSP16_Q_CONVERT(-0.320955),
      DSP16_Q_CONVERT(0.975702), DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(0.945607), DSP16_Q_CONVERT(-0.325310),
      DSP16_Q_CONVERT(0.975025), DSP16_Q_CONVERT(-0.222094), DSP16_Q_CONVERT(0.944100), DSP16_Q_CONVERT(-0.329658),
      DSP16_Q_CONVERT(0.974339), DSP16_Q_CONVERT(-0.225084), DSP16_Q_CONVERT(0.942573), DSP16_Q_CONVERT(-0.334000),
      DSP16_Q_CONVERT(0.973644), DSP16_Q_CONVERT(-0.228072), DSP16_Q_CONVERT(0.941026), DSP16_Q_CONVERT(-0.338334),
      DSP16_Q_CONVERT(0.972940), DSP16_Q_CONVERT(-0.231058), DSP16_Q_CONVERT(0.939459), DSP16_Q_CONVERT(-0.342661),
      DSP16_Q_CONVERT(0.972226), DSP16_Q_CONVERT(-0.234042), DSP16_Q_CONVERT(0.937872), DSP16_Q_CONVERT(-0.346980),
      DSP16_Q_CONVERT(0.971504), DSP16_Q_CONVERT(-0.237024), DSP16_Q_CONVERT(0.936266), DSP16_Q_CONVERT(-0.351293),
      DSP16_Q_CONVERT(0.970772), DSP16_Q_CONVERT(-0.240003), DSP16_Q_CONVERT(0.934639), DSP16_Q_CONVERT(-0.355598),
      DSP16_Q_CONVERT(0.970031), DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(0.932993), DSP16_Q_CONVERT(-0.359895),
      DSP16_Q_CONVERT(0.969281), DSP16_Q_CONVERT(-0.245955), DSP16_Q_CONVERT(0.931327), DSP16_Q_CONVERT(-0.364185),
      DSP16_Q_CONVERT(0.968522), DSP16_Q_CONVERT(-0.248928), DSP16_Q_CONVERT(0.929641), DSP16_Q_CONVERT(-0.368467),
      DSP16_Q_CONVERT(0.967754), DSP16_Q_CONVERT(-0.251898), DSP16_Q_CONVERT(0.927935), DSP16_Q_CONVERT(-0.372741),
      DSP16_Q_CONVERT(0.966976), DSP16_Q_CONVERT(-0.254866), DSP16_Q_CONVERT(0.926210), DSP16_Q_CONVERT(-0.377007),
      DSP16_Q_CONVERT(0.966190), DSP16_Q_CONVERT(-0.257831), DSP16_Q_CONVERT(0.924465), DSP16_Q_CONVERT(-0.381266),
      DSP16_Q_CONVERT(0.965394), DSP16_Q_CONVERT(-0.260794), DSP16_Q_CONVERT(0.922701), DSP16_Q_CONVERT(-0.385516),
      DSP16_Q_CONVERT(0.964590), DSP16_Q_CONVERT(-0.263755), DSP16_Q_CONVERT(0.920917), DSP16_Q_CONVERT(-0.389758),
      DSP16_Q_CONVERT(0.963776), DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(0.919114), DSP16_Q_CONVERT(-0.393992),
      DSP16_Q_CONVERT(0.962953), DSP16_Q_CONVERT(-0.269668), DSP16_Q_CONVERT(0.917291), DSP16_Q_CONVERT(-0.398218),
      DSP16_Q_CONVERT(0.962121), DSP16_Q_CONVERT(-0.272621), DSP16_Q_CONVERT(0.915449), DSP16_Q_CONVERT(-0.402435),
      DSP16_Q_CONVERT(0.961280), DSP16_Q_CONVERT(-0.275572), DSP16_Q_CONVERT(0.913587), DSP16_Q_CONVERT(-0.406643),
      DSP16_Q_CONVERT(0.960431), DSP16_Q_CONVERT(-0.278520), DSP16_Q_CONVERT(0.911706), DSP16_Q_CONVERT(-0.410843),
      DSP16_Q_CONVERT(0.959572), DSP16_Q_CONVERT(-0.281465), DSP16_Q_CONVERT(0.909806), DSP16_Q_CONVERT(-0.415034),
      DSP16_Q_CONVERT(0.958703), DSP16_Q_CONVERT(-0.284408), DSP16_Q_CONVERT(0.907886), DSP16_Q_CONVERT(-0.419217),
      DSP16_Q_CONVERT(0.957826), DSP16_Q_CONVERT(-0.287347), DSP16_Q_CONVERT(0.905947), DSP16_Q_CONVERT(-0.423390),
      DSP16_Q_CONVERT(0.956940), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555),
      DSP16_Q_CONVERT(0.956045), DSP16_Q_CONVERT(-0.293219), DSP16_Q_CONVERT(0.902012), DSP16_Q_CONVERT(-0.431711),
      DSP16_Q_CONVERT(0.955141), DSP16_Q_CONVERT(-0.296151), DSP16_Q_CONVERT(0.900016), DSP16_Q_CONVERT(-0.435857),
      DSP16_Q_CONVERT(0.954228), DSP16_Q_CONVERT(-0.299080), DSP16_Q_CONVERT(0.898001), DSP16_Q_CONVERT(-0.439994),
      DSP16_Q_CONVERT(0.953306), DSP16_Q_CONVERT(-0.302006), DSP16_Q_CONVERT(0.895966), DSP16_Q_CONVERT(-0.444122),
      DSP16_Q_CONVERT(0.952375), DSP16_Q_CONVERT(-0.304929), DSP16_Q_CONVERT(0.893913), DSP16_Q_CONVERT(-0.448241),
      DSP16_Q_CONVERT(0.951435), DSP16_Q_CONVERT(-0.307850), DSP16_Q_CONVERT(0.891841), DSP16_Q_CONVERT(-0.452350),
      DSP16_Q_CONVERT(0.950486), DSP16_Q_CONVERT(-0.310767), DSP16_Q_CONVERT(0.889750), DSP16_Q_CONVERT(-0.456449),
      DSP16_Q_CONVERT(0.949528), DSP16_Q_CONVERT(-0.313682), DSP16_Q_CONVERT(0.887640), DSP16_Q_CONVERT(-0.460539),
      DSP16_Q_CONVERT(0.948561), DSP16_Q_CONVERT(-0.316593), DSP16_Q_CONVERT(0.885511), DSP16_Q_CONVERT(-0.464619),
      DSP16_Q_CONVERT(0.947586), DSP16_Q_CONVERT(-0.319502), DSP16_Q_CONVERT(0.883363), DSP16_Q_CONVERT(-0.468689),
      DSP16_Q_CONVERT(0.946601), DSP16_Q_CONVERT(-0.322408), DSP16_Q_CONVERT(0.881197), DSP16_Q_CONVERT(-0.472749),
      DSP16_Q_CONVERT(0.945607), DSP16_Q_CONVERT(-0.325310), DSP16_Q_CONVERT(0.879012), DSP16_Q_CONVERT(-0.476799),
      DSP16_Q_CONVERT(0.944605), DSP16_Q_CONVERT(-0.328210), DSP16_Q_CONVERT(0.876809), DSP16_Q_CONVERT(-0.480839),
      DSP16_Q_CONVERT(0.943593), DSP16_Q_CONVERT(-0.331106), DSP16_Q_CONVERT(0.874587), DSP16_Q_CONVERT(-0.484869),
      DSP16_Q_CONVERT(0.942573), DSP16_Q_CONVERT(-0.334000), DSP16_Q_CONVERT(0.872346), DSP16_Q_CONVERT(-0.488889),
      DSP16_Q_CONVERT(0.941544), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(0.870087), DSP16_Q_CONVERT(-0.492898),
      DSP16_Q_CONVERT(0.940506), DSP16_Q_CONVERT(-0.339777), DSP16_Q_CONVERT(0.867809), DSP16_Q_CONVERT(-0.496897),
      DSP16_Q_CONVERT(0.939459), DSP16_Q_CONVERT(-0.342661), DSP16_Q_CONVERT(0.865514), DSP16_Q_CONVERT(-0.500885),
      DSP16_Q_CONVERT(0.938404), DSP16_Q_CONVERT(-0.345541), DSP16_Q_CONVERT(0.863199), DSP16_Q_CONVERT(-0.504863),
      DSP16_Q_CONVERT(0.937339), DSP16_Q_CONVERT(-0.348419), DSP16_Q_CONVERT(0.860867), DSP16_Q_CONVERT(-0.508830),
      DSP16_Q_CONVERT(0.936266), DSP16_Q_CONVERT(-0.351293), DSP16_Q_CONVERT(0.858516), DSP16_Q_CONVERT(-0.512786),
      DSP16_Q_CONVERT(0.935184), DSP16_Q_CONVERT(-0.354164), DSP16_Q_CONVERT(0.856147), DSP16_Q_CONVERT(-0.516732),
      DSP16_Q_CONVERT(0.934093), DSP16_Q_CONVERT(-0.357031), DSP16_Q_CONVERT(0.853760), DSP16_Q_CONVERT(-0.520666),
      DSP16_Q_CONVERT(0.932993), DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(0.851355), DSP16_Q_CONVERT(-0.524590),
      DSP16_Q_CONVERT(0.931884), DSP16_Q_CONVERT(-0.362756), DSP16_Q_CONVERT(0.848932), DSP16_Q_CONVERT(-0.528502),
      DSP16_Q_CONVERT(0.930767), DSP16_Q_CONVERT(-0.365613), DSP16_Q_CONVERT(0.846491), DSP16_Q_CONVERT(-0.532403),
      DSP16_Q_CONVERT(0.929641), DSP16_Q_CONVERT(-0.368467), DSP16_Q_CONVERT(0.844032), DSP16_Q_CONVERT(-0.536293),
      DSP16_Q_CONVERT(0.928506), DSP16_Q_CONVERT(-0.371317), DSP16_Q_CONVERT(0.841555), DSP16_Q_CONVERT(-0.540171),
      DSP16_Q_CONVERT(0.927363), DSP16_Q_CONVERT(-0.374164), DSP16_Q_CONVERT(0.839060), DSP16_Q_CONVERT(-0.544039),
      DSP16_Q_CONVERT(0.926210), DSP16_Q_CONVERT(-0.377007), DSP16_Q_CONVERT(0.836548), DSP16_Q_CONVERT(-0.547894),
      DSP16_Q_CONVERT(0.925049), DSP16_Q_CONVERT(-0.379847), DSP16_Q_CONVERT(0.834018), DSP16_Q_CONVERT(-0.551738),
      DSP16_Q_CONVERT(0.923880), DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570),
      DSP16_Q_CONVERT(0.922701), DSP16_Q_CONVERT(-0.385516), DSP16_Q_CONVERT(0.828904), DSP16_Q_CONVERT(-0.559391),
      DSP16_Q_CONVERT(0.921514), DSP16_Q_CONVERT(-0.388345), DSP16_Q_CONVERT(0.826321), DSP16_Q_CONVERT(-0.563199),
      DSP16_Q_CONVERT(0.920318), DSP16_Q_CONVERT(-0.391170), DSP16_Q_CONVERT(0.823721), DSP16_Q_CONVERT(-0.566996),
      DSP16_Q_CONVERT(0.919114), DSP16_Q_CONVERT(-0.393992), DSP16_Q_CONVERT(0.821103), DSP16_Q_CONVERT(-0.570781),
      DSP16_Q_CONVERT(0.917901), DSP16_Q_CONVERT(-0.396810), DSP16_Q_CONVERT(0.818467), DSP16_Q_CONVERT(-0.574553),
      DSP16_Q_CONVERT(0.916679), DSP16_Q_CONVERT(-0.399624), DSP16_Q_CONVERT(0.815814), DSP16_Q_CONVERT(-0.578314),
      DSP16_Q_CONVERT(0.915449), DSP16_Q_CONVERT(-0.402435), DSP16_Q_CONVERT(0.813144), DSP16_Q_CONVERT(-0.582062),
      DSP16_Q_CONVERT(0.914210), DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(0.810457), DSP16_Q_CONVERT(-0.585798),
      DSP16_Q_CONVERT(0.912962), DSP16_Q_CONVERT(-0.408044), DSP16_Q_CONVERT(0.807753), DSP16_Q_CONVERT(-0.589521),
      DSP16_Q_CONVERT(0.911706), DSP16_Q_CONVERT(-0.410843), DSP16_Q_CONVERT(0.805031), DSP16_Q_CONVERT(-0.593232),
      DSP16_Q_CONVERT(0.910441), DSP16_Q_CONVERT(-0.413638), DSP16_Q_CONVERT(0.802293), DSP16_Q_CONVERT(-0.596931),
      DSP16_Q_CONVERT(0.909168), DSP16_Q_CONVERT(-0.416430), DSP16_Q_CONVERT(0.799537), DSP16_Q_CONVERT(-0.600616),
      DSP16_Q_CONVERT(0.907886), DSP16_Q_CONVERT(-0.419217), DSP16_Q_CONVERT(0.796765), DSP16_Q_CONVERT(-0.604290),
      DSP16_Q_CONVERT(0.906596), DSP16_Q_CONVERT(-0.422000), DSP16_Q_CONVERT(0.793975), DSP16_Q_CONVERT(-0.607950),
      DSP16_Q_CONVERT(0.905297), DSP16_Q_CONVERT(-0.424780), DSP16_Q_CONVERT(0.791169), DSP16_Q_CONVERT(-0.611597),
      DSP16_Q_CONVERT(0.903989), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(0.788346), DSP16_Q_CONVERT(-0.615232),
      DSP16_Q_CONVERT(0.902673), DSP16_Q_CONVERT(-0.430326), DSP16_Q_CONVERT(0.785507), DSP16_Q_CONVERT(-0.618853),
      DSP16_Q_CONVERT(0.901349), DSP16_Q_CONVERT(-0.433094), DSP16_Q_CONVERT(0.782651), DSP16_Q_CONVERT(-0.622461),
      DSP16_Q_CONVERT(0.900016), DSP16_Q_CONVERT(-0.435857), DSP16_Q_CONVERT(0.779778), DSP16_Q_CONVERT(-0.626056),
      DSP16_Q_CONVERT(0.898674), DSP16_Q_CONVERT(-0.438616), DSP16_Q_CONVERT(0.776888), DSP16_Q_CONVERT(-0.629638),
      DSP16_Q_CONVERT(0.897325), DSP16_Q_CONVERT(-0.441371), DSP16_Q_CONVERT(0.773983), DSP16_Q_CONVERT(-0.633207),
      DSP16_Q_CONVERT(0.895966), DSP16_Q_CONVERT(-0.444122), DSP16_Q_CONVERT(0.771061), DSP16_Q_CONVERT(-0.636762),
      DSP16_Q_CONVERT(0.894599), DSP16_Q_CONVERT(-0.446869), DSP16_Q_CONVERT(0.768122), DSP16_Q_CONVERT(-0.640303),
      DSP16_Q_CONVERT(0.893224), DSP16_Q_CONVERT(-0.449611), DSP16_Q_CONVERT(0.765167), DSP16_Q_CONVERT(-0.643832),
      DSP16_Q_CONVERT(0.891841), DSP16_Q_CONVERT(-0.452350), DSP16_Q_CONVERT(0.762196), DSP16_Q_CONVERT(-0.647346),
      DSP16_Q_CONVERT(0.890449), DSP16_Q_CONVERT(-0.455084), DSP16_Q_CONVERT(0.759209), DSP16_Q_CONVERT(-0.650847),
      DSP16_Q_CONVERT(0.889048), DSP16_Q_CONVERT(-0.457813), DSP16_Q_CONVERT(0.756206), DSP16_Q_CONVERT(-0.654334),
      DSP16_Q_CONVERT(0.887640), DSP16_Q_CONVERT(-0.460539), DSP16_Q_CONVERT(0.753187), DSP16_Q_CONVERT(-0.657807),
      DSP16_Q_CONVERT(0.886223), DSP16_Q_CONVERT(-0.463260), DSP16_Q_CONVERT(0.750152), DSP16_Q_CONVERT(-0.661266),
      DSP16_Q_CONVERT(0.884797), DSP16_Q_CONVERT(-0.465976), DSP16_Q_CONVERT(0.747101), DSP16_Q_CONVERT(-0.664711),
      DSP16_Q_CONVERT(0.883363), DSP16_Q_CONVERT(-0.468689), DSP16_Q_CONVERT(0.744034), DSP16_Q_CONVERT(-0.668142),
      DSP16_Q_CONVERT(0.881921), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559),
      DSP16_Q_CONVERT(0.880471), DSP16_Q_CONVERT(-0.474100), DSP16_Q_CONVERT(0.737853), DSP16_Q_CONVERT(-0.674962),
      DSP16_Q_CONVERT(0.879012), DSP16_Q_CONVERT(-0.476799), DSP16_Q_CONVERT(0.734739), DSP16_Q_CONVERT(-0.678350),
      DSP16_Q_CONVERT(0.877545), DSP16_Q_CONVERT(-0.479494), DSP16_Q_CONVERT(0.731609), DSP16_Q_CONVERT(-0.681724),
      DSP16_Q_CONVERT(0.876070), DSP16_Q_CONVERT(-0.482184), DSP16_Q_CONVERT(0.728464), DSP16_Q_CONVERT(-0.685084),
      DSP16_Q_CONVERT(0.874587), DSP16_Q_CONVERT(-0.484869), DSP16_Q_CONVERT(0.725304), DSP16_Q_CONVERT(-0.688429),
      DSP16_Q_CONVERT(0.873095), DSP16_Q_CONVERT(-0.487550), DSP16_Q_CONVERT(0.722128), DSP16_Q_CONVERT(-0.691759),
      DSP16_Q_CONVERT(0.871595), DSP16_Q_CONVERT(-0.490226), DSP16_Q_CONVERT(0.718937), DSP16_Q_CONVERT(-0.695075),
      DSP16_Q_CONVERT(0.870087), DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(0.715731), DSP16_Q_CONVERT(-0.698376),
      DSP16_Q_CONVERT(0.868571), DSP16_Q_CONVERT(-0.495565), DSP16_Q_CONVERT(0.712509), DSP16_Q_CONVERT(-0.701663),
      DSP16_Q_CONVERT(0.867046), DSP16_Q_CONVERT(-0.498228), DSP16_Q_CONVERT(0.709273), DSP16_Q_CONVERT(-0.704934),
      DSP16_Q_CONVERT(0.865514), DSP16_Q_CONVERT(-0.500885), DSP16_Q_CONVERT(0.706021), DSP16_Q_CONVERT(-0.708191),
      DSP16_Q_CONVERT(0.863973), DSP16_Q_CONVERT(-0.503538), DSP16_Q_CONVERT(0.702755), DSP16_Q_CONVERT(-0.711432),
      DSP16_Q_CONVERT(0.862424), DSP16_Q_CONVERT(-0.506187), DSP16_Q_CONVERT(0.699473), DSP16_Q_CONVERT(-0.714659),
      DSP16_Q_CONVERT(0.860867), DSP16_Q_CONVERT(-0.508830), DSP16_Q_CONVERT(0.696177), DSP16_Q_CONVERT(-0.717870),
      DSP16_Q_CONVERT(0.859302), DSP16_Q_CONVERT(-0.511469), DSP16_Q_CONVERT(0.692866), DSP16_Q_CONVERT(-0.721066),
      DSP16_Q_CONVERT(0.857729), DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(0.689541), DSP16_Q_CONVERT(-0.724247),
      DSP16_Q_CONVERT(0.856147), DSP16_Q_CONVERT(-0.516732), DSP16_Q_CONVERT(0.686200), DSP16_Q_CONVERT(-0.727413),
      DSP16_Q_CONVERT(0.854558), DSP16_Q_CONVERT(-0.519356), DSP16_Q_CONVERT(0.682846), DSP16_Q_CONVERT(-0.730563),
      DSP16_Q_CONVERT(0.852961), DSP16_Q_CONVERT(-0.521975), DSP16_Q_CONVERT(0.679476), DSP16_Q_CONVERT(-0.733697),
      DSP16_Q_CONVERT(0.851355), DSP16_Q_CONVERT(-0.524590), DSP16_Q_CONVERT(0.676093), DSP16_Q_CONVERT(-0.736817),
      DSP16_Q_CONVERT(0.849742), DSP16_Q_CONVERT(-0.527199), DSP16_Q_CONVERT(0.672695), DSP16_Q_CONVERT(-0.739920),
      DSP16_Q_CONVERT(0.848120), DSP16_Q_CONVERT(-0.529804), DSP16_Q_CONVERT(0.669283), DSP16_Q_CONVERT(-0.743008),
      DSP16_Q_CONVERT(0.846491), DSP16_Q_CONVERT(-0.532403), DSP16_Q_CONVERT(0.665856), DSP16_Q_CONVERT(-0.746080),
      DSP16_Q_CONVERT(0.844854), DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(0.662416), DSP16_Q_CONVERT(-0.749136),
      DSP16_Q_CONVERT(0.843208), DSP16_Q_CONVERT(-0.537587), DSP16_Q_CONVERT(0.658961), DSP16_Q_CONVERT(-0.752177),
      DSP16_Q_CONVERT(0.841555), DSP16_Q_CONVERT(-0.540171), DSP16_Q_CONVERT(0.655493), DSP16_Q_CONVERT(-0.755201),
      DSP16_Q_CONVERT(0.839894), DSP16_Q_CONVERT(-0.542751), DSP16_Q_CONVERT(0.652011), DSP16_Q_CONVERT(-0.758210),
      DSP16_Q_CONVERT(0.838225), DSP16_Q_CONVERT(-0.545325), DSP16_Q_CONVERT(0.648514), DSP16_Q_CONVERT(-0.761202),
      DSP16_Q_CONVERT(0.836548), DSP16_Q_CONVERT(-0.547894), DSP16_Q_CONVERT(0.645005), DSP16_Q_CONVERT(-0.764179),
      DSP16_Q_CONVERT(0.834863), DSP16_Q_CONVERT(-0.550458), DSP16_Q_CONVERT(0.641481), DSP16_Q_CONVERT(-0.767139),
      DSP16_Q_CONVERT(0.833170), DSP16_Q_CONVERT(-0.553017), DSP16_Q_CONVERT(0.637944), DSP16_Q_CONVERT(-0.770083),
      DSP16_Q_CONVERT(0.831470), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010),
      DSP16_Q_CONVERT(0.829761), DSP16_Q_CONVERT(-0.558119), DSP16_Q_CONVERT(0.630829), DSP16_Q_CONVERT(-0.775922),
      DSP16_Q_CONVERT(0.828045), DSP16_Q_CONVERT(-0.560662), DSP16_Q_CONVERT(0.627252), DSP16_Q_CONVERT(-0.778817),
      DSP16_Q_CONVERT(0.826321), DSP16_Q_CONVERT(-0.563199), DSP16_Q_CONVERT(0.623661), DSP16_Q_CONVERT(-0.781695),
      DSP16_Q_CONVERT(0.824589), DSP16_Q_CONVERT(-0.565732), DSP16_Q_CONVERT(0.620057), DSP16_Q_CONVERT(-0.784557),
      DSP16_Q_CONVERT(0.822850), DSP16_Q_CONVERT(-0.568259), DSP16_Q_CONVERT(0.616440), DSP16_Q_CONVERT(-0.787402),
      DSP16_Q_CONVERT(0.821103), DSP16_Q_CONVERT(-0.570781), DSP16_Q_CONVERT(0.612810), DSP16_Q_CONVERT(-0.790230),
      DSP16_Q_CONVERT(0.819348), DSP16_Q_CONVERT(-0.573297), DSP16_Q_CONVERT(0.609167), DSP16_Q_CONVERT(-0.793042),
      DSP16_Q_CONVERT(0.817585), DSP16_Q_CONVERT(-0.575808), DSP16_Q_CONVERT(0.605511), DSP16_Q_CONVERT(-0.795837),
      DSP16_Q_CONVERT(0.815814), DSP16_Q_CONVERT(-0.578314), DSP16_Q_CONVERT(0.601842), DSP16_Q_CONVERT(-0.798615),
      DSP16_Q_CONVERT(0.814036), DSP16_Q_CONVERT(-0.580814), DSP16_Q_CONVERT(0.598161), DSP16_Q_CONVERT(-0.801376),
      DSP16_Q_CONVERT(0.812251), DSP16_Q_CONVERT(-0.583309), DSP16_Q_CONVERT(0.594466), DSP16_Q_CONVERT(-0.804120),
      DSP16_Q_CONVERT(0.810457), DSP16_Q_CONVERT(-0.585798), DSP16_Q_CONVERT(0.590760), DSP16_Q_CONVERT(-0.806848),
      DSP16_Q_CONVERT(0.808656), DSP16_Q_CONVERT(-0.588282), DSP16_Q_CONVERT(0.587040), DSP16_Q_CONVERT(-0.809558),
      DSP16_Q_CONVERT(0.806848), DSP16_Q_CONVERT(-0.590760), DSP16_Q_CONVERT(0.583309), DSP16_Q_CONVERT(-0.812251),
      DSP16_Q_CONVERT(0.805031), DSP16_Q_CONVERT(-0.593232), DSP16_Q_CONVERT(0.579565), DSP16_Q_CONVERT(-0.814926),
      DSP16_Q_CONVERT(0.803208), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(0.575808), DSP16_Q_CONVERT(-0.817585),
      DSP16_Q_CONVERT(0.801376), DSP16_Q_CONVERT(-0.598161), DSP16_Q_CONVERT(0.572040), DSP16_Q_CONVERT(-0.820226),
      DSP16_Q_CONVERT(0.799537), DSP16_Q_CONVERT(-0.600616), DSP16_Q_CONVERT(0.568259), DSP16_Q_CONVERT(-0.822850),
      DSP16_Q_CONVERT(0.797691), DSP16_Q_CONVERT(-0.603067), DSP16_Q_CONVERT(0.564466), DSP16_Q_CONVERT(-0.825456),
      DSP16_Q_CONVERT(0.795837), DSP16_Q_CONVERT(-0.605511), DSP16_Q_CONVERT(0.560662), DSP16_Q_CONVERT(-0.828045),
      DSP16_Q_CONVERT(0.793975), DSP16_Q_CONVERT(-0.607950), DSP16_Q_CONVERT(0.556845), DSP16_Q_CONVERT(-0.830616),
      DSP16_Q_CONVERT(0.792107), DSP16_Q_CONVERT(-0.610383), DSP16_Q_CONVERT(0.553017), DSP16_Q_CONVERT(-0.833170),
      DSP16_Q_CONVERT(0.790230), DSP16_Q_CONVERT(-0.612810), DSP16_Q_CONVERT(0.549177), DSP16_Q_CONVERT(-0.835706),
      DSP16_Q_CONVERT(0.788346), DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(0.545325), DSP16_Q_CONVERT(-0.838225),
      DSP16_Q_CONVERT(0.786455), DSP16_Q_CONVERT(-0.617647), DSP16_Q_CONVERT(0.541462), DSP16_Q_CONVERT(-0.840725),
      DSP16_Q_CONVERT(0.784557), DSP16_Q_CONVERT(-0.620057), DSP16_Q_CONVERT(0.537587), DSP16_Q_CONVERT(-0.843208),
      DSP16_Q_CONVERT(0.782651), DSP16_Q_CONVERT(-0.622461), DSP16_Q_CONVERT(0.533701), DSP16_Q_CONVERT(-0.845673),
      DSP16_Q_CONVERT(0.780737), DSP16_Q_CONVERT(-0.624859), DSP16_Q_CONVERT(0.529804), DSP16_Q_CONVERT(-0.848120),
      DSP16_Q_CONVERT(0.778817), DSP16_Q_CONVERT(-0.627252), DSP16_Q_CONVERT(0.525895), DSP16_Q_CONVERT(-0.850549),
      DSP16_Q_CONVERT(0.776888), DSP16_Q_CONVERT(-0.629638), DSP16_Q_CONVERT(0.521975), DSP16_Q_CONVERT(-0.852961),
      DSP16_Q_CONVERT(0.774953), DSP16_Q_CONVERT(-0.632019), DSP16_Q_CONVERT(0.518045), DSP16_Q_CONVERT(-0.855354),
      DSP16_Q_CONVERT(0.773010), DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729),
      DSP16_Q_CONVERT(0.771061), DSP16_Q_CONVERT(-0.636762), DSP16_Q_CONVERT(0.510150), DSP16_Q_CONVERT(-0.860085),
      DSP16_Q_CONVERT(0.769103), DSP16_Q_CONVERT(-0.639124), DSP16_Q_CONVERT(0.506187), DSP16_Q_CONVERT(-0.862424),
      DSP16_Q_CONVERT(0.767139), DSP16_Q_CONVERT(-0.641481), DSP16_Q_CONVERT(0.502212), DSP16_Q_CONVERT(-0.864744),
      DSP16_Q_CONVERT(0.765167), DSP16_Q_CONVERT(-0.643832), DSP16_Q_CONVERT(0.498228), DSP16_Q_CONVERT(-0.867046),
      DSP16_Q_CONVERT(0.763188), DSP16_Q_CONVERT(-0.646176), DSP16_Q_CONVERT(0.494232), DSP16_Q_CONVERT(-0.869330),
      DSP16_Q_CONVERT(0.761202), DSP16_Q_CONVERT(-0.648514), DSP16_Q_CONVERT(0.490226), DSP16_Q_CONVERT(-0.871595),
      DSP16_Q_CONVERT(0.759209), DSP16_Q_CONVERT(-0.650847), DSP16_Q_CONVERT(0.486210), DSP16_Q_CONVERT(-0.873842),
      DSP16_Q_CONVERT(0.757209), DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(0.482184), DSP16_Q_CONVERT(-0.876070),
      DSP16_Q_CONVERT(0.755201), DSP16_Q_CONVERT(-0.655493), DSP16_Q_CONVERT(0.478147), DSP16_Q_CONVERT(-0.878280),
      DSP16_Q_CONVERT(0.753187), DSP16_Q_CONVERT(-0.657807), DSP16_Q_CONVERT(0.474100), DSP16_Q_CONVERT(-0.880471),
      DSP16_Q_CONVERT(0.751165), DSP16_Q_CONVERT(-0.660114), DSP16_Q_CONVERT(0.470043), DSP16_Q_CONVERT(-0.882643),
      DSP16_Q_CONVERT(0.749136), DSP16_Q_CONVERT(-0.662416), DSP16_Q_CONVERT(0.465976), DSP16_Q_CONVERT(-0.884797),
      DSP16_Q_CONVERT(0.747101), DSP16_Q_CONVERT(-0.664711), DSP16_Q_CONVERT(0.461900), DSP16_Q_CONVERT(-0.886932),
      DSP16_Q_CONVERT(0.745058), DSP16_Q_CONVERT(-0.667000), DSP16_Q_CONVERT(0.457813), DSP16_Q_CONVERT(-0.889048),
      DSP16_Q_CONVERT(0.743008), DSP16_Q_CONVERT(-0.669283), DSP16_Q_CONVERT(0.453717), DSP16_Q_CONVERT(-0.891146),
      DSP16_Q_CONVERT(0.740951), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(0.449611), DSP16_Q_CONVERT(-0.893224),
      DSP16_Q_CONVERT(0.738887), DSP16_Q_CONVERT(-0.673829), DSP16_Q_CONVERT(0.445496), DSP16_Q_CONVERT(-0.895284),
      DSP16_Q_CONVERT(0.736817), DSP16_Q_CONVERT(-0.676093), DSP16_Q_CONVERT(0.441371), DSP16_Q_CONVERT(-0.897325),
      DSP16_Q_CONVERT(0.734739), DSP16_Q_CONVERT(-0.678350), DSP16_Q_CONVERT(0.437237), DSP16_Q_CONVERT(-0.899346),
      DSP16_Q_CONVERT(0.732654), DSP16_Q_CONVERT(-0.680601), DSP16_Q_CONVERT(0.433094), DSP16_Q_CONVERT(-0.901349),
      DSP16_Q_CONVERT(0.730563), DSP16_Q_CONVERT(-0.682846), DSP16_Q_CONVERT(0.428941), DSP16_Q_CONVERT(-0.903332),
      DSP16_Q_CONVERT(0.728464), DSP16_Q_CONVERT(-0.685084), DSP16_Q_CONVERT(0.424780), DSP16_Q_CONVERT(-0.905297),
      DSP16_Q_CONVERT(0.726359), DSP16_Q_CONVERT(-0.687315), DSP16_Q_CONVERT(0.420609), DSP16_Q_CONVERT(-0.907242),
      DSP16_Q_CONVERT(0.724247), DSP16_Q_CONVERT(-0.689541), DSP16_Q_CONVERT(0.416430), DSP16_Q_CONVERT(-0.909168),
      DSP16_Q_CONVERT(0.722128), DSP16_Q_CONVERT(-0.691759), DSP16_Q_CONVERT(0.412241), DSP16_Q_CONVERT(-0.911075),
      DSP16_Q_CONVERT(0.720003), DSP16_Q_CONVERT(-0.693971), DSP16_Q_CONVERT(0.408044), DSP16_Q_CONVERT(-0.912962),
      DSP16_Q_CONVERT(0.717870), DSP16_Q_CONVERT(-0.696177), DSP16_Q_CONVERT(0.403838), DSP16_Q_CONVERT(-0.914830),
      DSP16_Q_CONVERT(0.715731), DSP16_Q_CONVERT(-0.698376), DSP16_Q_CONVERT(0.399624), DSP16_Q_CONVERT(-0.916679),
      DSP16_Q_CONVERT(0.713585), DSP16_Q_CONVERT(-0.700569), DSP16_Q_CONVERT(0.395401), DSP16_Q_CONVERT(-0.918508),
      DSP16_Q_CONVERT(0.711432), DSP16_Q_CONVERT(-0.702755), DSP16_Q_CONVERT(0.391170), DSP16_Q_CONVERT(-0.920318),
      DSP16_Q_CONVERT(0.709273), DSP16_Q_CONVERT(-0.704934), DSP16_Q_CONVERT(0.386931), DSP16_Q_CONVERT(-0.922109),
      DSP16_Q_CONVERT(0.707107), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880),
      DSP16_Q_CONVERT(0.704934), DSP16_Q_CONVERT(-0.709273), DSP16_Q_CONVERT(0.378428), DSP16_Q_CONVERT(-0.925631),
      DSP16_Q_CONVERT(0.702755), DSP16_Q_CONVERT(-0.711432), DSP16_Q_CONVERT(0.374164), DSP16_Q_CONVERT(-0.927363),
      DSP16_Q_CONVERT(0.700569), DSP16_Q_CONVERT(-0.713585), DSP16_Q_CONVERT(0.369892), DSP16_Q_CONVERT(-0.929075),
      DSP16_Q_CONVERT(0.698376), DSP16_Q_CONVERT(-0.715731), DSP16_Q_CONVERT(0.365613), DSP16_Q_CONVERT(-0.930767),
      DSP16_Q_CONVERT(0.696177), DSP16_Q_CONVERT(-0.717870), DSP16_Q_CONVERT(0.361326), DSP16_Q_CONVERT(-0.932440),
      DSP16_Q_CONVERT(0.693971), DSP16_Q_CONVERT(-0.720003), DSP16_Q_CONVERT(0.357031), DSP16_Q_CONVERT(-0.934093),
      DSP16_Q_CONVERT(0.691759), DSP16_Q_CONVERT(-0.722128), DSP16_Q_CONVERT(0.352729), DSP16_Q_CONVERT(-0.935726),
      DSP16_Q_CONVERT(0.689541), DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(0.348419), DSP16_Q_CONVERT(-0.937339),
      DSP16_Q_CONVERT(0.687315), DSP16_Q_CONVERT(-0.726359), DSP16_Q_CONVERT(0.344101), DSP16_Q_CONVERT(-0.938932),
      DSP16_Q_CONVERT(0.685084), DSP16_Q_CONVERT(-0.728464), DSP16_Q_CONVERT(0.339777), DSP16_Q_CONVERT(-0.940506),
      DSP16_Q_CONVERT(0.682846), DSP16_Q_CONVERT(-0.730563), DSP16_Q_CONVERT(0.335445), DSP16_Q_CONVERT(-0.942060),
      DSP16_Q_CONVERT(0.680601), DSP16_Q_CONVERT(-0.732654), DSP16_Q_CONVERT(0.331106), DSP16_Q_CONVERT(-0.943593),
      DSP16_Q_CONVERT(0.678350), DSP16_Q_CONVERT(-0.734739), DSP16_Q_CONVERT(0.326760), DSP16_Q_CONVERT(-0.945107),
      DSP16_Q_CONVERT(0.676093), DSP16_Q_CONVERT(-0.736817), DSP16_Q_CONVERT(0.322408), DSP16_Q_CONVERT(-0.946601),
      DSP16_Q_CONVERT(0.673829), DSP16_Q_CONVERT(-0.738887), DSP16_Q_CONVERT(0.318048), DSP16_Q_CONVERT(-0.948075),
      DSP16_Q_CONVERT(0.671559), DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(0.313682), DSP16_Q_CONVERT(-0.949528),
      DSP16_Q_CONVERT(0.669283), DSP16_Q_CONVERT(-0.743008), DSP16_Q_CONVERT(0.309309), DSP16_Q_CONVERT(-0.950962),
      DSP16_Q_CONVERT(0.667000), DSP16_Q_CONVERT(-0.745058), DSP16_Q_CONVERT(0.304929), DSP16_Q_CONVERT(-0.952375),
      DSP16_Q_CONVERT(0.664711), DSP16_Q_CONVERT(-0.747101), DSP16_Q_CONVERT(0.300543), DSP16_Q_CONVERT(-0.953768),
      DSP16_Q_CONVERT(0.662416), DSP16_Q_CONVERT(-0.749136), DSP16_Q_CONVERT(0.296151), DSP16_Q_CONVERT(-0.955141),
      DSP16_Q_CONVERT(0.660114), DSP16_Q_CONVERT(-0.751165), DSP16_Q_CONVERT(0.291752), DSP16_Q_CONVERT(-0.956494),
      DSP16_Q_CONVERT(0.657807), DSP16_Q_CONVERT(-0.753187), DSP16_Q_CONVERT(0.287347), DSP16_Q_CONVERT(-0.957826),
      DSP16_Q_CONVERT(0.655493), DSP16_Q_CONVERT(-0.755201), DSP16_Q_CONVERT(0.282937), DSP16_Q_CONVERT(-0.959139),
      DSP16_Q_CONVERT(0.653173), DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(0.278520), DSP16_Q_CONVERT(-0.960431),
      DSP16_Q_CONVERT(0.650847), DSP16_Q_CONVERT(-0.759209), DSP16_Q_CONVERT(0.274097), DSP16_Q_CONVERT(-0.961702),
      DSP16_Q_CONVERT(0.648514), DSP16_Q_CONVERT(-0.761202), DSP16_Q_CONVERT(0.269668), DSP16_Q_CONVERT(-0.962953),
      DSP16_Q_CONVERT(0.646176), DSP16_Q_CONVERT(-0.763188), DSP16_Q_CONVERT(0.265234), DSP16_Q_CONVERT(-0.964184),
      DSP16_Q_CONVERT(0.643832), DSP16_Q_CONVERT(-0.765167), DSP16_Q_CONVERT(0.260794), DSP16_Q_CONVERT(-0.965394),
      DSP16_Q_CONVERT(0.641481), DSP16_Q_CONVERT(-0.767139), DSP16_Q_CONVERT(0.256349), DSP16_Q_CONVERT(-0.966584),
      DSP16_Q_CONVERT(0.639124), DSP16_Q_CONVERT(-0.769103), DSP16_Q_CONVERT(0.251898), DSP16_Q_CONVERT(-0.967754),
      DSP16_Q_CONVERT(0.636762), DSP16_Q_CONVERT(-0.771061), DSP16_Q_CONVERT(0.247442), DSP16_Q_CONVERT(-0.968903),
      DSP16_Q_CONVERT(0.634393), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031),
      DSP16_Q_CONVERT(0.632019), DSP16_Q_CONVERT(-0.774953), DSP16_Q_CONVERT(0.238514), DSP16_Q_CONVERT(-0.971139),
      DSP16_Q_CONVERT(0.629638), DSP16_Q_CONVERT(-0.776888), DSP16_Q_CONVERT(0.234042), DSP16_Q_CONVERT(-0.972226),
      DSP16_Q_CONVERT(0.627252), DSP16_Q_CONVERT(-0.778817), DSP16_Q_CONVERT(0.229565), DSP16_Q_CONVERT(-0.973293),
      DSP16_Q_CONVERT(0.624859), DSP16_Q_CONVERT(-0.780737), DSP16_Q_CONVERT(0.225084), DSP16_Q_CONVERT(-0.974339),
      DSP16_Q_CONVERT(0.622461), DSP16_Q_CONVERT(-0.782651), DSP16_Q_CONVERT(0.220598), DSP16_Q_CONVERT(-0.975365),
      DSP16_Q_CONVERT(0.620057), DSP16_Q_CONVERT(-0.784557), DSP16_Q_CONVERT(0.216107), DSP16_Q_CONVERT(-0.976370),
      DSP16_Q_CONVERT(0.617647), DSP16_Q_CONVERT(-0.786455), DSP16_Q_CONVERT(0.211611), DSP16_Q_CONVERT(-0.977354),
      DSP16_Q_CONVERT(0.615232), DSP16_Q_CONVERT(-0.788346), DSP16_Q_CONVERT(0.207111), DSP16_Q_CONVERT(-0.978317),
      DSP16_Q_CONVERT(0.612810), DSP16_Q_CONVERT(-0.790230), DSP16_Q_CONVERT(0.202607), DSP16_Q_CONVERT(-0.979260),
      DSP16_Q_CONVERT(0.610383), DSP16_Q_CONVERT(-0.792107), DSP16_Q_CONVERT(0.198098), DSP16_Q_CONVERT(-0.980182),
      DSP16_Q_CONVERT(0.607950), DSP16_Q_CONVERT(-0.793975), DSP16_Q_CONVERT(0.193586), DSP16_Q_CONVERT(-0.981083),
      DSP16_Q_CONVERT(0.605511), DSP16_Q_CONVERT(-0.795837), DSP16_Q_CONVERT(0.189069), DSP16_Q_CONVERT(-0.981964),
      DSP16_Q_CONVERT(0.603067), DSP16_Q_CONVERT(-0.797691), DSP16_Q_CONVERT(0.184548), DSP16_Q_CONVERT(-0.982824),
      DSP16_Q_CONVERT(0.600616), DSP16_Q_CONVERT(-0.799537), DSP16_Q_CONVERT(0.180023), DSP16_Q_CONVERT(-0.983662),
      DSP16_Q_CONVERT(0.598161), DSP16_Q_CONVERT(-0.801376), DSP16_Q_CONVERT(0.175494), DSP16_Q_CONVERT(-0.984480),
      DSP16_Q_CONVERT(0.595699), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(0.170962), DSP16_Q_CONVERT(-0.985278),
      DSP16_Q_CONVERT(0.593232), DSP16_Q_CONVERT(-0.805031), DSP16_Q_CONVERT(0.166426), DSP16_Q_CONVERT(-0.986054),
      DSP16_Q_CONVERT(0.590760), DSP16_Q_CONVERT(-0.806848), DSP16_Q_CONVERT(0.161886), DSP16_Q_CONVERT(-0.986809),
      DSP16_Q_CONVERT(0.588282), DSP16_Q_CONVERT(-0.808656), DSP16_Q_CONVERT(0.157343), DSP16_Q_CONVERT(-0.987544),
      DSP16_Q_CONVERT(0.585798), DSP16_Q_CONVERT(-0.810457), DSP16_Q_CONVERT(0.152797), DSP16_Q_CONVERT(-0.988258),
      DSP16_Q_CONVERT(0.583309), DSP16_Q_CONVERT(-0.812251), DSP16_Q_CONVERT(0.148248), DSP16_Q_CONVERT(-0.988950),
      DSP16_Q_CONVERT(0.580814), DSP16_Q_CONVERT(-0.814036), DSP16_Q_CONVERT(0.143695), DSP16_Q_CONVERT(-0.989622),
      DSP16_Q_CONVERT(0.578314), DSP16_Q_CONVERT(-0.815814), DSP16_Q_CONVERT(0.139139), DSP16_Q_CONVERT(-0.990273),
      DSP16_Q_CONVERT(0.575808), DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(0.134581), DSP16_Q_CONVERT(-0.990903),
      DSP16_Q_CONVERT(0.573297), DSP16_Q_CONVERT(-0.819348), DSP16_Q_CONVERT(0.130019), DSP16_Q_CONVERT(-0.991511),
      DSP16_Q_CONVERT(0.570781), DSP16_Q_CONVERT(-0.821103), DSP16_Q_CONVERT(0.125455), DSP16_Q_CONVERT(-0.992099),
      DSP16_Q_CONVERT(0.568259), DSP16_Q_CONVERT(-0.822850), DSP16_Q_CONVERT(0.120888), DSP16_Q_CONVERT(-0.992666),
      DSP16_Q_CONVERT(0.565732), DSP16_Q_CONVERT(-0.824589), DSP16_Q_CONVERT(0.116319), DSP16_Q_CONVERT(-0.993212),
      DSP16_Q_CONVERT(0.563199), DSP16_Q_CONVERT(-0.826321), DSP16_Q_CONVERT(0.111747), DSP16_Q_CONVERT(-0.993737),
      DSP16_Q_CONVERT(0.560662), DSP16_Q_CONVERT(-0.828045), DSP16_Q_CONVERT(0.107172), DSP16_Q_CONVERT(-0.994240),
      DSP16_Q_CONVERT(0.558119), DSP16_Q_CONVERT(-0.829761), DSP16_Q_CONVERT(0.102596), DSP16_Q_CONVERT(-0.994723),
      DSP16_Q_CONVERT(0.555570), DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185),
      DSP16_Q_CONVERT(0.553017), DSP16_Q_CONVERT(-0.833170), DSP16_Q_CONVERT(0.093436), DSP16_Q_CONVERT(-0.995625),
      DSP16_Q_CONVERT(0.550458), DSP16_Q_CONVERT(-0.834863), DSP16_Q_CONVERT(0.088854), DSP16_Q_CONVERT(-0.996045),
      DSP16_Q_CONVERT(0.547894), DSP16_Q_CONVERT(-0.836548), DSP16_Q_CONVERT(0.084269), DSP16_Q_CONVERT(-0.996443),
      DSP16_Q_CONVERT(0.545325), DSP16_Q_CONVERT(-0.838225), DSP16_Q_CONVERT(0.079682), DSP16_Q_CONVERT(-0.996820),
      DSP16_Q_CONVERT(0.542751), DSP16_Q_CONVERT(-0.839894), DSP16_Q_CONVERT(0.075094), DSP16_Q_CONVERT(-0.997176),
      DSP16_Q_CONVERT(0.540171), DSP16_Q_CONVERT(-0.841555), DSP16_Q_CONVERT(0.070505), DSP16_Q_CONVERT(-0.997511),
      DSP16_Q_CONVERT(0.537587), DSP16_Q_CONVERT(-0.843208), DSP16_Q_CONVERT(0.065913), DSP16_Q_CONVERT(-0.997825),
      DSP16_Q_CONVERT(0.534998), DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(0.061321), DSP16_Q_CONVERT(-0.998118),
      DSP16_Q_CONVERT(0.532403), DSP16_Q_CONVERT(-0.846491), DSP16_Q_CONVERT(0.056727), DSP16_Q_CONVERT(-0.998390),
      DSP16_Q_CONVERT(0.529804), DSP16_Q_CONVERT(-0.848120), DSP16_Q_CONVERT(0.052132), DSP16_Q_CONVERT(-0.998640),
      DSP16_Q_CONVERT(0.527199), DSP16_Q_CONVERT(-0.849742), DSP16_Q_CONVERT(0.047535), DSP16_Q_CONVERT(-0.998870),
      DSP16_Q_CONVERT(0.524590), DSP16_Q_CONVERT(-0.851355), DSP16_Q_CONVERT(0.042938), DSP16_Q_CONVERT(-0.999078),
      DSP16_Q_CONVERT(0.521975), DSP16_Q_CONVERT(-0.852961), DSP16_Q_CONVERT(0.038340), DSP16_Q_CONVERT(-0.999265),
      DSP16_Q_CONVERT(0.519356), DSP16_Q_CONVERT(-0.854558), DSP16_Q_CONVERT(0.033741), DSP16_Q_CONVERT(-0.999431),
      DSP16_Q_CONVERT(0.516732), DSP16_Q_CONVERT(-0.856147), DSP16_Q_CONVERT(0.029142), DSP16_Q_CONVERT(-0.999575),
      DSP16_Q_CONVERT(0.514103), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(0.024541), DSP16_Q_CONVERT(-0.999699),
      DSP16_Q_CONVERT(0.511469), DSP16_Q_CONVERT(-0.859302), DSP16_Q_CONVERT(0.019940), DSP16_Q_CONVERT(-0.999801),
      DSP16_Q_CONVERT(0.508830), DSP16_Q_CONVERT(-0.860867), DSP16_Q_CONVERT(0.015339), DSP16_Q_CONVERT(-0.999882),
      DSP16_Q_CONVERT(0.506187), DSP16_Q_CONVERT(-0.862424), DSP16_Q_CONVERT(0.010738), DSP16_Q_CONVERT(-0.999942),
      DSP16_Q_CONVERT(0.503538), DSP16_Q_CONVERT(-0.863973), DSP16_Q_CONVERT(0.006136), DSP16_Q_CONVERT(-0.999981),
      DSP16_Q_CONVERT(0.500885), DSP16_Q_CONVERT(-0.865514), DSP16_Q_CONVERT(0.001534), DSP16_Q_CONVERT(-0.999999),
      DSP16_Q_CONVERT(0.498228), DSP16_Q_CONVERT(-0.867046), DSP16_Q_CONVERT(-0.003068), DSP16_Q_CONVERT(-0.999995),
      DSP16_Q_CONVERT(0.495565), DSP16_Q_CONVERT(-0.868571), DSP16_Q_CONVERT(-0.007670), DSP16_Q_CONVERT(-0.999971),
      DSP16_Q_CONVERT(0.492898), DSP16_Q_CONVERT(-0.870087), DSP16_Q_CONVERT(-0.012272), DSP16_Q_CONVERT(-0.999925),
      DSP16_Q_CONVERT(0.490226), DSP16_Q_CONVERT(-0.871595), DSP16_Q_CONVERT(-0.016873), DSP16_Q_CONVERT(-0.999858),
      DSP16_Q_CONVERT(0.487550), DSP16_Q_CONVERT(-0.873095), DSP16_Q_CONVERT(-0.021474), DSP16_Q_CONVERT(-0.999769),
      DSP16_Q_CONVERT(0.484869), DSP16_Q_CONVERT(-0.874587), DSP16_Q_CONVERT(-0.026075), DSP16_Q_CONVERT(-0.999660),
      DSP16_Q_CONVERT(0.482184), DSP16_Q_CONVERT(-0.876070), DSP16_Q_CONVERT(-0.030675), DSP16_Q_CONVERT(-0.999529),
      DSP16_Q_CONVERT(0.479494), DSP16_Q_CONVERT(-0.877545), DSP16_Q_CONVERT(-0.035274), DSP16_Q_CONVERT(-0.999378),
      DSP16_Q_CONVERT(0.476799), DSP16_Q_CONVERT(-0.879012), DSP16_Q_CONVERT(-0.039873), DSP16_Q_CONVERT(-0.999205),
      DSP16_Q_CONVERT(0.474100), DSP16_Q_CONVERT(-0.880471), DSP16_Q_CONVERT(-0.044471), DSP16_Q_CONVERT(-0.999011),
      DSP16_Q_CONVERT(0.471397), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.998795),
      DSP16_Q_CONVERT(0.468689), DSP16_Q_CONVERT(-0.883363), DSP16_Q_CONVERT(-0.053664), DSP16_Q_CONVERT(-0.998559),
      DSP16_Q_CONVERT(0.465976), DSP16_Q_CONVERT(-0.884797), DSP16_Q_CONVERT(-0.058258), DSP16_Q_CONVERT(-0.998302),
      DSP16_Q_CONVERT(0.463260), DSP16_Q_CONVERT(-0.886223), DSP16_Q_CONVERT(-0.062852), DSP16_Q_CONVERT(-0.998023),
      DSP16_Q_CONVERT(0.460539), DSP16_Q_CONVERT(-0.887640), DSP16_Q_CONVERT(-0.067444), DSP16_Q_CONVERT(-0.997723),
      DSP16_Q_CONVERT(0.457813), DSP16_Q_CONVERT(-0.889048), DSP16_Q_CONVERT(-0.072035), DSP16_Q_CONVERT(-0.997402),
      DSP16_Q_CONVERT(0.455084), DSP16_Q_CONVERT(-0.890449), DSP16_Q_CONVERT(-0.076624), DSP16_Q_CONVERT(-0.997060),
      DSP16_Q_CONVERT(0.452350), DSP16_Q_CONVERT(-0.891841), DSP16_Q_CONVERT(-0.081211), DSP16_Q_CONVERT(-0.996697),
      DSP16_Q_CONVERT(0.449611), DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(-0.085797), DSP16_Q_CONVERT(-0.996313),
      DSP16_Q_CONVERT(0.446869), DSP16_Q_CONVERT(-0.894599), DSP16_Q_CONVERT(-0.090381), DSP16_Q_CONVERT(-0.995907),
      DSP16_Q_CONVERT(0.444122), DSP16_Q_CONVERT(-0.895966), DSP16_Q_CONVERT(-0.094963), DSP16_Q_CONVERT(-0.995481),
      DSP16_Q_CONVERT(0.441371), DSP16_Q_CONVERT(-0.897325), DSP16_Q_CONVERT(-0.099544), DSP16_Q_CONVERT(-0.995033),
      DSP16_Q_CONVERT(0.438616), DSP16_Q_CONVERT(-0.898674), DSP16_Q_CONVERT(-0.104122), DSP16_Q_CONVERT(-0.994565),
      DSP16_Q_CONVERT(0.435857), DSP16_Q_CONVERT(-0.900016), DSP16_Q_CONVERT(-0.108697), DSP16_Q_CONVERT(-0.994075),
      DSP16_Q_CONVERT(0.433094), DSP16_Q_CONVERT(-0.901349), DSP16_Q_CONVERT(-0.113271), DSP16_Q_CONVERT(-0.993564),
      DSP16_Q_CONVERT(0.430326), DSP16_Q_CONVERT(-0.902673), DSP16_Q_CONVERT(-0.117842), DSP16_Q_CONVERT(-0.993032),
      DSP16_Q_CONVERT(0.427555), DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(-0.992480),
      DSP16_Q_CONVERT(0.424780), DSP16_Q_CONVERT(-0.905297), DSP16_Q_CONVERT(-0.126977), DSP16_Q_CONVERT(-0.991906),
      DSP16_Q_CONVERT(0.422000), DSP16_Q_CONVERT(-0.906596), DSP16_Q_CONVERT(-0.131540), DSP16_Q_CONVERT(-0.991311),
      DSP16_Q_CONVERT(0.419217), DSP16_Q_CONVERT(-0.907886), DSP16_Q_CONVERT(-0.136101), DSP16_Q_CONVERT(-0.990695),
      DSP16_Q_CONVERT(0.416430), DSP16_Q_CONVERT(-0.909168), DSP16_Q_CONVERT(-0.140658), DSP16_Q_CONVERT(-0.990058),
      DSP16_Q_CONVERT(0.413638), DSP16_Q_CONVERT(-0.910441), DSP16_Q_CONVERT(-0.145213), DSP16_Q_CONVERT(-0.989400),
      DSP16_Q_CONVERT(0.410843), DSP16_Q_CONVERT(-0.911706), DSP16_Q_CONVERT(-0.149765), DSP16_Q_CONVERT(-0.988722),
      DSP16_Q_CONVERT(0.408044), DSP16_Q_CONVERT(-0.912962), DSP16_Q_CONVERT(-0.154313), DSP16_Q_CONVERT(-0.988022),
      DSP16_Q_CONVERT(0.405241), DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.158858), DSP16_Q_CONVERT(-0.987301),
      DSP16_Q_CONVERT(0.402435), DSP16_Q_CONVERT(-0.915449), DSP16_Q_CONVERT(-0.163400), DSP16_Q_CONVERT(-0.986560),
      DSP16_Q_CONVERT(0.399624), DSP16_Q_CONVERT(-0.916679), DSP16_Q_CONVERT(-0.167938), DSP16_Q_CONVERT(-0.985798),
      DSP16_Q_CONVERT(0.396810), DSP16_Q_CONVERT(-0.917901), DSP16_Q_CONVERT(-0.172473), DSP16_Q_CONVERT(-0.985014),
      DSP16_Q_CONVERT(0.393992), DSP16_Q_CONVERT(-0.919114), DSP16_Q_CONVERT(-0.177004), DSP16_Q_CONVERT(-0.984210),
      DSP16_Q_CONVERT(0.391170), DSP16_Q_CONVERT(-0.920318), DSP16_Q_CONVERT(-0.181532), DSP16_Q_CONVERT(-0.983385),
      DSP16_Q_CONVERT(0.388345), DSP16_Q_CONVERT(-0.921514), DSP16_Q_CONVERT(-0.186055), DSP16_Q_CONVERT(-0.982539),
      DSP16_Q_CONVERT(0.385516), DSP16_Q_CONVERT(-0.922701), DSP16_Q_CONVERT(-0.190575), DSP16_Q_CONVERT(-0.981673),
      DSP16_Q_CONVERT(0.382683), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.980785),
      DSP16_Q_CONVERT(0.379847), DSP16_Q_CONVERT(-0.925049), DSP16_Q_CONVERT(-0.199602), DSP16_Q_CONVERT(-0.979877),
      DSP16_Q_CONVERT(0.377007), DSP16_Q_CONVERT(-0.926210), DSP16_Q_CONVERT(-0.204109), DSP16_Q_CONVERT(-0.978948),
      DSP16_Q_CONVERT(0.374164), DSP16_Q_CONVERT(-0.927363), DSP16_Q_CONVERT(-0.208612), DSP16_Q_CONVERT(-0.977999),
      DSP16_Q_CONVERT(0.371317), DSP16_Q_CONVERT(-0.928506), DSP16_Q_CONVERT(-0.213110), DSP16_Q_CONVERT(-0.977028),
      DSP16_Q_CONVERT(0.368467), DSP16_Q_CONVERT(-0.929641), DSP16_Q_CONVERT(-0.217604), DSP16_Q_CONVERT(-0.976037),
      DSP16_Q_CONVERT(0.365613), DSP16_Q_CONVERT(-0.930767), DSP16_Q_CONVERT(-0.222094), DSP16_Q_CONVERT(-0.975025),
      DSP16_Q_CONVERT(0.362756), DSP16_Q_CONVERT(-0.931884), DSP16_Q_CONVERT(-0.226578), DSP16_Q_CONVERT(-0.973993),
      DSP16_Q_CONVERT(0.359895), DSP16_Q_CONVERT(-0.932993), DSP16_Q_CONVERT(-0.231058), DSP16_Q_CONVERT(-0.972940),
      DSP16_Q_CONVERT(0.357031), DSP16_Q_CONVERT(-0.934093), DSP16_Q_CONVERT(-0.235533), DSP16_Q_CONVERT(-0.971866),
      DSP16_Q_CONVERT(0.354164), DSP16_Q_CONVERT(-0.935184), DSP16_Q_CONVERT(-0.240003), DSP16_Q_CONVERT(-0.970772),
      DSP16_Q_CONVERT(0.351293), DSP16_Q_CONVERT(-0.936266), DSP16_Q_CONVERT(-0.244468), DSP16_Q_CONVERT(-0.969657),
      DSP16_Q_CONVERT(0.348419), DSP16_Q_CONVERT(-0.937339), DSP16_Q_CONVERT(-0.248928), DSP16_Q_CONVERT(-0.968522),
      DSP16_Q_CONVERT(0.345541), DSP16_Q_CONVERT(-0.938404), DSP16_Q_CONVERT(-0.253382), DSP16_Q_CONVERT(-0.967366),
      DSP16_Q_CONVERT(0.342661), DSP16_Q_CONVERT(-0.939459), DSP16_Q_CONVERT(-0.257831), DSP16_Q_CONVERT(-0.966190),
      DSP16_Q_CONVERT(0.339777), DSP16_Q_CONVERT(-0.940506), DSP16_Q_CONVERT(-0.262275), DSP16_Q_CONVERT(-0.964993),
      DSP16_Q_CONVERT(0.336890), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(-0.963776),
      DSP16_Q_CONVERT(0.334000), DSP16_Q_CONVERT(-0.942573), DSP16_Q_CONVERT(-0.271145), DSP16_Q_CONVERT(-0.962538),
      DSP16_Q_CONVERT(0.331106), DSP16_Q_CONVERT(-0.943593), DSP16_Q_CONVERT(-0.275572), DSP16_Q_CONVERT(-0.961280),
      DSP16_Q_CONVERT(0.328210), DSP16_Q_CONVERT(-0.944605), DSP16_Q_CONVERT(-0.279993), DSP16_Q_CONVERT(-0.960002),
      DSP16_Q_CONVERT(0.325310), DSP16_Q_CONVERT(-0.945607), DSP16_Q_CONVERT(-0.284408), DSP16_Q_CONVERT(-0.958703),
      DSP16_Q_CONVERT(0.322408), DSP16_Q_CONVERT(-0.946601), DSP16_Q_CONVERT(-0.288816), DSP16_Q_CONVERT(-0.957385),
      DSP16_Q_CONVERT(0.319502), DSP16_Q_CONVERT(-0.947586), DSP16_Q_CONVERT(-0.293219), DSP16_Q_CONVERT(-0.956045),
      DSP16_Q_CONVERT(0.316593), DSP16_Q_CONVERT(-0.948561), DSP16_Q_CONVERT(-0.297616), DSP16_Q_CONVERT(-0.954686),
      DSP16_Q_CONVERT(0.313682), DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(-0.302006), DSP16_Q_CONVERT(-0.953306),
      DSP16_Q_CONVERT(0.310767), DSP16_Q_CONVERT(-0.950486), DSP16_Q_CONVERT(-0.306390), DSP16_Q_CONVERT(-0.951906),
      DSP16_Q_CONVERT(0.307850), DSP16_Q_CONVERT(-0.951435), DSP16_Q_CONVERT(-0.310767), DSP16_Q_CONVERT(-0.950486),
      DSP16_Q_CONVERT(0.304929), DSP16_Q_CONVERT(-0.952375), DSP16_Q_CONVERT(-0.315138), DSP16_Q_CONVERT(-0.949046),
      DSP16_Q_CONVERT(0.302006), DSP16_Q_CONVERT(-0.953306), DSP16_Q_CONVERT(-0.319502), DSP16_Q_CONVERT(-0.947586),
      DSP16_Q_CONVERT(0.299080), DSP16_Q_CONVERT(-0.954228), DSP16_Q_CONVERT(-0.323859), DSP16_Q_CONVERT(-0.946105),
      DSP16_Q_CONVERT(0.296151), DSP16_Q_CONVERT(-0.955141), DSP16_Q_CONVERT(-0.328210), DSP16_Q_CONVERT(-0.944605),
      DSP16_Q_CONVERT(0.293219), DSP16_Q_CONVERT(-0.956045), DSP16_Q_CONVERT(-0.332553), DSP16_Q_CONVERT(-0.943084),
      DSP16_Q_CONVERT(0.290285), DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.941544),
      DSP16_Q_CONVERT(0.287347), DSP16_Q_CONVERT(-0.957826), DSP16_Q_CONVERT(-0.341219), DSP16_Q_CONVERT(-0.939984),
      DSP16_Q_CONVERT(0.284408), DSP16_Q_CONVERT(-0.958703), DSP16_Q_CONVERT(-0.345541), DSP16_Q_CONVERT(-0.938404),
      DSP16_Q_CONVERT(0.281465), DSP16_Q_CONVERT(-0.959572), DSP16_Q_CONVERT(-0.349856), DSP16_Q_CONVERT(-0.936803),
      DSP16_Q_CONVERT(0.278520), DSP16_Q_CONVERT(-0.960431), DSP16_Q_CONVERT(-0.354164), DSP16_Q_CONVERT(-0.935184),
      DSP16_Q_CONVERT(0.275572), DSP16_Q_CONVERT(-0.961280), DSP16_Q_CONVERT(-0.358463), DSP16_Q_CONVERT(-0.933544),
      DSP16_Q_CONVERT(0.272621), DSP16_Q_CONVERT(-0.962121), DSP16_Q_CONVERT(-0.362756), DSP16_Q_CONVERT(-0.931884),
      DSP16_Q_CONVERT(0.269668), DSP16_Q_CONVERT(-0.962953), DSP16_Q_CONVERT(-0.367040), DSP16_Q_CONVERT(-0.930205),
      DSP16_Q_CONVERT(0.266713), DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.371317), DSP16_Q_CONVERT(-0.928506),
      DSP16_Q_CONVERT(0.263755), DSP16_Q_CONVERT(-0.964590), DSP16_Q_CONVERT(-0.375586), DSP16_Q_CONVERT(-0.926787),
      DSP16_Q_CONVERT(0.260794), DSP16_Q_CONVERT(-0.965394), DSP16_Q_CONVERT(-0.379847), DSP16_Q_CONVERT(-0.925049),
      DSP16_Q_CONVERT(0.257831), DSP16_Q_CONVERT(-0.966190), DSP16_Q_CONVERT(-0.384100), DSP16_Q_CONVERT(-0.923291),
      DSP16_Q_CONVERT(0.254866), DSP16_Q_CONVERT(-0.966976), DSP16_Q_CONVERT(-0.388345), DSP16_Q_CONVERT(-0.921514),
      DSP16_Q_CONVERT(0.251898), DSP16_Q_CONVERT(-0.967754), DSP16_Q_CONVERT(-0.392582), DSP16_Q_CONVERT(-0.919717),
      DSP16_Q_CONVERT(0.248928), DSP16_Q_CONVERT(-0.968522), DSP16_Q_CONVERT(-0.396810), DSP16_Q_CONVERT(-0.917901),
      DSP16_Q_CONVERT(0.245955), DSP16_Q_CONVERT(-0.969281), DSP16_Q_CONVERT(-0.401030), DSP16_Q_CONVERT(-0.916065),
      DSP16_Q_CONVERT(0.242980), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(-0.914210),
      DSP16_Q_CONVERT(0.240003), DSP16_Q_CONVERT(-0.970772), DSP16_Q_CONVERT(-0.409444), DSP16_Q_CONVERT(-0.912335),
      DSP16_Q_CONVERT(0.237024), DSP16_Q_CONVERT(-0.971504), DSP16_Q_CONVERT(-0.413638), DSP16_Q_CONVERT(-0.910441),
      DSP16_Q_CONVERT(0.234042), DSP16_Q_CONVERT(-0.972226), DSP16_Q_CONVERT(-0.417824), DSP16_Q_CONVERT(-0.908528),
      DSP16_Q_CONVERT(0.231058), DSP16_Q_CONVERT(-0.972940), DSP16_Q_CONVERT(-0.422000), DSP16_Q_CONVERT(-0.906596),
      DSP16_Q_CONVERT(0.228072), DSP16_Q_CONVERT(-0.973644), DSP16_Q_CONVERT(-0.426168), DSP16_Q_CONVERT(-0.904644),
      DSP16_Q_CONVERT(0.225084), DSP16_Q_CONVERT(-0.974339), DSP16_Q_CONVERT(-0.430326), DSP16_Q_CONVERT(-0.902673),
      DSP16_Q_CONVERT(0.222094), DSP16_Q_CONVERT(-0.975025), DSP16_Q_CONVERT(-0.434476), DSP16_Q_CONVERT(-0.900683),
      DSP16_Q_CONVERT(0.219101), DSP16_Q_CONVERT(-0.975702), DSP16_Q_CONVERT(-0.438616), DSP16_Q_CONVERT(-0.898674),
      DSP16_Q_CONVERT(0.216107), DSP16_Q_CONVERT(-0.976370), DSP16_Q_CONVERT(-0.442747), DSP16_Q_CONVERT(-0.896646),
      DSP16_Q_CONVERT(0.213110), DSP16_Q_CONVERT(-0.977028), DSP16_Q_CONVERT(-0.446869), DSP16_Q_CONVERT(-0.894599),
      DSP16_Q_CONVERT(0.210112), DSP16_Q_CONVERT(-0.977677), DSP16_Q_CONVERT(-0.450981), DSP16_Q_CONVERT(-0.892534),
      DSP16_Q_CONVERT(0.207111), DSP16_Q_CONVERT(-0.978317), DSP16_Q_CONVERT(-0.455084), DSP16_Q_CONVERT(-0.890449),
      DSP16_Q_CONVERT(0.204109), DSP16_Q_CONVERT(-0.978948), DSP16_Q_CONVERT(-0.459177), DSP16_Q_CONVERT(-0.888345),
      DSP16_Q_CONVERT(0.201105), DSP16_Q_CONVERT(-0.979570), DSP16_Q_CONVERT(-0.463260), DSP16_Q_CONVERT(-0.886223),
      DSP16_Q_CONVERT(0.198098), DSP16_Q_CONVERT(-0.980182), DSP16_Q_CONVERT(-0.467333), DSP16_Q_CONVERT(-0.884081),
      DSP16_Q_CONVERT(0.195090), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.881921),
      DSP16_Q_CONVERT(0.192080), DSP16_Q_CONVERT(-0.981379), DSP16_Q_CONVERT(-0.475450), DSP16_Q_CONVERT(-0.879743),
      DSP16_Q_CONVERT(0.189069), DSP16_Q_CONVERT(-0.981964), DSP16_Q_CONVERT(-0.479494), DSP16_Q_CONVERT(-0.877545),
      DSP16_Q_CONVERT(0.186055), DSP16_Q_CONVERT(-0.982539), DSP16_Q_CONVERT(-0.483527), DSP16_Q_CONVERT(-0.875329),
      DSP16_Q_CONVERT(0.183040), DSP16_Q_CONVERT(-0.983105), DSP16_Q_CONVERT(-0.487550), DSP16_Q_CONVERT(-0.873095),
      DSP16_Q_CONVERT(0.180023), DSP16_Q_CONVERT(-0.983662), DSP16_Q_CONVERT(-0.491563), DSP16_Q_CONVERT(-0.870842),
      DSP16_Q_CONVERT(0.177004), DSP16_Q_CONVERT(-0.984210), DSP16_Q_CONVERT(-0.495565), DSP16_Q_CONVERT(-0.868571),
      DSP16_Q_CONVERT(0.173984), DSP16_Q_CONVERT(-0.984749), DSP16_Q_CONVERT(-0.499557), DSP16_Q_CONVERT(-0.866281),
      DSP16_Q_CONVERT(0.170962), DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(-0.503538), DSP16_Q_CONVERT(-0.863973),
      DSP16_Q_CONVERT(0.167938), DSP16_Q_CONVERT(-0.985798), DSP16_Q_CONVERT(-0.507509), DSP16_Q_CONVERT(-0.861646),
      DSP16_Q_CONVERT(0.164913), DSP16_Q_CONVERT(-0.986308), DSP16_Q_CONVERT(-0.511469), DSP16_Q_CONVERT(-0.859302),
      DSP16_Q_CONVERT(0.161886), DSP16_Q_CONVERT(-0.986809), DSP16_Q_CONVERT(-0.515418), DSP16_Q_CONVERT(-0.856939),
      DSP16_Q_CONVERT(0.158858), DSP16_Q_CONVERT(-0.987301), DSP16_Q_CONVERT(-0.519356), DSP16_Q_CONVERT(-0.854558),
      DSP16_Q_CONVERT(0.155828), DSP16_Q_CONVERT(-0.987784), DSP16_Q_CONVERT(-0.523283), DSP16_Q_CONVERT(-0.852159),
      DSP16_Q_CONVERT(0.152797), DSP16_Q_CONVERT(-0.988258), DSP16_Q_CONVERT(-0.527199), DSP16_Q_CONVERT(-0.849742),
      DSP16_Q_CONVERT(0.149765), DSP16_Q_CONVERT(-0.988722), DSP16_Q_CONVERT(-0.531104), DSP16_Q_CONVERT(-0.847307),
      DSP16_Q_CONVERT(0.146730), DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(-0.844854),
      DSP16_Q_CONVERT(0.143695), DSP16_Q_CONVERT(-0.989622), DSP16_Q_CONVERT(-0.538880), DSP16_Q_CONVERT(-0.842383),
      DSP16_Q_CONVERT(0.140658), DSP16_Q_CONVERT(-0.990058), DSP16_Q_CONVERT(-0.542751), DSP16_Q_CONVERT(-0.839894),
      DSP16_Q_CONVERT(0.137620), DSP16_Q_CONVERT(-0.990485), DSP16_Q_CONVERT(-0.546610), DSP16_Q_CONVERT(-0.837387),
      DSP16_Q_CONVERT(0.134581), DSP16_Q_CONVERT(-0.990903), DSP16_Q_CONVERT(-0.550458), DSP16_Q_CONVERT(-0.834863),
      DSP16_Q_CONVERT(0.131540), DSP16_Q_CONVERT(-0.991311), DSP16_Q_CONVERT(-0.554294), DSP16_Q_CONVERT(-0.832321),
      DSP16_Q_CONVERT(0.128498), DSP16_Q_CONVERT(-0.991710), DSP16_Q_CONVERT(-0.558119), DSP16_Q_CONVERT(-0.829761),
      DSP16_Q_CONVERT(0.125455), DSP16_Q_CONVERT(-0.992099), DSP16_Q_CONVERT(-0.561931), DSP16_Q_CONVERT(-0.827184),
      DSP16_Q_CONVERT(0.122411), DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.565732), DSP16_Q_CONVERT(-0.824589),
      DSP16_Q_CONVERT(0.119365), DSP16_Q_CONVERT(-0.992850), DSP16_Q_CONVERT(-0.569521), DSP16_Q_CONVERT(-0.821977),
      DSP16_Q_CONVERT(0.116319), DSP16_Q_CONVERT(-0.993212), DSP16_Q_CONVERT(-0.573297), DSP16_Q_CONVERT(-0.819348),
      DSP16_Q_CONVERT(0.113271), DSP16_Q_CONVERT(-0.993564), DSP16_Q_CONVERT(-0.577062), DSP16_Q_CONVERT(-0.816701),
      DSP16_Q_CONVERT(0.110222), DSP16_Q_CONVERT(-0.993907), DSP16_Q_CONVERT(-0.580814), DSP16_Q_CONVERT(-0.814036),
      DSP16_Q_CONVERT(0.107172), DSP16_Q_CONVERT(-0.994240), DSP16_Q_CONVERT(-0.584554), DSP16_Q_CONVERT(-0.811355),
      DSP16_Q_CONVERT(0.104122), DSP16_Q_CONVERT(-0.994565), DSP16_Q_CONVERT(-0.588282), DSP16_Q_CONVERT(-0.808656),
      DSP16_Q_CONVERT(0.101070), DSP16_Q_CONVERT(-0.994879), DSP16_Q_CONVERT(-0.591997), DSP16_Q_CONVERT(-0.805940),
      DSP16_Q_CONVERT(0.098017), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.803208),
      DSP16_Q_CONVERT(0.094963), DSP16_Q_CONVERT(-0.995481), DSP16_Q_CONVERT(-0.599389), DSP16_Q_CONVERT(-0.800458),
      DSP16_Q_CONVERT(0.091909), DSP16_Q_CONVERT(-0.995767), DSP16_Q_CONVERT(-0.603067), DSP16_Q_CONVERT(-0.797691),
      DSP16_Q_CONVERT(0.088854), DSP16_Q_CONVERT(-0.996045), DSP16_Q_CONVERT(-0.606731), DSP16_Q_CONVERT(-0.794907),
      DSP16_Q_CONVERT(0.085797), DSP16_Q_CONVERT(-0.996313), DSP16_Q_CONVERT(-0.610383), DSP16_Q_CONVERT(-0.792107),
      DSP16_Q_CONVERT(0.082740), DSP16_Q_CONVERT(-0.996571), DSP16_Q_CONVERT(-0.614022), DSP16_Q_CONVERT(-0.789289),
      DSP16_Q_CONVERT(0.079682), DSP16_Q_CONVERT(-0.996820), DSP16_Q_CONVERT(-0.617647), DSP16_Q_CONVERT(-0.786455),
      DSP16_Q_CONVERT(0.076624), DSP16_Q_CONVERT(-0.997060), DSP16_Q_CONVERT(-0.621260), DSP16_Q_CONVERT(-0.783605),
      DSP16_Q_CONVERT(0.073565), DSP16_Q_CONVERT(-0.997290), DSP16_Q_CONVERT(-0.624859), DSP16_Q_CONVERT(-0.780737),
      DSP16_Q_CONVERT(0.070505), DSP16_Q_CONVERT(-0.997511), DSP16_Q_CONVERT(-0.628446), DSP16_Q_CONVERT(-0.777853),
      DSP16_Q_CONVERT(0.067444), DSP16_Q_CONVERT(-0.997723), DSP16_Q_CONVERT(-0.632019), DSP16_Q_CONVERT(-0.774953),
      DSP16_Q_CONVERT(0.064383), DSP16_Q_CONVERT(-0.997925), DSP16_Q_CONVERT(-0.635578), DSP16_Q_CONVERT(-0.772036),
      DSP16_Q_CONVERT(0.061321), DSP16_Q_CONVERT(-0.998118), DSP16_Q_CONVERT(-0.639124), DSP16_Q_CONVERT(-0.769103),
      DSP16_Q_CONVERT(0.058258), DSP16_Q_CONVERT(-0.998302), DSP16_Q_CONVERT(-0.642657), DSP16_Q_CONVERT(-0.766154),
      DSP16_Q_CONVERT(0.055195), DSP16_Q_CONVERT(-0.998476), DSP16_Q_CONVERT(-0.646176), DSP16_Q_CONVERT(-0.763188),
      DSP16_Q_CONVERT(0.052132), DSP16_Q_CONVERT(-0.998640), DSP16_Q_CONVERT(-0.649681), DSP16_Q_CONVERT(-0.760207),
      DSP16_Q_CONVERT(0.049068), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(-0.757209),
      DSP16_Q_CONVERT(0.046003), DSP16_Q_CONVERT(-0.998941), DSP16_Q_CONVERT(-0.656651), DSP16_Q_CONVERT(-0.754195),
      DSP16_Q_CONVERT(0.042938), DSP16_Q_CONVERT(-0.999078), DSP16_Q_CONVERT(-0.660114), DSP16_Q_CONVERT(-0.751165),
      DSP16_Q_CONVERT(0.039873), DSP16_Q_CONVERT(-0.999205), DSP16_Q_CONVERT(-0.663564), DSP16_Q_CONVERT(-0.748119),
      DSP16_Q_CONVERT(0.036807), DSP16_Q_CONVERT(-0.999322), DSP16_Q_CONVERT(-0.667000), DSP16_Q_CONVERT(-0.745058),
      DSP16_Q_CONVERT(0.033741), DSP16_Q_CONVERT(-0.999431), DSP16_Q_CONVERT(-0.670422), DSP16_Q_CONVERT(-0.741980),
      DSP16_Q_CONVERT(0.030675), DSP16_Q_CONVERT(-0.999529), DSP16_Q_CONVERT(-0.673829), DSP16_Q_CONVERT(-0.738887),
      DSP16_Q_CONVERT(0.027608), DSP16_Q_CONVERT(-0.999619), DSP16_Q_CONVERT(-0.677222), DSP16_Q_CONVERT(-0.735779),
      DSP16_Q_CONVERT(0.024541), DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(-0.680601), DSP16_Q_CONVERT(-0.732654),
      DSP16_Q_CONVERT(0.021474), DSP16_Q_CONVERT(-0.999769), DSP16_Q_CONVERT(-0.683965), DSP16_Q_CONVERT(-0.729514),
      DSP16_Q_CONVERT(0.018407), DSP16_Q_CONVERT(-0.999831), DSP16_Q_CONVERT(-0.687315), DSP16_Q_CONVERT(-0.726359),
      DSP16_Q_CONVERT(0.015339), DSP16_Q_CONVERT(-0.999882), DSP16_Q_CONVERT(-0.690651), DSP16_Q_CONVERT(-0.723188),
      DSP16_Q_CONVERT(0.012272), DSP16_Q_CONVERT(-0.999925), DSP16_Q_CONVERT(-0.693971), DSP16_Q_CONVERT(-0.720003),
      DSP16_Q_CONVERT(0.009204), DSP16_Q_CONVERT(-0.999958), DSP16_Q_CONVERT(-0.697278), DSP16_Q_CONVERT(-0.716801),
      DSP16_Q_CONVERT(0.006136), DSP16_Q_CONVERT(-0.999981), DSP16_Q_CONVERT(-0.700569), DSP16_Q_CONVERT(-0.713585),
      DSP16_Q_CONVERT(0.003068), DSP16_Q_CONVERT(-0.999995), DSP16_Q_CONVERT(-0.703845), DSP16_Q_CONVERT(-0.710353),
      DSP16_Q_CONVERT(0.000000), DSP16_Q_CONVERT(-1.000000), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.707107),
      DSP16_Q_CONVERT(-0.003068), DSP16_Q_CONVERT(-0.999995), DSP16_Q_CONVERT(-0.710353), DSP16_Q_CONVERT(-0.703845),
      DSP16_Q_CONVERT(-0.006136), DSP16_Q_CONVERT(-0.999981), DSP16_Q_CONVERT(-0.713585), DSP16_Q_CONVERT(-0.700569),
      DSP16_Q_CONVERT(-0.009204), DSP16_Q_CONVERT(-0.999958), DSP16_Q_CONVERT(-0.716801), DSP16_Q_CONVERT(-0.697278),
      DSP16_Q_CONVERT(-0.012272), DSP16_Q_CONVERT(-0.999925), DSP16_Q_CONVERT(-0.720003), DSP16_Q_CONVERT(-0.693971),
      DSP16_Q_CONVERT(-0.015339), DSP16_Q_CONVERT(-0.999882), DSP16_Q_CONVERT(-0.723188), DSP16_Q_CONVERT(-0.690651),
      DSP16_Q_CONVERT(-0.018407), DSP16_Q_CONVERT(-0.999831), DSP16_Q_CONVERT(-0.726359), DSP16_Q_CONVERT(-0.687315),
      DSP16_Q_CONVERT(-0.021474), DSP16_Q_CONVERT(-0.999769), DSP16_Q_CONVERT(-0.729514), DSP16_Q_CONVERT(-0.683965),
      DSP16_Q_CONVERT(-0.024541), DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(-0.732654), DSP16_Q_CONVERT(-0.680601),
      DSP16_Q_CONVERT(-0.027608), DSP16_Q_CONVERT(-0.999619), DSP16_Q_CONVERT(-0.735779), DSP16_Q_CONVERT(-0.677222),
      DSP16_Q_CONVERT(-0.030675), DSP16_Q_CONVERT(-0.999529), DSP16_Q_CONVERT(-0.738887), DSP16_Q_CONVERT(-0.673829),
      DSP16_Q_CONVERT(-0.033741), DSP16_Q_CONVERT(-0.999431), DSP16_Q_CONVERT(-0.741980), DSP16_Q_CONVERT(-0.670422),
      DSP16_Q_CONVERT(-0.036807), DSP16_Q_CONVERT(-0.999322), DSP16_Q_CONVERT(-0.745058), DSP16_Q_CONVERT(-0.667000),
      DSP16_Q_CONVERT(-0.039873), DSP16_Q_CONVERT(-0.999205), DSP16_Q_CONVERT(-0.748119), DSP16_Q_CONVERT(-0.663564),
      DSP16_Q_CONVERT(-0.042938), DSP16_Q_CONVERT(-0.999078), DSP16_Q_CONVERT(-0.751165), DSP16_Q_CONVERT(-0.660114),
      DSP16_Q_CONVERT(-0.046003), DSP16_Q_CONVERT(-0.998941), DSP16_Q_CONVERT(-0.754195), DSP16_Q_CONVERT(-0.656651),
      DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(-0.653173),
      DSP16_Q_CONVERT(-0.052132), DSP16_Q_CONVERT(-0.998640), DSP16_Q_CONVERT(-0.760207), DSP16_Q_CONVERT(-0.649681),
      DSP16_Q_CONVERT(-0.055195), DSP16_Q_CONVERT(-0.998476), DSP16_Q_CONVERT(-0.763188), DSP16_Q_CONVERT(-0.646176),
      DSP16_Q_CONVERT(-0.058258), DSP16_Q_CONVERT(-0.998302), DSP16_Q_CONVERT(-0.766154), DSP16_Q_CONVERT(-0.642657),
      DSP16_Q_CONVERT(-0.061321), DSP16_Q_CONVERT(-0.998118), DSP16_Q_CONVERT(-0.769103), DSP16_Q_CONVERT(-0.639124),
      DSP16_Q_CONVERT(-0.064383), DSP16_Q_CONVERT(-0.997925), DSP16_Q_CONVERT(-0.772036), DSP16_Q_CONVERT(-0.635578),
      DSP16_Q_CONVERT(-0.067444), DSP16_Q_CONVERT(-0.997723), DSP16_Q_CONVERT(-0.774953), DSP16_Q_CONVERT(-0.632019),
      DSP16_Q_CONVERT(-0.070505), DSP16_Q_CONVERT(-0.997511), DSP16_Q_CONVERT(-0.777853), DSP16_Q_CONVERT(-0.628446),
      DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(-0.997290), DSP16_Q_CONVERT(-0.780737), DSP16_Q_CONVERT(-0.624859),
      DSP16_Q_CONVERT(-0.076624), DSP16_Q_CONVERT(-0.997060), DSP16_Q_CONVERT(-0.783605), DSP16_Q_CONVERT(-0.621260),
      DSP16_Q_CONVERT(-0.079682), DSP16_Q_CONVERT(-0.996820), DSP16_Q_CONVERT(-0.786455), DSP16_Q_CONVERT(-0.617647),
      DSP16_Q_CONVERT(-0.082740), DSP16_Q_CONVERT(-0.996571), DSP16_Q_CONVERT(-0.789289), DSP16_Q_CONVERT(-0.614022),
      DSP16_Q_CONVERT(-0.085797), DSP16_Q_CONVERT(-0.996313), DSP16_Q_CONVERT(-0.792107), DSP16_Q_CONVERT(-0.610383),
      DSP16_Q_CONVERT(-0.088854), DSP16_Q_CONVERT(-0.996045), DSP16_Q_CONVERT(-0.794907), DSP16_Q_CONVERT(-0.606731),
      DSP16_Q_CONVERT(-0.091909), DSP16_Q_CONVERT(-0.995767), DSP16_Q_CONVERT(-0.797691), DSP16_Q_CONVERT(-0.603067),
      DSP16_Q_CONVERT(-0.094963), DSP16_Q_CONVERT(-0.995481), DSP16_Q_CONVERT(-0.800458), DSP16_Q_CONVERT(-0.599389),
      DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.595699),
      DSP16_Q_CONVERT(-0.101070), DSP16_Q_CONVERT(-0.994879), DSP16_Q_CONVERT(-0.805940), DSP16_Q_CONVERT(-0.591997),
      DSP16_Q_CONVERT(-0.104122), DSP16_Q_CONVERT(-0.994565), DSP16_Q_CONVERT(-0.808656), DSP16_Q_CONVERT(-0.588282),
      DSP16_Q_CONVERT(-0.107172), DSP16_Q_CONVERT(-0.994240), DSP16_Q_CONVERT(-0.811355), DSP16_Q_CONVERT(-0.584554),
      DSP16_Q_CONVERT(-0.110222), DSP16_Q_CONVERT(-0.993907), DSP16_Q_CONVERT(-0.814036), DSP16_Q_CONVERT(-0.580814),
      DSP16_Q_CONVERT(-0.113271), DSP16_Q_CONVERT(-0.993564), DSP16_Q_CONVERT(-0.816701), DSP16_Q_CONVERT(-0.577062),
      DSP16_Q_CONVERT(-0.116319), DSP16_Q_CONVERT(-0.993212), DSP16_Q_CONVERT(-0.819348), DSP16_Q_CONVERT(-0.573297),
      DSP16_Q_CONVERT(-0.119365), DSP16_Q_CONVERT(-0.992850), DSP16_Q_CONVERT(-0.821977), DSP16_Q_CONVERT(-0.569521),
      DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.824589), DSP16_Q_CONVERT(-0.565732),
      DSP16_Q_CONVERT(-0.125455), DSP16_Q_CONVERT(-0.992099), DSP16_Q_CONVERT(-0.827184), DSP16_Q_CONVERT(-0.561931),
      DSP16_Q_CONVERT(-0.128498), DSP16_Q_CONVERT(-0.991710), DSP16_Q_CONVERT(-0.829761), DSP16_Q_CONVERT(-0.558119),
      DSP16_Q_CONVERT(-0.131540), DSP16_Q_CONVERT(-0.991311), DSP16_Q_CONVERT(-0.832321), DSP16_Q_CONVERT(-0.554294),
      DSP16_Q_CONVERT(-0.134581), DSP16_Q_CONVERT(-0.990903), DSP16_Q_CONVERT(-0.834863), DSP16_Q_CONVERT(-0.550458),
      DSP16_Q_CONVERT(-0.137620), DSP16_Q_CONVERT(-0.990485), DSP16_Q_CONVERT(-0.837387), DSP16_Q_CONVERT(-0.546610),
      DSP16_Q_CONVERT(-0.140658), DSP16_Q_CONVERT(-0.990058), DSP16_Q_CONVERT(-0.839894), DSP16_Q_CONVERT(-0.542751),
      DSP16_Q_CONVERT(-0.143695), DSP16_Q_CONVERT(-0.989622), DSP16_Q_CONVERT(-0.842383), DSP16_Q_CONVERT(-0.538880),
      DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(-0.534998),
      DSP16_Q_CONVERT(-0.149765), DSP16_Q_CONVERT(-0.988722), DSP16_Q_CONVERT(-0.847307), DSP16_Q_CONVERT(-0.531104),
      DSP16_Q_CONVERT(-0.152797), DSP16_Q_CONVERT(-0.988258), DSP16_Q_CONVERT(-0.849742), DSP16_Q_CONVERT(-0.527199),
      DSP16_Q_CONVERT(-0.155828), DSP16_Q_CONVERT(-0.987784), DSP16_Q_CONVERT(-0.852159), DSP16_Q_CONVERT(-0.523283),
      DSP16_Q_CONVERT(-0.158858), DSP16_Q_CONVERT(-0.987301), DSP16_Q_CONVERT(-0.854558), DSP16_Q_CONVERT(-0.519356),
      DSP16_Q_CONVERT(-0.161886), DSP16_Q_CONVERT(-0.986809), DSP16_Q_CONVERT(-0.856939), DSP16_Q_CONVERT(-0.515418),
      DSP16_Q_CONVERT(-0.164913), DSP16_Q_CONVERT(-0.986308), DSP16_Q_CONVERT(-0.859302), DSP16_Q_CONVERT(-0.511469),
      DSP16_Q_CONVERT(-0.167938), DSP16_Q_CONVERT(-0.985798), DSP16_Q_CONVERT(-0.861646), DSP16_Q_CONVERT(-0.507509),
      DSP16_Q_CONVERT(-0.170962), DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(-0.863973), DSP16_Q_CONVERT(-0.503538),
      DSP16_Q_CONVERT(-0.173984), DSP16_Q_CONVERT(-0.984749), DSP16_Q_CONVERT(-0.866281), DSP16_Q_CONVERT(-0.499557),
      DSP16_Q_CONVERT(-0.177004), DSP16_Q_CONVERT(-0.984210), DSP16_Q_CONVERT(-0.868571), DSP16_Q_CONVERT(-0.495565),
      DSP16_Q_CONVERT(-0.180023), DSP16_Q_CONVERT(-0.983662), DSP16_Q_CONVERT(-0.870842), DSP16_Q_CONVERT(-0.491563),
      DSP16_Q_CONVERT(-0.183040), DSP16_Q_CONVERT(-0.983105), DSP16_Q_CONVERT(-0.873095), DSP16_Q_CONVERT(-0.487550),
      DSP16_Q_CONVERT(-0.186055), DSP16_Q_CONVERT(-0.982539), DSP16_Q_CONVERT(-0.875329), DSP16_Q_CONVERT(-0.483527),
      DSP16_Q_CONVERT(-0.189069), DSP16_Q_CONVERT(-0.981964), DSP16_Q_CONVERT(-0.877545), DSP16_Q_CONVERT(-0.479494),
      DSP16_Q_CONVERT(-0.192080), DSP16_Q_CONVERT(-0.981379), DSP16_Q_CONVERT(-0.879743), DSP16_Q_CONVERT(-0.475450),
      DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.471397),
      DSP16_Q_CONVERT(-0.198098), DSP16_Q_CONVERT(-0.980182), DSP16_Q_CONVERT(-0.884081), DSP16_Q_CONVERT(-0.467333),
      DSP16_Q_CONVERT(-0.201105), DSP16_Q_CONVERT(-0.979570), DSP16_Q_CONVERT(-0.886223), DSP16_Q_CONVERT(-0.463260),
      DSP16_Q_CONVERT(-0.204109), DSP16_Q_CONVERT(-0.978948), DSP16_Q_CONVERT(-0.888345), DSP16_Q_CONVERT(-0.459177),
      DSP16_Q_CONVERT(-0.207111), DSP16_Q_CONVERT(-0.978317), DSP16_Q_CONVERT(-0.890449), DSP16_Q_CONVERT(-0.455084),
      DSP16_Q_CONVERT(-0.210112), DSP16_Q_CONVERT(-0.977677), DSP16_Q_CONVERT(-0.892534), DSP16_Q_CONVERT(-0.450981),
      DSP16_Q_CONVERT(-0.213110), DSP16_Q_CONVERT(-0.977028), DSP16_Q_CONVERT(-0.894599), DSP16_Q_CONVERT(-0.446869),
      DSP16_Q_CONVERT(-0.216107), DSP16_Q_CONVERT(-0.976370), DSP16_Q_CONVERT(-0.896646), DSP16_Q_CONVERT(-0.442747),
      DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(-0.975702), DSP16_Q_CONVERT(-0.898674), DSP16_Q_CONVERT(-0.438616),
      DSP16_Q_CONVERT(-0.222094), DSP16_Q_CONVERT(-0.975025), DSP16_Q_CONVERT(-0.900683), DSP16_Q_CONVERT(-0.434476),
      DSP16_Q_CONVERT(-0.225084), DSP16_Q_CONVERT(-0.974339), DSP16_Q_CONVERT(-0.902673), DSP16_Q_CONVERT(-0.430326),
      DSP16_Q_CONVERT(-0.228072), DSP16_Q_CONVERT(-0.973644), DSP16_Q_CONVERT(-0.904644), DSP16_Q_CONVERT(-0.426168),
      DSP16_Q_CONVERT(-0.231058), DSP16_Q_CONVERT(-0.972940), DSP16_Q_CONVERT(-0.906596), DSP16_Q_CONVERT(-0.422000),
      DSP16_Q_CONVERT(-0.234042), DSP16_Q_CONVERT(-0.972226), DSP16_Q_CONVERT(-0.908528), DSP16_Q_CONVERT(-0.417824),
      DSP16_Q_CONVERT(-0.237024), DSP16_Q_CONVERT(-0.971504), DSP16_Q_CONVERT(-0.910441), DSP16_Q_CONVERT(-0.413638),
      DSP16_Q_CONVERT(-0.240003), DSP16_Q_CONVERT(-0.970772), DSP16_Q_CONVERT(-0.912335), DSP16_Q_CONVERT(-0.409444),
      DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.405241),
      DSP16_Q_CONVERT(-0.245955), DSP16_Q_CONVERT(-0.969281), DSP16_Q_CONVERT(-0.916065), DSP16_Q_CONVERT(-0.401030),
      DSP16_Q_CONVERT(-0.248928), DSP16_Q_CONVERT(-0.968522), DSP16_Q_CONVERT(-0.917901), DSP16_Q_CONVERT(-0.396810),
      DSP16_Q_CONVERT(-0.251898), DSP16_Q_CONVERT(-0.967754), DSP16_Q_CONVERT(-0.919717), DSP16_Q_CONVERT(-0.392582),
      DSP16_Q_CONVERT(-0.254866), DSP16_Q_CONVERT(-0.966976), DSP16_Q_CONVERT(-0.921514), DSP16_Q_CONVERT(-0.388345),
      DSP16_Q_CONVERT(-0.257831), DSP16_Q_CONVERT(-0.966190), DSP16_Q_CONVERT(-0.923291), DSP16_Q_CONVERT(-0.384100),
      DSP16_Q_CONVERT(-0.260794), DSP16_Q_CONVERT(-0.965394), DSP16_Q_CONVERT(-0.925049), DSP16_Q_CONVERT(-0.379847),
      DSP16_Q_CONVERT(-0.263755), DSP16_Q_CONVERT(-0.964590), DSP16_Q_CONVERT(-0.926787), DSP16_Q_CONVERT(-0.375586),
      DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.928506), DSP16_Q_CONVERT(-0.371317),
      DSP16_Q_CONVERT(-0.269668), DSP16_Q_CONVERT(-0.962953), DSP16_Q_CONVERT(-0.930205), DSP16_Q_CONVERT(-0.367040),
      DSP16_Q_CONVERT(-0.272621), DSP16_Q_CONVERT(-0.962121), DSP16_Q_CONVERT(-0.931884), DSP16_Q_CONVERT(-0.362756),
      DSP16_Q_CONVERT(-0.275572), DSP16_Q_CONVERT(-0.961280), DSP16_Q_CONVERT(-0.933544), DSP16_Q_CONVERT(-0.358463),
      DSP16_Q_CONVERT(-0.278520), DSP16_Q_CONVERT(-0.960431), DSP16_Q_CONVERT(-0.935184), DSP16_Q_CONVERT(-0.354164),
      DSP16_Q_CONVERT(-0.281465), DSP16_Q_CONVERT(-0.959572), DSP16_Q_CONVERT(-0.936803), DSP16_Q_CONVERT(-0.349856),
      DSP16_Q_CONVERT(-0.284408), DSP16_Q_CONVERT(-0.958703), DSP16_Q_CONVERT(-0.938404), DSP16_Q_CONVERT(-0.345541),
      DSP16_Q_CONVERT(-0.287347), DSP16_Q_CONVERT(-0.957826), DSP16_Q_CONVERT(-0.939984), DSP16_Q_CONVERT(-0.341219),
      DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.336890),
      DSP16_Q_CONVERT(-0.293219), DSP16_Q_CONVERT(-0.956045), DSP16_Q_CONVERT(-0.943084), DSP16_Q_CONVERT(-0.332553),
      DSP16_Q_CONVERT(-0.296151), DSP16_Q_CONVERT(-0.955141), DSP16_Q_CONVERT(-0.944605), DSP16_Q_CONVERT(-0.328210),
      DSP16_Q_CONVERT(-0.299080), DSP16_Q_CONVERT(-0.954228), DSP16_Q_CONVERT(-0.946105), DSP16_Q_CONVERT(-0.323859),
      DSP16_Q_CONVERT(-0.302006), DSP16_Q_CONVERT(-0.953306), DSP16_Q_CONVERT(-0.947586), DSP16_Q_CONVERT(-0.319502),
      DSP16_Q_CONVERT(-0.304929), DSP16_Q_CONVERT(-0.952375), DSP16_Q_CONVERT(-0.949046), DSP16_Q_CONVERT(-0.315138),
      DSP16_Q_CONVERT(-0.307850), DSP16_Q_CONVERT(-0.951435), DSP16_Q_CONVERT(-0.950486), DSP16_Q_CONVERT(-0.310767),
      DSP16_Q_CONVERT(-0.310767), DSP16_Q_CONVERT(-0.950486), DSP16_Q_CONVERT(-0.951906), DSP16_Q_CONVERT(-0.306390),
      DSP16_Q_CONVERT(-0.313682), DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(-0.953306), DSP16_Q_CONVERT(-0.302006),
      DSP16_Q_CONVERT(-0.316593), DSP16_Q_CONVERT(-0.948561), DSP16_Q_CONVERT(-0.954686), DSP16_Q_CONVERT(-0.297616),
      DSP16_Q_CONVERT(-0.319502), DSP16_Q_CONVERT(-0.947586), DSP16_Q_CONVERT(-0.956045), DSP16_Q_CONVERT(-0.293219),
      DSP16_Q_CONVERT(-0.322408), DSP16_Q_CONVERT(-0.946601), DSP16_Q_CONVERT(-0.957385), DSP16_Q_CONVERT(-0.288816),
      DSP16_Q_CONVERT(-0.325310), DSP16_Q_CONVERT(-0.945607), DSP16_Q_CONVERT(-0.958703), DSP16_Q_CONVERT(-0.284408),
      DSP16_Q_CONVERT(-0.328210), DSP16_Q_CONVERT(-0.944605), DSP16_Q_CONVERT(-0.960002), DSP16_Q_CONVERT(-0.279993),
      DSP16_Q_CONVERT(-0.331106), DSP16_Q_CONVERT(-0.943593), DSP16_Q_CONVERT(-0.961280), DSP16_Q_CONVERT(-0.275572),
      DSP16_Q_CONVERT(-0.334000), DSP16_Q_CONVERT(-0.942573), DSP16_Q_CONVERT(-0.962538), DSP16_Q_CONVERT(-0.271145),
      DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.266713),
      DSP16_Q_CONVERT(-0.339777), DSP16_Q_CONVERT(-0.940506), DSP16_Q_CONVERT(-0.964993), DSP16_Q_CONVERT(-0.262275),
      DSP16_Q_CONVERT(-0.342661), DSP16_Q_CONVERT(-0.939459), DSP16_Q_CONVERT(-0.966190), DSP16_Q_CONVERT(-0.257831),
      DSP16_Q_CONVERT(-0.345541), DSP16_Q_CONVERT(-0.938404), DSP16_Q_CONVERT(-0.967366), DSP16_Q_CONVERT(-0.253382),
      DSP16_Q_CONVERT(-0.348419), DSP16_Q_CONVERT(-0.937339), DSP16_Q_CONVERT(-0.968522), DSP16_Q_CONVERT(-0.248928),
      DSP16_Q_CONVERT(-0.351293), DSP16_Q_CONVERT(-0.936266), DSP16_Q_CONVERT(-0.969657), DSP16_Q_CONVERT(-0.244468),
      DSP16_Q_CONVERT(-0.354164), DSP16_Q_CONVERT(-0.935184), DSP16_Q_CONVERT(-0.970772), DSP16_Q_CONVERT(-0.240003),
      DSP16_Q_CONVERT(-0.357031), DSP16_Q_CONVERT(-0.934093), DSP16_Q_CONVERT(-0.971866), DSP16_Q_CONVERT(-0.235533),
      DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(-0.932993), DSP16_Q_CONVERT(-0.972940), DSP16_Q_CONVERT(-0.231058),
      DSP16_Q_CONVERT(-0.362756), DSP16_Q_CONVERT(-0.931884), DSP16_Q_CONVERT(-0.973993), DSP16_Q_CONVERT(-0.226578),
      DSP16_Q_CONVERT(-0.365613), DSP16_Q_CONVERT(-0.930767), DSP16_Q_CONVERT(-0.975025), DSP16_Q_CONVERT(-0.222094),
      DSP16_Q_CONVERT(-0.368467), DSP16_Q_CONVERT(-0.929641), DSP16_Q_CONVERT(-0.976037), DSP16_Q_CONVERT(-0.217604),
      DSP16_Q_CONVERT(-0.371317), DSP16_Q_CONVERT(-0.928506), DSP16_Q_CONVERT(-0.977028), DSP16_Q_CONVERT(-0.213110),
      DSP16_Q_CONVERT(-0.374164), DSP16_Q_CONVERT(-0.927363), DSP16_Q_CONVERT(-0.977999), DSP16_Q_CONVERT(-0.208612),
      DSP16_Q_CONVERT(-0.377007), DSP16_Q_CONVERT(-0.926210), DSP16_Q_CONVERT(-0.978948), DSP16_Q_CONVERT(-0.204109),
      DSP16_Q_CONVERT(-0.379847), DSP16_Q_CONVERT(-0.925049), DSP16_Q_CONVERT(-0.979877), DSP16_Q_CONVERT(-0.199602),
      DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.195090),
      DSP16_Q_CONVERT(-0.385516), DSP16_Q_CONVERT(-0.922701), DSP16_Q_CONVERT(-0.981673), DSP16_Q_CONVERT(-0.190575),
      DSP16_Q_CONVERT(-0.388345), DSP16_Q_CONVERT(-0.921514), DSP16_Q_CONVERT(-0.982539), DSP16_Q_CONVERT(-0.186055),
      DSP16_Q_CONVERT(-0.391170), DSP16_Q_CONVERT(-0.920318), DSP16_Q_CONVERT(-0.983385), DSP16_Q_CONVERT(-0.181532),
      DSP16_Q_CONVERT(-0.393992), DSP16_Q_CONVERT(-0.919114), DSP16_Q_CONVERT(-0.984210), DSP16_Q_CONVERT(-0.177004),
      DSP16_Q_CONVERT(-0.396810), DSP16_Q_CONVERT(-0.917901), DSP16_Q_CONVERT(-0.985014), DSP16_Q_CONVERT(-0.172473),
      DSP16_Q_CONVERT(-0.399624), DSP16_Q_CONVERT(-0.916679), DSP16_Q_CONVERT(-0.985798), DSP16_Q_CONVERT(-0.167938),
      DSP16_Q_CONVERT(-0.402435), DSP16_Q_CONVERT(-0.915449), DSP16_Q_CONVERT(-0.986560), DSP16_Q_CONVERT(-0.163400),
      DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.987301), DSP16_Q_CONVERT(-0.158858),
      DSP16_Q_CONVERT(-0.408044), DSP16_Q_CONVERT(-0.912962), DSP16_Q_CONVERT(-0.988022), DSP16_Q_CONVERT(-0.154313),
      DSP16_Q_CONVERT(-0.410843), DSP16_Q_CONVERT(-0.911706), DSP16_Q_CONVERT(-0.988722), DSP16_Q_CONVERT(-0.149765),
      DSP16_Q_CONVERT(-0.413638), DSP16_Q_CONVERT(-0.910441), DSP16_Q_CONVERT(-0.989400), DSP16_Q_CONVERT(-0.145213),
      DSP16_Q_CONVERT(-0.416430), DSP16_Q_CONVERT(-0.909168), DSP16_Q_CONVERT(-0.990058), DSP16_Q_CONVERT(-0.140658),
      DSP16_Q_CONVERT(-0.419217), DSP16_Q_CONVERT(-0.907886), DSP16_Q_CONVERT(-0.990695), DSP16_Q_CONVERT(-0.136101),
      DSP16_Q_CONVERT(-0.422000), DSP16_Q_CONVERT(-0.906596), DSP16_Q_CONVERT(-0.991311), DSP16_Q_CONVERT(-0.131540),
      DSP16_Q_CONVERT(-0.424780), DSP16_Q_CONVERT(-0.905297), DSP16_Q_CONVERT(-0.991906), DSP16_Q_CONVERT(-0.126977),
      DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.122411),
      DSP16_Q_CONVERT(-0.430326), DSP16_Q_CONVERT(-0.902673), DSP16_Q_CONVERT(-0.993032), DSP16_Q_CONVERT(-0.117842),
      DSP16_Q_CONVERT(-0.433094), DSP16_Q_CONVERT(-0.901349), DSP16_Q_CONVERT(-0.993564), DSP16_Q_CONVERT(-0.113271),
      DSP16_Q_CONVERT(-0.435857), DSP16_Q_CONVERT(-0.900016), DSP16_Q_CONVERT(-0.994075), DSP16_Q_CONVERT(-0.108697),
      DSP16_Q_CONVERT(-0.438616), DSP16_Q_CONVERT(-0.898674), DSP16_Q_CONVERT(-0.994565), DSP16_Q_CONVERT(-0.104122),
      DSP16_Q_CONVERT(-0.441371), DSP16_Q_CONVERT(-0.897325), DSP16_Q_CONVERT(-0.995033), DSP16_Q_CONVERT(-0.099544),
      DSP16_Q_CONVERT(-0.444122), DSP16_Q_CONVERT(-0.895966), DSP16_Q_CONVERT(-0.995481), DSP16_Q_CONVERT(-0.094963),
      DSP16_Q_CONVERT(-0.446869), DSP16_Q_CONVERT(-0.894599), DSP16_Q_CONVERT(-0.995907), DSP16_Q_CONVERT(-0.090381),
      DSP16_Q_CONVERT(-0.449611), DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(-0.996313), DSP16_Q_CONVERT(-0.085797),
      DSP16_Q_CONVERT(-0.452350), DSP16_Q_CONVERT(-0.891841), DSP16_Q_CONVERT(-0.996697), DSP16_Q_CONVERT(-0.081211),
      DSP16_Q_CONVERT(-0.455084), DSP16_Q_CONVERT(-0.890449), DSP16_Q_CONVERT(-0.997060), DSP16_Q_CONVERT(-0.076624),
      DSP16_Q_CONVERT(-0.457813), DSP16_Q_CONVERT(-0.889048), DSP16_Q_CONVERT(-0.997402), DSP16_Q_CONVERT(-0.072035),
      DSP16_Q_CONVERT(-0.460539), DSP16_Q_CONVERT(-0.887640), DSP16_Q_CONVERT(-0.997723), DSP16_Q_CONVERT(-0.067444),
      DSP16_Q_CONVERT(-0.463260), DSP16_Q_CONVERT(-0.886223), DSP16_Q_CONVERT(-0.998023), DSP16_Q_CONVERT(-0.062852),
      DSP16_Q_CONVERT(-0.465976), DSP16_Q_CONVERT(-0.884797), DSP16_Q_CONVERT(-0.998302), DSP16_Q_CONVERT(-0.058258),
      DSP16_Q_CONVERT(-0.468689), DSP16_Q_CONVERT(-0.883363), DSP16_Q_CONVERT(-0.998559), DSP16_Q_CONVERT(-0.053664),
      DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.049068),
      DSP16_Q_CONVERT(-0.474100), DSP16_Q_CONVERT(-0.880471), DSP16_Q_CONVERT(-0.999011), DSP16_Q_CONVERT(-0.044471),
      DSP16_Q_CONVERT(-0.476799), DSP16_Q_CONVERT(-0.879012), DSP16_Q_CONVERT(-0.999205), DSP16_Q_CONVERT(-0.039873),
      DSP16_Q_CONVERT(-0.479494), DSP16_Q_CONVERT(-0.877545), DSP16_Q_CONVERT(-0.999378), DSP16_Q_CONVERT(-0.035274),
      DSP16_Q_CONVERT(-0.482184), DSP16_Q_CONVERT(-0.876070), DSP16_Q_CONVERT(-0.999529), DSP16_Q_CONVERT(-0.030675),
      DSP16_Q_CONVERT(-0.484869), DSP16_Q_CONVERT(-0.874587), DSP16_Q_CONVERT(-0.999660), DSP16_Q_CONVERT(-0.026075),
      DSP16_Q_CONVERT(-0.487550), DSP16_Q_CONVERT(-0.873095), DSP16_Q_CONVERT(-0.999769), DSP16_Q_CONVERT(-0.021474),
      DSP16_Q_CONVERT(-0.490226), DSP16_Q_CONVERT(-0.871595), DSP16_Q_CONVERT(-0.999858), DSP16_Q_CONVERT(-0.016873),
      DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(-0.870087), DSP16_Q_CONVERT(-0.999925), DSP16_Q_CONVERT(-0.012272),
      DSP16_Q_CONVERT(-0.495565), DSP16_Q_CONVERT(-0.868571), DSP16_Q_CONVERT(-0.999971), DSP16_Q_CONVERT(-0.007670),
      DSP16_Q_CONVERT(-0.498228), DSP16_Q_CONVERT(-0.867046), DSP16_Q_CONVERT(-0.999995), DSP16_Q_CONVERT(-0.003068),
      DSP16_Q_CONVERT(-0.500885), DSP16_Q_CONVERT(-0.865514), DSP16_Q_CONVERT(-0.999999), DSP16_Q_CONVERT(0.001534),
      DSP16_Q_CONVERT(-0.503538), DSP16_Q_CONVERT(-0.863973), DSP16_Q_CONVERT(-0.999981), DSP16_Q_CONVERT(0.006136),
      DSP16_Q_CONVERT(-0.506187), DSP16_Q_CONVERT(-0.862424), DSP16_Q_CONVERT(-0.999942), DSP16_Q_CONVERT(0.010738),
      DSP16_Q_CONVERT(-0.508830), DSP16_Q_CONVERT(-0.860867), DSP16_Q_CONVERT(-0.999882), DSP16_Q_CONVERT(0.015339),
      DSP16_Q_CONVERT(-0.511469), DSP16_Q_CONVERT(-0.859302), DSP16_Q_CONVERT(-0.999801), DSP16_Q_CONVERT(0.019940),
      DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(0.024541),
      DSP16_Q_CONVERT(-0.516732), DSP16_Q_CONVERT(-0.856147), DSP16_Q_CONVERT(-0.999575), DSP16_Q_CONVERT(0.029142),
      DSP16_Q_CONVERT(-0.519356), DSP16_Q_CONVERT(-0.854558), DSP16_Q_CONVERT(-0.999431), DSP16_Q_CONVERT(0.033741),
      DSP16_Q_CONVERT(-0.521975), DSP16_Q_CONVERT(-0.852961), DSP16_Q_CONVERT(-0.999265), DSP16_Q_CONVERT(0.038340),
      DSP16_Q_CONVERT(-0.524590), DSP16_Q_CONVERT(-0.851355), DSP16_Q_CONVERT(-0.999078), DSP16_Q_CONVERT(0.042938),
      DSP16_Q_CONVERT(-0.527199), DSP16_Q_CONVERT(-0.849742), DSP16_Q_CONVERT(-0.998870), DSP16_Q_CONVERT(0.047535),
      DSP16_Q_CONVERT(-0.529804), DSP16_Q_CONVERT(-0.848120), DSP16_Q_CONVERT(-0.998640), DSP16_Q_CONVERT(0.052132),
      DSP16_Q_CONVERT(-0.532403), DSP16_Q_CONVERT(-0.846491), DSP16_Q_CONVERT(-0.998390), DSP16_Q_CONVERT(0.056727),
      DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(-0.998118), DSP16_Q_CONVERT(0.061321),
      DSP16_Q_CONVERT(-0.537587), DSP16_Q_CONVERT(-0.843208), DSP16_Q_CONVERT(-0.997825), DSP16_Q_CONVERT(0.065913),
      DSP16_Q_CONVERT(-0.540171), DSP16_Q_CONVERT(-0.841555), DSP16_Q_CONVERT(-0.997511), DSP16_Q_CONVERT(0.070505),
      DSP16_Q_CONVERT(-0.542751), DSP16_Q_CONVERT(-0.839894), DSP16_Q_CONVERT(-0.997176), DSP16_Q_CONVERT(0.075094),
      DSP16_Q_CONVERT(-0.545325), DSP16_Q_CONVERT(-0.838225), DSP16_Q_CONVERT(-0.996820), DSP16_Q_CONVERT(0.079682),
      DSP16_Q_CONVERT(-0.547894), DSP16_Q_CONVERT(-0.836548), DSP16_Q_CONVERT(-0.996443), DSP16_Q_CONVERT(0.084269),
      DSP16_Q_CONVERT(-0.550458), DSP16_Q_CONVERT(-0.834863), DSP16_Q_CONVERT(-0.996045), DSP16_Q_CONVERT(0.088854),
      DSP16_Q_CONVERT(-0.553017), DSP16_Q_CONVERT(-0.833170), DSP16_Q_CONVERT(-0.995625), DSP16_Q_CONVERT(0.093436),
      DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(0.098017),
      DSP16_Q_CONVERT(-0.558119), DSP16_Q_CONVERT(-0.829761), DSP16_Q_CONVERT(-0.994723), DSP16_Q_CONVERT(0.102596),
      DSP16_Q_CONVERT(-0.560662), DSP16_Q_CONVERT(-0.828045), DSP16_Q_CONVERT(-0.994240), DSP16_Q_CONVERT(0.107172),
      DSP16_Q_CONVERT(-0.563199), DSP16_Q_CONVERT(-0.826321), DSP16_Q_CONVERT(-0.993737), DSP16_Q_CONVERT(0.111747),
      DSP16_Q_CONVERT(-0.565732), DSP16_Q_CONVERT(-0.824589), DSP16_Q_CONVERT(-0.993212), DSP16_Q_CONVERT(0.116319),
      DSP16_Q_CONVERT(-0.568259), DSP16_Q_CONVERT(-0.822850), DSP16_Q_CONVERT(-0.992666), DSP16_Q_CONVERT(0.120888),
      DSP16_Q_CONVERT(-0.570781), DSP16_Q_CONVERT(-0.821103), DSP16_Q_CONVERT(-0.992099), DSP16_Q_CONVERT(0.125455),
      DSP16_Q_CONVERT(-0.573297), DSP16_Q_CONVERT(-0.819348), DSP16_Q_CONVERT(-0.991511), DSP16_Q_CONVERT(0.130019),
      DSP16_Q_CONVERT(-0.575808), DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(-0.990903), DSP16_Q_CONVERT(0.134581),
      DSP16_Q_CONVERT(-0.578314), DSP16_Q_CONVERT(-0.815814), DSP16_Q_CONVERT(-0.990273), DSP16_Q_CONVERT(0.139139),
      DSP16_Q_CONVERT(-0.580814), DSP16_Q_CONVERT(-0.814036), DSP16_Q_CONVERT(-0.989622), DSP16_Q_CONVERT(0.143695),
      DSP16_Q_CONVERT(-0.583309), DSP16_Q_CONVERT(-0.812251), DSP16_Q_CONVERT(-0.988950), DSP16_Q_CONVERT(0.148248),
      DSP16_Q_CONVERT(-0.585798), DSP16_Q_CONVERT(-0.810457), DSP16_Q_CONVERT(-0.988258), DSP16_Q_CONVERT(0.152797),
      DSP16_Q_CONVERT(-0.588282), DSP16_Q_CONVERT(-0.808656), DSP16_Q_CONVERT(-0.987544), DSP16_Q_CONVERT(0.157343),
      DSP16_Q_CONVERT(-0.590760), DSP16_Q_CONVERT(-0.806848), DSP16_Q_CONVERT(-0.986809), DSP16_Q_CONVERT(0.161886),
      DSP16_Q_CONVERT(-0.593232), DSP16_Q_CONVERT(-0.805031), DSP16_Q_CONVERT(-0.986054), DSP16_Q_CONVERT(0.166426),
      DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(0.170962),
      DSP16_Q_CONVERT(-0.598161), DSP16_Q_CONVERT(-0.801376), DSP16_Q_CONVERT(-0.984480), DSP16_Q_CONVERT(0.175494),
      DSP16_Q_CONVERT(-0.600616), DSP16_Q_CONVERT(-0.799537), DSP16_Q_CONVERT(-0.983662), DSP16_Q_CONVERT(0.180023),
      DSP16_Q_CONVERT(-0.603067), DSP16_Q_CONVERT(-0.797691), DSP16_Q_CONVERT(-0.982824), DSP16_Q_CONVERT(0.184548),
      DSP16_Q_CONVERT(-0.605511), DSP16_Q_CONVERT(-0.795837), DSP16_Q_CONVERT(-0.981964), DSP16_Q_CONVERT(0.189069),
      DSP16_Q_CONVERT(-0.607950), DSP16_Q_CONVERT(-0.793975), DSP16_Q_CONVERT(-0.981083), DSP16_Q_CONVERT(0.193586),
      DSP16_Q_CONVERT(-0.610383), DSP16_Q_CONVERT(-0.792107), DSP16_Q_CONVERT(-0.980182), DSP16_Q_CONVERT(0.198098),
      DSP16_Q_CONVERT(-0.612810), DSP16_Q_CONVERT(-0.790230), DSP16_Q_CONVERT(-0.979260), DSP16_Q_CONVERT(0.202607),
      DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(-0.788346), DSP16_Q_CONVERT(-0.978317), DSP16_Q_CONVERT(0.207111),
      DSP16_Q_CONVERT(-0.617647), DSP16_Q_CONVERT(-0.786455), DSP16_Q_CONVERT(-0.977354), DSP16_Q_CONVERT(0.211611),
      DSP16_Q_CONVERT(-0.620057), DSP16_Q_CONVERT(-0.784557), DSP16_Q_CONVERT(-0.976370), DSP16_Q_CONVERT(0.216107),
      DSP16_Q_CONVERT(-0.622461), DSP16_Q_CONVERT(-0.782651), DSP16_Q_CONVERT(-0.975365), DSP16_Q_CONVERT(0.220598),
      DSP16_Q_CONVERT(-0.624859), DSP16_Q_CONVERT(-0.780737), DSP16_Q_CONVERT(-0.974339), DSP16_Q_CONVERT(0.225084),
      DSP16_Q_CONVERT(-0.627252), DSP16_Q_CONVERT(-0.778817), DSP16_Q_CONVERT(-0.973293), DSP16_Q_CONVERT(0.229565),
      DSP16_Q_CONVERT(-0.629638), DSP16_Q_CONVERT(-0.776888), DSP16_Q_CONVERT(-0.972226), DSP16_Q_CONVERT(0.234042),
      DSP16_Q_CONVERT(-0.632019), DSP16_Q_CONVERT(-0.774953), DSP16_Q_CONVERT(-0.971139), DSP16_Q_CONVERT(0.238514),
      DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(0.242980),
      DSP16_Q_CONVERT(-0.636762), DSP16_Q_CONVERT(-0.771061), DSP16_Q_CONVERT(-0.968903), DSP16_Q_CONVERT(0.247442),
      DSP16_Q_CONVERT(-0.639124), DSP16_Q_CONVERT(-0.769103), DSP16_Q_CONVERT(-0.967754), DSP16_Q_CONVERT(0.251898),
      DSP16_Q_CONVERT(-0.641481), DSP16_Q_CONVERT(-0.767139), DSP16_Q_CONVERT(-0.966584), DSP16_Q_CONVERT(0.256349),
      DSP16_Q_CONVERT(-0.643832), DSP16_Q_CONVERT(-0.765167), DSP16_Q_CONVERT(-0.965394), DSP16_Q_CONVERT(0.260794),
      DSP16_Q_CONVERT(-0.646176), DSP16_Q_CONVERT(-0.763188), DSP16_Q_CONVERT(-0.964184), DSP16_Q_CONVERT(0.265234),
      DSP16_Q_CONVERT(-0.648514), DSP16_Q_CONVERT(-0.761202), DSP16_Q_CONVERT(-0.962953), DSP16_Q_CONVERT(0.269668),
      DSP16_Q_CONVERT(-0.650847), DSP16_Q_CONVERT(-0.759209), DSP16_Q_CONVERT(-0.961702), DSP16_Q_CONVERT(0.274097),
      DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(-0.960431), DSP16_Q_CONVERT(0.278520),
      DSP16_Q_CONVERT(-0.655493), DSP16_Q_CONVERT(-0.755201), DSP16_Q_CONVERT(-0.959139), DSP16_Q_CONVERT(0.282937),
      DSP16_Q_CONVERT(-0.657807), DSP16_Q_CONVERT(-0.753187), DSP16_Q_CONVERT(-0.957826), DSP16_Q_CONVERT(0.287347),
      DSP16_Q_CONVERT(-0.660114), DSP16_Q_CONVERT(-0.751165), DSP16_Q_CONVERT(-0.956494), DSP16_Q_CONVERT(0.291752),
      DSP16_Q_CONVERT(-0.662416), DSP16_Q_CONVERT(-0.749136), DSP16_Q_CONVERT(-0.955141), DSP16_Q_CONVERT(0.296151),
      DSP16_Q_CONVERT(-0.664711), DSP16_Q_CONVERT(-0.747101), DSP16_Q_CONVERT(-0.953768), DSP16_Q_CONVERT(0.300543),
      DSP16_Q_CONVERT(-0.667000), DSP16_Q_CONVERT(-0.745058), DSP16_Q_CONVERT(-0.952375), DSP16_Q_CONVERT(0.304929),
      DSP16_Q_CONVERT(-0.669283), DSP16_Q_CONVERT(-0.743008), DSP16_Q_CONVERT(-0.950962), DSP16_Q_CONVERT(0.309309),
      DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(0.313682),
      DSP16_Q_CONVERT(-0.673829), DSP16_Q_CONVERT(-0.738887), DSP16_Q_CONVERT(-0.948075), DSP16_Q_CONVERT(0.318048),
      DSP16_Q_CONVERT(-0.676093), DSP16_Q_CONVERT(-0.736817), DSP16_Q_CONVERT(-0.946601), DSP16_Q_CONVERT(0.322408),
      DSP16_Q_CONVERT(-0.678350), DSP16_Q_CONVERT(-0.734739), DSP16_Q_CONVERT(-0.945107), DSP16_Q_CONVERT(0.326760),
      DSP16_Q_CONVERT(-0.680601), DSP16_Q_CONVERT(-0.732654), DSP16_Q_CONVERT(-0.943593), DSP16_Q_CONVERT(0.331106),
      DSP16_Q_CONVERT(-0.682846), DSP16_Q_CONVERT(-0.730563), DSP16_Q_CONVERT(-0.942060), DSP16_Q_CONVERT(0.335445),
      DSP16_Q_CONVERT(-0.685084), DSP16_Q_CONVERT(-0.728464), DSP16_Q_CONVERT(-0.940506), DSP16_Q_CONVERT(0.339777),
      DSP16_Q_CONVERT(-0.687315), DSP16_Q_CONVERT(-0.726359), DSP16_Q_CONVERT(-0.938932), DSP16_Q_CONVERT(0.344101),
      DSP16_Q_CONVERT(-0.689541), DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(-0.937339), DSP16_Q_CONVERT(0.348419),
      DSP16_Q_CONVERT(-0.691759), DSP16_Q_CONVERT(-0.722128), DSP16_Q_CONVERT(-0.935726), DSP16_Q_CONVERT(0.352729),
      DSP16_Q_CONVERT(-0.693971), DSP16_Q_CONVERT(-0.720003), DSP16_Q_CONVERT(-0.934093), DSP16_Q_CONVERT(0.357031),
      DSP16_Q_CONVERT(-0.696177), DSP16_Q_CONVERT(-0.717870), DSP16_Q_CONVERT(-0.932440), DSP16_Q_CONVERT(0.361326),
      DSP16_Q_CONVERT(-0.698376), DSP16_Q_CONVERT(-0.715731), DSP16_Q_CONVERT(-0.930767), DSP16_Q_CONVERT(0.365613),
      DSP16_Q_CONVERT(-0.700569), DSP16_Q_CONVERT(-0.713585), DSP16_Q_CONVERT(-0.929075), DSP16_Q_CONVERT(0.369892),
      DSP16_Q_CONVERT(-0.702755), DSP16_Q_CONVERT(-0.711432), DSP16_Q_CONVERT(-0.927363), DSP16_Q_CONVERT(0.374164),
      DSP16_Q_CONVERT(-0.704934), DSP16_Q_CONVERT(-0.709273), DSP16_Q_CONVERT(-0.925631), DSP16_Q_CONVERT(0.378428),
      DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.707107), DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(0.382683),
      DSP16_Q_CONVERT(-0.709273), DSP16_Q_CONVERT(-0.704934), DSP16_Q_CONVERT(-0.922109), DSP16_Q_CONVERT(0.386931),
      DSP16_Q_CONVERT(-0.711432), DSP16_Q_CONVERT(-0.702755), DSP16_Q_CONVERT(-0.920318), DSP16_Q_CONVERT(0.391170),
      DSP16_Q_CONVERT(-0.713585), DSP16_Q_CONVERT(-0.700569), DSP16_Q_CONVERT(-0.918508), DSP16_Q_CONVERT(0.395401),
      DSP16_Q_CONVERT(-0.715731), DSP16_Q_CONVERT(-0.698376), DSP16_Q_CONVERT(-0.916679), DSP16_Q_CONVERT(0.399624),
      DSP16_Q_CONVERT(-0.717870), DSP16_Q_CONVERT(-0.696177), DSP16_Q_CONVERT(-0.914830), DSP16_Q_CONVERT(0.403838),
      DSP16_Q_CONVERT(-0.720003), DSP16_Q_CONVERT(-0.693971), DSP16_Q_CONVERT(-0.912962), DSP16_Q_CONVERT(0.408044),
      DSP16_Q_CONVERT(-0.722128), DSP16_Q_CONVERT(-0.691759), DSP16_Q_CONVERT(-0.911075), DSP16_Q_CONVERT(0.412241),
      DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(-0.689541), DSP16_Q_CONVERT(-0.909168), DSP16_Q_CONVERT(0.416430),
      DSP16_Q_CONVERT(-0.726359), DSP16_Q_CONVERT(-0.687315), DSP16_Q_CONVERT(-0.907242), DSP16_Q_CONVERT(0.420609),
      DSP16_Q_CONVERT(-0.728464), DSP16_Q_CONVERT(-0.685084), DSP16_Q_CONVERT(-0.905297), DSP16_Q_CONVERT(0.424780),
      DSP16_Q_CONVERT(-0.730563), DSP16_Q_CONVERT(-0.682846), DSP16_Q_CONVERT(-0.903332), DSP16_Q_CONVERT(0.428941),
      DSP16_Q_CONVERT(-0.732654), DSP16_Q_CONVERT(-0.680601), DSP16_Q_CONVERT(-0.901349), DSP16_Q_CONVERT(0.433094),
      DSP16_Q_CONVERT(-0.734739), DSP16_Q_CONVERT(-0.678350), DSP16_Q_CONVERT(-0.899346), DSP16_Q_CONVERT(0.437237),
      DSP16_Q_CONVERT(-0.736817), DSP16_Q_CONVERT(-0.676093), DSP16_Q_CONVERT(-0.897325), DSP16_Q_CONVERT(0.441371),
      DSP16_Q_CONVERT(-0.738887), DSP16_Q_CONVERT(-0.673829), DSP16_Q_CONVERT(-0.895284), DSP16_Q_CONVERT(0.445496),
      DSP16_Q_CONVERT(-0.740951), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(0.449611),
      DSP16_Q_CONVERT(-0.743008), DSP16_Q_CONVERT(-0.669283), DSP16_Q_CONVERT(-0.891146), DSP16_Q_CONVERT(0.453717),
      DSP16_Q_CONVERT(-0.745058), DSP16_Q_CONVERT(-0.667000), DSP16_Q_CONVERT(-0.889048), DSP16_Q_CONVERT(0.457813),
      DSP16_Q_CONVERT(-0.747101), DSP16_Q_CONVERT(-0.664711), DSP16_Q_CONVERT(-0.886932), DSP16_Q_CONVERT(0.461900),
      DSP16_Q_CONVERT(-0.749136), DSP16_Q_CONVERT(-0.662416), DSP16_Q_CONVERT(-0.884797), DSP16_Q_CONVERT(0.465976),
      DSP16_Q_CONVERT(-0.751165), DSP16_Q_CONVERT(-0.660114), DSP16_Q_CONVERT(-0.882643), DSP16_Q_CONVERT(0.470043),
      DSP16_Q_CONVERT(-0.753187), DSP16_Q_CONVERT(-0.657807), DSP16_Q_CONVERT(-0.880471), DSP16_Q_CONVERT(0.474100),
      DSP16_Q_CONVERT(-0.755201), DSP16_Q_CONVERT(-0.655493), DSP16_Q_CONVERT(-0.878280), DSP16_Q_CONVERT(0.478147),
      DSP16_Q_CONVERT(-0.757209), DSP16_Q_CONVERT(-0.653173), DSP16_Q_CONVERT(-0.876070), DSP16_Q_CONVERT(0.482184),
      DSP16_Q_CONVERT(-0.759209), DSP16_Q_CONVERT(-0.650847), DSP16_Q_CONVERT(-0.873842), DSP16_Q_CONVERT(0.486210),
      DSP16_Q_CONVERT(-0.761202), DSP16_Q_CONVERT(-0.648514), DSP16_Q_CONVERT(-0.871595), DSP16_Q_CONVERT(0.490226),
      DSP16_Q_CONVERT(-0.763188), DSP16_Q_CONVERT(-0.646176), DSP16_Q_CONVERT(-0.869330), DSP16_Q_CONVERT(0.494232),
      DSP16_Q_CONVERT(-0.765167), DSP16_Q_CONVERT(-0.643832), DSP16_Q_CONVERT(-0.867046), DSP16_Q_CONVERT(0.498228),
      DSP16_Q_CONVERT(-0.767139), DSP16_Q_CONVERT(-0.641481), DSP16_Q_CONVERT(-0.864744), DSP16_Q_CONVERT(0.502212),
      DSP16_Q_CONVERT(-0.769103), DSP16_Q_CONVERT(-0.639124), DSP16_Q_CONVERT(-0.862424), DSP16_Q_CONVERT(0.506187),
      DSP16_Q_CONVERT(-0.771061), DSP16_Q_CONVERT(-0.636762), DSP16_Q_CONVERT(-0.860085), DSP16_Q_CONVERT(0.510150),
      DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(-0.634393), DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(0.514103),
      DSP16_Q_CONVERT(-0.774953), DSP16_Q_CONVERT(-0.632019), DSP16_Q_CONVERT(-0.855354), DSP16_Q_CONVERT(0.518045),
      DSP16_Q_CONVERT(-0.776888), DSP16_Q_CONVERT(-0.629638), DSP16_Q_CONVERT(-0.852961), DSP16_Q_CONVERT(0.521975),
      DSP16_Q_CONVERT(-0.778817), DSP16_Q_CONVERT(-0.627252), DSP16_Q_CONVERT(-0.850549), DSP16_Q_CONVERT(0.525895),
      DSP16_Q_CONVERT(-0.780737), DSP16_Q_CONVERT(-0.624859), DSP16_Q_CONVERT(-0.848120), DSP16_Q_CONVERT(0.529804),
      DSP16_Q_CONVERT(-0.782651), DSP16_Q_CONVERT(-0.622461), DSP16_Q_CONVERT(-0.845673), DSP16_Q_CONVERT(0.533701),
      DSP16_Q_CONVERT(-0.784557), DSP16_Q_CONVERT(-0.620057), DSP16_Q_CONVERT(-0.843208), DSP16_Q_CONVERT(0.537587),
      DSP16_Q_CONVERT(-0.786455), DSP16_Q_CONVERT(-0.617647), DSP16_Q_CONVERT(-0.840725), DSP16_Q_CONVERT(0.541462),
      DSP16_Q_CONVERT(-0.788346), DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(-0.838225), DSP16_Q_CONVERT(0.545325),
      DSP16_Q_CONVERT(-0.790230), DSP16_Q_CONVERT(-0.612810), DSP16_Q_CONVERT(-0.835706), DSP16_Q_CONVERT(0.549177),
      DSP16_Q_CONVERT(-0.792107), DSP16_Q_CONVERT(-0.610383), DSP16_Q_CONVERT(-0.833170), DSP16_Q_CONVERT(0.553017),
      DSP16_Q_CONVERT(-0.793975), DSP16_Q_CONVERT(-0.607950), DSP16_Q_CONVERT(-0.830616), DSP16_Q_CONVERT(0.556845),
      DSP16_Q_CONVERT(-0.795837), DSP16_Q_CONVERT(-0.605511), DSP16_Q_CONVERT(-0.828045), DSP16_Q_CONVERT(0.560662),
      DSP16_Q_CONVERT(-0.797691), DSP16_Q_CONVERT(-0.603067), DSP16_Q_CONVERT(-0.825456), DSP16_Q_CONVERT(0.564466),
      DSP16_Q_CONVERT(-0.799537), DSP16_Q_CONVERT(-0.600616), DSP16_Q_CONVERT(-0.822850), DSP16_Q_CONVERT(0.568259),
      DSP16_Q_CONVERT(-0.801376), DSP16_Q_CONVERT(-0.598161), DSP16_Q_CONVERT(-0.820226), DSP16_Q_CONVERT(0.572040),
      DSP16_Q_CONVERT(-0.803208), DSP16_Q_CONVERT(-0.595699), DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(0.575808),
      DSP16_Q_CONVERT(-0.805031), DSP16_Q_CONVERT(-0.593232), DSP16_Q_CONVERT(-0.814926), DSP16_Q_CONVERT(0.579565),
      DSP16_Q_CONVERT(-0.806848), DSP16_Q_CONVERT(-0.590760), DSP16_Q_CONVERT(-0.812251), DSP16_Q_CONVERT(0.583309),
      DSP16_Q_CONVERT(-0.808656), DSP16_Q_CONVERT(-0.588282), DSP16_Q_CONVERT(-0.809558), DSP16_Q_CONVERT(0.587040),
      DSP16_Q_CONVERT(-0.810457), DSP16_Q_CONVERT(-0.585798), DSP16_Q_CONVERT(-0.806848), DSP16_Q_CONVERT(0.590760),
      DSP16_Q_CONVERT(-0.812251), DSP16_Q_CONVERT(-0.583309), DSP16_Q_CONVERT(-0.804120), DSP16_Q_CONVERT(0.594466),
      DSP16_Q_CONVERT(-0.814036), DSP16_Q_CONVERT(-0.580814), DSP16_Q_CONVERT(-0.801376), DSP16_Q_CONVERT(0.598161),
      DSP16_Q_CONVERT(-0.815814), DSP16_Q_CONVERT(-0.578314), DSP16_Q_CONVERT(-0.798615), DSP16_Q_CONVERT(0.601842),
      DSP16_Q_CONVERT(-0.817585), DSP16_Q_CONVERT(-0.575808), DSP16_Q_CONVERT(-0.795837), DSP16_Q_CONVERT(0.605511),
      DSP16_Q_CONVERT(-0.819348), DSP16_Q_CONVERT(-0.573297), DSP16_Q_CONVERT(-0.793042), DSP16_Q_CONVERT(0.609167),
      DSP16_Q_CONVERT(-0.821103), DSP16_Q_CONVERT(-0.570781), DSP16_Q_CONVERT(-0.790230), DSP16_Q_CONVERT(0.612810),
      DSP16_Q_CONVERT(-0.822850), DSP16_Q_CONVERT(-0.568259), DSP16_Q_CONVERT(-0.787402), DSP16_Q_CONVERT(0.616440),
      DSP16_Q_CONVERT(-0.824589), DSP16_Q_CONVERT(-0.565732), DSP16_Q_CONVERT(-0.784557), DSP16_Q_CONVERT(0.620057),
      DSP16_Q_CONVERT(-0.826321), DSP16_Q_CONVERT(-0.563199), DSP16_Q_CONVERT(-0.781695), DSP16_Q_CONVERT(0.623661),
      DSP16_Q_CONVERT(-0.828045), DSP16_Q_CONVERT(-0.560662), DSP16_Q_CONVERT(-0.778817), DSP16_Q_CONVERT(0.627252),
      DSP16_Q_CONVERT(-0.829761), DSP16_Q_CONVERT(-0.558119), DSP16_Q_CONVERT(-0.775922), DSP16_Q_CONVERT(0.630829),
      DSP16_Q_CONVERT(-0.831470), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(-0.773010), DSP16_Q_CONVERT(0.634393),
      DSP16_Q_CONVERT(-0.833170), DSP16_Q_CONVERT(-0.553017), DSP16_Q_CONVERT(-0.770083), DSP16_Q_CONVERT(0.637944),
      DSP16_Q_CONVERT(-0.834863), DSP16_Q_CONVERT(-0.550458), DSP16_Q_CONVERT(-0.767139), DSP16_Q_CONVERT(0.641481),
      DSP16_Q_CONVERT(-0.836548), DSP16_Q_CONVERT(-0.547894), DSP16_Q_CONVERT(-0.764179), DSP16_Q_CONVERT(0.645005),
      DSP16_Q_CONVERT(-0.838225), DSP16_Q_CONVERT(-0.545325), DSP16_Q_CONVERT(-0.761202), DSP16_Q_CONVERT(0.648514),
      DSP16_Q_CONVERT(-0.839894), DSP16_Q_CONVERT(-0.542751), DSP16_Q_CONVERT(-0.758210), DSP16_Q_CONVERT(0.652011),
      DSP16_Q_CONVERT(-0.841555), DSP16_Q_CONVERT(-0.540171), DSP16_Q_CONVERT(-0.755201), DSP16_Q_CONVERT(0.655493),
      DSP16_Q_CONVERT(-0.843208), DSP16_Q_CONVERT(-0.537587), DSP16_Q_CONVERT(-0.752177), DSP16_Q_CONVERT(0.658961),
      DSP16_Q_CONVERT(-0.844854), DSP16_Q_CONVERT(-0.534998), DSP16_Q_CONVERT(-0.749136), DSP16_Q_CONVERT(0.662416),
      DSP16_Q_CONVERT(-0.846491), DSP16_Q_CONVERT(-0.532403), DSP16_Q_CONVERT(-0.746080), DSP16_Q_CONVERT(0.665856),
      DSP16_Q_CONVERT(-0.848120), DSP16_Q_CONVERT(-0.529804), DSP16_Q_CONVERT(-0.743008), DSP16_Q_CONVERT(0.669283),
      DSP16_Q_CONVERT(-0.849742), DSP16_Q_CONVERT(-0.527199), DSP16_Q_CONVERT(-0.739920), DSP16_Q_CONVERT(0.672695),
      DSP16_Q_CONVERT(-0.851355), DSP16_Q_CONVERT(-0.524590), DSP16_Q_CONVERT(-0.736817), DSP16_Q_CONVERT(0.676093),
      DSP16_Q_CONVERT(-0.852961), DSP16_Q_CONVERT(-0.521975), DSP16_Q_CONVERT(-0.733697), DSP16_Q_CONVERT(0.679476),
      DSP16_Q_CONVERT(-0.854558), DSP16_Q_CONVERT(-0.519356), DSP16_Q_CONVERT(-0.730563), DSP16_Q_CONVERT(0.682846),
      DSP16_Q_CONVERT(-0.856147), DSP16_Q_CONVERT(-0.516732), DSP16_Q_CONVERT(-0.727413), DSP16_Q_CONVERT(0.686200),
      DSP16_Q_CONVERT(-0.857729), DSP16_Q_CONVERT(-0.514103), DSP16_Q_CONVERT(-0.724247), DSP16_Q_CONVERT(0.689541),
      DSP16_Q_CONVERT(-0.859302), DSP16_Q_CONVERT(-0.511469), DSP16_Q_CONVERT(-0.721066), DSP16_Q_CONVERT(0.692866),
      DSP16_Q_CONVERT(-0.860867), DSP16_Q_CONVERT(-0.508830), DSP16_Q_CONVERT(-0.717870), DSP16_Q_CONVERT(0.696177),
      DSP16_Q_CONVERT(-0.862424), DSP16_Q_CONVERT(-0.506187), DSP16_Q_CONVERT(-0.714659), DSP16_Q_CONVERT(0.699473),
      DSP16_Q_CONVERT(-0.863973), DSP16_Q_CONVERT(-0.503538), DSP16_Q_CONVERT(-0.711432), DSP16_Q_CONVERT(0.702755),
      DSP16_Q_CONVERT(-0.865514), DSP16_Q_CONVERT(-0.500885), DSP16_Q_CONVERT(-0.708191), DSP16_Q_CONVERT(0.706021),
      DSP16_Q_CONVERT(-0.867046), DSP16_Q_CONVERT(-0.498228), DSP16_Q_CONVERT(-0.704934), DSP16_Q_CONVERT(0.709273),
      DSP16_Q_CONVERT(-0.868571), DSP16_Q_CONVERT(-0.495565), DSP16_Q_CONVERT(-0.701663), DSP16_Q_CONVERT(0.712509),
      DSP16_Q_CONVERT(-0.870087), DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(-0.698376), DSP16_Q_CONVERT(0.715731),
      DSP16_Q_CONVERT(-0.871595), DSP16_Q_CONVERT(-0.490226), DSP16_Q_CONVERT(-0.695075), DSP16_Q_CONVERT(0.718937),
      DSP16_Q_CONVERT(-0.873095), DSP16_Q_CONVERT(-0.487550), DSP16_Q_CONVERT(-0.691759), DSP16_Q_CONVERT(0.722128),
      DSP16_Q_CONVERT(-0.874587), DSP16_Q_CONVERT(-0.484869), DSP16_Q_CONVERT(-0.688429), DSP16_Q_CONVERT(0.725304),
      DSP16_Q_CONVERT(-0.876070), DSP16_Q_CONVERT(-0.482184), DSP16_Q_CONVERT(-0.685084), DSP16_Q_CONVERT(0.728464),
      DSP16_Q_CONVERT(-0.877545), DSP16_Q_CONVERT(-0.479494), DSP16_Q_CONVERT(-0.681724), DSP16_Q_CONVERT(0.731609),
      DSP16_Q_CONVERT(-0.879012), DSP16_Q_CONVERT(-0.476799), DSP16_Q_CONVERT(-0.678350), DSP16_Q_CONVERT(0.734739),
      DSP16_Q_CONVERT(-0.880471), DSP16_Q_CONVERT(-0.474100), DSP16_Q_CONVERT(-0.674962), DSP16_Q_CONVERT(0.737853),
      DSP16_Q_CONVERT(-0.881921), DSP16_Q_CONVERT(-0.471397), DSP16_Q_CONVERT(-0.671559), DSP16_Q_CONVERT(0.740951),
      DSP16_Q_CONVERT(-0.883363), DSP16_Q_CONVERT(-0.468689), DSP16_Q_CONVERT(-0.668142), DSP16_Q_CONVERT(0.744034),
      DSP16_Q_CONVERT(-0.884797), DSP16_Q_CONVERT(-0.465976), DSP16_Q_CONVERT(-0.664711), DSP16_Q_CONVERT(0.747101),
      DSP16_Q_CONVERT(-0.886223), DSP16_Q_CONVERT(-0.463260), DSP16_Q_CONVERT(-0.661266), DSP16_Q_CONVERT(0.750152),
      DSP16_Q_CONVERT(-0.887640), DSP16_Q_CONVERT(-0.460539), DSP16_Q_CONVERT(-0.657807), DSP16_Q_CONVERT(0.753187),
      DSP16_Q_CONVERT(-0.889048), DSP16_Q_CONVERT(-0.457813), DSP16_Q_CONVERT(-0.654334), DSP16_Q_CONVERT(0.756206),
      DSP16_Q_CONVERT(-0.890449), DSP16_Q_CONVERT(-0.455084), DSP16_Q_CONVERT(-0.650847), DSP16_Q_CONVERT(0.759209),
      DSP16_Q_CONVERT(-0.891841), DSP16_Q_CONVERT(-0.452350), DSP16_Q_CONVERT(-0.647346), DSP16_Q_CONVERT(0.762196),
      DSP16_Q_CONVERT(-0.893224), DSP16_Q_CONVERT(-0.449611), DSP16_Q_CONVERT(-0.643832), DSP16_Q_CONVERT(0.765167),
      DSP16_Q_CONVERT(-0.894599), DSP16_Q_CONVERT(-0.446869), DSP16_Q_CONVERT(-0.640303), DSP16_Q_CONVERT(0.768122),
      DSP16_Q_CONVERT(-0.895966), DSP16_Q_CONVERT(-0.444122), DSP16_Q_CONVERT(-0.636762), DSP16_Q_CONVERT(0.771061),
      DSP16_Q_CONVERT(-0.897325), DSP16_Q_CONVERT(-0.441371), DSP16_Q_CONVERT(-0.633207), DSP16_Q_CONVERT(0.773983),
      DSP16_Q_CONVERT(-0.898674), DSP16_Q_CONVERT(-0.438616), DSP16_Q_CONVERT(-0.629638), DSP16_Q_CONVERT(0.776888),
      DSP16_Q_CONVERT(-0.900016), DSP16_Q_CONVERT(-0.435857), DSP16_Q_CONVERT(-0.626056), DSP16_Q_CONVERT(0.779778),
      DSP16_Q_CONVERT(-0.901349), DSP16_Q_CONVERT(-0.433094), DSP16_Q_CONVERT(-0.622461), DSP16_Q_CONVERT(0.782651),
      DSP16_Q_CONVERT(-0.902673), DSP16_Q_CONVERT(-0.430326), DSP16_Q_CONVERT(-0.618853), DSP16_Q_CONVERT(0.785507),
      DSP16_Q_CONVERT(-0.903989), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(-0.615232), DSP16_Q_CONVERT(0.788346),
      DSP16_Q_CONVERT(-0.905297), DSP16_Q_CONVERT(-0.424780), DSP16_Q_CONVERT(-0.611597), DSP16_Q_CONVERT(0.791169),
      DSP16_Q_CONVERT(-0.906596), DSP16_Q_CONVERT(-0.422000), DSP16_Q_CONVERT(-0.607950), DSP16_Q_CONVERT(0.793975),
      DSP16_Q_CONVERT(-0.907886), DSP16_Q_CONVERT(-0.419217), DSP16_Q_CONVERT(-0.604290), DSP16_Q_CONVERT(0.796765),
      DSP16_Q_CONVERT(-0.909168), DSP16_Q_CONVERT(-0.416430), DSP16_Q_CONVERT(-0.600616), DSP16_Q_CONVERT(0.799537),
      DSP16_Q_CONVERT(-0.910441), DSP16_Q_CONVERT(-0.413638), DSP16_Q_CONVERT(-0.596931), DSP16_Q_CONVERT(0.802293),
      DSP16_Q_CONVERT(-0.911706), DSP16_Q_CONVERT(-0.410843), DSP16_Q_CONVERT(-0.593232), DSP16_Q_CONVERT(0.805031),
      DSP16_Q_CONVERT(-0.912962), DSP16_Q_CONVERT(-0.408044), DSP16_Q_CONVERT(-0.589521), DSP16_Q_CONVERT(0.807753),
      DSP16_Q_CONVERT(-0.914210), DSP16_Q_CONVERT(-0.405241), DSP16_Q_CONVERT(-0.585798), DSP16_Q_CONVERT(0.810457),
      DSP16_Q_CONVERT(-0.915449), DSP16_Q_CONVERT(-0.402435), DSP16_Q_CONVERT(-0.582062), DSP16_Q_CONVERT(0.813144),
      DSP16_Q_CONVERT(-0.916679), DSP16_Q_CONVERT(-0.399624), DSP16_Q_CONVERT(-0.578314), DSP16_Q_CONVERT(0.815814),
      DSP16_Q_CONVERT(-0.917901), DSP16_Q_CONVERT(-0.396810), DSP16_Q_CONVERT(-0.574553), DSP16_Q_CONVERT(0.818467),
      DSP16_Q_CONVERT(-0.919114), DSP16_Q_CONVERT(-0.393992), DSP16_Q_CONVERT(-0.570781), DSP16_Q_CONVERT(0.821103),
      DSP16_Q_CONVERT(-0.920318), DSP16_Q_CONVERT(-0.391170), DSP16_Q_CONVERT(-0.566996), DSP16_Q_CONVERT(0.823721),
      DSP16_Q_CONVERT(-0.921514), DSP16_Q_CONVERT(-0.388345), DSP16_Q_CONVERT(-0.563199), DSP16_Q_CONVERT(0.826321),
      DSP16_Q_CONVERT(-0.922701), DSP16_Q_CONVERT(-0.385516), DSP16_Q_CONVERT(-0.559391), DSP16_Q_CONVERT(0.828904),
      DSP16_Q_CONVERT(-0.923880), DSP16_Q_CONVERT(-0.382683), DSP16_Q_CONVERT(-0.555570), DSP16_Q_CONVERT(0.831470),
      DSP16_Q_CONVERT(-0.925049), DSP16_Q_CONVERT(-0.379847), DSP16_Q_CONVERT(-0.551738), DSP16_Q_CONVERT(0.834018),
      DSP16_Q_CONVERT(-0.926210), DSP16_Q_CONVERT(-0.377007), DSP16_Q_CONVERT(-0.547894), DSP16_Q_CONVERT(0.836548),
      DSP16_Q_CONVERT(-0.927363), DSP16_Q_CONVERT(-0.374164), DSP16_Q_CONVERT(-0.544039), DSP16_Q_CONVERT(0.839060),
      DSP16_Q_CONVERT(-0.928506), DSP16_Q_CONVERT(-0.371317), DSP16_Q_CONVERT(-0.540171), DSP16_Q_CONVERT(0.841555),
      DSP16_Q_CONVERT(-0.929641), DSP16_Q_CONVERT(-0.368467), DSP16_Q_CONVERT(-0.536293), DSP16_Q_CONVERT(0.844032),
      DSP16_Q_CONVERT(-0.930767), DSP16_Q_CONVERT(-0.365613), DSP16_Q_CONVERT(-0.532403), DSP16_Q_CONVERT(0.846491),
      DSP16_Q_CONVERT(-0.931884), DSP16_Q_CONVERT(-0.362756), DSP16_Q_CONVERT(-0.528502), DSP16_Q_CONVERT(0.848932),
      DSP16_Q_CONVERT(-0.932993), DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(-0.524590), DSP16_Q_CONVERT(0.851355),
      DSP16_Q_CONVERT(-0.934093), DSP16_Q_CONVERT(-0.357031), DSP16_Q_CONVERT(-0.520666), DSP16_Q_CONVERT(0.853760),
      DSP16_Q_CONVERT(-0.935184), DSP16_Q_CONVERT(-0.354164), DSP16_Q_CONVERT(-0.516732), DSP16_Q_CONVERT(0.856147),
      DSP16_Q_CONVERT(-0.936266), DSP16_Q_CONVERT(-0.351293), DSP16_Q_CONVERT(-0.512786), DSP16_Q_CONVERT(0.858516),
      DSP16_Q_CONVERT(-0.937339), DSP16_Q_CONVERT(-0.348419), DSP16_Q_CONVERT(-0.508830), DSP16_Q_CONVERT(0.860867),
      DSP16_Q_CONVERT(-0.938404), DSP16_Q_CONVERT(-0.345541), DSP16_Q_CONVERT(-0.504863), DSP16_Q_CONVERT(0.863199),
      DSP16_Q_CONVERT(-0.939459), DSP16_Q_CONVERT(-0.342661), DSP16_Q_CONVERT(-0.500885), DSP16_Q_CONVERT(0.865514),
      DSP16_Q_CONVERT(-0.940506), DSP16_Q_CONVERT(-0.339777), DSP16_Q_CONVERT(-0.496897), DSP16_Q_CONVERT(0.867809),
      DSP16_Q_CONVERT(-0.941544), DSP16_Q_CONVERT(-0.336890), DSP16_Q_CONVERT(-0.492898), DSP16_Q_CONVERT(0.870087),
      DSP16_Q_CONVERT(-0.942573), DSP16_Q_CONVERT(-0.334000), DSP16_Q_CONVERT(-0.488889), DSP16_Q_CONVERT(0.872346),
      DSP16_Q_CONVERT(-0.943593), DSP16_Q_CONVERT(-0.331106), DSP16_Q_CONVERT(-0.484869), DSP16_Q_CONVERT(0.874587),
      DSP16_Q_CONVERT(-0.944605), DSP16_Q_CONVERT(-0.328210), DSP16_Q_CONVERT(-0.480839), DSP16_Q_CONVERT(0.876809),
      DSP16_Q_CONVERT(-0.945607), DSP16_Q_CONVERT(-0.325310), DSP16_Q_CONVERT(-0.476799), DSP16_Q_CONVERT(0.879012),
      DSP16_Q_CONVERT(-0.946601), DSP16_Q_CONVERT(-0.322408), DSP16_Q_CONVERT(-0.472749), DSP16_Q_CONVERT(0.881197),
      DSP16_Q_CONVERT(-0.947586), DSP16_Q_CONVERT(-0.319502), DSP16_Q_CONVERT(-0.468689), DSP16_Q_CONVERT(0.883363),
      DSP16_Q_CONVERT(-0.948561), DSP16_Q_CONVERT(-0.316593), DSP16_Q_CONVERT(-0.464619), DSP16_Q_CONVERT(0.885511),
      DSP16_Q_CONVERT(-0.949528), DSP16_Q_CONVERT(-0.313682), DSP16_Q_CONVERT(-0.460539), DSP16_Q_CONVERT(0.887640),
      DSP16_Q_CONVERT(-0.950486), DSP16_Q_CONVERT(-0.310767), DSP16_Q_CONVERT(-0.456449), DSP16_Q_CONVERT(0.889750),
      DSP16_Q_CONVERT(-0.951435), DSP16_Q_CONVERT(-0.307850), DSP16_Q_CONVERT(-0.452350), DSP16_Q_CONVERT(0.891841),
      DSP16_Q_CONVERT(-0.952375), DSP16_Q_CONVERT(-0.304929), DSP16_Q_CONVERT(-0.448241), DSP16_Q_CONVERT(0.893913),
      DSP16_Q_CONVERT(-0.953306), DSP16_Q_CONVERT(-0.302006), DSP16_Q_CONVERT(-0.444122), DSP16_Q_CONVERT(0.895966),
      DSP16_Q_CONVERT(-0.954228), DSP16_Q_CONVERT(-0.299080), DSP16_Q_CONVERT(-0.439994), DSP16_Q_CONVERT(0.898001),
      DSP16_Q_CONVERT(-0.955141), DSP16_Q_CONVERT(-0.296151), DSP16_Q_CONVERT(-0.435857), DSP16_Q_CONVERT(0.900016),
      DSP16_Q_CONVERT(-0.956045), DSP16_Q_CONVERT(-0.293219), DSP16_Q_CONVERT(-0.431711), DSP16_Q_CONVERT(0.902012),
      DSP16_Q_CONVERT(-0.956940), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(-0.427555), DSP16_Q_CONVERT(0.903989),
      DSP16_Q_CONVERT(-0.957826), DSP16_Q_CONVERT(-0.287347), DSP16_Q_CONVERT(-0.423390), DSP16_Q_CONVERT(0.905947),
      DSP16_Q_CONVERT(-0.958703), DSP16_Q_CONVERT(-0.284408), DSP16_Q_CONVERT(-0.419217), DSP16_Q_CONVERT(0.907886),
      DSP16_Q_CONVERT(-0.959572), DSP16_Q_CONVERT(-0.281465), DSP16_Q_CONVERT(-0.415034), DSP16_Q_CONVERT(0.909806),
      DSP16_Q_CONVERT(-0.960431), DSP16_Q_CONVERT(-0.278520), DSP16_Q_CONVERT(-0.410843), DSP16_Q_CONVERT(0.911706),
      DSP16_Q_CONVERT(-0.961280), DSP16_Q_CONVERT(-0.275572), DSP16_Q_CONVERT(-0.406643), DSP16_Q_CONVERT(0.913587),
      DSP16_Q_CONVERT(-0.962121), DSP16_Q_CONVERT(-0.272621), DSP16_Q_CONVERT(-0.402435), DSP16_Q_CONVERT(0.915449),
      DSP16_Q_CONVERT(-0.962953), DSP16_Q_CONVERT(-0.269668), DSP16_Q_CONVERT(-0.398218), DSP16_Q_CONVERT(0.917291),
      DSP16_Q_CONVERT(-0.963776), DSP16_Q_CONVERT(-0.266713), DSP16_Q_CONVERT(-0.393992), DSP16_Q_CONVERT(0.919114),
      DSP16_Q_CONVERT(-0.964590), DSP16_Q_CONVERT(-0.263755), DSP16_Q_CONVERT(-0.389758), DSP16_Q_CONVERT(0.920917),
      DSP16_Q_CONVERT(-0.965394), DSP16_Q_CONVERT(-0.260794), DSP16_Q_CONVERT(-0.385516), DSP16_Q_CONVERT(0.922701),
      DSP16_Q_CONVERT(-0.966190), DSP16_Q_CONVERT(-0.257831), DSP16_Q_CONVERT(-0.381266), DSP16_Q_CONVERT(0.924465),
      DSP16_Q_CONVERT(-0.966976), DSP16_Q_CONVERT(-0.254866), DSP16_Q_CONVERT(-0.377007), DSP16_Q_CONVERT(0.926210),
      DSP16_Q_CONVERT(-0.967754), DSP16_Q_CONVERT(-0.251898), DSP16_Q_CONVERT(-0.372741), DSP16_Q_CONVERT(0.927935),
      DSP16_Q_CONVERT(-0.968522), DSP16_Q_CONVERT(-0.248928), DSP16_Q_CONVERT(-0.368467), DSP16_Q_CONVERT(0.929641),
      DSP16_Q_CONVERT(-0.969281), DSP16_Q_CONVERT(-0.245955), DSP16_Q_CONVERT(-0.364185), DSP16_Q_CONVERT(0.931327),
      DSP16_Q_CONVERT(-0.970031), DSP16_Q_CONVERT(-0.242980), DSP16_Q_CONVERT(-0.359895), DSP16_Q_CONVERT(0.932993),
      DSP16_Q_CONVERT(-0.970772), DSP16_Q_CONVERT(-0.240003), DSP16_Q_CONVERT(-0.355598), DSP16_Q_CONVERT(0.934639),
      DSP16_Q_CONVERT(-0.971504), DSP16_Q_CONVERT(-0.237024), DSP16_Q_CONVERT(-0.351293), DSP16_Q_CONVERT(0.936266),
      DSP16_Q_CONVERT(-0.972226), DSP16_Q_CONVERT(-0.234042), DSP16_Q_CONVERT(-0.346980), DSP16_Q_CONVERT(0.937872),
      DSP16_Q_CONVERT(-0.972940), DSP16_Q_CONVERT(-0.231058), DSP16_Q_CONVERT(-0.342661), DSP16_Q_CONVERT(0.939459),
      DSP16_Q_CONVERT(-0.973644), DSP16_Q_CONVERT(-0.228072), DSP16_Q_CONVERT(-0.338334), DSP16_Q_CONVERT(0.941026),
      DSP16_Q_CONVERT(-0.974339), DSP16_Q_CONVERT(-0.225084), DSP16_Q_CONVERT(-0.334000), DSP16_Q_CONVERT(0.942573),
      DSP16_Q_CONVERT(-0.975025), DSP16_Q_CONVERT(-0.222094), DSP16_Q_CONVERT(-0.329658), DSP16_Q_CONVERT(0.944100),
      DSP16_Q_CONVERT(-0.975702), DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(-0.325310), DSP16_Q_CONVERT(0.945607),
      DSP16_Q_CONVERT(-0.976370), DSP16_Q_CONVERT(-0.216107), DSP16_Q_CONVERT(-0.320955), DSP16_Q_CONVERT(0.947094),
      DSP16_Q_CONVERT(-0.977028), DSP16_Q_CONVERT(-0.213110), DSP16_Q_CONVERT(-0.316593), DSP16_Q_CONVERT(0.948561),
      DSP16_Q_CONVERT(-0.977677), DSP16_Q_CONVERT(-0.210112), DSP16_Q_CONVERT(-0.312225), DSP16_Q_CONVERT(0.950008),
      DSP16_Q_CONVERT(-0.978317), DSP16_Q_CONVERT(-0.207111), DSP16_Q_CONVERT(-0.307850), DSP16_Q_CONVERT(0.951435),
      DSP16_Q_CONVERT(-0.978948), DSP16_Q_CONVERT(-0.204109), DSP16_Q_CONVERT(-0.303468), DSP16_Q_CONVERT(0.952842),
      DSP16_Q_CONVERT(-0.979570), DSP16_Q_CONVERT(-0.201105), DSP16_Q_CONVERT(-0.299080), DSP16_Q_CONVERT(0.954228),
      DSP16_Q_CONVERT(-0.980182), DSP16_Q_CONVERT(-0.198098), DSP16_Q_CONVERT(-0.294685), DSP16_Q_CONVERT(0.955594),
      DSP16_Q_CONVERT(-0.980785), DSP16_Q_CONVERT(-0.195090), DSP16_Q_CONVERT(-0.290285), DSP16_Q_CONVERT(0.956940),
      DSP16_Q_CONVERT(-0.981379), DSP16_Q_CONVERT(-0.192080), DSP16_Q_CONVERT(-0.285878), DSP16_Q_CONVERT(0.958266),
      DSP16_Q_CONVERT(-0.981964), DSP16_Q_CONVERT(-0.189069), DSP16_Q_CONVERT(-0.281465), DSP16_Q_CONVERT(0.959572),
      DSP16_Q_CONVERT(-0.982539), DSP16_Q_CONVERT(-0.186055), DSP16_Q_CONVERT(-0.277046), DSP16_Q_CONVERT(0.960857),
      DSP16_Q_CONVERT(-0.983105), DSP16_Q_CONVERT(-0.183040), DSP16_Q_CONVERT(-0.272621), DSP16_Q_CONVERT(0.962121),
      DSP16_Q_CONVERT(-0.983662), DSP16_Q_CONVERT(-0.180023), DSP16_Q_CONVERT(-0.268191), DSP16_Q_CONVERT(0.963366),
      DSP16_Q_CONVERT(-0.984210), DSP16_Q_CONVERT(-0.177004), DSP16_Q_CONVERT(-0.263755), DSP16_Q_CONVERT(0.964590),
      DSP16_Q_CONVERT(-0.984749), DSP16_Q_CONVERT(-0.173984), DSP16_Q_CONVERT(-0.259313), DSP16_Q_CONVERT(0.965793),
      DSP16_Q_CONVERT(-0.985278), DSP16_Q_CONVERT(-0.170962), DSP16_Q_CONVERT(-0.254866), DSP16_Q_CONVERT(0.966976),
      DSP16_Q_CONVERT(-0.985798), DSP16_Q_CONVERT(-0.167938), DSP16_Q_CONVERT(-0.250413), DSP16_Q_CONVERT(0.968139),
      DSP16_Q_CONVERT(-0.986308), DSP16_Q_CONVERT(-0.164913), DSP16_Q_CONVERT(-0.245955), DSP16_Q_CONVERT(0.969281),
      DSP16_Q_CONVERT(-0.986809), DSP16_Q_CONVERT(-0.161886), DSP16_Q_CONVERT(-0.241492), DSP16_Q_CONVERT(0.970403),
      DSP16_Q_CONVERT(-0.987301), DSP16_Q_CONVERT(-0.158858), DSP16_Q_CONVERT(-0.237024), DSP16_Q_CONVERT(0.971504),
      DSP16_Q_CONVERT(-0.987784), DSP16_Q_CONVERT(-0.155828), DSP16_Q_CONVERT(-0.232550), DSP16_Q_CONVERT(0.972584),
      DSP16_Q_CONVERT(-0.988258), DSP16_Q_CONVERT(-0.152797), DSP16_Q_CONVERT(-0.228072), DSP16_Q_CONVERT(0.973644),
      DSP16_Q_CONVERT(-0.988722), DSP16_Q_CONVERT(-0.149765), DSP16_Q_CONVERT(-0.223589), DSP16_Q_CONVERT(0.974684),
      DSP16_Q_CONVERT(-0.989177), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(-0.219101), DSP16_Q_CONVERT(0.975702),
      DSP16_Q_CONVERT(-0.989622), DSP16_Q_CONVERT(-0.143695), DSP16_Q_CONVERT(-0.214609), DSP16_Q_CONVERT(0.976700),
      DSP16_Q_CONVERT(-0.990058), DSP16_Q_CONVERT(-0.140658), DSP16_Q_CONVERT(-0.210112), DSP16_Q_CONVERT(0.977677),
      DSP16_Q_CONVERT(-0.990485), DSP16_Q_CONVERT(-0.137620), DSP16_Q_CONVERT(-0.205610), DSP16_Q_CONVERT(0.978634),
      DSP16_Q_CONVERT(-0.990903), DSP16_Q_CONVERT(-0.134581), DSP16_Q_CONVERT(-0.201105), DSP16_Q_CONVERT(0.979570),
      DSP16_Q_CONVERT(-0.991311), DSP16_Q_CONVERT(-0.131540), DSP16_Q_CONVERT(-0.196595), DSP16_Q_CONVERT(0.980485),
      DSP16_Q_CONVERT(-0.991710), DSP16_Q_CONVERT(-0.128498), DSP16_Q_CONVERT(-0.192080), DSP16_Q_CONVERT(0.981379),
      DSP16_Q_CONVERT(-0.992099), DSP16_Q_CONVERT(-0.125455), DSP16_Q_CONVERT(-0.187562), DSP16_Q_CONVERT(0.982253),
      DSP16_Q_CONVERT(-0.992480), DSP16_Q_CONVERT(-0.122411), DSP16_Q_CONVERT(-0.183040), DSP16_Q_CONVERT(0.983105),
      DSP16_Q_CONVERT(-0.992850), DSP16_Q_CONVERT(-0.119365), DSP16_Q_CONVERT(-0.178514), DSP16_Q_CONVERT(0.983937),
      DSP16_Q_CONVERT(-0.993212), DSP16_Q_CONVERT(-0.116319), DSP16_Q_CONVERT(-0.173984), DSP16_Q_CONVERT(0.984749),
      DSP16_Q_CONVERT(-0.993564), DSP16_Q_CONVERT(-0.113271), DSP16_Q_CONVERT(-0.169450), DSP16_Q_CONVERT(0.985539),
      DSP16_Q_CONVERT(-0.993907), DSP16_Q_CONVERT(-0.110222), DSP16_Q_CONVERT(-0.164913), DSP16_Q_CONVERT(0.986308),
      DSP16_Q_CONVERT(-0.994240), DSP16_Q_CONVERT(-0.107172), DSP16_Q_CONVERT(-0.160372), DSP16_Q_CONVERT(0.987057),
      DSP16_Q_CONVERT(-0.994565), DSP16_Q_CONVERT(-0.104122), DSP16_Q_CONVERT(-0.155828), DSP16_Q_CONVERT(0.987784),
      DSP16_Q_CONVERT(-0.994879), DSP16_Q_CONVERT(-0.101070), DSP16_Q_CONVERT(-0.151281), DSP16_Q_CONVERT(0.988491),
      DSP16_Q_CONVERT(-0.995185), DSP16_Q_CONVERT(-0.098017), DSP16_Q_CONVERT(-0.146730), DSP16_Q_CONVERT(0.989177),
      DSP16_Q_CONVERT(-0.995481), DSP16_Q_CONVERT(-0.094963), DSP16_Q_CONVERT(-0.142177), DSP16_Q_CONVERT(0.989841),
      DSP16_Q_CONVERT(-0.995767), DSP16_Q_CONVERT(-0.091909), DSP16_Q_CONVERT(-0.137620), DSP16_Q_CONVERT(0.990485),
      DSP16_Q_CONVERT(-0.996045), DSP16_Q_CONVERT(-0.088854), DSP16_Q_CONVERT(-0.133061), DSP16_Q_CONVERT(0.991108),
      DSP16_Q_CONVERT(-0.996313), DSP16_Q_CONVERT(-0.085797), DSP16_Q_CONVERT(-0.128498), DSP16_Q_CONVERT(0.991710),
      DSP16_Q_CONVERT(-0.996571), DSP16_Q_CONVERT(-0.082740), DSP16_Q_CONVERT(-0.123933), DSP16_Q_CONVERT(0.992291),
      DSP16_Q_CONVERT(-0.996820), DSP16_Q_CONVERT(-0.079682), DSP16_Q_CONVERT(-0.119365), DSP16_Q_CONVERT(0.992850),
      DSP16_Q_CONVERT(-0.997060), DSP16_Q_CONVERT(-0.076624), DSP16_Q_CONVERT(-0.114795), DSP16_Q_CONVERT(0.993389),
      DSP16_Q_CONVERT(-0.997290), DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(-0.110222), DSP16_Q_CONVERT(0.993907),
      DSP16_Q_CONVERT(-0.997511), DSP16_Q_CONVERT(-0.070505), DSP16_Q_CONVERT(-0.105647), DSP16_Q_CONVERT(0.994404),
      DSP16_Q_CONVERT(-0.997723), DSP16_Q_CONVERT(-0.067444), DSP16_Q_CONVERT(-0.101070), DSP16_Q_CONVERT(0.994879),
      DSP16_Q_CONVERT(-0.997925), DSP16_Q_CONVERT(-0.064383), DSP16_Q_CONVERT(-0.096490), DSP16_Q_CONVERT(0.995334),
      DSP16_Q_CONVERT(-0.998118), DSP16_Q_CONVERT(-0.061321), DSP16_Q_CONVERT(-0.091909), DSP16_Q_CONVERT(0.995767),
      DSP16_Q_CONVERT(-0.998302), DSP16_Q_CONVERT(-0.058258), DSP16_Q_CONVERT(-0.087326), DSP16_Q_CONVERT(0.996180),
      DSP16_Q_CONVERT(-0.998476), DSP16_Q_CONVERT(-0.055195), DSP16_Q_CONVERT(-0.082740), DSP16_Q_CONVERT(0.996571),
      DSP16_Q_CONVERT(-0.998640), DSP16_Q_CONVERT(-0.052132), DSP16_Q_CONVERT(-0.078153), DSP16_Q_CONVERT(0.996941),
      DSP16_Q_CONVERT(-0.998795), DSP16_Q_CONVERT(-0.049068), DSP16_Q_CONVERT(-0.073565), DSP16_Q_CONVERT(0.997290),
      DSP16_Q_CONVERT(-0.998941), DSP16_Q_CONVERT(-0.046003), DSP16_Q_CONVERT(-0.068974), DSP16_Q_CONVERT(0.997618),
      DSP16_Q_CONVERT(-0.999078), DSP16_Q_CONVERT(-0.042938), DSP16_Q_CONVERT(-0.064383), DSP16_Q_CONVERT(0.997925),
      DSP16_Q_CONVERT(-0.999205), DSP16_Q_CONVERT(-0.039873), DSP16_Q_CONVERT(-0.059790), DSP16_Q_CONVERT(0.998211),
      DSP16_Q_CONVERT(-0.999322), DSP16_Q_CONVERT(-0.036807), DSP16_Q_CONVERT(-0.055195), DSP16_Q_CONVERT(0.998476),
      DSP16_Q_CONVERT(-0.999431), DSP16_Q_CONVERT(-0.033741), DSP16_Q_CONVERT(-0.050600), DSP16_Q_CONVERT(0.998719),
      DSP16_Q_CONVERT(-0.999529), DSP16_Q_CONVERT(-0.030675), DSP16_Q_CONVERT(-0.046003), DSP16_Q_CONVERT(0.998941),
      DSP16_Q_CONVERT(-0.999619), DSP16_Q_CONVERT(-0.027608), DSP16_Q_CONVERT(-0.041406), DSP16_Q_CONVERT(0.999142),
      DSP16_Q_CONVERT(-0.999699), DSP16_Q_CONVERT(-0.024541), DSP16_Q_CONVERT(-0.036807), DSP16_Q_CONVERT(0.999322),
      DSP16_Q_CONVERT(-0.999769), DSP16_Q_CONVERT(-0.021474), DSP16_Q_CONVERT(-0.032208), DSP16_Q_CONVERT(0.999481),
      DSP16_Q_CONVERT(-0.999831), DSP16_Q_CONVERT(-0.018407), DSP16_Q_CONVERT(-0.027608), DSP16_Q_CONVERT(0.999619),
      DSP16_Q_CONVERT(-0.999882), DSP16_Q_CONVERT(-0.015339), DSP16_Q_CONVERT(-0.023008), DSP16_Q_CONVERT(0.999735),
      DSP16_Q_CONVERT(-0.999925), DSP16_Q_CONVERT(-0.012272), DSP16_Q_CONVERT(-0.018407), DSP16_Q_CONVERT(0.999831),
      DSP16_Q_CONVERT(-0.999958), DSP16_Q_CONVERT(-0.009204), DSP16_Q_CONVERT(-0.013805), DSP16_Q_CONVERT(0.999905),
      DSP16_Q_CONVERT(-0.999981), DSP16_Q_CONVERT(-0.006136), DSP16_Q_CONVERT(-0.009204), DSP16_Q_CONVERT(0.999958),
      DSP16_Q_CONVERT(-0.999995), DSP16_Q_CONVERT(-0.003068), DSP16_Q_CONVERT(-0.004602), DSP16_Q_CONVERT(0.999989)
    };
#  endif

//! \}

#else
#  error This N-point FFT is not supported. Make sure you are using a 4^n-point FFT, else check the DSP16_N_TWIDDLE_FACTORS constant.
#endif
