/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief Generic 16-bit lowpass FIR filter design function.
 *
 * This file contains the code of the lowpass FIR filter design.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 *****************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */

#include "dsp.h"

#if defined(FORCE_ALL_GENERICS) || \
    defined(FORCE_GENERIC_FILT16_LPFIRDESIGN) || \
    !defined(TARGET_SPECIFIC_FILT16_LPFIRDESIGN)

#ifndef Max
# define Max(a, b)           (((a) > (b)) ?  (a) : (b))
#endif

#define DSP16_MUL_TO_DSP32(num1, num2) \
  ((DSP32_QB > DSP16_QB*2)? \
  ((((S32) (num1)) * ((S32) (num2))) << (DSP32_QB - DSP16_QB*2)) \
  :((((S32) (num1)) * ((S32) (num2))) >> (DSP16_QB*2 - DSP32_QB)))

#define DSP32_DIV_TO_DSP16(num1, num2) \
  ((DSP32_QB > DSP16_QB*2)? \
  (((num1) >> (DSP32_QB - DSP16_QB*2)) / ((S32) (num2))) \
  :(((num1) << (DSP16_QB*2 - DSP32_QB)) / ((S32) (num2))))

void dsp16_filt_lpfirdesign(dsp16_t *c, int fc, int fs, int order, dsp16_win_fct_t dsp16_win_fct, dsp_filt_design_options_t options)
{
  S32 n_fc;
  S32 s32_temp1;
  S32 s32_temp2;
  dsp16_t dsp16_temp;
  int num, i;
  dsp32_t dsp32_temp;
  dsp16_t norm;

  if (!order)
    return;
  // Calculate normalized cutoff frequency mulitplied by 2
  n_fc = (((S32) fc) << (DSP16_QB + 1)) / fs;
  num = order / 2 + 1;

  if (options == DSP_FILT_DESIGN_OPTIONS_NORMALIZE)
  {
    // Calculate ideal lowpass filter impulse response coefficients:
    c[0] = (n_fc / 2);

    for(i=1; i<order; i++)
    {
      s32_temp1 = (S32) DSP_Q(32 - DSP16_QB, DSP16_QB, (double) i + 1.) / num;
      s32_temp2 = (S32) DSP_Q(32 - DSP16_QB, DSP16_QB, CST_PI * ((double) i + 1.)) / num;
      s32_temp2 >>= 1;
      dsp32_temp = dsp32_op_sin(DSP16_MUL_TO_DSP32(s32_temp1, n_fc));
      dsp16_temp = DSP32_DIV_TO_DSP16(dsp32_temp, s32_temp2);
      c[i] = dsp16_temp;
    }

    norm = dsp16_op_abs(dsp16_vect_max(c, order));
    norm = Max(norm, dsp16_op_abs(dsp16_vect_min(c, order)));

    // Create the window or fills the buffer with ones
    if (dsp16_win_fct)
      dsp16_win_fct(c, order);
    else
    {
      for(i=0; i<order; i++)
        c[i] = DSP16_Q(1.);
    }

    c[0] = dsp16_op_div(dsp16_op_mul(c[0], n_fc), norm);
    for(i=1; i<order; i++)
    {
      s32_temp1 = (S32) DSP_Q(32 - DSP16_QB, DSP16_QB, (double) i + 1.) / num;
      s32_temp2 = (S32) DSP_Q(32 - DSP16_QB, DSP16_QB, CST_PI * ((double) i + 1.));
      s32_temp2 = (((S64) s32_temp2) * norm) >> DSP16_QB;
      s32_temp2 = s32_temp2 / num + 1;
      dsp32_temp = dsp32_op_sin(DSP16_MUL_TO_DSP32(s32_temp1, n_fc));
      dsp16_temp = DSP32_DIV_TO_DSP16(dsp32_temp, s32_temp2);
      c[i] = dsp16_op_mul(c[i], dsp16_temp);
    }
  }
  else
  {
    // Create the window or fills the buffer with ones
    if (dsp16_win_fct)
      dsp16_win_fct(c, order);
    else
    {
      for(i=0; i<order; i++)
        c[i] = DSP16_Q(1.);
    }
    c[0] = dsp16_op_mul(c[0], n_fc);
    for(i=1; i<order; i++)
    {
      s32_temp1 = (S32) DSP_Q(32 - DSP16_QB, DSP16_QB, (double) i + 1.) / num;
      s32_temp2 = (S32) DSP_Q(32 - DSP16_QB, DSP16_QB, CST_PI * ((double) i + 1.)) / num;
      dsp32_temp = dsp32_op_sin(DSP16_MUL_TO_DSP32(s32_temp1, n_fc));
      dsp16_temp = DSP32_DIV_TO_DSP16(dsp32_temp, s32_temp2);
      c[i] = dsp16_op_mul(c[i], dsp16_temp);
    }
  }
}

#endif
