/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file ******************************************************************
 *
 * \brief Autobaud detection.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ***************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
#include "compiler.h"
#include "autobaud.h"

#define USE_BUILT_IN_64_BITS_OPERATORS    FALSE

int autobaud_detect_osc_cmp(const struct autobaud_cfg_osc_cmp_t *cfg)
{
  int delay_osc_ref, delay_osc_target;
  int counter_ref, counter_target;
  int estimated_freq_hz;
  int previous_estimated_freq_hz;
  Bool waiting;

  // Compute approximate delays to read counter values
  cfg->osc_ref_counter_reset();
  delay_osc_ref = cfg->osc_ref_counter_value();
  cfg->osc_target_counter_reset();
  delay_osc_target = cfg->osc_target_counter_value();

  cfg->osc_ref_counter_reset();
  cfg->osc_target_counter_reset();

  estimated_freq_hz = 0;
  counter_ref = 0;
  counter_target = 0;
  do
  {
    int previous_counter_ref, previous_counter_target;
    // For a 64-bit mutliplication
    unsigned long a, b, c, d, temp1, temp2, temp3;
    
    waiting = TRUE;
    // Assigns previosu values
    previous_estimated_freq_hz = estimated_freq_hz;
    previous_counter_ref = counter_ref;
    previous_counter_target = counter_target;
    // Reads the timer counters
    counter_ref = cfg->osc_ref_counter_value();
    counter_target = cfg->osc_target_counter_value();
    // Makes sure the counter are not wrapped
    if (previous_counter_ref > counter_ref ||
        previous_counter_target > counter_target)
      break;
    if (counter_target <= delay_osc_target || counter_ref <= delay_osc_ref)
      continue;
    // Substracts the delay due to the functions' call
    counter_ref -= delay_osc_ref;
    counter_target -= delay_osc_target;
    // Estimates the frequency
#if defined(USE_BUILT_IN_64_BITS_OPERATORS) && USE_BUILT_IN_64_BITS_OPERATORS == TRUE
    estimated_freq_hz = ((unsigned long long) counter_target * cfg->osc_ref_freq_hz) / counter_ref;
#else
    // 64-bit Mutliplication
    a = (counter_target >> 16);
    b = (counter_target & 0xFFFF);
    c = (cfg->osc_ref_freq_hz >> 16);
    d = (cfg->osc_ref_freq_hz & 0xFFFF);
    temp1 = a*c;
    temp2 = a*d + b*c;
    temp3 = b*d;
    // counter_target * cfg->osc_ref_freq_hz = a*2^32 + b
    a = temp1 + (temp2 >> 16);
    b = ((temp2 & 0xFFFF) << 16) + temp3;
    estimated_freq_hz = 0;
    for (c=0; c<8; c++)
    {
      a = ((a << 4) + (b >> 28));
      b <<= 4;
      estimated_freq_hz = (estimated_freq_hz << 4) + (a / counter_ref);
      a = a % counter_ref;
    }
#endif
    if (!estimated_freq_hz)
      continue;
    waiting = FALSE;

    // Stay in the loop until the estimated frequency stabilizes
  } while (waiting || abs(previous_estimated_freq_hz - estimated_freq_hz) > cfg->convergence_rate);

   return estimated_freq_hz;
}

int autobaud_match_frequency(int freq_hz, const struct autobaud_matching_freq_t *match)
{
  int i, current_candidate = -1;
  unsigned int current_excursion = (unsigned int) -1;

  for (i=0; i<match->nb_entries; i++)
  {
    if (current_excursion > abs(freq_hz - match->freq_hz[i]))
    {
      current_excursion = (unsigned int) abs(freq_hz - match->freq_hz[i]);
      current_candidate = i;
    }
  }
  if (current_candidate == -1)
    return freq_hz;
  return match->freq_hz[current_candidate];
}
