/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 * \brief Timer/Counter example 3.
 *
 * This example will start a timer/counter and generate a "tick" interrupt each
 * millisecond.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a TC module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/*! \page License
 * Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
 
 /*! \mainpage
 * \section intro Introduction
 * This is the documentation for the data structures, functions, variables,
 * defines, enums, and typedefs for the TC driver.\n It also gives an example of
 * the usage of the TC module.
 *
 * This example will start a timer/counter and generate a "tick" interrupt each millisecond.
 *
 * The selected timer input clock is the internal clock labelled TC3
 * referred to as TIMER_CLOCK3 in the datasheet. TIMER_CLOCK3 is connected to fPBA / 8 (see datasheet).
 *
 * The 16-bit timer/counter channel will cycle from 0x0000 to RC. RC is initialized
 * to (fPBA / 8) / 1000, so that an interrupt will be triggered every 1 ms. Upon interrupt,
 * a GPIO pin is toggled thus producing a square signal of frequency 500 Hz.
 * Check the toggled GPIO pin on an oscilloscope and see the square signal at frequency 500 Hz.
 * \note
 * - On AT32UC3A0512, AT32UC3B0256 and AT32UC3A3256, the toggled GPIO pin is GPIO3 which is mapped on PA03.
 * - On AT32UC3C0512 the toggled GPIO pin is GPIO3 which is mapped on PA08.
 * - On AT32UC3L064, the toggled GPIO pin is GPIO21 which is mapped on PA21; with
 *   the STK600 + RCUC3L0 routing card, PA21 is routed to STK600.PORTC.PC5.
 *
 * Also, a message is displayed on USART1 and the 'Timer:' value is updated every second.
 * The displayed message is:\n
 * <code> ATMEL \n AVR32 UC3 - TC example \n Timer: 721000 ms </code>
 *
 * \section files Main Files
 * - tc.c: TC driver;
 * - tc.h: TC driver header file;
 * - tc_example3.c: TC example 3.
 *
 * \section compinfo Compilation Info
 * This software was written for the GNU GCC for AVR32 and IAR Systems compiler
 * for AVR32. Other compilers may or may not work.
 *
 * \section deviceinfo Device Info
 * All AVR32 devices with a TC module can be used. This example has been tested
 * with the following board:
 * - EVK1100 evaluation kit
 * - EVK1101 evaluation kit
 * - AT32UC3C-EK evaluation kit
 * - EVK1104 evaluation kit
 * - STK600 starter kit with the RCUC3L0 routing card, 12MHz crystal in the STK600's crystal socket
 * - AT32UC3L-EK evaluation kit
 *
 * \section setupinfo Setup Information
 * This example has been tested with the following configuration:
 * - EVK1100, EVK1101, AT32UC3C-EK, EVK1104, AT32UC3L-EK evaluation kits; STK600
 *   starter kit with the RCUC3L0 routing card, 12MHz crystal in the STK600's crystal socket.
 * - CPU clock: 
 *            -- 12 MHz : EVK1100, EVK1101, EVK1104, EVK1105, AT32UC3L-EK evaluation kits; STK600+RCUC3L routing card;
 *            -- 16 MHz : AT32UC3C-EK;
 * - USART1 (on EVK1100 or EVK1101) connected to a PC serial port via a standard
 *   RS232 DB9 cable, or USART1 (on EVK1104), or USART3 (on AT32UC3L-EK) abstracted
 *   with a USB CDC connection to a PC; STK600 usart port for the STK600+RCUC3L
 *   setup (connect STK600.PORTE.PE2 to STK600.RS232 SPARE.TXD and STK600.PORTE.PE3
 *   to STK600.RS232 SPARE.RXD);
 * - PC terminal settings:
 *   - 57600 bps,
 *   - 8 data bits,
 *   - no parity bit,
 *   - 1 stop bit,
 *   - no flow control.
 *
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.atmel.com/products/AVR32/">Atmel AVR32</A>.\n
 * Support and FAQ: http://support.atmel.no/
 */


#include <avr32/io.h>
#if defined (__GNUC__)
#  include "intc.h"
#endif
#include "compiler.h"
#include "board.h"
#include "power_clocks_lib.h"
#include "gpio.h"
#include "tc.h"
#include "usart.h"


#define TC_CHANNEL    0

/*! \name USART Settings
 */
//! @{
#if BOARD == EVK1100
#  define EXAMPLE_TC                  (&AVR32_TC)
#  define EXAMPLE_TC_IRQ_GROUP        AVR32_TC_IRQ_GROUP
#  define EXAMPLE_TC_IRQ              AVR32_TC_IRQ0
#  define EXAMPLE_USART               (&AVR32_USART1)
#  define EXAMPLE_USART_RX_PIN        AVR32_USART1_RXD_0_0_PIN
#  define EXAMPLE_USART_RX_FUNCTION   AVR32_USART1_RXD_0_0_FUNCTION
#  define EXAMPLE_USART_TX_PIN        AVR32_USART1_TXD_0_0_PIN
#  define EXAMPLE_USART_TX_FUNCTION   AVR32_USART1_TXD_0_0_FUNCTION
#  define EXAMPLE_TOGGLED_PIN         AVR32_PIN_PA03
#  define FPBA                        FOSC0
#elif BOARD == EVK1101
#  define EXAMPLE_TC                  (&AVR32_TC)
#  define EXAMPLE_TC_IRQ_GROUP        AVR32_TC_IRQ_GROUP
#  define EXAMPLE_TC_IRQ              AVR32_TC_IRQ0
#  define EXAMPLE_USART               (&AVR32_USART1)
#  define EXAMPLE_USART_RX_PIN        AVR32_USART1_RXD_0_0_PIN
#  define EXAMPLE_USART_RX_FUNCTION   AVR32_USART1_RXD_0_0_FUNCTION
#  define EXAMPLE_USART_TX_PIN        AVR32_USART1_TXD_0_0_PIN
#  define EXAMPLE_USART_TX_FUNCTION   AVR32_USART1_TXD_0_0_FUNCTION
#  define EXAMPLE_TOGGLED_PIN         AVR32_PIN_PA03
#  define FPBA                        FOSC0
#elif BOARD == UC3C_EK
#  define EXAMPLE_TC                  (&AVR32_TC0)
#  define EXAMPLE_TC_IRQ_GROUP        AVR32_TC0_IRQ_GROUP
#  define EXAMPLE_TC_IRQ              AVR32_TC0_IRQ0
#  define EXAMPLE_USART               (&AVR32_USART2)
#  define EXAMPLE_USART_RX_PIN        AVR32_USART2_RXD_0_1_PIN
#  define EXAMPLE_USART_RX_FUNCTION   AVR32_USART2_RXD_0_1_FUNCTION
#  define EXAMPLE_USART_TX_PIN        AVR32_USART2_TXD_0_1_PIN
#  define EXAMPLE_USART_TX_FUNCTION   AVR32_USART2_TXD_0_1_FUNCTION
#  define EXAMPLE_TOGGLED_PIN         AVR32_PIN_PA08
#  define FPBA                        FOSC0
#elif BOARD == EVK1104
#  define EXAMPLE_TC                  (&AVR32_TC0)
#  define EXAMPLE_TC_IRQ_GROUP        AVR32_TC_IRQ_GROUP
#  define EXAMPLE_TC_IRQ              AVR32_TC0_IRQ0
#  define EXAMPLE_USART               (&AVR32_USART1)
#  define EXAMPLE_USART_RX_PIN        AVR32_USART1_RXD_0_0_PIN
#  define EXAMPLE_USART_RX_FUNCTION   AVR32_USART1_RXD_0_0_FUNCTION
#  define EXAMPLE_USART_TX_PIN        AVR32_USART1_TXD_0_0_PIN
#  define EXAMPLE_USART_TX_FUNCTION   AVR32_USART1_TXD_0_0_FUNCTION
#  define EXAMPLE_TOGGLED_PIN         AVR32_PIN_PA03
#  define FPBA                        FOSC0
#elif BOARD == STK600_RCUC3L0
#  define EXAMPLE_TC                  (&AVR32_TC1)
#  define EXAMPLE_TC_IRQ_GROUP        AVR32_TC1_IRQ_GROUP
#  define EXAMPLE_TC_IRQ              AVR32_TC1_IRQ0
#  define EXAMPLE_USART               (&AVR32_USART1)
#  define EXAMPLE_USART_RX_PIN        AVR32_USART1_RXD_0_1_PIN
#  define EXAMPLE_USART_RX_FUNCTION   AVR32_USART1_RXD_0_1_FUNCTION
// For the RX pin, connect STK600.PORTE.PE3 to STK600.RS232 SPARE.RXD
#  define EXAMPLE_USART_TX_PIN        AVR32_USART1_TXD_0_1_PIN
#  define EXAMPLE_USART_TX_FUNCTION   AVR32_USART1_TXD_0_1_FUNCTION
// For the TX pin, connect STK600.PORTE.PE2 to STK600.RS232 SPARE.TXD
#  define EXAMPLE_TOGGLED_PIN         AVR32_PIN_PA21
// With the STK600 + RCUC3L0 routing card, PA21 is routed to STK600.PORTC.PC5.
#  define FPBA                        FOSC0
#elif BOARD == UC3L_EK
#  define EXAMPLE_TC                  (&AVR32_TC1)
#  define EXAMPLE_TC_IRQ_GROUP        AVR32_TC1_IRQ_GROUP
#  define EXAMPLE_TC_IRQ              AVR32_TC1_IRQ0
#  define EXAMPLE_USART               (&AVR32_USART3)
#  define EXAMPLE_USART_RX_PIN        AVR32_USART3_RXD_0_0_PIN
#  define EXAMPLE_USART_RX_FUNCTION   AVR32_USART3_RXD_0_0_FUNCTION
#  define EXAMPLE_USART_TX_PIN        AVR32_USART3_TXD_0_0_PIN
#  define EXAMPLE_USART_TX_FUNCTION   AVR32_USART3_TXD_0_0_FUNCTION
#  define EXAMPLE_TOGGLED_PIN         AVR32_PIN_PA21 // LED0
#  define EXAMPLE_TARGET_DFLL_FREQ_HZ   96000000  // DFLL target frequency, in Hz
#  define EXAMPLE_TARGET_MCUCLK_FREQ_HZ 12000000  // MCU clock target frequency, in Hz
#  define EXAMPLE_TARGET_PBACLK_FREQ_HZ 12000000  // PBA clock target frequency, in Hz
#  define FPBA                          EXAMPLE_TARGET_PBACLK_FREQ_HZ
    /*! \name Parameters to pcl_configure_clocks().
     */
    //! @{
    static scif_gclk_opt_t gc_dfllif_ref_opt = { SCIF_GCCTRL_SLOWCLOCK, 0, OFF };
    static pcl_freq_param_t pcl_dfll_freq_param =
    {
      .main_clk_src = PCL_MC_DFLL0,
      .cpu_f        = EXAMPLE_TARGET_MCUCLK_FREQ_HZ,
      .pba_f        = EXAMPLE_TARGET_PBACLK_FREQ_HZ,
      .pbb_f        = EXAMPLE_TARGET_PBACLK_FREQ_HZ,
      .dfll_f       = EXAMPLE_TARGET_DFLL_FREQ_HZ,
      .pextra_params = &gc_dfllif_ref_opt
    };
    //! @}
#endif

#if !defined(EXAMPLE_USART)             || \
    !defined(EXAMPLE_USART_RX_PIN)      || \
    !defined(EXAMPLE_USART_RX_FUNCTION) || \
    !defined(EXAMPLE_USART_TX_PIN)      || \
    !defined(EXAMPLE_USART_TX_FUNCTION)
#  error The USART preprocessor configuration to use in this example is missing.
#endif

#if !defined(EXAMPLE_TC)             || \
    !defined(EXAMPLE_TC_IRQ_GROUP)      || \
    !defined(EXAMPLE_TC_IRQ)
#  error The TC preprocessor configuration to use in this example is missing.
#endif

#if !defined(EXAMPLE_TOGGLED_PIN)
#  error The preprocessor configuration of the GPIO pin to toggle is missing.
#endif
//! @}


// To specify we have to print a new time
volatile static int print_sec = 1;

volatile U32 tc_tick = 0;


/*! \brief TC interrupt.
 */
#if defined (__GNUC__)
__attribute__((__interrupt__))
#elif defined (__ICCAVR32__)
#pragma handler = EXAMPLE_TC_IRQ_GROUP, 1
__interrupt
#endif
static void tc_irq(void)
{
  // Increment the ms seconds counter
  tc_tick++;

  // Clear the interrupt flag. This is a side effect of reading the TC SR.
  tc_read_sr(EXAMPLE_TC, TC_CHANNEL);

  // specify that an interrupt has been raised
  print_sec = 1;
  // Toggle a GPIO pin (this pin is used as a regular GPIO pin).
  gpio_tgl_gpio_pin(EXAMPLE_TOGGLED_PIN);
}


/*! \brief Converts the given number to an ASCII decimal representation.
 */
static char *print_i(char *str, int n)
{
    int i = 10;

    str[i] = '\0';
    do
    {
      str[--i] = '0' + n%10;
      n /= 10;
    }while(n);

    return &str[i];
}


/*! \brief Main function:
 *  - Configure the CPU to run at 12MHz
 *  - Configure the USART
 *  - Register the TC interrupt (GCC only)
 *  - Configure, enable the CPCS (RC compare match) interrupt, and start a TC channel in waveform mode
 *  - In an infinite loop, update the USART message every second.
 */
int main(void)
{
  char temp[20];
  char *ptemp;

  volatile avr32_tc_t *tc = EXAMPLE_TC;

  // Options for waveform genration.
  static const tc_waveform_opt_t WAVEFORM_OPT =
  {
    .channel  = TC_CHANNEL,                        // Channel selection.

    .bswtrg   = TC_EVT_EFFECT_NOOP,                // Software trigger effect on TIOB.
    .beevt    = TC_EVT_EFFECT_NOOP,                // External event effect on TIOB.
    .bcpc     = TC_EVT_EFFECT_NOOP,                // RC compare effect on TIOB.
    .bcpb     = TC_EVT_EFFECT_NOOP,                // RB compare effect on TIOB.

    .aswtrg   = TC_EVT_EFFECT_NOOP,                // Software trigger effect on TIOA.
    .aeevt    = TC_EVT_EFFECT_NOOP,                // External event effect on TIOA.
    .acpc     = TC_EVT_EFFECT_NOOP,                // RC compare effect on TIOA: toggle.
    .acpa     = TC_EVT_EFFECT_NOOP,                // RA compare effect on TIOA: toggle (other possibilities are none, set and clear).

    .wavsel   = TC_WAVEFORM_SEL_UP_MODE_RC_TRIGGER,// Waveform selection: Up mode with automatic trigger(reset) on RC compare.
    .enetrg   = FALSE,                             // External event trigger enable.
    .eevt     = 0,                                 // External event selection.
    .eevtedg  = TC_SEL_NO_EDGE,                    // External event edge selection.
    .cpcdis   = FALSE,                             // Counter disable when RC compare.
    .cpcstop  = FALSE,                             // Counter clock stopped with RC compare.

    .burst    = FALSE,                             // Burst signal selection.
    .clki     = FALSE,                             // Clock inversion.
    .tcclks   = TC_CLOCK_SOURCE_TC3                // Internal source clock 3, connected to fPBA / 8.
  };

  static const tc_interrupt_t TC_INTERRUPT =
  {
    .etrgs = 0,
    .ldrbs = 0,
    .ldras = 0,
    .cpcs  = 1,
    .cpbs  = 0,
    .cpas  = 0,
    .lovrs = 0,
    .covfs = 0
  };

#if BOARD == UC3L_EK
  // Note: on the AT32UC3L-EK board, there is no crystal/external clock connected
  // to the OSC0 pinout XIN0/XOUT0. We shall then program the DFLL and switch the
  // main clock source to the DFLL.
  pcl_configure_clocks(&pcl_dfll_freq_param);
  // Note: since it is dynamically computing the appropriate field values of the
  // configuration registers from the parameters structure, this function is not
  // optimal in terms of code size. For a code size optimal solution, it is better
  // to create a new function from pcl_configure_clocks_dfll0() and modify it
  // to use preprocessor computation from pre-defined target frequencies.
#else
  // Configure Osc0 in crystal mode (i.e. use of an external crystal source, with
  // frequency FOSC0) with an appropriate startup time then switch the main clock
  // source to Osc0.
  pcl_switch_to_osc(PCL_OSC0, FOSC0, OSC0_STARTUP);
#endif

  static const gpio_map_t USART_GPIO_MAP =
  {
    {EXAMPLE_USART_RX_PIN, EXAMPLE_USART_RX_FUNCTION},
    {EXAMPLE_USART_TX_PIN, EXAMPLE_USART_TX_FUNCTION}
  };

  // USART options.
  static const usart_options_t USART_OPTIONS =
  {
    .baudrate     = 57600,
    .charlength   = 8,
    .paritytype   = USART_NO_PARITY,
    .stopbits     = USART_1_STOPBIT,
    .channelmode  = 0
  };

  // Assign GPIO pins to USART.
  gpio_enable_module(USART_GPIO_MAP, sizeof(USART_GPIO_MAP) / sizeof(USART_GPIO_MAP[0]));

  // Initialize USART in RS232 mode at 12MHz.
  usart_init_rs232(EXAMPLE_USART, &USART_OPTIONS, FPBA);

  // Welcome sentence.
  usart_write_line(EXAMPLE_USART, "\x1B[2J\x1B[H\r\nATMEL\r\n");
  usart_write_line(EXAMPLE_USART, "AVR32 UC3 - TC example\r\n");

  Disable_global_interrupt();

  // The INTC driver has to be used only for GNU GCC for AVR32.
#if defined (__GNUC__)
  // Initialize interrupt vectors.
  INTC_init_interrupts();

  // Register the RTC interrupt handler to the interrupt controller.
  INTC_register_interrupt(&tc_irq, EXAMPLE_TC_IRQ, AVR32_INTC_INT0);
#endif

  Enable_global_interrupt();

  // Initialize the timer/counter.
  tc_init_waveform(tc, &WAVEFORM_OPT);         // Initialize the timer/counter waveform.

  // Set the compare triggers.
  // Remember TC counter is 16-bits, so counting second is not possible with fPBA = 12 MHz.
  // We configure it to count ms.
  // We want: (1/(fPBA/8)) * RC = 0.001 s, hence RC = (fPBA/8) / 1000 = 1500 to get an interrupt every 1 ms.
  tc_write_rc(tc, TC_CHANNEL, (FPBA / 8) / 1000); // Set RC value.

  tc_configure_interrupts(tc, TC_CHANNEL, &TC_INTERRUPT);

  // Start the timer/counter.
  tc_start(tc, TC_CHANNEL);                    // And start the timer/counter.

  while(1)
  {
    // Update the display on USART every second.
    if ((print_sec) && (!(tc_tick%1000)))
    {
      // Set cursor to the position (1; 5)
      usart_write_line(EXAMPLE_USART, "\x1B[5;1H");
      ptemp = print_i(temp, tc_tick);
      usart_write_line(EXAMPLE_USART, "Timer: ");
      usart_write_line(EXAMPLE_USART, ptemp);
      usart_write_line(EXAMPLE_USART, " ms");
      print_sec = 0;
    }
  }
}
