/* This header file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief HMATRIX driver.
 *
 * This file defines a useful set of functions for the HMATRIX module on AVR32
 * devices.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a HMATRIX module.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
#ifndef _HMATRIX_H_
#define _HMATRIX_H_

#include <avr32/io.h>
#include "compiler.h"


/*!
 * Access schemes for selecting default master type accces
 */
enum e_granting_schemes {
  no_default_master,
  last_accessed_master,
  fixed_default_master
  };
typedef enum e_granting_schemes granting_scheme_t;

/*!
 * Burst type polixy
 */
enum e_burst_types {
  infinite = 0,
  single = 1,
  four_beat = 2,
  eight_beat = 3,
  sixteen_beat = 4
  };
typedef enum e_burst_types burst_type_t;

/*!
 * Arbitration for bus access
 */
enum e_arbitration_types {
  round_robin = 0,
  fixed_priority = 1
  };
typedef enum e_arbitration_types arbitration_type_t;

#ifdef CPU_TYPE
#if CPU_TYPE==ap7000

/*!
 * All HSB masters for the AVR32AP7000 series
 */
enum e_hsb_masters {
  cpu_data_cache = 0,
  cpu_instruction_cache = 1,
  hsb_hsb_bridge = 2,
  isi = 3,
  usb = 4,
  lcd = 5,
  macb0 = 6,
  macb1 = 7,
  dmac0 = 8,
  dmac1 = 9
  };
typedef enum e_hsb_masters hsb_master_t;

/*!
 * HSB slaves for the AVR32AP7000 series
 */
enum e_hsb_slaves {
  sram0 = 0,
  sram1 = 1,
  pba = 2,
  pbb = 3,
  ebi = 4,
  usb_data = 5,
  lcd_configuration = 6,
  dmac_configuration = 7
  };
typedef enum e_hsb_slaves hsb_slave_t;

/*!
 * Alternative memory devices for the AVR32AP7000 series
 */
enum e_memory_devices {
  SDRAM = 1,
  NAND = 3,
  CompactFlash0 = 4,
  CompactFlash1 = 5
  };
typedef enum e_memory_devices memory_devices_t;
#else
#error "High Speed Bus Matrix operations for your chosen CPU is not supported"
#endif
#endif

/*!
 * This function sets the burst type used for a  given master.
 * \param master  Which master to access
 * \param burst_type  The burst policy to use
 * \return none
 */
void hmatrix_set_master_burst_type(hsb_master_t master, burst_type_t burst_type);

/*!
 * This function will configure a HSB slave
 * \param slave    The slave to modify
 * \param default_master_type  Access type (granting scheme)
 * \param default_master  Set the default master. Only active if default_master_type is set to "fixed_default_master"
 * \param arb_type  Select arbitration scheme
 * \return none
 */
void hmatrix_configure_slave(hsb_slave_t slave, \
        U8 max_cycles_per_burst, \
        granting_scheme_t default_master_type, \
        hsb_master_t default_master, \
        arbitration_type_t arb_type );

/*!
 * This function can enable or disable the address decoding for the given master
 * \param master  Which master to access
 * \param enable  Enable / disable address decoding
 * \return none
 */
U32 hmatrix_master_remap_address_decoding(hsb_master_t master, bool enable);

/*!
 * This function will set the master access priority for the given slave
 * \param slave    Which slave to modify
 * \param master   Set priority for this master
 * \param priority The desired priority. Range: 0 through 15
 * \return status  SUCCESS or INVALID_ARGUMENT
 */
U32 hmatrix_slave_set_master_access_priority( hsb_slave_t slave,
            hsb_master_t master,
            U8 priority);

#endif  // _HMATRIX_H_
