/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief AVR32 UC3L FlashVault basic example: this file contains the 3rd-party
 * application that accesses public API functions of the FlashVault. This
 * 3rd-party application also configures and handles the COUNT/COMPARE match
 * interrupt. The purpose of this example is to show that no matter when this
 * interrupt occurs (whether while executing code in the FlashVault or in the
 * 3rd-party application) it will safely be handled by the interrupt handler of
 * the 3rd-party application.
 *
 * - Compiler:           GCC for AVR32
 * - Supported devices:  All AVR32UC3 devices that implement a secure execution state.
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
 
 /*! \mainpage
 * \section intro Introduction
 * This is the documentation for the Secure State basic example of usage.
 *
 * The Secure State allows execution of secure or secret code alongside 3rd-party
 * code on the same processor. The secret code will execute in the secure state,
 * and therefore be protected from hacking or readout by the 3rd-party code.
 *
 * Recommended reading: the Secure State chapter in the AVR32 UC Technical
 * Reference manual.
 *
 * \section applidescription Basic Example description
 * A 3rd-party application accesses public API functions exposed by the FlashVault.\n
 * This 3rd-party application also configures and handles the COUNT/COMPARE match
 * interrupt.\n The purpose of this example is to show that no matter when this
 * interrupt occurs (whether while executing code in the FlashVault or in the
 * 3rd-party application) it will safely be handled by the interrupt handler of
 * the 3rd-party code. This is happening because the FlashVault implementation
 * temporarily switches back to the 3rd-party code if an interrupt occurs while
 * the execution is in the FlashVault.
 *
 * <b>Visual behavior: </b>
 * - the 3rd-party application forever toggles LED1
 * - when a COUNT/COMPARE match interrupt occurs, LED0 toggles (use an oscilloscope
 *   to see the toggling because the interrupt occurs so often that LED0 appears
 *   to be steady on)
 * - Every time a COUNT/COMPARE match interrupt occurs while the execution is in
 *   the FlashVault, LED2 is toggled.
 *
 * \section files Main Files
 * - thirdparty_app.c: the 3rd-party application that accesses the FlashVault public API
 *   and handles the COUNT/COMPARE match interrupt.
 * - thirdparty_crt0.S: the 3rd-party C runtime startup assembly file (GCC)
 * - flashvault.S: contains the Secure world source code (boot process, SSCALL
 *   handling to public API functions, interrupt handler, public API functions)
 * - flashvault.h: FlashVault public API accessible by a third-party application
 * - link_uc3l064.lds: custom linker script to separate the FlashVault area and
 *   the 3rd-party application area
 *
 * \section compinfo Compilation Info
 * This software was written for the GCC for AVR32. Other compilers may or may not work.
 *
 * \section configinfo Configuration Information
 * This example has been tested with the following configuration:
 * - STK600+RCUC3L0 routing card; AT32UC3L-EK
 * - CPU clock: default (i.e. on RCOsc);
 * - For the STK600+RCUC3L0 setup, apply the following connections:
 *   - STK600.PORTA.PA4 -> STK600.LEDS.LED0
 *   - STK600.PORTA.PA5 -> STK600.LEDS.LED1
 *   - STK600.PORTA.PA6 -> STK600.LEDS.LED2
 *   - STK600.PORTA.PA7 -> STK600.LEDS.LED3
 *
 * \section loadcode Loading The Code
 * This is a step-by-step description of the steps to follow to load the code
 * and enable the FlashVault:
 * - avr32program chiperase  <b>: erase the flash and the gp fuses</b>
 * - avr32program erase -o -finternal\@0x80000000,64kB <b>: erase the user page</b>
 * - avr32program -pavrone program uc3l064-basic_flashvault_example.elf -euv -finternal\@0x80000000,64kB <b>: program the code</b>
 * - avr32program -pavrone writefuses -finternal\@0x80000000,64kB gp=0xFFEFFFFF <b>: set the SECURE fuses to SS on/SSD off</b> \n
 *   <b>OR</b> \n
 *   avr32program -pavrone writefuses -finternal\@0x80000000,64kB gp=0xFFDFFFFF <b>: set the SECURE fuses to SS on/SSD on</b>
 * - <b>reset the target to start the show.</b>
 *
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.atmel.com/products/AVR32/">Atmel AVR32</A>.\n
 * Support and FAQ: http://support.atmel.no/
 */

#include <avr32/io.h>
#include "board.h"
#include "intc.h"
#include "cycle_counter.h"
#include "gpio.h"
#include "flashvault.h"


#define NB_CLOCK_CYCLE_DELAY_SHORT    1000  // 8.69 ms if fCPU==115kHz

//******************************************************************************
//***
//***                T H I R D - P A R T Y  A P P L I C A T I O N
//***
//******************************************************************************

__attribute__((__section__(".thirdparty_data")))
static volatile unsigned char bFirstInterrupt = FALSE;

// Counter of COUNT/COMPARE matches.
__attribute__((__section__(".thirdparty_data")))
static volatile unsigned int u32NbCompareIrqTrigger = 0;

// Created just to check that dummy gets placed in the .bss.
static volatile int dummy;

// COUNT/COMPARE match interrupt handler
// GCC-specific syntax to declare an interrupt handler. The interrupt handler
// registration is done in the main function using the INTC software driver module.
__attribute__((__interrupt__)) static void compare_irq_handler(void)
{
  bFirstInterrupt = TRUE;
  // Count the number of times this IRQ handler is called.
  u32NbCompareIrqTrigger++;
  
  dummy++;
  
  // Toggle LED0.
  gpio_tgl_gpio_pin(LED0_GPIO);
  
  // Clear the pending interrupt(writing a value to the COMPARE register clears
  // any pending compare interrupt requests). Schedule the COUNT&COMPARE match
  // interrupt to happen every NB_CLOCK_CYCLE_DELAY_SHORT cycles.
  Set_sys_compare(NB_CLOCK_CYCLE_DELAY_SHORT);
}


static void delay(int d)
{
  volatile int i = d;
  while(i--);
}


//**
//** The non-secure world application
//**
__attribute__((__used__))
int main(void)
{
  U32 u32CompareVal;
  U32 u32CountVal;


  // Disable all interrupts.
  Disable_global_interrupt();

  INTC_init_interrupts();

  // Register the compare interrupt handler to the interrupt controller.
  // compare_irq_handler is the interrupt handler to register.
  // AVR32_CORE_COMPARE_IRQ is the IRQ of the interrupt handler to register.
  // AVR32_INTC_INT0 is the interrupt priority level to assign to the group of this IRQ.
  // void INTC_register_interrupt(__int_handler handler, unsigned int irq, unsigned int int_level);
  INTC_register_interrupt(&compare_irq_handler, AVR32_CORE_COMPARE_IRQ, AVR32_INTC_INT0);

  // Enable all interrupts.
  Enable_global_interrupt();

  // Schedule the COUNT&COMPARE match interrupt in NB_CLOCK_CYCLE_DELAY_SHORT
  // clock cycles from now.
  u32CountVal = Get_sys_count();

  u32CompareVal = u32CountVal + NB_CLOCK_CYCLE_DELAY_SHORT; // WARNING: MUST FIT IN 32bits.
  // If u32CompareVal ends up to be 0, make it 1 so that the COMPARE and exception
  // generation feature does not get disabled.
  if(0 == u32CompareVal)
  {
    u32CompareVal++;
  }

  Set_sys_compare(u32CompareVal); // GO

  // Wait for the first COUNT/COMPARE match interrupt to trigger.
  while(bFirstInterrupt == FALSE);

  while(1)
  {
    // Toggle led1. According to the FlashVault API, we must fill R8 with the
    // target function id before calling SSCALL.
    __asm__ __volatile__ (                                                     \
    /* Fill the api vector id. */                                              \
    "mov    r8, %[TGL_LED1]\n\t"                                                  \
                                                                               \
    /* Call the secure world. */                                               \
    ".int %[SSCALL]"                                                              \
    :                                                                          \
    : [SSCALL] "i" (INST_SSCALL),                                              \
      [TGL_LED1] "i" (FLASHVAULT_API_TGL_LED1)                                  \
    );

    // Insert a delay
    delay(5000);
  }
  return 0; // Never reached
}
