/* This file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief AVR32 UC3L FlashVault basic example: Secure world implementation offering
 * a public API; this implementation also temporarily switches back to outside the
 * secure world when an interrupt occurs.
 *
 * - Compiler:           GCC for AVR32
 * - Supported devices:  All AVR32UC3 devices that implement a secure execution state.
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */

#include <avr32/io.h>
#include "board.h"
#include "flashvault.h"

//! @{
//! \verbatim

//**
//** Pre-requesites: SSE fuse bit is 0 and SSDE fuse bit is 1 (i.e. Secure State
//** enabled, Secure State debug disabled) or SSE fuse bit is 1 and SSDE fuse bit
//** is 0 (i.e. Secure State enabled, Secure State Debug enabled)
//**
//** This module is a Secure world implementation offering a public API (to
//** toggle led1).
//**


//! Secure State default status
#define SEC_EXE_STATUS_DEFAULT      0x000000AA
//! Secure State interrupted status
#define SEC_EXE_STATUS_INTERRUPTED  0x00000011

//******************************************************************************
//***
//***              S E C U R E   W O R L D  =  F L A S H V A U L T
//***
//******************************************************************************
  // This must be linked @ 0x80000000 if it is to be run upon reset.
  .section  .reset, "ax", @progbits

  .global _start
  .type _start, @function
_start: // reset @ == 0x80000000
  // Jump to the C runtime startup routine.
  // lda.w   pc, sec_boot  // Use the linker option -mno-relax when using this pseudo-instruction.
  rjmp   sec_boot

  //**
  //** Secure State Event Handling table.
  //**
.org 0x4  // SSCALL handler entry point
  rjmp   sec_sscall_handle
.org 0x8  // Exceptions handler entry point
  rjmp   sec_exceptions_handle
.org 0xc  // NMI handler entry point
  rjmp   sec_nmi_handle
.org 0x10 // BREAKPOINT handler entry point
  rjmp   sec_bp_handle
.org 0x14 // Interrupts handler entry point
  rjmp   sec_irq_handle


  //**
  //** Boot
  //**
sec_boot:
  // Set the Secure Sections size to 1kB
  mov    r8, SECURE_FLASH_SIZE
  mtsr   AVR32_SS_ADRR, r8
  mtsr   AVR32_SS_ADRF, r8

  // Init the Secure State Stack pointer
  mov    sp, _e_sec_stack

  // Init the SS_RAR register to the start address of the zone outside the FlashVault.
  mov    r8, lo(AVR32_FLASH_ADDRESS+SECURE_FLASH_SIZE)
  orh    r8, hi(AVR32_FLASH_ADDRESS+SECURE_FLASH_SIZE)
  mtsr   AVR32_SS_RAR, r8

  // Init the SS_RSR register to Secure State Inactive (clear the SS bit of the
  // current SR register).
  mfsr   r8, AVR32_SR
  cbr    r8, 31
  mtsr   AVR32_SS_RSR, r8
  
  // Load initialized data having a global lifetime from the data LMA.
  lda.w  r0, sec_data
  lda.w  r1, sec_edata
  cp     r0, r1
  brhs   sec_idata_load_loop_end
  lda.w  r2, sec_data_lma
sec_idata_load_loop:
  ld.d   r4, r2++
  st.d   r0++, r4
  cp     r0, r1
  brlo   sec_idata_load_loop
sec_idata_load_loop_end:

  // Clear uninitialized data having a global lifetime in the blank static storage section.
  lda.w  r0, sec_bss_start
  lda.w  r1, sec_bss_end
  cp     r0, r1
  brhs   sec_udata_clear_loop_end
  mov    r2, 0
  mov    r3, 0
sec_udata_clear_loop:
  st.d   r0++, r2
  cp     r0, r1
  brlo   sec_udata_clear_loop
sec_udata_clear_loop_end:

  // Set the execution status of the secure world as default
  mov    r0, SEC_EXE_STATUS_DEFAULT
  mtsr   AVR32_SS_STATUS, r0

  // Switch to the application outside the FlashVault.
  .int INST_RETSS
  // Should never reached that point.


  //**
  //** SSCALL handler
  //** R8: public api vector number
  //**
sec_sscall_handle:
  // Temporary save of r0 before modifying it.
  st.w --sp, r0
  // Check if we're back from an interrupt handling performed in the application
  // outside the FlashVault.
  // If so we should resume the secure execution at the point where it was
  // interrupted.
  mfsr   r0, AVR32_SS_STATUS
  cp     r0, SEC_EXE_STATUS_INTERRUPTED
  // Restore r0
  ld.w   r0, sp++
  breq   sec_irq_restore_exe

  // We must save all registers on the stack else the application outside the
  // FlashVault may fail when it resumes!
  pushm  r0-r12, lr

  // Save SS_RAR and SS_RSR in secure memory before enabling IRQ
  mov    r0, ss_rar_save
  mfsr   r1, AVR32_SS_RAR
  st.w   r0[0], r1
  mov    r0, ss_rsr_save
  mfsr   r1, AVR32_SS_RSR
  st.w   r0[0], r1
  // Enable INT0 interrupts.
  csrf   AVR32_SR_GM
  csrf   AVR32_SR_I0M

  // Make sure R8 holds one of the public api vector number
  cp     r8, FLASHVAULT_API_TGL_LED1
  breq   sec_pub_tgl_led1
  // Go back to the application outside the FlashVault.
  mov    r12, FAIL
  // Expected fall-back to sec_exit()

  //**
  //** Common routine to leave the secure mode:
  //** Mask interrupts, restore RAR/RSR. Return value in r12
  //** Use the following addresses (RAR @0(i.e. in ss_rar_save), RSR @4(i.e. in ss_rsr_save))
  //** Cannot use stack since we need random access when handling interrupts.
  //**
sec_exit:
  // Disable all interrupts
  ssrf   AVR32_SR_GM
  // Restore RSR
  mov    r0, ss_rsr_save
  ld.w   r1, r0[0]
  mtsr   AVR32_SS_RSR, r1
  // Restore RAR
  mov    r0, ss_rar_save
  ld.w   r1, r0[0]
  sub    r1, -2 // Update return address to the instruction after SSCALL
  mtsr   AVR32_SS_RAR, r1
  // We must restore all registers from the stack else the application outside
  // the FlashVault may fail when it resumes!
  ldm    sp++, r0-r12, lr
  .int INST_RETSS


  //**
  //** Restore a previously interrupted secure mode execution:
  //** - Set the secure mode execution status back to default
  //** - Restore the SR & PC from the secure stack to resume the secure code execution
  //**
sec_irq_restore_exe:
  // Pop SR and PC to interrupted secure mode application SSCALL from stack.
  // Set the secure mode execution status back to default.
  mov    r0, SEC_EXE_STATUS_DEFAULT
  mtsr   AVR32_SS_STATUS, r0
  // Restore the SR & PC from the secure stack.
  // IRQs will then be reenabled due to SR restore.
  // What will happen if IRQ is received immediately? Must this be done atomically?
  ld.w   r0, sp++
  mtsr   AVR32_SR, r0
  // We must restore all registers from the stack else the secure code application
  // will fail when it resumes!
  sub    sp, 15*4           // Reset the SP to the appropriate position before popping R0-R12, LR.
  ldm    sp++, r0-r12, lr
  sub    sp, -1*4           // Reset the SP to the appropriate position before popping SS_RAR.
  ld.w   pc, sp++


  //**
  //**  Exceptions handler, NMI handler, breakpoint handler.
  //**  These are not expected in this appli; so just turn led3 on.
  //**
sec_exceptions_handle:
sec_nmi_handle:
sec_bp_handle:
  rcall  sec_set_led3_on
  rjmp   $


  //**
  //** Interrupts handler:
  //** Secure code was interrupted. Save required state on the secure stack, and
  //** return to the application outside the FlashVault. Write status info in
  //** SS_STATUS allowing the restore of the secure code execution once we get 
  //** back from the interrupt handling performed outside the FlashVault.
  //**
sec_irq_handle:
  // We must save all registers on the stack else the secure code application
  // will fail when it resumes!
  sub    sp, 2*4            // Save room for RAR and RSR
  pushm  r0-r12,lr          // Push R0-R12 & LR on the stack
  sub    sp, -16*4          // Reset the SP to the appropriate position before saving RAR & RSR.
  // Put RSR and RAR of the interrupted secure application on the secure stack.
  mfsr   r0, AVR32_SS_RAR
  st.w   --sp, r0
  mfsr   r0, AVR32_SS_RSR
  st.w   --sp, r0

  // Toggle LED2 to provide a visual information that this case occured.
  rcall  sec_tgl_led2

  // Update SS_STATUS with a value indicating that the secure code was interrupted.
  mov    r0, SEC_EXE_STATUS_INTERRUPTED
  mtsr   AVR32_SS_STATUS, r0

  // Restore the SS_RAR & SS_RSR from the original SSCALL call (these were saved
  // in the SSCALL handler in the ss_rar_save & ss_rsr_save variables).
  // Note that SS_RAR will then be set to the application outside the FlashVault
  // on the SSCALL instruction that was issued to reach the FlashVault API.
  mov   r0, ss_rar_save
  ld.w  r1, r0[0]
  mtsr  AVR32_SS_RAR, r1
  mov   r0, ss_rsr_save
  ld.w  r1, r0[0]
  mtsr  AVR32_SS_RSR, r1
  // Return to the application outside the FlashVault to handle the IRQ.
  // NOTE: In the world outside the FlashVault, once the IRQ handling is done,
  // rete will be executed which will restore execution on the SSCALL instruction
  // which will switch execution to the FlashVault in the SSCALL handler sec_sscall_handle().
  .int INST_RETSS




  //**
  //** Public API: sec_pub_tgl_led1
  //**
sec_pub_tgl_led1:
  // Set the GPIO bit in the GPER register to enable the GPIO.
  mov    r10, 1 << (LED1_GPIO & 0x1F)
  mov    r9, AVR32_GPIO_ADDRESS + 0x200*(LED1_GPIO >> 5)

sec_tgl_led_retss:
  // Set the GPIO bit in the GPER register to enable the GPIO.
  st.w   r9[AVR32_GPIO_GPERS], r10
  // Toggle the GPIO bit in the OVR register.
  st.w   r9[AVR32_GPIO_OVRT], r10
  // Set the GPIO bit in ODER register to enable the GPIO output driver for that pin.
  st.w   r9[AVR32_GPIO_ODERS], r10
  // Go back to the application outside the FlashVault.
  mov    r12, PASS
  rjmp   sec_exit


  //**
  //** Private API (not accessible from outside the secure world):
  //** sec_set_led2_on, sec_set_led3_on, sec_tgl_led2
  //**
sec_set_led2_on:
  // Set the GPIO bit in the GPER register to enable the GPIO.
  mov    r10, 1 << (LED2_GPIO & 0x1F)
  mov    r9, AVR32_GPIO_ADDRESS + 0x200*(LED2_GPIO >> 5)
  
sec_set_led_on_retal:
  // Set the GPIO bit in the GPER register to enable the GPIO.
  st.w   r9[AVR32_GPIO_GPERS], r10
  // Clear the GPIO bit in the OVR register to drive a low level when in output mode.
  st.w   r9[AVR32_GPIO_OVRC], r10
  // Set the GPIO bit in ODER register to enable the GPIO output driver for that pin.
  st.w   r9[AVR32_GPIO_ODERS], r10
  retal  r12

sec_set_led3_on:
  // Set the GPIO bit in the GPER register to enable the GPIO.
  mov    r10, 1 << (LED3_GPIO & 0x1F)
  mov    r9, AVR32_GPIO_ADDRESS + 0x200*(LED3_GPIO >> 5)
  rjmp   sec_set_led_on_retal
  
sec_tgl_led2:
  // Set the GPIO bit in the GPER register to enable the GPIO.
  mov    r10, 1 << (LED2_GPIO & 0x1F)
  mov    r9, AVR32_GPIO_ADDRESS + 0x200*(LED2_GPIO >> 5)
  // Set the GPIO bit in the GPER register to enable the GPIO.
  st.w   r9[AVR32_GPIO_GPERS], r10
  // Toggle the GPIO bit in the OVR register.
  st.w   r9[AVR32_GPIO_OVRT], r10
  // Set the GPIO bit in ODER register to enable the GPIO output driver for that pin.
  st.w   r9[AVR32_GPIO_ODERS], r10
  retal  r12


  .section  .sec_data, "aw", @progbits

  .global ss_rar_save
  .type ss_rar_save, @object
  .org 0x0
ss_rar_save: .word

  .global ss_rsr_save
  .type ss_rsr_save, @object
  // . = . + 4 // equivalent to .org 0x4
  .org 0x4
ss_rsr_save: .word

//! \endverbatim
//! @}
