/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief Example of usage of the Software Framework MPU software driver.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 *****************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
/*! \mainpage
 * \section intro Introduction
 * This is the documentation for the data structures, functions, variables,
 * defines, enums, and typedefs for the MPU driver.\n It also gives an example of
 * the usage of the MPU module.
 *
 * The MPU allows the user to divide the memory space into different protection
 * regions. These regions have a user-defined size and start at a user-defined
 * address. The different regions can have different access privileges attributes.
 *
 * You can also read the Memory Protection Unit section in the AVR32 UC technical
 * reference guide.
 *
 * <b>Example operating mode: </b>
 * - The example starts by displaying the number of MPU entries.
 * - The memory areas are then set up in the following manner:
 * <table>
 * <CAPTION><em>Access permissions of used memory areas</em></CAPTION>
 * <tr>
 *  <td>  </td>
 *  <td> Access Permissions </td>
 * </tr>
 * <tr>
 *  <td> RAM   [0x00000000 - 0x00000F00[, Region 1, subregions 0 to 14 </td>
 *  <td> Read / Write </td>
 * </tr>
 * <tr>
 *  <td> RAM   [0x00000F00 - 0x00001000[, Region 1, subregion 15 </td>
 *  <td> Read </td>
 * </tr>
 * <tr>
 *  <td> Stack [&stack - &stack + stack size[, Region 2, all subregions </td>
 *  <td> Read / Write </td>
 * </tr>
 * <tr>
 *  <td> Flash [0x80000000 - 0x8000C000[, Region 0, subregions 0 to 11</td>
 *  <td> Read / eXecute </td>
 * </tr>
 * <tr>
 *  <td> Flash [0x8000C000 - 0x80010000[, Region 0, subregions 12 to 15 </td>
 *  <td> Read </td>
 * </tr>
 * <tr>
 *  <td> PBA memory map [0xFFFF0000 - 0xFFFFFFFF], Region 3, all subregions </td>
 *  <td> Read / Write </td>
 * </tr>
 * </table>
 * - The application then performs a write access to Region 1/subregion 15. That
 * triggers a DTLB write protection exception => LED3 lights up.
 * - The application then calls a function in Region 0/Subregion 12. That triggers
 * an ITLB protection exception => LED2 lights up.
 * - End of example: LED0 & LED1 alternatively blink forever.
 *
 * \section files Main Files
 * - mpu.c: MPU driver;
 * - mpu.h: MPU driver header file;
 * - mpu_example.c: MPU example .
 * - mpu_exception.S: asm file for exception support (GCC)
 * - mpu_exception.s82: asm file for exception support and declaration of function specially located (IAR)
 *
 * \section compinfo Compilation Info
 * This software was written for the GNU GCC for AVR32 and IAR Systems compiler
 * for AVR32. Other compilers may or may not work.
 *
 * \section Configuration Information
 * This example has been tested with the following configuration:
 * - EVK1100, EVK1101, EVK1104, EVK1105 evalutation kit, STK600+RCUC3L0 routing card;
 * - CPU clock: 12 MHz; for the STK600+RCUC3L0 setup, plug a 12MHz crystal in the
 *   crystal socket (near the PROGRAM button) and switch the CLOCK selector to XTAL.
 * - USART1 (on EVK1100 or EVK1101) connected to a PC serial port via a standard
 *   RS232 DB9 cable, or USART0 (on EVK1105) or USART1 (on EVK1104) abstracted
 *   with a USB CDC connection to a PC; STK600 usart port for the STK600+RCUC3L
 *   setup (connect STK600.PE2 to STK600.RS232 SPARE.TXD and STK600.PE3 to
 *   STK600.RS232 SPARE.RXD)
 * - STK600 LEDs for the STK600+RCUC3L setup, apply the following connections:
 *   - STK600.PORTA.PA4 -> STK600.LEDS.LED0
 *   - STK600.PORTA.PA5 -> STK600.LEDS.LED1
 *   - STK600.PORTA.PA6 -> STK600.LEDS.LED2
 *   - STK600.PORTA.PA7 -> STK600.LEDS.LED3
 *   - STK600.PORTA.PB0 -> STK600.LEDS.LED4
 *   - STK600.PORTA.PB1 -> STK600.LEDS.LED5
 *   - STK600.PORTA.PB2 -> STK600.LEDS.LED6
 *   - STK600.PORTA.PB3 -> STK600.LEDS.LED7
 * - PC terminal settings:
 *   - 57600 bps,
 *   - 8 data bits,
 *   - no parity bit,
 *   - 1 stop bit,
 *   - no flow control.
 *
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.atmel.com/products/AVR32/">Atmel AVR32</A>.\n
 * Support and FAQ: http://support.atmel.no/

*/

#include <avr32/io.h>
#include "compiler.h"
#include "print_funcs.h"
#include "gpio.h"
#include "mpu.h"
#include "power_clocks_lib.h"
#include "cycle_counter.h"


// Number of regions considered in this example.
#define NB_CONFIG_MPU_ENTRIES       4

#define REGION_0_FLASH_INDEX        0
#define REGION_1_RAM_INDEX          1
#define REGION_2_STACK_INDEX        2
#define REGION_3_PERIPHERALS_INDEX  3

// Allocation for NB_CONFIG_MPU_ENTRIES DMPU entries
mpu_entry_t dmpu_entries[NB_CONFIG_MPU_ENTRIES];

extern void ForbiddenFunc();


#if defined (__GNUC__)
/*!
 * ForbiddenFunc is placed in the section .ForbiddenText. This region is configured
 * as non-executable in test_memory_area().
 * .ForbiddenText is a section created for this example, placed in a well-known
 * location in Flash. cf linker script bundled with the example.
 */
__attribute__ ((__naked__, __section__(".ForbiddenText")))
#elif __ICCAVR32__
#pragma shadow_registers = full
#pragma location="FORBIDDENTEXT"
#endif
void ForbiddenFunc(void) /*@"FORBIDDENTEXT"*/
{
#if defined (__GNUC__)
  __asm__ __volatile__ (
    ".balign 0x4\n\t"

    "nop\n\t"         /* We'll get an MPU exception here. */
    "mov     pc,lr"   /* This is setting the PC to the instruction following the call of ForbiddenFunc. */
    ::
  );
#elif defined (__ICCAVR32__)
  __asm__ __volatile__ (
    "nop\n\t"         /* We'll get an MPU exception here. */
    "mov     pc,lr"   /* This is setting the PC to the instruction following the call of ForbiddenFunc. */
  );
#endif
}


/*! \brief Handle exception; called by the _evba assembly function.
 *
 * AVR32_EVBA_OFFSET_ITLB_PROT  AVR32_EVBA_OFFSET_DTLB_PROT_x
 *
 * \param *sp               pointer to stack before the exception (provided through R12). At this point, sp[0]=SR, sp[1]=PC.
 * \param exception_address address of the offending instruction
 * \param exception_cause   exception cause (== (EVBA offset of the exception)>>2)
 */
void handle_exception(unsigned int *sp, unsigned long exception_address, unsigned int exception_cause)
{
  /* Test exception cause register AVR32_ECR. AVR32_ECR is updated with a value
   * equal to the 9 lowest bits of the EVBA offset of the exception shifted 2
   * bits to the right. */

  switch(exception_cause)
  {
    case AVR32_EVBA_OFFSET_TLB_MULTIPLE/4:
      // The TLB Multiple Hit Exception is generated when an access hits in
      // multiple MPU regions. This is usually caused by programming error.
      // Turn LED0 on.
      ((avr32_gpio_port_t *)&AVR32_GPIO.port[LED0_GPIO >> 5])->ovrc  = 1 << (LED0_GPIO & 0x1F);
      disable_mpu();  // Disable the MPU to avoid to trigger the exception again!
      break;

    case AVR32_EVBA_OFFSET_ITLB_MISS/4:
      // The ITLB Miss exception is generated when the MPU is enabled and the
      // instruction memory access does not hit in any regions.
      // Turn LED1 on.
      ((avr32_gpio_port_t *)&AVR32_GPIO.port[LED1_GPIO >> 5])->ovrc  = 1 << (LED1_GPIO & 0x1F);
      disable_mpu();  // Disable the MPU to avoid to trigger the exception again!
      break;

    case AVR32_EVBA_OFFSET_ITLB_PROT/4:
      // The ITLB Protection exception is generated when the instruction memory
      // access violates the access rights specified by the protection region in
      // which the address lies.
      // Turn LED2 on.
      ((avr32_gpio_port_t *)&AVR32_GPIO.port[LED2_GPIO >> 5])->ovrc  = 1 << (LED2_GPIO & 0x1F);
      disable_mpu();  // Disable the MPU to avoid to trigger the exception again!
      break;

    case AVR32_EVBA_OFFSET_DTLB_MISS_R/4:
      // The DTLB Read Miss exception is generated when the MPU is enabled and
      // the data memory read access does not hit in any regions.
#if BOARD == EVK1100 || BOARD == STK600_RCUC3L0
      // Turn LED6 on.
      ((avr32_gpio_port_t *)&AVR32_GPIO.port[LED6_GPIO >> 5])->ovrc  = 1 << (LED6_GPIO & 0x1F);
#endif
      disable_mpu();  // Disable the MPU to avoid to trigger the exception again!
      break;

    case AVR32_EVBA_OFFSET_DTLB_MISS_W/4:
      // The DTLB Write Miss exception is generated when the MPU is enabled and
      // the data memory write access does not hit in any regions.
#if BOARD == EVK1100 || BOARD == STK600_RCUC3L0
      // Turn LED4 on.
      ((avr32_gpio_port_t *)&AVR32_GPIO.port[LED4_GPIO >> 5])->ovrc  = 1 << (LED4_GPIO & 0x1F);
#endif
      disable_mpu();  // Disable the MPU to avoid to trigger the exception again!
      break;

    case AVR32_EVBA_OFFSET_DTLB_PROT_R/4:
      // The DTLB Protection exception is generated when the data memory read
      // violates the access rights specified by the protection region in which
      // the address lies.
#if BOARD == EVK1100 || BOARD == STK600_RCUC3L0
      // Turn LED5 on.
      ((avr32_gpio_port_t *)&AVR32_GPIO.port[LED5_GPIO >> 5])->ovrc  = 1 << (LED5_GPIO & 0x1F);
#endif
      disable_mpu();  // Disable the MPU to avoid to trigger the exception again!
      break;

    case AVR32_EVBA_OFFSET_DTLB_PROT_W/4:
      // The DTLB Protection exception is generated when the data memory write
      // violates the access rights specified by the protection region in which
      // the address lies.
      // Turn LED3 on.
      ((avr32_gpio_port_t *)&AVR32_GPIO.port[LED3_GPIO >> 5])->ovrc  = 1 << (LED3_GPIO & 0x1F);
      // Update the return from exception PC address to the next instruction,
      // to avoid triggering the exception again.
      // Check if the instruction that generated the exception is extended
      // (4 bytes) or compact (2 bytes) to find out the address of the next
      // instruction which we should return to. All extended intructions have the
      // 3 MSb always set, while this pattern is forbidden for compact instructions.
      if ( (*((unsigned char *)exception_address) & 0xE0 ) == 0xE0 )
        // Extended instruction.
        sp[1] = exception_address + 4;
      else
        // Compact instruction
        sp[1] = exception_address + 2;
      break;

    default: // Unexpected
#if BOARD == EVK1100 || BOARD == STK600_RCUC3L0
      // Toggle LED7.
      ((avr32_gpio_port_t *)&AVR32_GPIO.port[LED7_GPIO >> 5])->ovrt  = 1 << (LED7_GPIO & 0x1F);
#endif
      break;
  }
}


static void ToggleLedsInfinitly(void)
{
   volatile unsigned long i;


   LED_On( LED0 | LED1 | LED2 | LED3 );
   while(1)
   {
      i=100000;
      while(i--);
      LED_Toggle( LED0 | LED1 | LED2 | LED3 );
   }
}


static void DisplayLedChaserInfinitly(void)
{
   volatile unsigned long i;
   U8  u8LedMap = 0x01;


   while(1)
   {
      // Turn the current LED on only and move to next LED.
      LED_Display_Field(LED0 | LED1, u8LedMap);
      i=100000;
      while(i--);
      u8LedMap = max((U8)(u8LedMap << 1) & 0x0F, 0x01);
   }
}


/*! \brief Test a memory area by assigning different permissions to
 * segments inside this area.
 *
 *  \return 0
 */
void test_memory_area(void)
{
  avr32_config1_t config1;    // configuration register
  U8  u8NbMpuEntries;         // Number of MPU entries.
#if defined (__GNUC__)
  extern void _stack;         // Start address of the stack.
  extern void __stack_size__; // Size of the stack.
#elif defined (__ICCAVR32__)
  #pragma segment = "SSTACK"
#endif
  eRegionSize RegionSize;     // Region size
  U32 u32kBSizeValue;         // A region size expressed in kB


  // Read the CONFIG1 register to check the MPU configuration.
  *(U32 *)&config1 = (U32)Get_system_register(AVR32_CONFIG1);
  u8NbMpuEntries = config1.dmmusz;
  print_dbg("\r\nNumber of MPU entries: 0x");
  print_dbg_hex(u8NbMpuEntries);  // Should be 8 on UC3.

  print_dbg("\r\nSetting up all MPU entries:");


  //###
  // Configuring Region 0: the Flash memory area occupied by this application.
  // Since this program is linked into flash, we want to define the Flash memory
  // protection scheme as being Readable and Executable (RX).
  // The flash memory starts at address 0x80000000 (this info is found in the part
  // specific header file found in the compiler installation tree; e.g. in uc3a0512.h
  // for an AT32UC3A0512 part).
  dmpu_entries[REGION_0_FLASH_INDEX].addr = AVR32_FLASH_ADDRESS;
  // We estimate the size of this application to be less than 64kB. So we define
  // region 0 size as being 64kB.
  dmpu_entries[REGION_0_FLASH_INDEX].size = MPU_REGION_SIZE_64KB;
  // Deem this protection scheme as valid.
  dmpu_entries[REGION_0_FLASH_INDEX].valid = 1;
  if( set_mpu_entry(&dmpu_entries[REGION_0_FLASH_INDEX], REGION_0_FLASH_INDEX) )
    ToggleLedsInfinitly();  // Error

  // Set Access Permission A to Read/Execute access in both privileged
  // and unprivileged mode.
  set_access_permissions(REGION_0_FLASH_INDEX, MPU_APRA_ID, MPU_PRIVRX_UNPRIVRX);
  // Set Access Permission B to Read access in privileged mode and to
  // None in unprivileged mode.
  set_access_permissions(REGION_0_FLASH_INDEX, MPU_APRB_ID, MPU_PRIVR_UNPRIVNONE);
  // Each region is divided into 16 sub-regions. So in the case of this region
  // (region 0) we are currently setting up, each sub-region is 4kB in size.
  // All subregions that hold our application should be in EXECUTE access; except
  // for the ForbiddenFunc function which we implicitly put towards the end of
  // the .text section, for test purpose.
  // We estimated that our application should fit in much less that 64kB. So we
  // configure the subregions 0 to 11 to use the permission access Read/eXecute
  // as defined in the MPUAPRA register and the remaining subregions (12 to 15)
  // to use the permission access Read. Each subregion is mapped to a bit: when
  // the bit is set to 0, the subregion permission access is taken from the MPUAPRA
  // register (previously defined as MPU_PRIVRX_UNPRIVRX for region 0); if the
  // bit is set to 1; the subregion permission access is taken from the MPUAPRB
  // register (previously set to MPU_PRIVR_UNPRIVNONE for region 0).
  // We purposedly placed the ForbiddenFunc function in one of the subregion 12
  // to 15; so that when the code jumps to this location, we should get an ITLB
  // exception.
  select_subregion(REGION_0_FLASH_INDEX, 0xF000);


  //###
  // Configuring Region 1: the internal RAM memory used by this application.
  // The RAM memory starts at address 0x00000000 (this info is found in the part
  // specific header file found in the compiler installation tree; e.g. in uc3a0512.h
  // for an AT32UC3A0512 part).
  dmpu_entries[REGION_1_RAM_INDEX].addr = AVR32_SRAM_ADDRESS;
  // We estimate the RAM footprint of this application to be less than 4kB. So
  // we define region 1 size as being 4kB in size.
  dmpu_entries[REGION_1_RAM_INDEX].size = MPU_REGION_SIZE_4KB;
  // Deem this protection scheme as valid.
  dmpu_entries[REGION_1_RAM_INDEX].valid = 1;
  if( set_mpu_entry(&dmpu_entries[REGION_1_RAM_INDEX], REGION_1_RAM_INDEX) )
    ToggleLedsInfinitly();  // Error

  // Set Access Permission A to Read/Write access in privileged mode and to None
  // in unprivileged mode.
  set_access_permissions(REGION_1_RAM_INDEX, MPU_APRA_ID, MPU_PRIVRW_UNPRIVNONE);
  // Set Access Permission B to Read access in privileged mode and to None in
  // unprivileged mode.
  set_access_permissions(REGION_1_RAM_INDEX, MPU_APRB_ID, MPU_PRIVR_UNPRIVNONE);
  // Each region is divided into 16 sub-regions. So in the case of region 1 we
  // are currently setting up, each sub-region is 256B in size.
  // All subregions that our application uses should be in Read/Write access;
  // except for the last subregion (in [0x00000F00, 0x00001000[) (for test purpose).
  // We estimated that our application should use much less than 4kB. So we
  // configure the subregions 0 to 14 to use the permission access Read/Write
  // as defined in the MPUAPRA register and the remaining subregion (15) to use
  // the permission access Read.
  // Each subregion is mapped to a bit: when the bit is set to 0, the subregion
  // permission access is taken from the MPUAPRA register (previously defined as
  // MPU_PRIVRW_UNPRIVNONE for region 1); if the  bit is set to 1; the subregion
  // permission access is taken from the MPUAPRB  register (previously set to
  // MPU_PRIVR_UNPRIVNONE for region 1).
  // For the sake of this example, the application will try to do a write access
  // at an address in the range [0x00000F00, 0x00001000[: we should then get a
  // DTLB exception.
  select_subregion(REGION_1_RAM_INDEX, 0x8000);


  //###
  // Configuring Region 2: the memory used by the stack.
  // According to the linker script and to the linker step, the stack is placed
  // in INTRAM (internal RAM memory) after the RAM area explicitly used by the
  // application.
  // The stack is used upon exceptions (some registers are saved on the stack; cf
  // doc32000 "AVR32 Architecture Manual Complete" section 7.1 Event Handling in
  // AVR32A); the stack may be used when doing function calls also. Since this
  // application intends to generate exceptions, we must make sure that the memory
  // area used by the stack is Read/Write!
#if defined (__GNUC__)
  dmpu_entries[REGION_2_STACK_INDEX].addr = (unsigned int)&_stack;
#elif defined (__ICCAVR32__)
  dmpu_entries[REGION_2_STACK_INDEX].addr = (unsigned int)__segment_begin( "SSTACK" );
#endif
  // WARNING NOTE: there are limitations concerning the region size; see doc32002
  // "AVR32UC Technical Reference Manual Complete" Table 6-1. "Protection region
  // sizes implied by the Size field". The mpu_convert_kbsize_to_eregionsize()
  // makes sure this part of the spec is respected.
#if defined (__GNUC__)
  u32kBSizeValue = (U32)&__stack_size__ >> 10;
#elif defined (__ICCAVR32__)
  u32kBSizeValue = ((U32)__segment_end( "SSTACK" ) - (U32)__segment_begin( "SSTACK" )) >> 10;
#endif

  if(KO == mpu_convert_kbsize_to_eregionsize(&RegionSize, u32kBSizeValue))
    ToggleLedsInfinitly();  // Error
  dmpu_entries[REGION_2_STACK_INDEX].size = RegionSize;
  // Deem this protection scheme as valid.
  dmpu_entries[REGION_2_STACK_INDEX].valid = 1;
  if( set_mpu_entry(&dmpu_entries[REGION_2_STACK_INDEX], REGION_2_STACK_INDEX) )
    ToggleLedsInfinitly();  // Error

  // Set Access Permission A to Read access in privileged mode and to None in
  // unprivileged mode.
  set_access_permissions(REGION_2_STACK_INDEX, MPU_APRA_ID, MPU_PRIVR_UNPRIVNONE);
  // Set Access Permission B to Read/Write access in privileged mode and to None
  // in unprivileged mode.
  set_access_permissions(REGION_2_STACK_INDEX, MPU_APRB_ID, MPU_PRIVRW_UNPRIVNONE);
  // Set all subregions of the stack to Read/Write access (i.e. use the Access Permissions B).
  select_subregion(REGION_2_STACK_INDEX,0xFFFF);


  //###
  // Configuring Region 3: the peripherals memory mapping.
  // We're using the DEBUG module to output traces to USART1, we're using the GPIO
  // module => we have to define access permissions to the region where the USART1
  // & GPIO peripherals registers are mapped. We'll set this region as Read/Write
  // because we don't want to generate MPU exceptions when using a peripheral.
  // The USART1 & GPIO peripherals are on the same Peripheral Bus (PBA). We'll
  // configure all peripherals on PBA with the same access rights.
  // The PBA physical memory map starts at address 0xFFFF0000 (this info is found
  // in the datasheet in the table "AT32UC3x Physical Memory Map".
  // The PDCA module happens to be the module that is mapped at the start of the
  // PBA memory map.
  dmpu_entries[REGION_3_PERIPHERALS_INDEX].addr = AVR32_PDCA_ADDRESS;
  // The size of the PBA memory map is 64kB.
  dmpu_entries[REGION_3_PERIPHERALS_INDEX].size = MPU_REGION_SIZE_64KB;
  // Deem this protection scheme as valid.
  dmpu_entries[REGION_3_PERIPHERALS_INDEX].valid = 1;
  if( set_mpu_entry(&dmpu_entries[REGION_3_PERIPHERALS_INDEX], REGION_3_PERIPHERALS_INDEX) )
    ToggleLedsInfinitly();  // Error

  // Set Access Permission A to Read/Write access in privileged mode and to None
  // in unprivileged mode.
  set_access_permissions(REGION_3_PERIPHERALS_INDEX, MPU_APRA_ID, MPU_PRIVRW_UNPRIVNONE);
  // We don't set the Access Permission B because we won't use it.
  // Set all 16 subregions of the PBA memory map to Read/Write access.
  select_subregion(REGION_3_PERIPHERALS_INDEX,0x0000);


  enable_mpu(); // Enable the MPU address checking.
}


/* \brief Main entry point
 * This is an example of how to setup the AVR32 UC3 MPU with protected regions.
 */

int main(void)
{
  Enable_global_exception();
  Enable_global_interrupt();

  // Clear the LEDs.
#if BOARD == EVK1100 || BOARD == EVK1101 || BOARD == EVK1104 || BOARD == EVK1105 || BOARD == STK600_RCUC3L0
  gpio_set_gpio_pin(LED0_GPIO); gpio_set_gpio_pin(LED1_GPIO);
  gpio_set_gpio_pin(LED2_GPIO); gpio_set_gpio_pin(LED3_GPIO);
#endif
#if BOARD == EVK1100 || BOARD == STK600_RCUC3L0
  gpio_set_gpio_pin(LED4_GPIO); gpio_set_gpio_pin(LED5_GPIO);
  gpio_set_gpio_pin(LED6_GPIO); gpio_set_gpio_pin(LED7_GPIO);
#endif

  // Configure Osc0 in crystal mode (i.e. use of an external crystal source, with
  // frequency FOSC0) with an appropriate startup time then switch the main clock
  // source to Osc0.
  pcl_switch_to_osc(PCL_OSC0, FOSC0, OSC0_STARTUP);

  // Init the trace-on-usart1 module (under UTILS/DEBUG/).
  init_dbg_rs232(FOSC0);
  print_dbg("\n\n## This is the AT32UC3 MPU example ##");

  // Set up and enable the MPU to protect the following memory areas:
  //                                             --------------------
  //                                            | Access permissions |
  //  ------------------------------------------|--------------------|
  // | Flash [0x80000000 - 0x8000C000[          | Read / eXecute     |
  // | Region 0, subregions 0 to 11             |                    |
  // |------------------------------------------|--------------------|
  // | Flash [0x8000C000 - 0x80010000[          | Read               |
  // | Region 0, subregions 12 to 15            |                    |
  // |------------------------------------------|--------------------|
  // | RAM   [0x00000000 - 0x00000F00[          | Read / Write       |
  // | Region 1, subregions 0 to 14             |                    |
  // |------------------------------------------|--------------------|
  // | RAM   [0x00000F00 - 0x00001000[          | Read               |
  // | Region 1, subregion 15                   |                    |
  // |------------------------------------------|--------------------|
  // | Stack [&stack - &stack + stack size[     | Read / Write       |
  // | Region 2, all subregions                 |                    |
  //  ---------------------------------------------------------------|
  // | PBA memory map [0xFFFF0000 - 0xFFFFFFFF] | Read / Write       |
  // | Region 3, all subregions                 |                    |
  //  ---------------------------------------------------------------
  test_memory_area();

  print_dbg("\nTesting the Write protection of the area [0x00000F00 - 0x00001000[");
  // The address 0x00000F00 is mpu-configured as Read only. We perform a Write
  // access there, so that should trigger a DTLB write protection exception
  // => LED3 lights up.
  *( (volatile int*) (0x00000F00))  = 0x55AA55AA;

  // Wait two seconds before doing the next test, to clearly see that LED3 lights
  // up first then LED2 does.
  cpu_delay_ms(2000, FOSC0);


  print_dbg("\nTesting the eXecute protection of the area [0x8000C000 - 0x80010000[");
  // The memory area [0x8000C000 - 0x80010000[ is mpu-configured as Read only.
  // We forced the ForbiddenFunc() function to be mapped in this area. Calling
  // this function should thus trigger an ITLB protection exception => LED2 lights up.
  ForbiddenFunc();

  // LED0 & LED1 blink alternatively forever.
  DisplayLedChaserInfinitly();

  return(0);
}
