/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief CAN Driver Example 1.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with CANIF modules.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/*! \page License
 * Copyright (c) 2006-2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*! \mainpage
 * \section intro Introduction
 * This documents data structures, functions, variables, defines, enums, and
 * typedefs in the software for the CAN driver.
 *
 * The given example is a CAN driver example using two CAN channels available.
 * A mailbox is defined by the user and all messages are sent easily.
 *
 * \section compinfo Compilation Info
 * This software was written for the GNU GCC for AVR32 and IAR Systems compiler
 * for AVR32. Other compilers may or may not work.
 *
 * \section files Main Files
 * - canif.c: CAN low level driver.
 * - canif.h: CAN low level header file.
 * - can.c: CAN High level driver.
 * - can.h: CAN High level header file.
 * - conf_can.h: Configuration file for the CAN Lib.
 * - can_example1.c: CAN Driver example.
 * - conf_can_example.h: Configuration file for the CAN example.
 *
 * \section configinfo Configuration Information
 * This example has been tested with the following configuration:
 * - UC3C_EK evaluation kit;
 * - CPU clock: 16 MHz;
 * This example demonstrates that the application can be waked-up from the CAN.
 * - Plug a DB9 cable on CAN1 DB9 connector and an external CAN node to setup a 'point to point' network.
 * - Configure the baudrate to 1Mbps.
 * - Setup the external CAN node to transmit message.
 * - Start the example.
 * - The CPU is in sleep mode (STATIC) at the beginning of the application and then is waked up after the reception
 *   of the message from the external CAN node
 * 
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.atmel.com/products/AVR32/">Atmel AVR32</A>.\n
 * Support and FAQ: http://support.atmel.no/
 */
 
#include <stddef.h>
#include <stdio.h>
#include <avr32/io.h>
#include "compiler.h"
#include "board.h"
#include "power_clocks_lib.h"
#include "gpio.h"
#include "pm_uc3c.h"
#include "scif_uc3c.h"
#include "intc.h"
#include "can.h"
#include "canif.h"
#if defined (__GNUC__) && defined (__AVR32__)
  #include "nlao_cpu.h"
  #include "nlao_usart.h"
#endif
#include "usart.h"

#include "conf_can_example.h"

//! Local allocation for MOB buffer
volatile can_msg_t mob_ram_ch0[NB_MOB_CHANNEL];  

//! Local function to prepare RX and TX buffers
void can_example_prepare_data_to_send(U8 mode);


volatile U8 nb_message_received_on_channel0 = 0;
//! Call Back called by can_drv
void can_out_callback_channel0(U8 handle, U8 event)
{
   gpio_tgl_gpio_pin(LED0_GPIO);

   // Reception Only
   appli_rx_msg.can_msg->data.u64 = can_get_mob_data(0,handle).u64;
   appli_rx_msg.can_msg->id = can_get_mob_id(0,handle);
   appli_rx_msg.dlc = can_get_mob_dlc(0,handle);
   appli_rx_msg.status = event;
   can_mob_free(0,handle);       
   nb_message_received_on_channel0 = 1;       

}

/*! \brief Initializes MCU exceptions.
 */
static void init_exceptions(void)
{
#if defined (__GNUC__) && defined (__AVR32__)
  // Import the Exception Vector Base Address.
  extern void _evba;

  // Load the Exception Vector Base Address in the corresponding system
  // register.
  Set_system_register(AVR32_EVBA, (int)&_evba);
#endif

  // Enable exceptions globally.
  Enable_global_exception();
}

/*! \brief Initializes the MCU system clocks.
 */
static void init_sys_clocks(void)
{
  // Switch the main clock to OSC0 
    scif_configure_osc_crystalmode(SCIF_OSC0, FOSC0);
    scif_enable_osc(SCIF_OSC0, OSC0_STARTUP, true);
    pm_set_mclk_source(PM_CLK_SRC_OSC0);

  // Setup the generic clock for CAN
  scif_gc_setup(AVR32_SCIF_GCLK_CANIF, 
                SCIF_GCCTRL_OSC0, 
                AVR32_SCIF_GC_NO_DIV_CLOCK, 
                0);  
  // Now enable the generic clock
  scif_gc_enable(AVR32_SCIF_GCLK_CANIF);
  
#if defined (__GNUC__) && defined (__AVR32__)
  // Give the used PBA clock frequency to Newlib (only for GCC), so it can work properly.
  set_cpu_hz(FPBA_HZ);
#endif
}

/*! \brief Initializes STDIO.
 */
static void init_stdio(void)
{

  static const gpio_map_t STDIO_USART_GPIO_MAP =
  {
    {STDIO_USART_RX_PIN, STDIO_USART_RX_FUNCTION},
    {STDIO_USART_TX_PIN, STDIO_USART_TX_FUNCTION}
  };

  static const usart_options_t STDIO_USART_OPTIONS =
  {
    .baudrate     = STDIO_USART_BAUDRATE,
    .charlength   = 8,
    .paritytype   = USART_NO_PARITY,
    .stopbits     = USART_1_STOPBIT,
    .channelmode  = USART_NORMAL_CHMODE
  };

  gpio_enable_module(STDIO_USART_GPIO_MAP,
                     sizeof(STDIO_USART_GPIO_MAP) / sizeof(STDIO_USART_GPIO_MAP[0]));
  
  usart_init_rs232(STDIO_USART, &STDIO_USART_OPTIONS, FPBA_HZ);
  
#if defined (__GNUC__) && defined (__AVR32__)
  // Initialize the USART used for STDIO.
  set_usart_base((void *)STDIO_USART);
#endif
}


/*! \brief Initializes MCU interrupts.
 */
static void init_interrupts(void)
{
  // Initialize interrupt handling.
  INTC_init_interrupts();

  // Enable interrupts globally.
  Enable_global_interrupt();
}


/*! \brief Low-level initialization routine called during startup, before the
 *         \ref main function.
 */
#if defined (__GNUC__) && defined (__AVR32__)
int _init_startup(void)
#elif defined (__ICCAVR32__)
int __low_level_init(void)
#endif
{
  init_exceptions();
  init_sys_clocks();
  init_stdio();
  init_interrupts();

  // EWAVR32: Request initialization of data segments.
  // GCC: Don't-care value.
  return 1;
}


int main(void)
{
  // Configure standard I/O streams as unbuffered.
#if defined (__GNUC__) && defined (__AVR32__)
  setbuf(stdin, NULL);
#endif
  setbuf(stdout, NULL);
  
  // Disable all interrupts.
  Disable_global_interrupt();

  // Initialize interrupt vectors.
  INTC_init_interrupts();

  static const gpio_map_t CAN_GPIO_MAP =
  {
    {AVR32_CANIF_RXLINE_0_0_PIN, AVR32_CANIF_RXLINE_0_0_FUNCTION},
    {AVR32_CANIF_TXLINE_0_0_PIN, AVR32_CANIF_TXLINE_0_0_FUNCTION}
  };
  // Assign GPIO to CAN.
  gpio_enable_module(CAN_GPIO_MAP,
                     sizeof(CAN_GPIO_MAP) / sizeof(CAN_GPIO_MAP[0]));
                       
  // Initialize channel 0
  can_init(0,
           ((U32)&mob_ram_ch0[0]),
           CANIF_CHANNEL_MODE_LISTENING,
           can_out_callback_channel0);  

  // Enable all interrupts.
  Enable_global_interrupt();

  printf("\nUC3C CAN Examples 1\n");

  printf(CAN_Wakeup);
  
  // Initialize CAN Channel 0
  can_init(0,
           ((U32)&mob_ram_ch0[0]),
           CANIF_CHANNEL_MODE_NORMAL,
           can_out_callback_channel0);
           
  // Allocate one mob for RX 
  appli_rx_msg.handle = can_mob_alloc(0);
  
  // Initialize RX message
  can_rx(0,
         appli_rx_msg.handle,
         appli_rx_msg.req_type,
         appli_rx_msg.can_msg);

  //---------SLEEP MODE PROCEDURE-------------
  // Disable CAN Channel 0
  CANIF_clr_reset(0);
  // Wait CAN Channel 0 is disabled
  while(!CANIF_channel_enable_status(0));
  // Enable Wake-Up Mode
  CANIF_enable_wakeup(0);
  // Go to sleep mode.
  SLEEP(AVR32_PM_SMODE_STATIC);
  // Clear Interrupt Flag
  CANIF_clr_interrupt_status(0);
  // Disable Wake-Up Mode
  CANIF_disable_wakeup(0);
  //---------SLEEP MODE PROCEDURE-------------
  printf(CAN_WakeupD);    
  // Initialize again CAN Channel 0
  can_init(0,
           ((U32)&mob_ram_ch0[0]),
           CANIF_CHANNEL_MODE_NORMAL,
           can_out_callback_channel0);
           
  // Allocate one mob for RX 
  appli_rx_msg.handle = can_mob_alloc(0);

  // Initialize RX message  
  can_rx(0,
         appli_rx_msg.handle,
         appli_rx_msg.req_type,
         appli_rx_msg.can_msg);
        
  while(1);

}
