/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief ADCIFA driver for AVR UC3.
 *
 * This file defines a useful set of functions for ADC on AVR UC3 devices.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR UC3
 * - Supported devices:  All AVR UC3 devices with an ADCIFA can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 *****************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */

#include <avr32/io.h>
#include "compiler.h"
#include "adcifa.h"

void adcifa_get_calibration_data( volatile avr32_adcifa_t * adcifa,
                                  adcifa_opt_t * p_adcifa_opt,
                                  U8 sequencer)
{
  unsigned int* calibration_data_adc = (unsigned int*)AVR32_FLASHC_CALIBRATION_FIRST_WORD_ADDRESS;
  unsigned int* calibration_data_seq = (unsigned int*)AVR32_FLASHC_CALIBRATION_SECOND_WORD_ADDRESS;
  
  // Get ADC Offset Calibration
  p_adcifa_opt->offset_calibration_value = (*calibration_data_adc >> 24);
  
  // Get ADC Gain Calibration
  p_adcifa_opt->gain_calibration_value = ((*calibration_data_adc >> 8) & AVR32_ADCIFA_ADCCAL_GCAL_MASK) ;
   
  if (sequencer == 0)
  {   
    // Get Sample & Hold Gain Calibration
    p_adcifa_opt->sh_calibration_value = *calibration_data_seq;
  }
  else
  {
    // Get Sample & Hold Gain Calibration
    p_adcifa_opt->sh_calibration_value = (*calibration_data_seq >> 16);
  }
}                                
U8 adcifa_configure(volatile avr32_adcifa_t * adcifa, 
                    adcifa_opt_t * p_adcifa_opt,
                    unsigned long pb_hz)
{

 
  // Sanity Check
  Assert( adcifa!=NULL );

  // Set the ADC configuration
  AVR32_ADCIFA.cfg  = (p_adcifa_opt->sleep_mode_enable << AVR32_ADCIFA_CFG_SLEEP)
                    | (p_adcifa_opt->single_sequencer_mode << AVR32_ADCIFA_CFG_SSMQ)
                    | (p_adcifa_opt->free_running_mode_enable << AVR32_ADCIFA_CFG_SSMQ)
                    | (p_adcifa_opt->reference_source << AVR32_ADCIFA_CFG_RS)
                    | (p_adcifa_opt->sample_and_hold_disable << AVR32_ADCIFA_CFG_SHD)
                    | (((ADCIFA_START_UP_TIME * (p_adcifa_opt->frequency/1000))/32000) << AVR32_ADCIFA_CFG_SUT);

  // Configure Clock  (rounded up)
  adcifa->ckdiv = ((pb_hz / (2*p_adcifa_opt->frequency)) << AVR32_ADCIFA_CKDIV_CNT_OFFSET) & AVR32_ADCIFA_CKDIV_CNT_MASK; 

  // Set ADC Offset Calibration
  ADCIFA_set_offset_calibration(p_adcifa_opt->gain_calibration_value);

  // Set ADC Gain Calibration
  ADCIFA_set_gain_calibration(p_adcifa_opt->offset_calibration_value);

  // Set Sample & Hold Gain Calibration
  ADCIFA_set_sh_gain_calibration(p_adcifa_opt->sh_calibration_value);
  
  // Enable ADCIFA
  ADCIFA_enable();

  // Wait Startup Time
  while(1) {
   if (ADCIFA_is_startup_time()) break;
  }

  return ADCIFA_CONFIGURATION_ACCEPTED;
}

U8 adcifa_configure_sequencer(volatile avr32_adcifa_t * adcifa,
                              U8 sequencer,
                              adcifa_sequencer_opt_t * p_adcifa_sequencer_opt,
                              adcifa_sequencer_conversion_opt_t * p_adcifa_sequencer_conversion_opt)
{
  U8 g[8]={0};
  U8 mp[8]={0};
  U8 mn[8]={0};
  U8 i;
  // Sanity Check
  Assert( adcifa!=NULL );

  // Switch case with sequencer
  switch (sequencer)
  {
    // Sequencer 0
    case 0:
      // Configure Sequencer 0
      ADCIFA_configure_sequencer_0((p_adcifa_sequencer_opt->convnb-1), 
                                   p_adcifa_sequencer_opt->resolution, 
                                   p_adcifa_sequencer_opt->trigger_selection, 
                                   p_adcifa_sequencer_opt->start_of_conversion, 
                                   p_adcifa_sequencer_opt->oversampling, 
                                   p_adcifa_sequencer_opt->half_word_adjustment, 
                                   p_adcifa_sequencer_opt->software_acknowledge);

      // Configure Gain for Sequencer 0
      for (i = 0; i < p_adcifa_sequencer_opt->convnb; i++)
        g[i] = p_adcifa_sequencer_conversion_opt[i].gain;
      ADCIFA_configure_sh0gain(g[7], g[6], g[5], g[4], g[3], g[2], g[1], g[0]);
      
      // Configure Mux for Sequencer 0
      for (i = 0; i < p_adcifa_sequencer_opt->convnb; i++)
      {
        mp[i] = p_adcifa_sequencer_conversion_opt[i].channel_p;
        mn[i] = p_adcifa_sequencer_conversion_opt[i].channel_n;
      }
      ADCIFA_configure_muxsel0p(mp[7], mp[6], mp[5], mp[4], mp[3], mp[2], mp[1], mp[0]);
      ADCIFA_configure_muxsel0n(mn[7], mn[6], mn[5], mn[4], mn[3], mn[2], mn[1], mn[0]);
      break;
    // Sequencer 1
    case 1:
      // Configure Sequencer 1
      ADCIFA_configure_sequencer_1(p_adcifa_sequencer_opt->convnb-1, 
                                   p_adcifa_sequencer_opt->resolution, 
                                   p_adcifa_sequencer_opt->trigger_selection, 
                                   p_adcifa_sequencer_opt->start_of_conversion, 
                                   p_adcifa_sequencer_opt->oversampling, 
                                   p_adcifa_sequencer_opt->half_word_adjustment, 
                                   p_adcifa_sequencer_opt->software_acknowledge);

      // Configure Gain for Sequencer 1
      for (i = 0; i < p_adcifa_sequencer_opt->convnb; i++)
        g[i] = p_adcifa_sequencer_conversion_opt[i].gain;
      ADCIFA_configure_sh1gain(g[7], g[6], g[5], g[4], g[3], g[2], g[1], g[0]);
      
      // Configure Mux for Sequencer 1
      for (i = 0 ; i < p_adcifa_sequencer_opt->convnb; i++)
      {
        mp[i] = p_adcifa_sequencer_conversion_opt[i].channel_p;
        mn[i] = p_adcifa_sequencer_conversion_opt[i].channel_n;
      }
      ADCIFA_configure_muxsel1p(mp[7], mp[6], mp[5], mp[4], mp[3], mp[2], mp[1], mp[0]);
      ADCIFA_configure_muxsel1n(mn[7], mn[6], mn[5], mn[4], mn[3], mn[2], mn[1], mn[0]);
      break;
    default:
      break;
  }
  return ADCIFA_CONFIGURATION_ACCEPTED;
}

void adcifa_start_sequencer(volatile avr32_adcifa_t * adcifa, U8 sequencer)
{
  // Sanity Check
  Assert( adcifa!=NULL );

  // Switch Sequencer
  switch (sequencer)
  {  
    case ADCIFA_SEQ0:
      ADCIFA_softsoc_sequencer(AVR32_ADCIFA_CR_SOC0_MASK);
      break;
    case ADCIFA_SEQ1:
      ADCIFA_softsoc_sequencer(AVR32_ADCIFA_CR_SOC1_MASK);
      break;
    case ADCIFA_SEQ0_SEQ1:
      ADCIFA_softsoc_sequencer(AVR32_ADCIFA_CR_SOC0_MASK | AVR32_ADCIFA_CR_SOC1_MASK);
      break;
    default:
      break;  
  }
}

Bool adcifa_check_eoc(volatile avr32_adcifa_t * adcifa, U8 sequencer)
{
  // Sanity Check
  Assert( adcifa!=NULL );

  // get SR register : EOC bit for channel
  switch(sequencer)
  {
    case 0:
      return ((ADCIFA_is_eoc_sequencer_0()) ? HIGH : LOW);
    case 1:
      return ((ADCIFA_is_eoc_sequencer_1()) ? HIGH : LOW);
    default:
      break;
  }
  return LOW;
}

Bool adcifa_check_eos(volatile avr32_adcifa_t * adcifa, U8 sequencer)
{
  // Sanity Check
  Assert( adcifa!=NULL );

  // get SR register : EOS bit for channel
  switch(sequencer)
  {
    case 0:
      return ((ADCIFA_is_eos_sequencer_0()) ? HIGH : LOW);
    case 1:
      return ((ADCIFA_is_eos_sequencer_1()) ? HIGH : LOW);
    default:
      break;
  }
  return LOW;
}


U8 adcifa_get_values_from_sequencer(volatile avr32_adcifa_t * adcifa,
                                    U8 sequencer,
                                    adcifa_sequencer_opt_t * p_adcifa_sequencer_opt,
                                    S16* adcifa_values)
{
  U8 i;
  // Sanity Check
  Assert( adcifa!=NULL );

  // wait for end of sequence
  if(adcifa_check_eos(adcifa, sequencer) != HIGH)
    return ADCIFA_STATUS_NOT_COMPLETED;
  
  switch(sequencer)
  {
    case 0:
      // Read values from Sequencer 0
      for (i = 0; i < p_adcifa_sequencer_opt->convnb; i++)
      {
          adcifa_values[i] = ADCIFA_read_resx_sequencer_0(i);
      }
      break;
    case 1:
      // Read values from Sequencer 1
      for (i = 0; i < p_adcifa_sequencer_opt->convnb; i++)
      {
          adcifa_values[i] = ADCIFA_read_resx_sequencer_1(i);
      }
      break;
    default:
      break;
  }

  return ADCIFA_STATUS_COMPLETED;
}

