/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*! \page License
 * Copyright (C) 2009, H&D Wireless AB All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of H&D Wireless AB may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY H&D WIRELESS AB ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*!
 * \file *********************************************************************
 *
 * \brief HTTP Server WiFi Demo Application
 *
 * This file provides a an example application for the H&D Wireless AB
 * WiFi driver.
 *
 * - Compiler:           GNU GCC for AVR32
 * - Supported devices:  
 *   \li SPB104 + EVK1104 (SDIO)
 *   \li SPB104 + EVK1105 (SPI)
 *   \li SPB104 + EVK1104 (SPI + irq)
 *   \li SPB105 + EVK1105 (SPI)
 *   \li SPB104 + EVK1100
 *   \li SPB104 + EVK1101
 * - AppNote:
 *
 * \author               H&D Wireless AB \n
 *
 *****************************************************************************/

/*! \mainpage http_server_demo HTTP Server WiFi Demo Application
 * \section http_server_intro Introduction
 * This is the documentation for the WiFi HTTP Server demo application.
 * This application demonstrates the use of the (o)WL API &copy;
 *
 * Information on the SPB104 and SPB105 devices can be found here:
 * http://www.hd-wireless.se
 *
 * \section files Main Files
 * - avr32_sdio.c : SDIO HAL
 * - avr32_spi.c : SPI HAL
 * - clocks.c : Board clock setup
 * - cmd_wl.c : Console command handlers
 * - conf_*.h : Board component configurations
 * - console.c : Serial-USB console
 * - fs.c : Filesystem stub for HTTP server file access
 * - fsdata.c : Filesystem data image
 * - fw_download.c : WiFi firmware download implementation for boards with 
 *                   the firmware image embedded in the application.
 * - fw_download_extflash.c : WiFi firmware download implementation for boards
 *                            with the firmware image in external flash.
 * - gui.c : Generic GUI implementation
 * - gui_getstring.c : GUI input for encryption keys
 * - httpd.c : HTTP Server
 * - http_server_gui.c : HTTP Server-specific GUI handling
 * - lwipopts.h : lwIP configuration tailored for this application
 * - main.c : Application entry point
 * - nor_flash.c : NOR flash access functions (used for WiFi firmware access
 *                 when the firmware is stored in external flash)
 * - ping.c : ICMP echo implementation
 * - printf-stdarg.c : String formatting functions
 * - startup.c : Board configuration
 * - timer.c : Timer implementation
 * - ttcp.c : TTCP traffic source/sink implementation for throughput measurements
 * - util.c : Utility functions
 * - wl_cm.c : Simple WiFi Connection Manager implementation
 * - wl_util.c : WiFi-specific utility functions
 *
 * \section compilinfo Compilation Information
 * This software is written for GNU GCC for AVR32. Other compilers may or may not work.
 *
 * \section deviceinfo Device Information
 * - Supported devices:  
 *   \li SPB104 + EVK1104 (SDIO)
 *   \li SPB104 + EVK1105 (SPI)
 *   \li SPB104 + EVK1104 (SPI + irq)
 *   \li SPB105 + EVK1105 (SPI)
 *   \li SPB104 + EVK1100
 *   \li SPB104 + EVK1101
 *
 * \section configinfo Configuration Information
 * This example has been tested with the following configuration:
 * - EVK1100 evaluation kits with SPB104 card.
 * - EVK1101 evaluation kits with SPB104 card.
 * - EVK1104 evaluation kits with SPB104 card.
 * - EVK1105 evaluation kits with SPB104 card.
 *
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.hd-wireless.se/">H&D Wireless</A>.\n
 * Support and FAQ: http://www.atmel.com/
 */

#include <compiler.h>
#include "board.h"
#include "gpio.h"
#include "adc.h"

#include "wl_api.h"
#include "wl_cm.h"

#include "lwip/init.h"
#include "lwip/dhcp.h"
#include "lwip/tcp.h"
#include "netif/etharp.h"
#include "netif/wlif.h"

#include "startup.h"
#include "trace.h"
#include "fw_download.h"
#include "timer.h"
#include "wl_util.h"
#include "util.h"
#include "cmd_wl.h"
#include "httpd.h"
#include "ping.h"
#include "ttcp.h"
#include "gui.h"

#include "http_server_gui.h"

struct http_server {
	struct netif *netif;
	uint8_t wl_init_complete;
};

/**
 *
 */
static void
tcp_tmr_cb(void *ctx)
{
	tcp_tmr();
}


/**
 *
 */
static void
etharp_tmr_cb(void *ctx)
{
	etharp_tmr();
}


/**
 *
 */
static void
dhcp_fine_tmr_cb(void *ctx)
{
	dhcp_fine_tmr();
}


/**
 *
 */
static void
dhcp_coarse_tmr_cb(void *ctx)
{
	dhcp_coarse_tmr();
}


/**
 *
 */
static void
wl_cm_conn_cb(struct wl_network_t* net, void* ctx)
{
	struct http_server* hs = ctx;

        gui_link_up_cb();
        printk("link up, connected to \"%s\"\n", net->ssid.ssid);
        printk("requesting dhcp ... ");

        dhcp_start(hs->netif);
}


/**
 *
 */
static void
wl_cm_disconn_cb(void* ctx)
{
	struct http_server* hs = ctx;

        gui_link_down_cb();

        if (netif_is_up(hs->netif)) {
                printk("link down, release dhcp\n");
                dhcp_release(hs->netif);
                dhcp_stop(hs->netif);
        } else {
                printk("link down\n");
        }
}


/**
 *
 */
static void
ip_status_cb(struct netif* netif)
{
        if (netif_is_up(netif)) {
                gui_status_up_cb();
                printk("bound to %s\n", ip2str(netif->ip_addr));
                printk("starting httpd ... ");
                if (httpd_start() == ERR_OK)
                        printk("ok\n");
                else
                        printk("fail\n");

        }
        else {
                gui_status_down_cb();
                printk("stopping httpd\n");
                httpd_stop();
        }
}


/**
 *
 */
void 
adc_init(void)
{
        static const gpio_map_t ADC_GPIO_MAP = {
                { AVR32_ADC_AD_0_PIN, AVR32_ADC_AD_0_FUNCTION },
        };
        volatile avr32_adc_t *adc = &AVR32_ADC;

        gpio_enable_module(ADC_GPIO_MAP,
                           sizeof(ADC_GPIO_MAP) / sizeof(ADC_GPIO_MAP[0]));
        AVR32_ADC.mr |= 0x1 << AVR32_ADC_MR_PRESCAL_OFFSET;
        adc_configure(adc);
        adc_enable(adc, 0);
}


/**
 *
 */
void
led_init(void)
{
	gpio_enable_gpio_pin(LED0_GPIO);
	gpio_enable_gpio_pin(LED1_GPIO);
	gpio_enable_gpio_pin(LED2_GPIO);
	gpio_enable_gpio_pin(LED3_GPIO);
        LED_Toggle(LED0);
        LED_Toggle(LED1);
        LED_Toggle(LED2);
        LED_Toggle(LED3);
}


/**
 *
 */
void 
poll(struct http_server* hs)
{
        /* this will trigger any scheduled timer callbacks */
        timer_poll();

        /* handle console input */
        console_poll();

        /* wl api 'tick' */
        wl_poll(timer_get_ms());

        /* lwip driver poll */
        wlif_poll(hs->netif);

#ifdef WITH_GUI
        gui_exec(timer_get_ms());
#endif
}


/**
 *
 */
void 
wl_init_complete_cb(void* ctx) 
{
	struct http_server *hs = ctx;
        struct ip_addr ipaddr, netmask, gw;
	wl_err_t wl_status;
	
        IP4_ADDR(&gw, 0,0,0,0);
        IP4_ADDR(&ipaddr, 0,0,0,0);
        IP4_ADDR(&netmask, 0,0,0,0);
        
	/* add wl to lwip interface list and set as default */
        hs->netif = netif_add(hs->netif, &ipaddr, &netmask, &gw, NULL,
			      wlif_init, /* init */
			      ethernet_input /* handles ARP and IP packets */);
	ASSERT(hs->netif, "failed to add netif");
        netif_set_default(hs->netif);
        netif_set_status_callback(hs->netif, ip_status_cb);

	/* register lwip timer callbacks for tcp, arp and dhcp protocols */
        timer_sched_timeout_cb(5000, TIMEOUT_PERIODIC, 
			       etharp_tmr_cb, hs);
        timer_sched_timeout_cb(TCP_TMR_INTERVAL, TIMEOUT_PERIODIC, 
			       tcp_tmr_cb, hs);
	timer_sched_timeout_cb(DHCP_FINE_TIMER_MSECS, TIMEOUT_PERIODIC, 
			       dhcp_fine_tmr_cb, hs);
        timer_sched_timeout_cb(DHCP_COARSE_TIMER_MSECS, TIMEOUT_PERIODIC,
			       dhcp_coarse_tmr_cb, hs);

	/* initialize shell */
        console_init();
        console_add_cmd("scan", cmd_scan, NULL);
        console_add_cmd("connect", cmd_connect, NULL);
        console_add_cmd("setkey", cmd_setkey, NULL);
        console_add_cmd("status", cmd_status, NULL);
        console_add_cmd("powersave", cmd_power, NULL);
        console_add_cmd("psconf", cmd_psconf, NULL);
        console_add_cmd("ping", cmd_ping, NULL);
        console_add_cmd("ttcp", cmd_ttcp, NULL);
#ifdef WITH_WPA
        console_add_cmd("wpass", cmd_setpass, NULL);
        console_add_cmd("dpass", cmd_delpass, NULL);
#endif
        
	/* start connection manager */
	wl_status = wl_cm_start(NULL, wl_cm_conn_cb, wl_cm_disconn_cb, hs);
	ASSERT(wl_status == WL_SUCCESS, "failed to init wl conn mgr");

        /* init devices used by httpd */
        adc_init();
        led_init();

        gui_start();
}

/**
 *
 */
int
main(void)
{
	wl_err_t wl_status;
	int status;
	struct http_server *hs;

        startup_init();

	hs = calloc(1, sizeof(struct http_server));
	ASSERT(hs, "out of memory");

	hs->netif = calloc(1, sizeof(struct netif));
	ASSERT(hs->netif, "out of memory");

        timer_init(NULL, NULL);
        lwip_init();
        
	status = fw_download_init();
	ASSERT(status == 0, "failed to prepare for firmware download\n");

        wl_status = wl_init(hs, fw_download_cb, wl_init_complete_cb);
        switch (wl_status) {
        case WL_SUCCESS:
                /* ok */
                break;

        case WL_CARD_FAILURE:
                printk("Could not detect wl device, aborting\n");
                return -1;

        case WL_FIRMWARE_LENGTH:
                printk("Invalid firmware length, aborting\n");
                return -1;

        case WL_FIRMWARE_INVALID:
                printk("Invalid firmware data, aborting\n");
                return -1;

        default:
                printk("Failed to start wl initialization\n");
                return -1;
        }

	/* start main loop */
        for (;;)
                poll(hs);
}
