/* This header file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

// This file has been prepared for Doxygen automatic documentation generation.

#ifndef _RTOUCH_H_
#define _RTOUCH_H_



/*============================ INCLUDES ======================================*/
#include <stdint.h>
#include <stdbool.h>



/*============================ TYPES =========================================*/

//! Info for one calibration point, raw samples and actual panel position.
typedef struct rtouch_calibration_point_struct
{
    uint16_t panelX; //!< Actual panel X coordinate.
    uint16_t panelY; //!< Actual panel Y coordinate.
    uint16_t rawX; //!< Raw X sample value.
    uint16_t rawY; //!< Raw Y sample value.
} rtouch_calibration_point_t;

//!  Collection of the three calibration points required to compute calibration.
typedef struct rtouch_calibration_points_struct
{
    rtouch_calibration_point_t point1;
    rtouch_calibration_point_t point2;
    rtouch_calibration_point_t point3;
} rtouch_calibration_points_t;

//! Calibration matrix coefficients, computed from three calibration points.
typedef struct rtouch_calibration_matrix_struct
{
    int32_t A;
	int32_t B;
	int32_t C;
    int32_t D;
	int32_t E;
	int32_t F;
    int32_t K;
} rtouch_calibration_matrix_t;

//! Event type used by event struct TOUCH_event_t.
typedef enum rtouch_event_type_enum
{
	RTOUCH_NO_EVENT,
	RTOUCH_PRESS,
	RTOUCH_MOVE,
	RTOUCH_RELEASE
} rtouch_event_type_t;


//! User-friendly touch event structure.
typedef struct rtouch_event_struct
{
	rtouch_event_type_t type;
	int16_t rawX; //!< Raw X sample.
	int16_t rawY; //!< Raw Y sample.
	int16_t panelX; //!< Calibrated panel X position.
	int16_t panelY; //!< Calibrated panel Y position.
} rtouch_event_t;


typedef void (* rtouch_event_handler_t)( rtouch_event_t const * event );



/*============================ PROTOTYPES ====================================*/

#ifdef __cplusplus
extern "C" {
#endif


//! Initialize touch panel driver.
void rtouch_init(void);
//! Disable driver.
void rtouch_disable(void);
//! Enable driver.
void rtouch_enable(void);

//! Get last event.
void rtouch_get_event(rtouch_event_t * event);
//! True if touched.
bool rtouch_is_touched(void);

//! Compute a calibration matrix from three calibration points.
void rtouch_compute_calibration_matrix(
	rtouch_calibration_points_t const * points,
	rtouch_calibration_matrix_t * matrix );
//! Assign a calibration matrix to the driver.
void rtouch_set_calibration_matrix(rtouch_calibration_matrix_t const *source);
//! Get the current calibration matrix.
void rtouch_get_calibration_matrix(rtouch_calibration_matrix_t *destination);

//! Set new event handler.
void rtouch_set_event_handler(rtouch_event_handler_t handler);
//! Get current event handler.
rtouch_event_handler_t rtouch_get_event_handler(void);


#ifdef __cplusplus
} /* extern "C" */
#endif


#endif
/* EOF */
