/* This header file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief NAND flash driver for AVR32 with local bus support or EBI.
 *
 * This file contains the driver for NAND flash driver.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a CPU local bus can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 *****************************************************************************/

/* Copyright (c) 2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _NAND_H
#define _NAND_H

/*! \brief FCPU must be configured to match the system configuration. */
#define FCPU				48000000

/*! \brief Interface the flash using GPIO and CPU local bus. */
#define	NAND_BUS_TYPE_GPIO		1

/*! \brief Do not use error correction code when interfacing the flash. */
#define NAND_ECC_NONE			1
/*! \brief Use software error correction code when interfacing the flash. */
#define NAND_ECC_SW			2

/*! \brief Define to match the interface used. */
#define NAND_BUS_TYPE			NAND_BUS_TYPE_GPIO
/*! \brief Define to wanted ECC type. */
#define NAND_ECC_TYPE			NAND_ECC_SW
/*! \brief Define to enable marking bad blocks permanently in the NAND flash.
 *         This will write the bad block byte in the spare area for blocks
 *         where commands fail.
 */
/*#define NAND_MARK_BLOCKS_BAD		1*/
/*! \brief Define to enable the function to permanently marked bad blocks in
 *         the NAND flash. This will erase a block, even if marked bad.
 */
/*#define NAND_MARK_BLOCKS_OK		1*/

/*! \brief Define to enable always active chip enable (CE). */
/* #define NAND_CE_ALWAYS_ACTIVE	1 */

/*! \brief Nanoseconds per CPU tick. */
#define NS_PER_TICK					(1000000 / (FCPU / 1000))

/*! \brief NAND flash driver commands. */
enum {
	NAND_CMD_READ	= 1,
	NAND_CMD_RANDOM_READ,
	NAND_CMD_ID,
	NAND_CMD_RESET,
	NAND_CMD_PROGRAM,
	NAND_CMD_RANDOM_PROGRAM,
	NAND_CMD_PROGRAM_COMPLETE,
	NAND_CMD_ERASE,
	NAND_CMD_STATUS,
} nand_cmd_t;

/*! \brief NAND flash driver return values. */
enum {
	SUCCESS = 0,
	/*! \brief I/O error return value. */
	EIO,
	/*! \brief Busy error return value. */
	EBUSY,
	/*! \brief Invalid argument error return value. */
	EINVAL,
	/*! \brief Timeout error return value. */
	ETIMEDOUT,
	/*! \brief ECC error return value. */
	ENANDECC_BADECC,
	/*! \brief Correctable ECC error return value. */
	ENANDECC_CORRECTABLE,
	/*! \brief Uncorrectable ECC error return value. */
	ENANDECC_UNCORRECTABLE,
} nand_return_values_t;

/*! \brief Struct for defining the spare area (out of bounce) in a page. */
struct nand_out_of_bounce_layout {
	/*! \brief Spare are total size. */
	unsigned char	size;
	/*! \brief Number of bytes for ECC data. */
	unsigned char	eccbytes;
	/*! \brief Array of offsets where ECC data can be stored. */
	unsigned char	eccpos[24];
};

/*! \brief NAND flash read page command start. */
#define NAND_COMMAND_READ1				0x00
/*! \brief NAND flash read page command end. */
#define NAND_COMMAND_READ2				0x30
/*! \brief NAND flash read ID command. */
#define NAND_COMMAND_READID				0x90
/*! \brief NAND flash reset command. */
#define NAND_COMMAND_RESET				0xFF
/*! \brief NAND flash program page command start. */
#define NAND_COMMAND_PAGEPROG1				0x80
/*! \brief NAND flash program page command end. */
#define NAND_COMMAND_PAGEPROG2				0x10
/*! \brief NAND flash erase block command start. */
#define NAND_COMMAND_ERASE1				0x60
/*! \brief NAND flash erase block command end. */
#define NAND_COMMAND_ERASE2				0xD0
/*! \brief NAND flash read status command. */
#define NAND_COMMAND_STATUS				0x70
/*! \brief NAND flash random program page command start. */
#define NAND_COMMAND_RANDOM_PAGEPROG			0x85
/*! \brief NAND flash random read page command start. */
#define NAND_COMMAND_RANDOM_READ1			0x05
/*! \brief NAND flash random read page command end. */
#define NAND_COMMAND_RANDOM_READ2			0xE0

/*! \brief Offset in spare area where bad block information is located for large page devices. */
#define NAND_LARGE_BAD_BLOCK_POSITION			0
/*! \brief Offset in spare area where bad block information is located for small page devices. */
#define NAND_SMALL_BAD_BLOCK_POSITION			5

/*! \brief NAND flash block ok. */
#define NAND_BLOCK_OK				0
/*! \brief NAND flash block bad and must not be used. */
#define NAND_BLOCK_BAD				1

/*! \brief NAND flash information about maker, device, size and timing.
*/
struct nand_info {
	/*! \brief Out of bounce layout information. */
	struct nand_out_of_bounce_layout *oob;

	/*! \brief Maker code. */
	unsigned char	maker_code;
	/*! \brief Device code. */
	unsigned char	device_code;

	/*! \brief Page size in bytes. */
	unsigned int	page_size;
	/*! \brief Number of positions to shift when converting an offset in
	 *         a block to page. Used when calculating NAND flash address.
	 */ 
	unsigned int	page_shift;
	/*! \brief Number of pages per block. */
	unsigned int	pages_per_block;
	/*! \brief Block size in bytes. */
	unsigned int	block_size;
	/*! \brief Number of positions to shift when converting block number
	 *         to NAND flash address.
	 */ 
	unsigned int	block_shift;
	/*! \brief Number of blocks. */
	unsigned int	num_blocks;
	/*! \brief NAND flash I/O bus width in bits. */
	unsigned int	bus_width;

	/*! \brief NAND flash number of planes. */
	unsigned char	num_planes;
	/*! \brief NAND flash plane size. */
	unsigned long	plane_size;

	/*! \brief Offset to bad block position in page spare area. */
	unsigned char	badblock_offset;

	/*! \brief NAND flash WE low pulse time. */
	unsigned int	t_wp;
	/*! \brief NAND flash address to data loading time. */
	unsigned int	t_adl;
	/*! \brief NAND flash ALE hold time. */
	unsigned int	t_alh;
	/*! \brief NAND flash ALE setup timing. */
	unsigned int	t_als;
	/*! \brief NAND flash WE high hold time. */
	unsigned int	t_wh;
	/*! \brief NAND flash read cycle time. */
	unsigned int	t_rc;
};

/*! \brief Array of status on all blocks, defining which are bad.
*/
struct nand_bad_table {
	unsigned char				*block_status;
};

/*! \brief NAND flash driver data.
*/
struct nand_driver_data {
	struct nand_info		info;
	struct nand_bad_table		bad_table;

	/*! \brief GPIO line connected to NAND CE signal. */
	int				gpio_ce;
	/*! \brief GPIO line connected to NAND R/B signal. */
	int				gpio_rb;
	/*! \brief GPIO line connected to NAND RE signal. */
	int				gpio_re;
	/*! \brief GPIO line connected to NAND WE signal. */
	int				gpio_we;

	/*! \brief GPIO line connected to NAND WP signal, not mandatory. */
	int				gpio_wp;

	/*! \brief GPIO line connected to NAND ALE signal. */
	int				gpio_ale;
	/*! \brief GPIO line connected to NAND CLE signal. */
	int				gpio_cle;

	/*! \brief GPIO port number on CPU local bus used for I/O access. */
	int				gpio_io_port;
	/*! \brief GPIO port mask on CPU local bus used for I/O access. */
	unsigned long			gpio_io_mask;
	/*! \brief GPIO port address on CPU local bus used for I/O access. */
	unsigned long			*gpio_io_address;
	/*! \brief GPIO port offset to first GPIO line used for I/O access. */
	unsigned long			gpio_io_offset;
	/*! \brief GPIO port size used for I/O access, i.e. number of bits. */
	unsigned long			gpio_io_size;
};

int nand_init(struct nand_driver_data *nfd);

int nand_create_badblocks_table(struct nand_driver_data *nfd);

int nand_erase(struct nand_driver_data *nfd, const unsigned long block);
int nand_read(struct nand_driver_data *nfd, const unsigned long block,
		const unsigned long offset, unsigned char *buf,
		const unsigned short count);
int nand_read_raw(struct nand_driver_data *nfd, const unsigned long block,
		const unsigned long offset, unsigned char *buf,
		const unsigned short count);
int nand_write(struct nand_driver_data *nfd, const unsigned long block,
		const unsigned long offset, const unsigned char *buf,
		const unsigned short count);
int nand_write_raw(struct nand_driver_data *nfd, const unsigned long block,
		const unsigned long offset, const unsigned char *buf,
		const unsigned short count);

#endif /* _NAND_H */
