/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief NAND flash GPIO driver for AVR32 with local bus support.
 *
 * This file contains the example for NAND flash GPIO driver.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a CPU local bus can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 *****************************************************************************/

/* Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
#include "compiler.h"
#include "print_funcs.h"
#include "flashc.h"
#include "pm.h"
#include "gpio.h"
#include "nand.h"
#include "nand_gpio_example_data.h"

/* Defines for where to put the example data in the NAND flash device. */
#define BLOCK_OFFSET	134
#define PAGE_OFFSET	4096

/*! \brief Offset to where the bit error is introduced in the flash. */
#define ERROR_OFFSET	4215

/* OSC0 startup time RCOSC periods. */
#define OSC0_STARTUP    AVR32_PM_OSCCTRL0_STARTUP_2048_RCOSC
/* OSC0 frequency 12 MHz. */
#define FOSC0			12000000
/* PLL0 frequency is 192 MHz. */
#define FPLL0			192000000
/* CPU frequency is 48 MHz. */
#define FCPU			48000000
/* HSB frequency is 48 MHz. */
#define FHSB			48000000
/* PBA frequency is 48 MHz. */
#define FPBA			48000000
/* PBB frequency is 48 MHz. */
#define FPBB			48000000

/*! \brief Compare data in two pointers for a given number of bytes.
 *
 *  This helper function compares data in two pointers and returns the index
 *  of where they differ.
 *
 *  \param orig Pointer to the original data.
 *  \param compare Pointer to the data to compare to orig.
 *  \param count Number of bytes to compare.
 *
 *  \return 0 on success, or the index of not equal data starting on 1.
 */
int compare_data(const unsigned char *orig, const unsigned char *compare,
		unsigned short count)
{
	int i;

	for (i = 0; i < count; i++) {
		if (orig[i] != compare[i]) {
			print_dbg("\r\nERROR: read 0x");
			print_dbg_char_hex(compare[i]);
			print_dbg(", expected 0x");
			print_dbg_char_hex(orig[i]);
			print_dbg(" at index ");
			print_dbg_ulong(i);
			print_dbg("\r\n");
			return i + 1;
		}
	}

	return 0;
}

/*! \brief Start PLL0 and switch main clock to PLL0 output.
 *
 *  \param pm Pointer to the Power Manager.
 */
void switch_to_pll0(volatile avr32_pm_t* pm)
{
	/* Switch main clock to OSC0 */
	pm_switch_to_osc0(pm, FOSC0, OSC0_STARTUP);

	/* PM, PLL#, mul, div, OSC#, lock count. */
	pm_pll_setup(pm, 0, 15, 1, 0, 16);

	/* PM, PLL#, range, no output div by 2, disable wide bandith mode. */
	pm_pll_set_option(pm, 0, 0, 0, 0);

	pm_pll_enable(pm, 0);

	pm_wait_for_pll0_locked(pm) ;

	/* PM, PBA div enable, PBA div, PBB div enable, PBB div, HSB div enable, HSB div. */
	pm_cksel(pm, 1, 1, 1, 1, 1, 1);

	/* HSB/CPU clock is above 30 MHz so we need 1 wait state. */
	flashc_set_wait_state(1);

	/* Switch main clock to 48MHz */
	pm_switch_to_clock(pm, AVR32_PM_MCSEL_PLL0);
}

/*! \brief Main function for NAND GPIO driver.
 *
 *  \return Only on failure.
 */
int main(void)
{
	int offset;
	int block;
	int retval;
	unsigned char *buffer;
	unsigned char *block_status;
	volatile avr32_pm_t* pm = &AVR32_PM;
	struct nand_driver_data nfd;

	/* Switch the main clock to PLL0. */
	switch_to_pll0(pm);

	/* Init USART for debug messages. */
	init_dbg_rs232(FPBA);

	print_dbg("AVR32 NAND GPIO example application\r\n");

	/*
	 * TODO: these GPIO values must match hardware setup.
	 */
	nfd.gpio_ce		= AVR32_PIN_PA27;
	nfd.gpio_re		= AVR32_PIN_PB00;
	nfd.gpio_rb		= AVR32_PIN_PA02;
	nfd.gpio_we		= AVR32_PIN_PB07;
	nfd.gpio_cle		= AVR32_PIN_PB01;
	nfd.gpio_ale		= AVR32_PIN_PB04;
	nfd.gpio_wp		= AVR32_PIN_PA28;
	nfd.gpio_io_port	= 1; /* PORT B              */
	nfd.gpio_io_offset	= 8; /* PB08 to PB15        */
	nfd.gpio_io_size	= 8; /* 8-bit I/O interface */

	/*
	 * TODO: These timing values in nanoseconds must match the minimum timing
	 *       values from the NAND flash device datasheet.
	 */
	nfd.info.t_adl = 100;
	nfd.info.t_alh = 5;
	nfd.info.t_als = 12;
	nfd.info.t_rc  = 25;
	nfd.info.t_wh  = 10;
	nfd.info.t_wp  = 12;

	nfd.gpio_io_address = (unsigned long *)(AVR32_GPIO_LOCAL_ADDRESS
			+ (0x100 * nfd.gpio_io_port));

	/* Turn on the local bus. */
	Set_system_register(AVR32_CPUCR, Get_system_register(AVR32_CPUCR) |
			AVR32_CPUCR_LOCEN_MASK);

	/*
	 * Initialize the NAND GPIO driver and read the NAND flash ID.
	 */

	retval = nand_init(&nfd);
	if (retval) {
		print_dbg("ERROR: failed to init NAND flash\r\n");
		return retval;
	}

	block_status = malloc(nfd.info.num_blocks);
	if (!block_status) {
		print_dbg("ERROR: could not allocate badblock table\r\n");
		return -1;
	}

	nfd.bad_table.block_status = block_status;

	buffer = malloc(nfd.info.page_size);
	if (!buffer) {
		print_dbg("ERROR: could not allocate buffer\r\n");
		return -1;
	}

	print_dbg(" - NAND flash initialized\r\n");
	print_dbg("    - maker code : ");
	print_dbg_char_hex(nfd.info.maker_code);
	print_dbg("\r\n");
	print_dbg("    - device code: ");
	print_dbg_char_hex(nfd.info.device_code);
	print_dbg("\r\n");
	print_dbg("    - page size  : ");
	print_dbg_ulong(nfd.info.page_size);
	print_dbg(" bytes\r\n");
	print_dbg("    - spare size : ");
	print_dbg_ulong(nfd.info.oob->size);
	print_dbg(" bytes\r\n");
	print_dbg("    - block size : ");
	print_dbg_ulong(nfd.info.block_size);
	print_dbg(" bytes\r\n");
	print_dbg("    - block num  : ");
	print_dbg_ulong(nfd.info.num_blocks);
	print_dbg(" bytes\r\n");
	print_dbg("    - plane num  : ");
	print_dbg_ulong(nfd.info.num_planes);
	print_dbg("\r\n");
	print_dbg("    - plane size : ");
	print_dbg_ulong(nfd.info.plane_size);
	print_dbg(" bytes\r\n");
	print_dbg("    - flash size : ");
	print_dbg_ulong(nfd.info.num_blocks * nfd.info.block_size / 1048576);
	print_dbg(" MB\r\n");
	print_dbg("    - bus width  : ");
	print_dbg_ulong(nfd.info.bus_width);
	print_dbg("-bit\r\n");

	/*
	 * Create the NAND flash badblocks table.
	 */

	print_dbg(" - Creating NAND flash badblocks table\r\n");

	retval = nand_create_badblocks_table(&nfd);
	if (retval < 0) {
		print_dbg("ERROR: could not create badblocks table\r\n");
		return -1;
	}
	else if (retval) {
		print_dbg("    - INFO: number of bad blocks: ");
		print_dbg_ulong(retval);
		print_dbg("\r\n");
	}

	print_dbg("    - Done\r\n");

	/*
	 * Erase the NAND flash.
	 */
	print_dbg(" - Erasing NAND flash\r\n    ");

	for (block = 0; block < nfd.info.num_blocks; block++) {
		retval = nand_erase(&nfd, block);
		if (retval) {
			print_dbg("(");
			print_dbg_ulong(block);
			print_dbg(" bad)");
		}
		if (!(block % 128)) {
			print_dbg(".");
		}
	}

	print_dbg("\r\n    - Done\r\n");

	/*
	 * Check that all NAND flash blocks are erased.
	 */
	print_dbg(" - Checking if NAND flash is erased\r\n    ");

	for (block = 0; block < nfd.info.num_blocks; block++) {
		for (offset = 0; offset < nfd.info.block_size;
				offset += nfd.info.page_size) {
			int index = 0;

			retval = nand_read_raw(&nfd, block, offset, buffer, 2);

			if (retval < 0) {
				print_dbg("(");
				print_dbg_ulong(block);
				print_dbg(" bad)");
				break;
			}

			do {
				if (buffer[index] != 0xff) {
					print_dbg("(");
					print_dbg_ulong(block);
					print_dbg(",");
					print_dbg_ulong(offset);
					print_dbg(",");
					print_dbg_ulong(index);
					print_dbg(" unerased)");
					break;
				}
			} while (++index < 2);
		}

		if (!(block % 128)) {
			print_dbg(".");
		}
	}

	print_dbg("\r\n    - Done\r\n");

	/*
	 * Program known data to a NAND flash.
	 */
	print_dbg(" - Programming test data to NAND flash\r\n    ");

	for (offset = PAGE_OFFSET;
			offset < (NAND_GPIO_EXAMPLE_DATA_SIZE + PAGE_OFFSET);
			offset += nfd.info.page_size) {
		retval = nand_write(&nfd, BLOCK_OFFSET, offset,
				&data[offset - PAGE_OFFSET],
				nfd.info.page_size);
		if (retval < 0) {
			print_dbg("(");
			print_dbg_ulong(BLOCK_OFFSET);
			print_dbg(",");
			print_dbg_ulong(offset);
			print_dbg(" bad)");
			break;
		}

		print_dbg(".");
	}

	print_dbg("\r\n    - Done\r\n");

	/*
	 * Read back known data from the NAND flash.
	 */
	print_dbg(" - Read test data from NAND flash and compare\r\n    ");

	for (offset = PAGE_OFFSET;
			offset < (NAND_GPIO_EXAMPLE_DATA_SIZE + PAGE_OFFSET);
			offset += nfd.info.page_size) {
		retval = nand_read(&nfd, BLOCK_OFFSET, offset,
				buffer, nfd.info.page_size);

		if (retval < 0) {
			print_dbg("\r\nERROR: block ");
			print_dbg_ulong(BLOCK_OFFSET);
			print_dbg(", offset ");
			print_dbg_ulong(offset);
			print_dbg(" is bad");
			break;
		} else {
			retval = compare_data(&data[offset - PAGE_OFFSET],
					buffer, nfd.info.page_size);
			if (retval) {
				break;
			}
		}

		print_dbg(".");
	}

	if (retval > 0) {
		print_dbg("ERROR: compare mismatch failed at block ");
		print_dbg_ulong(BLOCK_OFFSET);
		print_dbg(", offset ");
		print_dbg_ulong(offset);
		print_dbg(", index ");
		print_dbg_ulong(retval - 1);
		print_dbg("\r\n");
	}
	else if (retval < 0) {
		print_dbg("\r\nERROR: compare mismatch failed\r\n");
	} else {
		print_dbg("\r\n    - INFO: compare success\r\n    - Done\r\n");
	}

	/*
	 * Introduce a single bit error in the NAND flash.
	 */
	print_dbg(" - Introducing a single bit error in NAND flash\r\n");

	nand_read_raw(&nfd, BLOCK_OFFSET, PAGE_OFFSET + ERROR_OFFSET, buffer, 1);

	print_dbg("   - Read 0x");
	print_dbg_hex(buffer[0]);
	print_dbg("\r\n");

	buffer[0] = data[ERROR_OFFSET] ^ (1 << 2);

	print_dbg("   - Introducing bit error\r\n");

	nand_write_raw(&nfd, BLOCK_OFFSET, PAGE_OFFSET + ERROR_OFFSET, buffer, 1);
	nand_read_raw(&nfd, BLOCK_OFFSET, PAGE_OFFSET + ERROR_OFFSET, buffer, 1);

	print_dbg("   - Read 0x");
	print_dbg_hex(buffer[0]);
	print_dbg("\r\n");

	/*
	 * Read back known data from the NAND flash.
	 */
	print_dbg(" - Read test data from NAND flash and compare\r\n    ");

	for (offset = PAGE_OFFSET;
			offset < (NAND_GPIO_EXAMPLE_DATA_SIZE + PAGE_OFFSET);
			offset += nfd.info.page_size) {
		retval = nand_read(&nfd, BLOCK_OFFSET, offset,
				buffer, nfd.info.page_size);

		if (retval < 0) {
			print_dbg("\r\nERROR: block ");
			print_dbg_ulong(BLOCK_OFFSET);
			print_dbg(", offset ");
			print_dbg_ulong(offset);
			print_dbg(" is bad");
			break;
		} else {
			retval = compare_data(&data[offset - PAGE_OFFSET],
					buffer, nfd.info.page_size);
			if (retval) {
				break;
			}
		}

		print_dbg(".");
	}

	if (retval > 0) {
		print_dbg("ERROR: compare mismatch failed at block ");
		print_dbg_ulong(BLOCK_OFFSET);
		print_dbg(", offset ");
		print_dbg_ulong(offset);
		print_dbg(", index ");
		print_dbg_ulong(retval - 1);
		print_dbg("\r\n");
	}
	else if (retval < 0) {
		print_dbg("\r\nERROR: compare mismatch failed\r\n");
	} else {
		print_dbg("\r\n    - INFO: compare success\r\n    - Done\r\n");
	}

	for (;;) {
		/* Game over */
	}
}
