/* This source file is part of the ATMEL AVR32-UC3-SoftwareFramework-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief Main file of the CS2200 example.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/*! \mainpage AVR32 Software Framework for CS2200 example
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */

//_____  I N C L U D E S ___________________________________________________

#if (defined __GNUC__)
# include "nlao_cpu.h"
# include "nlao_usart.h"
#endif
#include <stdio.h>
#include "compiler.h"
#include "preprocessor.h"
#include "board.h"
#include "print_funcs.h"
#include "flashc.h"
#include "gpio.h"
#include "intc.h"
#include "pm.h"
#include "twi.h"
#include "usart.h"
#include "main.h"
#include "conf_cs2200.h"
#include "cycle_counter.h"
#include "cs2200.h"
#include "et024006dhu.h"


//_____ D E F I N I T I O N S ______________________________________________

/*! \name System Clock Frequencies
 */
//! @{
pm_freq_param_t   pm_freq_param=
{
   .cpu_f  =       FCPU_HZ
,  .pba_f    =     FPBA_HZ
,  .osc0_f     =   FOSC0
,  .osc0_startup = OSC0_STARTUP
};
//! @}

static void mmi_print_line(char *string, et024006_color_t fg_color, et024006_color_t bg_color);


#if __GNUC__ && __AVR32__

/*! \brief Replaces Newlib's implementation of the standard \c wcstombs function
 *         because of portability issues with \c wchar_t.
 *
 * \warning This function must be used with the `-fshort-wchar' code generation
 *          option.
 */
size_t wcstombs(char *s, const wchar_t *pwcs, size_t n)
{
  size_t count = 0;

  while (n--)
  {
    if ((*s++ = (char)*pwcs++) == '\0')
      break;
    count++;
  }

  return count;
}

#endif


/*! \brief Initializes MCU exceptions.
 */
static void init_exceptions(void)
{
#if __GNUC__ && __AVR32__
  // Import the Exception Vector Base Address.
  extern void _evba;

  // Load the Exception Vector Base Address in the corresponding system
  // register.
  Set_system_register(AVR32_EVBA, (int)&_evba);
#endif

  // Enable exceptions globally.
  Enable_global_exception();
}


/*! \brief Initializes the HSB bus matrix.
 */
static void init_hmatrix(void)
{
  union
  {
    unsigned long                 scfg;
    avr32_hmatrix_scfg_t          SCFG;
  } u_avr32_hmatrix_scfg;

  // For the internal-flash HMATRIX slave, use last master as default.
  u_avr32_hmatrix_scfg.scfg = AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_FLASH];
  u_avr32_hmatrix_scfg.SCFG.defmstr_type = AVR32_HMATRIX_DEFMSTR_TYPE_LAST_DEFAULT;
  AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_FLASH] = u_avr32_hmatrix_scfg.scfg;

  // For the internal-SRAM HMATRIX slave, use last master as default.
  u_avr32_hmatrix_scfg.scfg = AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_SRAM];
  u_avr32_hmatrix_scfg.SCFG.defmstr_type = AVR32_HMATRIX_DEFMSTR_TYPE_LAST_DEFAULT;
  AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_SRAM] = u_avr32_hmatrix_scfg.scfg;

  // For the USBB DPRAM HMATRIX slave, use last master as default.
  u_avr32_hmatrix_scfg.scfg = AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_USBB_DPRAM];
  u_avr32_hmatrix_scfg.SCFG.defmstr_type = AVR32_HMATRIX_DEFMSTR_TYPE_LAST_DEFAULT;
  AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_USBB_DPRAM] = u_avr32_hmatrix_scfg.scfg;

  // For the EBI HMATRIX slave, use last master as default.
  u_avr32_hmatrix_scfg.scfg = AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_EBI];
  u_avr32_hmatrix_scfg.SCFG.defmstr_type = AVR32_HMATRIX_DEFMSTR_TYPE_LAST_DEFAULT;
  AVR32_HMATRIX.scfg[AVR32_HMATRIX_SLAVE_EBI] = u_avr32_hmatrix_scfg.scfg;
}


/*! \brief Initializes the MCU system clocks.
 */
static void init_sys_clocks(void)
{
  // Switch to OSC0 to speed up the booting
  pm_switch_to_osc0(&AVR32_PM, FOSC0, OSC0_STARTUP);

  // Start oscillator1
  pm_enable_osc1_crystal(&AVR32_PM, FOSC0);
  //
  pm_enable_clk1(&AVR32_PM, OSC0_STARTUP);

  // Set PLL0 (fed from OSC0 = 12 MHz) to 132 MHz
  pm_pll_setup(&AVR32_PM, 0,  // pll.
      10,  // mul.
      1,   // div.
      0,   // osc.
      16); // lockcount.

  // Set PLL operating range and divider (fpll = fvco/2)
  // -> PLL0 output = 66 MHz
  pm_pll_set_option(&AVR32_PM, 0, // pll.
      1,  // pll_freq.
      1,  // pll_div2.
      0); // pll_wbwdisable.

  // start PLL0 and wait for the lock
  pm_pll_enable(&AVR32_PM, 0);
  pm_wait_for_pll0_locked(&AVR32_PM);
  // Set all peripheral clocks torun at master clock rate
  pm_cksel(&AVR32_PM,
      0,   // pbadiv.
      0,   // pbasel.
      0,   // pbbdiv.
      0,   // pbbsel.
      0,   // hsbdiv.
      0);  // hsbsel.

  // Set one waitstate for the flash
  flashc_set_wait_state(1);

  // Switch to PLL0 as the master clock
  pm_switch_to_clock(&AVR32_PM, AVR32_PM_MCCTRL_MCSEL_PLL0);


#if __GNUC__ && __AVR32__
  // Give the used PBA clock frequency to Newlib Addon library, so it can work properly.
  set_cpu_hz(FPBA_HZ);
#endif
}


/*! \brief Initializes STDIO.
 */
static void init_stdio(void)
{
#if __GNUC__ && __AVR32__

  static const gpio_map_t STDIO_USART_GPIO_MAP =
  {
    {STDIO_USART_RX_PIN, STDIO_USART_RX_FUNCTION},
    {STDIO_USART_TX_PIN, STDIO_USART_TX_FUNCTION}
  };

  // Initialize the USART used for STDIO.
  set_usart_base((void *)STDIO_USART);
  gpio_enable_module(STDIO_USART_GPIO_MAP,
                     sizeof(STDIO_USART_GPIO_MAP) / sizeof(STDIO_USART_GPIO_MAP[0]));
  usart_init(STDIO_USART_BAUDRATE);

#elif __ICCAVR32__

  static const gpio_map_t STDIO_USART_GPIO_MAP =
  {
    {STDIO_USART_RX_PIN, STDIO_USART_RX_FUNCTION},
    {STDIO_USART_TX_PIN, STDIO_USART_TX_FUNCTION}
  };

  static const usart_options_t STDIO_USART_OPTIONS =
  {
    .baudrate     = STDIO_USART_BAUDRATE,
    .charlength   = 8,
    .paritytype   = USART_NO_PARITY,
    .stopbits     = USART_1_STOPBIT,
    .channelmode  = USART_NORMAL_CHMODE
  };

  // Initialize the USART used for STDIO.
  extern volatile avr32_usart_t *volatile stdio_usart_base;
  stdio_usart_base = STDIO_USART;
  gpio_enable_module(STDIO_USART_GPIO_MAP,
                     sizeof(STDIO_USART_GPIO_MAP) / sizeof(STDIO_USART_GPIO_MAP[0]));
  usart_init_rs232(STDIO_USART, &STDIO_USART_OPTIONS, FPBA_HZ);

#endif
}

  #if __GNUC__

/*! \brief Low-level initialization routine called during startup, before the
 *         main function.
 *
 * This version comes in replacement to the default one provided by the Newlib
 * add-ons library.
 * Newlib add-ons' _init_startup only calls init_exceptions, but Newlib add-ons'
 * exception and interrupt vectors are defined in the same section and Newlib
 * add-ons' interrupt vectors are not compatible with the interrupt management
 * of the INTC module.
 * More low-level initializations are besides added here.
 */
int _init_startup(void)
{
  // Import the Exception Vector Base Address.
  extern void _evba;

  // Load the Exception Vector Base Address in the corresponding system register.
  Set_system_register(AVR32_EVBA, (int)&_evba);

  // Enable exceptions.
  Enable_global_exception();

  // Initialize interrupt handling.
  INTC_init_interrupts();

  // Don't-care value for GCC.
  return 1;
}

  #elif __ICCAVR32__

/*! \brief Low-level initialization routine called during startup, before the
 *         main function.
 */
int __low_level_init(void)
{
  // Enable exceptions.
  Enable_global_exception();

  // Initialize interrupt handling.
  INTC_init_interrupts();

  // Request initialization of data segments.
  return 1;
}

  #endif  // Compiler


/*! \brief Initializes the two-wire interface using the internal RCOSC.
 */
static void init_twi(U32 fpba_hz)
{
  volatile U32 i;

  static const gpio_map_t CS2200_TWI_GPIO_MAP =
  {
    {AVR32_TWI_SCL_0_0_PIN, AVR32_TWI_SCL_0_0_FUNCTION},
    {AVR32_TWI_SDA_0_0_PIN, AVR32_TWI_SDA_0_0_FUNCTION}
  };

  static twi_options_t CS2200_TWI_OPTIONS =
  {
    .speed  = CS2200_TWI_MASTER_SPEED,
    .chip   = CS2200_TWI_SLAVE_ADDRESS
  };
  CS2200_TWI_OPTIONS.pba_hz = fpba_hz;

  gpio_enable_module(CS2200_TWI_GPIO_MAP,
                     sizeof(CS2200_TWI_GPIO_MAP) / sizeof(CS2200_TWI_GPIO_MAP[0]));
  twi_master_init(CS2200_TWI, &CS2200_TWI_OPTIONS);

}


/*! \brief Main function. Execution starts here.
 *
 * \retval 42 Fatal error.
 */
int main(void)
{
  U32 iter=0;
  U32 cs2200_out_freq=11289600;
  static Bool b_sweep_up=TRUE;
  static U32 freq_step=0;

  // Initialize the TWI using the internal RCOSC
  init_twi(AVR32_PM_RCOSC_FREQUENCY);

  // Initialize the CS2200 and produce a default 12MHz frequency.
  cs2200_setup(11289600);

  // Initializes the MCU system clocks
  init_sys_clocks();

  // Initialize the TWI
  init_twi(FPBA_HZ);

  // Set CPU and PBA clock
  if( PM_FREQ_STATUS_FAIL==pm_configure_clocks(&pm_freq_param) )
     return 42;

  // Initialize the USART for debug purpose.
  init_stdio();

  // Initialize the HMatrix.
  init_hmatrix();

  // Initialize USART link.
  init_dbg_rs232(pm_freq_param.pba_f);
  
  // Initialize the LCD.
  et024006_Init( pm_freq_param.cpu_f, pm_freq_param.cpu_f /*HSB*/);

  // Clear the display i.e. make it black
  et024006_DrawFilledRect(0, 0, ET024006_WIDTH, ET024006_HEIGHT, BLACK );

  // Set the backlight.
  gpio_set_gpio_pin(ET024006DHU_BL_PIN);

  mmi_print_line("CS2200 example", WHITE, BLACK);

  // Generate a 12.288 MHz frequency out of the CS2200.
  mmi_print_line("Output 12.288 MHz", WHITE, BLACK);
  cs2200_freq_clk_out(_32_BITS_RATIO(12288000));
  cpu_delay_ms( 10000, FCPU_HZ);

  // Generate a 11.2896 MHz frequency out of the CS2200.
  mmi_print_line("Output 11.2896 MHz", WHITE, BLACK);
  cs2200_freq_clk_out(_32_BITS_RATIO(cs2200_out_freq));
  cpu_delay_ms( 10000, FCPU_HZ);

  mmi_print_line("Sweep from 11.2896 MHz steps of 100 PPM", WHITE, BLACK);
  freq_step = PPM(cs2200_out_freq, 100);

  while(1)
  {
    U32 ratio;
 
    if(b_sweep_up)
    {
      if( iter<=10 )
      {
        mmi_print_line("Add 100 PPM", WHITE, BLACK);
        iter++;
        cs2200_out_freq += freq_step;
        ratio = _32_BITS_RATIO(cs2200_out_freq);
        cs2200_freq_clk_adjust((U16)ratio);
        cpu_delay_ms( 1000, FCPU_HZ);
        while( twi_is_busy() ); 
      }
      else
        b_sweep_up=FALSE;
    }

    if(!b_sweep_up)
    {
      if( iter>0 )
      {
        mmi_print_line("Sub 100 PPM", WHITE, BLACK);
        iter--;
        cs2200_out_freq -= freq_step;
        ratio = _32_BITS_RATIO(cs2200_out_freq);
        cs2200_freq_clk_adjust((U16)ratio);
        cpu_delay_ms( 1000, FCPU_HZ);
        while( twi_is_busy() ); 
      }
      else
        b_sweep_up=TRUE;
    }
  }
}

/*! \brief Prints status information to the display and to stdio.
 */
static void mmi_print_line(char *string, et024006_color_t fg_color, et024006_color_t bg_color)
{
    static U8 current_line = 1;
    static U8 clear_lines = 0;

    if(clear_lines || current_line == 23)
    {
      et024006_DrawFilledRect(10, (current_line) * 10, 320, 10, bg_color);
      clear_lines = 1;
    }

    et024006_PrintString(string, (const unsigned char*) &FONT6x8,
                         10, 10 * current_line, fg_color, bg_color);
    
    current_line = current_line % 23;
    current_line++;

    print_dbg(string);
    print_dbg("\n");
}
