/* This source file is part of the ATMEL AVR32-SoftwareFramework-AT32UC3-1.6.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief Audio Codec Example.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with PDCA and USART modules.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/*! \page License
 * Copyright (c) 2009 Atmel Corporation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of Atmel may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * 4. This software may only be redistributed and used in connection with an Atmel
 * AVR product.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT ARE
 * EXPRESSLY AND SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 *
 */
/*! \mainpage
 * \section intro Introduction
 * This documents data structures, functions, variables, defines, enums, and
 * typedefs in the software for the Audio Codec driver.
 *
 * The given example is a transfer of an input streaming buffer of audio samples on an output
 * using the Audio Codec support. All the transferts use the SSC channel of the PDCA.
 *
 * \section compinfo Compilation Info
 * This software was written for the GNU GCC for AVR32 and IAR Systems compiler
 * for AVR32. Other compilers may or may not work.
 *
 * \section files Main Files
 * - tlv320aic23b.c: Audio Codec driver.
 * - tlv320aic23b.h: Audio Codec header file.
 * - conf_tlv320aic23b.h: Configuration file for the Audio Codec application. 
 * - tlv320aic23b_example.c: Audio Codec application example.
 *
 * \section configinfo Configuration Information
 * This example has been tested with the following configuration:
 * - EVK1101 evaluation kit with evk1101_extansion_board;
 * - CPU clock: 12 MHz;
 *
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.atmel.com/products/AVR32/">Atmel AVR32</A>.\n
 * Support and FAQ: http://support.atmel.no/
 */

#include <avr32/io.h>
#include <stdio.h>
#include <string.h>
#include "compiler.h"
#include "preprocessor.h"

#include "flashc.h"
#include "pm.h"
#include "gpio.h"
#include "intc.h"
#include "board.h"
#include "spi.h"

#include "conf_tlv320aic23b.h"  
#include "tlv320aic23b.h"
#include "audio.h"

//! Sample Count Value
#define SAMPLE_COUNT                      2048

S16 sound_buffer1[SAMPLE_COUNT];
S16 sound_buffer2[SAMPLE_COUNT];

void dac_reload_callback(void);
void dac_overrun_callback(void);
void adc_underrun_callback(void);
void adc_reload_callback(void);

void master_callback(U32 arg)
{
  if( arg == AUDIO_DAC_OUT_OF_SAMPLE_CB )
  {
    dac_overrun_callback();
  }

  else if( arg == AUDIO_DAC_RELOAD_CB )
  {
    dac_reload_callback();
  }

  else if( arg == AUDIO_ADC_OUT_OF_SAMPLE_CB )
  {
    adc_underrun_callback();;
  }

  else if( arg == AUDIO_ADC_RELOAD_CB )
  {
    adc_reload_callback();;
  }
}


void dac_reload_callback(void)
{
  // Nothing todo
}

void dac_overrun_callback(void)
{
  // Nothing todo
}


void adc_underrun_callback(void)
{
  // Nothing todo
}


void adc_reload_callback(void)
{
  // Nothing todo
}

/*! \brief Initializes Audio Codec in SPI
 */
static void aic23b_resources_init(void)
{

  static const gpio_map_t AIC23B_SPI_GPIO_MAP =
  {
    {AIC23B_SPI_NPCS_PIN, AIC23B_SPI_NPCS_FUNCTION},  // Chip Select for DAC Audio NPCS.
    {AIC23B_SPI_SCK_PIN,  AIC23B_SPI_SCK_FUNCTION },  // SPI Clock.
    {AIC23B_SPI_MISO_PIN, AIC23B_SPI_MISO_FUNCTION},  // MISO.
    {AIC23B_SPI_MOSI_PIN, AIC23B_SPI_MOSI_FUNCTION},  // MOSI.
  };

  // SPI options.
  static const spi_options_t AIC23B_SPI_OPTIONS =
  {
    .reg          = AIC23B_SPI_NPCS,
    .baudrate     = AIC23B_SPI_MASTER_SPEED,
    .bits         = AIC23B_CTRL_SIZE,
    .spck_delay   = 0,
    .trans_delay  = 0,
    .stay_act     = 0,
    .spi_mode     = 3,
    .modfdis      = 1
  };

  // Assign I/Os to SPI.
  gpio_enable_module(AIC23B_SPI_GPIO_MAP,
                     sizeof(AIC23B_SPI_GPIO_MAP) / sizeof(AIC23B_SPI_GPIO_MAP[0]));

  // Initialize as master.
  spi_initMaster(AIC23B_SPI, &AIC23B_SPI_OPTIONS);

  // Set selection mode: variable_ps, pcs_decode, delay.
  spi_selectionMode(AIC23B_SPI, 0, 0, 0);

  // Enable SPI.
  spi_enable(AIC23B_SPI);

  // Initialize data flash with SPI clock Osc0.
  spi_setupChipReg(AIC23B_SPI, &AIC23B_SPI_OPTIONS, FOSC0);

  
}
int main(void)
{
  // Switch to OSC0 as source of main clock.
  pm_switch_to_osc0(&AVR32_PM, FOSC0, OSC0_STARTUP);

  // Initialize interrupt handling.
  INTC_init_interrupts();

  // Enable interrupts globally.
  Enable_global_interrupt();
  
  // Audio Codec Ressource Init
  aic23b_resources_init();
   
  // Codec Start
  aic23b_codec_start(DEFAULT_DAC_SAMPLE_RATE_HZ,
                    DEFAULT_DAC_NUM_CHANNELS,
                    DEFAULT_DAC_BITS_PER_SAMPLE,
                    DEFAULT_DAC_SWAP_CHANNELS,
                    master_callback,
                      AUDIO_DAC_OUT_OF_SAMPLE_CB
                    | AUDIO_DAC_RELOAD_CB,
                    FOSC0);

  // Record of buffer 1
  aic23b_adc_input(sound_buffer1,(SAMPLE_COUNT/2));
  unsigned char step = 1;
  while (1)
  {    
    LED_Toggle(LED0);
    switch(step)
    {
    case 0:
        // Wait until the end of reception of buffer 1 
        while(!aic23b_adc_input(sound_buffer1,0));
        // Play buffer 1 after complete reception
        aic23b_dac_output(sound_buffer1,(SAMPLE_COUNT/2));        
        // Wait until the end of transmission of buffer 1
        while(!aic23b_dac_output(sound_buffer1,0));
        // Record of buffer 2        
        aic23b_adc_input(sound_buffer2,(SAMPLE_COUNT/2)); 
        step = 1;
        break;
    case 1:
        // Wait until the end of reception of buffer 2       
        while(!aic23b_adc_input(sound_buffer2,0));
        // Play buffer 2 after complete reception
        aic23b_dac_output(sound_buffer2,(SAMPLE_COUNT/2));
        // Wait until the end of transmission of buffer 2
        while(!aic23b_dac_output(sound_buffer2,0));        
        // Record of buffer 2        
        aic23b_adc_input(sound_buffer1,(SAMPLE_COUNT/2));        
        step = 0;
        break;        
    }
  } 
}
