/* This source file is part of the ATMEL AVR32-SoftwareFramework-AT32AP7000-1.0.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief SSC I2S example application.
 *
 * This file gives an example of using the SSC in I2S mode.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with an SSC module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 *****************************************************************************/

/*! \page License
 * Copyright (C) 2006-2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*! \mainpage
 * \section intro Introduction
 * This documents data structures, functions, variables, defines, enums, and
 * typedefs in the software for application note AVR32x. It also gives an
 * example of the usage of the SSC driver generating I2S output.
 *
 * \section files Main Files
 *  - ssc_i2s.c: SSC I2S driver
 *  - ssc_i2s.h: SSC I2S driver header file
 *  - ssc_i2s_example.c: SSC I2S example application.
 *
 * \section compinfo Compilation Info
 * This software was written for the GNU GCC for AVR32 and IAR Systems compiler
 * for AVR32. Other compilers may or may not work.
 *
 * \section deviceinfo Device Info
 * All AVR32 devices with an SSC module can be used. This example has been tested
 * with the following setup:
 *
 * - STK1000 with a oscilloscope probing the I2S signals.
 *
 * \section setupinfo Setup Information
 * CPU speed: <i>20 MHz</i>
 *
 * \section applicationUsage Application Usage
 *
 * \subsection applicationUsageMaster Master mode
 *
 * \section contactinfo Contact Information
 * For further information, visit
 * <A href="http://www.atmel.com/products/AVR32/">Atmel AVR32</A>.\n
 * Support and FAQ: http://support.atmel.no/
 */


#include <avr32/io.h>
#include "board.h"
#include "pm_at32ap7000.h"
#include "gpio.h"
#include "ssc_i2s.h"


/*! \brief Main function, application starts executing here after
 *         initializing the stack pointer.
 */
int main(void)
{
#if BOARD == STK1000
  #define EXAMPLE_SSC (&AVR32_SSC0)
  #define EXAMPLE_SSC_TX_CLOCK_PIN            AVR32_SSC0_TX_CLOCK_0_PIN
  #define EXAMPLE_SSC_TX_CLOCK_FUNCTION       AVR32_SSC0_TX_CLOCK_0_FUNCTION
  #define EXAMPLE_SSC_TX_DATA_PIN             AVR32_SSC0_TX_DATA_0_PIN
  #define EXAMPLE_SSC_TX_DATA_FUNCTION        AVR32_SSC0_TX_DATA_0_FUNCTION
  #define EXAMPLE_SSC_TX_FRAME_SYNC_PIN       AVR32_SSC0_TX_FRAME_SYNC_0_PIN
  #define EXAMPLE_SSC_TX_FRAME_SYNC_FUNCTION  AVR32_SSC0_TX_FRAME_SYNC_0_FUNCTION
#elif BOARD == NGW100
  #define EXAMPLE_SSC (&AVR32_SSC0)
  #define EXAMPLE_SSC_TX_CLOCK_PIN            AVR32_SSC0_TX_CLOCK_0_PIN
  #define EXAMPLE_SSC_TX_CLOCK_FUNCTION       AVR32_SSC0_TX_CLOCK_0_FUNCTION
  #define EXAMPLE_SSC_TX_DATA_PIN             AVR32_SSC0_TX_DATA_0_PIN
  #define EXAMPLE_SSC_TX_DATA_FUNCTION        AVR32_SSC0_TX_DATA_0_FUNCTION
  #define EXAMPLE_SSC_TX_FRAME_SYNC_PIN       AVR32_SSC0_TX_FRAME_SYNC_0_PIN
  #define EXAMPLE_SSC_TX_FRAME_SYNC_FUNCTION  AVR32_SSC0_TX_FRAME_SYNC_0_FUNCTION
#elif BOARD == EVK1100
  #define EXAMPLE_SSC (&AVR32_SSC)
  #define EXAMPLE_SSC_TX_CLOCK_PIN            AVR32_SSC_TX_CLOCK_0_PIN
  #define EXAMPLE_SSC_TX_CLOCK_FUNCTION       AVR32_SSC_TX_CLOCK_0_FUNCTION
  #define EXAMPLE_SSC_TX_DATA_PIN             AVR32_SSC_TX_DATA_0_PIN
  #define EXAMPLE_SSC_TX_DATA_FUNCTION        AVR32_SSC_TX_DATA_0_FUNCTION
  #define EXAMPLE_SSC_TX_FRAME_SYNC_PIN       AVR32_SSC_TX_FRAME_SYNC_0_PIN
  #define EXAMPLE_SSC_TX_FRAME_SYNC_FUNCTION  AVR32_SSC_TX_FRAME_SYNC_0_FUNCTION
#endif

  static const gpio_map_t SSC_GPIO_MAP =
  {
    {EXAMPLE_SSC_TX_CLOCK_PIN,      EXAMPLE_SSC_TX_CLOCK_FUNCTION     },
    {EXAMPLE_SSC_TX_DATA_PIN,       EXAMPLE_SSC_TX_DATA_FUNCTION      },
    {EXAMPLE_SSC_TX_FRAME_SYNC_PIN, EXAMPLE_SSC_TX_FRAME_SYNC_FUNCTION}
  };

  volatile avr32_ssc_t *ssc = EXAMPLE_SSC;

  // Switch main clock to external oscillator 0 (crystal).
  //pm_switch_to_osc0(&AVR32_PM, FOSC0, OSC0_STARTUP);

  // Assign GPIO to SSC.
  gpio_enable_module(SSC_GPIO_MAP, sizeof(SSC_GPIO_MAP) / sizeof(SSC_GPIO_MAP[0]));

  // SSC init in I2S mode.
	ssc_i2s_init(ssc, 11025, 8, 8, SSC_I2S_MODE_STEREO_OUT, FOSC0);

	unsigned char leds = 0x00;

	while (1)
  {
		leds <<= 1;
		leds ^= (leds ^ 0x08) >> 3;
    LED_Display_Field(LED_MONO0_GREEN |
                      LED_MONO1_GREEN |
                      LED_MONO2_GREEN |
                      LED_MONO3_GREEN,
                      leds);

    while (!(ssc_i2s_get_status(ssc) & AVR32_SSC_SR_TXRDY_MASK));
		ssc_i2s_transfer(ssc, 0xA3);
	}
}
