/* This source file is part of the ATMEL AVR32-SoftwareFramework-AT32AP7000-1.0.0 Release */

/*This file has been prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief This example demonstrates PIO functionality
 *
 * This examples demonstrates how to use some pins as input and some as output.
 * It routes the status from a set of pins to other pins. See the documentation for
 * settings and how to deploy this example on target.
 *
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with a PIO module can be used.
 *                       The example is written for AP7000 and STK1000.
 * - AppNote:            AVR32111 - Using the AVR32 PIO Controller
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support email: avr32@atmel.com
 *
 *****************************************************************************/

/* Copyright (C) 2006-2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include "pio.h"

#ifdef __GNUC__
  #include <avr32/io.h>
#elif __ICCAVR32__
  #include <avr32/ioap7000.h>
#else
  #error No known compiler used
#endif

#define INPUT_MASK 0x000000ff
#define OUTPUT_MASK 0x000000ff

#define SUCCESS 0


/*!
 * This function will read the dip switches on the STK1000 and put the status
 * out on the leds. To get it to work, you must connect the input and output
 * correctly on the STK1000. The input (switches) header marked J25 must be
 * connected to the header labeled J1 (PORTB[0..7]). While the output (leds)
 * header marked J15 must be connected to the header marked J3 (PORTC[0..7])
 * \return SUCCESS
 */

int main( void )
{
  volatile avr32_pio_t *piob = &AVR32_PIOB;
  volatile avr32_pio_t *pioc = &AVR32_PIOC;
  unsigned int input, output;

  pioc->per = OUTPUT_MASK;
  piob->per = INPUT_MASK;

  pioc->oer = OUTPUT_MASK;
  pioc->puer = OUTPUT_MASK;
  pioc->codr = OUTPUT_MASK;
  piob->codr = INPUT_MASK;

  while(1){
    input = ( piob->pdsr & INPUT_MASK); /* get input */
    output = ( pioc->pdsr & OUTPUT_MASK); /* get output */

    if ( output != input){
      pioc->codr = OUTPUT_MASK;  /* clear output */
      pioc->sodr = (~input & OUTPUT_MASK); /* set output from input */
    }

  }
}
