/****************************************************************************
 * Copyright (c) 2009 Atmel Corporation
 *
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the license which accompanies this code.
 * 
 * Contributors:
 *      Atmel Norway AS - Initial API and implementation  
 ****************************************************************************/
package com.atmel.avr32.sf.core.templates;

import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;

import org.eclipse.cdt.core.templateengine.TemplateCore;
import org.eclipse.cdt.core.templateengine.process.ProcessArgument;
import org.eclipse.cdt.core.templateengine.process.ProcessFailureException;
import org.eclipse.cdt.core.templateengine.process.ProcessRunner;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.osgi.framework.Bundle;

import com.atmel.avr32.sf.core.IFrameworkDataWorkingCopy;
import com.atmel.avr32.sf.core.SFCorePlugin;

/**
 * This template process will import one or more files from a given software
 * framework.
 * 
 * <pre>
 * &lt;process type=&quot;com.atmel.avr32.sf.core.importFiles&quot;&gt;
 *   &lt;simple name=&quot;projectName&quot; value=&quot;$(projectName)&quot; /&gt;
 *   &lt;simple name=&quot;frameworkId&quot; value=&quot;com.atmel.avr32.sf.core.ui.test.framework&quot; /&gt;
 *     &lt;complex-array name=&quot;files&quot;&gt;
 *       &lt;element&gt;
 *         &lt;simple name=&quot;source&quot; value=&quot;main.c&quot; /&gt;
 *         &lt;simple name=&quot;target&quot; value=&quot;src/main.c&quot; /&gt;
 *       &lt;/element&gt;
 *    &lt;/complex-array&gt;
 * &lt;/process&gt;
 * </pre>
 * 
 * @author Torkild Ulvøy Resheim
 * @since 2.3
 */
public class ImportFiles extends ProcessRunner {

	private static final String SOURCE = "source";

	@Override
	public void process(TemplateCore template, ProcessArgument[] args,
			String processId, IProgressMonitor monitor)
			throws ProcessFailureException {
		String projectName = args[0].getSimpleValue();
		String frameworkId = args[1].getSimpleValue();

		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(
				projectName);

		IConfigurationElement frameworkDeclaration = SFCorePlugin.getDefault()
				.getFramework(frameworkId).getConfiguration();

		ProcessArgument[][] files = args[2].getComplexArrayValue();
		for (int i = 0; i < files.length; i++) {
			ProcessArgument[] file = files[i];
			String source = file[0].getSimpleValue();
			String target = file[1].getSimpleValue();
			try {
				String root = frameworkDeclaration.getAttribute(SOURCE);
				Bundle bundle = Platform.getBundle(frameworkDeclaration
						.getContributor().getName());
				doImport(bundle, frameworkDeclaration, project, root + source,
						target, monitor);
			} catch (Exception e) {
				throw new ProcessFailureException("Could not import file.", e);
			}
		}
	}

	@SuppressWarnings("unchecked")
	private void doImport(Bundle bundle,
			IConfigurationElement frameworkDeclaration, IProject project,
			String source, String target, IProgressMonitor monitor)
			throws CoreException, IOException {
		if (SFCorePlugin.getDefault().isDebugging()) {
			System.out.println("Template import \"" + source + "\" to \""
					+ target + "\"");
		}
		if (source.endsWith("/")) {
			Enumeration<String> items = bundle.getEntryPaths(source);
			while (items.hasMoreElements()) {
				// Obtain the full path of the item to be imported
				String name = items.nextElement();
				IPath dst = new Path(name);
				// We're interested in the last segments.
				dst = dst.makeRelativeTo(new Path(source));
				// Make sure we ignore hidden stuff
				if (!dst.lastSegment().startsWith(".")) {
					doImport(bundle, frameworkDeclaration, project, name,
							target + dst.toPortableString(), monitor);
				}
			}
		} else {
			importOperation(project, frameworkDeclaration, source, target,
					monitor);
		}
	}

	private final static QualifiedName qname = new QualifiedName(
			"com.atmel.avr32.sf.core.imported", "imported"); //$NON-NLS-1$ //$NON-NLS-2$

	private void importOperation(IProject project,
			IConfigurationElement frameworkDeclaration, String sourceFile,
			String targetFile, IProgressMonitor monitor) throws CoreException,
			IOException {
		Bundle bundle = Platform.getBundle(frameworkDeclaration
				.getContributor().getName());
		Path p = new Path(sourceFile);
		final IFile to = project.getFile(targetFile);
		String[] segments = to.getProjectRelativePath().segments();
		// Copied into a sub folder, make sure that the folder structure
		// actually exists.
		if (segments.length > 1) {
			IFolder root = null;
			for (int x = 0; x < segments.length - 1; x++) {
				if (x == 0)
					root = project.getFolder(segments[x]);
				else
					root = root.getFolder(segments[x]);
				if (!root.exists()) {
					root.create(true, true, monitor);
					root.refreshLocal(IResource.DEPTH_INFINITE, monitor);
				}
			}
		}
		InputStream is = FileLocator.openStream(bundle, p, false);
		// Delete the old copy of the file
		if (to.exists()) {
			to.delete(true, monitor);
		}
		to.create(is, 0, monitor);
		is.close();
		to.setLocalTimeStamp(bundle.getLastModified());
		to.setPersistentProperty(qname, Boolean.toString(true));
		IFrameworkDataWorkingCopy data = SFCorePlugin
				.getDataWorkingCopy(project);
		String root = frameworkDeclaration.getAttribute(SOURCE);
		data.setIsFrameworkFile(to, frameworkDeclaration.getAttribute("id"),
				frameworkDeclaration.getAttribute("version"), (new Path(
						sourceFile)).makeRelativeTo(new Path(root))
						.toPortableString(), true);
		SFCorePlugin.storeData(data, monitor);
		to.refreshLocal(IResource.DEPTH_ZERO, monitor);
	}
}
