/****************************************************************************
 * Copyright (c) 2009 Atmel Corporation
 *
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the license which accompanies this code.
 * 
 * Contributors:
 *      Atmel Norway AS - Initial API and implementation  
 ****************************************************************************/
package com.atmel.avr32.sf.core.internal.filesystem;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.IFileSystem;
import org.eclipse.core.filesystem.provider.FileInfo;
import org.eclipse.core.filesystem.provider.FileStore;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;

import com.atmel.avr32.sf.core.SFCorePlugin;
import com.atmel.avr32.sf.core.SoftwareFramework;

/**
 * 
 * @since 2.3.0
 * @author Torkild Ulvøy Resheim
 */
public class FrameworkFileStore extends FileStore {

	private Bundle bundle = null;
	private SoftwareFramework framework = null;
	private String id = null;
	private String name;
	private FrameworkFileStore parent;
	private String path = null;
	private URI uri = null;

	/**
	 * 
	 * @param uri
	 */
	public FrameworkFileStore(URI uri) {
		this.uri = uri;
		id = uri.getPath();
		if (id.startsWith("/")) {
			id = id.substring(1); // remove root
		} else {
			id = uri.getPath();
		}
		framework = SFCorePlugin.getDefault().getFramework(id);
		if (framework != null) {
			bundle = Platform.getBundle(framework.getConfiguration()
					.getContributor().getName());
			if (uri.getQuery() != null) {
				path = uri.getQuery();
				name = path.substring(path.lastIndexOf('/') + 1);
			} else {
				path = "";
				name = framework.getName();
			}
		}
	}

	@Override
	public String[] childNames(int options, IProgressMonitor monitor)
			throws CoreException {
		try {
			File f = getFile();
			if (f.isDirectory()) {
				ArrayList<String> names = new ArrayList<String>();
				String[] list = f.list();
				if (list != null) {
					for (String string : list) {
						if (!string.startsWith(".")) {
							names.add(string);
						}
					}
					return names.toArray(new String[names.size()]);
				} else {
					System.err.println("Cannot find children of " + f);
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
		return EMPTY_STRING_ARRAY;
	}

	@Override
	public IFileInfo fetchInfo(int options, IProgressMonitor monitor)
			throws CoreException {
		FileInfo result = new FileInfo(getName());
		try {
			File f = getFile();
			if (f != null) {
				result.setExists(framework != null);
				result.setDirectory(f.isDirectory());
				result.setLength(f.length());
				result.setAttribute(EFS.ATTRIBUTE_HIDDEN, true);
			}
		} catch (IOException e) {
		}
		return result;
	}

	public Bundle getBundle() {
		return bundle;
	}

	@Override
	public IFileStore getChild(String name) {
		try {
			URI uri = new URI(SFCorePlugin.FILESYSTEM_SCHEME, null, null, -1,
					"/" + id, path + "/" + name, null);
			return new FrameworkFileStore(uri);
		} catch (URISyntaxException e) {
			e.printStackTrace();
		}
		return null;
	}

	private File getFile() throws IOException {
		if (framework == null) {
			return null;
		}
		IPath folder = Path.ROOT.addTrailingSeparator().append(
				framework.getSource()).append(path);
		URL bundlePath = FileLocator.find(bundle, folder, null);
		URL root = FileLocator.toFileURL(bundlePath);
		File f = new File(root.getPath());
		return f;
	}

	@Override
	public IFileSystem getFileSystem() {
		return new FrameworkFileSystem();
	}

	public SoftwareFramework getFramework() {
		return framework;
	}

	@Override
	public String getName() {
		return name;
	}

	@Override
	public IFileStore getParent() {
		return parent;
	}

	@Override
	public InputStream openInputStream(int options, IProgressMonitor monitor)
			throws CoreException {
		try {
			Path file = new Path(framework.getSource() + "/" + path);
			return FileLocator.openStream(bundle, file, false);
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}

	@Override
	public File toLocalFile(int options, IProgressMonitor monitor)
			throws CoreException {
		// if (options != EFS.CACHE)
		try {
			return getFile();
		} catch (IOException e) {
			e.printStackTrace();
			return null;
		}
		/*
		 * return super.toLocalFile(options, monitor);
		 */
	}

	@Override
	public URI toURI() {
		return uri;
	}
}
