package com.atmel.avr32.sf.core;

import java.io.InputStream;
import java.text.MessageFormat;
import java.util.ArrayList;

import org.eclipse.cdt.managedbuilder.core.BuildException;
import org.eclipse.cdt.managedbuilder.core.IConfiguration;
import org.eclipse.cdt.managedbuilder.core.IManagedBuildInfo;
import org.eclipse.cdt.managedbuilder.core.IOption;
import org.eclipse.cdt.managedbuilder.core.ITool;
import org.eclipse.cdt.managedbuilder.core.ManagedBuildManager;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.osgi.framework.Bundle;

/**
 * This type is capable of executing a software framework wizard without having
 * access to a user interface.
 * 
 * @author Torkild U. Resheim
 * @since 2.3
 */
public class WizardExecutor extends FrameworkConfigurationParser {
	private enum Overwrite {
		ALWAYS, NEVER, UNDETERMINED
	}

	private static final String BLANK = ""; //$NON-NLS-1$

	private static final String FRAMEWORK = "framework"; //$NON-NLS-1$
	/** QName indicating a framework file */
	private final static QualifiedName qname = new QualifiedName(
			"com.atmel.avr32.sf.core.imported", "imported"); //$NON-NLS-1$ //$NON-NLS-2$

	private static final String VERSION = "version"; //$NON-NLS-1$

	/** The framework data for the project */
	private IFrameworkDataWorkingCopy data;

	Overwrite fOverwrite = Overwrite.UNDETERMINED;

	private OperationFilterScanner scanner;

	/** The wizard configuration element */
	private IConfigurationElement wizardElement;

	/**
	 * Creates a new instance of the wizard executor using the
	 * {@link IConfigurationElement} containing the wizard declaration along
	 * with a working copy of the framework metadata for the project.
	 * 
	 * @param root
	 *            the wizard declaration root.
	 * @param data
	 *            the framework data for the project.
	 * @param project
	 */
	public WizardExecutor(IConfigurationElement root,
			IFrameworkDataWorkingCopy data) {
		this.wizardElement = root;
		this.data = data;
		scanner = new OperationFilterScanner(data, root);
	}

	/**
	 * 
	 * @param key
	 * @param value
	 * @param configuration
	 * @throws BuildException
	 * @since 2.0
	 */
	private void addDefinedSymbol(String key, String value,
			IConfiguration configuration) throws BuildException {
		ITool[] tools = configuration.getTools();
		for (ITool tool : tools) {
			for (IOption option : tool.getOptions()) {
				if (option.getId() != null && option.getId().startsWith(key)) {
					String[] values = option.getDefinedSymbols();
					setMBSListValue(value, configuration, tool, option, values);
				}
			} // for
		} // for
	}

	/**
	 * @param value
	 * @param configuration
	 * @since 2.0
	 */
	private void addIncludeFolder(String value, IConfiguration configuration)
			throws BuildException {
		ITool[] tools = configuration.getTools();
		for (ITool tool : tools) {
			for (IOption option : tool.getOptions()) {
				if (option.getId() != null
						&& (option.getId().startsWith(C_INCLUDE_PATH_ID)
								|| option.getId().startsWith(
										ASM_INCLUDE_PATH_ID) || option.getId()
								.startsWith(ASM2_INCLUDE_PATH_ID))) {
					String[] values = option.getIncludePaths();
					setMBSListValue(value, configuration, tool, option, values);
				}
			} // for
		} // for
	}

	/**
	 * 
	 * @param value
	 * @param configuration
	 * @throws BuildException
	 * @since 2.0
	 */
	private void addLibrary(String value, IConfiguration configuration)
			throws BuildException {
		ITool[] tools = configuration.getTools();
		for (ITool tool : tools) {
			for (IOption option : tool.getOptions()) {
				if (option.getId() != null
						&& option.getId().startsWith(LIBRARY_ID)) {
					String[] values = option.getLibraries();
					setMBSListValue(value, configuration, tool, option, values);
				}
			} // for
		} // for
	}

	/**
	 * 
	 * @param frameworkDeclaration
	 * @param operation
	 * @param monitor
	 * @return
	 * @since 2.1
	 */
	private IStatus addLibraryOperation(
			IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor) {
		try {
			IConfiguration[] configs = getConfigurations();
			for (IConfiguration configuration : configs) {
				addLibrary(operation.getAttribute(ATTR_NAME), configuration);
			} // for
			ManagedBuildManager.saveBuildInfo(data.getProject(), true);
		} catch (Exception e) {
			return error(operation, Messages.WizardExecutor_CouldNotAddLibrary,
					e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * 
	 * @param value
	 * @param configuration
	 * @throws BuildException
	 * @since 2.0
	 */
	private void addLibraryPathFolder(String value, IConfiguration configuration)
			throws BuildException {
		ITool[] tools = configuration.getTools();
		for (ITool tool : tools) {
			for (IOption option : tool.getOptions()) {
				if (option.getId() != null
						&& option.getId().startsWith(LIBARY_PATH_ID)) {
					String[] values = option.getBasicStringListValue();
					setMBSListValue(value, configuration, tool, option, values);
				}
			} // for
		} // for
	}

	/**
	 * 
	 * @param frameworkDeclaration
	 * @param operation
	 * @param monitor
	 * @return
	 * @since 2.1
	 */
	private IStatus addLibraryPathOperation(
			IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor) {
		try {
			for (IConfiguration configuration : getConfigurations()) {
				addLibraryPathFolder(operation.getAttribute(ATTR_PATH),
						configuration);
			} // for
			ManagedBuildManager.saveBuildInfo(data.getProject(), true);
		} catch (Exception e) {
			return error(operation,
					Messages.WizardExecutor_CouldNotAddLibraryPath, e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * 
	 * @param frameworkDeclaration
	 *            the software framework declaration
	 * @param operation
	 *            the current operation element
	 * @return the operation status
	 * @since 2.0
	 */
	private IStatus defineOperation(IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor) {
		try {
			IConfiguration[] configs = getConfigurations();
			for (IConfiguration configuration : configs) {
				addDefinedSymbol(SYMBOL_ID,
						operation.getAttribute(ATTR_SYMBOL), configuration);
			} // for
			ManagedBuildManager.saveBuildInfo(data.getProject(), true);
		} catch (Exception e) {
			return error(operation, Messages.WizardExecutor_DefineSymbolError,
					e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * Determines if an existing file can be overwritten when executing the
	 * wizard. Subclasses may re-implement.
	 * 
	 * @return whether or not a file may be overwritten.
	 */
	protected int canOverwrite(IResource resource) {
		return 0;
	}

	/**
	 * Determines if an existing file can be overwritten when executing the
	 * wizard. Subclasses may re-implement.
	 * 
	 * @return whether or not a file may be overwritten.
	 */
	protected boolean canDelete(IResource resource) {
		return true;
	}

	/**
	 * Removes the file or folder specified in the <i>path</i> attribute of the
	 * operation.
	 * 
	 * @param operation
	 *            The current operation element
	 * @return The operation status
	 * @since 2.0
	 */
	private IStatus deleteCommand(IConfigurationElement operation,
			IProgressMonitor monitor) {
		IResource r = data.getProject().findMember(
				operation.getAttribute(ATTR_PATH));
		try {
			if (r != null) {
				// Handle that the file may not be registered as a framework
				// file
				if (!data.isFrameworkFile(r) && r instanceof IFile) {
					if (!canDelete(r)) {
						return Status.OK_STATUS;
					}
				}
			}
			r.delete(true, monitor);
			data.setIsFrameworkFile(r, null, null, null, false);
			// Delete parent folders if they are empty
			r = r.getParent();
			while (!(r instanceof IProject)) {
				IFolder folder = (IFolder) r;
				if (folder.members().length == 0) {
					folder.delete(true, monitor);
				}
				r = r.getParent();
			}
		} catch (Exception e) {
			return error(operation, Messages.WizardExecutor_CouldNotRemoveFile,
					e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * Uses the current operation along with a message and an optional exception
	 * to format a error message.
	 * 
	 * @param operation
	 *            The current operation.
	 * @param message
	 *            A message.
	 * @param e
	 *            An exception, may be <b>null</b>.
	 * @return
	 * @since 2.0
	 */
	private IStatus error(IConfigurationElement operation, String message,
			Exception e) {
		if (e != null) {
			return new Status(
					IStatus.ERROR,
					SFCorePlugin.PLUGIN_ID,
					MessageFormat
							.format(
									Messages.WizardExecutor_ProblemMessage,
									new Object[] {
											operation.getName(),
											getOptionAttribute(operation, LABEL),
											getOptionAttribute(operation, ID),
											message }), e);
		} else {
			return new Status(
					IStatus.ERROR,
					SFCorePlugin.PLUGIN_ID,
					MessageFormat
							.format(
									Messages.WizardExecutor_ProblemMessage,
									new Object[] {
											operation.getName(),
											getOptionAttribute(operation, LABEL),
											getOptionAttribute(operation, ID),
											message }));

		}
	}

	/**
	 * Scans through the wizard declaration specified for this instance and
	 * executes the operations that matches the supplied values.
	 * 
	 * @param monitor
	 *            A progress monitor.
	 * @return The execution status.
	 * @since 2.0
	 */
	public IStatus execute(IProgressMonitor monitor) {
		IConfigurationElement declaration = getFrameworkDeclaration();
		MultiStatus ms = new MultiStatus(SFCorePlugin.PLUGIN_ID, 0,
				Messages.WizardExecutor_StatusTitle, null);
		try {
			ArrayList<IConfigurationElement> operations = scanner
					.getFilteredCommands(OperationFilterScanner.NONE);
			monitor.beginTask(Messages.WizardExecutor_TaskName, operations
					.size());
			for (IConfigurationElement element : operations) {
				execute(monitor, declaration, ms, element);
				monitor.worked(1);
			}
		} catch (CoreException e) {
			return new Status(IStatus.ERROR, SFCorePlugin.PLUGIN_ID,
					"Error when executing operations", e); //$NON-NLS-1$
		} catch (BuildException e) {
			return new Status(IStatus.ERROR, SFCorePlugin.PLUGIN_ID,
					"Error when executing operations", e); //$NON-NLS-1$
		}
		if (ms.isOK()) {
			ms.add(SFCorePlugin.storeData(data, monitor));
		}
		return ms;
	}

	/**
	 * 
	 * @param monitor
	 * @param declaration
	 *            the framework declaration
	 * @param ms
	 * @param element
	 * @throws CoreException
	 */
	private void execute(IProgressMonitor monitor,
			IConfigurationElement declaration, MultiStatus ms,
			IConfigurationElement element) throws CoreException {
		Operation op = getOperation(element);
		switch (op) {
		case IMPORT:
			ms.add(importCommand(declaration, element, monitor));
			break;
		case DELETE:
			ms.add(deleteCommand(element, monitor));
			break;
		case INCLUDE:
			ms.add(includeCommand(declaration, element, monitor));
			break;
		case ADD_LIBRARY_PATH:
			ms.add(addLibraryPathOperation(declaration, element, monitor));
			break;
		case REMOVE_LIBRARY_PATH:
			ms.add(removeLibraryPathCommand(declaration, element, monitor));
			break;
		case ADD_LIBRARY:
			ms.add(addLibraryOperation(declaration, element, monitor));
			break;
		case REMOVE_LIBRARY:
			ms.add(removeLibraryCommand(declaration, element, monitor));
			break;
		case RM_INCLUDE:
			ms.add(removeIncludeCommand(declaration, element, monitor));
			break;
		case DEFINE:
			ms.add(defineOperation(declaration, element, monitor));
			break;
		case RM_DEFINE:
			ms.add(removeDefineCommand(declaration, element, monitor));
			break;
		}
	}

	/**
	 * 
	 * @return
	 * @since 2.0
	 */
	private IConfiguration[] getConfigurations() {
		IManagedBuildInfo info = ManagedBuildManager.getBuildInfo(data
				.getProject());
		IConfiguration[] configs = info.getManagedProject().getConfigurations();
		return configs;
	}

	/**
	 * Finds and returns the framework declaration that the wizard declaration
	 * is referring to.
	 * 
	 * @return The framework declaration
	 * @since 2.0
	 */
	private IConfigurationElement getFrameworkDeclaration() {
		final String frameworkId = wizardElement.getAttribute(FRAMEWORK);
		return SFCorePlugin.getDefault().getFramework(frameworkId)
				.getConfiguration();
	}

	/**
	 * 
	 * @param operation
	 * @param attribute
	 * @return
	 * @since 2.0
	 */
	private String getOptionAttribute(IConfigurationElement operation,
			String attribute) {
		String value = ((IConfigurationElement) (((IConfigurationElement) operation
				.getParent()).getParent())).getAttribute(attribute);
		return (value == null ? "<unknown>" : value); //$NON-NLS-1$
	}

	/**
	 * If a workspace copy of the file already exists the MD5 checksum is
	 * calculated for both the workspace and the framework file. If there is a
	 * mismatch the user is will be asked whether to overwrite or not.
	 * 
	 * @param frameworkDeclaration
	 *            The framework declaration.
	 * @param operation
	 *            The current operation.
	 * @return The operation status.
	 * @throws CoreException
	 * @since 2.0
	 */
	private IStatus importCommand(IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor)
			throws CoreException {
		Bundle bundle = Platform.getBundle(frameworkDeclaration
				.getContributor().getName());
		String source = frameworkDeclaration.getAttribute(ATTR_SOURCE);
		Path p = new Path(source + IPath.SEPARATOR
				+ operation.getAttribute(ATTR_PATH));
		final IFile to = data.getProject().getFile(
				operation.getAttribute(ATTR_DEST));
		// Handle that the file may already exist.
		if (to.exists()) {
			// If the file is the same we don't need to do anything
			if (SFCorePlugin.getDefault().checksum(to).equals(
					SFCorePlugin.getDefault().checksum(bundle, p))) {
				// It's the same file. No need to overwrite.
				return Status.OK_STATUS;
			} else
				switch (fOverwrite) {
				case UNDETERMINED:
					// Don't overwrite this time.
					switch (canOverwrite(to)) {
					case 0: // Yes
						break;
					case 1: // No
						return Status.OK_STATUS;
					case 2: // Yes to all
						fOverwrite = Overwrite.ALWAYS;
						break;
					case 3: // No to all
						fOverwrite = Overwrite.NEVER;
						return Status.OK_STATUS;
					default:
						break;
					}
					break;

				case NEVER:
					return Status.OK_STATUS;
				default:
					// Simply continue
				}
		}
		String[] segments = to.getProjectRelativePath().segments();
		// Copied into a sub folder, make sure that the folder structure
		// actually exists.
		try {
			if (segments.length > 1) {
				IFolder root = null;
				for (int x = 0; x < segments.length - 1; x++) {
					if (x == 0)
						root = data.getProject().getFolder(segments[x]);
					else
						root = root.getFolder(segments[x]);
					if (!root.exists()) {
						root.create(true, true, monitor);
						root.refreshLocal(IResource.DEPTH_INFINITE, monitor);
					}
				}
			}
		} catch (Exception e) {
			return error(operation,
					Messages.WizardExecutor_CouldNotCreateFolders, e);
		}
		try {
			InputStream is = FileLocator.openStream(bundle, p, false);
			// Delete the old copy of the file
			if (to.exists()) {
				to.delete(true, monitor);
			}
			to.create(is, 0, monitor);
			is.close();
			to.setLocalTimeStamp(bundle.getLastModified());
			to.setPersistentProperty(qname, Boolean.toString(true));
			data.setIsFrameworkFile(to, frameworkDeclaration.getAttribute(ID),
					frameworkDeclaration.getAttribute(VERSION), operation
							.getAttribute(ATTR_PATH), true);
			to.refreshLocal(IResource.DEPTH_ZERO, monitor);
		} catch (Exception e) {
			return error(operation, MessageFormat.format(
					Messages.WizardExecutor_CouldNotCopyFile,
					new Object[] { p }), e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * 
	 * @param frameworkDeclaration
	 *            The software framework declaration
	 * @param operation
	 *            The current operation element
	 * @return The operation status
	 * @since 2.0
	 */
	private IStatus includeCommand(IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor) {
		try {
			for (IConfiguration configuration : getConfigurations()) {
				addIncludeFolder(operation.getAttribute(ATTR_PATH),
						configuration);
			} // for
			ManagedBuildManager.saveBuildInfo(data.getProject(), true);
		} catch (Exception e) {
			return error(operation, Messages.WizardExecutor_CouldNoteResolve, e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * 
	 * @param key
	 * @param value
	 * @param configuration
	 * @throws BuildException
	 * @since 2.0
	 */
	private void removeDefinedSymbol(String key, String value,
			IConfiguration configuration) throws BuildException {
		ITool[] tools = configuration.getTools();
		for (ITool tool : tools) {
			for (IOption option : tool.getOptions()) {
				if (option.getId() != null && option.getId().startsWith(key)) {
					String[] values = option.getDefinedSymbols();
					ArrayList<String> newValues = new ArrayList<String>();
					unsetMBSListValue(value, configuration, tool, option,
							values, newValues);
				}
			} // for
		} // for
	}

	/**
	 * 
	 * @param frameworkDeclaration
	 *            The software framework declaration
	 * @param operation
	 *            The current operation element
	 * @return The operation status
	 * @since 2.0
	 */
	private IStatus removeDefineCommand(
			IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor) {
		try {
			for (IConfiguration configuration : getConfigurations()) {
				removeDefinedSymbol(SYMBOL_ID, operation
						.getAttribute(ATTR_SYMBOL), configuration);
			} // for
			ManagedBuildManager.saveBuildInfo(data.getProject(), true);
		} catch (Exception e) {
			return error(operation, Messages.WizardExecutor_RemoveSymbolError,
					e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * @param value
	 * @param configuration
	 * @since 2.0
	 */
	private void removeIncludeFolder(String value, IConfiguration configuration)
			throws BuildException {
		ITool[] tools = configuration.getTools();
		for (ITool tool : tools) {
			for (IOption option : tool.getOptions()) {
				if (option.getId() != null
						&& (option.getId().startsWith(C_INCLUDE_PATH_ID))
						|| option.getId().startsWith(ASM_INCLUDE_PATH_ID)
						|| option.getId().startsWith(ASM2_INCLUDE_PATH_ID)) {
					String[] values = option.getIncludePaths();
					ArrayList<String> newValues = new ArrayList<String>();
					unsetMBSListValue(value, configuration, tool, option,
							values, newValues);
				}
			} // for
		} // for
	}

	/**
	 * 
	 * @param frameworkDeclaration
	 *            The software framework declaration
	 * @param operation
	 *            The current operation element
	 * @return The operation status
	 * @since 2.0
	 */
	private IStatus removeIncludeCommand(
			IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor) {
		try {
			for (IConfiguration configuration : getConfigurations()) {
				removeIncludeFolder(operation.getAttribute(ATTR_PATH),
						configuration);
			} // for
			ManagedBuildManager.saveBuildInfo(data.getProject(), true);
		} catch (Exception e) {
			return error(operation,
					Messages.WizardExecutor_CouldNotRemoveInclude, e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * 
	 * @param value
	 * @param configuration
	 * @throws BuildException
	 * @since 2.1
	 */
	private void removeLibrary(String value, IConfiguration configuration)
			throws BuildException {
		ITool[] tools = configuration.getTools();
		for (ITool tool : tools) {
			for (IOption option : tool.getOptions()) {
				if (option.getId() != null
						&& option.getId().startsWith(LIBRARY_ID)) {
					String[] values = option.getLibraries();
					ArrayList<String> newValues = new ArrayList<String>();
					unsetMBSListValue(value, configuration, tool, option,
							values, newValues);
				}
			} // for
		} // for
	}

	/**
	 * 
	 * @param frameworkDeclaration
	 * @param operation
	 * @param monitor
	 * @return
	 * @since 2.1
	 */
	private IStatus removeLibraryCommand(
			IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor) {
		try {
			for (IConfiguration configuration : getConfigurations()) {
				removeLibrary(operation.getAttribute(ATTR_NAME), configuration);
			} // for
			ManagedBuildManager.saveBuildInfo(data.getProject(), true);
		} catch (Exception e) {
			return error(operation,
					Messages.WizardExecutor_CouldNotRemoveLibrary, e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * Removes the specified library search path from the configuration.
	 * 
	 * @param value
	 *            the path to remove
	 * @param configuration
	 *            the configuration to move the path from
	 * @throws BuildException
	 * @since 2.1
	 */
	private void removeLibraryPath(String value, IConfiguration configuration)
			throws BuildException {
		ITool[] tools = configuration.getTools();
		for (ITool tool : tools) {
			for (IOption option : tool.getOptions()) {
				if (option.getId() != null
						&& option.getId().startsWith(LIBARY_PATH_ID)) {
					String[] values = option.getBasicStringListValue();
					ArrayList<String> newValues = new ArrayList<String>();
					unsetMBSListValue(value, configuration, tool, option,
							values, newValues);
				}
			} // for
		} // for
	}

	/**
	 * 
	 * @param frameworkDeclaration
	 * @param operation
	 * @param monitor
	 * @return
	 * @since 2.1
	 */
	private IStatus removeLibraryPathCommand(
			IConfigurationElement frameworkDeclaration,
			IConfigurationElement operation, IProgressMonitor monitor) {
		try {
			for (IConfiguration configuration : getConfigurations()) {
				removeLibraryPath(operation.getAttribute(ATTR_PATH),
						configuration);
			} // for
			ManagedBuildManager.saveBuildInfo(data.getProject(), true);
		} catch (Exception e) {
			return error(operation,
					Messages.WizardExecutor_CouldNotRemoveLibraryPath, e);
		}
		return Status.OK_STATUS;
	}

	/**
	 * 
	 * @param value
	 * @param configuration
	 * @param tool
	 * @param option
	 * @param values
	 * @throws BuildException
	 * @since 2.0
	 */
	private void setMBSListValue(String value, IConfiguration configuration,
			ITool tool, IOption option, String[] values) throws BuildException {
		// We're not setting the same value twice
		for (String string : values) {
			if (string.equals(value))
				return;
		}
		String[] newValues = new String[values.length + 1];
		System.arraycopy(values, 0, newValues, 1, values.length);
		newValues[0] = value;
		configuration.setOption(tool, option, newValues);
	}

	/**
	 * 
	 * @param value
	 * @param configuration
	 * @param tool
	 * @param option
	 * @param values
	 * @param newValues
	 * @throws BuildException
	 * @since 2.1
	 */
	private void unsetMBSListValue(String value, IConfiguration configuration,
			ITool tool, IOption option, String[] values,
			ArrayList<String> newValues) throws BuildException {
		// For some reason includes added by the compiler itself
		// show up as blank strings in this list. We must be careful
		// not to add the blank strings back as this will create
		// invalid items.
		for (String string : values) {
			if (!string.equals(value) && !string.equals(BLANK)) {
				newValues.add(string);
			}
		}
		configuration.setOption(tool, option, newValues
				.toArray(new String[newValues.size()]));
	}

}
