/****************************************************************************
 * Copyright (c) 2009 Atmel Corporation
 *
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the license which accompanies this code.
 * 
 * Contributors:
 *      Atmel Norway AS - Initial API and implementation  
 ****************************************************************************/
package com.atmel.avr32.sf.core;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.text.Collator;
import java.util.ArrayList;
import java.util.Collections;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;

import com.atmel.avr32.managedbuilder.ManagedbuildPlugin;

/**
 * This type represents a software framework declaration and also has utility
 * methods for accessing the framework implementation.
 * 
 * @author Torkild Ulvøy Resheim
 * @since 2.3.0
 */
public class SoftwareFramework {

	private static final String PARTLIST_COMMENT = "#"; //$NON-NLS-1$
	private static final String PARTS = "partlist"; //$NON-NLS-1$

	private String name;
	private String id;
	private String source;
	private String version;
	private boolean visible;

	/**
	 * 
	 * @return <code>true</code> if the framework is visible in the UI.
	 * @since 2.4
	 */
	public boolean showInNavigator() {
		return visible;
	}

	private IConfigurationElement configuration;

	public IConfigurationElement getConfiguration() {
		return configuration;
	}

	public String getVersion() {
		return version;
	}

	public SoftwareFramework(IConfigurationElement config) {
		name = config.getAttribute("name");
		id = config.getAttribute("id");
		version = config.getAttribute("version");
		configuration = config;
		source = config.getAttribute("source");
		if (config.getAttribute("showInNavigator") != null) {
			visible = Boolean.parseBoolean(config
					.getAttribute("showInNavigator"));
		}
	}

	public String getSource() {
		return source;
	}

	public String getName() {
		return name;
	}

	public String getId() {
		return id;
	}

	public boolean isCompatible(IProject project) {
		String projectMCU = ManagedbuildPlugin.getProjectMCU(project);
		String[] list = getSupportedParts();
		if (list.length == 0) {
			return true;
		}
		for (String string : list) {
			if (string.equals(projectMCU)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Returns a list over supported parts for the tool.
	 * 
	 * @return Part ID list
	 */
	public String[] getSupportedParts() {
		ArrayList<String> templist = new ArrayList<String>();
		Assert.isNotNull(configuration,
				"The configuration element must not be null"); //$NON-NLS-1$
		String pluginID = configuration.getContributor().getName();
		Path path = new Path(configuration.getAttribute(PARTS));
		try {
			InputStream st = FileLocator.openStream(Platform
					.getBundle(pluginID), path, false);
			BufferedReader br = new BufferedReader(new InputStreamReader(st));
			String line;
			while ((line = br.readLine()) != null) {
				if (!line.startsWith(PARTLIST_COMMENT) && line.length() > 0)
					templist.add(line.trim());
			}
			br.close();
			st.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		Collections.sort(templist, Collator.getInstance());
		return templist.toArray(new String[templist.size()]);
	}

	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append(name);
		sb.append(" (");
		sb.append(id);
		sb.append(')');
		return sb.toString();
	}
}
