/****************************************************************************
 * Copyright (c) 2008 Atmel Corporation
 *
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the license which accompanies this code.
 * 
 * Contributors:
 *		Atmel Norway AS - Initial API and implementation  
 ****************************************************************************/
package com.atmel.avr32.sf.core;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.MessageDigest;
import java.text.MessageFormat;
import java.util.HashMap;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.ISaveContext;
import org.eclipse.core.resources.ISaveParticipant;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

import com.atmel.avr32.AVR32CorePlugin;
import com.atmel.avr32.sf.core.IFrameworkConfigurationParser.Operation;

/**
 * Core software framework plug-in. This type manages the framework metadata.
 * 
 * @author Torkild Ulvøy Resheim
 * @since 2.0
 */
public class SFCorePlugin extends Plugin implements IResourceChangeListener {

	public static final String FILESYSTEM_SCHEME = "framework";

	private static final String ATTR_SOURCE = "source"; //$NON-NLS-1$

	private static final String ATTR_VERSION = "version"; //$NON-NLS-1$

	public static final String WIZARD_EXTENSION_ID = "com.atmel.avr32.sf.core.wizard"; //$NON-NLS-1$

	/**
	 * Returns the software framework wizard with the specified identifier
	 * 
	 * @param id
	 * @since 2.3
	 */
	public IConfigurationElement getWizardDescription(String id) {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint extensionPoint = registry
				.getExtensionPoint(SFCorePlugin.WIZARD_EXTENSION_ID);
		IConfigurationElement wizards[] = extensionPoint
				.getConfigurationElements();
		for (IConfigurationElement wizard : wizards) {
			if (wizard.getAttribute(ID).equals(id)) {
				return wizard;
			}
		}
		return null;
	}

	/**
	 * This type is used to determine if a software framework file in the
	 * workspace has been changed and will update the framework metadata
	 * accordingly. Note that this visitor will <b>only</b> handle files that
	 * are associated with a software framework. All other files will be
	 * ignored.
	 * 
	 * @author Torkild Ulvøy Resheim
	 * @since 2.0
	 */
	class FrameworkDataUpdater implements IResourceDeltaVisitor,
			IResourceVisitor {

		IProject closingProject = null;

		/**
		 * Remove the file from the framework metadata if the file is a
		 * framework file.
		 * 
		 * @param resource
		 *            The file to remove
		 * @since 2.0
		 */
		private void removeFile(IResource resource) {
			IFrameworkData data = getData(resource.getProject());
			if (data != null && data.isFrameworkFile(resource)) {
				((FrameworkProjectData) data).removeFile(resource);
			}
		}

		/**
		 * Update the framework related metadata of the resource (file) if it is
		 * a framework related file.
		 * 
		 * @param resource
		 *            the file to update
		 * @throws CoreException
		 */
		private void updateMetadata(IResource resource) throws CoreException {
			IFrameworkData data = getData(resource.getProject());
			if (data == null)
				return;
			if (data.isFrameworkFile(resource)) {
				IFrameworkFileDescriptor fd = ((FrameworkProjectData) data)
						.getFileDescriptor(resource);
				IConfigurationElement declaration = getFrameworkDeclaration(fd
						.getFrameworkId());
				if (declaration != null) {
					String source = declaration.getAttribute(ATTR_SOURCE);
					Path path = new Path(source + IPath.SEPARATOR
							+ fd.getSource());
					Bundle bundle = Platform
							.getBundle(declaration.getDeclaringExtension()
									.getContributor().getName());
					// Make sure the version is updated
					((FrameworkFileDescriptor) fd).setVersion(declaration
							.getAttribute(ATTR_VERSION));

					if (!checksum((IFile) resource).equals(
							checksum(bundle, path))) {
						// The file is different from the reference
						((FrameworkFileDescriptor) fd).setChanged(true);
					} else {
						// The file is equal to the reference
						// So we must clear the changed flag
						((FrameworkFileDescriptor) fd).setChanged(false);
					}
					fireDataChangedEvent(resource);
					// In this case the "framework" attribute of the entry is
					// probably wrong but we want to clear the file anyway.
				} else {
					((FrameworkProjectData) data).removeFile(resource);
					fireDataChangedEvent(resource);
				}
			}
		}

		public boolean visit(IResourceDelta delta) throws CoreException {
			IResource resource = delta.getResource();
			if ((resource instanceof IFile)) {
				if (!resource.getName().equals(
						FrameworkProjectData.FRAMEWORK_DATA_FILE)) {
					switch (delta.getKind()) {
					case IResourceDelta.REMOVED:
						if (closingProject == null)
							removeFile(resource);
						break;
					case IResourceDelta.CHANGED:
						updateMetadata(resource);
						break;
					}
				}
			} else if ((resource instanceof IProject)) {
				closingProject = null;
				switch (delta.getKind()) {
				case IResourceDelta.CHANGED:
					if ((delta.getFlags() & IResourceDelta.OPEN) == IResourceDelta.OPEN) {
						closingProject = delta.getResource().getProject();
					}
					break;
				case IResourceDelta.REMOVED:
					dataSets.remove(delta.getResource().getProject());
					break;
				}
			}
			return true;
		}

		public boolean visit(IResource resource) throws CoreException {
			updateMetadata(resource);
			return false;
		}
	}

	/**
	 * We use this type to store the framework metadata when the workbench
	 * requests that we do so.
	 * 
	 * @author Torkild Ulvøy Resheim
	 */
	class SaveParticipant implements ISaveParticipant {

		public void doneSaving(ISaveContext context) {
		}

		public void prepareToSave(ISaveContext context) throws CoreException {
		}

		public void rollback(ISaveContext context) {
			// not supported
		}

		public void saving(ISaveContext context) throws CoreException {
			for (IFrameworkData data : dataSets.values()) {
				((FrameworkProjectData) data)
						.storeData(new NullProgressMonitor());
			}
		}

	}

	/** List of projects and data sets */
	private static HashMap<IProject, IFrameworkData> dataSets;

	private static ListenerList fListeners;

	public static final String FRAMEWORK_EXTENSION_ID = "com.atmel.avr32.sf.core.framework"; //$NON-NLS-1$

	private static final String HEX_PREFIX = "0x"; //$NON-NLS-1$

	private static final String ID = "id"; //$NON-NLS-1$

	private static final String MESSAGE_DIGEST_TYPE = "MD5"; //$NON-NLS-1$

	/** The shared instance */
	private static SFCorePlugin plugin;

	/** The plug-in identifier */
	public final static String PLUGIN_ID = "com.atmel.avr32.sf.core"; //$NON-NLS-1$

	private static FrameworkDataUpdater updater;

	/**
	 * Returns the framework data for the specified project. If the project is
	 * not a AVR32 project, <b>null</b> is returned.
	 * 
	 * @param project
	 *            The project to retrieve the data for
	 * @return The framework data or <b>null</b>
	 * @since 2.0
	 */
	public static IFrameworkData getData(IProject project) {
		try {
			if (project.isAccessible()
					&& project.hasNature(AVR32CorePlugin.NATURE_ID)) {
				IFrameworkData data = dataSets.get(project);
				if (data == null) {
					data = new FrameworkProjectData(project);
					dataSets.put(project, data);
				}
				return data;
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * Returns a working copy of the framework data for the specified project.
	 * <i>storeData(IFrameworkData data, IProgressMonitor monitor)</i> must be
	 * called in order to commit changes and persist these.
	 * 
	 * @param project
	 *            The project to retrieve the data for
	 * @return The framework data
	 * @since 2.0
	 */
	public static IFrameworkDataWorkingCopy getDataWorkingCopy(IProject project) {
		return new FrameworkDataWorkingCopy(getData(project));
	}

	/**
	 * Returns the default singleton instance of this plug-in.
	 * 
	 * @return The plug-in instance
	 * @since 2.0
	 */
	public static SFCorePlugin getDefault() {
		if (plugin == null) {
			plugin = new SFCorePlugin();
		}
		return plugin;
	}

	/**
	 * Updates the real framework metadata with information from the working
	 * copy and saves the data. This method should only be called internally;
	 * from the WizardExecutor or other functions that successfully performed an
	 * operation.
	 * 
	 * @param data
	 *            The framework data working copy
	 * @param monitor
	 *            A progress monitor
	 * @return The status
	 * @since 2.0
	 */
	public static IStatus storeData(IFrameworkDataWorkingCopy data,
			IProgressMonitor monitor) {
		FrameworkProjectData real = ((FrameworkProjectData) getData(data
				.getProject()));
		data.updateData(real);
		return real.storeData(monitor);
	}

	public SFCorePlugin() {
		super();
		plugin = this;
		updater = new FrameworkDataUpdater();
		fListeners = new ListenerList();
	}

	public void addListener(IFrameworkDataChangedListener listener) {
		fListeners.add(listener);
	}

	/**
	 * Computes a MD5 checksum for the file at <i>path</i> in the specified
	 * plug-in bundle.
	 * 
	 * @param bundle
	 *            The bundle the file is in
	 * @param path
	 *            The path to the file
	 * @return The MD5 checksum
	 * @throws CoreException
	 * @since 2.0
	 */
	public String checksum(Bundle bundle, Path path) throws CoreException {
		try {
			InputStream fin = FileLocator.openStream(bundle, path, false);
			java.security.MessageDigest md5er = MessageDigest
					.getInstance(MESSAGE_DIGEST_TYPE);
			byte[] buffer = new byte[1024];
			int read;
			do {
				read = fin.read(buffer);
				if (read > 0)
					md5er.update(buffer, 0, read);
			} while (read != -1);
			fin.close();
			byte[] digest = md5er.digest();
			if (digest == null)
				return null;
			String strDigest = HEX_PREFIX;
			for (int i = 0; i < digest.length; i++) {
				strDigest += Integer.toString((digest[i] & 0xff) + 0x100, 16)
						.substring(1).toUpperCase();
			}
			return strDigest;
		} catch (Exception e) {
			throw new CoreException(
					new Status(
							IStatus.ERROR,
							PLUGIN_ID,
							MessageFormat
									.format(
											"Could not locate framework file \"{0}\" in bundle \"{1}\"",
											new Object[] {
													path.toPortableString(),
													bundle.getSymbolicName() }),
							e));
		}
	}

	/**
	 * Tests if the framework source relative path exists in the framework with
	 * the given identifier.
	 * 
	 * @param id
	 *            the framework identifier
	 * @param path
	 *            the file path
	 * @return <b>true</b> if the path exists
	 */
	public boolean isFrameworkFile(String id, IPath path) {
		if (path.isEmpty()) {
			return false;
		}
		IConfigurationElement declaration = getFrameworkDeclaration(id);
		if (declaration == null)
			return false;
		String source = declaration.getAttribute(ATTR_SOURCE);
		IPath sourcePath = new Path(source).append(path);
		String bundleId = declaration.getDeclaringExtension().getContributor()
				.getName();
		Bundle bundle = Platform.getBundle(bundleId);
		if (bundle == null)
			return false;
		URL url = FileLocator.find(bundle, sourcePath, null);
		return (url != null);
	}

	/**
	 * Attempts to determine if the source file of given file is a configuration
	 * file. This can happen in two ways:
	 * <ul>
	 * <li>The source file is explicitly specified as configuration file.</li>
	 * <li>The source file belongs to a folder that is specified as a
	 * configuration folder</li>
	 * </ul>
	 * 
	 * @param file
	 *            the <i>workspace</i> file to test
	 * @return <b>true</b> if a configuration file
	 * @since 2.1
	 */
	public boolean isFrameworkConfigurationFile(IFile file) {
		IFrameworkFileDescriptor fd = getData(file.getProject())
				.getFileDescriptor(file);
		String frameworkId = fd.getFrameworkId();
		IPath sourcePath = new Path(fd.getSource()).removeLastSegments(1);
		IConfigurationElement config = getFrameworkDeclaration(frameworkId);
		IConfigurationElement[] layouts = config.getChildren("layout"); //$NON-NLS-1$
		if (layouts.length > 0) {
			IConfigurationElement[] files = layouts[0].getChildren("file"); //$NON-NLS-1$
			for (IConfigurationElement f : files) {
				if (f.getAttribute("type").equals("configuration")) { //$NON-NLS-1$ //$NON-NLS-2$
					if (f.getAttribute("path").equals(fd.getSource())) { //$NON-NLS-1$
						return true;
					}
				}
			}
			IConfigurationElement[] directories = layouts[0]
					.getChildren("directory"); //$NON-NLS-1$
			for (IConfigurationElement dir : directories) {
				if (dir.getAttribute("type").equals("configuration")) { //$NON-NLS-1$ //$NON-NLS-2$
					if (dir.getAttribute("path").equals(sourcePath.toString())) { //$NON-NLS-1$
						return true;
					}
				}
			}
		}

		return false;
	}

	/**
	 * Calculates the MD5 checksum for a file.
	 * 
	 * @param file
	 *            The file to calculate the checksum for.
	 * @return The MD5 checksum
	 * @since 2.0
	 */
	public String checksum(IFile file) {
		try {
			InputStream fin = file.getContents(true);
			java.security.MessageDigest md5er = MessageDigest
					.getInstance(MESSAGE_DIGEST_TYPE);
			byte[] buffer = new byte[1024];
			int read;
			do {
				read = fin.read(buffer);
				if (read > 0)
					md5er.update(buffer, 0, read);
			} while (read != -1);
			fin.close();
			byte[] digest = md5er.digest();
			if (digest == null)
				return null;
			String strDigest = HEX_PREFIX;
			for (int i = 0; i < digest.length; i++) {
				strDigest += Integer.toString((digest[i] & 0xff) + 0x100, 16)
						.substring(1).toUpperCase();
			}
			return strDigest;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * 
	 * @param resource
	 * @since 2.0.0
	 */
	private void fireDataChangedEvent(IResource resource) {
		Object[] listeners = fListeners.getListeners();
		for (int i = 0; i < listeners.length; i++) {
			((IFrameworkDataChangedListener) listeners[i])
					.frameworkDataChanged(resource);
		}
	}

	/**
	 * Attempts to find the configuration element representing the extension
	 * declaration for the framework.
	 * 
	 * @param frameworkId
	 *            The framework identifier
	 * @return The configuration element
	 * @since 2.0.0
	 */
	private IConfigurationElement getFrameworkDeclaration(String frameworkId) {
		for (IConfigurationElement framework : getFrameworkDeclarations()) {
			String id = framework.getAttribute(ID);
			if (id.equals(frameworkId)) {
				return framework;
			}
		}
		return null;
	}

	private static HashMap<String, SoftwareFramework> fFrameworks = new HashMap<String, SoftwareFramework>();

	/**
	 * Finds and returns the given software framework. Including the
	 * configuration elements of this framework.
	 * 
	 * @param id
	 *            the framework identifier
	 * @return the software framework
	 */
	public SoftwareFramework getFramework(String id) {
		return fFrameworks.get(id);
	}

	public SoftwareFramework[] getFrameworks() {
		return fFrameworks.values().toArray(
				new SoftwareFramework[fFrameworks.size()]);
	}

	/**
	 * 
	 * 
	 * @return
	 */
	private IConfigurationElement[] getFrameworkDeclarations() {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint extensionPoint = registry
				.getExtensionPoint(FRAMEWORK_EXTENSION_ID);
		IConfigurationElement frameworks[] = extensionPoint
				.getConfigurationElements();
		return frameworks;
	}

	/**
	 * Uses the given file to determine if it stems from a software framework
	 * and returns the input stream of the framework implementation if it can be
	 * found.
	 * 
	 * @param resource
	 *            the file to calculate the input stream from.
	 * @return the input stream or <b>null</b> if not found.
	 * @throws IOException
	 * @since 2.1.0
	 */
	public InputStream getReferenceImplementationStream(IResource resource)
			throws IOException {

		IFrameworkFileDescriptor fd = getData(resource.getProject())
				.getFileDescriptor(resource);

		// Return null if this is not a software framework file
		if (fd == null) {
			return null;
		}

		IConfigurationElement frameworkDeclaration = getFrameworkDeclaration(fd
				.getFrameworkId());

		// Return null if the framework the file comes from cannot be found
		if (frameworkDeclaration == null) {
			return null;
		}

		Bundle bundle = Platform.getBundle(frameworkDeclaration
				.getContributor().getName());
		String source = frameworkDeclaration.getAttribute("source"); //$NON-NLS-1$
		Path p = new Path(source + IPath.SEPARATOR + fd.getSource());
		InputStream is = FileLocator.openStream(bundle, p, false);
		return is;
	}

	/**
	 * @since 2.0.0
	 */
	public void resourceChanged(IResourceChangeEvent event) {
		try {
			event.getDelta().accept(updater);
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Use to associate a single file with a software framework. The file will
	 * be examined and flagged as changed/unchanged accordingly.
	 * 
	 * @param file
	 *            the file
	 * @param id
	 *            the software framework identifier
	 * @param version
	 *            the software framework version
	 * @param path
	 *            the software framework path
	 * @since 2.1
	 */
	public void associate(IFile file, String id, String version, String path) {
		IFrameworkDataWorkingCopy data = getDataWorkingCopy(file.getProject());
		data.setIsFrameworkFile(file, id, version, path, true);
		storeData(data, new NullProgressMonitor());
		try {
			file.accept(updater);
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Removes a file from software framework context.
	 * 
	 * @param file
	 *            the file to disassociate
	 * @since 2.1
	 */
	public void disAssociate(IFile file) {
		updater.removeFile(file);
		fireDataChangedEvent(file);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.core.runtime.Plugin#start(org.osgi.framework.BundleContext)
	 */
	@Override
	public void start(BundleContext context) throws Exception {
		super.start(context);
		dataSets = new HashMap<IProject, IFrameworkData>();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this,
				IResourceChangeEvent.POST_BUILD);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this,
				IResourceChangeEvent.POST_CHANGE);
		ISaveParticipant saveParticipant = new SaveParticipant();
		ResourcesPlugin.getWorkspace()
				.addSaveParticipant(this, saveParticipant);
		IConfigurationElement[] elements = getFrameworkDeclarations();
		for (IConfigurationElement configurationElement : elements) {
			SoftwareFramework sf = new SoftwareFramework(configurationElement);
			fFrameworks.put(sf.getId(), sf);
		}
	}

	@Override
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		ResourcesPlugin.getWorkspace().removeSaveParticipant(this);
	}

	public static String describe(IConfigurationElement operation) {
		StringBuilder sb = new StringBuilder();

		Operation op = FrameworkConfigurationParser.getOperation(operation);
		switch (op) {
		case IMPORT:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_Import,
									new Object[] {
											operation
													.getAttribute(IFrameworkConfigurationParser.ATTR_PATH),
											operation
													.getAttribute(IFrameworkConfigurationParser.ATTR_DEST) }));
			break;
		case DELETE:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_Delete,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_PATH) }));
			break;
		case INCLUDE:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_Include,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_PATH) }));
			break;
		case ADD_LIBRARY_PATH:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_AddLibraryPath,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_PATH) }));
			break;
		case REMOVE_LIBRARY_PATH:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_RemoveLibraryPath,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_PATH) }));
			break;
		case ADD_LIBRARY:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_AddLibrary,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_NAME) }));
			break;
		case REMOVE_LIBRARY:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_RemoveLibrary,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_NAME) }));
			break;
		case RM_INCLUDE:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_RemoveInclude,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_PATH) }));
			break;
		case DEFINE:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_Define,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_SYMBOL) }));
			break;
		case RM_DEFINE:
			sb
					.append(MessageFormat
							.format(
									Messages.SummaryPage_RemoveDefine,
									new Object[] { operation
											.getAttribute(IFrameworkConfigurationParser.ATTR_SYMBOL) }));
			break;
		}

		return sb.toString();
	}

}