/****************************************************************************
 * Copyright (c) 2008 Atmel Corporation
 *
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the license which accompanies this code.
 * 
 * Contributors:
 *		Atmel Norway AS - Initial API and implementation  
 ****************************************************************************/

package com.atmel.avr32.sf.core;

import java.text.MessageFormat;
import java.util.ArrayList;

import org.eclipse.cdt.managedbuilder.core.BuildException;
import org.eclipse.cdt.managedbuilder.core.IConfiguration;
import org.eclipse.cdt.managedbuilder.core.IManagedBuildInfo;
import org.eclipse.cdt.managedbuilder.core.IOption;
import org.eclipse.cdt.managedbuilder.core.ITool;
import org.eclipse.cdt.managedbuilder.core.ManagedBuildManager;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.Status;

/**
 * Framework operation scanner which will go through all specified operations
 * and filter these according to supplied settings.
 * 
 * @author Torkild Ulvøy Resheim
 * @since 2.0
 */
public class OperationFilterScanner extends FrameworkConfigurationParser {
	private static final String ATTR_ID = "id"; //$NON-NLS-1$

	/** Hide include operations */
	public static final int HIDE_INCLUDES = 0x001;

	/** Do not filter */
	public static final int NONE = 0x000;

	/** The project framework data */
	private IFrameworkData data;

	/** A list of commands that will be executed */
	private ArrayList<IConfigurationElement> fCommandsToExcute;

	/** A list of operations whose commands will be executed */
	private ArrayList<IConfigurationElement> fOperationsToExecute;

	private IConfigurationElement root;

	/**
	 * 
	 * @param data
	 *            the framework data for the project
	 * @param root
	 *            wizard declaration root
	 */
	public OperationFilterScanner(IFrameworkData data,
			IConfigurationElement root) {
		super();
		this.data = data;
		this.root = root;
		fOperationsToExecute = new ArrayList<IConfigurationElement>();
		fCommandsToExcute = new ArrayList<IConfigurationElement>();
	}

	/**
	 * 
	 * @return all managed build configurations for the project
	 * @since 2.1
	 */
	private IConfiguration[] getConfigurations() {
		IManagedBuildInfo info = ManagedBuildManager.getBuildInfo(data
				.getProject());
		IConfiguration[] configs = info.getManagedProject().getConfigurations();
		return configs;
	}

	/**
	 * Locates and returns the operation with the given identifier.
	 * 
	 * @param id
	 * @param root
	 * @return
	 */
	private IConfigurationElement findMacro(String id) {
		IConfigurationElement[] children = root.getChildren();
		for (IConfigurationElement iConfigurationElement : children) {
			if (iConfigurationElement.getName().equals(MACRO)) {
				String t = iConfigurationElement.getAttribute(ID);
				if (t != null && t.equals(id)) {
					return iConfigurationElement;
				}
			}
		}
		return null;
	}

	/**
	 * Iterates through all operations and determines if they should apply given
	 * the specified filter options.
	 * 
	 * @param options
	 *            filter options
	 * @return A list of operations that applies
	 * @throws BuildException
	 * @since 2.0
	 */
	public ArrayList<IConfigurationElement> getFilteredCommands(int options)
			throws CoreException, BuildException {
		fOperationsToExecute.clear();
		fCommandsToExcute.clear();
		int size = -1;
		while (size != fOperationsToExecute.size()) {
			size = fOperationsToExecute.size();
			scan(root);
		}
		// As long as we have operations
		for (IConfigurationElement operation : fOperationsToExecute) {
			filterOperations(options, operation);
		}
		return fCommandsToExcute;

	}

	private ArrayList<IConfigurationElement> filterOperations(int options,
			IConfigurationElement operation) throws CoreException,
			BuildException {
		Assert.isNotNull(operation);
		ArrayList<IConfigurationElement> added = new ArrayList<IConfigurationElement>();
		String parent = "first option"; //$NON-NLS-1$
		if (SFCorePlugin.getDefault().isDebugging()) {
			System.out.println("Filtering commands from " + operation.getName()
					+ " \"" + operation.getAttribute(ID) + "\"");
		}
		for (IConfigurationElement command : operation.getChildren()) {
			Operation op = null;
			try {
				op = getOperation(command);
			} catch (IllegalArgumentException e) {
				throw new CoreException(
						new Status(
								IStatus.ERROR,
								SFCorePlugin.PLUGIN_ID,
								MessageFormat
										.format(
												"Not an recognized operation \"{0}\" following option {1}", //$NON-NLS-1$
												new Object[] {
														command.getName(),
														parent })));
			}
			switch (op) {

			case IMPORT:
				if (!isImportDuplicate(command) && isImportPerformable(command)) {
					fCommandsToExcute.add(command);
					if (SFCorePlugin.getDefault().isDebugging()) {
						System.out.println("  Will execute command \""
								+ SFCorePlugin.describe(command) + "\".");
					}
				}
				break;

			case DELETE:
				if (!isDeleteOverridden(command) && !isDeleteDuplicate(command)
						&& isDeletePerformable(command)) {
					fCommandsToExcute.add(command);
					if (SFCorePlugin.getDefault().isDebugging()) {
						System.out.println("  Will execute command \""
								+ SFCorePlugin.describe(command) + "\".");
					}
				}
				break;

			case INCLUDE:
				if ((options & HIDE_INCLUDES) != HIDE_INCLUDES) {
					if (!isIncludeDuplicate(command)
							&& isIncludePerformable(command)) {
						fCommandsToExcute.add(command);
						if (SFCorePlugin.getDefault().isDebugging()) {
							System.out.println("  Will execute command \""
									+ SFCorePlugin.describe(command) + "\".");
						}
					}
				}
				break;

			case ADD_LIBRARY_PATH:
				if (!isAddLibraryPathDuplicate(command)
						&& isAddLibraryPathPerformable(command)) {
					fCommandsToExcute.add(command);
					if (SFCorePlugin.getDefault().isDebugging()) {
						System.out.println("  Will execute command \""
								+ SFCorePlugin.describe(command) + "\".");
					}
				}
				break;

			case REMOVE_LIBRARY_PATH:
				if (!isRemoveLibraryPathDuplicate(command)
						&& !isRemoveLibraryPathOverridden(command)
						&& isRemoveLibraryPathPerformable(command)) {
					fCommandsToExcute.add(command);
					if (SFCorePlugin.getDefault().isDebugging()) {
						System.out.println("  Will execute command \""
								+ SFCorePlugin.describe(command) + "\".");
					}
				}
				break;

			case ADD_LIBRARY:
				if (!isAddLibraryDuplicate(command)
						&& isAddLibraryPerformable(command)) {
					fCommandsToExcute.add(command);
					if (SFCorePlugin.getDefault().isDebugging()) {
						System.out.println("  Will execute command \""
								+ SFCorePlugin.describe(command) + "\".");
					}
				}
				break;

			case REMOVE_LIBRARY:
				if (!isRemoveLibraryDuplicate(command)
						&& !isRemoveLibraryOverridden(command)
						&& isRemoveLibraryPerformable(command)) {
					fCommandsToExcute.add(command);
					if (SFCorePlugin.getDefault().isDebugging()) {
						System.out.println("  Will execute command \""
								+ SFCorePlugin.describe(command) + "\".");
					}
				}
				break;

			case RM_INCLUDE:
				if ((options & HIDE_INCLUDES) != HIDE_INCLUDES) {
					if (!isRmIncludeDuplicate(command)
							&& !isRemoveIncludeOverridden(command)
							&& isRemoveIncludePerformable(command)) {
						fCommandsToExcute.add(command);
						if (SFCorePlugin.getDefault().isDebugging()) {
							System.out.println("  Will execute command \""
									+ SFCorePlugin.describe(command) + "\".");
						}
					}
				}
				break;

			case DEFINE:
				if (!isDefineDuplicate(command) && isDefinePerformable(command)) {
					fCommandsToExcute.add(command);
					if (SFCorePlugin.getDefault().isDebugging()) {
						System.out.println("  Will execute command \""
								+ SFCorePlugin.describe(command) + "\".");
					}
				}
				break;

			case RM_DEFINE:
				if (!isRmDefineDuplicate(command)
						&& !isRemoveDefineOverridden(command)
						&& isRemoveDefinePerformable(command)) {
					fCommandsToExcute.add(command);
					if (SFCorePlugin.getDefault().isDebugging()) {
						System.out.println("  Will execute command \""
								+ SFCorePlugin.describe(command) + "\".");
					}
				}
				break;
			}
			parent = ((IConfigurationElement) ((IConfigurationElement) command
					.getParent()).getParent()).getAttribute(ATTR_ID);
		}
		return added;
	}

	/**
	 * Tests if the remove defined symbol operation should be performed. If the
	 * symbol is not defined in one of the tools, <b>false</b> is returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the operation should be performed
	 * @throws BuildException
	 * @since 2.1
	 */
	private boolean isRemoveDefinePerformable(IConfigurationElement operation)
			throws BuildException {
		return !(isDefinePerformable(operation));
	}

	/**
	 * Tests if the define symbol operation should be performed. If the symbol
	 * is already defined in one of the tools, <b>false</b> is returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the operation should be performed
	 * @throws BuildException
	 * @since 2.1
	 */
	private boolean isDefinePerformable(IConfigurationElement operation)
			throws BuildException {
		String value = operation.getAttribute(ATTR_SYMBOL);
		for (IConfiguration configuration : getConfigurations()) {
			ITool[] tools = configuration.getTools();
			for (ITool tool : tools) {
				for (IOption option : tool.getOptions()) {
					if (option.getId() != null
							&& (option.getId().startsWith(SYMBOL_ID))) {
						String[] values = option.getDefinedSymbols();
						for (String string : values) {
							if (string.equals(value)) {
								return false;
							}
						}
					}
				} // for
			} // for
		} // for
		return true;
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>define</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.1
	 */
	private boolean isAddLibraryDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(ADD_LIBRARY));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(ADD_LIBRARY)) {
				if (operation.getAttribute(ATTR_NAME).equals(
						element.getAttribute(ATTR_NAME)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>add-library-path</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.1
	 */
	private boolean isAddLibraryPathDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(ADD_LIBRARY_PATH));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(ADD_LIBRARY_PATH)) {
				if (operation.getAttribute(ATTR_PATH).equals(
						element.getAttribute(ATTR_PATH)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Tests if the add library path operation should be performed. If the
	 * folder is already included in one of the tools, <b>false</b> is returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the operation should be performed
	 * @throws BuildException
	 * @since 2.1
	 */
	private boolean isAddLibraryPathPerformable(IConfigurationElement operation)
			throws BuildException {
		String value = operation.getAttribute(ATTR_PATH);
		for (IConfiguration configuration : getConfigurations()) {
			ITool[] tools = configuration.getTools();
			for (ITool tool : tools) {
				for (IOption option : tool.getOptions()) {
					if (option.getId() != null
							&& (option.getId().startsWith(LIBARY_PATH_ID))) {
						String[] values = option.getBasicStringListValue();
						for (String string : values) {
							if (string.equals(value)) {
								return false;
							}
						}
					}
				} // for
			} // for
		} // for
		return true;
	}

	/**
	 * Tests if the add library operation should be performed. If the library is
	 * already included in <i>one</i> of the tools, <b>false</b> is returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the operation should be performed
	 * @throws BuildException
	 * @since 2.1
	 */
	private boolean isAddLibraryPerformable(IConfigurationElement operation)
			throws BuildException {
		String value = operation.getAttribute(ATTR_NAME);
		for (IConfiguration configuration : getConfigurations()) {
			ITool[] tools = configuration.getTools();
			for (ITool tool : tools) {
				for (IOption option : tool.getOptions()) {
					if (option.getId() != null
							&& (option.getId().startsWith(LIBRARY_ID))) {
						String[] values = option.getLibraries();
						for (String string : values) {
							if (string.equals(value)) {
								return false;
							}
						}
					}
				} // for
			} // for
		} // for
		return true;
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>define</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.0
	 */
	private boolean isDefineDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(DEFINE));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(DEFINE)) {
				if (operation.getAttribute(ATTR_SYMBOL).equals(
						element.getAttribute(ATTR_SYMBOL)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>include</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.0
	 */
	private boolean isDeleteDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(DELETE));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(DELETE)) {
				if (operation.getAttribute(ATTR_PATH).equals(
						element.getAttribute(ATTR_PATH)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Tests if the <i>delete</i> operation is overridden by an <i>import</i> in
	 * which case <b>True</b> is returned.
	 * 
	 * @param element
	 * @return
	 * @since 2.0
	 */
	private boolean isDeleteOverridden(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(DELETE));
		for (IConfigurationElement opRoot : fOperationsToExecute) {
			for (IConfigurationElement command : opRoot.getChildren()) {
				if (command != element && command.getName().equals(IMPORT)) {
					if (command.getAttribute(ATTR_DEST).equals(
							element.getAttribute(ATTR_PATH))) {
						if (SFCorePlugin.getDefault().isDebugging()) {
							System.out
									.println("Removal of  \""
											+ element.getAttribute(ATTR_PATH)
											+ "\" is overridden by \"import\" command.");
						}
						return true;
					}
				}
			}
		}
		return false;
	}

	/**
	 * Tests if the delete operation can be executed. If the file that is to be
	 * deleted does not exist, <b>false</b> will be returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the operation should be performed
	 * @since 2.1
	 */
	private boolean isDeletePerformable(IConfigurationElement operation) {
		IResource r = data.getProject().findMember(
				operation.getAttribute(ATTR_PATH));
		return (r != null && r.exists());
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>include</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.0
	 */
	private boolean isImportDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(IMPORT));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(IMPORT)) {
				if (operation.getAttribute(ATTR_DEST).equals(
						element.getAttribute(ATTR_DEST)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Tests if the import operation should be performed. If the file already
	 * exists and is unchanged, <b>false</b> is returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the operation should be performed
	 * @since 2.1
	 */
	private boolean isImportPerformable(IConfigurationElement operation) {
		final IFile to = data.getProject().getFile(
				operation.getAttribute(ATTR_DEST));
		// Handle that the file may already exist.
		if (to.exists()) {
			IFrameworkFileDescriptor desc = data.getFileDescriptor(to);
			if (desc != null && desc.isChanged()) {
				return true;
			} else
				return false;
		}
		return true;
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>include</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.0
	 */
	private boolean isIncludeDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(INCLUDE));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(INCLUDE)) {
				if (operation.getAttribute(ATTR_PATH).equals(
						element.getAttribute(ATTR_PATH)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Tests if the include operation should be performed. If the folder is
	 * already included in one of the tools, <b>true</b> is returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the operation should be performed
	 * @throws BuildException
	 * @since 2.1
	 */
	private boolean isIncludePerformable(IConfigurationElement operation)
			throws BuildException {
		String value = operation.getAttribute(ATTR_PATH);
		for (IConfiguration configuration : getConfigurations()) {
			ITool[] tools = configuration.getTools();
			for (ITool tool : tools) {
				for (IOption option : tool.getOptions()) {
					if (option.getId() != null
							&& (option.getId().startsWith(C_INCLUDE_PATH_ID)
									|| option.getId().startsWith(
											ASM_INCLUDE_PATH_ID) || option
									.getId().startsWith(ASM2_INCLUDE_PATH_ID))) {
						String[] values = option.getIncludePaths();
						for (String string : values) {
							if (string.equals(value)) {
								return false;
							}
						}
					}
				} // for
			} // for
		} // for
		return true;
	}

	/**
	 * Goes trough all operations and returns <b>True</b> if the
	 * <i>rm-define</i> operation is overridden by an active <i>define</i>
	 * operation. In which case the <i>rm-define</i> operation should not apply.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation is being overridden
	 * @since 2.0
	 */
	private boolean isRemoveDefineOverridden(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(RM_DEFINE));
		testOverride(DEFINE, ATTR_SYMBOL, element);
		return false;
	}

	/**
	 * Goes trough all operations and returns <b>True</b> if the
	 * <i>rm-include</i> operation is overridden by an active <i>include</i>
	 * operation. In which case the <i>rm-include</i> operation should not
	 * apply.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation is being overridden
	 * @since 2.0
	 */
	private boolean isRemoveIncludeOverridden(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(RM_INCLUDE));
		return testOverride(INCLUDE, ATTR_PATH, element);
	}

	/**
	 * Tests if the remove include operation should be performed. If the folder
	 * is included in one of the tools, <b>true</b> is returned.
	 * 
	 * @param operation
	 * 
	 * @return <b>true</b> if the remove include path operation should be
	 *         performed
	 * 
	 * @throws BuildException
	 */
	private boolean isRemoveIncludePerformable(IConfigurationElement operation)
			throws BuildException {
		return (!isIncludePerformable(operation));
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>remove-library</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.1
	 */
	private boolean isRemoveLibraryDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(REMOVE_LIBRARY));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(REMOVE_LIBRARY)) {
				if (operation.getAttribute(ATTR_NAME).equals(
						element.getAttribute(ATTR_NAME)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Goes trough all operations and returns <b>True</b> if the
	 * <i>remove-library</i> operation is overridden by an active
	 * <i>add-library</i> operation. In which case the <i>remove-library</i>
	 * operation should not apply.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation is being overridden
	 * @since 2.1
	 */
	private boolean isRemoveLibraryOverridden(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(REMOVE_LIBRARY));
		return testOverride(ADD_LIBRARY, ATTR_NAME, element);
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>remove-library-path</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.1
	 */
	private boolean isRemoveLibraryPathDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(REMOVE_LIBRARY_PATH));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(REMOVE_LIBRARY_PATH)) {
				if (operation.getAttribute(ATTR_PATH).equals(
						element.getAttribute(ATTR_PATH)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Goes trough all operations and returns <b>True</b> if the
	 * <i>remove-library-path</i> operation is overridden by an active
	 * <i>add-library-path</i> operation. In which case the
	 * <i>remove-library-path</i> operation should not apply.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation is being overridden
	 * @since 2.1
	 */
	private boolean isRemoveLibraryPathOverridden(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(REMOVE_LIBRARY_PATH));
		return testOverride(ADD_LIBRARY_PATH, ATTR_PATH, element);
	}

	/**
	 * Tests if the remove library path operation should be performed. If the
	 * folder is already included in one of the tools, <b>false</b> is returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the library path should be performed
	 * @throws BuildException
	 * @since 2.1
	 */
	private boolean isRemoveLibraryPathPerformable(
			IConfigurationElement operation) throws BuildException {
		return (!isAddLibraryPathPerformable(operation));
	}

	/**
	 * Tests if the remove library operation should be performed. If the library
	 * is included in one of the tools, <b>true</b> is returned.
	 * 
	 * @param operation
	 *            the operation to test
	 * @return <b>true</b> if the library path should be performed
	 * @throws BuildException
	 * @since 2.1
	 */
	private boolean isRemoveLibraryPerformable(IConfigurationElement operation)
			throws BuildException {
		return (!isAddLibraryPerformable(operation));
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>rm-define</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.0
	 */
	private boolean isRmDefineDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(RM_DEFINE));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(RM_DEFINE)) {
				if (operation.getAttribute(ATTR_SYMBOL).equals(
						element.getAttribute(ATTR_SYMBOL)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Goes trough all already filtered operations and returns <b>True</b> if an
	 * equal <i>rm-include</i> operation has already been added.
	 * 
	 * @param element
	 *            The element to test
	 * @return <b>True</b> if the operation has already been defined
	 * @since 2.0
	 */
	private boolean isRmIncludeDuplicate(IConfigurationElement element) {
		Assert.isTrue(element.getName().equals(RM_INCLUDE));
		for (IConfigurationElement operation : fCommandsToExcute) {
			if (operation.getName().equals(RM_INCLUDE)) {
				if (operation.getAttribute(ATTR_PATH).equals(
						element.getAttribute(ATTR_PATH)))
					return true;
			}
		}
		return false;
	}

	/**
	 * Recurses through the configuration element, looking for options that
	 * shall result in an operation. Feed the configuration root to begin with.
	 * As a result of this <b>fOperation</b> will be populated with the
	 * configuration elements of operations that will be executed.
	 * 
	 * @param parent
	 *            the parent element.
	 * @throws CoreException
	 * @throws InvalidRegistryObjectException
	 * @since 2.0
	 */
	private void scan(IConfigurationElement parent)
			throws InvalidRegistryObjectException, CoreException {
		IConfigurationElement[] children = parent.getChildren();
		for (IConfigurationElement child : children) {
			if (child.getName().equals(OPERATION)) {
				final String id = parent.getAttribute(ID);
				final String value = data.getValue(id);
				if (child.getAttribute(VALUE) == null
						|| (value != null && value.equals(child
								.getAttribute(VALUE)))) {
					if (!fOperationsToExecute.contains(child)) {
						fOperationsToExecute.add(child);
						if (SFCorePlugin.getDefault().isDebugging()) {
							System.out.println("Adding operation \""
									+ ((IConfigurationElement) child
											.getParent()).getAttribute(ID)
									+ "=" + child.getAttribute(VALUE)
									+ "\" to executable.");
						}
					}
					// Determine if we have any macros that also must be added
					IConfigurationElement[] commands = child.getChildren();
					for (IConfigurationElement command : commands) {
						if (command.getName().equals(EXECUTE)) {
							IConfigurationElement macro = findMacro(command
									.getAttribute(MACRO));
							if (macro != null) {
								if (!fOperationsToExecute.contains(macro)) {
									fOperationsToExecute.add(macro);
									if (SFCorePlugin.getDefault().isDebugging()) {
										System.out.println("Adding macro \""
												+ macro.getAttribute(ID)
												+ "\" to executable.");
									}
								}
							} else {
								throw new CoreException(
										new Status(
												IStatus.ERROR,
												SFCorePlugin.PLUGIN_ID,
												MessageFormat
														.format(
																"Could not find referenced macro \"{0}\"",
																new Object[] { command
																		.getAttribute(MACRO) })));

							}
						}
					}
				}
			}
			scan(child);
		} // for
	}

	/**
	 * Tests if an operation exists with the specified operation and attribute
	 * name which value equals the value of the same attribute in the supplied
	 * element.
	 * 
	 * @param opName
	 * @param element
	 * @return
	 * @since 2.1
	 */
	private boolean testOverride(String opName, String attrName,
			IConfigurationElement element) {
		for (IConfigurationElement opRoot : fOperationsToExecute) {
			for (IConfigurationElement operation : opRoot.getChildren()) {
				if (operation.getName().equals(opName)) {
					if (operation.getAttribute(attrName).equals(
							element.getAttribute(attrName)))
						return true;
				}
			}
		}
		return false;
	}
}
