/****************************************************************************
 * Copyright (c) 2008 Atmel Corporation
 *
 * All rights reserved. This program and the accompanying materials are made 
 * available under the terms of the license which accompanies this code.
 * 
 * Contributors:
 *		Atmel Norway AS - Initial API and implementation  
 ****************************************************************************/
package com.atmel.avr32.sf.core;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;


/**
 * 
 * @author Torkild Ulvøy Resheim
 * @since 2.0
 */
public class FrameworkProjectData implements IFrameworkData {
    public static final String FRAMEWORK_DATA_FILE = ".framework"; //$NON-NLS-1$

    /** The project that the framework data is for */
    protected IProject project;

    /** Option values */
    protected HashMap<String, String> values;

    protected HashMap<IPath, FrameworkFileDescriptor> descriptors;

    public IFrameworkFileDescriptor getFileDescriptor(IResource resource) {
        FrameworkFileDescriptor desc = descriptors.get(resource
                .getProjectRelativePath());
        return desc;
    }

    /**
     * Clear the framework metadata for information belonging to the specified
     * file.
     * 
     * @param resource
     *            The file to clear the data for.
     * @since 2.0
     */
    public void removeFile(IResource resource) {
        descriptors.remove(resource.getProjectRelativePath());
    }

    public FrameworkProjectData() {
        values = new HashMap<String, String>();
        descriptors = new HashMap<IPath, FrameworkFileDescriptor>();
    }

    public FrameworkProjectData(IProject project) {
        this();
        this.project = project;
        restoreData();
    }

    void addFrameworkDescriptor(FrameworkFileDescriptor descriptor) {
        descriptors.put(descriptor.getPath(), descriptor);
    }

    /**
     * @return the project
     */
    public IProject getProject() {
        return project;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.atmel.avr32.sf.core.IFrameworkData#getValue(java.lang.String)
     */
    public String getValue(String key) {
        return values.get(key);
    }

    public HashMap<String, String> getValues() {
        return values;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.atmel.avr32.sf.core.IFrameworkData#isFrameworkFile(org.eclipse.core.resources.IResource)
     */
    public boolean isFrameworkFile(IResource resource) {
        return (descriptors.get(resource.getProjectRelativePath()) != null);
    }

    private InputStream openContentStream() {
        StringBuilder data = new StringBuilder();
        data.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"); //$NON-NLS-1$
        data.append("<?fileVersion 1.0.0?>\n"); //$NON-NLS-1$
        data.append("<framework-data>\n"); //$NON-NLS-1$
        data.append("  <options>\n"); //$NON-NLS-1$
        for (String key : values.keySet()) {
            data.append("    <option id=\""); //$NON-NLS-1$
            data.append(key);
            data.append("\" value=\""); //$NON-NLS-1$
            data.append(values.get(key));
            data.append("\"/>\n"); //$NON-NLS-1$
        }
        data.append("  </options>\n"); //$NON-NLS-1$
        data.append("  <files>\n"); //$NON-NLS-1$
        for (IPath path : descriptors.keySet()) {
            IFrameworkFileDescriptor descriptor = descriptors.get(path);
            data.append("    <file path=\""); //$NON-NLS-1$
            data.append(path.toPortableString());
            data.append("\" framework=\""); //$NON-NLS-1$
            data.append(descriptor.getFrameworkId());
            data.append("\" version=\""); //$NON-NLS-1$
            data.append(descriptor.getVersion());
            data.append("\" source=\""); //$NON-NLS-1$
            data.append(descriptor.getSource());
            data.append("\" changed=\""); //$NON-NLS-1$
            data.append(descriptor.isChanged());
            data.append("\"/>\n"); //$NON-NLS-1$
        }
        data.append("  </files>\n"); //$NON-NLS-1$
        data.append("</framework-data>\n"); //$NON-NLS-1$
        return new ByteArrayInputStream(data.toString().getBytes());
    }

    public IStatus restoreData() {
        final IFile file = project.getFile(FRAMEWORK_DATA_FILE);
        try {
            FrameworkDataReader ts = new FrameworkDataReader(this);
            SAXParserFactory factory = SAXParserFactory.newInstance();
            SAXParser parser = factory.newSAXParser();
            XMLReader reader = parser.getXMLReader();
            reader.setContentHandler(ts);
            InputSource input = new InputSource(file.getContents());
            // Make sure we clear all previous values
            descriptors.clear();
            values.clear();
            reader.parse(input);
        } catch (IOException e) {
            return new Status(IStatus.ERROR, SFCorePlugin.PLUGIN_ID,
                    Messages.FrameworkData_RestoreError, e);
        } catch (SAXException e) {
            return new Status(IStatus.ERROR, SFCorePlugin.PLUGIN_ID,
                    Messages.FrameworkData_RestoreError, e);
        } catch (ParserConfigurationException e) {
            return new Status(IStatus.ERROR, SFCorePlugin.PLUGIN_ID,
                    Messages.FrameworkData_RestoreError, e);
        } catch (CoreException e) {
            return new Status(IStatus.ERROR, SFCorePlugin.PLUGIN_ID,
                    Messages.FrameworkData_RestoreError, e);
        }
        return Status.OK_STATUS;
    }

    public void setIsFrameworkFile(IResource resource, String framework,
            String version, String source, boolean value) {
        if (value) {
            FrameworkFileDescriptor descriptor = (FrameworkFileDescriptor) getFileDescriptor(resource);
            if (descriptor == null) {
                descriptor = new FrameworkFileDescriptor(resource);
            }
            descriptor.setUpdated(System.currentTimeMillis());
            descriptor.setFramework(framework);
            descriptor.setVersion(version);
            descriptor.setSource(source);
            descriptors.put(resource.getProjectRelativePath(), descriptor);
        } else {
            descriptors.remove(resource.getProjectRelativePath());
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.atmel.avr32.sf.core.IFrameworkData#setValue(java.lang.String,
     *      java.lang.String)
     */
    public void setValue(String id, String value) {
        values.put(id, value);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.atmel.avr32.sf.core.IFrameworkData#storeData(org.eclipse.core.runtime.IProgressMonitor)
     */
    public IStatus storeData(IProgressMonitor monitor) {
        if (project.isAccessible()) {
            final IFile file = project.getFile(FRAMEWORK_DATA_FILE);
            // We should never get here as we don't want to store the framework
            // metadata when a project is being saved.
            if (!project.exists())
                return Status.OK_STATUS;
            try {
                InputStream stream = openContentStream();
                if (file.exists()) {
                    file.setContents(stream, true, true, monitor);

                } else {
                    file.create(stream, true, monitor);
                }
                stream.close();
            } catch (IOException e) {
                e.printStackTrace();
                return new Status(IStatus.ERROR, SFCorePlugin.PLUGIN_ID,
                        Messages.FrameworkData_StoreError, e);
            } catch (CoreException e) {
                e.printStackTrace();
                return new Status(IStatus.ERROR, SFCorePlugin.PLUGIN_ID,
                        Messages.FrameworkData_StoreError, e);
            }
            return Status.OK_STATUS;
        }
        return Status.CANCEL_STATUS;
    }

    public IFrameworkFileDescriptor[] getFileDescriptors() {
        return descriptors.values().toArray(
                new IFrameworkFileDescriptor[descriptors.size()]);
    }
}
