/* This source file is part of the ATMEL AVR32-SoftwareFramework-AT32AP7000-1.0.0 Release */

/*This file is prepared for Doxygen automatic documentation generation.*/
/*! \file *********************************************************************
 *
 * \brief SDRAMC on EBI example application.
 *
 * - Compiler:           IAR EWAVR32 and GNU GCC for AVR32
 * - Supported devices:  All AVR32 devices with an SDRAMC module can be used.
 * - AppNote:
 *
 * \author               Atmel Corporation: http://www.atmel.com \n
 *                       Support and FAQ: http://support.atmel.no/
 *
 ******************************************************************************/

/*! \page License
 * Copyright (C) 2006-2008, Atmel Corporation All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. The name of ATMEL may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ATMEL ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY AND
 * SPECIFICALLY DISCLAIMED. IN NO EVENT SHALL ATMEL BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*! \mainpage
 * \section intro Introduction
 * This is the documentation for the data structures, functions, variables,
 * defines, enums, and typedefs in the software for application note AVR32102.
 * It also gives an examples of the usage of the SRAM Controller driver with 
 * the hardware found on the STK1000. Setup, initialization and testing is 
 * included.
 *
 * \section files Main Files
 * - sdramc_at32ap7000.c: sdram controller driver
 * - sdramc_at32ap7000.h: sdram controller driver header file
 * - sdramc_example.c: sdram controller example application.
 *
 * \section compinfo Compilation Info
 * This software was written for the GNU GCC for AVR32 and IAR Systems compiler
 * for AVR32. Other compilers may or may not work.
 *
 * \section deviceinfo Device Info
 * All AVR32 devices with a SDRAM Controller can be used. This example has been tested
 * with the following setup:
 *
 * - STK1000 development kit
 * - AP7000 top board for STK1000 development kit
 * 
 * No hardware specific settings are needed on the STK1000
 *
 * \section setupinfo Setup Information
 * CPU speed: <i>20 MHz</i>
 *
 *
 * \section applicationUsage Application Usage
 *
 * This example initializes the SDRAM found on the STK1000, tests it and then returns 
 * the number of errors found. To use this code in your own application, make a pointer 
 * to the base address of the SDRAM and access it like you would normally access a 
 * memory mapped device.
 *
 * When recompiling a project, make sure to run make clean before recompiling
 *
 * \section contactinfo Contact Info
 * For more info about Atmel AVR32 visit
 * <A href="http://www.atmel.com/products/AVR32/" >Atmel AVR32</A> \n
 * <A href="http://www.atmel.com/dyn/products/app_notes.asp?family_id=682">
 * AVR32 Application Notes</A>\n
 * Support mail: avr32@atmel.com
 */

#include "print_funcs.h"
#include "pm_at32ap7000.h"
#include "board.h"
#include "gpio.h"
#include "sdramc_at32ap7000.h"


/*!
 * This function will setup the the SDRAM controller, intialize the SDRAM found on the STK1000 and test it.
 * \return The number of errors
 */
int main( void )
{
  U32 i, sdram_size, tmp;
  U32 noErrors=0;
  volatile U32 *sdram = (void *) BOARD_SDRAM_BASE;

#if BOARD == STK1000
  // Reset PM. Makes sure we get the expected clocking after a soft reset (e.g.: JTAG reset)
  pm_reset();

  // Start PLL0 giving 80 MHz clock
  pm_pll_opt_t pll_opt ={
    .pll_id = 0,
    .mul = 4,
    .div = 1,
    .osc_id = 0,
    .count = 16,
    .wait_for_lock = 1,
  };
  pm_start_pll(&pll_opt);

  // Divide HSB by 2, PBB by 2 and PBA by 4 to keep them below maximum ratings
  pm_set_clock_domain_scaler(PM_HSB_DOMAIN, 2);
  pm_set_clock_domain_scaler(PM_PBB_DOMAIN, 2);
  pm_set_clock_domain_scaler(PM_PBA_DOMAIN, 4);

  pm_set_mclk_source(PM_PLL0);
#endif

  init_dbg_rs232(pm_read_module_freq_hz(PM_PBA_USART1));
  print_dbg("Board running at ");
  print_dbg_ulong(pm_get_mclk_freq_hz()/1000000);
  print_dbg(" MHz\n");
  print_dbg("Initializing SDRAM...");
  sdramc_init(pm_read_module_freq_hz(PM_PBB_HSDRAMC));
  print_dbg("done\n\n");

  /* setup sdram_size */
  sdram_size = 1 << (SDRAM_COL_BITS + SDRAM_ROW_BITS + SDRAM_BANK_BITS + 2);

/* test the chip */
  for (i = 0; i < sdram_size; i++){

    sdram[i] = i;
    tmp = sdram[i];

    if(i != tmp) {
      noErrors++;
      print_dbg("ERROR at 0x");
      print_dbg_hex((U32) &sdram[i]);
      print_dbg("\n");
      }
    }  
  if (noErrors == 0)
    print_dbg("SDRAM test successfully completed\n");
  else{
    print_dbg("SDRAM test completed with ");
    print_dbg_ulong(noErrors);
    print_dbg(" errors out of ");
    print_dbg_ulong(i);
    print_dbg(" tests\n");
  }

  while(1);
}
