/**************************************************
 *
 * Declaration of system and registers for AVR32
 * IAR C/C++ Compiler and IAR Assembler.
 *  
 * Copyright 2005-2006 IAR Systems. All rights reserved.
 *
 * $Revision: 2.2 $
 *
 **************************************************/

#ifndef __IOAVR32_H
#define __IOAVR32_H

#if (!__AAVR32__ && !__ICCAVR32__)
#error "File ioavr32.h is for use with ICCAVR32/AAVR32 only."
#endif

#include "avr32/abi.h"

#ifdef __ICCAVR32__
#pragma language=extended

/**************************************************
 *   System registers
 **************************************************/

typedef struct avr32_sr_t {
    unsigned int                 : 2;
    unsigned int h               : 1;
    unsigned int j               : 1;
    unsigned int dm              : 1;
    unsigned int d               : 1;
    unsigned int                 : 1;
    unsigned int m               : 3;
    unsigned int em              : 1;
    unsigned int i3m             : 1;
    unsigned int i2m             : 1;
    unsigned int i1m             : 1;
    unsigned int i0m             : 1;
    unsigned int gm              : 1;
    unsigned int r               : 1;
    unsigned int t               : 1;
    unsigned int                 : 8;
    unsigned int l               : 1;
    unsigned int q               : 1;
    unsigned int v               : 1;
    unsigned int n               : 1;
    unsigned int z               : 1;
    unsigned int c               : 1;
} avr32_sr_t;

#if __CORE__ == __AVR32A__
typedef struct avr32_cpucr_t {
    unsigned int                 :15;
    unsigned int locen           : 1;
    unsigned int spl             : 5;
    unsigned int cpl             : 5;
    unsigned int cop             : 5;
    unsigned int sie             : 1;
} avr32_cpucr_t;
#elif __CORE__ == __AVR32B__
typedef struct avr32_cpucr_t {
    unsigned int cop7en          : 1;
    unsigned int cop6en          : 1;
    unsigned int cop5en          : 1;
    unsigned int cop4en          : 1;
    unsigned int cop3en          : 1;
    unsigned int cop2en          : 1;
    unsigned int cop1en          : 1;
    unsigned int cop0en          : 1;
    unsigned int                 :17;
    unsigned int im              : 1;
    unsigned int iee             : 1;
    unsigned int ibe             : 1;
    unsigned int re              : 1;
    unsigned int fe              : 1;
    unsigned int be              : 1;
    unsigned int bi              : 1;
} avr32_cpucr_t;
#endif

typedef struct avr32_config0_t {
    unsigned int processorid     : 8;
    unsigned int                 : 4;
    unsigned int processorrevision : 4;
    unsigned int at              : 3;
    unsigned int ar              : 3;
    unsigned int mmut            : 3;
    unsigned int f               : 1;
    unsigned int j               : 1;
    unsigned int p               : 1;
    unsigned int o               : 1;
    unsigned int s               : 1;
    unsigned int d               : 1;
    unsigned int r               : 1;
} avr32_config0_t;

typedef struct avr32_config1_t {
    unsigned int immusz          : 6;
    unsigned int dmmusz          : 6;
    unsigned int iset            : 4;
    unsigned int ilsz            : 3;
    unsigned int iass            : 3;
    unsigned int dset            : 4;
    unsigned int dlsz            : 3;
    unsigned int dass            : 3;
} avr32_config1_t;

typedef struct avr32_tlbehi_t {
    unsigned int vpn             :22;
    unsigned int v               : 1;
    unsigned int i               : 1;
    unsigned int asid            : 8;
} avr32_tlbehi_t;

typedef struct avr32_tlbelo_t {
    unsigned int pfn             :22;
    unsigned int c               : 1;
    unsigned int g               : 1;
    unsigned int b               : 1;
    unsigned int ap              : 3;
    unsigned int sz              : 2;
    unsigned int d               : 1;
    unsigned int w               : 1;
} avr32_tlbelo_t;

typedef struct avr32_mmucr_t {
    unsigned int                 :12;
    unsigned int drp             : 6;
    unsigned int dla             : 6;
    unsigned int                 : 3;
    unsigned int s               : 1;
    unsigned int n               : 1;
    unsigned int i               : 1;
    unsigned int m               : 1;
    unsigned int e               : 1;
} avr32_mmucr_t;

typedef struct avr32_pccr_t {
    unsigned int                 : 8;
    unsigned int conf1           : 6;
    unsigned int conf0           : 6;
    unsigned int                 : 1;
    unsigned int f1              : 1;
    unsigned int f0              : 1;
    unsigned int fc              : 1;
    unsigned int                 : 1;
    unsigned int ie1             : 1;
    unsigned int ie0             : 1;
    unsigned int iec             : 1;
    unsigned int s               : 1;
    unsigned int c               : 1;
    unsigned int r               : 1;
    unsigned int e               : 1;
} avr32_pccr_t;

typedef struct avr32_mpuar_t {
    unsigned int base            :20;
    unsigned int                 : 6;
    unsigned int size            : 5;
    unsigned int v               : 1;
} avr32_mpuar_t;

typedef struct avr32_mpupsr_t {
    unsigned int                 :16;
    unsigned int p15             : 1;
    unsigned int p14             : 1;
    unsigned int p13             : 1;
    unsigned int p12             : 1;
    unsigned int p11             : 1;
    unsigned int p10             : 1;
    unsigned int p9              : 1;
    unsigned int p8              : 1;
    unsigned int p7              : 1;
    unsigned int p6              : 1;
    unsigned int p5              : 1;
    unsigned int p4              : 1;
    unsigned int p3              : 1;
    unsigned int p2              : 1;
    unsigned int p1              : 1;
    unsigned int p0              : 1;
} avr32_mpupsr_t;

typedef struct avr32_mpucrx_t {
    unsigned int                 :24;
    unsigned int c7              : 1;
    unsigned int c6              : 1;
    unsigned int c5              : 1;
    unsigned int c4              : 1;
    unsigned int c3              : 1;
    unsigned int c2              : 1;
    unsigned int c1              : 1;
    unsigned int c0              : 1;
} avr32_mpucrx_t;

typedef struct avr32_mpubrx_t {
    unsigned int                 :24;
    unsigned int b7              : 1;
    unsigned int b6              : 1;
    unsigned int b5              : 1;
    unsigned int b4              : 1;
    unsigned int b3              : 1;
    unsigned int b2              : 1;
    unsigned int b1              : 1;
    unsigned int b0              : 1;
} avr32_mpubrx_t;

typedef struct avr32_mpuaprx_t {
    unsigned int ap7             : 4;
    unsigned int ap6             : 4;
    unsigned int ap5             : 4;
    unsigned int ap4             : 4;
    unsigned int ap3             : 4;
    unsigned int ap2             : 4;
    unsigned int ap1             : 4;
    unsigned int ap0             : 4;
} avr32_mpuaprx_t;

typedef struct avr32_mpucr_t {
    unsigned int                 :31;
    unsigned int e               : 1;
} avr32_mpucr_t;

#if __CORE__ == __AVR32A__
#define AVR32A(s) s
#define AVR32B(s) unsigned long : 32
#elif __CORE__ == __AVR32B__
#define AVR32A(s) unsigned long : 32
#define AVR32B(s) s
#else
#error "Unknown AVR32 core"
#endif

typedef struct avr32_core_t {
        /* 0x000 */
        union {
          unsigned long         sr;
          avr32_sr_t            SR;
        };
        unsigned long           evba;
        unsigned long           acba;
        union {
          unsigned long         cpucr;
          avr32_cpucr_t         CPUCR;
        };
        unsigned long     const ecr;

        /* 0x014 */
AVR32B (unsigned long           rsr_sup);
AVR32B (unsigned long           rsr_int0);
AVR32B (unsigned long           rsr_int1);
AVR32B (unsigned long           rsr_int2);
AVR32B (unsigned long           rsr_int3);
AVR32B (unsigned long           rsr_ex);
AVR32B (unsigned long           rsr_nmi);
        unsigned long           rsr_dbg;

        /* 0x034 */
AVR32B (unsigned long           rar_sup);
AVR32B (unsigned long           rar_int0);
AVR32B (unsigned long           rar_int1);
AVR32B (unsigned long           rar_int2);
AVR32B (unsigned long           rar_int3);
AVR32B (unsigned long           rar_ex);
AVR32B (unsigned long           rar_nmi);
        unsigned long           rar_dbg ;

        /* 0x054 */
        unsigned long     const jecr;
        unsigned long           josp;
        unsigned long           java_lv0;
        unsigned long           java_lv1;
        unsigned long           java_lv2;
        unsigned long           java_lv3;
        unsigned long           java_lv4;
        unsigned long           java_lv5;
        unsigned long           java_lv6;
        unsigned long           java_lv7;
        unsigned long           jtba;
        unsigned long           jbcr;

        /* 0x084 */
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;

        /* 0x0A0 */
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;

        /* 0x0C0 */
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;

        /* 0x0E0 */
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;
        unsigned long           : 32;

        /* 0x100 */
        union {
          unsigned long   const config0;
          avr32_config0_t const CONFIG0;
        };
        union {
          unsigned long   const config1;
          avr32_config1_t const CONFIG1;
        };

        /* 0x108 */
        unsigned long           count;
        unsigned long           compare;

        /* 0x110 */
        union {
          unsigned long         tlbehi;
          avr32_tlbehi_t        TLBEHI;
        };
        union {
          unsigned long         tlbelo;
          avr32_tlbelo_t        TLBELO;
        };
        unsigned long           ptbr;
        unsigned long     const tlbear;
        union {
          unsigned long         mmucr;
          avr32_mmucr_t         MMUCR;
        };
        unsigned long           tlbarlo;
        unsigned long           tlbarhi;

        /* 0x12C */
        unsigned long           pccnt;
        unsigned long           pcnt0;
        unsigned long           pcnt1;
        union {
          unsigned long         pccr;
          avr32_pccr_t          PCCR;
        };

        /* 0x13C */
        unsigned long     const bear;

        /* 0x140 */
        union {
          unsigned long         mpuar0;
          avr32_mpuar_t         MPUAR0;
        };
        union {
          unsigned long         mpuar1;
          avr32_mpuar_t         MPUAR1;
        };
        union {
          unsigned long         mpuar2;
          avr32_mpuar_t         MPUAR2;
        };
        union {
          unsigned long         mpuar3;
          avr32_mpuar_t         MPUAR3;
        };
        union {
          unsigned long         mpuar4;
          avr32_mpuar_t         MPUAR4;
        };
        union {
          unsigned long         mpuar5;
          avr32_mpuar_t         MPUAR5;
        };
        union {
          unsigned long         mpuar6;
          avr32_mpuar_t         MPUAR6;
        };
        union {
          unsigned long         mpuar7;
          avr32_mpuar_t         MPUAR7;
        };
        union {
          unsigned long         mpupsr0;
          avr32_mpupsr_t        MPUPSR0;
        };
        union {
          unsigned long         mpupsr1;
          avr32_mpupsr_t        MPUPSR1;
        };
        union {
          unsigned long         mpupsr2;
          avr32_mpupsr_t        MPUPSR2;
        };
        union {
          unsigned long         mpupsr3;
          avr32_mpupsr_t        MPUPSR3;
        };
        union {
          unsigned long         mpupsr4;
          avr32_mpupsr_t        MPUPSR4;
        };
        union {
          unsigned long         mpupsr5;
          avr32_mpupsr_t        MPUPSR5;
        };
        union {
          unsigned long         mpupsr6;
          avr32_mpupsr_t        MPUPSR6;
        };
        union {
          unsigned long         mpupsr7;
          avr32_mpupsr_t        MPUPSR7;
        };
        union {
          unsigned long         mpucra;
          avr32_mpucrx_t        MPUCRA;
        };
        union {
          unsigned long         mpucrb;
          avr32_mpucrx_t        MPUCRB;
        };
        union {
          unsigned long         mpubra;
          avr32_mpubrx_t        MPUBRA;
        };
        union {
          unsigned long         mpubrb;
          avr32_mpubrx_t        MPUBRB;
        };
        union {
          unsigned long         mpuapra;
          avr32_mpuaprx_t       MPUAPRA;
        };
        union {
          unsigned long         mpuaprb;
          avr32_mpuaprx_t       MPUAPRB;
        };
        union {
          unsigned long         mpucr;
          avr32_mpucr_t         MPUCR;
        };
} avr32_core_t;

extern __no_init __sysreg volatile avr32_core_t AVR32_CORE @ 0x000;

#pragma language=default
#endif  /* __IAR_SYSTEMS_ICC__  */

#define AVR32_CONFIG0_AR                                       10
#define AVR32_CONFIG0_AR_MASK                          0x00001c00
#define AVR32_CONFIG0_AR_OFFSET                                10
#define AVR32_CONFIG0_AR_SIZE                                   3
#define AVR32_CONFIG0_AT                                       13
#define AVR32_CONFIG0_AT_AVR32A                        0x00000000
#define AVR32_CONFIG0_AT_AVR32B                        0x00000001
#define AVR32_CONFIG0_AT_MASK                          0x0000e000
#define AVR32_CONFIG0_AT_OFFSET                                13
#define AVR32_CONFIG0_AT_SIZE                                   3
#define AVR32_CONFIG0_D                                         1
#define AVR32_CONFIG0_D_MASK                           0x00000002
#define AVR32_CONFIG0_D_OFFSET                                  1
#define AVR32_CONFIG0_D_SIZE                                    1
#define AVR32_CONFIG0_F                                         6
#define AVR32_CONFIG0_F_MASK                           0x00000040
#define AVR32_CONFIG0_F_OFFSET                                  6
#define AVR32_CONFIG0_F_SIZE                                    1
#define AVR32_CONFIG0_J                                         5
#define AVR32_CONFIG0_J_MASK                           0x00000020
#define AVR32_CONFIG0_J_OFFSET                                  5
#define AVR32_CONFIG0_J_SIZE                                    1
#define AVR32_CONFIG0_MMUT                                      7
#define AVR32_CONFIG0_MMUT_ITLB_DTLB                   0x00000001
#define AVR32_CONFIG0_MMUT_MASK                        0x00000380
#define AVR32_CONFIG0_MMUT_MPU                         0x00000003
#define AVR32_CONFIG0_MMUT_NONE                        0x00000000
#define AVR32_CONFIG0_MMUT_OFFSET                               7
#define AVR32_CONFIG0_MMUT_SHARED_TLB                  0x00000002
#define AVR32_CONFIG0_MMUT_SIZE                                 3
#define AVR32_CONFIG0_O                                         3
#define AVR32_CONFIG0_O_MASK                           0x00000008
#define AVR32_CONFIG0_O_OFFSET                                  3
#define AVR32_CONFIG0_O_SIZE                                    1
#define AVR32_CONFIG0_P                                         4
#define AVR32_CONFIG0_PROCESSORID                              24
#define AVR32_CONFIG0_PROCESSORID_AP7                  0x00000001
#define AVR32_CONFIG0_PROCESSORID_MASK                 0xff000000
#define AVR32_CONFIG0_PROCESSORID_OFFSET                       24
#define AVR32_CONFIG0_PROCESSORID_SIZE                          8
#define AVR32_CONFIG0_PROCESSORID_UC3                  0x00000002
#define AVR32_CONFIG0_PROCESSORREVISION                        16
#define AVR32_CONFIG0_PROCESSORREVISION_MASK           0x000f0000
#define AVR32_CONFIG0_PROCESSORREVISION_OFFSET                 16
#define AVR32_CONFIG0_PROCESSORREVISION_SIZE                    4
#define AVR32_CONFIG0_P_MASK                           0x00000010
#define AVR32_CONFIG0_P_OFFSET                                  4
#define AVR32_CONFIG0_P_SIZE                                    1
#define AVR32_CONFIG0_R                                         0
#define AVR32_CONFIG0_R_MASK                           0x00000001
#define AVR32_CONFIG0_R_OFFSET                                  0
#define AVR32_CONFIG0_R_SIZE                                    1
#define AVR32_CONFIG0_S                                         2
#define AVR32_CONFIG0_S_MASK                           0x00000004
#define AVR32_CONFIG0_S_OFFSET                                  2
#define AVR32_CONFIG0_S_SIZE                                    1

#define AVR32_CONFIG1_DASS                                      0
#define AVR32_CONFIG1_DASS_MASK                        0x00000007
#define AVR32_CONFIG1_DASS_OFFSET                               0
#define AVR32_CONFIG1_DASS_SIZE                                 3
#define AVR32_CONFIG1_DLSZ                                      3
#define AVR32_CONFIG1_DLSZ_MASK                        0x00000038
#define AVR32_CONFIG1_DLSZ_OFFSET                               3
#define AVR32_CONFIG1_DLSZ_SIZE                                 3
#define AVR32_CONFIG1_DMMUSZ                                   20
#define AVR32_CONFIG1_DMMUSZ_MASK                      0x03f00000
#define AVR32_CONFIG1_DMMUSZ_OFFSET                            20
#define AVR32_CONFIG1_DMMUSZ_SIZE                               6
#define AVR32_CONFIG1_DSET                                      6
#define AVR32_CONFIG1_DSET_MASK                        0x000003c0
#define AVR32_CONFIG1_DSET_OFFSET                               6
#define AVR32_CONFIG1_DSET_SIZE                                 4
#define AVR32_CONFIG1_IASS                                     10
#define AVR32_CONFIG1_IASS_MASK                        0x00001c00
#define AVR32_CONFIG1_IASS_OFFSET                              10
#define AVR32_CONFIG1_IASS_SIZE                                 3
#define AVR32_CONFIG1_ILSZ                                     13
#define AVR32_CONFIG1_ILSZ_MASK                        0x0000e000
#define AVR32_CONFIG1_ILSZ_OFFSET                              13
#define AVR32_CONFIG1_ILSZ_SIZE                                 3
#define AVR32_CONFIG1_IMMUSZ                                   26
#define AVR32_CONFIG1_IMMUSZ_MASK                      0xfc000000
#define AVR32_CONFIG1_IMMUSZ_OFFSET                            26
#define AVR32_CONFIG1_IMMUSZ_SIZE                               6
#define AVR32_CONFIG1_ISET                                     16
#define AVR32_CONFIG1_ISET_MASK                        0x000f0000
#define AVR32_CONFIG1_ISET_OFFSET                              16
#define AVR32_CONFIG1_ISET_SIZE                                 4

#if __CORE__ == __AVR32A__
#define AVR32_CPUCR_COP                                         1
#define AVR32_CPUCR_COP_MASK                           0x0000003e
#define AVR32_CPUCR_COP_OFFSET                                  1
#define AVR32_CPUCR_COP_SIZE                                    5
#define AVR32_CPUCR_CPL                                         6
#define AVR32_CPUCR_CPL_MASK                           0x000007c0
#define AVR32_CPUCR_CPL_OFFSET                                  6
#define AVR32_CPUCR_CPL_SIZE                                    5
#define AVR32_CPUCR_LOCEN                                      16
#define AVR32_CPUCR_LOCEN_MASK                         0x00010000
#define AVR32_CPUCR_LOCEN_OFFSET                               16
#define AVR32_CPUCR_LOCEN_SIZE                                  1
#define AVR32_CPUCR_SIE                                         0
#define AVR32_CPUCR_SIE_MASK                           0x00000001
#define AVR32_CPUCR_SIE_OFFSET                                  0
#define AVR32_CPUCR_SIE_SIZE                                    1
#define AVR32_CPUCR_SPL                                        11
#define AVR32_CPUCR_SPL_MASK                           0x0000f800
#define AVR32_CPUCR_SPL_OFFSET                                 11
#define AVR32_CPUCR_SPL_SIZE                                    5
#elif __CORE__ == __AVR32B__
#define AVR32_CPUCR_BE                                          1
#define AVR32_CPUCR_BE_MASK                            0x00000002
#define AVR32_CPUCR_BE_OFFSET                                   1
#define AVR32_CPUCR_BE_SIZE                                     1
#define AVR32_CPUCR_BI                                          0
#define AVR32_CPUCR_BI_MASK                            0x00000001
#define AVR32_CPUCR_BI_OFFSET                                   0
#define AVR32_CPUCR_BI_SIZE                                     1
#define AVR32_CPUCR_COP0EN                                     24
#define AVR32_CPUCR_COP0EN_MASK                        0x01000000
#define AVR32_CPUCR_COP0EN_OFFSET                              24
#define AVR32_CPUCR_COP0EN_SIZE                                 1
#define AVR32_CPUCR_COP1EN                                     25
#define AVR32_CPUCR_COP1EN_MASK                        0x02000000
#define AVR32_CPUCR_COP1EN_OFFSET                              25
#define AVR32_CPUCR_COP1EN_SIZE                                 1
#define AVR32_CPUCR_COP2EN                                     26
#define AVR32_CPUCR_COP2EN_MASK                        0x04000000
#define AVR32_CPUCR_COP2EN_OFFSET                              26
#define AVR32_CPUCR_COP2EN_SIZE                                 1
#define AVR32_CPUCR_COP3EN                                     27
#define AVR32_CPUCR_COP3EN_MASK                        0x08000000
#define AVR32_CPUCR_COP3EN_OFFSET                              27
#define AVR32_CPUCR_COP3EN_SIZE                                 1
#define AVR32_CPUCR_COP4EN                                     28
#define AVR32_CPUCR_COP4EN_MASK                        0x10000000
#define AVR32_CPUCR_COP4EN_OFFSET                              28
#define AVR32_CPUCR_COP4EN_SIZE                                 1
#define AVR32_CPUCR_COP5EN                                     29
#define AVR32_CPUCR_COP5EN_MASK                        0x20000000
#define AVR32_CPUCR_COP5EN_OFFSET                              29
#define AVR32_CPUCR_COP5EN_SIZE                                 1
#define AVR32_CPUCR_COP6EN                                     30
#define AVR32_CPUCR_COP6EN_MASK                        0x40000000
#define AVR32_CPUCR_COP6EN_OFFSET                              30
#define AVR32_CPUCR_COP6EN_SIZE                                 1
#define AVR32_CPUCR_COP7EN                                     31
#define AVR32_CPUCR_COP7EN_MASK                        0x80000000
#define AVR32_CPUCR_COP7EN_OFFSET                              31
#define AVR32_CPUCR_COP7EN_SIZE                                 1
#define AVR32_CPUCR_FE                                          2
#define AVR32_CPUCR_FE_MASK                            0x00000004
#define AVR32_CPUCR_FE_OFFSET                                   2
#define AVR32_CPUCR_FE_SIZE                                     1
#define AVR32_CPUCR_IBE                                         4
#define AVR32_CPUCR_IBE_MASK                           0x00000010
#define AVR32_CPUCR_IBE_OFFSET                                  4
#define AVR32_CPUCR_IBE_SIZE                                    1
#define AVR32_CPUCR_IEE                                         5
#define AVR32_CPUCR_IEE_MASK                           0x00000020
#define AVR32_CPUCR_IEE_OFFSET                                  5
#define AVR32_CPUCR_IEE_SIZE                                    1
#define AVR32_CPUCR_IM                                          6
#define AVR32_CPUCR_IM_MASK                            0x00000040
#define AVR32_CPUCR_IM_OFFSET                                   6
#define AVR32_CPUCR_IM_SIZE                                     1
#define AVR32_CPUCR_RE                                          3
#define AVR32_CPUCR_RE_MASK                            0x00000008
#define AVR32_CPUCR_RE_OFFSET                                   3
#define AVR32_CPUCR_RE_SIZE                                     1
#endif

#define AVR32_ECR_ECR                                           0
#define AVR32_ECR_ECR_ADDR_ALIGN_R                     0x0000000d
#define AVR32_ECR_ECR_ADDR_ALIGN_W                     0x0000000e
#define AVR32_ECR_ECR_ADDR_ALIGN_X                     0x00000005
#define AVR32_ECR_ECR_BUS_ERROR_READ                   0x00000003
#define AVR32_ECR_ECR_BUS_ERROR_WRITE                  0x00000002
#define AVR32_ECR_ECR_COPROC_ABSENT                    0x0000000c
#define AVR32_ECR_ECR_DEBUG                            0x00000007
#define AVR32_ECR_ECR_DTLB_MODIFIED                    0x00000011
#define AVR32_ECR_ECR_FPE                              0x0000000b
#define AVR32_ECR_ECR_ILLEGAL_OPCODE                   0x00000008
#define AVR32_ECR_ECR_MASK                             0xffffffff
#define AVR32_ECR_ECR_NMI                              0x00000004
#define AVR32_ECR_ECR_OFFSET                                    0
#define AVR32_ECR_ECR_PRIVILEGE_VIOLATION              0x0000000a
#define AVR32_ECR_ECR_PROTECTION_R                     0x0000000f
#define AVR32_ECR_ECR_PROTECTION_W                     0x00000010
#define AVR32_ECR_ECR_PROTECTION_X                     0x00000006
#define AVR32_ECR_ECR_SIZE                                     32
#define AVR32_ECR_ECR_TLB_MISS_R                       0x00000018
#define AVR32_ECR_ECR_TLB_MISS_W                       0x0000001c
#define AVR32_ECR_ECR_TLB_MISS_X                       0x00000014
#define AVR32_ECR_ECR_TLB_MULTIPLE                     0x00000001
#define AVR32_ECR_ECR_UNIMPL_INSTRUCTION               0x00000009
#define AVR32_ECR_ECR_UNRECOVERABLE                    0x00000000

#define AVR32_MMUCR_DLA                                         8
#define AVR32_MMUCR_DLA_MASK                           0x00003f00
#define AVR32_MMUCR_DLA_OFFSET                                  8
#define AVR32_MMUCR_DLA_SIZE                                    6
#define AVR32_MMUCR_DRP                                        14
#define AVR32_MMUCR_DRP_MASK                           0x000fc000
#define AVR32_MMUCR_DRP_OFFSET                                 14
#define AVR32_MMUCR_DRP_SIZE                                    6
#define AVR32_MMUCR_E                                           0
#define AVR32_MMUCR_E_MASK                             0x00000001
#define AVR32_MMUCR_E_OFFSET                                    0
#define AVR32_MMUCR_E_SIZE                                      1
#define AVR32_MMUCR_I                                           2
#define AVR32_MMUCR_I_MASK                             0x00000004
#define AVR32_MMUCR_I_OFFSET                                    2
#define AVR32_MMUCR_I_SIZE                                      1
#define AVR32_MMUCR_M                                           1
#define AVR32_MMUCR_M_MASK                             0x00000002
#define AVR32_MMUCR_M_OFFSET                                    1
#define AVR32_MMUCR_M_PRIVATE                          0x00000000
#define AVR32_MMUCR_M_SHARED                           0x00000001
#define AVR32_MMUCR_M_SIZE                                      1
#define AVR32_MMUCR_N                                           3
#define AVR32_MMUCR_N_MASK                             0x00000008
#define AVR32_MMUCR_N_OFFSET                                    3
#define AVR32_MMUCR_N_SIZE                                      1
#define AVR32_MMUCR_S                                           4
#define AVR32_MMUCR_S_MASK                             0x00000010
#define AVR32_MMUCR_S_OFFSET                                    4
#define AVR32_MMUCR_S_SIZE                                      1

#define AVR32_MPUAPRX_AP0                                       0
#define AVR32_MPUAPRX_AP0_MASK                         0x0000000f
#define AVR32_MPUAPRX_AP0_OFFSET                                0
#define AVR32_MPUAPRX_AP0_PRIV_NONE_UNPRIV_NONE        0x0000000a
#define AVR32_MPUAPRX_AP0_PRIV_RWX_UNPRIV_NONE         0x00000003
#define AVR32_MPUAPRX_AP0_PRIV_RWX_UNPRIV_RWX          0x00000007
#define AVR32_MPUAPRX_AP0_PRIV_RW_UNPRIV_NONE          0x00000002
#define AVR32_MPUAPRX_AP0_PRIV_RW_UNPRIV_R             0x00000008
#define AVR32_MPUAPRX_AP0_PRIV_RW_UNPRIV_RW            0x00000006
#define AVR32_MPUAPRX_AP0_PRIV_RW_UNPRIV_RX            0x00000009
#define AVR32_MPUAPRX_AP0_PRIV_RX_UNPRIV_NONE          0x00000001
#define AVR32_MPUAPRX_AP0_PRIV_RX_UNPRIV_RX            0x00000005
#define AVR32_MPUAPRX_AP0_PRIV_R_UNPRIV_NONE           0x00000000
#define AVR32_MPUAPRX_AP0_PRIV_R_UNPRIV_R              0x00000004
#define AVR32_MPUAPRX_AP0_SIZE                                  4
#define AVR32_MPUAPRX_AP1                                       4
#define AVR32_MPUAPRX_AP1_MASK                         0x000000f0
#define AVR32_MPUAPRX_AP1_OFFSET                                4
#define AVR32_MPUAPRX_AP1_PRIV_NONE_UNPRIV_NONE        0x0000000a
#define AVR32_MPUAPRX_AP1_PRIV_RWX_UNPRIV_NONE         0x00000003
#define AVR32_MPUAPRX_AP1_PRIV_RWX_UNPRIV_RWX          0x00000007
#define AVR32_MPUAPRX_AP1_PRIV_RW_UNPRIV_NONE          0x00000002
#define AVR32_MPUAPRX_AP1_PRIV_RW_UNPRIV_R             0x00000008
#define AVR32_MPUAPRX_AP1_PRIV_RW_UNPRIV_RW            0x00000006
#define AVR32_MPUAPRX_AP1_PRIV_RW_UNPRIV_RX            0x00000009
#define AVR32_MPUAPRX_AP1_PRIV_RX_UNPRIV_NONE          0x00000001
#define AVR32_MPUAPRX_AP1_PRIV_RX_UNPRIV_RX            0x00000005
#define AVR32_MPUAPRX_AP1_PRIV_R_UNPRIV_NONE           0x00000000
#define AVR32_MPUAPRX_AP1_PRIV_R_UNPRIV_R              0x00000004
#define AVR32_MPUAPRX_AP1_SIZE                                  4
#define AVR32_MPUAPRX_AP2                                       8
#define AVR32_MPUAPRX_AP2_MASK                         0x00000f00
#define AVR32_MPUAPRX_AP2_OFFSET                                8
#define AVR32_MPUAPRX_AP2_PRIV_NONE_UNPRIV_NONE        0x0000000a
#define AVR32_MPUAPRX_AP2_PRIV_RWX_UNPRIV_NONE         0x00000003
#define AVR32_MPUAPRX_AP2_PRIV_RWX_UNPRIV_RWX          0x00000007
#define AVR32_MPUAPRX_AP2_PRIV_RW_UNPRIV_NONE          0x00000002
#define AVR32_MPUAPRX_AP2_PRIV_RW_UNPRIV_R             0x00000008
#define AVR32_MPUAPRX_AP2_PRIV_RW_UNPRIV_RW            0x00000006
#define AVR32_MPUAPRX_AP2_PRIV_RW_UNPRIV_RX            0x00000009
#define AVR32_MPUAPRX_AP2_PRIV_RX_UNPRIV_NONE          0x00000001
#define AVR32_MPUAPRX_AP2_PRIV_RX_UNPRIV_RX            0x00000005
#define AVR32_MPUAPRX_AP2_PRIV_R_UNPRIV_NONE           0x00000000
#define AVR32_MPUAPRX_AP2_PRIV_R_UNPRIV_R              0x00000004
#define AVR32_MPUAPRX_AP2_SIZE                                  4
#define AVR32_MPUAPRX_AP3                                      12
#define AVR32_MPUAPRX_AP3_MASK                         0x0000f000
#define AVR32_MPUAPRX_AP3_OFFSET                               12
#define AVR32_MPUAPRX_AP3_PRIV_NONE_UNPRIV_NONE        0x0000000a
#define AVR32_MPUAPRX_AP3_PRIV_RWX_UNPRIV_NONE         0x00000003
#define AVR32_MPUAPRX_AP3_PRIV_RWX_UNPRIV_RWX          0x00000007
#define AVR32_MPUAPRX_AP3_PRIV_RW_UNPRIV_NONE          0x00000002
#define AVR32_MPUAPRX_AP3_PRIV_RW_UNPRIV_R             0x00000008
#define AVR32_MPUAPRX_AP3_PRIV_RW_UNPRIV_RW            0x00000006
#define AVR32_MPUAPRX_AP3_PRIV_RW_UNPRIV_RX            0x00000009
#define AVR32_MPUAPRX_AP3_PRIV_RX_UNPRIV_NONE          0x00000001
#define AVR32_MPUAPRX_AP3_PRIV_RX_UNPRIV_RX            0x00000005
#define AVR32_MPUAPRX_AP3_PRIV_R_UNPRIV_NONE           0x00000000
#define AVR32_MPUAPRX_AP3_PRIV_R_UNPRIV_R              0x00000004
#define AVR32_MPUAPRX_AP3_SIZE                                  4
#define AVR32_MPUAPRX_AP4                                      16
#define AVR32_MPUAPRX_AP4_MASK                         0x000f0000
#define AVR32_MPUAPRX_AP4_OFFSET                               16
#define AVR32_MPUAPRX_AP4_PRIV_NONE_UNPRIV_NONE        0x0000000a
#define AVR32_MPUAPRX_AP4_PRIV_RWX_UNPRIV_NONE         0x00000003
#define AVR32_MPUAPRX_AP4_PRIV_RWX_UNPRIV_RWX          0x00000007
#define AVR32_MPUAPRX_AP4_PRIV_RW_UNPRIV_NONE          0x00000002
#define AVR32_MPUAPRX_AP4_PRIV_RW_UNPRIV_R             0x00000008
#define AVR32_MPUAPRX_AP4_PRIV_RW_UNPRIV_RW            0x00000006
#define AVR32_MPUAPRX_AP4_PRIV_RW_UNPRIV_RX            0x00000009
#define AVR32_MPUAPRX_AP4_PRIV_RX_UNPRIV_NONE          0x00000001
#define AVR32_MPUAPRX_AP4_PRIV_RX_UNPRIV_RX            0x00000005
#define AVR32_MPUAPRX_AP4_PRIV_R_UNPRIV_NONE           0x00000000
#define AVR32_MPUAPRX_AP4_PRIV_R_UNPRIV_R              0x00000004
#define AVR32_MPUAPRX_AP4_SIZE                                  4
#define AVR32_MPUAPRX_AP5                                      20
#define AVR32_MPUAPRX_AP5_MASK                         0x00f00000
#define AVR32_MPUAPRX_AP5_OFFSET                               20
#define AVR32_MPUAPRX_AP5_PRIV_NONE_UNPRIV_NONE        0x0000000a
#define AVR32_MPUAPRX_AP5_PRIV_RWX_UNPRIV_NONE         0x00000003
#define AVR32_MPUAPRX_AP5_PRIV_RWX_UNPRIV_RWX          0x00000007
#define AVR32_MPUAPRX_AP5_PRIV_RW_UNPRIV_NONE          0x00000002
#define AVR32_MPUAPRX_AP5_PRIV_RW_UNPRIV_R             0x00000008
#define AVR32_MPUAPRX_AP5_PRIV_RW_UNPRIV_RW            0x00000006
#define AVR32_MPUAPRX_AP5_PRIV_RW_UNPRIV_RX            0x00000009
#define AVR32_MPUAPRX_AP5_PRIV_RX_UNPRIV_NONE          0x00000001
#define AVR32_MPUAPRX_AP5_PRIV_RX_UNPRIV_RX            0x00000005
#define AVR32_MPUAPRX_AP5_PRIV_R_UNPRIV_NONE           0x00000000
#define AVR32_MPUAPRX_AP5_PRIV_R_UNPRIV_R              0x00000004
#define AVR32_MPUAPRX_AP5_SIZE                                  4
#define AVR32_MPUAPRX_AP6                                      24
#define AVR32_MPUAPRX_AP6_MASK                         0x0f000000
#define AVR32_MPUAPRX_AP6_OFFSET                               24
#define AVR32_MPUAPRX_AP6_PRIV_NONE_UNPRIV_NONE        0x0000000a
#define AVR32_MPUAPRX_AP6_PRIV_RWX_UNPRIV_NONE         0x00000003
#define AVR32_MPUAPRX_AP6_PRIV_RWX_UNPRIV_RWX          0x00000007
#define AVR32_MPUAPRX_AP6_PRIV_RW_UNPRIV_NONE          0x00000002
#define AVR32_MPUAPRX_AP6_PRIV_RW_UNPRIV_R             0x00000008
#define AVR32_MPUAPRX_AP6_PRIV_RW_UNPRIV_RW            0x00000006
#define AVR32_MPUAPRX_AP6_PRIV_RW_UNPRIV_RX            0x00000009
#define AVR32_MPUAPRX_AP6_PRIV_RX_UNPRIV_NONE          0x00000001
#define AVR32_MPUAPRX_AP6_PRIV_RX_UNPRIV_RX            0x00000005
#define AVR32_MPUAPRX_AP6_PRIV_R_UNPRIV_NONE           0x00000000
#define AVR32_MPUAPRX_AP6_PRIV_R_UNPRIV_R              0x00000004
#define AVR32_MPUAPRX_AP6_SIZE                                  4
#define AVR32_MPUAPRX_AP7                                      28
#define AVR32_MPUAPRX_AP7_MASK                         0xf0000000
#define AVR32_MPUAPRX_AP7_OFFSET                               28
#define AVR32_MPUAPRX_AP7_PRIV_NONE_UNPRIV_NONE        0x0000000a
#define AVR32_MPUAPRX_AP7_PRIV_RWX_UNPRIV_NONE         0x00000003
#define AVR32_MPUAPRX_AP7_PRIV_RWX_UNPRIV_RWX          0x00000007
#define AVR32_MPUAPRX_AP7_PRIV_RW_UNPRIV_NONE          0x00000002
#define AVR32_MPUAPRX_AP7_PRIV_RW_UNPRIV_R             0x00000008
#define AVR32_MPUAPRX_AP7_PRIV_RW_UNPRIV_RW            0x00000006
#define AVR32_MPUAPRX_AP7_PRIV_RW_UNPRIV_RX            0x00000009
#define AVR32_MPUAPRX_AP7_PRIV_RX_UNPRIV_NONE          0x00000001
#define AVR32_MPUAPRX_AP7_PRIV_RX_UNPRIV_RX            0x00000005
#define AVR32_MPUAPRX_AP7_PRIV_R_UNPRIV_NONE           0x00000000
#define AVR32_MPUAPRX_AP7_PRIV_R_UNPRIV_R              0x00000004
#define AVR32_MPUAPRX_AP7_SIZE                                  4

#define AVR32_MPUAR_BASE                                       12
#define AVR32_MPUAR_BASE_MASK                          0xfffff000
#define AVR32_MPUAR_BASE_OFFSET                                12
#define AVR32_MPUAR_BASE_SIZE                                  20
#define AVR32_MPUAR_SIZE                                        1
#define AVR32_MPUAR_SIZE_MASK                          0x0000003e
#define AVR32_MPUAR_SIZE_OFFSET                                 1
#define AVR32_MPUAR_SIZE_SIZE                                   5
#define AVR32_MPUAR_V                                           0
#define AVR32_MPUAR_V_MASK                             0x00000001
#define AVR32_MPUAR_V_OFFSET                                    0
#define AVR32_MPUAR_V_SIZE                                      1

#define AVR32_MPUBRX_B0                                         0
#define AVR32_MPUBRX_B0_MASK                           0x00000001
#define AVR32_MPUBRX_B0_OFFSET                                  0
#define AVR32_MPUBRX_B0_SIZE                                    1
#define AVR32_MPUBRX_B1                                         1
#define AVR32_MPUBRX_B1_MASK                           0x00000002
#define AVR32_MPUBRX_B1_OFFSET                                  1
#define AVR32_MPUBRX_B1_SIZE                                    1
#define AVR32_MPUBRX_B2                                         2
#define AVR32_MPUBRX_B2_MASK                           0x00000004
#define AVR32_MPUBRX_B2_OFFSET                                  2
#define AVR32_MPUBRX_B2_SIZE                                    1
#define AVR32_MPUBRX_B3                                         3
#define AVR32_MPUBRX_B3_MASK                           0x00000008
#define AVR32_MPUBRX_B3_OFFSET                                  3
#define AVR32_MPUBRX_B3_SIZE                                    1
#define AVR32_MPUBRX_B4                                         4
#define AVR32_MPUBRX_B4_MASK                           0x00000010
#define AVR32_MPUBRX_B4_OFFSET                                  4
#define AVR32_MPUBRX_B4_SIZE                                    1
#define AVR32_MPUBRX_B5                                         5
#define AVR32_MPUBRX_B5_MASK                           0x00000020
#define AVR32_MPUBRX_B5_OFFSET                                  5
#define AVR32_MPUBRX_B5_SIZE                                    1
#define AVR32_MPUBRX_B6                                         6
#define AVR32_MPUBRX_B6_MASK                           0x00000040
#define AVR32_MPUBRX_B6_OFFSET                                  6
#define AVR32_MPUBRX_B6_SIZE                                    1
#define AVR32_MPUBRX_B7                                         7
#define AVR32_MPUBRX_B7_MASK                           0x00000080
#define AVR32_MPUBRX_B7_OFFSET                                  7
#define AVR32_MPUBRX_B7_SIZE                                    1

#define AVR32_MPUCRX_C0                                         0
#define AVR32_MPUCRX_C0_MASK                           0x00000001
#define AVR32_MPUCRX_C0_OFFSET                                  0
#define AVR32_MPUCRX_C0_SIZE                                    1
#define AVR32_MPUCRX_C1                                         1
#define AVR32_MPUCRX_C1_MASK                           0x00000002
#define AVR32_MPUCRX_C1_OFFSET                                  1
#define AVR32_MPUCRX_C1_SIZE                                    1
#define AVR32_MPUCRX_C2                                         2
#define AVR32_MPUCRX_C2_MASK                           0x00000004
#define AVR32_MPUCRX_C2_OFFSET                                  2
#define AVR32_MPUCRX_C2_SIZE                                    1
#define AVR32_MPUCRX_C3                                         3
#define AVR32_MPUCRX_C3_MASK                           0x00000008
#define AVR32_MPUCRX_C3_OFFSET                                  3
#define AVR32_MPUCRX_C3_SIZE                                    1
#define AVR32_MPUCRX_C4                                         4
#define AVR32_MPUCRX_C4_MASK                           0x00000010
#define AVR32_MPUCRX_C4_OFFSET                                  4
#define AVR32_MPUCRX_C4_SIZE                                    1
#define AVR32_MPUCRX_C5                                         5
#define AVR32_MPUCRX_C5_MASK                           0x00000020
#define AVR32_MPUCRX_C5_OFFSET                                  5
#define AVR32_MPUCRX_C5_SIZE                                    1
#define AVR32_MPUCRX_C6                                         6
#define AVR32_MPUCRX_C6_MASK                           0x00000040
#define AVR32_MPUCRX_C6_OFFSET                                  6
#define AVR32_MPUCRX_C6_SIZE                                    1
#define AVR32_MPUCRX_C7                                         7
#define AVR32_MPUCRX_C7_MASK                           0x00000080
#define AVR32_MPUCRX_C7_OFFSET                                  7
#define AVR32_MPUCRX_C7_SIZE                                    1

#define AVR32_MPUCR_E                                           0
#define AVR32_MPUCR_E_MASK                             0x00000001
#define AVR32_MPUCR_E_OFFSET                                    0
#define AVR32_MPUCR_E_SIZE                                      1

#define AVR32_MPUPSR_P0                                         0
#define AVR32_MPUPSR_P0_MASK                           0x00000001
#define AVR32_MPUPSR_P0_OFFSET                                  0
#define AVR32_MPUPSR_P0_SIZE                                    1
#define AVR32_MPUPSR_P1                                         1
#define AVR32_MPUPSR_P10                                       10
#define AVR32_MPUPSR_P10_MASK                          0x00000400
#define AVR32_MPUPSR_P10_OFFSET                                10
#define AVR32_MPUPSR_P10_SIZE                                   1
#define AVR32_MPUPSR_P11                                       11
#define AVR32_MPUPSR_P11_MASK                          0x00000800
#define AVR32_MPUPSR_P11_OFFSET                                11
#define AVR32_MPUPSR_P11_SIZE                                   1
#define AVR32_MPUPSR_P12                                       12
#define AVR32_MPUPSR_P12_MASK                          0x00001000
#define AVR32_MPUPSR_P12_OFFSET                                12
#define AVR32_MPUPSR_P12_SIZE                                   1
#define AVR32_MPUPSR_P13                                       13
#define AVR32_MPUPSR_P13_MASK                          0x00002000
#define AVR32_MPUPSR_P13_OFFSET                                13
#define AVR32_MPUPSR_P13_SIZE                                   1
#define AVR32_MPUPSR_P14                                       14
#define AVR32_MPUPSR_P14_MASK                          0x00004000
#define AVR32_MPUPSR_P14_OFFSET                                14
#define AVR32_MPUPSR_P14_SIZE                                   1
#define AVR32_MPUPSR_P15                                       15
#define AVR32_MPUPSR_P15_MASK                          0x00008000
#define AVR32_MPUPSR_P15_OFFSET                                15
#define AVR32_MPUPSR_P15_SIZE                                   1
#define AVR32_MPUPSR_P1_MASK                           0x00000002
#define AVR32_MPUPSR_P1_OFFSET                                  1
#define AVR32_MPUPSR_P1_SIZE                                    1
#define AVR32_MPUPSR_P2                                         2
#define AVR32_MPUPSR_P2_MASK                           0x00000004
#define AVR32_MPUPSR_P2_OFFSET                                  2
#define AVR32_MPUPSR_P2_SIZE                                    1
#define AVR32_MPUPSR_P3                                         3
#define AVR32_MPUPSR_P3_MASK                           0x00000008
#define AVR32_MPUPSR_P3_OFFSET                                  3
#define AVR32_MPUPSR_P3_SIZE                                    1
#define AVR32_MPUPSR_P4                                         4
#define AVR32_MPUPSR_P4_MASK                           0x00000010
#define AVR32_MPUPSR_P4_OFFSET                                  4
#define AVR32_MPUPSR_P4_SIZE                                    1
#define AVR32_MPUPSR_P5                                         5
#define AVR32_MPUPSR_P5_MASK                           0x00000020
#define AVR32_MPUPSR_P5_OFFSET                                  5
#define AVR32_MPUPSR_P5_SIZE                                    1
#define AVR32_MPUPSR_P6                                         6
#define AVR32_MPUPSR_P6_MASK                           0x00000040
#define AVR32_MPUPSR_P6_OFFSET                                  6
#define AVR32_MPUPSR_P6_SIZE                                    1
#define AVR32_MPUPSR_P7                                         7
#define AVR32_MPUPSR_P7_MASK                           0x00000080
#define AVR32_MPUPSR_P7_OFFSET                                  7
#define AVR32_MPUPSR_P7_SIZE                                    1
#define AVR32_MPUPSR_P8                                         8
#define AVR32_MPUPSR_P8_MASK                           0x00000100
#define AVR32_MPUPSR_P8_OFFSET                                  8
#define AVR32_MPUPSR_P8_SIZE                                    1
#define AVR32_MPUPSR_P9                                         9
#define AVR32_MPUPSR_P9_MASK                           0x00000200
#define AVR32_MPUPSR_P9_OFFSET                                  9
#define AVR32_MPUPSR_P9_SIZE                                    1

#define AVR32_PCCR_C                                            2
#define AVR32_PCCR_CONF0                                       12
#define AVR32_PCCR_CONF0_ACACHE_HIT                    0x00000015
#define AVR32_PCCR_CONF0_ACACHE_MISS                   0x00000016
#define AVR32_PCCR_CONF0_BR_EXECUTED                   0x00000005
#define AVR32_PCCR_CONF0_BR_MISPREDICTED               0x00000006
#define AVR32_PCCR_CONF0_BTB_HIT                       0x00000017
#define AVR32_PCCR_CONF0_CACHE_ACCESS_CYC              0x00000013
#define AVR32_PCCR_CONF0_CACHE_ACCESS_OCC              0x00000012
#define AVR32_PCCR_CONF0_CACHE_STALL_CYC               0x00000011
#define AVR32_PCCR_CONF0_CACHE_STALL_OCC               0x00000010
#define AVR32_PCCR_CONF0_DCACHE_LINE_WB                0x00000014
#define AVR32_PCCR_CONF0_DCACHE_READ_MISS_CYC          0x0000000b
#define AVR32_PCCR_CONF0_DCACHE_READ_MISS_OCC          0x0000000a
#define AVR32_PCCR_CONF0_DCACHE_WBUFF_FULL_STALL_CYC   0x00000009
#define AVR32_PCCR_CONF0_DCACHE_WBUFF_FULL_STALL_OCC   0x00000008
#define AVR32_PCCR_CONF0_DHAZARD_STALL                 0x00000002
#define AVR32_PCCR_CONF0_DTLB_MISS                     0x00000004
#define AVR32_PCCR_CONF0_DUTLB_MISS                    0x00000019
#define AVR32_PCCR_CONF0_ICACHE_MISS                   0x00000000
#define AVR32_PCCR_CONF0_IFETCH_STALL                  0x00000001
#define AVR32_PCCR_CONF0_INSN_EXECUTED                 0x00000007
#define AVR32_PCCR_CONF0_ITLB_MISS                     0x00000003
#define AVR32_PCCR_CONF0_IUTLB_MISS                    0x00000018
#define AVR32_PCCR_CONF0_MASK                          0x0003f000
#define AVR32_PCCR_CONF0_OFFSET                                12
#define AVR32_PCCR_CONF0_RACCESS_CYC                   0x0000000f
#define AVR32_PCCR_CONF0_RACCESS_OCC                   0x0000000e
#define AVR32_PCCR_CONF0_SIZE                                   6
#define AVR32_PCCR_CONF0_WACCESS_CYC                   0x0000000d
#define AVR32_PCCR_CONF0_WACCESS_OCC                   0x0000000c
#define AVR32_PCCR_CONF1                                       18
#define AVR32_PCCR_CONF1_ACACHE_HIT                    0x00000015
#define AVR32_PCCR_CONF1_ACACHE_MISS                   0x00000016
#define AVR32_PCCR_CONF1_BR_EXECUTED                   0x00000005
#define AVR32_PCCR_CONF1_BR_MISPREDICTED               0x00000006
#define AVR32_PCCR_CONF1_BTB_HIT                       0x00000017
#define AVR32_PCCR_CONF1_CACHE_ACCESS_CYC              0x00000013
#define AVR32_PCCR_CONF1_CACHE_ACCESS_OCC              0x00000012
#define AVR32_PCCR_CONF1_CACHE_STALL_CYC               0x00000011
#define AVR32_PCCR_CONF1_CACHE_STALL_OCC               0x00000010
#define AVR32_PCCR_CONF1_DCACHE_LINE_WB                0x00000014
#define AVR32_PCCR_CONF1_DCACHE_READ_MISS_CYC          0x0000000b
#define AVR32_PCCR_CONF1_DCACHE_READ_MISS_OCC          0x0000000a
#define AVR32_PCCR_CONF1_DCACHE_WBUFF_FULL_STALL_CYC   0x00000009
#define AVR32_PCCR_CONF1_DCACHE_WBUFF_FULL_STALL_OCC   0x00000008
#define AVR32_PCCR_CONF1_DHAZARD_STALL                 0x00000002
#define AVR32_PCCR_CONF1_DTLB_MISS                     0x00000004
#define AVR32_PCCR_CONF1_DUTLB_MISS                    0x00000019
#define AVR32_PCCR_CONF1_ICACHE_MISS                   0x00000000
#define AVR32_PCCR_CONF1_IFETCH_STALL                  0x00000001
#define AVR32_PCCR_CONF1_INSN_EXECUTED                 0x00000007
#define AVR32_PCCR_CONF1_ITLB_MISS                     0x00000003
#define AVR32_PCCR_CONF1_IUTLB_MISS                    0x00000018
#define AVR32_PCCR_CONF1_MASK                          0x00fc0000
#define AVR32_PCCR_CONF1_OFFSET                                18
#define AVR32_PCCR_CONF1_RACCESS_CYC                   0x0000000f
#define AVR32_PCCR_CONF1_RACCESS_OCC                   0x0000000e
#define AVR32_PCCR_CONF1_SIZE                                   6
#define AVR32_PCCR_CONF1_WACCESS_CYC                   0x0000000d
#define AVR32_PCCR_CONF1_WACCESS_OCC                   0x0000000c
#define AVR32_PCCR_C_MASK                              0x00000004
#define AVR32_PCCR_C_OFFSET                                     2
#define AVR32_PCCR_C_SIZE                                       1
#define AVR32_PCCR_E                                            0
#define AVR32_PCCR_E_MASK                              0x00000001
#define AVR32_PCCR_E_OFFSET                                     0
#define AVR32_PCCR_E_SIZE                                       1
#define AVR32_PCCR_F0                                           9
#define AVR32_PCCR_F0_MASK                             0x00000200
#define AVR32_PCCR_F0_OFFSET                                    9
#define AVR32_PCCR_F0_SIZE                                      1
#define AVR32_PCCR_F1                                          10
#define AVR32_PCCR_F1_MASK                             0x00000400
#define AVR32_PCCR_F1_OFFSET                                   10
#define AVR32_PCCR_F1_SIZE                                      1
#define AVR32_PCCR_FC                                           8
#define AVR32_PCCR_FC_MASK                             0x00000100
#define AVR32_PCCR_FC_OFFSET                                    8
#define AVR32_PCCR_FC_SIZE                                      1
#define AVR32_PCCR_IE0                                          5
#define AVR32_PCCR_IE0_MASK                            0x00000020
#define AVR32_PCCR_IE0_OFFSET                                   5
#define AVR32_PCCR_IE0_SIZE                                     1
#define AVR32_PCCR_IE1                                          6
#define AVR32_PCCR_IE1_MASK                            0x00000040
#define AVR32_PCCR_IE1_OFFSET                                   6
#define AVR32_PCCR_IE1_SIZE                                     1
#define AVR32_PCCR_IEC                                          4
#define AVR32_PCCR_IEC_MASK                            0x00000010
#define AVR32_PCCR_IEC_OFFSET                                   4
#define AVR32_PCCR_IEC_SIZE                                     1
#define AVR32_PCCR_R                                            1
#define AVR32_PCCR_R_MASK                              0x00000002
#define AVR32_PCCR_R_OFFSET                                     1
#define AVR32_PCCR_R_SIZE                                       1
#define AVR32_PCCR_S                                            3
#define AVR32_PCCR_S_MASK                              0x00000008
#define AVR32_PCCR_S_OFFSET                                     3
#define AVR32_PCCR_S_SIZE                                       1

#define AVR32_SR_C                                              0
#define AVR32_SR_C_MASK                                0x00000001
#define AVR32_SR_C_OFFSET                                       0
#define AVR32_SR_C_SIZE                                         1
#define AVR32_SR_D                                             26
#define AVR32_SR_DM                                            27
#define AVR32_SR_DM_MASK                               0x08000000
#define AVR32_SR_DM_OFFSET                                     27
#define AVR32_SR_DM_SIZE                                        1
#define AVR32_SR_D_MASK                                0x04000000
#define AVR32_SR_D_OFFSET                                      26
#define AVR32_SR_D_SIZE                                         1
#define AVR32_SR_EM                                            21
#define AVR32_SR_EM_MASK                               0x00200000
#define AVR32_SR_EM_OFFSET                                     21
#define AVR32_SR_EM_SIZE                                        1
#define AVR32_SR_GM                                            16
#define AVR32_SR_GM_MASK                               0x00010000
#define AVR32_SR_GM_OFFSET                                     16
#define AVR32_SR_GM_SIZE                                        1
#define AVR32_SR_H                                             29
#define AVR32_SR_H_MASK                                0x20000000
#define AVR32_SR_H_OFFSET                                      29
#define AVR32_SR_H_SIZE                                         1
#define AVR32_SR_I0M                                           17
#define AVR32_SR_I0M_MASK                              0x00020000
#define AVR32_SR_I0M_OFFSET                                    17
#define AVR32_SR_I0M_SIZE                                       1
#define AVR32_SR_I1M                                           18
#define AVR32_SR_I1M_MASK                              0x00040000
#define AVR32_SR_I1M_OFFSET                                    18
#define AVR32_SR_I1M_SIZE                                       1
#define AVR32_SR_I2M                                           19
#define AVR32_SR_I2M_MASK                              0x00080000
#define AVR32_SR_I2M_OFFSET                                    19
#define AVR32_SR_I2M_SIZE                                       1
#define AVR32_SR_I3M                                           20
#define AVR32_SR_I3M_MASK                              0x00100000
#define AVR32_SR_I3M_OFFSET                                    20
#define AVR32_SR_I3M_SIZE                                       1
#define AVR32_SR_J                                             28
#define AVR32_SR_J_MASK                                0x10000000
#define AVR32_SR_J_OFFSET                                      28
#define AVR32_SR_J_SIZE                                         1
#define AVR32_SR_L                                              5
#define AVR32_SR_L_MASK                                0x00000020
#define AVR32_SR_L_OFFSET                                       5
#define AVR32_SR_L_SIZE                                         1
#define AVR32_SR_M                                             22
#define AVR32_SR_M_APP                                 0x00000000
#define AVR32_SR_M_EX                                  0x00000006
#define AVR32_SR_M_INT0                                0x00000002
#define AVR32_SR_M_INT1                                0x00000003
#define AVR32_SR_M_INT2                                0x00000004
#define AVR32_SR_M_INT3                                0x00000005
#define AVR32_SR_M_MASK                                0x01c00000
#define AVR32_SR_M_NMI                                 0x00000007
#define AVR32_SR_M_OFFSET                                      22
#define AVR32_SR_M_SIZE                                         3
#define AVR32_SR_M_SUP                                 0x00000001
#define AVR32_SR_N                                              2
#define AVR32_SR_N_MASK                                0x00000004
#define AVR32_SR_N_OFFSET                                       2
#define AVR32_SR_N_SIZE                                         1
#define AVR32_SR_Q                                              4
#define AVR32_SR_Q_MASK                                0x00000010
#define AVR32_SR_Q_OFFSET                                       4
#define AVR32_SR_Q_SIZE                                         1
#define AVR32_SR_R                                             15
#define AVR32_SR_R_MASK                                0x00008000
#define AVR32_SR_R_OFFSET                                      15
#define AVR32_SR_R_SIZE                                         1
#define AVR32_SR_T                                             14
#define AVR32_SR_T_MASK                                0x00004000
#define AVR32_SR_T_OFFSET                                      14
#define AVR32_SR_T_SIZE                                         1
#define AVR32_SR_V                                              3
#define AVR32_SR_V_MASK                                0x00000008
#define AVR32_SR_V_OFFSET                                       3
#define AVR32_SR_V_SIZE                                         1
#define AVR32_SR_Z                                              1
#define AVR32_SR_Z_MASK                                0x00000002
#define AVR32_SR_Z_OFFSET                                       1
#define AVR32_SR_Z_SIZE                                         1

#define AVR32_TLBEHI_ASID                                       0
#define AVR32_TLBEHI_ASID_MASK                         0x000000ff
#define AVR32_TLBEHI_ASID_OFFSET                                0
#define AVR32_TLBEHI_ASID_SIZE                                  8
#define AVR32_TLBEHI_I                                          8
#define AVR32_TLBEHI_I_MASK                            0x00000100
#define AVR32_TLBEHI_I_OFFSET                                   8
#define AVR32_TLBEHI_I_SIZE                                     1
#define AVR32_TLBEHI_V                                          9
#define AVR32_TLBEHI_VPN                                       10
#define AVR32_TLBEHI_VPN_MASK                          0xfffffc00
#define AVR32_TLBEHI_VPN_OFFSET                                10
#define AVR32_TLBEHI_VPN_SIZE                                  22
#define AVR32_TLBEHI_V_MASK                            0x00000200
#define AVR32_TLBEHI_V_OFFSET                                   9
#define AVR32_TLBEHI_V_SIZE                                     1

#define AVR32_TLBELO_AP                                         4
#define AVR32_TLBELO_AP_MASK                           0x00000070
#define AVR32_TLBELO_AP_OFFSET                                  4
#define AVR32_TLBELO_AP_PRIV_RWX_UNPRIV_NONE           0x00000003
#define AVR32_TLBELO_AP_PRIV_RWX_UNPRIV_RWX            0x00000007
#define AVR32_TLBELO_AP_PRIV_RW_UNPRIV_NONE            0x00000002
#define AVR32_TLBELO_AP_PRIV_RW_UNPRIV_RW              0x00000006
#define AVR32_TLBELO_AP_PRIV_RX_UNPRIV_NONE            0x00000001
#define AVR32_TLBELO_AP_PRIV_RX_UNPRIV_RX              0x00000005
#define AVR32_TLBELO_AP_PRIV_R_UNPRIV_NONE             0x00000000
#define AVR32_TLBELO_AP_PRIV_R_UNPRIV_R                0x00000004
#define AVR32_TLBELO_AP_SIZE                                    3
#define AVR32_TLBELO_B                                          7
#define AVR32_TLBELO_B_MASK                            0x00000080
#define AVR32_TLBELO_B_OFFSET                                   7
#define AVR32_TLBELO_B_SIZE                                     1
#define AVR32_TLBELO_C                                          9
#define AVR32_TLBELO_C_MASK                            0x00000200
#define AVR32_TLBELO_C_OFFSET                                   9
#define AVR32_TLBELO_C_SIZE                                     1
#define AVR32_TLBELO_D                                          1
#define AVR32_TLBELO_D_MASK                            0x00000002
#define AVR32_TLBELO_D_OFFSET                                   1
#define AVR32_TLBELO_D_SIZE                                     1
#define AVR32_TLBELO_G                                          8
#define AVR32_TLBELO_G_MASK                            0x00000100
#define AVR32_TLBELO_G_OFFSET                                   8
#define AVR32_TLBELO_G_SIZE                                     1
#define AVR32_TLBELO_PFN                                       10
#define AVR32_TLBELO_PFN_MASK                          0xfffffc00
#define AVR32_TLBELO_PFN_OFFSET                                10
#define AVR32_TLBELO_PFN_SIZE                                  22
#define AVR32_TLBELO_SZ                                         2
#define AVR32_TLBELO_SZ_1KB                            0x00000000
#define AVR32_TLBELO_SZ_1MB                            0x00000003
#define AVR32_TLBELO_SZ_4KB                            0x00000001
#define AVR32_TLBELO_SZ_64KB                           0x00000002
#define AVR32_TLBELO_SZ_MASK                           0x0000000c
#define AVR32_TLBELO_SZ_OFFSET                                  2
#define AVR32_TLBELO_SZ_SIZE                                    2
#define AVR32_TLBELO_W                                          0
#define AVR32_TLBELO_W_MASK                            0x00000001
#define AVR32_TLBELO_W_OFFSET                                   0
#define AVR32_TLBELO_W_SIZE                                     1

#ifdef __IAR_SYSTEMS_ASM__

/**************************************************
 *   System registers
 **************************************************/

AVR32_SR        DEFINE  0x000
AVR32_EVBA      DEFINE  0x004
AVR32_ACBA      DEFINE  0x008
AVR32_CPUCR     DEFINE  0x00c
AVR32_ECR       DEFINE  0x010
AVR32_RSR_SUP   DEFINE  0x014
AVR32_RSR_INT0  DEFINE  0x018
AVR32_RSR_INT1  DEFINE  0x01c
AVR32_RSR_INT2  DEFINE  0x020
AVR32_RSR_INT3  DEFINE  0x024
AVR32_RSR_EX    DEFINE  0x028
AVR32_RSR_NMI   DEFINE  0x02c
AVR32_RSR_DBG   DEFINE  0x030
AVR32_RAR_SUP   DEFINE  0x034
AVR32_RAR_INT0  DEFINE  0x038
AVR32_RAR_INT1  DEFINE  0x03c
AVR32_RAR_INT2  DEFINE  0x040
AVR32_RAR_INT3  DEFINE  0x044
AVR32_RAR_EX    DEFINE  0x048
AVR32_RAR_NMI   DEFINE  0x04c
AVR32_RAR_DBG   DEFINE  0x050
AVR32_JECR      DEFINE  0x054
AVR32_JOSP      DEFINE  0x058
AVR32_JAVA_LV0  DEFINE  0x05c
AVR32_JAVA_LV1  DEFINE  0x060
AVR32_JAVA_LV2  DEFINE  0x064
AVR32_JAVA_LV3  DEFINE  0x068
AVR32_JAVA_LV4  DEFINE  0x06c
AVR32_JAVA_LV5  DEFINE  0x070
AVR32_JAVA_LV6  DEFINE  0x074
AVR32_JAVA_LV7  DEFINE  0x078
AVR32_JTBA      DEFINE  0x07c
AVR32_JBCR      DEFINE  0x080

AVR32_CONFIG0   DEFINE  0x100
AVR32_CONFIG1   DEFINE  0x104
AVR32_COUNT     DEFINE  0x108
AVR32_COMPARE   DEFINE  0x10c
AVR32_TLBEHI    DEFINE  0x110
AVR32_TLBELO    DEFINE  0x114
AVR32_PTBR      DEFINE  0x118
AVR32_TLBEAR    DEFINE  0x11c
AVR32_MMUCR     DEFINE  0x120
AVR32_TLBARLO   DEFINE  0x124
AVR32_TLBARHI   DEFINE  0x128
AVR32_PCCNT     DEFINE  0x12c
AVR32_PCNT0     DEFINE  0x130
AVR32_PCNT1     DEFINE  0x134
AVR32_PCCR      DEFINE  0x138
AVR32_BEAR      DEFINE  0x13c
AVR32_MPUAR0    DEFINE  0x140
AVR32_MPUAR1    DEFINE  0x144
AVR32_MPUAR2    DEFINE  0x148
AVR32_MPUAR3    DEFINE  0x14c
AVR32_MPUAR4    DEFINE  0x150
AVR32_MPUAR5    DEFINE  0x154
AVR32_MPUAR6    DEFINE  0x158
AVR32_MPUAR7    DEFINE  0x15c
AVR32_MPUPSR0   DEFINE  0x160
AVR32_MPUPSR1   DEFINE  0x164
AVR32_MPUPSR2   DEFINE  0x168
AVR32_MPUPSR3   DEFINE  0x16c
AVR32_MPUPSR4   DEFINE  0x170
AVR32_MPUPSR5   DEFINE  0x174
AVR32_MPUPSR6   DEFINE  0x178
AVR32_MPUPSR7   DEFINE  0x17c
AVR32_MPUCRA    DEFINE  0x180
AVR32_MPUCRB    DEFINE  0x184
AVR32_MPUBRA    DEFINE  0x188
AVR32_MPUBRB    DEFINE  0x18c
AVR32_MPUAPRA   DEFINE  0x190
AVR32_MPUAPRB   DEFINE  0x194
AVR32_MPUCR     DEFINE  0x198

#endif /* __IAR_SYSTEMS_ASM__*/

#endif /* __IOAVR32_H */
