// Associated include file : Graphics/Colors.H

#include "common/common.h"
#include "drivers/drivers.h"
#include "mecanism/mecanism.h"
#include "graphics/grx.h"
#include "graphics/colors.h"

#include <math.h>

// ----- Global variables

int    ColorMode;
TColor NoSysColor[5];

int Palette13[13][3] =
  { {   0,   0,   0 },
    {   0,   0, 191 },
    {   0, 170, 170 },
    {  85, 255, 255 },
    {  48, 140,  43 },
    {  60, 182,  60 },
    {  85, 255,  85 },
    { 153,  44,  44 },
    { 198,  60,  60 },
    { 250,  80,  80 },
    {  90,  90,  90 },
    { 170, 170, 170 },
    { 255, 255, 255 } };

  int Palette3[3][3] =
  { {  70,  70,  70 },
    { 140, 140, 140 },
    { 200, 200, 200 } };

// ----- Local variables

#ifdef _TURBOC_
  #define rbits    4
  #define gbits    4
  #define bbits    4
  #define accurate 2
  #define tot      4
  #define tr       16
  #define tg       16
  #define tb       16
  #define FN       "%s/datas/turbo%d.col"
#else
  #define rbits    5
  #define gbits    5
  #define bbits    5
  #define accurate 4
  #define tot      16
  #define tr       32
  #define tg       32
  #define tb       32
  #define FN       "%s/datas/gnu%d.col"
#endif

#define V_Table(r,g,b,n) Table[(n)+tot*((b)+tb*((g)+tg*(r)))]

// ----- Object TDither : Used when in True Color (do nothing)

TDither::TDither()
{ }

TDither::~TDither()
{ }

TColor TDither::GetRGBColor(int, int, int R, int G, int B)
{ return GrAllocColor(R,G,B);
}

// ----- Object TGosselinkDither : Used when in 16 or 256 colors

class TGosselinkDither : public TDither
{ // --- Datas
  protected:
    byte *Table;
  // --- Functions
  public:
  // Constructor / Destructor
    TGosselinkDither();
    virtual ~TGosselinkDither();
  // Usefull method
    virtual TColor GetRGBColor(int X, int Y, int R, int G, int B);
};

TGosselinkDither::TGosselinkDither()
{ char   FileName[200];
  TDisk *FileIn;
  // Kepp memory space for the Table
  Table=new byte[tr*tb*tg*tot];
  // Build the file name
  if (ColorMode==colMODE_16) sprintf(FileName,FN,SwordPath,16);
                        else sprintf(FileName,FN,SwordPath,256);
  // Load the table
  FileIn=new TDisk(FileName,stOpen);
  FileIn->Read(Table,tr*tb*tg*tot);
  delete FileIn;
}

TGosselinkDither::~TGosselinkDither()
{ delete Table;
}

TColor TGosselinkDither::GetRGBColor(int X, int Y, int R, int G, int B)
{ return V_Table( ( R >> (8-rbits)) & (tr-1),
                  ( G >> (8-gbits)) & (tg-1),
                  ( B >> (8-bbits)) & (tb-1),
                  (X & (accurate-1)) + (Y & (accurate-1)) * accurate
                );
}

// ----- Local vars and funcs

TDither *Dither;

// SetSystemColors : Setup the system colors GrayXXX and adjust
//    corresponding NoSysColor values.
//    return TRUE if it is necessary to redraw the screen.

boolean SetSystemColors(void)
{ int i;
  switch(ColorMode)
  { case colMODE_16 :
      for(i=0;i<3;i++)
        GrSetColor(13+i,Palette3[i][0],Palette3[i][1],Palette3[i][2]);
      NoSysColor[1]=13;
      NoSysColor[2]=14;
      NoSysColor[3]=15;
      return FALSE;
    case colMODE_256 :
      for(i=0;i<3;i++)
        GrSetColor(252+i,Palette3[i][0],Palette3[i][1],Palette3[i][2]);
      NoSysColor[1]=252;
      NoSysColor[2]=253;
      NoSysColor[3]=254;
      return FALSE;
    case colMODE_TrueColor :
      for(i=0;i<3;i++)
        NoSysColor[i+1]=GrAllocColor(Palette3[i][0],Palette3[i][1],Palette3[i][2]);
     return TRUE;
  }
  return FALSE;
}

// SetDitherColors : Setup the colors used for dithering

void SetDitherColors(void)
{ int i;
  switch(ColorMode)
  { case colMODE_16 :
      for(i=0;i<13;i++)
        GrSetColor(i,Palette13[i][0],Palette13[i][1],Palette13[i][2]);
      break;
    case colMODE_256 :
	   for(i=0;i<252;i++)
        GrSetColor(i, ((i/42) % 6)*255/5,
                      ((i/ 6) % 7)*255/6,
                      (i     % 6)*255/5);
      break;
  }
}

// ----- Global functions

void InitColors(void)
{ int i;
  // Get the mode for color gestion
  switch(GrNumColors())
  { case 16  : ColorMode=colMODE_16;        break;
    case 256 : ColorMode=colMODE_256;       break;
    default  : ColorMode=colMODE_TrueColor; break;
  }
  Debug("  COLORS...    ColorMode      = %d -> ",ColorMode);
  // Initialize and show the palette entries when not in TrueColor
  switch(ColorMode)
  { case colMODE_16  :
      // Allocate all color cells
      // (Black and White are allocated by default. There are 14 free entries
      // to allocate).
      for(i=0;i<14;i++) GrAllocCell();
      NoSysColor[0]=0;
      NoSysColor[4]=12;
      Debug(" 16 colors\n");
      break;
    case colMODE_256 :
      // Allocate all color cells
      // (Black and White are allocated by default. There are 254 free entries
      // to allocate).
      for(i=0;i<254;i++) GrAllocCell();
      NoSysColor[0]=0;
      NoSysColor[4]=251;
      Debug(" 256 colors\n");
      break;
    case colMODE_TrueColor :
      NoSysColor[0]=GrAllocColor(0,0,0);
      NoSysColor[4]=GrAllocColor(255,255,255);
      Debug(" True color\n");
      break;
  }
  SetSystemColors();
  SetDitherColors();
  // Initialize the Dithering system
  switch(ColorMode)
  { case colMODE_16  :
    case colMODE_256 :
      Dither=new TGosselinkDither();
      break;
    case colMODE_TrueColor :
      Dither=new TDither();
      break;
  }
}

void DoneColors(void)
{ delete Dither;
}

boolean ChangeSystemColor(int n, int r, int g, int b)
{ if ((n<0)||(n>2)) return FALSE;
  Palette3[n][0]=r;
  Palette3[n][1]=g;
  Palette3[n][2]=b;
  return SetSystemColors();
}

void RGB2HSV(byte r, byte g, byte b, byte& h, byte& s, byte& v)
{ byte  max,min;
  float th;
  if (r>g) max=r; else max=g;
  if (b>max) max=b;
  if (r<g) min=r; else min=g;
  if (b<min) min=b;
  v=max;
  if (max) s=(byte)((float)(max-min)/max*255); else s=0;
  if (s)
  { if (r==max) th=(float)(g-b)/(max-min);
    if (g==max) th=2+(float)(b-r)/(max-min);
    if (b==max) th=4+(float)(r-g)/(max-min);
    th*=60;
    if (th<0) th+=360;
    h=(byte)(th*255./360.);
  } else h=0;
}

void HSV2RGB(byte h, byte s, byte v, byte& r, byte& g, byte& b)
{ float th,ts;
  byte  max,min;
  byte  me1,me2,j;
  int   i;
  if (s)
  { // h, en degrs et s, 0->1
    th=(float)h*360./255.;
    ts=(float)s/255.;
    // max,min
    max=v;
    min=(byte)((float)max*(1.-ts));
    //
    if (th>300) th-=360;
    th/=60;
    i=(int)floor(th);
    j=(i+1) & 0xFE;
    //
    me1=(byte)(min-(th-j)*(max-min));
    me2=(byte)(min+(th-j)*(max-min));
    switch(i)
    { case -1: r=max; g=min; b=me1; break;
      case  0: r=max; g=me2; b=min; break;
      case  1: r=me1; g=max; b=min; break;
      case  2: r=min; g=max; b=me2; break;
      case  3: r=min; g=me1; b=max; break;
      case  4: r=me2; g=min; b=max; break;
      default: Debug("ERROR in HSV2RGB. i out of interval\n"); break;
    }
  }
  else
  { r=g=b=v;
  }
}

void RGB2CYM(byte r, byte g, byte b, byte& c, byte& y, byte& m)
{ c=255-r;
  y=255-b;
  m=255-g;
}

void CYM2RGB(byte c, byte y, byte m, byte& r, byte& g, byte& b)
{ r=255-c;
  b=255-y;
  g=255-m;
}

void RGB2CYMK(byte r, byte g, byte b, byte& c, byte& y, byte& m, byte& k)
{ RGB2CYM(r,g,b,c,y,m);
  if (c<y) k=c; else k=y;
  if (m<k) k=m;
  if (k!=255)
  { float tk=(255.-(float)k)/255.;
    c=(byte)((float)(c-k)/tk);
    y=(byte)((float)(y-k)/tk);
    m=(byte)((float)(m-k)/tk);
  }
  else
  { c=y=m=0;
  }
}

void CYMK2RGB(byte c, byte y, byte m, byte k, byte& r, byte& g, byte& b)
{ float tk=(255.-(float)k)/255.;
  c=(byte)((float)c*tk+k);
  y=(byte)((float)y*tk+k);
  m=(byte)((float)m*tk+k);
  CYM2RGB(c,y,m,r,g,b);
}