// Copyright (C) 1996 Keith Whitwell.
// This file may only be copied under the terms of the GNU Library General
// Public License - see the file COPYING in the lib3d distribution.

#ifndef ModelBuilderClass
#define ModelBuilderClass

#include <Lib3d/Model.H>
#include <Lib3d/internals/Array.H>
#include <Lib3d/internals/Polygon.H>
#include <Lib3d/internals/Debuggable.H>
#include <Lib3d/internals/Material.H>

// Class to build an Model via a series of lowlevel function calls.

class BoundingBoxExact;

class ModelBuilder : public Debuggable
{
public:
    ModelBuilder();
    ~ModelBuilder();

    enum { Vertices = 0,
	   VertexNormals = 1,
	   PolygonNormals = 2 };

    // Extract the model info. from an nff file. (see also WorldBuilder)
    bool readNFF( istream &in );

    // Extract the model info. from an obj file.
    bool readObj( istream &in );

    // Extract the model info. from a TAGL .geom file.
    bool readGeom( istream &in );

    uint addMaterial(const Vector3 &colour, 
		     float Ka, float Kd, 
		     float c1, float c2, float c3);
    uint addMaterial(const Material &mat);

    // The add... methods search for a previous matching entry to
    // optimize the object generated.  
    //
    // The search is exact by default, but in future it will accept
    // near misses with the setSearchTolerance() method.
    //
    // This behaviour can be disabled with the setSearchBehaviour()
    // method.


    void setSearchBehaviour( uint buffer, bool flag );

    // Not implemented.
    void setSearchTolerance( uint buffer, float error );

    //
    uint  setVertexNormal( Vector3& ); 
    uint  setVertexNormal( uint );
    uint  setPolygonNormal( Vector3& );
    uint  setPolygonNormal( uint );

    void  setPolygonMaterial( uint );
    void  setModelTexture( Texture * );

    void  calculatePolygonNormals(); // future ones, that is.
    void  calculateVertexNormals(); // future ones, that is.
    void  setColourfulMode( float Ka = .3, float Kd = 1 ); 

    uint  addVertex( const Vector3& );            // obsolete
    uint  addVertex( float x, float y, float z ); // obsolete
    uint  addPolygon( uint nr, const Vector3* );  // obsolete
    uint  addPolygon( uint nr, const uint* );	  // obsolete

    uint  addVertex( float, float, float, float, float );
    uint  addVertex( const Vector3&, float, float );
    uint  addTriangle( uint a, uint b, uint c );
    uint  addTriangle( const Vector3 v[3] );

    void startModel();
    Model *endModel();


protected:
    uint  searchVertex( const Vector3& ) const;
    uint  searchVertexNormal( const Vector3& ) const;
    uint  searchPolygonNormal( const Vector3& ) const;

    void  cylinderWrap( const BoundingBoxExact *box );
    void  sphereWrap( const BoundingBoxExact *box );
    void  stupidWrap( const BoundingBoxExact *box );

protected:
    struct NormalTmp {
	NormalTmp() : polyNormalSum(0,0,0), nr(0) {}
	Vector3 polyNormalSum;
	int nr;
    };

    uint polygonNormal;
    uint vertexNormal;
    uint material;

    float searchError[3];
    uint  searchFlag[3];

    Array<Vertex>    vertices;
    Array<Polygon>   polygons;
    Array<Normal>    vertexNormals;
    Array<NormalTmp> vertexNormalScratch;
    Array<Normal>    polygonNormals;
    Array<Material>  materials;

    bool calculatePolygonNormalMode;
    bool calculateVertexNormalMode;
    uint calculateVertexNormalBase;
    bool colourfulMode;

    Texture *texture;
};



#endif







