#include <ctype.h>
#include <go32.h>
#include <keys.h>
#include <stdio.h>
#include <sys/farptr.h>

#include "buffer.h"
#include "colour.h"
#include "desktop.h"
#include "dialog.h"
#include "error.h"
#include "event.h"
#include "input.h"
#include "menu.h"
#include "refresh.h"

/*

	menu.c

		Menu system

		C-Desktop
		Copyright (C)1998, Brett Porter.

*/

/*
	FUNCTION PROTOTYPES
*/

static	void		MenuObject_Draw( T_MenuObject *aMenu, T_EventDrawParameters *aParams );
static	bool		MenuObject_EventHandler( void *aMenuObject, T_EventRec *aEvent );
static	bool		MenuObject_EnableCommand( T_MenuObject *aMenuObject, int aID, bool aEnabled );
static	void		MenuObject_Kill( T_MenuObject *aMenuObject );
static	bool		MenuObject_HandleKeypress( T_MenuObject *aMenuObject, int aKeypress );
static	void		MenuObject_ActivateSelected( T_MenuObject *aMenuObject );
static	void		MenuObject_MoveUp( T_MenuObject *aMenuObject );
static	void		MenuObject_MoveDown( T_MenuObject *aMenuObject );
static	bool		MenuObject_FindHotKey( T_MenuObject *aMenuObject, int aKeypress );

static	T_MenuItem	*MenuObject_GetSelectedItem( T_MenuObject *aMenuObject );

static	bool		MenuBar_EventHandler( void *aMenuBar, T_EventRec *aEvent );
static	void		MenuBar_Kill( T_MenuBar *aMenuBar );
static	void		MenuBar_Draw( T_MenuBar *aMenuBar, T_EventDrawParameters *aParams );
static	bool		MenuBar_LeftMouseClick( T_MenuBar *aMenuBar, int aColumn );
static	void		MenuBar_SelectLeftMenu( T_Group *aGroup );
static	void		MenuBar_SelectRightMenu( T_Group *aGroup );
static	bool		MenuBar_HandleKeypress( T_MenuBar *aMenuBar, int aKeypress );

/*
   EXTERNAL VARIABLES
*/

extern	const char	*kCommandDescriptions[];

/*
	FUNCTION DEFINITIONS
*/

T_MenuObject	*MenuObject_Initialise( int aID, const char *aName )
{
	int	i;
	bool	lFound = false;
	T_MenuObject	*lObject;

	P_InitVar( lObject, T_MenuObject );

	lObject->fID = aID;
	lObject->fSelected = 0;
	lObject->fEnabled = false;

	i = 0;
	while ( aName[i] != EOS )
	{
		if ( aName[i] == '&' && aName[i+1] != '&' )
		{
			if ( lFound )
			{
				FatalError( "Two hotkeys defined in menu title" );
			}
			i++;
			lObject->fHotKey = aName[i];
			if ( aName[i] == EOS )
			{
				FatalError( "Can't define EOS as a hotkey" );
			}
			lFound = true;
			lObject->fHotKeyPosition = i;
		}
		i++;
	}
	if ( !lFound )
	{
		lObject->fHotKey = EOS;
	}

	P_CreateStr( lObject->fName, aName );

	DesktopObj_Initialise( lObject, MenuObject_EventHandler, true );

	return lObject;
}

void	MenuObject_AddMenuItem( T_MenuObject *aObject, int aID, const char *aName, const char *aKeyName )
{
	int	i;
	bool	lFound = false;
	T_MenuItem	*lCurrent = aObject->fMenuItemList;

	if ( lCurrent == NULL )
	{
		P_InitVar( aObject->fMenuItemList, T_MenuItem );
		lCurrent = aObject->fMenuItemList;
	}
	else
	{
		while ( lCurrent->fNext != NULL )
		{
			lCurrent = lCurrent->fNext;
		}
		P_InitVar( lCurrent->fNext, T_MenuItem );
		lCurrent = lCurrent->fNext;
	}
	lCurrent->fNext = NULL;

	lCurrent->fID = aID;
	lCurrent->fIsLine = false;
	lCurrent->fEnabled = false;

	i = 0;
	while ( aName[i] != EOS )
	{
		if ( aName[i] == '&' && aName[i+1] != '&' )
		{
			if ( lFound )
			{
				FatalError( "Two hotkeys defined in menu item" );
			}
			i++;
			lCurrent->fHotKey = aName[i];
			if ( aName[i] == EOS )
			{
				FatalError( "Can't define EOS as a hotkey" );
			}
			lFound = true;
			lCurrent->fHotKeyPosition = i;
		}
		i++;
	}
	if ( !lFound )
	{
		lCurrent->fHotKey = EOS;
	}

	P_CreateStr( lCurrent->fName, aName );
	P_CreateStr( lCurrent->fKeyName, aKeyName );

	if (( signed )strlen( aKeyName ) > aObject->fMaxKeyNameLength )
	{
		aObject->fMaxKeyNameLength = strlen( aKeyName );
	}

	if (( signed )strlen( aName ) > aObject->fMaxNameLength )
	{
		aObject->fMaxNameLength = strlen( aName );
		if ( lFound )
		{
			aObject->fMaxNameLength--;	/* remove the & */
		}
	}
	aObject->fNumItems++;
}

bool	MenuObject_CheckClick( int aColumn, int aRow )
{
/*
	int	lItemClicked, i, j, lNewSelected;

	T_MenuObject	*lMenu;

	if ( aColumn > gOpenMenu->fX1 && aColumn < gOpenMenu->fX2 && aRow > gOpenMenu->fY1 && aRow < gOpenMenu->fY2 )
	{
		MenuObject_ShowSelectionBar( false );
		lItemClicked = aRow-gOpenMenu->fY1-1;

		lNewSelected = -1;

		for ( i = 0, j = 0; i < gOpenMenu->fY2-gOpenMenu->fY1-1; i++, j++ )
		{
			if ( i == lItemClicked )
			{
				lNewSelected = j;
			}
			if ( gOpenMenu->fMenuItems[j]->fLineAfterThis )
			{
				i++;
			}
		}
		if ( lNewSelected != -1 && gOpenMenu->fMenuItems[lNewSelected]->fIsEnabled )
		{
			gOpenMenu->fSelected = lNewSelected;
			MenuObject_ShowSelectionBar( true );
			lMenu = gOpenMenu;
			MenuObject_CloseOpenMenu();
			/// submenus should be supported, somehow.... for now the menuid is just interpreted as a command ///
			Event_HandleCommand( lMenu->fMenuItems[lNewSelected]->fID );
		}
		else
		{
			MenuObject_ShowSelectionBar( true );
		}
		return true;
	}
	if ( aColumn == gOpenMenu->fX1 || aColumn == gOpenMenu->fX2 || aRow == gOpenMenu->fY1 || aRow == gOpenMenu->fY2 )
	{
		return true;
	}
*/
	return false; /// fix this too
}

void	MenuObject_MoveUp( T_MenuObject *aMenuObject )
{
	T_MenuItem	*lSelected;

	do
	{
		aMenuObject->fSelected--;
		if ( aMenuObject->fSelected < 0 )
		{
			aMenuObject->fSelected = aMenuObject->fNumItems-1;
		}
		lSelected = MenuObject_GetSelectedItem( aMenuObject );
	}
	while ( !lSelected->fEnabled || lSelected->fIsLine );

	Group_RedrawObject( aMenuObject->fOwner->fOwner, aMenuObject );

	Desktop_SetStatusString( kCommandDescriptions[lSelected->fID] );
}

void	MenuObject_MoveDown( T_MenuObject *aMenuObject )
{
	T_MenuItem	*lSelected;

	do
	{
		aMenuObject->fSelected++;
		if ( aMenuObject->fSelected == aMenuObject->fNumItems )
		{
			aMenuObject->fSelected = 0;
		}
		lSelected = MenuObject_GetSelectedItem( aMenuObject );
	}
	while ( !lSelected->fEnabled || lSelected->fIsLine );

	Group_RedrawObject( aMenuObject->fOwner->fOwner, aMenuObject );

	Desktop_SetStatusString( kCommandDescriptions[lSelected->fID] );
}

void	MenuObject_Kill( T_MenuObject *aMenuObject )
{
	T_MenuItem	*lItem = aMenuObject->fMenuItemList, *lPrev;

	while ( lItem != NULL )
	{
		free( lItem->fName );
		free( lItem->fKeyName );
		lPrev = lItem;
		lItem = lItem->fNext;
		free( lPrev );
	}
	free( aMenuObject->fName );
	free( aMenuObject );
}

bool	MenuObject_FindHotKey( T_MenuObject *aMenuObject, int aKeypress )
{
	T_MenuItem	*lItem = aMenuObject->fMenuItemList;
	bool	lRetVal = false;
	int	lSelected = 0;

	while ( lItem != NULL )
	{
		if ( toupper( lItem->fHotKey ) == aKeypress && lItem->fEnabled )
		{
			aMenuObject->fSelected = lSelected;
			Desktop_SetStatusString( kCommandDescriptions[MenuObject_GetSelectedItem( aMenuObject )->fID] );
			MenuObject_ActivateSelected( aMenuObject );
			lRetVal = true;
			break;
		}
		lItem = lItem->fNext;
		lSelected++;
	}
	return lRetVal;
}

void	MenuObject_AddLine( T_MenuObject *aObject )
{
	T_MenuItem	*lCurrent = aObject->fMenuItemList;

	if ( lCurrent == NULL )
	{
		P_InitVar( aObject->fMenuItemList, T_MenuItem );
		lCurrent = aObject->fMenuItemList;
	}
	else
	{
		while ( lCurrent->fNext != NULL )
		{
			lCurrent = lCurrent->fNext;
		}
		P_InitVar( lCurrent->fNext, T_MenuItem );
		lCurrent = lCurrent->fNext;
	}
	lCurrent->fNext = NULL;

	lCurrent->fID = -1;
	lCurrent->fIsLine = true;
	lCurrent->fEnabled = false;
	lCurrent->fHotKey = EOS;

	lCurrent->fName = NULL;
	lCurrent->fKeyName = NULL;

	aObject->fNumItems++;
}

/*****************************************************************************

    Function: MenuBar_Initialise

    Description: Create a new menu bar at the specified physical screen location
    Parameters: The rectangle to define the menu bar within.
	 				NOTE! This must not only encompass the bar, but all objects added to it.
    Return: Pointer to the created menu bar

*****************************************************************************/

T_MenuBar	*MenuBar_Initialise( int aX1, int aY1, int aX2, int aY2 )
{
	T_MenuBar	*lMenuBar;

	P_InitVar( lMenuBar, T_MenuBar );

	DesktopObj_Initialise( lMenuBar, MenuBar_EventHandler, false );

	lMenuBar->fCurXPos = 0;

	lMenuBar->fGroup = Group_Initialise( aX1, aY1, aX2, aY2, FLAG_NONE );

	return lMenuBar;
}

void	MenuBar_Kill( T_MenuBar *aMenuBar )
{
	Group_Kill( aMenuBar->fGroup );
	free( aMenuBar );
}

void	MenuBar_AddMenuObject( T_MenuBar *aMenuBar, T_MenuObject *aObject )
{
	int	lStrLen, lWidth;

	lWidth = aObject->fMaxKeyNameLength+aObject->fMaxNameLength+4;

	Group_AddObject( aMenuBar->fGroup, aObject, aMenuBar->fCurXPos, 0, aMenuBar->fCurXPos+lWidth, aObject->fNumItems+2, FLAG_NONE );

	lStrLen = strlen( aObject->fName )+2;
	if ( aObject->fHotKey != EOS )
	{
		lStrLen--;
	}
	aMenuBar->fCurXPos += lStrLen;
}

void	MenuBar_Draw( T_MenuBar *aMenuBar, T_EventDrawParameters *aParams )
{
	char	lStr[aParams->fBuffer->fWidth+1];
	T_Buffer	*lBuffer = aParams->fBuffer;
        int     lX1, lY1, lX2;
	T_ObjectRec		*lRec;

	/// more than one line?

	lX1 = aParams->fX;
	lY1 = aParams->fY;
	lX2 = lX1+aMenuBar->fWidth-1;

	memset( lStr, ' ', lBuffer->fWidth );
	Buffer_Print( lBuffer, lX1, lY1, lX2, lY1, lStr, COLOUR_MENU_TEXT );

	lRec = aMenuBar->fGroup->fObjectList;

	while ( lRec != NULL )
	{
		DesktopObj_Draw( lRec, aMenuBar->fGroup->fX1-lX1, aMenuBar->fGroup->fY1-lY1, lBuffer );

		lRec = lRec->fNext;
	}
}

bool	MenuBar_LeftMouseClick( T_MenuBar *aMenuBar, int aColumn )
{
/*	int	i = 0, lStrLen;
	T_MenuObject	*lMenu = NULL;
	T_MenuBar		*lMenuBar = aMenuBar;

	while ( i < lMenuBar->fNumMenus )
	{
		lMenu = lMenuBar->fMenuObjects[i];
		lStrLen = strlen( lMenu->fName1 )+strlen( lMenu->fName2 )+2;
		if ( lMenu->fHotKey )
		{
			lStrLen++;
		}
		if ( aColumn >= lMenu->fX1 && aColumn < lMenu->fX1+lStrLen )
		{
			MenuObject_Open( i, lMenuBar );
		}
		i++;
	} */
	return true; /// different format...
}

void	MenuBar_SelectLeftMenu( T_Group *aGroup )
{
	T_ObjectRec	*lRec, *lTarget = aGroup->fFocused;

	do
	{
		lRec = aGroup->fObjectList;

		while ( lRec->fNext != lTarget && lRec->fNext != NULL )
		{
			lRec = lRec->fNext;
		}
		lTarget = lRec;
	}
	while ((( T_MenuObject* )lRec->fObject.fData )->fEnabled == false );

	DesktopObj_Focus( lRec );
}

void	MenuBar_SelectRightMenu( T_Group *aGroup )
{
	T_ObjectRec	*lRec = aGroup->fFocused;

	do
	{
		lRec = lRec->fNext;

		if ( lRec == NULL )
		{
			lRec = aGroup->fObjectList;
		}
	}
	while ((( T_MenuObject* )lRec->fObject.fData )->fEnabled == false );

	DesktopObj_Focus( lRec );
}

bool	MenuBar_EventHandler( void *aMenuBar, T_EventRec *aEvent )
{
	bool	lRetVal = false;

	T_Group	*lGroup = (( T_MenuBar* )aMenuBar )->fGroup;

	switch ( aEvent->fMessage )
	{
		case	EVENT_LEFTMOUSECLICK:
			lRetVal = MenuBar_LeftMouseClick( aMenuBar, aEvent->fParameters.fMouseClick.fRow );
			break;
		case	EVENT_KEYPRESS:
			lRetVal = MenuBar_HandleKeypress( aMenuBar, aEvent->fParameters.fKeypress );
			break;
		case	EVENT_KILL:
			MenuBar_Kill( aMenuBar );
			lRetVal = true;
			break;
		case	EVENT_DRAW:
			MenuBar_Draw( aMenuBar, &aEvent->fParameters.fDrawParams );
			lRetVal = true;
			break;
		case	EVENT_COMMANDENABLED:
			if ( Group_Broadcast( lGroup, aEvent ))
			{
				/// redraw?
			}
			lRetVal = true;
			break;
		case	EVENT_RIGHTMOUSECLICK:
			break;
		case	EVENT_CANFOCUS:
			lRetVal = false;
			break;
		case	EVENT_SHOW:
		case	EVENT_HIDE:
			break;
		default:
			FatalError( "Unhandled message" );
			break;
	}
	return lRetVal;
}

bool	MenuObject_EventHandler( void *aMenuObject, T_EventRec *aEvent )
{
	bool	lRetVal = false;

	T_MenuObject	*lMenuObject = aMenuObject;

	switch ( aEvent->fMessage )
	{
		case	EVENT_KILL:
			MenuObject_Kill( aMenuObject );
			lRetVal = true;
			break;
		case	EVENT_COMMANDENABLED:
			if ( MenuObject_EnableCommand( aMenuObject, aEvent->fParameters.fCommandStatus.fID, aEvent->fParameters.fCommandStatus.fEnabled ))
			{
				/// redraw?
				lRetVal = true;
			}
			break;
		case	EVENT_KEYPRESS:
			lRetVal = MenuObject_HandleKeypress( aMenuObject, aEvent->fParameters.fKeypress );
			break;
		case	EVENT_CANFOCUS:
			lRetVal = true;
			break;
		case	EVENT_FOCUS:
      	if ( lMenuObject->fOwner->fOwner->fVisible )
         {
				Desktop_SetStatusString( kCommandDescriptions[MenuObject_GetSelectedItem( aMenuObject )->fID] );
         }
			lRetVal = true;
			break;
		case	EVENT_UNFOCUS:
			Desktop_SetStatusString( "" );
			lRetVal = true;
			break;
		case	EVENT_DRAW:
			MenuObject_Draw( aMenuObject, &aEvent->fParameters.fDrawParams );
			lRetVal = true;
			break;
		case	EVENT_SHOW:
         if ( lMenuObject->fFocused )
         {
				Desktop_SetStatusString( kCommandDescriptions[MenuObject_GetSelectedItem( aMenuObject )->fID] );
         }
         lRetVal = true;
			break;
		case	EVENT_HIDE:
			Desktop_SetStatusString( "" );
         lRetVal = true;
			break;
		default:
			FatalError( "Unhandled message" );
			break;
	}
	return lRetVal;
}

bool	MenuObject_EnableCommand( T_MenuObject *aMenuObject, int aID, bool aEnabled )
{
	T_MenuItem	*lCurrent = aMenuObject->fMenuItemList;
	bool	lRetVal = false;

	aMenuObject->fEnabled = false;

	while ( lCurrent != NULL )
	{
		if ( lCurrent->fID == aID )
		{
			lCurrent->fEnabled = aEnabled;

			lRetVal = true;
		}
		if ( lCurrent->fEnabled )
		{
			aMenuObject->fEnabled = true;
			if ( lRetVal )
			{
				break;
			}
		}
		lCurrent = lCurrent->fNext;
	}
	return lRetVal;
}

bool	MenuObject_HandleKeypress( T_MenuObject *aMenuObject, int aKeypress )
{
	bool	lRetVal = false;

	if ( !aMenuObject->fEnabled )
	{
		return false;
	}
	if ( aKeypress & 0x100 )
	{
		if ( toupper( kScanToASCII[aKeypress-0x100] ) == aMenuObject->fHotKey )
		{
			DesktopObj_Focus( aMenuObject->fOwner );
			return true;
		}
	}
	if ( aMenuObject->fFocused )
	{
		switch ( aKeypress )
		{
			case	K_Return:
				MenuObject_ActivateSelected( aMenuObject );
				lRetVal = true;
				break;
			case	K_EUp:
				MenuObject_MoveUp( aMenuObject );
				lRetVal = true;
				break;
			case	K_EDown:
				MenuObject_MoveDown( aMenuObject );
				lRetVal = true;
				break;
			case	K_ELeft:
				MenuBar_SelectLeftMenu( aMenuObject->fOwner->fOwner );
				lRetVal = true;
				break;
			case	K_ERight:
				MenuBar_SelectRightMenu( aMenuObject->fOwner->fOwner );
				lRetVal = true;
				break;
			default:
				lRetVal = MenuObject_FindHotKey( aMenuObject, toupper( aKeypress ));
				break;
		}
	}
	return lRetVal;
}

void	MenuObject_Draw( T_MenuObject *aMenu, T_EventDrawParameters *aParams )
{
	T_Buffer	*lBuffer = aParams->fBuffer;
	char	lStr[40];	/// 40 big enough?
	int	lX1, lX2, lY1, lY2;
	int	lColour1, lColour2, lColour3, i, j, lItemCount;
	T_MenuItem	*lItem = aMenu->fMenuItemList;
	bool	lVisible = aMenu->fOwner->fOwner->fVisible;

	lX1 = aParams->fX;
	lY1 = aParams->fY;
	lX2 = lX1+aMenu->fWidth-1;
	lY2 = lY1+aMenu->fHeight-1;

	lColour1 = aMenu->fEnabled ? COLOUR_MENU_TEXT : COLOUR_MENU_DISABLED;
	lColour2 = aMenu->fEnabled ? COLOUR_MENU_HILITE : COLOUR_MENU_DISABLED;

	if ( lVisible && aMenu->fFocused )
	{
		lColour1 = COLOUR_MENU_SELECT;
		lColour2 = COLOUR_MENU_HILITE_SEL;
	}

	aMenu->fName[aMenu->fHotKeyPosition-1] = EOS;
	sprintf( lStr, " %s", aMenu->fName );
	Buffer_Print( lBuffer, lX1, lY1, lX2, lY1, lStr, lColour1 );
	aMenu->fName[aMenu->fHotKeyPosition-1] = '&';

	if ( aMenu->fHotKey != EOS )
	{
		i = lX1+aMenu->fHotKeyPosition;
		lStr[0] = aMenu->fHotKey;
		lStr[1] = EOS;
		Buffer_Print( lBuffer, i, lY1, lX2, lY1, lStr, lColour2 );

		i++;
		sprintf( lStr, "%s ", aMenu->fName+aMenu->fHotKeyPosition+1 );
		Buffer_Print( lBuffer, i, lY1, lX2, lY1, lStr, lColour1 );
	}
	i = lX1+strlen( aMenu->fName )+2;

	if ( aMenu->fHotKey != EOS )
	{
		i--;
	}

	memset( lStr, ' ', aMenu->fWidth );
	Buffer_Print( lBuffer, i, lY1, lX2, lY1, lStr, COLOUR_MENU_TEXT );

	lY1++;

	if ( lVisible && aMenu->fFocused )
	{
		lStr[aMenu->fWidth] = EOS;
		memset( lStr, '', aMenu->fWidth );
		lStr[0] = '';
		lStr[aMenu->fWidth-1] = '';
		Buffer_Print( lBuffer, lX1, lY1, lX2, lY1, lStr, COLOUR_MENU_FRAME );

		lStr[0] = '';
		lStr[aMenu->fWidth-1] = '';
		Buffer_Print( lBuffer, lX1, lY1+aMenu->fHeight-2, lX2, lY1+aMenu->fHeight-2, lStr, COLOUR_MENU_FRAME );

		i = lY1+1;
		lItemCount = 0;
		while ( lItem != NULL )
		{
			if ( !lItem->fIsLine )
			{
				lColour1 = lItem->fEnabled ? COLOUR_MENU_TEXT : COLOUR_MENU_DISABLED;
				lColour2 = lItem->fEnabled ? COLOUR_MENU_HILITE : COLOUR_MENU_DISABLED;
				lColour3 = lItem->fEnabled ? COLOUR_MENU_KEYNAME : COLOUR_MENU_DISABLED;

				if ( aMenu->fSelected == lItemCount )
				{
					lColour1 = COLOUR_MENU_SELECT;
					lColour2 = COLOUR_MENU_HILITE_SEL;
					lColour3 = COLOUR_MENU_KEYNAME_SEL;
				}

				Buffer_Print( lBuffer, lX1, i, lX2, i, "", COLOUR_MENU_FRAME );
				Buffer_Print( lBuffer, lX1+aMenu->fWidth-1, i, lX2, i, "", COLOUR_MENU_FRAME );

				lItem->fName[lItem->fHotKeyPosition-1] = EOS;
				sprintf( lStr, " %-*s", aMenu->fWidth-3, lItem->fName );
				Buffer_Print( lBuffer, lX1+1, i, lX2, i, lStr, lColour1 );
				lItem->fName[lItem->fHotKeyPosition-1] = '&';

				j = lX1+lItem->fHotKeyPosition+1;
				if ( lItem->fHotKey != EOS )
				{
					lStr[0] = lItem->fHotKey;
					lStr[1] = EOS;
					Buffer_Print( lBuffer, j++, i, lX2, i, lStr, lColour2 );
					Buffer_Print( lBuffer, j, i, lX2, i, lItem->fName+lItem->fHotKeyPosition+1, lColour1 );
				}
				Buffer_Print( lBuffer, lX1+aMenu->fWidth-2-strlen( lItem->fKeyName ), i, lX2, i, lItem->fKeyName, lColour3 );
			}
			else
			{
				memset( lStr, '', aMenu->fWidth );
				lStr[aMenu->fWidth] = EOS;
				lStr[0] = '';
				lStr[aMenu->fWidth-1] = '';
				Buffer_Print( lBuffer, lX1, i, lX2, i, lStr, COLOUR_MENU_FRAME );
			}
			lItemCount++;
			i++;

			lItem = lItem->fNext;
		}
	}
}

void	MenuObject_ActivateSelected( T_MenuObject *aMenuObject )
{
	T_Group	*lGroup = aMenuObject->fOwner->fOwner;

	lGroup->fCloseModalCommand = ENDMODAL_EXECUTECOMMAND;
	lGroup->fCloseModalID = MenuObject_GetSelectedItem( aMenuObject )->fID;

	Event_SendMessage( MSG_CLOSEMODAL );
}

T_MenuItem	*MenuObject_GetSelectedItem( T_MenuObject *aMenuObject )
{
	int	i = aMenuObject->fSelected;
	T_MenuItem	*lItem = aMenuObject->fMenuItemList;

	while ( i )
	{
		lItem = lItem->fNext;
		i--;
	}
	return lItem;
}

bool	MenuBar_HandleKeypress( T_MenuBar *aMenuBar, int aKeypress )
{
	T_ObjectRec		*lRec = aMenuBar->fGroup->fObjectList;
	T_MenuObject	*lMenu;

	if ( aKeypress & 0x100 )
	{
		while ( lRec != NULL )
		{
			lMenu = lRec->fObject.fData;

			if ( lMenu->fEnabled )
			{
				if ( toupper( kScanToASCII[aKeypress-0x100] ) == lMenu->fHotKey )
				{
					DesktopObj_Focus( lRec );
					Group_GoModal( aMenuBar->fGroup );
					return true;
				}
			}
         lRec = lRec->fNext;
		}
	}
	return false;
}

