#include <conio.h>
#include <crt0.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>

#include "appl.h"
#include "config.h"
#include "desktop.h"
#include "dialog.h"
#include "dlgobj.h"
#include "error.h"
#include "event.h"
#include "input.h"
#include "misc.h"
#include "mouse.h"
#include "refresh.h"

/*

	appl.c

		Application.

		C-Desktop
		Copyright (C)1998, Brett Porter.

*/

/*
	# CONFIGURATION PARAMTER IDS #
*/

#define	CFG_USE_MOUSE					0

/*
	# FUNCTION PROTOTYPES #
*/

extern	void	Application_Idle( void );
extern	void	Application_CustomInitialise( void );
extern   void  Application_CustomTerminate( void );
extern	bool	Application_CustomCheckConfigVar( T_ConfigVar *aVar, bool *lRetVal );
extern	void	Application_CustomRegisterConfigVars( void );
extern	void	Application_CustomRegisterEventCommands( void );

static	void	Application_Initialise( void );
static	void	Application_GoToBaseDirectory( void );
static	void	Application_Terminate( void );
static	void	Application_RegisterEventCommands( void );

extern   void	Colour_Initialise( void );
extern   void	Colour_Terminate( void );
extern	void	Event_Terminate( void );
extern	void	Desktop_Terminate( void );
extern	void	Config_Shutdown( void );
extern	void	Refresh_Terminate( void );

/*
	# FUNCTION DEFINITIONS #
*/

/*****************************************************************************

    Function: Application_GoToBaseDirectory

    Description: Retrieve executable's path from argv[0] and change to that directory
    Parameters: None
    Return: None

*****************************************************************************/

void	Application_GoToBaseDirectory( void )
{
	char	*lProgLoc =	__crt0_argv[0];
	int i = strlen( lProgLoc )-1;
	while ( lProgLoc[i] != '/' )
		i--;
	lProgLoc[i] = EOS;
	chdir( lProgLoc );
	lProgLoc[i] = '/';
}

/*****************************************************************************

    Function: Application_NOFP

    Description: Handles SIGNOFP: if program tries floating point op's when none is present.
    Parameters: the signal (SIGNOFP)
    Return: None

*****************************************************************************/

void	Application_NOFP( int aSignal )
{
	Error( "Sorry, but this program requires a floating-point processor." );
	exit( -1 );
}

/*****************************************************************************

    Function: Application_Initialise

    Description: Initialises all components of the application and C-Desktop library.
    Parameters: None
    Return: None

*****************************************************************************/

void	Application_Initialise( void )
{
	Application_GoToBaseDirectory();

	signal( SIGNOFP, Application_NOFP );
///	signal( SIGINT, SIG_IGN );
	signal( SIGALRM, SIG_IGN );
	signal( SIGABRT, SIG_IGN );

	Config_Initialise();
	Refresh_Initialise();
	Colour_Initialise();
	Desktop_Initialise();
	Event_Initialise();

	Config_LoadConfigFile();
	Config_ParseCommandLine( __crt0_argc-1, &__crt0_argv[1] );

	Mouse_Initialise();

	Application_RegisterEventCommands();

	Event_EnableCommand( CMD_EXIT, true );

	Screen_SetVideoMode();
	Screen_SetCursorType( _NOCURSOR );

	Application_CustomInitialise();
}

/*****************************************************************************

    Function: Application_Execute

    Description: Called by any program using C-Desktop after Application_Initialise. Sets the desktop modal, which takes over.
    Parameters: None
    Return: None

*****************************************************************************/

void	Application_Execute( void )
{
	Application_Initialise();

	Group_GoModal( gDesktop );

   Application_Terminate();
}

/*****************************************************************************

    Function: Application_Wait

    Description:
    	Wait for a message to be generated by some component of the application. Poll input and refresh screen while waiting.

    Parameters: The message to wait for
    Return: None

*****************************************************************************/

void	Application_Wait( E_MessageType aWaitCondition )
{
	while ( !Event_CheckMessage( aWaitCondition ))
	{
		Input_Check();
		Refresh_Update();
		Application_Idle();
	}
}

/*****************************************************************************

    Function: Application_Terminate

    Description: Cleans up after the application
    Parameters: None
    Return: None

*****************************************************************************/

void	Application_Terminate( void )
{
   Application_CustomTerminate();

	Config_Shutdown();

	signal( SIGNOFP, SIG_DFL );
	signal( SIGINT, SIG_DFL );
	signal( SIGALRM, SIG_DFL );
	signal( SIGABRT, SIG_DFL );

	Event_Terminate();
   Desktop_Terminate();
   Colour_Terminate();
   Refresh_Terminate();

   Application_GoToBaseDirectory();
}

/*****************************************************************************

    Function: Application_CheckFileOverwrite

    Description: Shows a Yes/No dialog asking to overwrite a file if it already existed.
    Parameters: The filename to check and the default response
    Return: True if didn't exist or can overwrite, otherwise false.

*****************************************************************************/

bool	Application_CheckFileOverwrite( const char *aFilename, bool aDefault )
{
	if ( __file_exists( aFilename ))
	{
		if ( Application_ConfirmDialog( "That file already exists. Overwrite it?", aDefault ) == ENDMODAL_YES )
		{
			return true;
		}
		return false;
	}
	return true;
}

/*****************************************************************************

    Function: Application_ConfirmDialog

    Description: A standard Yes/No confirmation dialog box.
    Parameters: The string to be displayed and the default button.
    Return: The close modal command (ENDMODAL_YES, ENDMODAL_NO or ENDMODAL_CLOSE)

*****************************************************************************/

int	Application_ConfirmDialog( const char *aString, bool aYesDefault )
{
	T_Dialog		*lDlg;
	T_DlgString	*lString;
	T_DlgButton	*lYesButton, *lNoButton;
	int	lRetVal, lWidth, lButSpace1, lButSpace2, i, j, lNumLines;

	char	*lStr[20];

	i = -1;
	j = 0;
	lNumLines = 0;
	lWidth = 23;

	memset( lStr, 0, 81 );

	do
	{
		i++;
		if ( aString[i] == '\n' || aString[i] == EOS )
		{
			if ( lWidth < i-j+4 )
			{
				lWidth = i-j+4;
			}
			lStr[lNumLines] = ( char* )malloc( i-j+1 );
			strncpy( lStr[lNumLines], aString+j, i-j );
			lStr[lNumLines][i-j] = EOS;
			j = i+1;
			lNumLines++;
		}
	}
	while ( aString[i] != EOS );

	lButSpace1 = ( lWidth/2-9 )/2;
	lButSpace2 = ( lWidth*3/2-9 )/2;

	lDlg = Dialog_Initialise( "Confirmation", 0, 0, lWidth-1, 5+lNumLines, FLAG_CENTRE, Dialog_DefaultEventHandler );

	for ( i = lNumLines; i--; )
	{
		lString = DlgString_Initialise( lStr[i], false );
		Group_AddObject( lDlg->fGroup, lString, 2, 2+i, lWidth-3, 2+i, FLAG_CENTREX );
		free( lStr[i] );
	}

	lYesButton = DlgButton_Initialise( "&Yes", ENDMODAL_YES, true );
	Group_AddObject( lDlg->fGroup, lYesButton, lButSpace1, 3+lNumLines, lButSpace1+8, 4+lNumLines, FLAG_NONE );

	lNoButton = DlgButton_Initialise( "&No", ENDMODAL_NO, false );
	Group_AddObject( lDlg->fGroup, lNoButton, lButSpace2, 3+lNumLines, lButSpace2+8, 4+lNumLines, FLAG_NONE );

	if ( aYesDefault )
	{
		Group_FocusObject( lDlg->fGroup, lYesButton );
	}
	else
	{
		Group_FocusObject( lDlg->fGroup, lNoButton );
	}

	lRetVal = Dialog_ShowModal( lDlg );

	Dialog_Kill( lDlg );

	return lRetVal;
}

/*****************************************************************************

    Function: Application_MessageBox

    Description: Displays the given message in a dialog box with just an OK button
    Parameters: The message to display.
    Return: None.

*****************************************************************************/

void	Application_MessageBox( const char *aMessage )
{
	T_Dialog		*lDlg;
	T_DlgString	*lString;
	T_DlgButton	*lButton;

	int	lWidth, lHeight, lCurWidth;
	const char	*lPtr = aMessage;

	lWidth =	0;
	lHeight = 6;

	lCurWidth = 0;

	while ( *lPtr != EOS )
	{
		if ( *lPtr == '\n' )
		{
			if ( lCurWidth > lWidth )
			{
				lWidth = lCurWidth;
			}
			lCurWidth = 0;
			lHeight++;
		}
		lPtr++;
		lCurWidth++;
	}
	if ( lCurWidth > lWidth )
	{
		lWidth = lCurWidth;
	}

	lDlg = Dialog_Initialise( "Message", 0, 0, lWidth+2, lHeight, FLAG_CENTRE, Dialog_DefaultEventHandler );

	lString = DlgString_Initialise( aMessage, false );
	Group_AddObject( lDlg->fGroup, lString, 2, 2, lWidth, lHeight-4, FLAG_CENTREX );

	lButton = DlgButton_Initialise( "&Ok", ENDMODAL_OK, true );
	Group_AddObject( lDlg->fGroup, lButton, 0, lHeight-2, 8, lHeight-1, FLAG_CENTREX );

	Dialog_ShowModal( lDlg );

	Dialog_Kill( lDlg );
}

/*****************************************************************************

    Function: Application_RegisterConfigVars

    Description: Registers all of the configuration variables in an application.
    Parameters: None
    Return: None

*****************************************************************************/

void	Application_RegisterConfigVars( void )
{
   Config_RegisterVar( "Mouse", CFG_USE_MOUSE, "mouse", CFGVAR_STRING, "on", true );

	Application_CustomRegisterConfigVars();
}

/*****************************************************************************

    Function: Application_CheckConfigVar

    Description:
		Checks if a configuration variable has a valid value. Note that 1000->2000 are reserved for colours.

    Parameters: The variable
    Return: Whether it was valid or not.

*****************************************************************************/

bool	Application_CheckConfigVar( T_ConfigVar *aVar )
{
	char	lStr[255];
	bool	lRetVal = true;

	if ( aVar->fKey >= 1000 && aVar->fKey < 2000 )
	{
		if ( aVar->fValue.fInteger < 0 || aVar->fValue.fInteger > 255 )
		{
			lRetVal = false;
		}
		return lRetVal;
	}

	switch ( aVar->fKey )
	{
		case	CFG_USE_MOUSE:
			if ( strcasecmp( aVar->fValue.fString, "on" ) != 0 &&	strcasecmp( aVar->fValue.fString, "off" ) != 0 )
			{
				lRetVal = false;
			}
			break;
		default:
			if ( !Application_CustomCheckConfigVar( aVar, &lRetVal ))
			{
				sprintf( lStr,	"Unknown configuration variable: %s", aVar->fName );
				Error( lStr );
				lRetVal = false;
			}
			break;
	}
	return lRetVal;
}

/*****************************************************************************

    Function: Application_RegisterEventCommands

    Description: Register all the valid application commands with the C-Desktop library.
    Parameters: None
    Return: None

*****************************************************************************/

void	Application_RegisterEventCommands( void )
{
	Event_RegisterCommand( CMD_EXIT );

	Application_CustomRegisterEventCommands();
}

/*****************************************************************************

    Function: Application_AboutCDesktop

    Description: Show a dialog box with version and copyright information about C-Desktop.
    Parameters: None
    Return: None

*****************************************************************************/

void	Application_AboutCDesktop( void )
{
	T_Dialog		*lDlg;
	T_DlgString	*lString;
	T_DlgButton	*lButton;
	char	lStr[256];

	lDlg = Dialog_Initialise( "About C-Desktop", 0, 0, 39, 11, FLAG_CENTRE, Dialog_DefaultEventHandler );

	sprintf( lStr,	"This program utilises:\n\nThe C-Desktop Library\nVersion %d.%d.%d\n\nCopyright (c)1998, Brett Porter",
 				__CDESKTOP_VERSION_MAJOR__, __CDESKTOP_VERSION_MINOR__, __CDESKTOP_VERSION_BUGFIX__ );

	lString = DlgString_Initialise( lStr, true );
	Group_AddObject( lDlg->fGroup, lString, 1, 2, 58, 7, FLAG_CENTREX );

	lButton = DlgButton_Initialise( "&Ok", ENDMODAL_OK, true );
	Group_AddObject( lDlg->fGroup, lButton, 0, 9, 8, 10, FLAG_CENTREX );

	Dialog_ShowModal( lDlg );

	Dialog_Kill( lDlg );
}


