/*****************************************************************************

 HANGMAN version 1.0
 by Chris La Mantia

 This code is released into the public domain.

 Purpose: This simple adaptation of the classic paper-and-pencil game Hangman
          is used to demonstrate the use of Text Data Objects with the
          Allegro game programming library.  All of the strings used
          in the game (with the exception of error messages) are stored
          in an Allegro datafile as a single Text Data Object.  The game
          could easily be ported to other languages simply by creating
          a different datafile for each language.

*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <allegro.h>
#include <time.h>
#include <textdata.h>
#include "hangman.h"
#include "wordlist.h"

#define BLACK     255
#define RED         1
#define BLUE        4

char get_a_char(int resetflag);
void draw_scaffold(int part);

int main(void)
{
   int word;                  /* Index into word list                    */
   int quitflag=0;            /* Flag to indicate end of session         */
   int winflag;               /* Flag to indicate game win               */
   int matchflag;             /* Flag to indicate a successful guess     */
   int matchcount;            /* Counter for matched characters          */
   int errcount;              /* Counter for incorrect guesses           */
   int x;                     /* X coordinate of beginning of word       */
   int y=140;                 /* Y coordinate of word                    */
   int len;                   /* Word length                             */
   int index;                 /* Loop index                              */
   char charout[2]=" ";       /* Buffer to hold character for textout    */
   DATAFILE *gamedata;
   FONT *swiss;
   char **wordlist;           /* Pointer for wordlist                    */

   srandom((int)time(NULL));  /* Randomize random number seed            */
   allegro_init();            /* Initialize Allegro                      */
   install_textdata();        /* Install text data object (TDO) handlers */
   install_timer();
   install_keyboard();
   install_mouse();
   gamedata=load_datafile("hangman.dat");
   if(gamedata==NULL) {
      printf("HANGMAN Error: Cannot load HANGMAN.DAT\n");
      return(1);
   }
   wordlist=(char **)gamedata[WORDLIST].dat; /* Set pointer to base of TDO */
   swiss=(FONT *)gamedata[SWISS].dat;
   if(set_gfx_mode(GFX_AUTODETECT,320,200,0,0)<0) {
      printf("HANGMAN Error: %s\n",allegro_error);
      return(1);
   }
   set_pallete(desktop_pallete);

   /*** MAIN GAME LOOP ***/

   do{
      clear(screen);
      errcount=matchcount=0;
      get_a_char(1);

      /* Display game title from the text data object */
      textout_centre(screen,swiss,wordlist[TITLE],160,5,BLUE);

      /* Select a random word from the text data object */
      word=random()%(LASTWORD+1);

      len=strlen(wordlist[word]);
      x=SCREEN_W/2-(8*len+4*(len-1))/2;
      for(index=0;index<len;index++)
         hline(screen,x+index*12,y+10,x+index*12+8,RED);
      winflag=0;
      while(errcount<11&&winflag==0) {
         charout[0]=get_a_char(0);
         matchflag=0;
         for(index=0;index<len;index++) {
            if(charout[0]==wordlist[word][index]) {
               matchflag=1;
               textout(screen,font,charout,x+index*12,y,BLUE);
               matchcount++;
            }
         }
         if(matchflag==0)
            draw_scaffold(errcount++);
         if(matchcount==len)
            winflag=1;
      }
      for(index=0;index<len;index++) {
         charout[0]=wordlist[word][index];
         textout(screen,font,charout,x+index*12,y,RED);
      }
      /* Show game end dialog..  Note that all of the text in the dialog,
         including the button labels, comes from the TDO in the datafile. */
      if(alert(wordlist[LOSE_TEXT+winflag],wordlist[REPLAY],NULL,
               wordlist[YES_TEXT],wordlist[NO_TEXT],0,0)==2)
         quitflag=1;

   } while(quitflag==0);

   return 0;
}

char get_a_char(int resetflag)
{
static int picked[26];
int index;
int choice;

DIALOG char_dlg[] =
   {
       /* (dialog proc) (x) (y) (w) (h) (fg) (bg) (key)  (flags) (d1)(d2) (dp) */
       { d_button_proc,  4, 170, 12, 12, 255,   0,  'a', D_EXIT,   0,  0, "A"},
       { d_button_proc, 16, 170, 12, 12, 255,   0,  'b', D_EXIT,   0,  0, "B"},
       { d_button_proc, 28, 170, 12, 12, 255,   0,  'c', D_EXIT,   0,  0, "C"},
       { d_button_proc, 40, 170, 12, 12, 255,   0,  'd', D_EXIT,   0,  0, "D"},
       { d_button_proc, 52, 170, 12, 12, 255,   0,  'e', D_EXIT,   0,  0, "E"},
       { d_button_proc, 64, 170, 12, 12, 255,   0,  'f', D_EXIT,   0,  0, "F"},
       { d_button_proc, 76, 170, 12, 12, 255,   0,  'g', D_EXIT,   0,  0, "G"},
       { d_button_proc, 88, 170, 12, 12, 255,   0,  'h', D_EXIT,   0,  0, "H"},
       { d_button_proc,100, 170, 12, 12, 255,   0,  'i', D_EXIT,   0,  0, "I"},
       { d_button_proc,112, 170, 12, 12, 255,   0,  'j', D_EXIT,   0,  0, "J"},
       { d_button_proc,124, 170, 12, 12, 255,   0,  'k', D_EXIT,   0,  0, "K"},
       { d_button_proc,136, 170, 12, 12, 255,   0,  'l', D_EXIT,   0,  0, "L"},
       { d_button_proc,148, 170, 12, 12, 255,   0,  'm', D_EXIT,   0,  0, "M"},
       { d_button_proc,160, 170, 12, 12, 255,   0,  'n', D_EXIT,   0,  0, "N"},
       { d_button_proc,172, 170, 12, 12, 255,   0,  'o', D_EXIT,   0,  0, "O"},
       { d_button_proc,184, 170, 12, 12, 255,   0,  'p', D_EXIT,   0,  0, "P"},
       { d_button_proc,196, 170, 12, 12, 255,   0,  'q', D_EXIT,   0,  0, "Q"},
       { d_button_proc,208, 170, 12, 12, 255,   0,  'r', D_EXIT,   0,  0, "R"},
       { d_button_proc,220, 170, 12, 12, 255,   0,  's', D_EXIT,   0,  0, "S"},
       { d_button_proc,232, 170, 12, 12, 255,   0,  't', D_EXIT,   0,  0, "T"},
       { d_button_proc,244, 170, 12, 12, 255,   0,  'u', D_EXIT,   0,  0, "U"},
       { d_button_proc,256, 170, 12, 12, 255,   0,  'v', D_EXIT,   0,  0, "V"},
       { d_button_proc,268, 170, 12, 12, 255,   0,  'w', D_EXIT,   0,  0, "W"},
       { d_button_proc,280, 170, 12, 12, 255,   0,  'x', D_EXIT,   0,  0, "X"},
       { d_button_proc,292, 170, 12, 12, 255,   0,  'y', D_EXIT,   0,  0, "Y"},
       { d_button_proc,304, 170, 12, 12, 255,   0,  'z', D_EXIT,   0,  0, "Z"},
       { NULL }
   };

   if(resetflag==1) {
      for(index=0;index<26;index++){
         picked[index]=0;
      }
      return(0);
   }
   for(index=0;index<26;index++)
      if(picked[index]==1) {
         char_dlg[index].flags|=D_DISABLED;
         char_dlg[index].flags|=D_SELECTED;
      }
   do
      choice=do_dialog(char_dlg,-1);
   while(choice<0||choice>25);
   picked[choice]=1;
   return(choice+97);
}

void draw_scaffold(int part)
{
   int top=30;
   int center=SCREEN_W/2;

   switch(part) {
      case 0: hline(screen,center-25,top+70,center+25,BLACK);
              vline(screen,center-25,top+70,top+75,BLACK);
              vline(screen,center+25,top+70,top+75,BLACK);
              break;
      case 1: vline(screen,center,top,top+70,BLACK);
              break;
      case 2: hline(screen,center-35,top,center,BLACK);
              break;
      case 3: line(screen,center-7,top,center,top+7,BLACK);
              break;
      case 4: vline(screen,center-35,top,top+5,BLACK);
              break;
      case 5: circle(screen,center-35,top+10,5,RED);
              putpixel(screen,center-37,top+8,BLUE);
              putpixel(screen,center-33,top+8,BLUE);
              hline(screen,center-36,top+12,center-34,BLACK);
              break;
      case 6: vline(screen,center-35,top+16,top+27,RED);
              break;
      case 7: hline(screen,center-41,top+21,center-35,RED);
              break;
      case 8: hline(screen,center-35,top+21,center-29,RED);
              break;
      case 9: line(screen,center-35,top+27,center-40,top+36,RED);
              break;
      case 10:line(screen,center-35,top+27,center-30,top+36,RED);
              putpixel(screen,center-37,top+13,BLACK);
              putpixel(screen,center-33,top+13,BLACK);
              line(screen,center-38,top+7,center-36,top+9,BLACK);
              line(screen,center-34,top+7,center-32,top+9,BLACK);
              line(screen,center-38,top+9,center-36,top+7,BLACK);
              line(screen,center-34,top+9,center-32,top+7,BLACK);
              break;

   }
}
