/*
 * CPU.CPP - Contains CPU functions.
 * Copyright (C) 1998, 1999 Prashant TR
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING.TR for more details.
*/

// ID for this file.
#define _CPU_CC_

#include "cpu.h"

int pinrun(const char *, const char *);

int errflag = 0;
FILE *fp;

char far cpu__vendor[13] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ,0, 0, 0 };
char cmdline[20];

int index;

typedef struct processor_tag {
	char name[40];
	unsigned id;
} PROCESSOR;

PROCESSOR processor[][37] =
{
	// Intel Processors.
	{
		{ "80386SX", 			0x30 },
		{ "80386DX", 			0x34 },

		{ "80486DX 25/33",            	0x40 },
		{ "80486SX",            	0x140 },
		{ "80486DX-S 50",          	0x41 },
		{ "80486SX-S",			0x42 },
		{ "80486DX2-S",			0x43 },
		{ "80486SL",			0x44 },
		{ "80486DX2",			0x144 },
		{ "80486SX2-S",			0x45 },
		{ "80486 Overdrive",		0x46 },
		{ "80486DX2-WB",		0x47 },
		{ "80486DX4",			0x48 },
		{ "80486DX4-WB",		0x49 },

		{ "Pentium 60/66 A-step",	0x50 },
		{ "Pentium (P5) 60/66",		0x51 },
		{ "Pentium (P54C) 75-200",	0x52 },
		{ "Pentium (P54C)",		0x152 },
		{ "Pentium Overdrive PODP5V83",	0x53 },
		{ "Pentium (P54C)",		0x153 },
		{ "Pentium (P55C MMX)",		0x54 },
		{ "Pentium (CT)",		0x154 },
		{ "Pentium (P54CM)",		0x254 },
		{ "Pentium (MMX Overdrive)", 	0x55 },
		{ "Mobile Pentium (P54C) 75-200",0x57 },
		{ "Mobile Pentium MMX (P54CS)",	0x58 },

		{ "PentiumPro A-step",		0x60 },
		{ "PentiumPro",			0x61 },
		{ "Pentium II (Klamanth)",	0x63 },
		{ "Pentium II Overdrive", 	0x163 },
		{ "Pentium Pro (P55CT)",	0x164 },
		{ "Pentium II Overdrive",	0x264 },
		{ "Pentium II Xeon",		0x65 },
		{ "Celeron (Covington)",	0x65 },
		{ "Pentium II (Deschutes)",	0x65 },
		{ "Celeron (Mendocino)",	0x66 },
		{ "",				0xffff },
	},

	// Cyrix.
	// This isn't the way to do it for Cyrix. But for now...
	{
		{ "486SX",            		0x140 },
		{ "486DX2",			0x144 },
		{ "486DX",            		0x40 },
		{ "486DX-S",          		0x41 },
		{ "486SX-S",			0x42 },
		{ "486DX2-S",			0x43 },
		{ "486SL",			0x44 },
		{ "Gx86",			0x440 }, // Don't change.
		{ "486SX2-S",			0x45 },
		{ "P24D",			0x46 },
		{ "486DX4/IBM Blue Lighting DX3",0x47 },
		{ "486DX4-ODP",			0x48 },
		{ "P24T",			0x49 },

		{ "6x86",			0x52 },
		{ "6x86",			0x53 },
		{ "M-II/IBM Cyrix 6x86MX",	0x60 },
		{ "",				0xffff },
	},

	// AMD.
	{
		{ "386DXLV",           		0x30 },
		{ "386SXLV",			0x230 },

		{ "486DX2-S",			0x43 },
		{ "486DX2-WB",			0x47 },
		{ "486DX4",			0x48 },
		{ "486DX4-WB",			0x49 },
		{ "Am5x86-WT",			0x4e },
		{ "Am5x86-WB",			0x4ff },

		{ "K5/SSA5",			0x50 },
		{ "K5",				0x51 },
		{ "K5",				0x52 },
		{ "K5",				0x53 },
		{ "K6",				0x56 },
		{ "K6",				0x57 },
		{ "K6-II",			0x58 },
		{ "K6-III",			0x59 },
		{ "",				0xffff },
	},

	// UMC.
	{
		{ "U5D",          		0x41 },
		{ "U5S",			0x42 },
		{ "",				0xffff },
	},

	// Centaur (IDT).
	{
		{ "C6",           		0x54 },
		{ "C6-II",           		0x58 },
		{ "",				0xffff },
	},

	// TI.
	{
		{ "486DX2-S",			0x43 },
		{ "",				0xffff },
	},

	// NexGen.
	{
		{ "Nx586",			0x50 },
		{ "",				0xffff },
	},

};

// Write any string to the file and check for successfulness.
void writestring(const char *string)
{
 if (fprintf(fp, "%s", string) == EOF) {
    errflag = 2;
    checkerrors();
 }
}

int sysinfo()
{
 char output[256], *vendor = cpu__vendor;
 unsigned long cpu__family, cpu__model, cpu__stepping, has__cpuid;
 unsigned long cpu__flags, cpu__value;
 int known, vendorid, exitcode, i;
 float cpu__speed, p_rating, divcount;
 FILE *ftemp;

 known = 1; // Assume that we know the CPU.

 // Create output file.
 if ((fp = fopen("cpu.txt", "w")) == NULL) {
    errflag = 1;
    checkerrors();
 }

 // Get cpu information.
 writestring("\nCPU :\n");
 has__cpuid = has_idflag();
 cpu__family = get_cpu_family();
 if (has__cpuid) {
    cpu__model = get_cpu_model();
    cpu__stepping = get_cpu_stepping_id();
    vendor = get_cpu_vendor_name();
    cpu__flags = get_cpu_feature_flags();
    cpu__value = get_cpu_value() >> 4;
 }

 // Get the CPU name.
 strcpy(output, "");
 known = 0;

 // Check if we are on a Cyrix 486 or a TI 486.
 if ((!has__cpuid) && (is_cyrix486())) {
    char x;
    outportb(0x22, 0xff);
    x = inportb(0x23);
    outportb(0x22, 0xfe);
    inportb(0x23);
    vendorid = 5; // Assume TI.
    if (x & 0x80) strcat(output, "TI ");
    else {
	 strcat(output, "Cyrix ");
	 vendorid = 1;
    }

    if (cpu__family == 3) {
	if (is_386dx()) cpu__value = 0x34;
	else cpu__value = 0x30;
    }

    if (cpu__family == 4) {
	if (is_486dx()) cpu__value = 0x144;
	else cpu__value = 0x140;
    }
 }

 else if (!has__cpuid) {
	// Intel CPU.
	vendorid = 0;
	if (cpu__family == 3) {
		if (is_386dx()) cpu__value = 0x34;
		else cpu__value = 0x30;
	}

	if (cpu__family == 4) {
		if (is_486dx()) cpu__value = 0x144;
		else cpu__value = 0x140;
	}
 }

 // Remove mask bits if family <= 4.
 if ((has__cpuid) && (cpu__family <= 4)) cpu__value &= 0xfff;

 // Check CPU vendors.
 if (has__cpuid) {
	if (!strcmp(vendor, "GenuineIntel")) { strcat(output, "Intel ");
					vendorid = 0; known++; }
	else if (!strcmp(vendor, "CyrixInstead")) { strcat(output, "Cyrix ");
					     vendorid = 1; }
	else if (!strcmp(vendor, "AuthenticAMD")) { strcat(output, "AMD ");
					     vendorid = 2; }
	else if (!strcmp(vendor, "AMD ISBETTER")) { strcat(output, "AMD ");
					     vendorid = 2; }
	else if (!strcmp(vendor, "UMC UMC UMC ")) { strcat(output, "UMC ");
					     vendorid = 3; }
	else if (!strcmp(vendor, "CentaurHauls")) {
					     strcat(output, "Centaur (IDT) ");
					     vendorid = 4; }
	else if (!strcmp(vendor, "NexGenDriven")) { strcat(output, "NexGen ");
					     vendorid = 5; }
	else vendorid = -1;
 }

 // Write CPU family information.
 i = 0;
 if (vendorid != -1) {
	// Check for Pentium II Xeon, Celeron and Pentium II (D).
	if ((!vendorid) && (cpu__value == 0x65)) {
		if ((get_cpu_value() & 0xffff) == 0x652) index = 32;
		else
			switch (get_xeon_celeron_p2d()) {
				case 1:
					// Celeron.
					index = 33;
					break;
				case 2:
					// PII (Deschutes).
					index = 34;
					break;
			}
	}
	else if ((vendorid == 1) && (cpu__value == 0x44) &&
		(!cpu__stepping))
		// Cyrix Generic x86.
		index = 7;
	else {
		while ((processor[vendorid][i].id != 0xffff) &&
			(processor[vendorid][i].id != cpu__value)) i++;


		if (processor[vendorid][i].id != 0xffff) index = i;
		else index = -1;
	}
 }

 if (index != -1) strcat(output, processor[vendorid][index].name);
 else strcat(output, "(Unknown)");

 if ((!vendorid) && (cpu__family >= 5) && (cpu__flags & 0x800000L))
					strcat (output, " - MMX");
 // Write to file.
 writestring("\tComputer Name                  : IBM PC/AT or compatible\n");
 writestring("\tProcessor Identification       : ");
 writestring(output);
 writestring("\n");
 sprintf(output, "\tSelf-Identification support    : %s\n",
					(has__cpuid) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tCPU Family                     : %ld\n", cpu__family);
 writestring(output);

 // Get processor speed.
 if ((ftemp = fopen("cmdline.$$$", "wb")) == NULL)
	fprintf(stderr, "Error: Argument setup failed (SPEED.EXE)\r\n");
 else {
  fprintf(ftemp, "getspeed\r\n");
  fclose(ftemp);
  unlink("temp.$$$");
  exitcode = pinrun("sysinfo.pin", "speed.exe");
  if (exitcode == -1)
     fprintf(stderr, "Error: Could not spawn SPEED.EXE\r\n");
  else {
      FILE *fp;
      if ((fp = fopen("temp.$$$", "rb")) == NULL)
	 fprintf(stderr, "Error: Cannot open temporary file for reading\r\n");

      else if (fscanf(fp, "%f %f %f", &divcount, &cpu__speed, &p_rating) != 3)
	      fprintf(stderr, "Error: Cannot read temporary file\r\n");
      else {
	   fclose(fp);
	   unlink("temp.$$$");
	   sprintf(output, "\tProcessor Speed                : %2.1fMHz\n",
							     cpu__speed);
	   writestring(output);
	   sprintf(output, "\tP-Rating                       : PR%2.1f\n",
							     p_rating);
	   writestring(output);
      }
  }
 }

 // If no cpuid support, simply return.
 if (!has__cpuid) return 0;

 // Write this only if it has CPUID support.
 sprintf(output, "\tCPU Model                      : %ld\n", cpu__model);
 writestring(output);
 sprintf(output, "\tCPU Stepping ID                : %ld\n", cpu__stepping);
 writestring(output);
 sprintf(output, "\tCPU Vendor ID                  : %s\n", vendor);
 writestring(output);
 sprintf(output, "\tProcessor Type                 : %s\n",
						(cpu__value & 0x200) == 0 ?
							"Original OEM" :
						(cpu__value & 0x200) == 1 ?
							"Overdrive" :
							"Dual");
 writestring(output);
 sprintf(output, "\tCPU Feature flags              : %lX\r\n", cpu__flags);
 writestring(output);

 // Write CPU features.
 sprintf(output, "\tBuilt in Math Coprocessor      : %s\n",
				  (cpu__flags & 1) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tVirtual Mode Extensions        : %s\n",
				 (cpu__flags & 2) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tDebugging Extensions           : %s\n",
			      (cpu__flags & 4) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tPage Size Extensions           : %s\n",
			      (cpu__flags & 8) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tTime Stamp Counter             : %s\n",
			       (cpu__flags & 0x10) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tModel Specific Registers       : %s\n",
				   (cpu__flags & 0x20) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tMachine Check Exceptions       : %s\n",
				  (cpu__flags & 0x80) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tCMPXCHG8B Instruction          : %s\n",
			      (cpu__flags & 0x100) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tPage Global Enable             : %s\n",
				(cpu__flags & 0x2000) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tConditional Move Instructions  : %s\n",
				(cpu__flags & 0x8000L) ? "Yes" : "No");
 writestring(output);
 sprintf(output, "\tMMX Technology                 : %s\n",
			(cpu__flags & 0x800000L) ? "Yes" : "No");
 writestring(output);

 // For Intel.
 if (!vendorid) {
	sprintf(output, "\tPhysical Address Extensions    : %s\n",
				     (cpu__flags & 0x40) ? "Yes" : "No");
	writestring(output);
	sprintf(output, "\tBuilt in Advanced PIC          : %s\n",
			     (cpu__flags & 0x200) ? "Yes" : "No");
	writestring(output);
	sprintf(output, "\tFast System Calls              : %s\n",
			 (cpu__flags & 0x4000L) ? "Yes" : "No");
	writestring(output);
	sprintf(output, "\tMemory type Range registers    : %s\n",
				(cpu__flags & 0x1000) ? "Yes" : "No");
	writestring(output);
	sprintf(output, "\tMachine Check Architecture     : %s\n",
				  (cpu__flags & 0x4000) ? "Yes" : "No");
	writestring(output);
	sprintf(output, "\tPage Attribute Table           : %s\n",
				   (cpu__flags & 0x10000L) ? "Yes" : "No");
	writestring(output);
	sprintf(output, "\t36-bit Page Size Extension     : %s\n",
				(cpu__flags & 0x20000L) ? "Yes" : "No");
	writestring(output);
	sprintf(output, "\tFast Float save and Restore    : %s\n",
			 (cpu__flags & 0x1000000L) ? "Yes" : "No");
	writestring(output);
 }
 // For Cyrix.
 else if (vendorid == 1) {
	sprintf(output, "\tCyrix Extended MMX Technology  : %s\n",
			(cpu__flags & 0x800000L) ? "Yes" : "No");
	writestring(output);
 }
 // For AMD.
 else if (vendorid == 2) {
	sprintf(output, "\tFCMOV instructions             : %s\n",
			(cpu__flags & 0x10000L) ? "Yes" : "No");
	writestring(output);
	sprintf(output, "\tAMD3D technology               : %s\n",
			(cpu__flags & 0x80000000L) ? "Yes" : "No");
	writestring(output);
 }

 fclose(fp);

 return 0;
}

void open_stderr()
{
	fclose(stdout);
	fclose(&_streams[2]);
	if (fopen("nul", "wb") == NULL) exit(0x7f);
	if (fopen("nul", "wb") == NULL) exit(0x7f);
	if ((stderr = fopen("errors.$$$", "ab")) == NULL) exit(0x7f);
}

void get_cmdline()
{
 if ((fp = fopen("cmdline.$$$", "rb")) == NULL) exit (0x7f);

 if (fscanf(fp, "%s", cmdline) != 1) {
		fclose(fp);
		exit (0x7f);
 }

 fclose(fp);
 unlink("cmdline.$$$");

}

#pragma argsused

int main(int argc, char **argv)
{
 // Check if it was called with correct arguments.
 open_stderr();

 // Get command line.
 get_cmdline();

 if (!strcmp(cmdline, "sysinfo")) return(sysinfo());

 return 0;
}
