#! /bin/sh

top=`pwd`
archive=${1-gcc-3.3.6.tar.bz2}
export SHELL=$DJDIR/bin/sh.exe

AUTOCONF_OLD=${AUTOCONF_OLD-autoconf}
AUTOHEADER_OLD=${AUTOHEADER_OLD-autoheader}
ACLOCAL_OLD=${ACLOCAL_OLD-aclocal}
AUTOMAKE_OLD=${AUTOMAKE_OLD-automake}

echo "Removing old source and build directories (if present) ..."
rm -rf gnu || exit 1
mkdir gnu

################ Verify autoconf and automake versions ##############


echo "Checking autoconf and automake versions ..."
autotools_ok=true

autoconf_old_ver=`$AUTOCONF_OLD --version 2>/dev/null | sed 1q | sed -e 's#^[^0-9]*##'`
case $autoconf_old_ver in
    2.13) 
	;;
    *) 
        echo ""
	echo "Autoconf 2.13 required. Please set environment variable "
	echo "AUTOCONF_OLD to it (default value is autoconf)"
	echo "Current settings: AUTOCONF_OLD=$AUTOCONF_OLD VERSION=$autoconf_old_ver"
	autotools_ok=false
	;;
esac	 

autoheader_old_ver=`$AUTOHEADER_OLD --version 2>/dev/null | sed 1q | sed -e 's#^[^0-9]*##'`
case $autoheader_old_ver in
    2.13) 
	;;
    *) 
        echo ""
	echo "Autoheader 2.13 required. Please set environment variable "
	echo "AUTOHEADER_OLD to it (default value is autoheader)"
	echo "Current settings: AUTOHEADER_OLD=$AUTOHEADER_OLD VERSION=$autoheader_old_ver"
	autotools_ok=false
	;;
esac	 

automake_old_ver=`$AUTOMAKE_OLD --version 2>/dev/null | sed 1q | sed -e 's#^[^0-9]*##'`
case $automake_old_ver in
    1.[45]) 
	;;
    *) 
        echo ""
	echo "Automake 1.4 or 1.5 required (1.5 recommended). Please set"
	echo "environment variable AUTOMAKE_OLD to it (default value is automake)"
	echo "Current settings: AUTOMAKE_OLD=$AUTOMAKE_OLD VERSION=$automake_old_ver"
	autotools_ok=false
	;;
esac	 

aclocal_old_ver=`$ACLOCAL_OLD --version 2>/dev/null | sed 1q | sed -e 's#^[^0-9]*##'`
case $aclocal_old_ver in
    1.[45]) 
	;;
    *) 
        echo ""
	echo "Aclocal 1.4 or 1.5 required (1.5 recommended). Please set"
	echo "environment variable ACLOCAL_OLD to it (default value is aclocal)"
	echo "Current settings: ACLOCAL_OLD=$ACLOCAL_OLD VERSION=$aclocal_old_ver"
	autotools_ok=false
	;;
esac	 

if ! $autotools_ok ; then
    echo ""
    echo "Correct versions of autoconf and(or) automake are not available"
    echo "Please install them if needed or set needed environment variables"
    echo ""
    exit 1
fi    

#####################################################################

echo "Unpacking sources and preparing DJGPP source archive ..."
exec >unpack-`date +%y%m%d` 2>&1

rm -f $top/s-sources
rm -f $top/s-patch

rm -rf gnu
mkdir gnu
case $archive in 
      *.tgz | *.tar.gz )
         ( cd gnu && tar xzf $top/$archive && echo $archive >$top/s-sources )
         ;;
      *.tar.bz2 )
         ( cd gnu && tar xf $top/$archive --use=bzip2 && echo $archive >$top/s-sources )
         ;;
      *)
         echo Unknown archive format
	 exit 1	
         ;; 
esac

if ! [ -f s-sources ] ; then
    echo "Cannot unpack GCC sources"
    exit 1
fi

dir_name=`(cd gnu && ls -d gcc* | sed 1q; )`

version=`cat gnu/$dir_name/gcc/version.c |
    grep version_string | 
    sed -e 's,^.*[=],,' | 
    sed -e 's,[";],,g' | 
    sed -e 's,^[\ ],,g'`

short_version=`echo $version | sed -e 's,[\ ].*,,' -e 's,^[ ]+,,g'`
arcprefix=gcc`echo $short_version | sed -e 's,\.,,g'`
srcname=gcc-`echo $short_version | sed -e 's,^\(3\.[0-9][0-9]*\)\.\(..*\),\1\2,g'` 
version_status=
echo $version | grep -q experimental && version_status=-experimental
echo $version | grep -q prerelease && version_status=-prerelease

dir_name0=$dir_name
case $dir_name in
    gcc-3.[0-9]-*)
        ;;
    gcc-3.[0-9])
        ;;
    gcc-3.[0-9].[0-9]*)
        dir_name=`echo $dir_name | sed -e 's,\.,,2'`
        ;;
    gcc)
        dir_name=$srcname$version_status
        ;;
esac

if ! test "$dir_name" = "$dir_name0" ; then
    mv -f gnu/$dir_name0 gnu/$dir_name || exit 1
fi

if [ -d $top/gnu/$dir_name ] ; then
    gcc_dir=$top/gnu/$dir_name
else
    echo "Error: directory $top/gnu/$dir_name is not found"
    exit 1
fi

rm -rf manifest

#
#  Remove parts we don't need for DJGPP
#

rm -rf $gcc_dir/boehm-gc
rm -rf $gcc_dir/fastjar
rm -rf $gcc_dir/libffi
rm -rf $gcc_dir/libjava
rm -rf $gcc_dir/libchill
rm -rf $gcc_dir/zlib
#rm -rf $gcc_dir/gcc/ada
rm -rf $gcc_dir/gcc/java
rm -rf $gcc_dir/gcc/testsuite
rm -rf $gcc_dir/gcc/treelang

#
#  List of directories in which we should run autoconf or/and automake
#

automake_list=
aclocal_list=

#
#  Apply DJGPP related diffs and copy files 
#

for x in `(cd diffs/source && find . -type f | sed -e 's/\.\///g')`; do
    case $x in 
       *.diff | *.diff.in)
           cd $top
           diff=`pwd`/$x;
           fname=`echo $x | sed -e 's/\.diff//g'`
           echo "Applying patch to gcc/$fname ..."
           if [ -f $gcc_dir/$fname ] ; then
              cp -f $gcc_dir/$fname $gcc_dir/${fname}~1 
              if ! patch --verbose $gcc_dir/$fname <diffs/source/$x ; then
                 echo "Failed to apply patch for $gcc_dir/$fname"
                 exit 1
              fi
              case $fname in 
                  */acinclude.m4)
                     dir=`echo $gcc_dir/$fname | sed -e 's/\/acinclude.m4//g'`
                      aclocal_list="$aclocal_list $dir"
                      ;;
                  */Makefile.am)
                      dir=`echo $gcc_dir/$fname | sed -e 's/\/Makefile.am//g'`
                      automake_list="$automake_list $dir"
                      ;;
              esac
           else
              Echo "File $gcc_dir/$fname is not found"
           fi
           ;;
       *~*)
           ;;
       *)
           echo Copying $x
	   mkdir -p `dirname $gcc_dir/$x`		   	
           if ! cp -fv diffs/source/$x $gcc_dir/$x ; then
              echo "Failed to copy diffs/source/$x to $gcc_dir/$x"
              exit 1
           fi
           ;;
       esac
done

for dir in build.gcc install.gcc; do
    for x in `(cd diffs/$dir && find . -type f | sed -e 's/\.\///g')`; do
        mkdir -p `dirname gnu/$dir/$x`
	echo "Writting $dir/$x"
	cat diffs/$dir/$x |\
	sed -e "s:@SRCDIR@:$srcname$version_status:g" >gnu/$dir/$x
    done
done

#
#  Run aclocal where needed
#

for dir in $aclocal_list ; do
   if cd $dir ; then
      echo Running aclocal in `pwd`...
      if ! $ACLOCAL_OLD ; then
         echo "aclocal failed in directory `pwd`"
         cd $top
         exit 1
      fi
   fi
   cd $top
done

#
#  Run automake where needed
#

for dir in $automake_list; do
   if cd $dir ; then
      if ! [ -f configure.in ] ; then
         cd ..;
         if ! [ -f configure.in ] ; then
            cd ..
         fi
      fi
      echo Running automake in `pwd`...
      if ! $AUTOMAKE_OLD ; then
         echo "automake failed in directory `pwd`"
         cd $top
         exit 1
      fi
   fi
   cd $top
done

#
#  Run autoconf where needed
#

for c in `find $gcc_dir -name 'configure.in'`; do
   d=`echo $c | sed -e 's#/configure.in##g'`
   if cd $d ; then
      if grep -q AC_OUTPUT configure.in ; then
          echo Running autoconf in `pwd` ...;
          if ! $AUTOCONF_OLD ; then 
             echo "autoconf failed in directory `pwd`"
             cd $top
             exit 1
	  fi
      fi
   fi
   cd $top
done

#
#  Run autoheader where needed
#

if cd $gcc_dir/gcc ; then
   echo Running autoheader in `pwd` ...;
   if ! $AUTOHEADER_OLD ; then
      echo "autoheader failed in directory `pwd`"
      cd $top
      exit 1
   fi
   rm -f cstamp-h.in
   echo timestamp >cstamp-h.in
fi

if cd $gcc_dir/libf2c/libI77 ; then
   echo Running autoheader in `pwd` ...;
   if ! $AUTOHEADER_OLD ; then
      echo "autoheader failed in directory `pwd`"
      cd $top
      exit 1
   fi
   rm -f stamp-h.in
   echo timestamp > stamp-h.in
fi

if [ -f $gcc_dir/contrib/gcc_update ] ; then
   echo Running contrib/gcc_update --touch
   cd $gcc_dir;
   ./contrib/gcc_update --touch
   cd $top
fi

cd $top

mkdir manifest
echo ${arcprefix}s${version_status}.zip - GCC $version source archive for DJGPP >$top/manifest/${arcprefix}s${version_status}.ver
touch $top/manifest/${arcprefix}s${version_status}.mft

( find gnu -type f \
          -and -not -name '.cvsignore' \
          -and -not -name '*~*' \
          -and -not -name '*.orig' \
          -and -not -name '.brik' ;
  find manifest -type f; ) |\
  grep -v 'autom4te\.cache' |\
  sort >$top/manifest/${arcprefix}s${version_status}.mft

rm -f $top/${arcprefix}s${version_status}.zip
cat $top/manifest/${arcprefix}s${version_status}.mft | zip -9q@ $top/${arcprefix}s${version_status}.zip
