/*
 *  pnmpsnr.c: Compute error (RMSE, PSNR) between images
 *
 *
 *  Derived from pnmpnsmr by Ulrich Hafner, part of his fiasco package,
 *  On 2001.03.04.

 *  Copyright (C) 1994-2000 Ullrich Hafner <hafner@bigfoot.de>
 */

#include <stdio.h>
#include <math.h>

#include "pam.h"

#define MAXFILES 16

static double
square(const double arg) {
    return(arg*arg);
}


static void
validate_input(const struct pam pam1, const struct pam pam2) {

    if (pam1.width != pam2.width)
        pm_error("images are not the same width, so can't be compared.  "
                 "The first is %d columns wide, "
                 "while the second is %d columns wide.",
                 pam1.width, pam2.width);
    if (pam1.height != pam2.height)
        pm_error("images are not the same height, so can't be compared.  "
                 "The first is %d rows high, "
                 "while the second is %d rows high.",
                 pam1.height, pam2.height);

    if (strcmp(pam1.tuple_type, pam2.tuple_type) != 0)
        pm_error("images are not of the same type.  The tuple types are "
                 "'%s' and '%s', respectively.",
                 pam1.tuple_type, pam2.tuple_type);

    if (strcmp(pam1.tuple_type, PAM_PBM_TUPLETYPE) != 0 &&
        strcmp(pam1.tuple_type, PAM_PGM_TUPLETYPE) != 0 &&
        strcmp(pam1.tuple_type, PAM_PPM_TUPLETYPE) != 0)
        pm_error("Images are not of a PNM type.  Tuple type is '%s'",
                 pam1.tuple_type);
}



static void
psnr_grayscale(tuple ** const image1, const struct pam pam1,
               tuple ** const image2, const struct pam pam2,
               char filespec1[], char filespec2[]) {

    double norm;
    int row;
       
    norm = 0;  /* initial value */
    
    for (row = 0; row < pam1.height; row++) {
        int col;
        for (col = 0; col < pam1.width; col++)
            norm += square(image1[row][col][0] - image2[row][col][0]);
    }

    norm /= pam1.width * pam1.height;
    norm /= pam1.maxval * pam2.maxval;
    
    if (norm > 1e-9)
        pm_message("PSNR between %s and %s: %.2f dB",
                   filespec1, filespec2, 10 * log10(1/norm));
    else
        pm_message("Images %s and %s don't differ.",
                   filespec1, filespec2);
}


static void
psnr_color(tuple ** const image1, const struct pam pam1,
              tuple ** const image2, const struct pam pam2,
              char filespec1[], char filespec2[]) {

    double y_norm, cb_norm, cr_norm;
    int row;
    
    y_norm = cb_norm = cr_norm = 0.0;
    
    for (row = 0; row < pam1.height; row++) {
        int col;
        for (col = 0; col < pam1.width; col++) {
            double y1, y2, cb1, cb2, cr1, cr2;
            
            pnm_YCbCrtuple(image1[row][col], &y1, &cb1, &cr1);
            pnm_YCbCrtuple(image2[row][col], &y2, &cb2, &cr2);
            
            y_norm  += square(y1  - y2);
            cb_norm += square(cb1 - cb2);
            cr_norm += square(cr1 - cr2);
        }
    }
    /* Take mean per-pixel values */
    y_norm  /= pam1.width * pam1.height;
    cb_norm /= pam1.width * pam1.height;
    cr_norm /= pam1.width * pam1.height;

    /* Normalize all components to 0..1.0 */
    y_norm /= pam1.maxval * pam2.maxval;
    cb_norm /= pam1.maxval * pam2.maxval;
    cr_norm /= pam1.maxval * pam2.maxval;
        
    pm_message("PSNR between %s and %s:", filespec1, filespec2);
    if (y_norm > 1e-9)
        pm_message("Y  color component: %.2f dB", 10 * log10(1/y_norm));
    else
        pm_message("Y color component doesn't differ.");
    if (cb_norm > 1e-9)
        pm_message("Cb color component: %.2f dB", 10 * log10(1/cb_norm));
    else
        pm_message("Cb color component  doesn't differ.");
    if (cr_norm > 1e-9)
        pm_message("Cr color component: %.2f dB", 10 * log10(1/cr_norm));
    else
        pm_message("Cr color component doesn't differ.");
}



int
main (int argc, char **argv) {
    char *filespec1, *filespec2;  /* specs of two files to compare */
    FILE *file1, *file2;
    struct pam pam1, pam2;

    tuple **image1, **image2;  /* malloc'ed */

    pnm_init(&argc, argv);

    if (argc < 2) 
        pm_error("Takes two arguments:  specifications of the two files.");
    else {
        filespec1 = argv[1];
        filespec2 = argv[2];
    }
    
    file1 = pm_openr(filespec1);
    file2 = pm_openr(filespec2);

    image1 = pnm_readpam(file1, &pam1, sizeof(pam1));
    image2 = pnm_readpam(file2, &pam2, sizeof(pam2));

    validate_input(pam1, pam2);

    if (strcmp(pam1.tuple_type, PAM_PPM_TUPLETYPE) != 0) 
        psnr_grayscale(image1, pam1, image2, pam2, filespec1, filespec2);
    else 
        psnr_color(image1, pam1, image2, pam2, filespec1, filespec2);

    pnm_freepamarray(image1, &pam1);
    pnm_freepamarray(image2, &pam2);

    return 0;
}

