/*    EPZip - the simple free distribution preparing tool
**    Copyright (C) 1997  Esa Peuha

**    This program is free software; you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation; either version 2 of the License, or
**    (at your option) any later version.

**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.

**    You should have received a copy of the GNU General Public License
**    along with this program; if not, write to the Free Software
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include "epzip.h"

struct match
{
  unsigned short length, distance;
};

static unsigned short init_hash(char *);
static void update_hash(unsigned short *);

static void init_hash_table(void);
static void store_hash(unsigned short, unsigned int);
static struct match search_match(unsigned char *, unsigned short,
				 unsigned int, unsigned int);
static void free_hash_table(void);

static unsigned short comp_string(unsigned char *, unsigned char *,
				  unsigned short);

static void put_begin(void);
static void put_literal(unsigned char);
static void put_string(struct match);
static unsigned int put_end(void);

FILE *substfile;

void substitute(char *buffer, unsigned int length, struct file_el *value)
{
  unsigned int written = 0;

  substfile = value->fp = tmpfile();
  put_begin();
  if(length > 3)
    {
      unsigned int hashed = 0;
      struct match match_to_write, current_match;
      unsigned short current_hash = init_hash(buffer);
      init_hash_table();
      current_match.length = 0;

      while(hashed + 3 < length)
	{
	  match_to_write = current_match;
	  store_hash(current_hash, hashed++);
	  update_hash(&current_hash);
	  if(written < hashed + 1)
	    current_match = search_match(buffer, current_hash,
					 hashed, length);
	  if(written < hashed)
	    {
	      if(match_to_write.length == 0 ||
		 match_to_write.length < current_match.length)
		put_literal(buffer[written++]);
	      else
		{
		  put_string(match_to_write);
		  written += match_to_write.length;
		}
	    }
	}
      free_hash_table();
    }
  while(written < length)
    put_literal(buffer[written++]);
  value->elements = put_end();
}



unsigned char *hash_pointer;

static unsigned short init_hash(char *buffer)
{
  unsigned short value = 0;
  hash_pointer = buffer;
  update_hash(&value);
  update_hash(&value);
  update_hash(&value);
  return value;
}

static void update_hash(unsigned short *value)
{
  *value <<= 6;
  *value ^= *hash_pointer++;
}



struct hash_chain_member
{
  unsigned short prev, next, hash;
};

struct hash_chain_member *hash_chain;

unsigned short *latest_hash;

static void init_hash_table(void)
{
  unsigned int i;
  hash_chain = malloc(0x8000 * sizeof(struct hash_chain_member));
  if(hash_chain == NULL)
    {
      fprintf(stderr, "epzip: can't allocate hash table\n");
      exit(1);
    }
  for(i = 0; i < 0x8000; i++)
    hash_chain[i].prev = hash_chain[i].next = 0xffff;
  latest_hash = malloc(0x10000 * sizeof(unsigned short));
  if(latest_hash == NULL)
    {
      fprintf(stderr, "epzip: can't allocate hash table\n");
      exit(1);
    }
  for(i = 0; i < 0x10000; i++)
    latest_hash[i] = 0xffff;

}

static void store_hash(unsigned short hash, unsigned int offset)
{
  offset &= 0x7fff;

  if(hash_chain[offset].prev == 0xffff)
    {
      if(latest_hash[hash_chain[offset].hash] == offset)
	latest_hash[hash_chain[offset].hash] = 0xffff;
    }
  else
    hash_chain[hash_chain[offset].prev].next = 0xffff;

  hash_chain[offset].hash = hash;
  hash_chain[offset].prev = 0xffff;
  hash_chain[offset].next = latest_hash[hash];
  latest_hash[hash] = offset;
  if(hash_chain[offset].next != 0xffff)
    hash_chain[hash_chain[offset].next].prev = offset;
}

static struct match search_match(unsigned char *buffer, unsigned short hash,
				 unsigned int offset, unsigned int length)
{
  struct match value, temp;
  unsigned short modulo_offset = offset & 0x7fff, distance,
    hash_offset = latest_hash[hash];
  value.length = 0;
  while(1)
    {
      if(hash_offset == 0xffff)
	return value;
      if(hash_chain[hash_offset].hash != hash)
	{
	  fprintf(stderr, "epzip: internal error: hash chain malfunction\n");
	  exit(1);
	}
      distance = hash_offset < modulo_offset ? modulo_offset - hash_offset :
	modulo_offset + 0x8000 - hash_offset;
      if(buffer[offset] == buffer[offset - distance] &&
	 buffer[offset + 1] == buffer[offset - distance + 1] &&
	 buffer[offset + 2] == buffer[offset - distance + 2])
	{
	  temp.distance = distance;
	  temp.length = 3 + comp_string(buffer + offset + 3,
					buffer + offset - distance + 3,
					offset + 258 < length ? 255 :
					length - offset - 3);
	  if(temp.length > value.length)
	    value = temp;
	}
      hash_offset = hash_chain[hash_offset].next;
    }
}

static void free_hash_table(void)
{
  free(hash_chain);
  free(latest_hash);
}

static unsigned short comp_string(unsigned char *p1, unsigned char *p2,
				  unsigned short max)
{
  unsigned short i;
  for(i = 0; i < max; i++)
    if(p1[i] != p2[i])
      return i;
  return max;
}


unsigned int elements;

static void put_begin(void)
{
  elements = 0;
  bwrite_begin(substfile);
}

static void put_literal(unsigned char value)
{
  elements++;
  bwrite(0, 1);
  bwrite(value, 8);
}

static void put_string(struct match value)
{
  elements++;
  bwrite(1, 1);
  bwrite(value.length - 3, 8);
  bwrite(value.distance - 1, 15);
}

static unsigned int put_end(void)
{
  bwrite_end();
  return elements;
}
