/*
 * Copyright (c) 1996, 2003 VIA Networking, Inc. All rights reserved.
 *
 * This software may be redistributed and/or modified under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 *
 * File: vmns_drv.h
 *
 * Purpose: Header file defined the VMNS connection interface.
 *
 * Author: Chuang Liang-Shing, AJ Jiang
 *
 * Date: Jan 24, 2003
 *
 */


/*
*    Description:
*
*        This file defined the interface between driver and VMNS.
*
*/
#ifndef _VMNS_DRV_H
#define _VMNS_DRV_H

#ifdef __KERNEL__
#include <linux/netdevice.h>
#endif

#include "rhine_cfg.h"

#if (LINUX_VERSION_CODE < 0x2030d)
#define net_device device
#endif //#if (LINUX_VERSION_CODE < 0x2030d)

/***************************************************************************
*
*    This section communication interface between VMNS and device driver via
* ioctl calls.
*
****************************************************************************/

/* The SIOC code for VMNS*/

#define VMNS_DRV_SIOC           (SIOCDEVPRIVATE+8) /*????*/
#ifndef MAX_SIGNATURE_LEN
#define MAX_SIGNATURE_LEN       20
#define VMNS_SIGNATURE          "VMNS\0"
#define VMNS_DRV_SIGNATURE      "VMNS_DRV\0"
#define VMNS_CMD_SIGNATURE      "VMNS_CMD\0"
#define VMNS_FRAME_TYPE         0x8888
#endif

typedef
enum _VMNS_STATUS {
    VMNS_DRV_STATUS_OK=0,
    VMNS_DRV_STATUS_BUSY,
    VMNS_DRV_STATUS_FAIL
} VMNS_DRV_STATUS, *PVMNS_DRV_STATUS;

typedef
enum  _VMNS_DRV_OPCODE {
    VMNS_DRV_OPCODE_IDENTIFY,       /*To see if device driver support VMNS*/
    VMNS_DRV_OPCODE_VMNS_EXIT,      /*Tell driver I'm going to levave*/
    VMNS_DRV_OPCODE_COMMIT,
    VMNS_DRV_OPCODE_DISCONNECT,
    VMNS_DRV_OPCODE_QUERY_STATUS,   /*Query link speed/duplex/tx bufs/rd bufs status*/
    VMNS_DRV_OPCODE_SET_STATUS,
    VMNS_DRV_OPCODE_QUERY_CAP,      /*Query adapter's capabilities*/
    VMNS_DRV_OPCODE_SET_VLAN,       /*Set PQEN, VIDFR*/
    VMNS_DRV_OPCODE_SET_VCAM,       /*Set vcam, mcam, taged/untaged*/
    VMNS_DRV_OPCODE_GET_VCAM,       /*Get vcam, mcam, taged/untage status*/
    VMNS_DRV_OPCODE_SET_MCAM,
    VMNS_DRV_OPCODE_GET_MCAM,
    VMNS_DRV_OPCODE_SET_MODE,
    VMNS_DRV_OPCODE_GET_MODE,
    VMNS_DRV_OPCODE_GET_REG,        /*For debug purpose only*/
    VMNS_DRV_OPCODE_SET_REG,        /*For debug purpose only*/
    VMNS_DRV_OPCODE_SET_CALLBACK,
    VMNS_DRV_OPCODE_LAST
} VMNS_DRV_OPCODE, *PVMNS_DRV_OPCODE;

typedef
struct _VMNS_DRV_SIOC_HEADER {
    VMNS_DRV_OPCODE OpCode;
    VMNS_DRV_STATUS ret_status;
} VMNS_DRV_SIOC_HEADER, *PVMNS_DRV_SIOC_HEADER;

/*Structure for VMNS_DRV_OPCODE_INDENTIFY*/
typedef
struct _VMNS_DRV_SIOC_IDENTIFY {
    VMNS_DRV_SIOC_HEADER    hdr;
    char            Signature1[MAX_SIGNATURE_LEN];  /*Fill this with caller signature*/
    char            Signature2[MAX_SIGNATURE_LEN];  /*Fill this with callee signature*/
    version_t       version1;                       /*Fill this with caller version*/
    version_t       version2;
    unsigned char       revisionID;                 /*Fill this with callee version*/
} VMNS_DRV_SIOC_IDENTIFY, *PVMNS_DRV_SIOC_IDENTIFY;

/*Structures for setting CAMs*/
typedef
struct _VMNS_DRV_SIOC_SET_VCAM {
    VMNS_DRV_SIOC_HEADER    hdr;
    UINT                    offset;
    UINT                    num;
    UINT32                  cam_mask;
    union                   {
        unsigned short      vcam[32];
        unsigned char       mcam[32][6];
    } u;
} VMNS_DRV_SIOC_SET_VCAM, *PVMNS_DRV_SIOC_SET_VCAM;

typedef
VMNS_DRV_SIOC_SET_VCAM
VMNS_DRV_SIOC_GET_VCAM, *PVMNS_DRV_SIOC_GET_VCAM,
VMNS_DRV_SIOC_SET_MCAM, *PVMNS_DRV_SIOC_SET_MCAM,
VMNS_DRV_SIOC_GET_MCAM, *PVMNS_DRV_SIOC_GET_MCAM;

#define VMNS_DRV_MODE_DISAU     0x00000001UL
#define VMNS_DRV_MODE_DISAB     0x00000002UL
#define VMNS_DRV_MODE_DISAM     0x00000004UL
#define VMNS_DRV_MODE_VIDFR     0x00000008UL
#define VMNS_DRV_MODE_PQEN      0x00000010UL
#define VMNS_DRV_MODE_RTGOPT    0x00000020UL

typedef
struct _VMNS_DRV_SIOC_SET_MODE {
    VMNS_DRV_SIOC_HEADER    hdr;
    unsigned long           mode;
} VMNS_DRV_SIOC_SET_MODE, *PVMNS_DRV_SIOC_SET_MODE,
  VMNS_DRV_SIOC_GET_MODE, *PVMNS_DRV_SIOC_GET_MODE;

/*Structures for set/get reg*/
typedef
enum _vmns_reg_size {
    VMNS_REG_BYTE=1,
    VMNS_REG_WORD=2,
    VMNS_REG_DWORD=4,
    VMNS_REG_QWORD=8
} VMNS_REG_SIZE, *PVMNS_REG_SIZE;

typedef
enum _vmns_reg_type {
    VMNS_REG_TYPE_MAC,
    VMNS_REG_TYPE_PCI,
    VMNS_REG_TYPE_MII
}   VMNS_REG_TYPE, *PVMNS_REG_TYPE;

typedef
struct _VMNS_DRV_SIOC_SET_REG {
    VMNS_DRV_SIOC_HEADER    hdr;
    VMNS_REG_TYPE       type;
    UINT32              offset;
    VMNS_REG_SIZE       size;
    UCHAR               data[8];
} VMNS_DRV_SIOC_SET_REG, *PVMNS_DRV_SIOC_SET_REG;

typedef
VMNS_DRV_SIOC_SET_REG
VMNS_DRV_SIOC_GET_REG, *PVMNS_DRV_SIOC_GET_REG;

/*Structures for VMNS_DRV_OPCODE_QUERY_STATUS*/
typedef
enum _QUERY_STATUS_ {
    QUERY_LINK_STATUS,
    QUERY_SPEED_STATUS,
    SET_LINK_STATUS,
    SET_SPEED_STATUS,
} QUERY_DRV_STATUS_ID, *PQUERY_DRV_STATUS_ID;

typedef
enum _LINK_DUPLEX_STATUS {
    LINK_FAIL,
    LINK_HALF_DUPLEX,
    LINK_FULL_DUPLEX
} LINK_STATUS, *PLINK_STATUS;

typedef
enum _LINK_SPEED_STATUS {
    VMNS_SPEED_1,
    VMNS_SPEED_10,
    VMNS_SPEED_100,
    VMNS_SPEED_1000
} LINK_SPEED_STATUS, *PLINK_SPEED_STATUS;

typedef
struct _VMNS_DRV_SIOC_QUERY {
    VMNS_DRV_SIOC_HEADER    hdr;
    QUERY_DRV_STATUS_ID     status_id;
    union {
                            LINK_STATUS         link_status;
                            LINK_SPEED_STATUS   speed_status;
    } u;
} VMNS_DRV_SIOC_QUERY_STATUS, *PVMNS_DRV_SIOC_QUERY_STATUS,
    VMNS_DRV_SIOC_SET_STATUS, *PVMNS_DRV_SIOC_SET_STATUS;

/***************************************************************************
*
*    This section defines per packet data structure
*
****************************************************************************/
/*
    Note: Incoming packet protocol type must be VMNS_FRAME_TYPE
*/
typedef unsigned int ATT_INFO_TYPE;
#define ATT_INFO_TYPE_NONE      0x00000000UL
#define ATT_INFO_TYPE_TAG       0x00000001UL
#define ATT_INFO_TYPE_INST      0x00000002UL
#define ATT_INFO_TYPE_PRI       0x00000004UL
#define ATT_INFO_TYPE_VIDHIT    0x00000008UL

typedef
struct _vmns_attach_info {
    ATT_INFO_TYPE   type;
    unsigned int    protocol;
    unsigned short  vlan;
    unsigned char   priority;
    unsigned char   InstNo;
} VMNS_ATTACH_INFO, *PVMNS_ATTACH_INFO;

#define GET_VMNS_ATTACH_INFO(skb) \
            ((PVMNS_ATTACH_INFO) &((skb)->cb))

/***************************************************************************
*
*    This section defines a callback mechanism for driver to notify VMNS.
*
****************************************************************************/
typedef
enum _VMNS_DRV_EVENT {
    VMNS_EVENT_STATUS_CHANGE,
    VMNS_EVENT_LINK_FAIL,
    VMNS_EVENT_LINK_UP,
    VMNS_EVENT_HW_FAIL,
    VMNS_EVENT_STOP_TRANSMIT,   /*For flow control*/
    VMNS_EVENT_START_TRANSMIT,  /*For flow control*/
    VMNS_EVENT_RESERVED
} VMNS_DRV_EVENT, *PVMNS_DRV_EVENT;

typedef
void (*VMNS_CALLBACK_FUNC) (char *, VMNS_DRV_EVENT);

typedef
struct _VMNS_DRV_CALLBACK {
    VMNS_DRV_SIOC_HEADER    hdr;
    VMNS_CALLBACK_FUNC      func;
} VMNS_DRV_SIOC_CALLBACK, *PVMNS_DRV_SIOC_CALLBACK;


#endif
