/*
 * Copyright (c) 1996, 2003 VIA Networking, Inc. All rights reserved.
 *
 * This software may be redistributed and/or modified under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 *
 * File: rhine_vmns.c
 *
 * Purpose: Functions to connect to VMNS.
 *
 * Author: Chuang Liang-Shing, AJ Jiang
 *
 * Date: Jan 24, 2003
 *
 * Functions:
 *      vmns_drv_identify       -   Function to identify & connect to VMNS.
 *      vmns_drv_commit         -   Notifiy driver the VMNS has began to process.
 *      vmns_drv_disconnect     -   Function to disconnect from VMNS.
 *      vmns_drv_exit_vmns      -   Exit.
 *      vmns_drv_set_callback   -   Set call-back function to notify VMNS.
 *      vmns_drv_query_status   -   Queuy the current link status.
 *      vmns_drv_set_status     -   Set the current link status.
 *      vmns_drv_set_mcam       -   Set the multicast cam addresses.
 *      vmns_drv_get_mcam       -   Get the multicast cam addresses.
 *      vmns_drv_set_vcam       -   Set the VLAN ID cam addresses.
 *      vmns_drv_set_mode       -   Set the driver packet filter mode.
 *      vmns_drv_get_mode       -   Get the driver packet filter mode.
 *      vmns_drv_get_vcam       -   Get the VLAN ID cam addresses.
 *
 */


#include "rhine.h"
#include "rhine_cfg.h"
#include "vmns_drv.h"

typedef
VMNS_DRV_STATUS (*VMNS_IOCTL_FUNCTION) (struct net_device*, PVMNS_DRV_SIOC_HEADER);

#define DECLARE_FUNC(x) \
    static VMNS_DRV_STATUS x(struct net_device*,PVMNS_DRV_SIOC_HEADER)

DECLARE_FUNC(vmns_drv_identify);
DECLARE_FUNC(vmns_drv_commit);
DECLARE_FUNC(vmns_drv_disconnect);
DECLARE_FUNC(vmns_drv_exit_vmns);
DECLARE_FUNC(vmns_drv_set_callback);
DECLARE_FUNC(vmns_drv_query_status);
DECLARE_FUNC(vmns_drv_set_status);
DECLARE_FUNC(vmns_drv_set_mcam);
DECLARE_FUNC(vmns_drv_get_mcam);
DECLARE_FUNC(vmns_drv_set_vcam);
DECLARE_FUNC(vmns_drv_set_mode);
DECLARE_FUNC(vmns_drv_get_mode);
DECLARE_FUNC(vmns_drv_get_vcam);

#ifdef vmns_drv_read_reg
DECLARE_FUNC(vmns_drv_read_reg);
#endif

static
struct {
    VMNS_DRV_OPCODE         opcode;
    VMNS_IOCTL_FUNCTION     function;
}   func_table[]    =
{
    {VMNS_DRV_OPCODE_IDENTIFY,          vmns_drv_identify},
    {VMNS_DRV_OPCODE_COMMIT,            vmns_drv_commit},
    {VMNS_DRV_OPCODE_DISCONNECT,        vmns_drv_disconnect},
    {VMNS_DRV_OPCODE_VMNS_EXIT,         vmns_drv_exit_vmns},
    {VMNS_DRV_OPCODE_SET_CALLBACK,      vmns_drv_set_callback},
    {VMNS_DRV_OPCODE_SET_MCAM,          vmns_drv_set_mcam},
    {VMNS_DRV_OPCODE_GET_MCAM,          vmns_drv_get_mcam},
    {VMNS_DRV_OPCODE_SET_VCAM,          vmns_drv_set_vcam},
    {VMNS_DRV_OPCODE_GET_VCAM,          vmns_drv_get_vcam},
    {VMNS_DRV_OPCODE_SET_MODE,          vmns_drv_set_mode},
    {VMNS_DRV_OPCODE_GET_MODE,          vmns_drv_get_mode},
    {VMNS_DRV_OPCODE_QUERY_STATUS,      vmns_drv_query_status},
    {VMNS_DRV_OPCODE_SET_STATUS,        vmns_drv_set_status},
    {VMNS_DRV_OPCODE_LAST,              NULL}
};

int vmns_process_ioctl(struct net_device* dev, PVMNS_DRV_SIOC_HEADER pParams) {
    int i;
    for (i=0;func_table[i].opcode!=VMNS_DRV_OPCODE_LAST;i++)
        if (pParams->OpCode==func_table[i].opcode) {
            pParams->ret_status=func_table[i].function(dev,pParams);
            if (pParams->ret_status==VMNS_DRV_STATUS_OK)
                return 0;
            else
                return -EAGAIN;
        }
    return -EOPNOTSUPP;
}

static VMNS_DRV_STATUS
vmns_drv_identify(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO         pInfo = dev->priv;

    pInfo->flags|=RHINE_FLAGS_VMNS_CONNECTED;

    memcpy(((PVMNS_DRV_SIOC_IDENTIFY) hdr)->Signature2,
        VMNS_DRV_SIGNATURE,strlen(VMNS_DRV_SIGNATURE)+1);

    ((PVMNS_DRV_SIOC_IDENTIFY) hdr)->version2.major=MAJOR_VERSION;
    ((PVMNS_DRV_SIOC_IDENTIFY) hdr)->version2.minor=MINOR_VERSION;
    ((PVMNS_DRV_SIOC_IDENTIFY) hdr)->revisionID = pInfo->byRevId;

    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_exit_vmns(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO         pInfo = dev->priv;
    if (dev->stop(dev)!=0)
        return VMNS_DRV_STATUS_BUSY;

    pInfo->flags &= (~RHINE_FLAGS_VMNS_COMMITTED);
    pInfo->flags &= (~RHINE_FLAGS_VMNS_CONNECTED);
    pInfo->multicast_limit =32;
    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_commit(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO         pInfo = dev->priv;

    if (pInfo->flags & RHINE_FLAGS_VMNS_COMMITTED)
        return VMNS_DRV_STATUS_BUSY;

    pInfo->flags|=RHINE_FLAGS_VMNS_COMMITTED;

    pInfo->multicast_limit -=4;
    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_disconnect(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO         pInfo = dev->priv;
    PVMNS_DRV_PRIVATE   vpriv=GET_VMNS_PRIVATE(pInfo);


    pInfo->flags &=~(RHINE_FLAGS_VMNS_COMMITTED|RHINE_FLAGS_VMNS_CONNECTED);
    vpriv->notify=NULL;
    pInfo->multicast_limit =32;
    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_set_callback(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO         pInfo = dev->priv;
    PVMNS_DRV_PRIVATE   vpriv=GET_VMNS_PRIVATE(pInfo);
    PVMNS_DRV_SIOC_CALLBACK param=(PVMNS_DRV_SIOC_CALLBACK) hdr;
    vpriv->notify=param->func;
    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_query_status(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PVMNS_DRV_SIOC_QUERY_STATUS param=(PVMNS_DRV_SIOC_QUERY_STATUS) hdr;
    PRHINE_INFO pInfo=(PRHINE_INFO) dev->priv;

    U32 mii_status=mii_check_media_mode(pInfo->pMacRegs);

    switch(param->status_id) {
    case QUERY_LINK_STATUS:
        if (mii_status & RHINE_LINK_FAIL)
            param->u.link_status=LINK_FAIL;
        else if (mii_status & RHINE_DUPLEX_FULL)
            param->u.link_status=LINK_FULL_DUPLEX;
        else
            param->u.link_status=LINK_HALF_DUPLEX;

        break;
    case QUERY_SPEED_STATUS:
        if (mii_status & RHINE_SPEED_100)
            param->u.speed_status=VMNS_SPEED_100;
        else
            param->u.speed_status=VMNS_SPEED_10;
        break;
    default:
        return VMNS_DRV_STATUS_FAIL;
    }
    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_set_status(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PVMNS_DRV_SIOC_SET_STATUS param=(PVMNS_DRV_SIOC_SET_STATUS) hdr;
    switch(param->status_id) {
    case SET_LINK_STATUS:

        break;
    case SET_SPEED_STATUS:
        break;
    default:
        return VMNS_DRV_STATUS_FAIL;
    }
    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_set_mcam(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO         pInfo = dev->priv;
    PVMNS_DRV_SIOC_SET_MCAM param=(PVMNS_DRV_SIOC_SET_MCAM) hdr;
    U8  oldmask[8];
    PU8 pMask=(PU8) &param->cam_mask;
    int     i;

    mac_get_cam_mask(pInfo->pMacRegs, oldmask, RHINE_MULTICAST_CAM);

    //Only allow first 32 cam to be used by VMNS
    if ((param->offset+param->num) >32 ) {
        return VMNS_DRV_STATUS_FAIL;
    }

    for (i=param->offset;i<(param->offset+param->num);i++) {
        mac_set_cam(pInfo->pMacRegs, i, &param->u.mcam[i-param->offset][0],
            RHINE_MULTICAST_CAM);
        oldmask[i/8]|=(1<<(i & 7));
    }

    for (i=0;i<4;i++)
        oldmask[i]&=*pMask++;

    mac_set_cam_mask(pInfo->pMacRegs,oldmask, RHINE_MULTICAST_CAM);

    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_get_mcam(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO         pInfo = dev->priv;
    PVMNS_DRV_SIOC_GET_MCAM param=(PVMNS_DRV_SIOC_GET_MCAM) hdr;
    int i;
    U8  oldmask[8];
    PU8 pMask=(PU8) &param->cam_mask;

    if ((param->offset+param->num) >32 ) {
        return VMNS_DRV_STATUS_FAIL;
    }

    for (i=param->offset;i<(param->offset+param->num);i++)  {
        mac_get_cam(pInfo->pMacRegs, i, &param->u.mcam[i][0], RHINE_MULTICAST_CAM);
    }

    mac_get_cam_mask(pInfo->pMacRegs, oldmask, RHINE_MULTICAST_CAM);

    for (i=0;i<4;i++)
        *pMask++=oldmask[i];

    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_set_vcam(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PVMNS_DRV_SIOC_SET_VCAM param=(PVMNS_DRV_SIOC_SET_VCAM) hdr;
    PRHINE_INFO         pInfo = dev->priv;
    int i;
    U8  oldmask[8];
    PU8 pMask=(PU8) &param->cam_mask;

    mac_get_cam_mask(pInfo->pMacRegs, oldmask, RHINE_VLAN_ID_CAM);

    if ((param->offset+param->num) >32 ) {
        return VMNS_DRV_STATUS_FAIL;
    }

    for (i=param->offset;i<(param->offset+param->num);i++)  {
        mac_set_cam(pInfo->pMacRegs, i, (PU8) &param->u.vcam[i-param->offset],
            RHINE_VLAN_ID_CAM);
        oldmask[i/8]|=(1<<(i & 7));
    }

    for (i=0;i<4;i++)
        oldmask[i]&=*pMask++;

    mac_set_cam_mask(pInfo->pMacRegs,oldmask, RHINE_VLAN_ID_CAM);

    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_get_vcam(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO         pInfo = dev->priv;
    PVMNS_DRV_SIOC_GET_VCAM param=(PVMNS_DRV_SIOC_GET_VCAM) hdr;
    int i;
    U8  oldmask[8];
    PU8 pMask=(PU8) &param->cam_mask;

    if ((param->offset+param->num) >32 ) {
        return VMNS_DRV_STATUS_FAIL;
    }

    for (i=param->offset;i<param->offset+param->num;i++)  {
        mac_get_cam(pInfo->pMacRegs, i, (PU8) &param->u.vcam[i-param->offset],
            RHINE_VLAN_ID_CAM);
    }

    mac_get_cam_mask(pInfo->pMacRegs, oldmask, RHINE_VLAN_ID_CAM);
    for (i=0;i<4;i++)
        *pMask++=oldmask[i];
    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_set_mode(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO             pInfo = dev->priv;
    PVMNS_DRV_PRIVATE       vpriv=GET_VMNS_PRIVATE(pInfo);
    PVMNS_DRV_SIOC_SET_MODE param=(PVMNS_DRV_SIOC_SET_MODE) hdr;
    vmns_set_mode(dev,vpriv,param->mode);
    return VMNS_DRV_STATUS_OK;
}

static VMNS_DRV_STATUS
vmns_drv_get_mode(struct net_device* dev, PVMNS_DRV_SIOC_HEADER hdr) {
    PRHINE_INFO             pInfo = dev->priv;
    PVMNS_DRV_PRIVATE       vpriv=GET_VMNS_PRIVATE(pInfo);
    PVMNS_DRV_SIOC_GET_MODE param=(PVMNS_DRV_SIOC_GET_MODE) hdr;
    param->mode=vpriv->mode;
    return VMNS_DRV_STATUS_OK;
}

void vmns_set_mode(struct net_device* dev, PVMNS_DRV_PRIVATE vpriv,
    unsigned long mode) {
    PRHINE_INFO         pInfo = dev->priv;
    PMAC_REGS           pMacRegs=pInfo->pMacRegs;

    vpriv->mode=mode;
    if (mode & VMNS_DRV_MODE_DISAU)
        writel(CR0_DISAU,&pMacRegs->dwCR0Set);
    else
        writel(CR0_DISAU,&pMacRegs->dwCR0Clr);

    if (mode & VMNS_DRV_MODE_PQEN)
        BYTE_REG_BITS_ON(MCFG_PQEN,&pMacRegs->wMCFG);
    else
        BYTE_REG_BITS_OFF(MCFG_PQEN,&pMacRegs->wMCFG);

    if (mode & VMNS_DRV_MODE_RTGOPT)
        BYTE_REG_BITS_ON(MCFG_RTGOPT,&pMacRegs->wMCFG);
    else
        BYTE_REG_BITS_OFF(MCFG_RTGOPT,&pMacRegs->wMCFG);


    if (mode & VMNS_DRV_MODE_DISAB)
        BYTE_REG_BITS_OFF(RCR_AB,&pMacRegs->byRCR);
    else
        BYTE_REG_BITS_ON(RCR_AB,&pMacRegs->byRCR);

    if (mode & VMNS_DRV_MODE_DISAM)
        BYTE_REG_BITS_OFF(RCR_AM,&pMacRegs->byRCR);
    else
        BYTE_REG_BITS_ON(RCR_AM,&pMacRegs->byRCR);
}
