/*
 * Copyright (c) 1996, 2003 VIA Networking Technologies, Inc.
 * All rights reserved.
 *
 * This software is copyrighted by and is the sole property of
 * VIA Networking Technologies, Inc. This software may only be used
 * in accordance with the corresponding license agreement. Any unauthorized
 * use, duplication, transmission, distribution, or disclosure of this
 * software is expressly forbidden.
 *
 * This software is provided by VIA Networking Technologies, Inc. "as is"
 * and any express or implied warranties, including, but not limited to, the
 * implied warranties of merchantability and fitness for a particular purpose
 * are disclaimed. In no event shall VIA Networking Technologies, Inc.
 * be liable for any direct, indirect, incidental, special, exemplary, or
 * consequential damages.
 *
 *
 * File: velocity_main.c
 *
 * Purpose: Functions for Linux drver interfaces.
 *
 * Author: Chuang Liang-Shing, AJ Jiang
 *
 * Date: Jan 24, 2003
 *
 */


#undef __NO_VERSION__


#include "velocity.h"


static int              velocity_nics               =0;
static PVELOCITY_INFO   pVelocity3_Infos            =NULL;
static int              msglevel                    =MSG_LEVEL_INFO;


#ifdef  VELOCITY_ETHTOOL_IOCTL_SUPPORT
static int  velocity_ethtool_ioctl(struct net_device* dev, struct ifreq* ifr);
#endif

#ifdef SIOCGMIIPHY
static int  velocity_mii_ioctl(struct net_device* dev, struct ifreq* ifr, int cmd);
#endif


/*
    Define module options
*/

MODULE_AUTHOR("ZyXEL Communications corporation");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("ZyXEL GN600 series 1000Base-T Adapter Driver");

#define VELOCITY_PARAM(N,D) \
        static const int N[MAX_UINTS]=OPTION_DEFAULT;\
        MODULE_PARM(N, "1-" __MODULE_STRING(MAX_UINTS) "i");\
        MODULE_PARM_DESC(N, D);

#define RX_DESC_MIN     64
#define RX_DESC_MAX     255
#define RX_DESC_DEF     64
VELOCITY_PARAM(RxDescriptors,"Number of receive descriptors");

#define TX_DESC_MIN     16
#define TX_DESC_MAX     256
#define TX_DESC_DEF     64
VELOCITY_PARAM(TxDescriptors,"Number of transmit descriptors");

#define VLAN_ID_MIN     0
#define VLAN_ID_MAX     4095
#define VLAN_ID_DEF     0
/* VID_setting[] is used for setting the VID of NIC.
   0: default VID.
   1-4094: other VIDs.
*/
VELOCITY_PARAM(VID_setting,"802.1Q VLAN ID");

#define RX_THRESH_MIN   0
#define RX_THRESH_MAX   3
#define RX_THRESH_DEF   0
/* rx_thresh[] is used for controlling the receive fifo threshold.
   0: indicate the rxfifo threshold is 128 bytes.
   1: indicate the rxfifo threshold is 512 bytes.
   2: indicate the rxfifo threshold is 1024 bytes.
   3: indicate the rxfifo threshold is store & forward.
*/
VELOCITY_PARAM(rx_thresh,"Receive fifo threshold");

#define DMA_LENGTH_MIN  0
#define DMA_LENGTH_MAX  7
#define DMA_LENGTH_DEF  0

/* DMA_length[] is used for controlling the DMA length
   0: 8 DWORDs
   1: 16 DWORDs
   2: 32 DWORDs
   3: 64 DWORDs
   4: 128 DWORDs
   5: 256 DWORDs
   6: SF(flush till emply)
   7: SF(flush till emply)
*/
VELOCITY_PARAM(DMA_length,"DMA length");

#define TAGGING_DEF     0
/* enable_tagging[] is used for enabling 802.1Q VID tagging.
   0: disable VID seeting(default).
   1: enable VID setting.
*/
VELOCITY_PARAM(enable_tagging,"Enable 802.1Q tagging");

#define IP_ALIG_DEF     0
/* IP_byte_align[] is used for IP header DWORD byte aligned
   0: indicate the IP header won't be DWORD byte aligned.(Default) .
   1: indicate the IP header will be DWORD byte aligned.
      In some enviroment, the IP header should be DWORD byte aligned,
      or the packet will be droped when we receive it. (eg: IPVS)
*/
VELOCITY_PARAM(IP_byte_align,"Enable IP header dword aligned");

#ifdef VELOCITY_TX_CSUM_SUPPORT
#define TX_CSUM_DEF     1
/* txcsum_offload[] is used for setting the checksum offload ability of NIC.
   (We only support RX checksum offload now)
   0: disable csum_offload[checksum offload
   1: enable checksum offload. (Default)
*/
VELOCITY_PARAM(txcsum_offload,"Enable transmit packet checksum offload");
#endif

#define FLOW_CNTL_DEF   1
#define FLOW_CNTL_MIN   1
#define FLOW_CNTL_MAX   5

/* flow_control[] is used for setting the flow control ability of NIC.
   1: hardware deafult - AUTO (default). Use Hardware default value in ANAR.
   2: enable TX flow control.
   3: enable RX flow control.
   4: enable RX/TX flow control.
   5: disable
*/
VELOCITY_PARAM(flow_control,"Enable flow control ability");

#define MED_LNK_DEF 0
#define MED_LNK_MIN 0
#define MED_LNK_MAX 4
/* speed_duplex[] is used for setting the speed and duplex mode of NIC.
   0: indicate autonegotiation for both speed and duplex mode
   1: indicate 100Mbps half duplex mode
   2: indicate 100Mbps full duplex mode
   3: indicate 10Mbps half duplex mode
   4: indicate 10Mbps full duplex mode

   Note:
        if EEPROM have been set to the force mode, this option is ignored
            by driver.
*/
VELOCITY_PARAM(speed_duplex,"Setting the speed and duplex mode");

#define VAL_PKT_LEN_DEF     0
/* ValPktLen[] is used for setting the checksum offload ability of NIC.
   0: Receive frame with invalid layer 2 length (Default)
   1: Drop frame with invalid layer 2 length
*/
VELOCITY_PARAM(ValPktLen,"Receiving or Drop invalid 802.3 frame");

#define WOL_OPT_DEF     0
#define WOL_OPT_MIN     0
#define WOL_OPT_MAX     7
/* wol_opts[] is used for controlling wake on lan behavior.
   0: Wake up if recevied a magic packet. (Default)
   1: Wake up if link status is on/off.
   2: Wake up if recevied an arp packet.
   4: Wake up if recevied any unicast packet.
   Those value can be sumed up to support more than one option.
*/
VELOCITY_PARAM(wol_opts,"Wake On Lan options");

#define INT_WORKS_DEF   20
#define INT_WORKS_MIN   10
#define INT_WORKS_MAX   64

VELOCITY_PARAM(int_works,"Number of packets per interrupt services");

static int  velocity_found1(struct pci_dev *pcid, const struct pci_device_id *ent);
static BOOL velocity_init_info(struct pci_dev* pcid, PVELOCITY_INFO* ppInfo, PCHIP_INFO);
static void velocity_free_info(PVELOCITY_INFO pInfo);
static BOOL velocity_get_pci_info(PVELOCITY_INFO, struct pci_dev* pcid);
static void velocity_print_info(PVELOCITY_INFO pInfo);
static int  velocity_open(struct net_device *dev);
static int  velocity_change_mtu(struct net_device *dev,int mtu);
static int  velocity_xmit(struct sk_buff *skb, struct net_device *dev);
static void velocity_intr(int irq, void *dev_instance, struct pt_regs *regs);
static void velocity_set_multi(struct net_device *dev);
static struct net_device_stats *velocity_get_stats(struct net_device *dev);
static int  velocity_ioctl(struct net_device *dev, struct ifreq *rq, int cmd);
static int  velocity_close(struct net_device *dev);
static int  velocity_rx_srv(PVELOCITY_INFO pInfo,int status);
static BOOL velocity_receive_frame(PVELOCITY_INFO ,int idx);
static BOOL velocity_alloc_rx_buf(PVELOCITY_INFO, int idx);
static void velocity_init_registers(PVELOCITY_INFO pInfo, VELOCITY_INIT_TYPE);
static void velocity_free_tx_buf(PVELOCITY_INFO pInfo, PVELOCITY_TD_INFO );
static BOOL velocity_soft_reset(PVELOCITY_INFO pInfo);
static void mii_init(PVELOCITY_INFO pInfo, U32 mii_status);
static U32 velocity_get_opt_media_mode(PVELOCITY_INFO pInfo);
static void velocity_print_link_status(PVELOCITY_INFO pInfo);
static void SafeDisableMiiAutoPoll (PMAC_REGS pMacRegs);
static void velocity_shutdown(PVELOCITY_INFO pInfo);
static void enable_flow_control_ability(PVELOCITY_INFO pInfo);
static void  EnableMiiAutoPoll(PMAC_REGS pMacRegs);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM
static int velocity_notify_reboot(struct notifier_block *, unsigned long event, void *ptr);
static int velocity_suspend(struct pci_dev *pcid, u32 state);
static int velocity_resume(struct pci_dev *pcid);

struct notifier_block velocity_notifier = {
        notifier_call:  velocity_notify_reboot,
        next:           NULL,
        priority:       0
};

static int
velocity_netdev_event(struct notifier_block *nb,
    unsigned long notification, void *ptr);

static struct notifier_block velocity_inetaddr_notifier = {
    notifier_call: velocity_netdev_event,
    };

#endif //CONFIG_PM
#endif //LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)


static CHIP_INFO chip_info_table[]= {
    { CHIP_TYPE_VT6110, "ZyXEL GN600 series 1000Base-T Adapter",
        256, 1, 0x00FFFFFFUL},
    {0,NULL}
};

static struct pci_device_id velocity_id_table[] __devinitdata = {
{ 0x1106, 0x3119, PCI_ANY_ID, PCI_ANY_ID, 0, 0, (int)&chip_info_table[0]},
{ 0, }
};

static char* get_chip_name(int chip_id) {
    int i;
    for (i=0;chip_info_table[i].name!=NULL;i++)
        if (chip_info_table[i].chip_id==chip_id)
            break;
    return chip_info_table[i].name;
}

static void __devexit velocity_remove1(struct pci_dev *pcid)
{
    PVELOCITY_INFO pInfo=pci_get_drvdata(pcid);

    if (pInfo==NULL)
        return;
/*
    if (pInfo->flags & VELOCITY_FLAGS_WOL_ENABLED) {
        velocity_get_ip(pInfo);
        velocity_set_wol(pInfo);
        pci_enable_wake(pcid,3,1);
        pci_set_power_state(pcid,1);
    }
*/
#ifdef CONFIG_PROC_FS
    velocity_free_proc_entry(pInfo);
#endif
    velocity_free_info(pInfo);

}

static void
velocity_set_int_opt(int *opt, int val, int min, int max, int def,char* name,char* devname) {
    if (val==-1)
        *opt=def;
    else if (val<min || val>max) {
        VELOCITY_PRT(MSG_LEVEL_INFO, KERN_NOTICE "%s: the value of parameter %s is invalid, the valid range is (%d-%d)\n" ,
            devname,name, min,max);
        *opt=def;
    } else {
        VELOCITY_PRT(MSG_LEVEL_INFO, KERN_INFO "%s: set value of parameter %s to %d\n",
            devname, name, val);
        *opt=val;
    }
}

static void
velocity_set_bool_opt(PU32 opt, int val,BOOL def,U32 flag, char* name,char* devname) {
    (*opt)&=(~flag);
    if (val==-1)
        *opt|=(def ? flag : 0);
    else if (val<0 || val>1) {
        printk(KERN_NOTICE
            "%s: the value of parameter %s is invalid, the valid range is (0-1)\n",devname,name);
        *opt|=(def ? flag : 0);
    } else {
        printk(KERN_INFO "%s: set parameter %s to %s\n",
            devname,name , val ? "TRUE" : "FALSE");
        *opt|=(val ? flag : 0);
    }
}

static void
velocity_get_options(POPTIONS pOpts,int index,char* devname) {

    velocity_set_int_opt(&pOpts->rx_thresh,rx_thresh[index],
            RX_THRESH_MIN, RX_THRESH_MAX, RX_THRESH_DEF,"rx_thresh",devname);

    velocity_set_int_opt(&pOpts->DMA_length,DMA_length[index],
        DMA_LENGTH_MIN, DMA_LENGTH_MAX, DMA_LENGTH_DEF,"DMA_length",devname);

    velocity_set_int_opt(&pOpts->nRxDescs,RxDescriptors[index],
        RX_DESC_MIN, RX_DESC_MAX, RX_DESC_DEF, "RxDescriptors",devname);

    velocity_set_int_opt(&pOpts->nTxDescs,TxDescriptors[index],
        TX_DESC_MIN, TX_DESC_MAX, TX_DESC_DEF, "TxDescriptors",devname);

    velocity_set_int_opt(&pOpts->vid,VID_setting[index],
        VLAN_ID_MIN, VLAN_ID_MAX, VLAN_ID_DEF,"VID_setting",devname);

    velocity_set_bool_opt(&pOpts->flags,enable_tagging[index],
        TAGGING_DEF,VELOCITY_FLAGS_TAGGING, "enable_tagging",devname);

#ifdef VELOCITY_TX_CSUM_SUPPORT
    velocity_set_bool_opt(&pOpts->flags,txcsum_offload[index],
        TX_CSUM_DEF,VELOCITY_FLAGS_TX_CSUM,"txcsum_offload",devname);
#endif

    velocity_set_int_opt(&pOpts->flow_cntl,flow_control[index],
        FLOW_CNTL_MIN,FLOW_CNTL_MAX, FLOW_CNTL_DEF, "flow_control",devname);

    velocity_set_bool_opt(&pOpts->flags,IP_byte_align[index],
        IP_ALIG_DEF, VELOCITY_FLAGS_IP_ALIGN, "IP_byte_align",devname);

    velocity_set_bool_opt(&pOpts->flags,ValPktLen[index],
        VAL_PKT_LEN_DEF, VELOCITY_FLAGS_VAL_PKT_LEN, "ValPktLen",devname);

    velocity_set_int_opt((int*) &pOpts->spd_dpx,speed_duplex[index],
        MED_LNK_MIN, MED_LNK_MAX, MED_LNK_DEF,"Media link mode",devname);

    velocity_set_int_opt((int*) &pOpts->wol_opts,wol_opts[index],
        WOL_OPT_MIN, WOL_OPT_MAX, WOL_OPT_DEF,"Wake On Lan options",devname);

    velocity_set_int_opt((int*) &pOpts->int_works,int_works[index],
        INT_WORKS_MIN, INT_WORKS_MAX, INT_WORKS_DEF,"Interrupt service works",devname);

    pOpts->nRxDescs = (pOpts->nRxDescs & ~3);
}

void velocity_init_cam_filter(PVELOCITY_INFO pInfo) {
    PMAC_REGS   pMacRegs=pInfo->pMacRegs;

    // turn on MCFG_PQEN, turn off MCFG_RTGOPT
    WORD_REG_BITS_SET(MCFG_PQEN, MCFG_RTGOPT, &pMacRegs->wMCFG);
    WORD_REG_BITS_ON(MCFG_VIDFR, &pMacRegs->wMCFG);

    //Disable all CAM
    memset(pInfo->abyVCAMMask,0,sizeof(U8)*8);
    memset(pInfo->abyMCAMMask,0,sizeof(U8)*8);
    mac_set_cam_mask(pInfo->pMacRegs,pInfo->abyVCAMMask,VELOCITY_VLAN_ID_CAM);
    mac_set_cam_mask(pInfo->pMacRegs,pInfo->abyMCAMMask,VELOCITY_MULTICAST_CAM);
    
    //Enable first VCAM
    if (pInfo->flags & VELOCITY_FLAGS_TAGGING) {
        // if Tagging option is enabled and VLAN ID is not zero, then turn on MCFG_RTGOPT also
        if (pInfo->sOpts.vid != 0)
            WORD_REG_BITS_ON(MCFG_RTGOPT, &pMacRegs->wMCFG);
        
        mac_set_cam(pInfo->pMacRegs, 0, (PU8)&(pInfo->sOpts.vid), VELOCITY_VLAN_ID_CAM);
        pInfo->abyVCAMMask[0] |= 1;
        mac_set_cam_mask(pInfo->pMacRegs, pInfo->abyVCAMMask, VELOCITY_VLAN_ID_CAM);
    }
    else {
        U16	wTemp = 0;        
		mac_set_cam(pInfo->pMacRegs, 0, (PU8)&wTemp, VELOCITY_VLAN_ID_CAM);
	    wTemp = 1;
		mac_set_cam_mask(pInfo->pMacRegs, (PU8)&(wTemp), VELOCITY_VLAN_ID_CAM);
    }
}
//
// Initialiation of MAC registers
//
static void
velocity_init_registers(PVELOCITY_INFO pInfo, VELOCITY_INIT_TYPE InitType) {
    PMAC_REGS   pMacRegs=pInfo->pMacRegs;
    int         i,mii_status;

    mac_wol_reset(pInfo->pMacRegs);

    switch (InitType) {
    case VELOCITY_INIT_RESET:
    case VELOCITY_INIT_WOL:

        netif_stop_queue(pInfo->dev);
        mac_rx_queue_wake(pMacRegs);

        mii_status=velocity_get_opt_media_mode(pInfo);
        if (velocity_set_media_mode(pInfo,mii_status)!=VELOCITY_LINK_CHANGE) {
            velocity_print_link_status(pInfo);
            if (!(pInfo->mii_status & VELOCITY_LINK_FAIL))
                netif_wake_queue(pInfo->dev);
        }

        enable_flow_control_ability(pInfo);

        mac_clear_isr(pInfo->pMacRegs);
        writel(CR0_STOP, &pMacRegs->dwCR0Clr);
        writel((CR0_DPOLL|CR0_TXON|CR0_RXON|CR0_STRT),&pMacRegs->dwCR0Set);

        break;

    case VELOCITY_INIT_COLD:
    default:
        // do reset
        velocity_soft_reset(pInfo);
        mdelay(5);

        mac_eeprom_reload(pMacRegs);
        for (i = 0; i < 6; i++) {
            writeb(pInfo->dev->dev_addr[i], &(pMacRegs->abyPAR[i]));
        }
        mac_set_rx_thresh(pMacRegs,pInfo->sOpts.rx_thresh);
        mac_set_dma_length(pMacRegs,pInfo->sOpts.DMA_length);

        writeb(WOLCFG_SAM|WOLCFG_SAB, &pMacRegs->byWOLCFGSet);

        // back off algorithm use original IEEE standard
        BYTE_REG_BITS_SET(CFGB_OFSET,
            (CFGB_CRANDOM | CFGB_CAP | CFGB_MBA | CFGB_BAKOPT),
            &pMacRegs->byCFGB);


        // set packet filter
        // receive directed and broadcast address
        velocity_set_multi(pInfo->dev);
        
        // enable MII auto-polling
        EnableMiiAutoPoll(pMacRegs);

        pInfo->IntMask=INT_MASK_DEF;

        writel(cpu_to_le32(pInfo->rd_pool_dma),&pMacRegs->dwRDBaseLo);
        writew(pInfo->sOpts.nRxDescs-1,&pMacRegs->wRDCSize);
        mac_rx_queue_run(pMacRegs);
        mac_rx_queue_wake(pMacRegs);

        writew(pInfo->sOpts.nTxDescs-1,&pMacRegs->wTDCSize);

        for (i=0;i<pInfo->nTxQueues;i++) {
            writel(cpu_to_le32(pInfo->td_pool_dma[i]),
                &(pMacRegs->adwTDBaseLo[i]));
            mac_tx_queue_run(pMacRegs,i);
        }

        velocity_init_cam_filter(pInfo);

        init_flow_control_register(pInfo);

        writel(CR0_STOP, &pMacRegs->dwCR0Clr);
        writel((CR0_DPOLL|CR0_TXON|CR0_RXON|CR0_STRT), &pMacRegs->dwCR0Set);

        mii_status=velocity_get_opt_media_mode(pInfo);
        netif_stop_queue(pInfo->dev);
        mac_clear_isr(pInfo->pMacRegs);

        mii_init(pInfo, mii_status);

        if (velocity_set_media_mode(pInfo,mii_status)!=VELOCITY_LINK_CHANGE) {
            velocity_print_link_status(pInfo);
            if (!(pInfo->mii_status & VELOCITY_LINK_FAIL))
                netif_wake_queue(pInfo->dev);
        }
        
        enable_flow_control_ability(pInfo);
        mac_hw_mibs_init(pInfo->pMacRegs);
        mac_write_int_mask(pInfo->IntMask,pMacRegs);
        mac_clear_isr(pInfo->pMacRegs);

    } //switch
}

static BOOL velocity_soft_reset(PVELOCITY_INFO pInfo) {
    PMAC_REGS   pMacRegs=pInfo->pMacRegs;
    int i=0;

    writel(CR0_SFRST, &pMacRegs->dwCR0Set);

    for (i=0;i<W_MAX_TIMEOUT;i++) {
        udelay(5);
        if (!DWORD_REG_BITS_IS_ON(CR0_SFRST,&pMacRegs->dwCR0Set))
            break;
    }

    if (i==W_MAX_TIMEOUT) {
        writel(CR0_FORSRST, &pMacRegs->dwCR0Set);
        // delay 2ms
        mdelay(2);
    }
    return TRUE;
}


static int
velocity_found1(struct pci_dev *pcid, const struct pci_device_id *ent)
{
    static BOOL         bFirst=TRUE;
    struct net_device*  dev=NULL;
    int                 i;
    PCHIP_INFO          pChip_info=(PCHIP_INFO) ent->driver_data;
    PVELOCITY_INFO         pInfo;
    PMAC_REGS           pMacRegs;

    if (velocity_nics++>=MAX_UINTS) {
        printk(KERN_NOTICE VELOCITY_NAME ": already found %d NICs\n", velocity_nics);
        return -ENODEV;
    }

    dev=init_etherdev(dev,0);

    if (dev==NULL) {
        printk(KERN_ERR VELOCITY_NAME ": allocate net device failed");
        return -ENODEV;
    }

    if (bFirst) {
        printk(KERN_INFO "%s Ver. %s\n",VELOCITY_FULL_DRV_NAM, VELOCITY_VERSION);
        printk(KERN_INFO "Copyright (c) 2002, 2003 ZyXEL Communications corporation\n");
        bFirst=FALSE;
    }

    if (!velocity_init_info(pcid,&pInfo, pChip_info)) {
        return -ENOMEM;
    }

    pInfo->dev=dev;

    dev->priv=pInfo;
    dev->irq=pcid->irq;

    if (pci_enable_device(pcid)) {
        velocity_free_info(pInfo);
        return -ENODEV;
    }

    if (velocity_get_pci_info(pInfo,pcid)==FALSE) {
        printk(KERN_ERR VELOCITY_NAME ": Failed to find PCI device\n");
        velocity_free_info(pInfo);
        return -ENODEV;
    }

    velocity_create_proc_entry(pInfo);

    pMacRegs=ioremap(pInfo->memaddr & PCI_BASE_ADDRESS_MEM_MASK, pInfo->io_size);
    pInfo->pMacRegs=pMacRegs;

    mac_wol_reset(pInfo->pMacRegs);

    if (check_region(pInfo->ioaddr,pInfo->io_size)) {
        printk(KERN_ERR VELOCITY_NAME ": Failed to find PCI device\n");
        pInfo->ioaddr=0;
        velocity_free_info(pInfo);
        return -ENODEV;
    }

    request_region(pInfo->ioaddr,pInfo->io_size,VELOCITY_NAME);

    dev->base_addr=pInfo->ioaddr;

    for (i=0;i<6;i++)
        dev->dev_addr[i]=readb(&pMacRegs->abyPAR[i]);

    velocity_print_info(pInfo);

    velocity_get_options(&pInfo->sOpts,velocity_nics-1,dev->name);

    //Mask out the options cannot be set to the chip
    pInfo->sOpts.flags &= pChip_info->flags;

    //Enable the chip specified capbilities
    pInfo->flags=pInfo->sOpts.flags | (pChip_info->flags & 0xFF000000UL);

    pInfo->wol_opts=pInfo->sOpts.wol_opts;
    pInfo->flags|=VELOCITY_FLAGS_WOL_ENABLED;

    pInfo->dwPHYid=MII_GET_PHY_ID(pInfo->pMacRegs);

    dev->irq=pcid->irq;
    dev->open               = velocity_open;
    dev->hard_start_xmit    = velocity_xmit;
    dev->stop               = velocity_close;
    dev->get_stats          = velocity_get_stats;
    dev->set_multicast_list = velocity_set_multi;
    dev->do_ioctl           = velocity_ioctl;
    dev->change_mtu         = velocity_change_mtu;
#ifdef  VELOCITY_ZERO_COPY_SUPPORT
    dev->features           |= NETIF_F_SG;
#endif

#ifdef  VELOCITY_TX_CSUM_SUPPORT
    if (pInfo->flags & VELOCITY_FLAGS_TX_CSUM) {
        dev->features           |= NETIF_F_HW_CSUM;
    }
#endif

    pci_set_drvdata(pcid,pInfo);

    return 0;
}

static void velocity_print_info(PVELOCITY_INFO pInfo)
{
    struct net_device* dev=pInfo->dev;

    printk(KERN_INFO "%s: %s\n",dev->name, get_chip_name(pInfo->chip_id));
    printk(KERN_INFO "%s: MAC=%2.2X:%2.2X:%2.2X:%2.2X:%2.2X:%2.2X",
        dev->name,
        dev->dev_addr[0],dev->dev_addr[1],dev->dev_addr[2],
        dev->dev_addr[3],dev->dev_addr[4],dev->dev_addr[5]);

    printk(" IO=0x%lx Mem=0x%lx ",(ULONG) pInfo->ioaddr,(ULONG) pInfo->pMacRegs);
    printk(" IRQ=%d \n", pInfo->dev->irq);

}

static BOOL velocity_init_info(struct pci_dev* pcid, PVELOCITY_INFO* ppInfo,
    PCHIP_INFO pChip_info) {
    PVELOCITY_INFO p;

    *ppInfo=MALLOC(sizeof(VELOCITY_INFO),GFP_ATOMIC);

    if (*ppInfo==NULL)
        return FALSE;

    memset(*ppInfo,0,sizeof(VELOCITY_INFO));

    if (pVelocity3_Infos==NULL) {
        pVelocity3_Infos=*ppInfo;
    }
    else {
        for (p=pVelocity3_Infos;p->next!=NULL;p=p->next)
            do {} while (0);
        p->next=*ppInfo;
        (*ppInfo)->prev=p;
    }

    (*ppInfo)->pcid=pcid;
    (*ppInfo)->chip_id=pChip_info->chip_id;
    (*ppInfo)->io_size=pChip_info->io_size;
    (*ppInfo)->nTxQueues=pChip_info->nTxQueue;
    (*ppInfo)->multicast_limit=MCAM_SIZE;

#ifdef VMNS
    (*ppInfo)->vmns_priv=MALLOC(sizeof(VMNS_DRV_PRIVATE),GFP_ATOMIC);
    if ((*ppInfo)->vmns_priv==NULL)
        return FALSE;
    memset((*ppInfo)->vmns_priv,0,sizeof(VMNS_DRV_PRIVATE));
#endif

    spin_lock_init(&((*ppInfo)->lock));
    spin_lock_init(&((*ppInfo)->xmit_lock));

#ifdef CONFIG_PROC_FS
    velocity_init_proc_fs(*ppInfo);
#endif

    return TRUE;
}

static BOOL velocity_get_pci_info(PVELOCITY_INFO pInfo, struct pci_dev* pcid) {

    U16 pci_cmd;
    pci_read_config_byte(pcid, PCI_REVISION_ID, &pInfo->byRevId);
    pci_read_config_word(pcid, PCI_SUBSYSTEM_ID,&pInfo->SubSystemID);
    pci_read_config_word(pcid, PCI_SUBSYSTEM_VENDOR_ID, &pInfo->SubVendorID);
    pci_read_config_word(pcid, PCI_COMMAND, (u16 *) & (pci_cmd));

    pci_set_master(pcid);

    pInfo->ioaddr=pci_resource_start(pcid,0);
    pInfo->memaddr=pci_resource_start(pcid,1);

    pInfo->pcid=pcid;

    return TRUE;
}

static void velocity_free_info(PVELOCITY_INFO pInfo) {
    PVELOCITY_INFO         ptr;
    struct net_device*  dev=pInfo->dev;

    ASSERT(pInfo);

    if (pVelocity3_Infos==NULL)
        return;

    for (ptr=pVelocity3_Infos;ptr && (ptr!=pInfo);ptr=ptr->next)
            do {} while (0);

    if (ptr==pInfo) {
        if (ptr==pVelocity3_Infos)
            pVelocity3_Infos=ptr->next;
        else
            ptr->prev->next=ptr->next;
    }
    else {
        VELOCITY_PRT(MSG_LEVEL_ERR, KERN_ERR "vmns: info struct not found\n");
        return;
    }

#ifdef CONFIG_PROC_FS
    velocity_free_proc_fs(pInfo);
#endif

    if (dev)
        unregister_netdev(dev);

    if (pInfo->pMacRegs)
        iounmap(pInfo->pMacRegs);

    if (pInfo->ioaddr)
        release_region(pInfo->ioaddr,pInfo->io_size);

    if (dev)
        KFREE(dev);

    if (pInfo->pcid) {
        pci_set_drvdata(pInfo->pcid,NULL);
    }

#ifdef VMNS
    if (pInfo->vmns_priv)
        KFREE(pInfo->vmns_priv);
#endif

    KFREE(pInfo);

}

static BOOL velocity_init_rings(PVELOCITY_INFO pInfo) {
    int     i;

    /*allocate all RD/TD rings a single pool*/
    pInfo->pool=pci_alloc_consistent(pInfo->pcid,
                    pInfo->sOpts.nRxDescs * sizeof(RX_DESC) + 64 +
                    pInfo->sOpts.nTxDescs * sizeof(TX_DESC)*pInfo->nTxQueues,
                    &pInfo->pool_dma);

    if (pInfo->pool==NULL)  {
        printk(KERN_ERR "%s : allocate dma memory failed\n", pInfo->dev->name);
        return FALSE;
    }

    memset(pInfo->pool,0, pInfo->sOpts.nRxDescs * sizeof(RX_DESC) +
        pInfo->sOpts.nTxDescs * sizeof(TX_DESC)*pInfo->nTxQueues);

    pInfo->aRDRing=(PRX_DESC) (((U32) (((PU8) pInfo->pool) + 63)) & ~63);


    pInfo->rd_pool_dma=pInfo->pool_dma;

    pInfo->tx_bufs=pci_alloc_consistent(pInfo->pcid,
                pInfo->sOpts.nTxDescs * PKT_BUF_SZ*pInfo->nTxQueues,
                &pInfo->tx_bufs_dma);

    if (pInfo->tx_bufs==NULL) {
        printk(KERN_ERR "%s: allocate dma memory failed\n", pInfo->dev->name);
        pci_free_consistent(pInfo->pcid,
            pInfo->sOpts.nRxDescs * sizeof(RX_DESC) + 64 +
            pInfo->sOpts.nTxDescs * sizeof(TX_DESC)*pInfo->nTxQueues,
            pInfo->pool, pInfo->pool_dma);
        return FALSE;
    }

    memset(pInfo->tx_bufs,0,pInfo->sOpts.nTxDescs * PKT_BUF_SZ*pInfo->nTxQueues);

    for (i=0;i<pInfo->nTxQueues;i++) {

        pInfo->td_pool_dma[i]=pInfo->rd_pool_dma+
                pInfo->sOpts.nRxDescs*sizeof(RX_DESC)+
                pInfo->sOpts.nTxDescs*sizeof(TX_DESC)*i;

        pInfo->apTDRings[i]=(PTX_DESC) (((PU8) pInfo->aRDRing)+
                pInfo->sOpts.nRxDescs*sizeof(RX_DESC)+
                pInfo->sOpts.nTxDescs*sizeof(TX_DESC)*i);


    }

    return TRUE;
}

static void velocity_free_rings(PVELOCITY_INFO pInfo) {
    pci_free_consistent(pInfo->pcid,
        pInfo->sOpts.nRxDescs * sizeof(RX_DESC) + 64 +
        pInfo->sOpts.nTxDescs * sizeof(TX_DESC)*pInfo->nTxQueues,
        pInfo->pool, pInfo->pool_dma);

    if (pInfo->tx_bufs)
        pci_free_consistent(pInfo->pcid,
                pInfo->sOpts.nTxDescs * PKT_BUF_SZ*pInfo->nTxQueues,
                pInfo->tx_bufs, pInfo->tx_bufs_dma);

}

static BOOL velocity_init_rd_ring(PVELOCITY_INFO pInfo) {
    int i;
    PRX_DESC        pDesc;
    PVELOCITY_RD_INFO  pRDInfo;

    pInfo->aRDInfo=MALLOC(sizeof(VELOCITY_RD_INFO)*pInfo->sOpts.nRxDescs,
            GFP_KERNEL);
    memset(pInfo->aRDInfo,0,sizeof(VELOCITY_RD_INFO)*pInfo->sOpts.nRxDescs);

    /* Init the RD ring entries */
    for (i = 0; i < pInfo->sOpts.nRxDescs; i++) {

        pDesc=&(pInfo->aRDRing[i]);
        pRDInfo=&(pInfo->aRDInfo[i]);

        if (!velocity_alloc_rx_buf(pInfo, i)) {
            VELOCITY_PRT(MSG_LEVEL_ERR,KERN_ERR "%s: can not alloc rx bufs\n",
            pInfo->dev->name);
            return FALSE;
        }
        pDesc->rdesc0.f1Owner=OWNED_BY_NIC;
    }

    pInfo->iRDUsed=pInfo->iCurrRDIdx=0;
    return TRUE;
}

static void velocity_free_rd_ring(PVELOCITY_INFO pInfo) {
    int i;
    if (pInfo->aRDInfo==NULL)
        return;

    for (i = 0; i < pInfo->sOpts.nRxDescs; i++) {
        PVELOCITY_RD_INFO  pRDInfo=&(pInfo->aRDInfo[i]);
        if (pRDInfo->skb_dma) {
            pci_unmap_single(pInfo->pcid,pRDInfo->skb_dma,
               pInfo->rx_buf_sz, PCI_DMA_FROMDEVICE);
            pRDInfo->skb_dma=(dma_addr_t)NULL;
        }
        if (pRDInfo->skb) {
            dev_kfree_skb(pRDInfo->skb);
            pRDInfo->skb=NULL;
        }
    }

    if (pInfo->aRDInfo)
        KFREE(pInfo->aRDInfo);
    pInfo->aRDInfo=NULL;
}

static BOOL velocity_init_td_ring(PVELOCITY_INFO pInfo) {
    int i,j;
    dma_addr_t  curr;
    PTX_DESC        pDesc;
    PVELOCITY_TD_INFO  pTDInfo;

    /* Init the TD ring entries */
    for (j=0;j<pInfo->nTxQueues;j++) {
        curr=pInfo->td_pool_dma[j];

        pInfo->apTDInfos[j]=MALLOC(sizeof(VELOCITY_TD_INFO)*pInfo->sOpts.nTxDescs,
            GFP_KERNEL);

        memset(pInfo->apTDInfos[j],0,
            sizeof(VELOCITY_TD_INFO)*pInfo->sOpts.nTxDescs);

        for (i = 0; i < pInfo->sOpts.nTxDescs; i++, curr+=sizeof(TX_DESC)) {
            pDesc=&(pInfo->apTDRings[j][i]);
            pTDInfo=&(pInfo->apTDInfos[j][i]);
            pTDInfo->buf=pInfo->tx_bufs+(i+j)*PKT_BUF_SZ;
            pTDInfo->buf_dma=pInfo->tx_bufs_dma+(i+j)*PKT_BUF_SZ;
        }
        pInfo->aiTailTDIdx[j]=pInfo->aiCurrTDIdx[j]=pInfo->iTDUsed[j]=0;
    }
    return TRUE;
}

static void velocity_free_td_ring(PVELOCITY_INFO pInfo) {
    int i, j,k;

    for (j=0;j<pInfo->nTxQueues;j++) {

        if (pInfo->apTDInfos[j]==NULL)
            continue;

        for (i = 0; i < pInfo->sOpts.nTxDescs; i++) {
            PVELOCITY_TD_INFO  pTDInfo=&(pInfo->apTDInfos[j][i]);

            if (pTDInfo==NULL)
                continue;

            for (k=0;k<pTDInfo->nskb_dma;k++)
                if (pTDInfo->skb_dma[k]) {
                    pci_unmap_single(pInfo->pcid,pTDInfo->skb_dma[k],
                       pTDInfo->skb->len, PCI_DMA_TODEVICE);
                    pTDInfo->skb_dma[k]=(dma_addr_t)NULL;
                }

            if (pTDInfo->skb) {
                dev_kfree_skb(pTDInfo->skb);
                pTDInfo->skb=NULL;
            }

        }

        if (pInfo->apTDInfos[j]) {
            KFREE(pInfo->apTDInfos[j]);
            pInfo->apTDInfos[j]=NULL;
        }
    }

}

/*-----------------------------------------------------------------*/
static int velocity_rx_srv(PVELOCITY_INFO pInfo, int status) {
    PRX_DESC                pRD;
    struct net_device_stats* pStats=&pInfo->stats;
    PMAC_REGS               pMacRegs=pInfo->pMacRegs;
    int                     iCurrRDIdx=pInfo->iCurrRDIdx;
    int                     works=0;

    while (TRUE) {

        pRD=&(pInfo->aRDRing[iCurrRDIdx]);

        if ((pInfo->aRDInfo[iCurrRDIdx]).skb==NULL) {
            if (!velocity_alloc_rx_buf(pInfo,iCurrRDIdx))
                break;
        }

        if (works++>15)
            break;

        if (pRD->rdesc0.f1Owner==OWNED_BY_NIC)
            break;

        pInfo->adwRMONStats[RMON_Octets]+=pRD->rdesc0.f14Length;

        // don't drop CE or RL error frame although RXOK is off
        if ((pRD->rdesc0.wRSR & RSR_RXOK) || 
        	(!(pRD->rdesc0.wRSR & RSR_RXOK) && (pRD->rdesc0.wRSR & (RSR_CE | RSR_RL)))) {
            if (velocity_receive_frame(pInfo,iCurrRDIdx)) {
                if (!velocity_alloc_rx_buf(pInfo,iCurrRDIdx)) {
                    VELOCITY_PRT(MSG_LEVEL_ERR, KERN_ERR
                    "%s: can not allocate rx buf\n", pInfo->dev->name);
                    break;
                }
            } else {
                pStats->rx_dropped++;
            }
        } else {
            if (pRD->rdesc0.wRSR & RSR_CRC)
                pStats->rx_crc_errors++;
            if (pRD->rdesc0.wRSR & RSR_FAE)
                pStats->rx_frame_errors++;

            pStats->rx_dropped++;
        }

        pRD->f1IntCtlEn=1;

        if (++pInfo->iRDUsed>=4) {
            int i, iPrevRDIdx=iCurrRDIdx;
            for (i=0;i<4;i++) {
                if (--iPrevRDIdx<0)
                    iPrevRDIdx=pInfo->sOpts.nRxDescs-1;

                pRD=&(pInfo->aRDRing[iPrevRDIdx]);
                pRD->rdesc0.f1Owner=OWNED_BY_NIC;
            }
            writew(4,&(pMacRegs->wRBRDU));
            pInfo->iRDUsed-=4;
        }

        pInfo->dev->last_rx=jiffies;

        iCurrRDIdx++;
        if (iCurrRDIdx>=pInfo->sOpts.nRxDescs)
            iCurrRDIdx=0;
    }

    pInfo->iCurrRDIdx=iCurrRDIdx;
    VAR_USED(pStats);
    return works;
}

static inline void
velocity_rx_csum(PRX_DESC pRD, struct sk_buff* skb) {
    skb->ip_summed=CHECKSUM_NONE;

    if (pRD->rdesc1.byCSM & CSM_IPKT) {
            if (pRD->rdesc1.byCSM & CSM_IPOK) {
                if ((pRD->rdesc1.byCSM & CSM_TCPKT) || (pRD->rdesc1.byCSM & CSM_UDPKT)) {
                    if (!(pRD->rdesc1.byCSM & CSM_TUPOK)) {
                        return;
                    }
                }
                skb->ip_summed=CHECKSUM_UNNECESSARY;
            }
    }
}

#ifdef VMNS
static void inline
velocity_attach_vmns_info(PRX_DESC pRD,struct sk_buff* skb) {
    PVMNS_ATTACH_INFO pAttch = GET_VMNS_ATTACH_INFO(skb);
    pAttch->type = ATT_INFO_TYPE_NONE;

    if (pRD->rdesc0.wRSR & RSR_DETAG) {
        pAttch->type |=ATT_INFO_TYPE_TAG;

        if (!(pRD->rdesc0.wRSR & RSR_VIDM))
            pAttch->type |=ATT_INFO_TYPE_VIDHIT;

        pAttch->vlan=pRD->rdesc1.wPQTAG & 0xfff;
        pAttch->priority=(pRD->rdesc1.wPQTAG>>12) & 0x7;
    }

    if (pRD->rdesc1.byIPKT) {
        pAttch->type |= ATT_INFO_TYPE_INST;
        pAttch->InstNo = pRD->rdesc1.byIPKT;
    }

    pAttch->protocol=skb->protocol;
    skb->protocol=VMNS_FRAME_TYPE;
}
#endif

static BOOL
velocity_receive_frame(PVELOCITY_INFO pInfo, int idx) {
    struct net_device_stats* pStats=&pInfo->stats;
    PVELOCITY_RD_INFO  pRDInfo=&(pInfo->aRDInfo[idx]);
    PRX_DESC        pRD=&(pInfo->aRDRing[idx]);
    struct sk_buff* skb;


    if (pRD->rdesc0.wRSR & (RSR_STP|RSR_EDP)) {
        VELOCITY_PRT(MSG_LEVEL_VERBOSE,
            KERN_NOTICE " %s : the received frame span multple RDs\n",
            pInfo->dev->name);
        pStats->rx_length_errors++;
        return FALSE;
    }

    if (pRD->rdesc0.wRSR & RSR_MAR)
        pInfo->stats.multicast++;

    if (pRD->rdesc0.wRSR & RSR_BAR)
        pInfo->adwRMONStats[RMON_BroadcastPkts]++;

    skb=pRDInfo->skb;
    skb->dev=pInfo->dev;

    pci_unmap_single(pInfo->pcid,pRDInfo->skb_dma,
                pInfo->rx_buf_sz, PCI_DMA_FROMDEVICE);
    pRDInfo->skb_dma=(dma_addr_t)NULL;
    pRDInfo->skb=NULL;

    if (pInfo->flags & VELOCITY_FLAGS_IP_ALIGN) {
        int i;
        for (i = pRD->rdesc0.f14Length+4; i >= 0 ; i--)
            *(skb->data + i + 2) = *(skb->data + i);
        skb->data += 2;
        skb->tail += 2;
    }

    skb_put(skb,(pRD->rdesc0.f14Length-4));
    
    skb->protocol=eth_type_trans(skb, skb->dev);

    //drop frame not met IEEE 802.3
    if (pInfo->flags & VELOCITY_FLAGS_VAL_PKT_LEN) {
        if (pRD->rdesc0.wRSR & RSR_RL) {
            pStats->rx_length_errors++;
            return FALSE;
        }
    }

    velocity_rx_csum(pRD,skb);

#ifdef VMNS
    if (pInfo->flags & VELOCITY_FLAGS_VMNS_COMMITTED)
        velocity_attach_vmns_info(pRD, skb);
#endif

    pStats->rx_bytes+=skb->len;
    netif_rx(skb);

    return TRUE;
}

static BOOL velocity_alloc_rx_buf(PVELOCITY_INFO pInfo, int idx) {
    PRX_DESC        pRD=&(pInfo->aRDRing[idx]);
    PVELOCITY_RD_INFO  pRDInfo=&(pInfo->aRDInfo[idx]);

    pRDInfo->skb=dev_alloc_skb(pInfo->rx_buf_sz+64);

    if (pRDInfo->skb==NULL)
        return FALSE;

    ASSERT(pRDInfo->skb);
    skb_reserve(pRDInfo->skb,(U32) pRDInfo->skb->tail & 63);
    pRDInfo->skb->dev=pInfo->dev;
    pRDInfo->skb_dma=
        pci_map_single(pInfo->pcid, pRDInfo->skb->tail, pInfo->rx_buf_sz,
            PCI_DMA_FROMDEVICE);
    *((PU32) &(pRD->rdesc0)) = 0;

    pRD->f15BufLen=cpu_to_le32(pInfo->rx_buf_sz);
    pRD->f1IntCtlEn=1;

    pRD->dwBufAddrLo=cpu_to_le32(pRDInfo->skb_dma);
    pRD->wBufAddrHi=0;

    return TRUE;
}

static int velocity_tx_srv(PVELOCITY_INFO pInfo, U32 status) {
    PTX_DESC        pTD;
    int             iQNo;
    BOOL            bFull=FALSE;
    int             idx;
    int             works=0;
    PVELOCITY_TD_INFO  pTDInfo;
    struct net_device_stats* pStats=&pInfo->stats;

    for (iQNo=0;iQNo<pInfo->nTxQueues;iQNo++) {
        for (idx=pInfo->aiTailTDIdx[iQNo];
            pInfo->iTDUsed[iQNo]>0;
            idx=(idx+1) % pInfo->sOpts.nTxDescs) {

            //Get Tx Descriptor
            pTD=&(pInfo->apTDRings[iQNo][idx]);
            pTDInfo=&(pInfo->apTDInfos[iQNo][idx]);

            if (pTD->tdesc0.f1Owner==OWNED_BY_NIC)
                break;


            if ((works++>15))
                break;

            if (pTD->tdesc0.wTSR & TSR0_TERR) {
                pStats->tx_errors++;
                pStats->tx_dropped++;
                if (pTD->tdesc0.wTSR & TSR0_CDH)
                    pStats->tx_heartbeat_errors++;
                if (pTD->tdesc0.wTSR & TSR0_CRS)
                    pStats->tx_carrier_errors++;
                if (pTD->tdesc0.wTSR & TSR0_ABT)
                    pStats->tx_aborted_errors++;
                if (pTD->tdesc0.wTSR & TSR0_OWC)
                    pStats->tx_window_errors++;

            }
            else {
                pStats->tx_packets++;
                pStats->tx_bytes+=pTDInfo->skb->len;
            }

            velocity_free_tx_buf(pInfo,pTDInfo);
            pInfo->iTDUsed[iQNo]--;
        }

        pInfo->aiTailTDIdx[iQNo]=idx;

        if (AVAIL_TD(pInfo, iQNo) < 1 ) {
            bFull=TRUE;
        }
    }

    if (netif_queue_stopped(pInfo->dev) && (bFull==FALSE)
        && (!(pInfo->mii_status & VELOCITY_LINK_FAIL))) {
        netif_wake_queue(pInfo->dev);
    }
    return works;
}

static void
velocity_print_link_status(PVELOCITY_INFO pInfo) {

    if (pInfo->mii_status & VELOCITY_LINK_FAIL) {
        VELOCITY_PRT(MSG_LEVEL_INFO, KERN_NOTICE
            "%s: failed to detect cable link\n",pInfo->dev->name);
    }
    else {
        if (pInfo->sOpts.spd_dpx == SPD_DPX_AUTO) {
            VELOCITY_PRT(MSG_LEVEL_INFO, KERN_NOTICE
                "%s: Link autonegation",pInfo->dev->name);
                            
        	if (pInfo->mii_status & VELOCITY_SPEED_1000)
            	VELOCITY_PRT(MSG_LEVEL_INFO," speed 1000M bps");
	        else if (pInfo->mii_status & VELOCITY_SPEED_100)
    		    VELOCITY_PRT(MSG_LEVEL_INFO," speed 100M bps");
	        else
    		    VELOCITY_PRT(MSG_LEVEL_INFO," speed 10M bps");

	        if (pInfo->mii_status & VELOCITY_DUPLEX_FULL)
    	        VELOCITY_PRT(MSG_LEVEL_INFO, " full duplex\n");
        	else
            	VELOCITY_PRT(MSG_LEVEL_INFO, " half duplex\n");
    	}
    	else {
    		VELOCITY_PRT(MSG_LEVEL_INFO, KERN_NOTICE
            	"%s: Link forced",pInfo->dev->name);
            switch(pInfo->sOpts.spd_dpx) {
            	case SPD_DPX_100_HALF:
            		VELOCITY_PRT(MSG_LEVEL_INFO, " speed 100M bps half duplex\n");
            		break;
            	case SPD_DPX_100_FULL:
            		VELOCITY_PRT(MSG_LEVEL_INFO, " speed 100M bps full duplex\n");
            		break;
            	case SPD_DPX_10_HALF:
            		VELOCITY_PRT(MSG_LEVEL_INFO, " speed 10M bps half duplex\n");
            		break;
            	case SPD_DPX_10_FULL:
            		VELOCITY_PRT(MSG_LEVEL_INFO, " speed 10M bps full duplex\n");
            		break;
            	default:
            		break;
			}
		}
	}
}

static void velocity_error(PVELOCITY_INFO pInfo, int status) {

    if (status & ISR_TXSTLI) {
        PMAC_REGS   pMacRegs=pInfo->pMacRegs;

        printk("TD structure errror TDindex=%hx\n",
            readw(&pMacRegs->awTDIdx[0]));

        BYTE_REG_BITS_ON(TXESR_TDSTR,&pMacRegs->byTXESR);
        writew(TRDCSR_RUN, &pMacRegs->wTDCSRClr);
        netif_stop_queue(pInfo->dev);
    }

    if (status & ISR_SRCI) {
    	PMAC_REGS	pMacRegs=pInfo->pMacRegs;
    	BOOL		bLinked;

        if (pInfo->sOpts.spd_dpx == SPD_DPX_AUTO) {
            pInfo->mii_status = check_connectiontype(pInfo->pMacRegs);
            
            // if it's 3119, disable frame bursting in halfduplex mode and
            // enable it in fullduplex mode
            if (pInfo->byRevId < REV_ID_VT3216_A0) {            
            	if (pInfo->mii_status | VELOCITY_DUPLEX_FULL)
            		BYTE_REG_BITS_ON(TCR_TB2BDIS, &pMacRegs->byTCR);
				else
					BYTE_REG_BITS_OFF(TCR_TB2BDIS, &pMacRegs->byTCR);
			}
			
			// only enable CD heart beat counter in 10HD mode
            if (!(pInfo->mii_status & VELOCITY_DUPLEX_FULL) && (pInfo->mii_status & VELOCITY_SPEED_10)) {
                BYTE_REG_BITS_OFF(TESTCFG_HBDIS, &pMacRegs->byTESTCFG);
            }
            else {
                BYTE_REG_BITS_ON(TESTCFG_HBDIS, &pMacRegs->byTESTCFG);
            }
		}

		// get link status from PHYSR0                
        bLinked = readb(&pMacRegs->byPHYSR0) & PHYSR0_LINKGD;        				

        if (bLinked) {
        	pInfo->mii_status &= ~VELOCITY_LINK_FAIL;
        }
        else {
        	pInfo->mii_status |= VELOCITY_LINK_FAIL;
        }
            
        velocity_print_link_status(pInfo);
	    enable_flow_control_ability(pInfo);

		// re-enable auto-polling because SRCI will disable auto-polling
		EnableMiiAutoPoll(pMacRegs);

#ifdef VMNS
        if (pInfo->flags & VELOCITY_FLAGS_VMNS_COMMITTED) {
            PVMNS_DRV_PRIVATE pvmns_priv = pInfo->vmns_priv;
            if ((pInfo->flags & VELOCITY_FLAGS_VMNS_COMMITTED) && pvmns_priv->notify) {
                if (pInfo->mii_status & VELOCITY_LINK_FAIL)
                    pvmns_priv->notify(pInfo->dev->name, VMNS_EVENT_LINK_FAIL);
                else
                    pvmns_priv->notify(pInfo->dev->name, VMNS_EVENT_LINK_UP);
            }
        }
#endif

        if (pInfo->mii_status & VELOCITY_LINK_FAIL)
            netif_stop_queue(pInfo->dev);
        else
            netif_wake_queue(pInfo->dev);

    };

    if (status & ISR_MIBFI)
        velocity_update_hw_mibs(pInfo);

    if (status & ISR_LSTEI) {
        mac_rx_queue_wake(pInfo->pMacRegs);
    }


}

static void velocity_free_tx_buf(PVELOCITY_INFO pInfo, PVELOCITY_TD_INFO pTDInfo) {
    struct sk_buff* skb=pTDInfo->skb;
    int i;

    // Don't unmap the pre-allocaed tx_bufs
    if (pTDInfo->skb_dma && (pTDInfo->skb_dma[0] != pTDInfo->buf_dma))  {

        for (i=0;i<pTDInfo->nskb_dma;i++) {
#ifdef VELOCITY_ZERO_COPY_SUPPORT
            pci_unmap_single(pInfo->pcid,pTDInfo->skb_dma[i],
                pDesc->tdesc1.f15BufLen,PCI_DMA_TODEVICE);
#else
            pci_unmap_single(pInfo->pcid,pTDInfo->skb_dma[i],skb->len,
                PCI_DMA_TODEVICE);
#endif
            pTDInfo->skb_dma[i]=0;
        }
    }

    dev_kfree_skb_irq(skb);

    pTDInfo->skb=0;

}

static int  velocity_open(struct net_device *dev) {
    PVELOCITY_INFO pInfo=(PVELOCITY_INFO) dev->priv;
    int i;

#ifdef VMNS
    if (pInfo->flags & VELOCITY_FLAGS_OPENED)
        return -EBUSY;
#endif

    pInfo->rx_buf_sz=(dev->mtu <= 1504 ? PKT_BUF_SZ : dev->mtu + 32);

    if (!velocity_init_rings(pInfo)) {

        return -ENOMEM;
    }


    if (!velocity_init_rd_ring(pInfo))
        return -ENOMEM;

    if (!velocity_init_td_ring(pInfo))
        return -ENOMEM;

    velocity_init_registers(pInfo, VELOCITY_INIT_COLD);

    i=request_irq(pInfo->pcid->irq, &velocity_intr, SA_SHIRQ, dev->name, dev);

    if (i)
        return i;

    mac_enable_int(pInfo->pMacRegs);

    netif_start_queue(dev);

    pInfo->flags |=VELOCITY_FLAGS_OPENED;

    MOD_INC_USE_COUNT;
    return 0;
}

static int  velocity_change_mtu(struct net_device *dev,int new_mtu) {
    PVELOCITY_INFO pInfo=(PVELOCITY_INFO) dev->priv;
    int     flags;
    int     oldmtu=dev->mtu;

    if ((new_mtu<VELOCITY_MIN_MTU)||new_mtu>(VELOCITY_MAX_MTU)) {
        VELOCITY_PRT(MSG_LEVEL_ERR, KERN_NOTICE
                "%s: Invaild MTU\n",pInfo->dev->name);
        return -EINVAL;
    }

    if (new_mtu!=oldmtu) {
        spin_lock_irqsave(&pInfo->lock,flags);

        netif_stop_queue(dev);
        velocity_shutdown(pInfo);

        velocity_free_td_ring(pInfo);
        velocity_free_rd_ring(pInfo);

        dev->mtu=new_mtu;
        if (new_mtu>8192)
            pInfo->rx_buf_sz=9*1024;
        else if (new_mtu>4096)
            pInfo->rx_buf_sz=8192;
        else
            pInfo->rx_buf_sz=4*1024;

        if (!velocity_init_rd_ring(pInfo))
            return -ENOMEM;

        if (!velocity_init_td_ring(pInfo))
            return -ENOMEM;

        velocity_init_registers(pInfo, VELOCITY_INIT_COLD);

        mac_enable_int(pInfo->pMacRegs);
        netif_start_queue(dev);
        spin_unlock_irqrestore(&pInfo->lock,flags);
    }

    return 0;
}

static void velocity_shutdown(PVELOCITY_INFO pInfo) {
    PMAC_REGS pMacRegs=pInfo->pMacRegs;
    mac_disable_int(pInfo->pMacRegs);
    writel(CR0_STOP,&pMacRegs->dwCR0Set);
    writew(0xFFFF,&pMacRegs->wTDCSRClr);
    writeb(0xFF,&pMacRegs->byRDCSRClr);
    SafeDisableMiiAutoPoll(pInfo->pMacRegs);
    mac_clear_isr(pInfo->pMacRegs);
}

static int  velocity_close(struct net_device *dev) {
    PVELOCITY_INFO pInfo=(PVELOCITY_INFO) dev->priv;

    netif_stop_queue(dev);

    velocity_shutdown(pInfo);

    if (pInfo->flags & VELOCITY_FLAGS_WOL_ENABLED)
        velocity_get_ip(pInfo);

    MOD_DEC_USE_COUNT;

    if (dev->irq!=0)
        free_irq(dev->irq, dev);

    velocity_free_td_ring(pInfo);
    velocity_free_rd_ring(pInfo);
    velocity_free_rings(pInfo);

    pInfo->flags &=(~VELOCITY_FLAGS_OPENED);
    return 0;
}

static int  velocity_xmit(struct sk_buff *skb, struct net_device *dev) {
    PVELOCITY_INFO pInfo=dev->priv;
    int             iQNo=0;

    PTX_DESC        pTD;
    PVELOCITY_TD_INFO  pTDInfo;

    int             flags;
    int             iCurrTDIdx;

    spin_lock_irqsave(&pInfo->lock,flags);

    iCurrTDIdx=pInfo->aiCurrTDIdx[iQNo];
    pTD=&(pInfo->apTDRings[iQNo][iCurrTDIdx]);
    pTDInfo=&(pInfo->apTDInfos[iQNo][iCurrTDIdx]);

    pTD->tdesc1.f2TCPLS=TCPLS_NORMAL;
    pTD->tdesc1.byTCR=TCR0_TIC;
    pTD->aTdBufs[0].f1Queue=0;

    if (skb->len< ETH_ZLEN) {
        skb_linearize(skb,GFP_ATOMIC);
        memcpy(pTDInfo->buf,skb->data,skb->len);
        memset(pTDInfo->buf+skb->len,0,ETH_ZLEN-skb->len);
        pTDInfo->skb=skb;
        pTDInfo->skb_dma[0]=pTDInfo->buf_dma;
        pTD->tdesc0.f14PktSize=(skb->len >= ETH_ZLEN ? skb->len : ETH_ZLEN);
        pTD->aTdBufs[0].dwBufAddrLo=cpu_to_le32(pTDInfo->skb_dma[0]);
        pTD->aTdBufs[0].wBufAddrHi=0;
        pTD->aTdBufs[0].f14BufSize=pTD->tdesc0.f14PktSize;
        pTDInfo->nskb_dma=1;
        pTD->tdesc1.f4CMDZ=2;
    }
    else
#ifdef VELOCITY_ZERO_COPY_SUPPORT
    if (skb_shinfo(skb)->nr_frags>0) {
        int nfrags=skb_shinfo(skb)->nr_frags;
        pTDInfo->skb=skb;
        if (nfrags>6) {
            skb_linearize(skb,GFP_ATOMIC);
            memcpy(pTDInfo->buf,skb->data,skb->len);
            pTDInfo->skb_dma[0]=pTDInfo->buf_dma;
            pTD->tdesc0.f14PktSize=(skb->len >= ETH_ZLEN ? skb->len : ETH_ZLEN);
            pTD->aTdBufs[0].dwBufAddrLo=cpu_to_le32(pTDInfo->skb_dma[0]);
            pTD->aTdBufs[0].wBufAddrHi=0;
            pTD->aTdBufs[0].f14BufSize=pTD->tdesc0.f14PktSize;
            pTDInfo->nskb_dma=1;
            pTD->tdesc1.f4CMDZ=2;
        }
        else {
            int i=0;
            pTDInfo->nskb_dma=0;
            pTDInfo->skb_dma[i]=
                pci_map_single(pInfo->pcid, skb->data,
                    skb->len -skb->data_len, PCI_DMA_TODEVICE);

            pTD->tdesc0.f14PktSize=(skb->len >= ETH_ZLEN ? skb->len : ETH_ZLEN);

            pTD->aTdBufs[i].dwBufAddrLo=cpu_to_le32(pTDInfo->skb_dma);
            pTD->aTdBufs[i].wBufAddrHi=0;
            pTD->aTdBufs[i].f14BufSize=skb->len->skb->data_len;

            for (i=0;i<nfrags;i++) {
                skb_frag_t *frag = &skb_shinfo(skb)->frags[i];
                void* addr = ((void *) page_address(frag->page +
                            frag->page_offset));

                pTDInfo->skb_dma[i+1]=
                    pci_map_single(pInfo->pcid,addr,frag->size,PCI_DMA_TODEVICE);

                pTD->aTdBufs[i+1].dwBufAddrLo=cpu_to_le32(pTDInfo->skb_dma[i+1]);
                pTD->aTdBufs[i+1].wBufAddrHi=0;
                pTD->aTdBufs[i+1].f14BufSize=frag->size;
            }
            pTDInfo->nskb_dma=i-1;
            pTD->tdesc1.f4CMDZ=i;
        }

    }
    else
#endif
    {

        pTDInfo->skb=skb;
        pTDInfo->skb_dma[0] = pci_map_single(pInfo->pcid, skb->data, skb->len,
            PCI_DMA_TODEVICE);
        pTD->tdesc0.f14PktSize=(skb->len >= ETH_ZLEN ? skb->len : ETH_ZLEN);
        pTD->aTdBufs[0].dwBufAddrLo=cpu_to_le32(pTDInfo->skb_dma[0]);
        pTD->aTdBufs[0].wBufAddrHi=0;
        pTD->aTdBufs[0].f14BufSize=pTD->tdesc0.f14PktSize;
        pTDInfo->nskb_dma=1;
        pTD->tdesc1.f4CMDZ=2;
    }

#ifdef VMNS
    if (pInfo->flags & VELOCITY_FLAGS_VMNS_COMMITTED) {
        PVMNS_ATTACH_INFO pAttch = GET_VMNS_ATTACH_INFO(skb);
        if (pAttch->type & ATT_INFO_TYPE_TAG) {
            pTD->tdesc1.pqinf.f1CFI=0;
            pTD->tdesc1.pqinf.f12VID=(pAttch->vlan & 0xfff);
            pTD->tdesc1.pqinf.f3Priority=(pAttch->priority & 0x7);
            pTD->tdesc1.byTCR|=TCR0_VETAG;
        }
    }
    else
#endif
    if (pInfo->flags & VELOCITY_FLAGS_TAGGING) {
        pTD->tdesc1.pqinf.f12VID=(pInfo->sOpts.vid & 0xfff);
        pTD->tdesc1.pqinf.f3Priority=0;
        pTD->tdesc1.pqinf.f1CFI=0;
        pTD->tdesc1.byTCR|=TCR0_VETAG;
    }


#ifdef VELOCITY_TX_CSUM_SUPPORT
    if ((pInfo->flags & VELOCITY_FLAGS_TX_CSUM) &&
        (skb->ip_summed==CHECKSUM_HW)) {
        struct iphdr* ip=skb->nh.iph;
        if (ip->protocol==IPPROTO_TCP)
            pTD->tdesc1.byTCR|=TCR0_TCPCK;
        else if (ip->protocol==IPPROTO_UDP)
            pTD->tdesc1.byTCR|=(TCR0_UDPCK);
        pTD->tdesc1.byTCR|=TCR0_IPCK;
    }
#endif
    {

        int iPrevTDIdx=iCurrTDIdx-1;

        if (iPrevTDIdx<0)
            iPrevTDIdx=pInfo->sOpts.nTxDescs-1;

        pTD->tdesc0.f1Owner=OWNED_BY_NIC;

        pInfo->iTDUsed[iQNo]++;

        pInfo->aiCurrTDIdx[iQNo]=
            (iCurrTDIdx+1) % pInfo->sOpts.nTxDescs;

        if (AVAIL_TD(pInfo,iQNo)<1)
            netif_stop_queue(dev);

        pTD=&(pInfo->apTDRings[iQNo][iPrevTDIdx]);
        pTD->aTdBufs[0].f1Queue=1;
        mac_tx_queue_wake(pInfo->pMacRegs, iQNo);
    }


    dev->trans_start = jiffies;

    spin_unlock_irqrestore(&pInfo->lock,flags);
    return 0;
}

static void velocity_intr(int irq, void *dev_instance, struct pt_regs *regs) {
    struct net_device* dev=dev_instance;
    PVELOCITY_INFO pInfo=(PVELOCITY_INFO) dev->priv;
    U32             isr_status;
    int             max_count=0;
//  int             flags;


    if (!spin_trylock(&pInfo->lock))
        return;

    isr_status=mac_read_isr(pInfo->pMacRegs);

    if (isr_status==0) {
        spin_unlock(&pInfo->lock);
        return;
    }
    mac_disable_int(pInfo->pMacRegs);

    while (isr_status!=0) {

        mac_write_isr(pInfo->pMacRegs,isr_status);

        if (isr_status & (~(ISR_PRXI|ISR_PPRXI|ISR_PTXI|ISR_PPTXI))) {
            velocity_error(pInfo, isr_status);
        }

        if (isr_status & (ISR_PRXI|ISR_PPRXI))
            max_count+=velocity_rx_srv(pInfo, isr_status);

        if (isr_status & (ISR_PTXI|ISR_PPTXI))
            max_count+=velocity_tx_srv(pInfo, isr_status);

        isr_status=mac_read_isr(pInfo->pMacRegs);

        if (max_count>pInfo->sOpts.int_works)
                break;
    }
    spin_unlock(&pInfo->lock);
    mac_enable_int(pInfo->pMacRegs);

}



static unsigned const ethernet_polynomial = 0x04c11db7U;
static inline u32 ether_crc(int length, unsigned char *data)
{
    int crc = -1;

    while(--length >= 0) {
        unsigned char current_octet = *data++;
        int bit;
        for (bit = 0; bit < 8; bit++, current_octet >>= 1) {
            crc = (crc << 1) ^
                ((crc < 0) ^ (current_octet & 1) ? ethernet_polynomial : 0);
        }
    }
    return crc;
}

static void velocity_set_multi(struct net_device *dev) {
    PVELOCITY_INFO         pInfo       =   (PVELOCITY_INFO) dev->priv;
    PMAC_REGS           pMacRegs    =   pInfo->pMacRegs;
    u8                  rx_mode;
    int                 i;
    struct dev_mc_list  *mclist;

    if (dev->flags & IFF_PROMISC) {         /* Set promiscuous. */
        /* Unconditionally log net taps. */
        printk(KERN_NOTICE "%s: Promiscuous mode enabled.\n", dev->name);
        writel(0xffffffff, &pMacRegs->abyMARCAM[0]);
        writel(0xffffffff, &pMacRegs->abyMARCAM[4]);
        rx_mode = (RCR_AM|RCR_AB|RCR_PROM);
    }
    else if ((dev->mc_count > pInfo->multicast_limit)
        ||  (dev->flags & IFF_ALLMULTI)) {
        writel(0xffffffff, &pMacRegs->abyMARCAM[0]);
        writel(0xffffffff, &pMacRegs->abyMARCAM[4]);
        rx_mode = (RCR_AM|RCR_AB);
    }
    else {
        int offset=MCAM_SIZE-pInfo->multicast_limit;
        mac_get_cam_mask(pInfo->pMacRegs,pInfo->abyMCAMMask,VELOCITY_MULTICAST_CAM);

        for (i = 0, mclist = dev->mc_list; mclist && i < dev->mc_count;
            i++, mclist = mclist->next) {
            mac_set_cam(pInfo->pMacRegs,i+offset,mclist->dmi_addr,VELOCITY_MULTICAST_CAM);
            pInfo->abyMCAMMask[(offset+i)/8]|=1<<((offset+i) & 7);
        }

        mac_set_cam_mask(pInfo->pMacRegs,pInfo->abyMCAMMask,VELOCITY_MULTICAST_CAM);

        rx_mode=(RCR_AM|RCR_AB);
    }

    if (dev->mtu>1500)
        rx_mode|=RCR_AL;

    BYTE_REG_BITS_ON(rx_mode, &pMacRegs->byRCR);

}

static struct net_device_stats *velocity_get_stats(struct net_device *dev) {
    PVELOCITY_INFO pInfo=(PVELOCITY_INFO) dev->priv;

    spin_lock_irq(&pInfo->lock);
    velocity_update_hw_mibs(pInfo);
    spin_unlock_irq(&pInfo->lock);

    pInfo->stats.rx_packets=pInfo->adwHWMIBCounters[HW_MIB_ifRxAllPkts];
    pInfo->stats.rx_errors=pInfo->adwHWMIBCounters[HW_MIB_ifRxErrorPkts];
    pInfo->stats.rx_length_errors=pInfo->adwHWMIBCounters[HW_MIB_ifInRangeLengthErrors];

//  unsigned long   rx_dropped;     /* no space in linux buffers    */
    pInfo->stats.collisions=pInfo->adwHWMIBCounters[HW_MIB_ifTxEtherCollisions];
    /* detailed rx_errors: */
//  unsigned long   rx_length_errors;
//  unsigned long   rx_over_errors;     /* receiver ring buff overflow  */
    pInfo->stats.rx_crc_errors=pInfo->adwHWMIBCounters[HW_MIB_ifRxPktCRCE];
//  unsigned long   rx_frame_errors;    /* recv'd frame alignment error */
//  unsigned long   rx_fifo_errors;     /* recv'r fifo overrun      */
//  unsigned long   rx_missed_errors;   /* receiver missed packet   */

    /* detailed tx_errors */
//  unsigned long   tx_fifo_errors;

    return &pInfo->stats;
}


static int  velocity_ioctl(struct net_device *dev, struct ifreq *rq, int cmd) {
    switch(cmd) {
#ifdef VMNS
    case VMNS_DRV_SIOC: {
        PVMNS_DRV_SIOC_HEADER   pParams = (PVMNS_DRV_SIOC_HEADER) rq->ifr_data;
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        if (vmns_process_ioctl(dev, pParams) == 0)
            return 0;
        }
        return -EAGAIN;
#endif

#ifdef VELOCITY_ETHTOOL_IOCTL_SUPPORT
    case SIOCETHTOOL:
        return velocity_ethtool_ioctl(dev, rq);
        break;
#endif

#ifdef VELOCITY_MII_IOCTL_SUPPORT
    case SIOCGMIIPHY:       /* Get address of MII PHY in use. */
    case SIOCGMIIREG:       /* Read MII PHY register. */
    case SIOCSMIIREG:       /* Write to MII PHY register. */
        return velocity_mii_ioctl(dev, rq, cmd);
        break;
#endif

    default:
        return -EOPNOTSUPP;
    }
    return 0;
}



/*------------------------------------------------------------------*/

MODULE_DEVICE_TABLE(pci, velocity_id_table);

static struct pci_driver velocity_driver = {
        name:       VELOCITY_NAME,
        id_table:   velocity_id_table,
        probe:      velocity_found1,
        remove:     velocity_remove1,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM
        suspend:    velocity_suspend,
        resume:     velocity_resume,
#endif
#endif
};

static int __init velocity_init_module(void)
{
    int ret;
    ret=pci_module_init(&velocity_driver);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM
    register_inetaddr_notifier(&velocity_inetaddr_notifier);
    if(ret >= 0)
        register_reboot_notifier(&velocity_notifier);

#endif
#endif

    return ret;
}

static void __exit velocity_cleanup_module(void)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM
    unregister_reboot_notifier(&velocity_notifier);
    unregister_inetaddr_notifier(&velocity_inetaddr_notifier);
#endif
#endif
    pci_unregister_driver(&velocity_driver);
}

module_init(velocity_init_module);
module_exit(velocity_cleanup_module);


/************************************************************************
* MII access , media link mode setting functions
************************************************************************/
static void mii_init(PVELOCITY_INFO pInfo, U32 mii_status) {
    U16 wBMCR;

    switch (PHYID_GET_PHY_ID(pInfo->dwPHYid)) {
    case PHYID_CICADA_CS8201:
        // Reset to hardware default
        MII_REG_BITS_OFF((ANAR_ASMDIR|ANAR_PAUSE), MII_REG_ANAR, pInfo->pMacRegs);
        
        // turn on ECHODIS bit in NWay-forced full mode and turn off it in
        // NWay-forced half mode for NWay-forced v.s. legacy-forced issue
        if (pInfo->mii_status & VELOCITY_DUPLEX_FULL) 
        	MII_REG_BITS_ON(TCSR_ECHODIS, MII_REG_TCSR, pInfo->pMacRegs);
        else
            MII_REG_BITS_OFF(TCSR_ECHODIS, MII_REG_TCSR, pInfo->pMacRegs);
            
		// Turn on Link/Activity LED enable bit for CIS8201
		MII_REG_BITS_ON(PLED_LALBE, MII_REG_PLED, pInfo->pMacRegs);
	            
        break;
        
    case PHYID_VT3216_32BIT:
    case PHYID_VT3216_64BIT:
        // Reset to hardware default
        MII_REG_BITS_ON((ANAR_ASMDIR|ANAR_PAUSE), MII_REG_ANAR, pInfo->pMacRegs);
        
        // turn on ECHODIS bit in NWay-forced full mode and turn off it in
        // NWay-forced half mode for NWay-forced v.s. legacy-forced issue
        if (pInfo->mii_status & VELOCITY_DUPLEX_FULL) 
        	MII_REG_BITS_ON(TCSR_ECHODIS, MII_REG_TCSR, pInfo->pMacRegs);
        else
            MII_REG_BITS_OFF(TCSR_ECHODIS, MII_REG_TCSR, pInfo->pMacRegs);

        break;            

    case PHYID_MARVELL_1000:
    case PHYID_MARVELL_1000S:
        /* Assert CRS on Transmit */
        MII_REG_BITS_ON(PSCR_ACRSTX,MII_REG_PSCR,pInfo->pMacRegs);
        /*Reset to hardware default*/
        MII_REG_BITS_ON((ANAR_ASMDIR|ANAR_PAUSE),MII_REG_ANAR,pInfo->pMacRegs);
        break;
    default:
        ;
    }

    velocity_mii_read(pInfo->pMacRegs,MII_REG_BMCR,&wBMCR);
    if (wBMCR & BMCR_ISO) {
        wBMCR &=~BMCR_ISO;
        velocity_mii_write(pInfo->pMacRegs,MII_REG_BMCR, wBMCR);
    }

}

static void SafeDisableMiiAutoPoll (PMAC_REGS pMacRegs)
{
    WORD        ww;

    // turn off MAUTO
    writeb(0,&pMacRegs->byMIICR);
    for (ww = 0; ww < W_MAX_TIMEOUT; ww++) {
        udelay(1);
        if (BYTE_REG_BITS_IS_ON(MIISR_MIDLE, &pMacRegs->byMIISR))
            break;
    }
}

static void  EnableMiiAutoPoll(PMAC_REGS pMacRegs) {
    int ii;

    writeb(0,&(pMacRegs->byMIICR));
    writeb(MIIADR_SWMPL,&pMacRegs->byMIIADR);

    for (ii=0;ii<W_MAX_TIMEOUT; ii++) {
        udelay(1);
        if (BYTE_REG_BITS_IS_ON(MIISR_MIDLE, &pMacRegs->byMIISR))
            break;
    }

    writeb(MIICR_MAUTO,&pMacRegs->byMIICR);

    for (ii=0;ii<W_MAX_TIMEOUT; ii++) {
        udelay(1);
        if (!BYTE_REG_BITS_IS_ON(MIISR_MIDLE, &pMacRegs->byMIISR))
            break;
    }

}

BOOL velocity_mii_read(PMAC_REGS pMacRegs, U8 byIdx, PU16 pdata)
{
    WORD        ww;

    // disable MIICR_MAUTO, so that mii addr can be set normally
    SafeDisableMiiAutoPoll(pMacRegs);

    writeb(byIdx,&pMacRegs->byMIIADR);

    BYTE_REG_BITS_ON(MIICR_RCMD, &pMacRegs->byMIICR);

    for (ww = 0; ww < W_MAX_TIMEOUT; ww++) {
        if (!(readb(&pMacRegs->byMIICR) & MIICR_RCMD))
            break;
    }

    *pdata=readw(&pMacRegs->wMIIDATA);

    EnableMiiAutoPoll(pMacRegs);
    if (ww == W_MAX_TIMEOUT) {
        return FALSE;
    }
    return TRUE;
}

BOOL
velocity_mii_write (PMAC_REGS pMacRegs, BYTE byMiiAddr, WORD wData)
{
    WORD        ww;

    // disable MIICR_MAUTO, so that mii addr can be set normally
    SafeDisableMiiAutoPoll(pMacRegs);

    // MII reg offset
    writeb(byMiiAddr, &pMacRegs->byMIIADR);
    // set MII data
    writew(wData,&pMacRegs->wMIIDATA);

    // turn on MIICR_WCMD
    BYTE_REG_BITS_ON(MIICR_WCMD, &pMacRegs->byMIICR);

    // W_MAX_TIMEOUT is the timeout period
    for (ww = 0; ww < W_MAX_TIMEOUT; ww++) {
        udelay(5);
        if (!(readb(&pMacRegs->byMIICR) & MIICR_WCMD))
            break;
    }


    EnableMiiAutoPoll(pMacRegs);

    if (ww == W_MAX_TIMEOUT) {
        return FALSE;
    }

    return TRUE;
}

//
// Get the media mode stored in EEPROM or module options
//
static U32 velocity_get_opt_media_mode(PVELOCITY_INFO pInfo) {
    U32         status=0;
    
    switch (pInfo->sOpts.spd_dpx) {
    case SPD_DPX_AUTO:
        status=VELOCITY_AUTONEG_ENABLE;
        break;
    case SPD_DPX_100_FULL:
        status=VELOCITY_SPEED_100|VELOCITY_DUPLEX_FULL;
        break;
    case SPD_DPX_10_FULL:
        status=VELOCITY_SPEED_10|VELOCITY_DUPLEX_FULL;
        break;
    case SPD_DPX_100_HALF:
        status=VELOCITY_SPEED_100;
        break;
    case SPD_DPX_10_HALF:
        status=VELOCITY_SPEED_10;
        break;
    }
    pInfo->mii_status=status;
    return status;
}

static void mii_set_auto_on(PVELOCITY_INFO pInfo)
{
    if (MII_REG_BITS_IS_ON(BMCR_AUTO, MII_REG_BMCR ,pInfo->pMacRegs))
        MII_REG_BITS_ON(BMCR_REAUTO, MII_REG_BMCR ,pInfo->pMacRegs);
    else
        MII_REG_BITS_ON(BMCR_AUTO, MII_REG_BMCR, pInfo->pMacRegs);
}


/*
static void mii_set_auto_off(PVELOCITY_INFO pInfo)
{
    MII_REG_BITS_OFF(BMCR_AUTO, MII_REG_BMCR, pInfo->pMacRegs);
}
*/

static void set_mii_flow_control(PVELOCITY_INFO pInfo) {
    /*Enable or Disable PAUSE in ANAR*/
    switch(pInfo->sOpts.flow_cntl) {
    case FLOW_CNTL_TX:
        MII_REG_BITS_OFF(ANAR_PAUSE, MII_REG_ANAR, pInfo->pMacRegs);
        MII_REG_BITS_ON(ANAR_ASMDIR, MII_REG_ANAR, pInfo->pMacRegs);
        break;

    case FLOW_CNTL_RX:
        MII_REG_BITS_ON(ANAR_PAUSE, MII_REG_ANAR, pInfo->pMacRegs);
        MII_REG_BITS_ON(ANAR_ASMDIR, MII_REG_ANAR, pInfo->pMacRegs);
        break;

    case FLOW_CNTL_TX_RX:
        MII_REG_BITS_ON(ANAR_PAUSE, MII_REG_ANAR, pInfo->pMacRegs);
        MII_REG_BITS_ON(ANAR_ASMDIR, MII_REG_ANAR, pInfo->pMacRegs);
        break;

    case FLOW_CNTL_DISABLE:
        MII_REG_BITS_OFF(ANAR_PAUSE, MII_REG_ANAR, pInfo->pMacRegs);
        MII_REG_BITS_OFF(ANAR_ASMDIR, MII_REG_ANAR, pInfo->pMacRegs);
        break;
    default:
        break;
    }
}

int velocity_set_media_mode(PVELOCITY_INFO pInfo, U32 mii_status) {
    U32         curr_status;
    PMAC_REGS   pMacRegs=pInfo->pMacRegs;

    pInfo->mii_status=mii_check_media_mode(pInfo->pMacRegs);
    curr_status= pInfo->mii_status & (~VELOCITY_LINK_FAIL);

    //Set mii link status
    set_mii_flow_control(pInfo);
    
    /*
    //Check if new status is consisent with current status
    if (((mii_status & curr_status) & VELOCITY_AUTONEG_ENABLE)
        || (mii_status==curr_status)) {
        //pInfo->mii_status=mii_check_media_mode(pInfo->pMacRegs);
        pInfo->mii_status=check_connectiontype(pInfo->pMacRegs);
        VELOCITY_PRT(MSG_LEVEL_INFO, "Velocity link no change\n");
        return 0;
    }       
    */
    
    if (PHYID_GET_PHY_ID(pInfo->dwPHYid) == PHYID_CICADA_CS8201) {
        MII_REG_BITS_ON(AUXCR_MDPPS,MII_REG_AUXCR,pInfo->pMacRegs);
    }

    
    // if connection type is AUTO
    if (mii_status & VELOCITY_AUTONEG_ENABLE) {
        VELOCITY_PRT(MSG_LEVEL_INFO, "Velocity is AUTO mode\n");
    	// clear force MAC mode bit
    	BYTE_REG_BITS_OFF(CHIPGCR_FCMODE,&pMacRegs->byCHIPGCR);
        // set duplex mode of MAC according to duplex mode of MII
        MII_REG_BITS_ON(ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10,
                        MII_REG_ANAR, pInfo->pMacRegs);
        MII_REG_BITS_ON(G1000CR_1000FD|G1000CR_1000, MII_REG_G1000CR,
            pInfo->pMacRegs);
        MII_REG_BITS_ON(BMCR_SPEED1G, MII_REG_BMCR, pInfo->pMacRegs);

        // enable AUTO-NEGO mode
        mii_set_auto_on(pInfo);
        //pInfo->mii_status=check_connectiontype(pInfo->pMacRegs);
    }
    else {
        U16 wANAR;
        U8  byCHIPGCR;
        
        // 1. if it's 3119, disable frame bursting in halfduplex mode and
        //    enable it in fullduplex mode
        // 2. set correct MII/GMII and half/full duplex mode in CHIPGCR
        // 3. only enable CD heart beat counter in 10HD mode
        
        //mii_set_auto_off(pInfo);
        // set force MAC mode bit
        BYTE_REG_BITS_ON(CHIPGCR_FCMODE,&pMacRegs->byCHIPGCR);
        
        byCHIPGCR = readb(&pMacRegs->byCHIPGCR);
        byCHIPGCR &= ~CHIPGCR_FCGMII;
				
        if (mii_status & VELOCITY_DUPLEX_FULL) {
        	byCHIPGCR |= CHIPGCR_FCFDX;         
            writeb(byCHIPGCR, &pMacRegs->byCHIPGCR);
            VELOCITY_PRT(MSG_LEVEL_INFO, "set Velocity to forced full mode\n");
			if (pInfo->byRevId < REV_ID_VT3216_A0)
				BYTE_REG_BITS_OFF(TCR_TB2BDIS, &pMacRegs->byTCR);
		}
		else {
		    byCHIPGCR &= ~CHIPGCR_FCFDX;
	        VELOCITY_PRT(MSG_LEVEL_INFO, "set Velocity to forced half mode\n");
            writeb(byCHIPGCR, &pMacRegs->byCHIPGCR);
			if (pInfo->byRevId < REV_ID_VT3216_A0)
				BYTE_REG_BITS_ON(TCR_TB2BDIS, &pMacRegs->byTCR); 
		}
	
        MII_REG_BITS_OFF(G1000CR_1000FD|G1000CR_1000, MII_REG_G1000CR,
            pInfo->pMacRegs);
            
        if (!(mii_status & VELOCITY_DUPLEX_FULL) && (mii_status & VELOCITY_SPEED_10)) {
            BYTE_REG_BITS_OFF(TESTCFG_HBDIS, &pMacRegs->byTESTCFG);
        }
        else {
            BYTE_REG_BITS_ON(TESTCFG_HBDIS, &pMacRegs->byTESTCFG);
        }
        
        //MII_REG_BITS_OFF(BMCR_SPEED1G, MII_REG_BMCR, pInfo->pMacRegs);

        velocity_mii_read(pInfo->pMacRegs, MII_REG_ANAR, &wANAR);

        wANAR &= (~(ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10));

        if (mii_status & VELOCITY_SPEED_100) {
            if (mii_status & VELOCITY_DUPLEX_FULL) {
                wANAR |=ANAR_TXFD;
            }
            else {
                wANAR |=ANAR_TX;
            }
        }
        else {
            if (mii_status & VELOCITY_DUPLEX_FULL) {
                wANAR |=ANAR_10FD;
            }
            else {
                wANAR |=ANAR_10;
            }
        }
        velocity_mii_write(pInfo->pMacRegs,MII_REG_ANAR, wANAR);
        
        // enable AUTO-NEGO mode
        mii_set_auto_on(pInfo);
        //MII_REG_BITS_ON(BMCR_AUTO, MII_REG_BMCR, pInfo->pMacRegs);
    }
    //pInfo->mii_status=mii_check_media_mode(pInfo->pMacRegs);
    //pInfo->mii_status=check_connectiontype(pInfo->pMacRegs);
    return VELOCITY_LINK_CHANGE;
}

U32 mii_check_media_mode(PMAC_REGS pMacRegs) {
    U32                 status=0;
    U16                 wANAR;

    if (!MII_REG_BITS_IS_ON(BMSR_LNK,MII_REG_BMSR ,pMacRegs))
        status|=VELOCITY_LINK_FAIL;

    if (MII_REG_BITS_IS_ON(G1000CR_1000FD, MII_REG_G1000CR, pMacRegs))
        status|=VELOCITY_SPEED_1000|VELOCITY_DUPLEX_FULL;
    else if (MII_REG_BITS_IS_ON(G1000CR_1000, MII_REG_G1000CR, pMacRegs))
        status|=(VELOCITY_SPEED_1000);
    else {
        velocity_mii_read(pMacRegs, MII_REG_ANAR, &wANAR);
        if (wANAR & ANAR_TXFD)
            status|=(VELOCITY_SPEED_100|VELOCITY_DUPLEX_FULL);
        else if (wANAR & ANAR_TX)
            status|=VELOCITY_SPEED_100;
        else if (wANAR & ANAR_10FD)
            status|=(VELOCITY_SPEED_10|VELOCITY_DUPLEX_FULL);
        else
            status|=(VELOCITY_SPEED_10);
    }

    if (MII_REG_BITS_IS_ON(BMCR_AUTO,MII_REG_BMCR,pMacRegs)) {
        velocity_mii_read(pMacRegs, MII_REG_ANAR, &wANAR);
        if ((wANAR & (ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10))
            ==(ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10)) {
            if (MII_REG_BITS_IS_ON(G1000CR_1000|G1000CR_1000FD,
                MII_REG_G1000CR, pMacRegs))
                status|=VELOCITY_AUTONEG_ENABLE;
        }
    }

    return status;
}

U32 check_connectiontype(PMAC_REGS pMacRegs) {
    U32                 status=0;
    U8                  byPHYSR0;
    U16                 wANAR;
    byPHYSR0=readb(&pMacRegs->byPHYSR0);

	/*
    if (!(byPHYSR0 & PHYSR0_LINKGD))
        status|=VELOCITY_LINK_FAIL;
	*/

    if (byPHYSR0 & PHYSR0_FDPX)
        status|=VELOCITY_DUPLEX_FULL;

    if (byPHYSR0 & PHYSR0_SPDG)
        status|=VELOCITY_SPEED_1000;
    if (byPHYSR0 & PHYSR0_SPD10)
        status|=VELOCITY_SPEED_10;
    else
        status|=VELOCITY_SPEED_100;

    if (MII_REG_BITS_IS_ON(BMCR_AUTO,MII_REG_BMCR,pMacRegs)) {
        velocity_mii_read(pMacRegs, MII_REG_ANAR, &wANAR);
        if ((wANAR & (ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10))
            ==(ANAR_TXFD|ANAR_TX|ANAR_10FD|ANAR_10)) {
            if (MII_REG_BITS_IS_ON(G1000CR_1000|G1000CR_1000FD,
                MII_REG_G1000CR, pMacRegs))
                status|=VELOCITY_AUTONEG_ENABLE;
        }
    }

    return status;
}

static void enable_flow_control_ability(PVELOCITY_INFO pInfo) {

    PMAC_REGS   pMacRegs=pInfo->pMacRegs;

    switch (pInfo->sOpts.flow_cntl) {
    
        case FLOW_CNTL_DEFAULT:
            if (BYTE_REG_BITS_IS_ON(PHYSR0_RXFLC,&pMacRegs->byPHYSR0))
                writel(CR0_FDXRFCEN,&pMacRegs->dwCR0Set);
            else
                writel(CR0_FDXRFCEN,&pMacRegs->dwCR0Clr);

            if (BYTE_REG_BITS_IS_ON(PHYSR0_TXFLC,&pMacRegs->byPHYSR0))
                writel(CR0_FDXTFCEN,&pMacRegs->dwCR0Set);
            else
                writel(CR0_FDXTFCEN,&pMacRegs->dwCR0Clr);
            break;
        
        case FLOW_CNTL_TX:
            writel(CR0_FDXTFCEN, &pMacRegs->dwCR0Set);
            writel(CR0_FDXRFCEN, &pMacRegs->dwCR0Clr);
            break;
            
        case FLOW_CNTL_RX:
            writel(CR0_FDXRFCEN, &pMacRegs->dwCR0Set);
            writel(CR0_FDXTFCEN, &pMacRegs->dwCR0Clr);
            break;
            
        case FLOW_CNTL_TX_RX:
            writel(CR0_FDXTFCEN, &pMacRegs->dwCR0Set);
            writel(CR0_FDXRFCEN, &pMacRegs->dwCR0Set);
            break;
            
        case FLOW_CNTL_DISABLE:
            writel(CR0_FDXRFCEN, &pMacRegs->dwCR0Clr);
            writel(CR0_FDXTFCEN, &pMacRegs->dwCR0Clr);
            break;
            
        default:
            break;            
    }

}
/***************************************************************************
*    ETHTOOL ioctl support routine
****************************************************************************/

#ifdef VELOCITY_ETHTOOL_IOCTL_SUPPORT
static int  velocity_ethtool_ioctl(struct net_device* dev, struct ifreq* ifr) {
    struct ethtool_cmd  ecmd;
    PVELOCITY_INFO     pInfo=(PVELOCITY_INFO) dev->priv;
    PMAC_REGS           pMacRegs=pInfo->pMacRegs;

    if (copy_from_user(&ecmd, ifr->ifr_data, sizeof(ecmd.cmd)))
        return -EFAULT;

    switch (ecmd.cmd) {
    case ETHTOOL_GSET: {
        U32 status=check_connectiontype(pInfo->pMacRegs);
        ecmd.supported=
            SUPPORTED_TP|SUPPORTED_Autoneg|SUPPORTED_10baseT_Half|SUPPORTED_10baseT_Full
            |SUPPORTED_100baseT_Half|SUPPORTED_100baseT_Full|SUPPORTED_1000baseT_Half|SUPPORTED_1000baseT_Full;

        if (status & VELOCITY_SPEED_100)
            ecmd.speed=SPEED_100;
        else
            ecmd.speed=SPEED_10;

        ecmd.autoneg=(status & VELOCITY_AUTONEG_ENABLE) ? AUTONEG_ENABLE : AUTONEG_DISABLE;
        ecmd.port=PORT_TP;
        ecmd.transceiver=XCVR_INTERNAL;
        ecmd.phy_address=readb(&pMacRegs->byMIIADR) & 0x1F;

        if (status & VELOCITY_DUPLEX_FULL)
            ecmd.duplex=DUPLEX_FULL;
        else
            ecmd.duplex=DUPLEX_HALF;

        if(copy_to_user(ifr->ifr_data, &ecmd, sizeof(ecmd)))
               return -EFAULT;

        }
        break;

    case ETHTOOL_SSET: {
        U32 curr_status;
        U32 new_status=0;
        if (!capable(CAP_NET_ADMIN)){
                return -EPERM;
        }

/*      if (BYTE_REG_BITS_IS_ON(CFGC_MEDEN, &pMacRegs->byCFGC)) {
            printk(KERN_INFO
                "%s: The media mode have been forced by EEPROM utiltiy\n",dev->name);
            return -EPERM;
        }*/

        if (copy_from_user(&ecmd, ifr->ifr_data, sizeof(ecmd)))
               return -EFAULT;

        curr_status=check_connectiontype(pInfo->pMacRegs);
        curr_status&=(~VELOCITY_LINK_FAIL);

        new_status|=((ecmd.autoneg) ? VELOCITY_AUTONEG_ENABLE : 0);
        new_status|=((ecmd.speed==SPEED_100) ? VELOCITY_SPEED_100 : 0);
        new_status|=((ecmd.speed==SPEED_10) ? VELOCITY_SPEED_10 : 0);
        new_status|=((ecmd.duplex==DUPLEX_FULL) ? VELOCITY_DUPLEX_FULL : 0);

        if ((new_status & VELOCITY_AUTONEG_ENABLE) &&
            (new_status!=(curr_status| VELOCITY_AUTONEG_ENABLE)))
            return -EINVAL;

        velocity_set_media_mode(pInfo,new_status);
        }
        break;

#ifdef ETHTOOL_GLINK
    case ETHTOOL_GLINK: {
        struct ethtool_value info;
        memset((void *) &info, 0, sizeof (info));
        info.cmd = ETHTOOL_GLINK;
        info.data = BYTE_REG_BITS_IS_ON(PHYSR0_LINKGD,&pMacRegs->byPHYSR0)
                    ? FALSE : TRUE;

        if (copy_to_user(ifr->ifr_data, &info, sizeof (info)))
            return -EFAULT;
        }
        break;
#endif

#ifdef ETHTOOL_GDRVINFO
    case ETHTOOL_GDRVINFO:
        {
            struct ethtool_drvinfo info={ETHTOOL_GDRVINFO};
            strcpy(info.driver, VELOCITY_NAME);
            strcpy(info.version, VELOCITY_VERSION);
            strcpy(info.bus_info,pInfo->pcid->slot_name);
            if (copy_to_user(ifr->ifr_data, &info, sizeof(info)))
                return -EFAULT;
        }
        break;
#endif

#ifdef ETHTOOL_GWOL
    case ETHTOOL_GWOL: {

        struct ethtool_wolinfo  wol={ETHTOOL_GWOL};
        memset(&wol,0,sizeof(wol));

            wol.supported=WAKE_PHY | WAKE_MAGIC | WAKE_UCAST |
                    WAKE_ARP;
            wol.wolopts|=WAKE_MAGIC;
            if (pInfo->wol_opts & VELOCITY_WOL_PHY)
                wol.wolopts|=WAKE_PHY;
            if (pInfo->wol_opts & VELOCITY_WOL_UCAST)
                wol.wolopts|=WAKE_UCAST;
            if (pInfo->wol_opts & VELOCITY_WOL_ARP)
                wol.wolopts|=WAKE_ARP;

            memcpy(&wol.sopass,pInfo->wol_passwd,6);


        if (copy_to_user(ifr->ifr_data, &wol,sizeof(wol)))
            return -EFAULT;
        }
        break;
#endif

#ifdef ETHTOOL_SWOL
    case ETHTOOL_SWOL: {
        struct ethtool_wolinfo  wol;

        if (copy_from_user(&wol, ifr->ifr_data, sizeof(wol)))
            return -EFAULT;

        if (!(wol.wolopts & (WAKE_PHY|WAKE_MAGIC|WAKE_UCAST|WAKE_ARP)))
            return -EFAULT;
        pInfo->wol_opts=VELOCITY_WOL_MAGIC;

            if (wol.wolopts & WAKE_PHY) {
                pInfo->wol_opts|=VELOCITY_WOL_PHY;
                pInfo->flags |=VELOCITY_FLAGS_WOL_ENABLED;
            }
            if (wol.wolopts & WAKE_MAGIC) {
                pInfo->wol_opts|=VELOCITY_WOL_MAGIC;
                pInfo->flags |=VELOCITY_FLAGS_WOL_ENABLED;
            }
            if (wol.wolopts & WAKE_UCAST) {
                pInfo->wol_opts|=VELOCITY_WOL_UCAST;
                pInfo->flags |=VELOCITY_FLAGS_WOL_ENABLED;
            }
            if (wol.wolopts & WAKE_ARP) {
                pInfo->wol_opts|=VELOCITY_WOL_ARP;
                pInfo->flags |=VELOCITY_FLAGS_WOL_ENABLED;
            }
            memcpy(pInfo->wol_passwd,wol.sopass,6);

        if (copy_to_user(ifr->ifr_data, &wol,sizeof(wol)))
            return -EFAULT;
    }
    break;
#endif


#ifdef ETHTOOL_GMSGLVL
    case ETHTOOL_GMSGLVL: {
        struct ethtool_value edata={ETHTOOL_GMSGLVL};
        edata.data=msglevel;
        if (copy_to_user(ifr->ifr_data, &edata,sizeof(edata)))
            return -EFAULT;
        }
        break;
#endif

#ifdef ETHTOOL_SMSGLVL
    case ETHTOOL_SMSGLVL: {
        struct ethtool_value edata={ETHTOOL_SMSGLVL};
        if (copy_from_user(&edata, ifr->ifr_data, sizeof(edata)))
            return -EFAULT;
        msglevel=edata.data;
        }
        break;
#endif
    default:
        return -EOPNOTSUPP;
    }
    return 0;
}

#endif

/***************************************************************************
*    MII ioctl support routine
****************************************************************************/
#ifdef VELOCITY_MII_IOCTL_SUPPORT
static int  velocity_mii_ioctl(struct net_device* dev, struct ifreq* ifr,int cmd) {
    PVELOCITY_INFO pInfo=(PVELOCITY_INFO) dev->priv;
    PMAC_REGS       pMacRegs=pInfo->pMacRegs;
    int             flags;
    struct mii_ioctl_data* pMiiData=(struct mii_ioctl_data*) &(ifr->ifr_data);

    switch(cmd) {
    case SIOCGMIIPHY:
        pMiiData->phy_id=readb(&pMacRegs->byMIIADR) & 0x1f;
        break;

    case SIOCGMIIREG:
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        velocity_mii_read(pInfo->pMacRegs,pMiiData->reg_num & 0x1f, &(pMiiData->val_out));
        break;

    case SIOCSMIIREG:
        if (!capable(CAP_NET_ADMIN))
            return -EPERM;
        spin_lock_irqsave(&pInfo->lock,flags);
        velocity_mii_write(pInfo->pMacRegs,pMiiData->reg_num & 0x1f,pMiiData->val_in);
        spin_unlock_irqrestore(&pInfo->lock,flags);
        check_connectiontype(pInfo->pMacRegs);
        break;
    default:
          return -EOPNOTSUPP;
    }
    return 0;
}
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,9)
#ifdef CONFIG_PM
static int
velocity_notify_reboot(struct notifier_block *nb, unsigned long event, void *p)
{
    struct pci_dev *pdev = NULL;
    switch(event) {
    case SYS_DOWN:
    case SYS_HALT:
    case SYS_POWER_OFF:
        pci_for_each_dev(pdev) {
            if(pci_dev_driver(pdev) == &velocity_driver) {
                if (pci_get_drvdata(pdev))
                    velocity_suspend(pdev, 3);
            }
        }
    }
    return NOTIFY_DONE;
}

static int
velocity_suspend(struct pci_dev *pcid, u32 state)
{
    PVELOCITY_INFO pInfo=pci_get_drvdata(pcid);
    int flags;
    netif_stop_queue(pInfo->dev);
    spin_lock_irqsave(&pInfo->lock,flags);

    pci_save_state(pcid, pInfo->pci_state);

#ifdef ETHTOOL_GWOL

    if (pInfo->flags & VELOCITY_FLAGS_WOL_ENABLED) {
        velocity_get_ip(pInfo);
        velocity_save_context(pInfo,&pInfo->context);
        velocity_shutdown(pInfo);
        velocity_set_wol(pInfo);
        pci_enable_wake(pcid, 3, 1);
        pci_set_power_state(pcid, 3);
    } else {
        velocity_save_context(pInfo,&pInfo->context);
        velocity_shutdown(pInfo);
        pci_disable_device(pcid);
        pci_set_power_state(pcid, state);
    }
#else
    pci_disable_device(pcid);
    pci_set_power_state(pcid, state);
#endif
    spin_unlock_irqrestore(&pInfo->lock,flags);
    return 0;
}

static int
velocity_resume(struct pci_dev *pcid)
{
    PVELOCITY_INFO pInfo=pci_get_drvdata(pcid);
    int         flags;
    pci_set_power_state(pcid, 0);
    pci_enable_wake(pcid, 0, 0);
    pci_restore_state(pcid, pInfo->pci_state);

    mac_wol_reset(pInfo->pMacRegs);

    if (netif_running(pInfo->dev)) {
        int i;
        spin_lock_irqsave(&pInfo->lock,flags);
        velocity_restore_context(pInfo,&pInfo->context);
        velocity_init_registers(pInfo, VELOCITY_INIT_WOL);
        mac_disable_int(pInfo->pMacRegs);

        velocity_tx_srv(pInfo,0);

        for (i=0;i<pInfo->nTxQueues;i++) {
            if (pInfo->iTDUsed[i])  {
                mac_tx_queue_wake(pInfo->pMacRegs,i);
            }
        }
        spin_unlock_irqrestore(&pInfo->lock,flags);
        mac_enable_int(pInfo->pMacRegs);
        netif_start_queue(pInfo->dev);
    }

    return 0;
}

static int
velocity_netdev_event(struct notifier_block *nb,
    unsigned long notification, void *ptr) {
    struct in_ifaddr* ifa=(struct in_ifaddr*) ptr;
    struct net_device*  dev;
    PVELOCITY_INFO pInfo;

    if (ifa)  {
        dev=ifa->ifa_dev->dev;
        for (pInfo=pVelocity3_Infos;pInfo;pInfo=pInfo->next) {
            if (pInfo->dev==dev)
                velocity_get_ip(pInfo);
        }
    }
    return NOTIFY_DONE;
}
#endif
#endif

